"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;

var _opensearch_security_client = require("../../backend/opensearch_security_client");

var _tenant_resolver = require("../../multitenancy/tenant_resolver");

var _errors = require("../../errors");

var _common = require("../../../common");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;

    _defineProperty(this, "type", void 0);

    _defineProperty(this, "securityClient", void 0);

    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;

      // skip auth for APIs that do not require auth
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      }

      const authState = {}; // if browser request, auth logic is:
      //   1. check if request includes auth header or paramter(e.g. jwt in url params) is present, if so, authenticate with auth header.
      //   2. if auth header not present, check if auth cookie is present, if no cookie, send to authentication workflow
      //   3. verify whether auth cookie is valid, if not valid, send to authentication workflow
      //   4. if cookie is valid, pass to route handlers

      const authHeaders = {};
      let cookie;
      let authInfo; // if this is an REST API call, suppose the request includes necessary auth header
      // see https://www.elastic.co/guide/en/opensearch-dashboards/master/using-api.html

      if (this.requestIncludesAuthInfo(request)) {
        try {
          const additonalAuthHeader = await this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additonalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additonalAuthHeader);
          cookie = this.getCookie(request, authInfo); // set tenant from cookie if exist

          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();

          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }

          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // no auth header in request, try cookie
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }

        if (!cookie || !(await this.isValidCookie(cookie))) {
          // clear cookie
          this.sessionStorageFactory.asScoped(request).clear(); // for assets, we can still pass it to resource handler as notHandled.
          // marking it as authenticated may result in login pop up when auth challenge
          // is enabled.

          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          } // send to auth workflow


          return this.handleUnauthedRequest(request, response, toolkit);
        } // extend session expiration time


        if (this.config.session.keepalive) {
          cookie.expiryTime = Date.now() + this.config.session.ttl;
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } // cookie is valid
        // build auth header


        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additonalAuthHeader = await this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additonalAuthHeader);
      } // resolve tenant if necessary


      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo); // return 401 if no tenant available

          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          }

          authState.selectedTenant = tenant; // set tenant in header

          if (this.config.multitenancy.enabled && this.config.multitenancy.enable_aggregation_view) {
            // Store all saved objects in a single kibana index.
            Object.assign(authHeaders, {
              securitytenant: _common.GLOBAL_TENANT_SYMBOL
            });
          } else {
            Object.assign(authHeaders, {
              securitytenant: tenant
            });
          } // set tenant to cookie


          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);

          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }

            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }

          throw error;
        }
      }

      if (!authInfo) {
        authInfo = await this.securityClient.authinfo(request, authHeaders);
      }

      authState.authInfo = authInfo;
      return toolkit.authenticated({
        requestHeaders: authHeaders,
        state: authState
      });
    });

    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
    this.config = config;
  }

  authNotRequired(request) {
    const pathname = request.url.pathname;

    if (!pathname) {
      return false;
    } // allow requests to ignored routes


    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    } // allow requests to routes that doesn't require authentication


    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }

    return false;
  }

  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }

    const selectedTenant = (0, _tenant_resolver.resolveTenant)(request, authInfo.user_name, authInfo.roles, authInfo.tenants, this.config, cookie);
    return selectedTenant;
  }

  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  } // abstract functions for concrete auth types to implement


}

exports.AuthenticationType = AuthenticationType;

_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/api/core/capabilities', // FIXME: need to figureout how to bypass this API call
'/app/login']);

_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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