"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AnsiblePlaybook = void 0;
const path = require("path");
const vscode_uri_1 = require("vscode-uri");
const vscode_languageserver_1 = require("vscode-languageserver");
const commandRunner_1 = require("../utils/commandRunner");
/**
 * Acts as an interface to ansible-playbook command.
 */
class AnsiblePlaybook {
    /**
     *
     * @param connection establishes connection with the client
     * @param context provides workspace context of the client
     */
    constructor(connection, context) {
        var _a;
        this.connection = connection;
        this.context = context;
        this.useProgressTracker = false;
        this.useProgressTracker =
            !!((_a = context.clientCapabilities.window) === null || _a === void 0 ? void 0 : _a.workDoneProgress);
    }
    /**
     * Acts as an interface to ansible-playbook <file> --syntax-check command and a cache of its output.
     * ansible syntax-check may provide diagnostics for more than just the file for which
     * it was triggered, and this is reflected in the implementation.
     *
     * Perform ansible syntax-check for the given document.
     */
    doValidate(textDocument) {
        return __awaiter(this, void 0, void 0, function* () {
            const docPath = vscode_uri_1.URI.parse(textDocument.uri).path;
            let diagnostics = new Map();
            const progressTracker = this.useProgressTracker
                ? yield this.connection.window.createWorkDoneProgress()
                : {
                    begin: () => { },
                    done: () => { }, // eslint-disable-line @typescript-eslint/no-empty-function
                };
            const workingDirectory = vscode_uri_1.URI.parse(this.context.workspaceFolder.uri).path;
            const mountPaths = new Set([workingDirectory, path.dirname(docPath)]);
            const settings = yield this.context.documentSettings.get(textDocument.uri);
            progressTracker.begin("ansible syntax-check", undefined, "Processing files...");
            const commandRunner = new commandRunner_1.CommandRunner(this.connection, this.context, settings);
            try {
                // run ansible playbook syntax-check
                yield commandRunner.runCommand("ansible-playbook", `${docPath} --syntax-check`, workingDirectory, mountPaths);
            }
            catch (error) {
                if (error instanceof Error) {
                    const execError = error;
                    // This is the regex to extract the filename, line and column number from the stderr produced by syntax-check command
                    const ansibleSyntaxCheckRegex = /The error appears to be in '(?<filename>.*)': line (?<line>\d+), column (?<column>\d+)/;
                    const filteredErrorMessage = ansibleSyntaxCheckRegex.exec(execError.stderr);
                    diagnostics = filteredErrorMessage
                        ? this.processReport(execError.message, filteredErrorMessage.groups.filename, parseInt(filteredErrorMessage.groups.line), parseInt(filteredErrorMessage.groups.column))
                        : this.processReport(execError.message, docPath, 1, 1);
                    if (execError.stderr) {
                        this.connection.console.info(`[ansible syntax-check] ${execError.stderr}`);
                    }
                }
                else {
                    this.connection.console.error(`Exception in AnsibleSyntaxCheck service: ${JSON.stringify(error)}`);
                }
            }
            progressTracker.done();
            return diagnostics;
        });
    }
    processReport(result, fileName, line, column) {
        const diagnostics = new Map();
        if (!result) {
            this.connection.console.warn("Standard output from ansible syntax-check is suspiciously empty.");
            return diagnostics;
        }
        const start = {
            line: line - 1,
            character: column - 1,
        };
        const end = {
            line: line - 1,
            character: vscode_languageserver_1.integer.MAX_VALUE,
        };
        const range = {
            start,
            end,
        };
        const severity = vscode_languageserver_1.DiagnosticSeverity.Error;
        const locationUri = `file://${fileName}`;
        const fileDiagnostics = diagnostics.get(locationUri) || [];
        fileDiagnostics.push({
            message: result,
            range,
            severity,
            source: "Ansible",
        });
        diagnostics.set(locationUri, fileDiagnostics);
        return diagnostics;
    }
}
exports.AnsiblePlaybook = AnsiblePlaybook;
//# sourceMappingURL=ansiblePlaybook.js.map