/*
 * Copyright 2021 Palantir Technologies, Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import * as React from "react";
import { shallowCompareKeys } from "../../common/utils";
import { HotkeysDialog2 } from "../../components/hotkeys/hotkeysDialog2";
const initialHotkeysState = { hotkeys: [], isDialogOpen: false };
const noOpDispatch = () => null;
// N.B. we can remove this optional call guard once Blueprint depends on React 16
/**
 * A React context used to register and deregister hotkeys as components are mounted and unmounted in an application.
 * Users should take care to make sure that only _one_ of these is instantiated and used within an application, especially
 * if using global hotkeys.
 *
 * You will likely not be using this HotkeysContext directly, except in cases where you need to get a direct handle on an
 * exisitng context instance for advanced use cases involving nested HotkeysProviders.
 *
 * For more information, see the [HotkeysProvider documentation](https://blueprintjs.com/docs/#core/context/hotkeys-provider).
 */
export const HotkeysContext = React.createContext?.([initialHotkeysState, noOpDispatch]);
const hotkeysReducer = (state, action) => {
    switch (action.type) {
        case "ADD_HOTKEYS":
            // only pick up unique hotkeys which haven't been registered already
            const newUniqueHotkeys = [];
            for (const a of action.payload) {
                let isUnique = true;
                for (const b of state.hotkeys) {
                    isUnique &&= !shallowCompareKeys(a, b, { exclude: ["onKeyDown", "onKeyUp"] });
                }
                if (isUnique) {
                    newUniqueHotkeys.push(a);
                }
            }
            return {
                ...state,
                hotkeys: [...state.hotkeys, ...newUniqueHotkeys],
            };
        case "REMOVE_HOTKEYS":
            return {
                ...state,
                hotkeys: state.hotkeys.filter(key => action.payload.indexOf(key) === -1),
            };
        case "OPEN_DIALOG":
            return { ...state, isDialogOpen: true };
        case "CLOSE_DIALOG":
            return { ...state, isDialogOpen: false };
        default:
            return state;
    }
};
/**
 * Hotkeys context provider, necessary for the `useHotkeys` hook.
 */
export const HotkeysProvider = ({ children, dialogProps, renderDialog, value }) => {
    const hasExistingContext = value != null;
    const [state, dispatch] = value ?? React.useReducer(hotkeysReducer, initialHotkeysState);
    const handleDialogClose = React.useCallback(() => dispatch({ type: "CLOSE_DIALOG" }), []);
    const dialog = renderDialog?.(state, { handleDialogClose }) ?? (React.createElement(HotkeysDialog2, Object.assign({}, dialogProps, { isOpen: state.isDialogOpen, hotkeys: state.hotkeys, onClose: handleDialogClose })));
    // if we are working with an existing context, we don't need to generate our own dialog
    return (React.createElement(HotkeysContext.Provider, { value: [state, dispatch] },
        children,
        hasExistingContext ? undefined : dialog));
};
//# sourceMappingURL=hotkeysProvider.js.map