"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = __importDefault(require("assert"));
const path_1 = __importDefault(require("path"));
const typescript_1 = __importDefault(require("typescript"));
const documents_1 = require("../../../../src/lib/documents");
const ls_config_1 = require("../../../../src/ls-config");
const plugins_1 = require("../../../../src/plugins");
const TypeDefinitionProvider_1 = require("../../../../src/plugins/typescript/features/TypeDefinitionProvider");
const utils_1 = require("../../../../src/utils");
const testDir = path_1.default.join(__dirname, '..');
describe('TypeDefinitionProvider', () => {
    function getFullPath(filename) {
        return path_1.default.join(testDir, 'testfiles', 'typedefinition', filename);
    }
    function getUri(filename) {
        return (0, utils_1.pathToUrl)(getFullPath(filename));
    }
    function setup(filename) {
        const docManager = new documents_1.DocumentManager((textDocument) => new documents_1.Document(textDocument.uri, textDocument.text));
        const lsAndTsDocResolver = new plugins_1.LSAndTSDocResolver(docManager, [testDir], new ls_config_1.LSConfigManager());
        const provider = new TypeDefinitionProvider_1.TypeDefinitionProviderImpl(lsAndTsDocResolver);
        const filePath = getFullPath(filename);
        const document = docManager.openDocument({
            uri: (0, utils_1.pathToUrl)(filePath),
            text: typescript_1.default.sys.readFile(filePath) || ''
        });
        return { provider, document };
    }
    it('find type definition in TS file', async () => {
        const { document, provider } = setup('typedefinition.svelte');
        const typeDefs = await provider.getTypeDefinition(document, {
            line: 5,
            character: 15
        });
        assert_1.default.deepStrictEqual(typeDefs, [
            {
                range: {
                    start: {
                        line: 0,
                        character: 13
                    },
                    end: {
                        line: 0,
                        character: 30
                    }
                },
                uri: getUri('some-class.ts')
            }
        ]);
    });
    it('find type definition in same Svelte file', async () => {
        const { document, provider } = setup('typedefinition.svelte');
        const typeDefs = await provider.getTypeDefinition(document, {
            line: 6,
            character: 20
        });
        assert_1.default.deepStrictEqual(typeDefs, [
            {
                range: {
                    start: {
                        line: 3,
                        character: 10
                    },
                    end: {
                        line: 3,
                        character: 19
                    }
                },
                uri: getUri('typedefinition.svelte')
            }
        ]);
    });
    it('map definition of dts with declarationMap to source ', async () => {
        const { provider, document } = setup('../declaration-map/importing.svelte');
        const typeDefs = await provider.getTypeDefinition(document, { line: 1, character: 13 });
        assert_1.default.deepStrictEqual(typeDefs, [
            {
                range: {
                    end: { line: 1, character: 18 },
                    start: { line: 1, character: 16 }
                },
                uri: getUri('../declaration-map/declaration-map-project/index.ts')
            }
        ]);
    });
});
//# sourceMappingURL=TypeDefinitionProvider.test.js.map