"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createSys = void 0;
const language_core_1 = require("@volar/language-core");
const path_1 = require("path");
const utilities_1 = require("./typescript/utilities");
const dtsHost_1 = require("./dtsHost");
let currentCwd = '';
function createSys(ctx, ts, env, dtsHost) {
    let version = 0;
    const rootPath = env.uriToFileName(env.rootUri.toString());
    const sys = ts.sys;
    const root = {
        dirs: {},
        files: {},
        requested: false,
    };
    const promises = new Set();
    const fileWatcher = env.onDidChangeWatchedFiles?.(({ changes }) => {
        for (const change of changes) {
            const fileName = env.uriToFileName(change.uri);
            const dirName = path_1.posix.dirname(fileName);
            const baseName = path_1.posix.basename(fileName);
            const dir = getDir(dirName);
            if (dir.files[baseName]) { // is requested file
                version++;
                if (change.type === 1) {
                    dir.files[baseName] = { exists: true };
                }
                else if (change.type === 2) {
                    dir.files[baseName] = { exists: true };
                }
                else if (change.type === 3) {
                    dir.files[baseName] = { exists: false };
                }
            }
        }
    });
    return {
        get version() {
            return version;
        },
        dispose() {
            fileWatcher?.dispose();
        },
        args: sys?.args ?? [],
        newLine: sys?.newLine ?? '\n',
        useCaseSensitiveFileNames: sys?.useCaseSensitiveFileNames ?? false,
        realpath: sys?.realpath,
        write: sys?.write ?? (() => { }),
        writeFile: sys?.writeFile ?? (() => { }),
        createDirectory: sys?.createDirectory ?? (() => { }),
        exit: sys?.exit ?? (() => { }),
        getExecutingFilePath: sys?.getExecutingFilePath ?? (() => rootPath + '/__fake__.js'),
        getCurrentDirectory: () => rootPath,
        getModifiedTime,
        readFile,
        readDirectory,
        getDirectories,
        resolvePath,
        fileExists,
        directoryExists,
        async sync() {
            while (promises.size) {
                await Promise.all(promises);
            }
            return version;
        },
    };
    function resolvePath(fsPath) {
        if (sys) {
            if (currentCwd !== rootPath) {
                currentCwd = rootPath;
                // https://github.com/vuejs/language-tools/issues/2039
                // https://github.com/vuejs/language-tools/issues/2234
                if (sys.directoryExists(rootPath)) {
                    // https://github.com/vuejs/language-tools/issues/2480
                    try {
                        process.chdir(rootPath);
                    }
                    catch { }
                }
            }
            return sys.resolvePath(fsPath).replace(/\\/g, '/');
        }
        return path_1.posix.resolve(fsPath).replace(/\\/g, '/');
    }
    function getModifiedTime(fileName) {
        fileName = resolvePath(fileName);
        const dirPath = path_1.posix.dirname(fileName);
        const dir = getDir(dirPath);
        const name = path_1.posix.basename(fileName);
        const modifiedTime = dir.files[name]?.modifiedTime;
        if (modifiedTime !== undefined) {
            return new Date(modifiedTime);
        }
    }
    function readFile(fileName, encoding) {
        fileName = resolvePath(fileName);
        const dirPath = path_1.posix.dirname(fileName);
        const dir = getDir(dirPath);
        const name = path_1.posix.basename(fileName);
        readFileWorker(fileName, encoding, dir);
        return dir.files[name]?.text;
    }
    function directoryExists(dirName) {
        dirName = resolvePath(dirName);
        const dir = getDir(dirName);
        if (dirName === '/node_modules' && dtsHost) {
            dir.exists = true;
        }
        else if (dirName.startsWith('/node_modules/') && dtsHost && !(0, dtsHost_1.getPackageNameOfDtsPath)(dirName)) {
            dir.exists = true;
        }
        else if (dir.exists === undefined) {
            dir.exists = false;
            const result = dirName.startsWith('/node_modules/') && dtsHost
                ? dtsHost.stat(dirName)
                : env.fs?.stat(env.fileNameToUri(dirName));
            if (typeof result === 'object' && 'then' in result) {
                const promise = result;
                promises.add(promise);
                result.then(result => {
                    promises.delete(promise);
                    dir.exists = result?.type === 2;
                    if (dir.exists) {
                        version++;
                    }
                });
            }
            else {
                dir.exists = result?.type === 2;
            }
        }
        return dir.exists;
    }
    function fileExists(fileName) {
        fileName = resolvePath(fileName);
        const dirPath = path_1.posix.dirname(fileName);
        const baseName = path_1.posix.basename(fileName);
        const dir = getDir(dirPath);
        const file = dir.files[baseName] ??= {};
        if (file.exists === undefined) {
            file.exists = false;
            const result = fileName.startsWith('/node_modules/') && dtsHost
                ? dtsHost.stat(fileName)
                : env.fs?.stat(env.fileNameToUri(fileName));
            if (typeof result === 'object' && 'then' in result) {
                const promise = result;
                promises.add(promise);
                result.then(result => {
                    promises.delete(promise);
                    file.exists = result?.type === 1 || result?.type === 64;
                    if (file.exists) {
                        const time = Date.now();
                        file.modifiedTime = time !== file.modifiedTime ? time : time + 1;
                        version++;
                    }
                });
            }
            else {
                file.exists = result?.type === 1 || result?.type === 64;
            }
        }
        return file.exists;
    }
    // for import path completion
    function getDirectories(dirName) {
        dirName = resolvePath(dirName);
        readDirectoryWorker(dirName);
        const dir = getDir(dirName);
        return [...Object.entries(dir.dirs)].filter(([_, dir]) => dir.exists).map(([name]) => name);
    }
    function readDirectory(dirName, extensions, excludes, includes, depth) {
        dirName = resolvePath(dirName);
        let matches = (0, utilities_1.matchFiles)(dirName, extensions, excludes, includes, sys?.useCaseSensitiveFileNames ?? false, rootPath, depth, (dirPath) => {
            dirPath = resolvePath(dirPath);
            readDirectoryWorker(dirPath);
            const dir = getDir(dirPath);
            const virtualFiles = [];
            if (ctx) {
                for (const { root } of ctx.virtualFiles.allSources()) {
                    (0, language_core_1.forEachEmbeddedFile)(root, file => {
                        if (file.kind === language_core_1.FileKind.TypeScriptHostFile) {
                            const fileDirName = path_1.posix.dirname(file.fileName);
                            if (fileDirName.toLowerCase() === dirPath.toLowerCase()) {
                                virtualFiles.push(path_1.posix.basename(file.fileName));
                            }
                        }
                    });
                }
            }
            return {
                files: [
                    ...[...Object.entries(dir.files)].filter(([_, file]) => file.exists).map(([name]) => name),
                    ...virtualFiles,
                ],
                directories: [...Object.entries(dir.dirs)].filter(([_, dir]) => dir.exists).map(([name]) => name),
            };
        }, sys?.realpath ? (path => sys.realpath(path)) : (path => path));
        if (ctx) {
            matches = matches.map(match => {
                const [_, source] = ctx.virtualFiles.getVirtualFile(match);
                if (source) {
                    return source.fileName;
                }
                return match;
            });
        }
        return [...new Set(matches)];
    }
    function readFileWorker(fileName, encoding, dir) {
        const name = path_1.posix.basename(fileName);
        dir.files[name] ??= {};
        const file = dir.files[name];
        if (file.exists === false || file.requested) {
            return;
        }
        file.requested = true;
        const uri = env.fileNameToUri(fileName);
        const result = fileName.startsWith('/node_modules/') && dtsHost
            ? dtsHost.readFile(fileName)
            : env.fs?.readFile(uri, encoding);
        if (typeof result === 'object' && 'then' in result) {
            const promise = result;
            promises.add(promise);
            result.then(result => {
                promises.delete(promise);
                if (result !== undefined) {
                    file.exists = true;
                    file.text = result;
                    const time = Date.now();
                    file.modifiedTime = time !== file.modifiedTime ? time : time + 1;
                    version++;
                }
                else {
                    file.exists = false;
                }
            });
        }
        else if (result !== undefined) {
            file.exists = true;
            file.text = result;
            const time = Date.now();
            file.modifiedTime = time !== file.modifiedTime ? time : time + 1;
        }
        else {
            file.exists = false;
        }
    }
    function readDirectoryWorker(dirName) {
        const dir = getDir(dirName);
        if (dir.requested) {
            return;
        }
        dir.requested = true;
        const uri = env.fileNameToUri(dirName);
        const result = dirName.startsWith('/node_modules/') && dtsHost
            ? dtsHost.readDirectory(dirName)
            : env.fs?.readDirectory(uri);
        if (typeof result === 'object' && 'then' in result) {
            const promise = result;
            promises.add(promise);
            result.then((result) => {
                promises.delete(promise);
                if (onReadDirectoryResult(dir, result)) {
                    version++;
                }
            });
        }
        else {
            onReadDirectoryResult(dir, result ?? []);
        }
    }
    function onReadDirectoryResult(dir, result) {
        // See https://github.com/microsoft/TypeScript/blob/e1a9290051a3b0cbdfbadc3adbcc155a4641522a/src/compiler/sys.ts#L1853-L1857
        result = result.filter(([name]) => name !== '.' && name !== '..');
        let updated = false;
        for (const [name, fileType] of result) {
            if (fileType === 1 || fileType === 64) {
                dir.files[name] ??= {};
                if (!dir.files[name].exists) {
                    dir.files[name].exists = true;
                    updated = true;
                }
            }
            else if (fileType === 2) {
                const childDir = getDirFromDir(dir, name);
                if (!childDir.exists) {
                    childDir.exists = true;
                    updated = true;
                }
            }
        }
        return updated;
    }
    function getDir(dirName) {
        const dirNames = [];
        let currentDirPath = dirName;
        let currentDirName = path_1.posix.basename(currentDirPath);
        let lastDirName;
        while (lastDirName !== currentDirName) {
            lastDirName = currentDirName;
            dirNames.push(currentDirName);
            currentDirPath = path_1.posix.dirname(currentDirPath);
            currentDirName = path_1.posix.basename(currentDirPath);
        }
        let currentDir = root;
        for (let i = dirNames.length - 1; i >= 0; i--) {
            const nextDirName = dirNames[i];
            currentDir = getDirFromDir(currentDir, nextDirName);
        }
        return currentDir;
    }
    function getDirFromDir(dir, name) {
        let target = dir.dirs[name];
        if (!target) {
            target = {
                dirs: {},
                files: {},
            };
            dir.dirs[name] = target;
        }
        return target;
    }
}
exports.createSys = createSys;
//# sourceMappingURL=sys.js.map