"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createLanguageServiceHost = void 0;
const path_1 = require("path");
function createLanguageServiceHost(context, ts) {
    let lastProjectVersion;
    let tsProjectVersion = 0;
    const host = context.host;
    const virtualFiles = context.core.virtualFiles;
    const _tsHost = {
        fileExists: host.fileExists
            ? fileName => {
                const ext = fileName.substring(fileName.lastIndexOf('.'));
                if (ext === '.js'
                    || ext === '.ts'
                    || ext === '.jsx'
                    || ext === '.tsx') {
                    /**
                     * If try to access a external .vue file that outside of the project,
                     * the file will not process by language service host,
                     * so virtual file will not be created.
                     *
                     * We try to create virtual file here.
                     */
                    const sourceFileName = fileName.substring(0, fileName.lastIndexOf('.'));
                    if (!virtualFiles.hasSource(sourceFileName)) {
                        const scriptSnapshot = host.getScriptSnapshot(sourceFileName);
                        if (scriptSnapshot) {
                            virtualFiles.updateSource(sourceFileName, scriptSnapshot, host.getScriptLanguageId?.(sourceFileName));
                        }
                    }
                }
                if (virtualFiles.hasVirtualFile(fileName)) {
                    return true;
                }
                return !!host.fileExists?.(fileName);
            }
            : undefined,
        getProjectVersion: () => {
            return host.getTypeRootsVersion?.() + ':' + tsProjectVersion.toString();
        },
        getTypeRootsVersion: host.getTypeRootsVersion,
        getScriptFileNames,
        getScriptVersion,
        getScriptSnapshot,
        readDirectory: (_path, extensions, exclude, include, depth) => {
            const result = host.readDirectory?.(_path, extensions, exclude, include, depth) ?? [];
            for (const { fileName } of virtualFiles.allSources()) {
                const vuePath2 = path_1.posix.join(_path, path_1.posix.basename(fileName));
                if (path_1.posix.relative(_path.toLowerCase(), fileName.toLowerCase()).startsWith('..')) {
                    continue;
                }
                if (!depth && fileName.toLowerCase() === vuePath2.toLowerCase()) {
                    result.push(vuePath2);
                }
                else if (depth) {
                    result.push(vuePath2); // TODO: depth num
                }
            }
            return result;
        },
        getScriptKind(fileName) {
            if (ts) {
                if (virtualFiles.hasSource(fileName))
                    return ts.ScriptKind.Deferred;
                switch (path_1.posix.extname(fileName)) {
                    case '.js': return ts.ScriptKind.JS;
                    case '.jsx': return ts.ScriptKind.JSX;
                    case '.ts': return ts.ScriptKind.TS;
                    case '.tsx': return ts.ScriptKind.TSX;
                    case '.json': return ts.ScriptKind.JSON;
                    default: return ts.ScriptKind.Unknown;
                }
            }
            return 0;
        },
    };
    const scriptSnapshots = new Map();
    const virtualFileVersions = new Map();
    return new Proxy(_tsHost, {
        get: (target, property) => {
            update();
            return target[property] || host[property];
        },
    });
    function update() {
        const newProjectVersion = host.getProjectVersion?.();
        const shouldUpdate = newProjectVersion === undefined || newProjectVersion !== lastProjectVersion;
        if (!shouldUpdate)
            return;
        lastProjectVersion = newProjectVersion;
        const oldTsVirtualFileSnapshots = new Set();
        const oldOtherVirtualFileSnapshots = new Set();
        const newTsVirtualFileSnapshots = new Set();
        const newOtherVirtualFileSnapshots = new Set();
        for (const { root } of virtualFiles.allSources()) {
            forEachEmbeddedFile(root, embedded => {
                if (embedded.kind === 1) {
                    oldTsVirtualFileSnapshots.add(embedded.snapshot);
                }
                else {
                    oldOtherVirtualFileSnapshots.add(embedded.snapshot);
                }
            });
        }
        context.core.update();
        for (const { root } of virtualFiles.allSources()) {
            forEachEmbeddedFile(root, embedded => {
                if (embedded.kind === 1) {
                    newTsVirtualFileSnapshots.add(embedded.snapshot);
                }
                else {
                    newOtherVirtualFileSnapshots.add(embedded.snapshot);
                }
            });
        }
        if (!setEquals(oldTsVirtualFileSnapshots, newTsVirtualFileSnapshots)) {
            tsProjectVersion++;
        }
        else if (setEquals(oldOtherVirtualFileSnapshots, newOtherVirtualFileSnapshots)) {
            // no any meta language files update, it mean project version was update by source files this time
            tsProjectVersion++;
        }
    }
    function getScriptFileNames() {
        const tsFileNames = new Set();
        for (const { root: rootVirtualFile } of virtualFiles.allSources()) {
            forEachEmbeddedFile(rootVirtualFile, embedded => {
                if (embedded.kind === 1) {
                    tsFileNames.add(embedded.fileName); // virtual .ts
                }
            });
        }
        for (const fileName of host.getScriptFileNames()) {
            if (!virtualFiles.hasSource(fileName)) {
                tsFileNames.add(fileName); // .ts
            }
        }
        return [...tsFileNames];
    }
    function getScriptSnapshot(fileName) {
        const version = getScriptVersion(fileName);
        const cache = scriptSnapshots.get(fileName.toLowerCase());
        if (cache && cache[0] === version) {
            return cache[1];
        }
        const [virtualFile] = virtualFiles.getVirtualFile(fileName);
        if (virtualFile) {
            const snapshot = virtualFile.snapshot;
            scriptSnapshots.set(fileName.toLowerCase(), [version, snapshot]);
            return snapshot;
        }
        let tsScript = host.getScriptSnapshot(fileName);
        if (tsScript) {
            scriptSnapshots.set(fileName.toLowerCase(), [version, tsScript]);
            return tsScript;
        }
    }
    function getScriptVersion(fileName) {
        let [virtualFile, source] = virtualFiles.getVirtualFile(fileName);
        if (virtualFile && source) {
            let version = virtualFileVersions.get(virtualFile.fileName);
            if (!version) {
                version = {
                    value: 0,
                    virtualFileSnapshot: virtualFile.snapshot,
                    sourceFileSnapshot: source.snapshot,
                };
                virtualFileVersions.set(virtualFile.fileName, version);
            }
            else if (version.virtualFileSnapshot !== virtualFile.snapshot
                || (host.isTsc && version.sourceFileSnapshot !== source.snapshot) // fix https://github.com/johnsoncodehk/volar/issues/1082
            ) {
                version.value++;
                version.virtualFileSnapshot = virtualFile.snapshot;
                version.sourceFileSnapshot = source.snapshot;
            }
            return version.value.toString();
        }
        return host.getScriptVersion(fileName);
    }
}
exports.createLanguageServiceHost = createLanguageServiceHost;
function setEquals(a, b) {
    if (a.size !== b.size)
        return false;
    for (const item of a) {
        if (!b.has(item))
            return false;
    }
    return true;
}
function forEachEmbeddedFile(file, cb) {
    cb(file);
    for (const embeddedFile of file.embeddedFiles) {
        forEachEmbeddedFile(embeddedFile, cb);
    }
}
//# sourceMappingURL=languageServiceHost.js.map