// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/*
 * This public interface for constructing the additional authenticated data (AAD)
 * is provided for the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other than the
 * Encryption SDK for JavaScript.
 *
 * This AAD is used for the Body section
 * See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/body-aad-reference.html
 */
import BN from 'bn.js';
import { ContentAADString, ContentType } from './identifiers';
import { concatBuffers } from './concat_buffers';
import { uInt32BE } from './uint_util';
export function aadFactory(fromUtf8) {
    return {
        messageAADContentString,
        messageAAD,
    };
    function messageAADContentString({ contentType, isFinalFrame, }) {
        switch (contentType) {
            case ContentType.NO_FRAMING:
                return ContentAADString.NON_FRAMED_STRING_ID;
            case ContentType.FRAMED_DATA:
                return isFinalFrame
                    ? ContentAADString.FINAL_FRAME_STRING_ID
                    : ContentAADString.FRAME_STRING_ID;
            default:
                throw new Error('Unrecognized content type');
        }
    }
    function messageAAD(messageId, aadContentString, seqNum, contentLength) {
        return concatBuffers(messageId, fromUtf8(aadContentString), uInt32BE(seqNum), new Uint8Array(new BN(contentLength).toArray('be', 8)));
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWFkX2ZhY3RvcnkuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvYWFkX2ZhY3RvcnkudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsb0VBQW9FO0FBQ3BFLHNDQUFzQztBQUV0Qzs7Ozs7Ozs7R0FRRztBQUVILE9BQU8sRUFBRSxNQUFNLE9BQU8sQ0FBQTtBQUN0QixPQUFPLEVBQUUsZ0JBQWdCLEVBQUUsV0FBVyxFQUFFLE1BQU0sZUFBZSxDQUFBO0FBRTdELE9BQU8sRUFBRSxhQUFhLEVBQUUsTUFBTSxrQkFBa0IsQ0FBQTtBQUNoRCxPQUFPLEVBQUUsUUFBUSxFQUFFLE1BQU0sYUFBYSxDQUFBO0FBRXRDLE1BQU0sVUFBVSxVQUFVLENBQUMsUUFBdUM7SUFDaEUsT0FBTztRQUNMLHVCQUF1QjtRQUN2QixVQUFVO0tBQ1gsQ0FBQTtJQUVELFNBQVMsdUJBQXVCLENBQUMsRUFDL0IsV0FBVyxFQUNYLFlBQVksR0FJYjtRQUNDLFFBQVEsV0FBVyxFQUFFO1lBQ25CLEtBQUssV0FBVyxDQUFDLFVBQVU7Z0JBQ3pCLE9BQU8sZ0JBQWdCLENBQUMsb0JBQW9CLENBQUE7WUFDOUMsS0FBSyxXQUFXLENBQUMsV0FBVztnQkFDMUIsT0FBTyxZQUFZO29CQUNqQixDQUFDLENBQUMsZ0JBQWdCLENBQUMscUJBQXFCO29CQUN4QyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsZUFBZSxDQUFBO1lBQ3RDO2dCQUNFLE1BQU0sSUFBSSxLQUFLLENBQUMsMkJBQTJCLENBQUMsQ0FBQTtTQUMvQztJQUNILENBQUM7SUFFRCxTQUFTLFVBQVUsQ0FDakIsU0FBcUIsRUFDckIsZ0JBQWtDLEVBQ2xDLE1BQWMsRUFDZCxhQUFxQjtRQUVyQixPQUFPLGFBQWEsQ0FDbEIsU0FBUyxFQUNULFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUMxQixRQUFRLENBQUMsTUFBTSxDQUFDLEVBQ2hCLElBQUksVUFBVSxDQUFDLElBQUksRUFBRSxDQUFDLGFBQWEsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FDdkQsQ0FBQTtJQUNILENBQUM7QUFDSCxDQUFDIn0=