"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _constants = require("../utils/constants");

var _helpers = require("../utils/helpers");

var _DataStreamService = require("./DataStreamService");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class IndexService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "getIndices", async (context, request, response) => {
      try {
        // @ts-ignore
        const {
          from,
          size,
          sortField,
          sortDirection,
          terms,
          indices,
          dataStreams,
          showDataStreams,
          expandWildcards,
          exactSearch
        } = request.query;
        const params = {
          index: (0, _helpers.getSearchString)(terms, indices, dataStreams),
          format: "json",
          s: `${sortField}:${sortDirection}`
        };

        if (expandWildcards) {
          params.expand_wildcards = expandWildcards;
        }

        if (exactSearch) {
          params.index = exactSearch;
        }

        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const [recoverys, tasks, indicesResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("cat.recovery", {
          format: "json",
          detailed: true
        }).catch(() => []), callWithRequest("cat.tasks", {
          format: "json",
          detailed: true,
          actions: "indices:data/write/reindex"
        }).catch(() => []), callWithRequest("cat.indices", params), (0, _DataStreamService.getIndexToDataStreamMapping)({
          callAsCurrentUser: callWithRequest
        })]);
        const formattedTasks = tasks.map(item => {
          const {
            description
          } = item;
          const regexp = /reindex from \[([^\]]+)\] to \[([^\]]+)\]/i;
          const matchResult = description.match(regexp);

          if (matchResult) {
            const [, fromIndex, toIndex] = matchResult;
            return { ...item,
              fromIndex,
              toIndex
            };
          } else {
            return { ...item,
              fromIndex: "",
              toIndex: ""
            };
          }
        });
        const onGoingRecovery = recoverys.filter(item => item.stage !== "done"); // Augment the indices with their parent data stream name.

        indicesResponse.forEach(index => {
          index.data_stream = indexToDataStreamMapping[index.index] || null;
          let extraStatus = index.status;

          if (index.health === "green") {
            if (formattedTasks.find(item => item.toIndex === index.index)) {
              extraStatus = "reindex";
            }
          } else {
            if (onGoingRecovery.find(item => item.index === index.index)) {
              extraStatus = "recovery";
            }
          }

          if (extraStatus) {
            index.extraStatus = extraStatus;
          }
        });

        if (sortField === "status") {
          // add new more status to status field so we need to sort
          indicesResponse.sort((a, b) => {
            let flag;
            const aStatus = a.extraStatus;
            const bStatus = b.extraStatus;

            if (sortDirection === "asc") {
              flag = aStatus < bStatus;
            } else {
              flag = aStatus > bStatus;
            }

            return flag ? -1 : 1;
          });
        } // Filtering out indices that belong to a data stream. This must be done before pagination.


        const filteredIndices = showDataStreams ? indicesResponse : indicesResponse.filter(index => index.data_stream === null); // _cat doesn't support pagination, do our own in server pagination to at least reduce network bandwidth

        const fromNumber = parseInt(from, 10);
        const sizeNumber = parseInt(size, 10);
        const paginatedIndices = filteredIndices.slice(fromNumber, fromNumber + sizeNumber);
        const indexNames = paginatedIndices.map(value => value.index);
        const managedStatus = await this._getManagedStatus(request, indexNames); // NOTE: Cannot use response.ok due to typescript type checking

        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              indices: paginatedIndices.map(catIndex => ({ ...catIndex,
                managed: managedStatus[catIndex.index] ? "Yes" : "No",
                managedPolicy: managedStatus[catIndex.index]
              })),
              totalIndices: filteredIndices.length
            }
          }
        });
      } catch (err) {
        // Throws an error if there is no index matching pattern
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                indices: [],
                totalIndices: 0
              }
            }
          });
        }

        console.error("Index Management - IndexService - getIndices:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "_getManagedStatus", async (request, indexNames) => {
      try {
        const explainParamas = {
          index: indexNames.toString()
        };
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const explainResponse = await callWithRequest("ism.explain", explainParamas);
        const managed = {};

        for (const indexName in explainResponse) {
          if (indexName === "total_managed_indices") continue;
          const explain = explainResponse[indexName];
          managed[indexName] = explain["index.plugins.index_state_management.policy_id"] === null ? "" : explain["index.plugins.index_state_management.policy_id"];
        }

        return managed;
      } catch (err) {
        // otherwise it could be an unauthorized access error to config index or some other error
        // in which case we will return managed status N/A
        console.error("Index Management - IndexService - _getManagedStatus:", err);
        return indexNames.reduce((accu, value) => ({ ...accu,
          [value]: "N/A"
        }), {});
      }
    });

    _defineProperty(this, "applyPolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId
          }
        };
        const addResponse = await callWithRequest("ism.add", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - applyPolicy:", err); // return { ok: false, error: err.message };

        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    _defineProperty(this, "editRolloverAlias", async (context, request, response) => {
      try {
        const {
          alias,
          index
        } = request.body;
        const {
          callAsCurrentUser: callWithRequest
        } = this.osDriver.asScoped(request);
        const params = {
          index,
          body: {
            [_constants.Setting.RolloverAlias]: alias
          }
        };
        const rollOverResponse = await callWithRequest("indices.putSettings", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: rollOverResponse
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - editRolloverAlias", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }

}

exports.default = IndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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