"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = __importDefault(require("assert"));
const path_1 = require("path");
const sinon_1 = __importDefault(require("sinon"));
const typescript_1 = __importDefault(require("typescript"));
const vscode_languageserver_1 = require("vscode-languageserver");
const documents_1 = require("../../../../src/lib/documents");
const ls_config_1 = require("../../../../src/ls-config");
const UpdateImportsProvider_1 = require("../../../../src/plugins/typescript/features/UpdateImportsProvider");
const LSAndTSDocResolver_1 = require("../../../../src/plugins/typescript/LSAndTSDocResolver");
const utils_1 = require("../../../../src/utils");
const testDir = (0, path_1.join)(__dirname, '..');
const testFilesDir = (0, path_1.join)(testDir, 'testfiles', 'update-imports');
describe('UpdateImportsProviderImpl', () => {
    async function setup(filename, useCaseSensitiveFileNames) {
        const docManager = new documents_1.DocumentManager((textDocument) => new documents_1.Document(textDocument.uri, textDocument.text), { useCaseSensitiveFileNames });
        const lsAndTsDocResolver = new LSAndTSDocResolver_1.LSAndTSDocResolver(docManager, [(0, utils_1.pathToUrl)(testDir)], new ls_config_1.LSConfigManager(), { tsSystem: { ...typescript_1.default.sys, useCaseSensitiveFileNames } });
        const updateImportsProvider = new UpdateImportsProvider_1.UpdateImportsProviderImpl(lsAndTsDocResolver);
        const filePath = (0, path_1.join)(testFilesDir, filename);
        const fileUri = (0, utils_1.pathToUrl)(filePath);
        const document = docManager.openDocument({
            uri: fileUri,
            text: typescript_1.default.sys.readFile(filePath) || ''
        });
        await lsAndTsDocResolver.getLSAndTSDoc(document); // this makes sure ts ls knows the file
        return { updateImportsProvider, fileUri };
    }
    afterEach(() => sinon_1.default.restore());
    it('updates imports', async () => {
        const { updateImportsProvider, fileUri } = await setup('updateimports.svelte', typescript_1.default.sys.useCaseSensitiveFileNames);
        const workspaceEdit = await updateImportsProvider.updateImports({
            oldUri: (0, utils_1.pathToUrl)((0, path_1.join)(testFilesDir, 'imported.svelte')),
            newUri: (0, utils_1.pathToUrl)((0, path_1.join)(testFilesDir, 'documentation.svelte'))
        });
        assert_1.default.deepStrictEqual(workspaceEdit?.documentChanges, [
            vscode_languageserver_1.TextDocumentEdit.create(vscode_languageserver_1.OptionalVersionedTextDocumentIdentifier.create(fileUri, null), [
                vscode_languageserver_1.TextEdit.replace(vscode_languageserver_1.Range.create(vscode_languageserver_1.Position.create(1, 17), vscode_languageserver_1.Position.create(1, 34)), './documentation.svelte')
            ])
        ]);
    });
    async function testUpdateForFileCasingChanges(useCaseSensitiveFileNames) {
        const { updateImportsProvider, fileUri } = await setup('updateimports.svelte', useCaseSensitiveFileNames);
        const workspaceEdit = await updateImportsProvider.updateImports({
            oldUri: (0, utils_1.pathToUrl)((0, path_1.join)(testFilesDir, 'imported.svelte')),
            newUri: (0, utils_1.pathToUrl)((0, path_1.join)(testFilesDir, 'Imported.svelte'))
        });
        assert_1.default.deepStrictEqual(workspaceEdit?.documentChanges, [
            vscode_languageserver_1.TextDocumentEdit.create(vscode_languageserver_1.OptionalVersionedTextDocumentIdentifier.create(fileUri, null), [
                vscode_languageserver_1.TextEdit.replace(vscode_languageserver_1.Range.create(vscode_languageserver_1.Position.create(1, 17), vscode_languageserver_1.Position.create(1, 34)), './Imported.svelte')
            ])
        ]);
    }
    it('updates imports for file casing changes (case-sensitive)', async () => {
        await testUpdateForFileCasingChanges(true);
    });
    it('updates imports for file casing changes (case-insensitive)', async () => {
        await testUpdateForFileCasingChanges(false);
    });
});
//# sourceMappingURL=UpdateImportsProvider.test.js.map