#
# -*- coding: utf-8 -*-
# Copyright 2019 Red Hat
# GNU General Public License v3.0+
# (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)
"""
The eos lldp_global fact class
It is in this file the configuration is collected from the device
for a given resource, parsed, and the facts tree is populated
based on the configuration.
"""

from __future__ import absolute_import, division, print_function


__metaclass__ = type

import re

from copy import deepcopy

from ansible_collections.ansible.netcommon.plugins.module_utils.network.common import (
    utils,
)

from ansible_collections.arista.eos.plugins.module_utils.network.eos.argspec.lldp_global.lldp_global import (
    Lldp_globalArgs,
)


class Lldp_globalFacts(object):
    """The eos lldp_global fact class"""

    def __init__(self, module, subspec="config", options="options"):
        self._module = module
        self.argument_spec = Lldp_globalArgs.argument_spec
        spec = deepcopy(self.argument_spec)
        if subspec:
            if options:
                facts_argument_spec = spec[subspec][options]
            else:
                facts_argument_spec = spec[subspec]
        else:
            facts_argument_spec = spec

        self.generated_spec = utils.generate_dict(facts_argument_spec)

    def get_device_data(self, connection):
        return connection.get("show running-config | section lldp")

    def populate_facts(self, connection, ansible_facts, data=None):
        """Populate the facts for lldp_global
        :param connection: the device connection
        :param ansible_facts: Facts dictionary
        :param data: previously collected conf
        :rtype: dictionary
        :returns: facts
        """
        if not data:
            data = self.get_device_data(connection)

        obj = {}
        if data:
            obj.update(self.render_config(self.generated_spec, data))

        ansible_facts["ansible_network_resources"].pop("lldp_global", None)
        facts = {}
        if obj:
            params = utils.validate_config(self.argument_spec, {"config": obj})
            facts["lldp_global"] = utils.remove_empties(params["config"])
        else:
            facts["lldp_global"] = {}

        ansible_facts["ansible_network_resources"].update(facts)
        return ansible_facts

    def render_config(self, spec, conf):
        """
        Render config as dictionary structure and delete keys
          from spec for null values

        :param spec: The facts tree, generated from the argspec
        :param conf: The configuration
        :rtype: dictionary
        :returns: The generated config
        """
        config = deepcopy(spec)
        config["holdtime"] = utils.parse_conf_arg(conf, "hold-time")
        config["reinit"] = utils.parse_conf_arg(conf, "timer reinitialization")
        config["timer"] = utils.parse_conf_arg(conf, "timer")
        if config.get("timer") and "reinitialization" in config["timer"]:
            config["timer"] = None

        for match in re.findall(
            r"^(no)? lldp tlv transmit (\S+)",
            conf,
            re.MULTILINE,
        ):
            tlv_option = match[1].replace("-", "_")
            config["tlv_select"][tlv_option] = bool(match[0] != "no")

        return utils.remove_empties(config)
