/*******************************************************************************
 * Copyright (c) 2006, 2013 Wind River Systems and others.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Wind River Systems - initial API and implementation
 *     Navid Mehregani (TI) - Bug 289526 - Migrate the Restart feature to the new one, as supported by the platform
 *     Patrick Chuong (Texas Instruments) - Add support for icon overlay in the debug view (Bug 334566)
 *     Alvaro Sanchez-Leon (Ericsson AB) - Support for Step into selection (bug 244865)
 *******************************************************************************/
package org.eclipse.cdt.dsf.iss.ui;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.WeakHashMap;

import org.eclipse.cdt.dsf.gdb.internal.ui.GdbAdapterFactory;
import org.eclipse.cdt.dsf.gdb.launching.GdbLaunch;
import org.eclipse.cdt.dsf.iss.launching.IssGdbLaunch;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;

@SuppressWarnings("restriction")
public class GdbIssAdapterFactory extends GdbAdapterFactory {
	private static Map<IssGdbLaunch, IssGdbSessionAdapters> fgLaunchAdapterSets =
	        Collections.synchronizedMap(new HashMap<IssGdbLaunch, IssGdbSessionAdapters>());
	
    private static Map<ILaunch, IssGdbSessionAdapters> fgDisposedLaunchAdapterSets = new WeakHashMap<>();
    
    static void disposeAdapterSet(ILaunch launch) {
    	synchronized(fgLaunchAdapterSets) {
    		if (fgLaunchAdapterSets.containsKey(launch)) {
    			fgLaunchAdapterSets.remove(launch).dispose();
    			fgDisposedLaunchAdapterSets.put(launch, null);
    		}
    	}
    }
    
    public GdbIssAdapterFactory() {
        DebugPlugin.getDefault().getLaunchManager().addLaunchListener(this);
    }	
    
    
    /**
     * This method only actually returns adapters for the launch object.
     */
    @Override
	public <T> T getAdapter(Object adaptableObject, Class<T> adapterType) {
        if (!(adaptableObject instanceof GdbLaunch)) return null;

        IssGdbLaunch launch = (IssGdbLaunch)adaptableObject;

        // Check for valid session.  
        // Note: even if the session is no longer active, the adapter set 
        // should still be returned.  This is because the view model may still
        // need to show elements representing a terminated process/thread/etc.
        DsfSession session = launch.getSession();
        if (session == null) return null;

        // Find the correct set of adapters based on the launch session-ID.  If not found
        // it means that we have a new launch and new session, and we have to create a
        // new set of adapters.

        IssGdbSessionAdapters adapterSet;
        synchronized(fgLaunchAdapterSets) {
            // The adapter set for the given launch was already disposed.  
            // Return a null adapter.
            if (fgDisposedLaunchAdapterSets.containsKey(launch)) {
                return null;
            }
            adapterSet = fgLaunchAdapterSets.get(launch);
            if (adapterSet == null) {
            	// If the first time we attempt to create an adapterSet is once the session is
            	// already inactive, we should not create it and return null.
            	// This can happen, for example, when we run JUnit tests and we don't actually
            	// have a need for any adapters until the launch is actually being removed.
            	// Note that we must do this here because fgDisposedLaunchAdapterSets
            	// may not already know that the launch has been removed because of a race
            	// condition with the caller which is also processing a launchRemoved method.
            	// Bug 334687 
            	if (session.isActive() == false) {
            		return null;
            	}
                adapterSet = createGdbSessionAdapters(launch, session);
                fgLaunchAdapterSets.put(launch, adapterSet);
            }
        }
        
        // Returns the adapter type for the launch object.
        return adapterSet.getLaunchAdapter(adapterType);
    }
    
    @Override
    protected IssGdbSessionAdapters createGdbSessionAdapters(ILaunch launch, DsfSession session) {
    	return new IssGdbSessionAdapters(launch, session, getAdapterList());
    }
}
