/*******************************************************************************
 * Copyright (c) 2015 - 2020 Intel Corporation.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Intel Corporation - Implementation for Function Call History View
 *******************************************************************************/
package org.eclipse.cdt.dsf.iss.ui.fch;

import java.util.HashMap;
import org.eclipse.core.commands.Command;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.debug.internal.ui.DelegatingModelPresentation;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IModelChangedListener;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IModelDelta;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IModelProxy;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IPresentationContext;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IViewerInputRequestor;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IViewerInputUpdate;
import org.eclipse.debug.internal.ui.viewers.model.provisional.TreeModelViewer;
import org.eclipse.debug.internal.ui.viewers.model.provisional.ViewerInputService;
import org.eclipse.debug.internal.ui.views.DebugModelPresentationContext;
import org.eclipse.debug.ui.AbstractDebugView;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.contexts.AbstractDebugContextProvider;
import org.eclipse.debug.ui.contexts.DebugContextEvent;
import org.eclipse.debug.ui.contexts.IDebugContextListener;
import org.eclipse.debug.ui.contexts.IDebugContextService;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.commands.ICommandService;
import org.eclipse.ui.contexts.IContextActivation;
import org.eclipse.ui.contexts.IContextService;
import org.eclipse.ui.part.IPageBookViewPage;
import org.eclipse.ui.services.IEvaluationService;

@SuppressWarnings({ "restriction" })
public class FunctionCallHistoryView extends AbstractDebugView implements
IDebugContextListener {

	/**
	 * The ID of the view as specified by the extension.
	 */
	public static final String ID = "org.eclipse.cdt.dsf.iss.ui.fch.view"; //$NON-NLS-1$
	public static final String Context_ID = "org.eclipse.cdt.dsf.iss.ui.fch.context"; //$NON-NLS-1$

	private IDebugModelPresentation fPresentation = null;
	private IPresentationContext fPresentationContext = null;
	private IContextActivation fActivatedContext = null;
	private SelectionListener fScrollSelectionListener = null;
	private ScrollBar fscrollV = null;
	private ScrollBar fscrollH = null;
	private Tree fTree = null;
    private ViewerInputService fInputService;
    
    /**
     * Viewer input requester used to update the viewer once the viewer input has been
     * resolved.
     */
    private IViewerInputRequestor fRequester = new IViewerInputRequestor() {
         @Override
        public void viewerInputComplete(IViewerInputUpdate update) {
            if (!update.isCanceled()) {
                viewerInputUpdateComplete(update);
             }
         }
    };

    /**
     * Called when the viewer input update is completed.  Unlike
     * {@link #setViewerInput(Object)}, it allows overriding classes
     * to examine the context for which the update was calculated.
     *
     * @param update Completed viewer input update.
     */
    protected void viewerInputUpdateComplete(IViewerInputUpdate update) {
        setViewerInput(update.getInputElement());
    }

    /**
     * Sets the input to the viewer
     * @param context the object context
     */
    protected void setViewerInput(Object context) {
    	Object current = getViewer().getInput();

    	if (current == null && context == null) {
    		return;
    	}

    	if (current != null && current.equals(context)) {
    		return;
    	}
    	showViewer();
    	getViewer().setInput(context);
    	updateObjects();  
    }


	class TreeViewerContextProvider extends AbstractDebugContextProvider
	implements IModelChangedListener {

		private ISelection fContext = null;

		private TreeModelViewer fViewer = null;

		public TreeViewerContextProvider(TreeModelViewer viewer) {
			super(FunctionCallHistoryView.this);
			fViewer = viewer;
			fViewer.addModelChangedListener(this);

		}

		protected void dispose() {
			fContext = null;
			fViewer.removeModelChangedListener(this);
			fViewer = null;
		}

		/*
		 * (non-Javadoc)
		 *
		 * @see
		 * org.eclipse.debug.ui.contexts.IDebugContextProvider#getActiveContext
		 * ()
		 */
		@Override
		public synchronized ISelection getActiveContext() {
			return fContext;
		}

		protected void activate(ISelection selection) {
			synchronized (this) {
				fContext = selection;
			}
			fire(new DebugContextEvent(this, selection,
					DebugContextEvent.ACTIVATED));
		}

		/*
		 * (non-Javadoc)
		 *
		 * @see org.eclipse.debug.internal.ui.viewers.model.provisional.
		 * IModelChangedListener
		 * #modelChanged(org.eclipse.debug.internal.ui.viewers
		 * .model.provisional.IModelDelta)
		 */
		@Override
		public void modelChanged(IModelDelta delta, IModelProxy proxy) {
			IModelDelta[] childDeltas = delta.getChildDeltas();
			int childCount = childDeltas.length;
			TreePath[] rootPath = fViewer.getElementPaths(fViewer.getInput()); // There should always be only one element in TreePath since we are searching for root element
			if (childCount > 0 && rootPath.length > 0) {
				Object itemToSelect = fViewer.getChildElement(rootPath[0], childCount - 1);
				if(itemToSelect != null) {
					fViewer.setSelection(new StructuredSelection(itemToSelect), true);
					int selection = (childCount - 1 )* (fscrollV.getMaximum() - fscrollV.getMinimum());
					selection = selection/childCount;
					if (selection + fscrollV.getThumb() >= fscrollV.getMaximum())
						selection = fscrollV.getMaximum() - fscrollV.getThumb() - 2;
					else if (selection <= fscrollV.getMinimum())
						selection = fscrollV.getMinimum() + 2;
					fscrollV.setSelection(selection);
				}
			}
			if(fscrollH != null) {
				fscrollH.setVisible(true);
			}
		}

		public StructuredViewer getViewer() {
			return fViewer;
		}

	}


	/**
	 * Context provider
	 */
	private TreeViewerContextProvider fTreeViewerDebugContextProvider = null;

	private ISelectionChangedListener fTreeViewerSelectionChangedListener = new ISelectionChangedListener() {
		@Override
		public void selectionChanged(SelectionChangedEvent event) {
			fTreeViewerDebugContextProvider.activate(event.getSelection());
		}
	};

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.debug.ui.IDebugView#getPresentation(java.lang.String)
	 */
	@Override
	public IDebugModelPresentation getPresentation(String id) {
		return ((DelegatingModelPresentation) fPresentation)
				.getPresentation(id);
	}



	/**
	 * Returns the active debug context for this view based on the view's site
	 * IDs.
	 *
	 * @return Active debug context for this view.
	 *
	 * @since 3.7
	 */
	protected ISelection getDebugContext() {
		IViewSite site = (IViewSite) getSite();
		IDebugContextService contextService = 
				DebugUITools.getDebugContextManager().getContextService(site.getWorkbenchWindow());
		return contextService.getActiveContext();

	}

	@Override
	public void createPartControl(Composite parent) {
		super.createPartControl(parent);

		// Set the tree viewer as the selection provider to the default page.
		// The page book view handles switching the between page selection
		// providers as needed.
		((IPageBookViewPage) getDefaultPage()).getSite().setSelectionProvider(
				getViewer());
		DebugUITools.getDebugContextManager()
		.getContextService(getSite().getWorkbenchWindow())
		.addDebugContextProvider(fTreeViewerDebugContextProvider);
	}



	@Override
	protected Viewer createViewer(Composite parent) {
		fPresentation = new DelegatingModelPresentation();
		fPresentationContext = new DebugModelPresentationContext(ID, this,
				fPresentation);

		TreeModelViewer viewer = new TreeModelViewer(parent, SWT.MULTI
				| SWT.H_SCROLL | SWT.V_SCROLL | SWT.VIRTUAL | SWT.BORDER,
				fPresentationContext);
		addScrollbar(parent, viewer);
		viewer.addSelectionChangedListener(fTreeViewerSelectionChangedListener);
		fTreeViewerDebugContextProvider = new TreeViewerContextProvider(viewer);
		DebugUITools.getDebugContextManager().addDebugContextListener(this);
		fInputService = new ViewerInputService(viewer, fRequester);
		IContextService ctx = (IContextService) getSite().getService(IContextService.class);
		fActivatedContext = ctx.activateContext(Context_ID);
		contextActivated(getDebugContext());
		return viewer;
	}


	@Override
	public void dispose() {
		fPresentation.dispose();
		fPresentationContext.dispose();
		fPresentationContext = null;
		fTreeViewerDebugContextProvider.getViewer().removeSelectionChangedListener(fTreeViewerSelectionChangedListener);
		fTreeViewerSelectionChangedListener = null;
		DebugUITools.getDebugContextManager().getContextService(getSite().getWorkbenchWindow()).removeDebugContextProvider(fTreeViewerDebugContextProvider);
		fTreeViewerDebugContextProvider.dispose();
		fTreeViewerDebugContextProvider = null;
		DebugUITools.getDebugContextManager().removeDebugContextListener(this);
		IContextService ctx = (IContextService) getSite().getService(IContextService.class);

		if (fActivatedContext != null)
			ctx.deactivateContext(fActivatedContext);

		fActivatedContext = null;
		fInputService.dispose();
		super.dispose();
	}

	private void addScrollbar(Composite parent, TreeModelViewer viewer) {
		fTree = (Tree) viewer.getControl();
		fScrollSelectionListener = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				IEvaluationService evaluationService = 
						(IEvaluationService) getViewSite().getService(IEvaluationService.class);
				ScrollBar sb = (ScrollBar) e.getSource();
				
				final String FAST_NEXT_FCH_RECORD_ID = "org.eclipse.cdt.dsf.iss.ui.fch.command.FastNextFCHRecord"; //$NON-NLS-1$
				final String FAST_PREV_FCH_RECORD_ID = "org.eclipse.cdt.dsf.iss.ui.fch.command.FastPrevFCHRecord"; //$NON-NLS-1$
				
				if (sb.getSelection() + sb.getThumb() >= sb.getMaximum()) {
					try {
						ICommandService commandService = 
								(ICommandService) PlatformUI.getWorkbench().getService(ICommandService.class);
						Command next = commandService.getCommand(FAST_NEXT_FCH_RECORD_ID);
						next.executeWithChecks(new ExecutionEvent(next,	new HashMap<Object, Object>(), 
								null, evaluationService.getCurrentState()));
					} catch (Exception exc) {

					}
				} else if (sb.getSelection() <= sb.getMinimum()) {
					try {
						ICommandService commandService = 
								(ICommandService) PlatformUI.getWorkbench().getService(ICommandService.class);
						Command prev = commandService.getCommand(FAST_PREV_FCH_RECORD_ID);
						prev.executeWithChecks(new ExecutionEvent(prev,	new HashMap<Object, Object>(),
								null, evaluationService.getCurrentState()));
					} catch (Exception exc) {

					}
				}
			}
		};

		ScrollBar scrollV = fTree.getVerticalBar();
		if (scrollV != null) {
			scrollV.addSelectionListener(fScrollSelectionListener);
			fscrollV = scrollV;
		}

		ScrollBar scrollH = fTree.getHorizontalBar();

		if(scrollH != null) {
			scrollH.setEnabled(true);
			fscrollH = scrollH;
		}
	}

	@Override
	protected void createActions() {
	}

	@Override
	protected String getHelpContextId() {
		return null;
	}

	@Override
	protected void fillContextMenu(IMenuManager menu) {
	}

	@Override
	protected void configureToolBar(IToolBarManager tbm) {
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.debug.ui.AbstractDebugView#becomesVisible()
	 */
	@Override
	protected void becomesVisible() {
		super.becomesVisible();
		ISelection selection = getDebugContext();
		contextActivated(selection);
		fTreeViewerDebugContextProvider.activate(selection);
	}

	@Override
	protected void becomesHidden() {
		fInputService.resolveViewerInput(ViewerInputService.NULL_INPUT);
		super.becomesHidden();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see
	 * org.eclipse.debug.internal.ui.contexts.provisional.IDebugContextListener
	 * #contextEvent
	 * (org.eclipse.debug.internal.ui.contexts.provisional.DebugContextEvent)
	 */
	@Override
	public void debugContextChanged(DebugContextEvent event) {
		if ((event.getFlags() & DebugContextEvent.ACTIVATED) > 0) {
			if(!event.getSource().equals(this.fTreeViewerDebugContextProvider))
				contextActivated(event.getContext());
		}
	}

	@Override
	public void setFocus() {
		fTreeViewerDebugContextProvider.getViewer().getControl().setFocus();
	}

	/**
	 * Updates actions and sets the viewer input when a context is activated.
	 *
	 * @param selection
	 *            New selection to activate.
	 */
	protected void contextActivated(ISelection selection) {
		int threadId;
		
		if (!isAvailable() || !isVisible())
			return;

		if (selection instanceof IStructuredSelection) {
			Object source = ((IStructuredSelection)selection).getFirstElement();
			fInputService.resolveViewerInput(source);
		}
	}		
}
