/*******************************************************************************
 * Copyright (c) 2015 - 2020 Intel Corporation.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Intel Corporation - Implementation for Function Call History View
 *******************************************************************************/
package org.eclipse.cdt.dsf.iss.ui.viewmodel.launch;

import java.util.List;
import java.util.concurrent.RejectedExecutionException;

import org.eclipse.cdt.dsf.concurrent.ConfinedToDsfExecutor;
import org.eclipse.cdt.dsf.concurrent.DataRequestMonitor;
import org.eclipse.cdt.dsf.concurrent.DsfRunnable;
import org.eclipse.cdt.dsf.concurrent.RequestMonitor;
import org.eclipse.cdt.dsf.datamodel.DataModelInitializedEvent;
import org.eclipse.cdt.dsf.debug.service.IRunControl.IExecutionDMContext;
import org.eclipse.cdt.dsf.debug.service.IRunControl.ISuspendedDMEvent;
import org.eclipse.cdt.dsf.debug.ui.viewmodel.launch.AbstractExecutionContextVMNode;
import org.eclipse.cdt.dsf.debug.ui.viewmodel.launch.ILaunchVMConstants;
import org.eclipse.cdt.dsf.iss.service.IFunctionRecordDMContext;
import org.eclipse.cdt.dsf.iss.service.IIssReverseRunControl;
import org.eclipse.cdt.dsf.iss.service.IIssReverseRunControl.IFunctionRecordDMData;
import org.eclipse.cdt.dsf.iss.ui.fch.FCHUserEvent;
import org.eclipse.cdt.dsf.mi.service.IMIExecutionDMContext;
import org.eclipse.cdt.dsf.mi.service.command.events.MIRunningEvent;
import org.eclipse.cdt.dsf.service.DsfSession;
import org.eclipse.cdt.dsf.ui.concurrent.ViewerDataRequestMonitor;
import org.eclipse.cdt.dsf.ui.viewmodel.IVMContext;
import org.eclipse.cdt.dsf.ui.viewmodel.ModelProxyInstalledEvent;
import org.eclipse.cdt.dsf.ui.viewmodel.VMChildrenUpdate;
import org.eclipse.cdt.dsf.ui.viewmodel.VMDelta;
import org.eclipse.cdt.dsf.ui.viewmodel.datamodel.AbstractDMVMProvider;
import org.eclipse.cdt.dsf.ui.viewmodel.datamodel.IDMVMContext;
import org.eclipse.cdt.dsf.ui.viewmodel.properties.IElementPropertiesProvider;
import org.eclipse.cdt.dsf.ui.viewmodel.properties.IPropertiesUpdate;
import org.eclipse.cdt.dsf.ui.viewmodel.properties.LabelAttribute;
import org.eclipse.cdt.dsf.ui.viewmodel.properties.LabelColumnInfo;
import org.eclipse.cdt.dsf.ui.viewmodel.properties.LabelText;
import org.eclipse.cdt.dsf.ui.viewmodel.properties.PropertiesBasedLabelProvider;
import org.eclipse.core.runtime.Path;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IChildrenCountUpdate;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IChildrenUpdate;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IElementLabelProvider;
import org.eclipse.debug.internal.ui.viewers.model.provisional.ILabelUpdate;
import org.eclipse.debug.internal.ui.viewers.model.provisional.IModelDelta;

@SuppressWarnings("restriction")
public class FunctionRecordVMNode extends AbstractExecutionContextVMNode
		implements IElementLabelProvider, IElementPropertiesProvider {

	/**
	 * The label provider delegate. This VM node will delegate label updates to this
	 * provider which can be created by sub-classes.
	 *
	 * @since 2.0
	 */
	private int childrenCount = -1;
	private PropertiesBasedLabelProvider fLabelProvider;
	private String[] indentationArray = null;
	private final int window = 100;

	private void buildIndentationArray() {
		for (int i = 0; i < window; i++) {
			indentationArray[i] = ""; //$NON-NLS-1$
			for (int j = 0; j < i; j++)
				indentationArray[i] += "    "; //$NON-NLS-1$
		}
	}

	public FunctionRecordVMNode(AbstractDMVMProvider provider, DsfSession session) {
		super(provider, session, IExecutionDMContext.class);
		fLabelProvider = createLabelProvider();
		indentationArray = new String[window];
		buildIndentationArray();
	}

	private PropertiesBasedLabelProvider createLabelProvider() {
		PropertiesBasedLabelProvider provider = new PropertiesBasedLabelProvider();

		provider.setColumnInfo(PropertiesBasedLabelProvider.ID_COLUMN_NO_COLUMNS,
				new LabelColumnInfo(new LabelAttribute[] { new LabelText(MessagesForFCHVM.FCHVMNode_text_format,
						new String[] {
								// ILaunchVMConstants.PROP_ID,
								ILaunchVMConstants.PROP_FRAME_FUNCTION }) }));

		return provider;
	}

	@Override
	public int getDeltaFlags(Object event) {
		if (event instanceof ModelProxyInstalledEvent || event instanceof DataModelInitializedEvent) {
			return (IModelDelta.NO_CHANGE);
		} else if (event instanceof ISuspendedDMEvent) {
			return (IModelDelta.EXPAND);
		} else if (event instanceof DebugEvent) {
			DebugEvent dbgEv = (DebugEvent) event;
			if ((dbgEv.getKind() & FCHUserEvent.COLLAPSE) != 0)
				return IModelDelta.COLLAPSE;
			else if ((dbgEv.getKind() & (FCHUserEvent.SHOWNEXT | FCHUserEvent.SHOWPREV | FCHUserEvent.SHOWFASTNEXT
					| FCHUserEvent.SHOWFASTPREV)) != 0)
				return IModelDelta.CONTENT;
		} else if (event instanceof MIRunningEvent) {
			return (IModelDelta.COLLAPSE);
		}
		return IModelDelta.NO_CHANGE;
	}

	@Override
	public void update(final ILabelUpdate[] updates) {
		fLabelProvider.update(updates);
	}

	@SuppressWarnings("unused")
	private static class FCHVMContextInfo {
		final IVMContext fVMContext;
		final int fIndex;

		FCHVMContextInfo(IVMContext vmContext, int index) {
			fVMContext = vmContext;
			fIndex = index;
		}
	}

	/**
	 * @see IElementPropertiesProvider#update(IPropertiesUpdate[])
	 *
	 * @since 2.0
	 */
	@Override
	public void update(final IPropertiesUpdate[] updates) {
		try {
			getSession().getExecutor().execute(new DsfRunnable() {
				@Override
				public void run() {
					updatePropertiesInSessionThread(updates);
				}
			});
		} catch (RejectedExecutionException e) {
			for (IPropertiesUpdate update : updates) {
				handleFailedUpdate(update);
			}
		}
	}

	@Override
	protected void updateElementCountInSessionThread(final IChildrenCountUpdate update) {
		IIssReverseRunControl runControl = getServicesTracker().getService(IIssReverseRunControl.class);
		final IMIExecutionDMContext execDmc = findDmcInPath(update.getViewerInput(), update.getElementPath(),
				IMIExecutionDMContext.class);
		if (runControl == null) {
			handleFailedUpdate(update);
			return;
		}

		runControl.getFunctionCallCountVM(execDmc,
				new ViewerDataRequestMonitor<Integer>(getSession().getExecutor(), update) {
					@Override
					public void handleCompleted() {
						if (isSuccess()) {
							update.setChildCount(getData());
						} else {
							update.setChildCount(-1);
						}
						update.done();
						return;
					}

					@Override
					public void handleFailure() {
						update.setChildCount(-1);
						update.done();
					}
				});
	}

	protected void updatePropertiesInSessionThread(IPropertiesUpdate[] updates) {
		for (int i = 0; i < updates.length; i++) {
			final IPropertiesUpdate update = updates[i];

			final IIssReverseRunControl runControl = getServicesTracker().getService(IIssReverseRunControl.class);
			if (runControl == null) {
				handleFailedUpdate(update);
				return;
			}
			IFunctionRecordDMContext fchContext = findDmcInPath(update.getViewerInput(), update.getElementPath(),
					IFunctionRecordDMContext.class);

			int contextId = fchContext.getId();

			runControl.getFunctionCallRecordList(
					new ViewerDataRequestMonitor<IFunctionRecordDMData>(getExecutor(), update) {
						@Override
						public void handleCompleted() {
							if (!isSuccess()) {
								handleFailedUpdate(update);
								return;
							}
							fillFCHRecordProperties(update, getData(), runControl.getFunctionCallRecordMinStackDepth());
							update.done();
						}
					}, contextId);
		}

	}

	@Override
	public void buildDelta(Object event, final VMDelta parent, final int nodeOffset,
			final RequestMonitor requestMonitor) {

		IIssReverseRunControl runControl = getServicesTracker().getService(IIssReverseRunControl.class);

		final int shift = 1;
		final int fastshift = 10;
		if (runControl == null)
			return;

		if (event instanceof ModelProxyInstalledEvent || event instanceof DataModelInitializedEvent) {
			requestMonitor.done();
		} else if (event instanceof ISuspendedDMEvent) {
			buildDeltaForEvent(parent, nodeOffset, requestMonitor);
		} else if (event instanceof DebugEvent && (((DebugEvent) event).getKind() & FCHUserEvent.SHOWNEXT) != 0) {
			runControl.shiftWindowForward(shift);
			buildDeltaForCommand(parent, nodeOffset, requestMonitor, shift);
		} else if (event instanceof DebugEvent && (((DebugEvent) event).getKind() & FCHUserEvent.SHOWPREV) != 0) {
			runControl.shiftWindowBackward(shift);
			buildDeltaForCommand(parent, nodeOffset, requestMonitor, (-shift));
		} else if (event instanceof DebugEvent && (((DebugEvent) event).getKind() & FCHUserEvent.SHOWFASTNEXT) != 0) {
			runControl.shiftWindowForward(fastshift);
			buildDeltaForCommand(parent, nodeOffset, requestMonitor, fastshift);
		} else if (event instanceof DebugEvent && (((DebugEvent) event).getKind() & FCHUserEvent.SHOWFASTPREV) != 0) {
			runControl.shiftWindowBackward(fastshift);
			buildDeltaForCommand(parent, nodeOffset, requestMonitor, (-fastshift));
		} else if (event instanceof MIRunningEvent
				|| event instanceof DebugEvent && (((DebugEvent) event).getKind() & FCHUserEvent.COLLAPSE) != 0) {
			parent.setFlags(parent.getFlags() | IModelDelta.COLLAPSE);
			requestMonitor.done();
		}
	}

	/**
	 * Builds the delta in response the debug view being opened.
	 * <p>
	 * The default behavior is to retrieve the list of stack frames, and mark the
	 * top frame to be selected.
	 *
	 * @since 2.1
	 */
	protected void buildDeltaForEvent(final VMDelta parentDelta, final int nodeOffset, final RequestMonitor rm) {
		getVMProvider().updateNode(this, new VMChildrenUpdate(parentDelta, getVMProvider().getPresentationContext(), -1,
				-1, new DataRequestMonitor<List<Object>>(getExecutor(), rm) {
					@Override
					public void handleCompleted() {
						if (isSuccess() && getData().size() != 0) {
							List<?> retval = getData();
							for (int i = 0; i < retval.size(); i++) {
								IFunctionRecordDMContext ret = (IFunctionRecordDMContext) ((IDMVMContext) retval.get(i))
										.getDMContext();
								parentDelta.addNode(retval.get(i), i, (IModelDelta.EXPAND));
							}
						}
						rm.done();
					}

					@Override
					public void handleFailure() {
						rm.done();
					}
				}));
	}

	/**
	 * Builds the delta in response the debug view being opened.
	 * <p>
	 * The default behavior is to retrieve the list of stack frames, and mark the
	 * top frame to be selected.
	 *
	 * @since 2.1
	 */
	protected void buildDeltaForCommand(final VMDelta parentDelta, final int nodeOffset, final RequestMonitor rm,
			final int shift) {
		getVMProvider().updateNode(this, new VMChildrenUpdate(parentDelta, getVMProvider().getPresentationContext(), -1,
				-1, new DataRequestMonitor<List<Object>>(getExecutor(), rm) {
					@Override
					public void handleCompleted() {
						if (isSuccess() && getData().size() != 0) {
							List<?> retval = getData();
							if (retval.isEmpty()) {
								rm.done();
								return;
							}
							for (int i = 0; i < retval.size(); i++) {
								parentDelta.addNode(retval.get(i), i, (IModelDelta.EXPAND));
							}
						}
						rm.done();
					}

					@Override
					public void handleFailure() {
						rm.done();
					}
				}));
	}

	@Override
	public void getContextsForEvent(VMDelta parentDelta, Object e, final DataRequestMonitor<IVMContext[]> rm) {
		rm.done();
		return;
	}

	@Override
	@ConfinedToDsfExecutor("getSession().getExecutor()")
	protected void updateElementsInSessionThread(final IChildrenUpdate update) {

		IIssReverseRunControl runControl = getServicesTracker().getService(IIssReverseRunControl.class);
		final IMIExecutionDMContext execDmc = findDmcInPath(update.getViewerInput(), update.getElementPath(),
				IMIExecutionDMContext.class);

		if (runControl == null) {
			handleFailedUpdate(update);
			return;
		}

		runControl.getPartialFunctionCallRecordVM(execDmc,
				new ViewerDataRequestMonitor<IFunctionRecordDMContext[]>(getSession().getExecutor(), update) {
					@Override
					public void handleCompleted() {
						if (isSuccess() || (getData() instanceof IFunctionRecordDMContext[])) {
							childrenCount = getData().length;

							fillUpdateWithVMCs(update, getData());
						}
						update.done();
						return;
					}

					@Override
					public void handleFailure() {
						update.done();
					}
				});
	}

	protected void fillFCHRecordProperties(IPropertiesUpdate update, IFunctionRecordDMData data, int minStackDepth) {
		if (data == null)
			return;

		int minTab = 0;
		if (data.getLevel() != null && data.getLevel().length() > 0) {
			minTab = Integer.parseInt(data.getLevel());
			minTab = minTab - minStackDepth;
		}

		if (data.getName() != null && data.getName().length() > 0) {
			StringBuilder tabTemp = new StringBuilder();
			if (minTab < indentationArray.length) {
				tabTemp.append(indentationArray[minTab]);
			} else {
				for (int i = 0; i < minTab; i++) {
					tabTemp.append("    "); //$NON-NLS-1$
				}
			}
			String temp = data.getName();
			String label = temp.replaceFirst("[(](.*)", ""); //$NON-NLS-1$ //$NON-NLS-2$
			label += "()"; //$NON-NLS-1$

			boolean hasfileinfo = (data.getFile() != null) && (data.getFile().length() > 0);
			if (hasfileinfo) {
				String fileName = new Path(data.getFile()).lastSegment();
				label += " at " + fileName + ":" + String.valueOf(data.getBeginLine()); //$NON-NLS-1$ //$NON-NLS-2$
			}

			update.setProperty(ILaunchVMConstants.PROP_FRAME_FUNCTION, (tabTemp.toString() + label));
		}
	}
}
