/*******************************************************************************
 * Copyright (c) 2015 - 2020 Intel Corporation.
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *     Intel Corporation - Implementation for Function Call History View
 *******************************************************************************/
package org.eclipse.cdt.dsf.iss.ui.fch.command;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IAdapterManager;
import org.eclipse.core.runtime.Platform;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.actions.DebugCommandHandler;
import org.eclipse.debug.ui.contexts.DebugContextEvent;
import org.eclipse.debug.ui.contexts.IDebugContextListener;
import org.eclipse.debug.ui.contexts.IDebugContextService;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

public abstract class FCHRecordHandler extends DebugCommandHandler implements
        IDebugContextListener {
    private Object fActiveContext = null;
    protected Object fTargetAdapter = null;
    private IDebugContextService fContextService = null;
    private Class<?> IdbgCommandHandlerType = null;

    public FCHRecordHandler(String id, Class<?> cmdHandlertype) {
        // add debug context change listener, so we refresh UI on debug context
        // changes.

        IdbgCommandHandlerType = cmdHandlertype;
        IWorkbenchWindow window = PlatformUI.getWorkbench()
                .getActiveWorkbenchWindow();
        if (window != null) {
            fContextService = DebugUITools.getDebugContextManager()
                    .getContextService(window);
            if (fContextService != null) {
                fContextService.addPostDebugContextListener(this);

                // This constructor might be called after the launch, so we must
                // refresh here too.
                // This can happen if we activate the action set after the
                // launch.
                refresh(fContextService.getActiveContext());
            }
        }
    }

    @Override
    public void dispose() {
        // Must use the stored context service. If we try to fetch the service
        // again with the workbenchWindow, it may fail if the window is
        // already closed.
        if (fContextService != null) {
            fContextService.removePostDebugContextListener(this);
        }
        fContextService = null;
        fTargetAdapter = null;
        fActiveContext = null;
        IdbgCommandHandlerType = null;
        super.dispose();
    }

    @Override
    public void debugContextChanged(DebugContextEvent event) {
        if ((event.getFlags() & DebugContextEvent.ACTIVATED) > 0) {
            refresh(event.getContext());
        }
    }

    private void refresh(ISelection selection) {
        // refresh the UI elements associated with the changeTraceMethod
        // command.
        // certain operations, f.e. toggle of reverse debugging state, have
        // impact on icon, tooltip and activation of the trace method UI
        // elements.
        fTargetAdapter = null;
        fActiveContext = null;
        if (selection instanceof IStructuredSelection) {
            IStructuredSelection ss = (IStructuredSelection) selection;
            if (!ss.isEmpty()) {
                fActiveContext = ss.getFirstElement();
                if (fActiveContext instanceof IAdaptable) {
                    fTargetAdapter = getAdapter((IAdaptable) fActiveContext);
                }
            }
        }
    }

    private Object getAdapter(IAdaptable adaptable) {
        Object adapter = adaptable.getAdapter(IdbgCommandHandlerType);
        if (adapter == null) {
            IAdapterManager adapterManager = Platform.getAdapterManager();
            if (adapterManager
                    .hasAdapter(adaptable, getCommandType().getName())) {
                adapter = adapterManager.loadAdapter(adaptable,
                        IdbgCommandHandlerType.getName());
            }
        }
        return adapter;
    }

    @Override
    protected boolean getInitialEnablement() {
        return true;
    }

    @Override
    protected Class<?> getCommandType() {
        return IdbgCommandHandlerType;
    }
}
