/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.core.snapshots;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.hadoop.fs.Path;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CloudSolrClient;
import org.apache.solr.client.solrj.request.CollectionAdminRequest;
import org.apache.solr.client.solrj.response.CollectionAdminResponse;
import org.apache.solr.common.cloud.DocCollection;
import org.apache.solr.common.cloud.Replica;
import org.apache.solr.common.cloud.Slice;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.core.snapshots.CollectionSnapshotMetaData;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SolrSnapshotsTool
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private static final DateFormat dateFormat = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss z", Locale.getDefault());
    private static final String CREATE = "create";
    private static final String DELETE = "delete";
    private static final String LIST = "list";
    private static final String DESCRIBE = "describe";
    private static final String PREPARE_FOR_EXPORT = "prepare-snapshot-export";
    private static final String EXPORT_SNAPSHOT = "export";
    private static final String HELP = "help";
    private static final String COLLECTION = "c";
    private static final String TEMP_DIR = "t";
    private static final String DEST_DIR = "d";
    private static final String SOLR_ZK_ENSEMBLE = "z";
    private static final String HDFS_PATH_PREFIX = "p";
    private static final String BACKUP_REPO_NAME = "r";
    private static final String ASYNC_REQ_ID = "i";
    private static final List<String> OPTION_HELP_ORDER = Arrays.asList("create", "delete", "list", "describe", "prepare-snapshot-export", "export", "help", "z", "c", "d", "r", "i", "t", "p");
    private final CloudSolrClient solrClient;

    public SolrSnapshotsTool(String solrZkEnsemble) {
        this.solrClient = new CloudSolrClient.Builder(Collections.singletonList(solrZkEnsemble), Optional.empty()).build();
    }

    @Override
    public void close() throws IOException {
        if (this.solrClient != null) {
            this.solrClient.close();
        }
    }

    public void createSnapshot(String collectionName, String snapshotName) {
        CollectionAdminRequest.CreateSnapshot createSnap = new CollectionAdminRequest.CreateSnapshot(collectionName, snapshotName);
        try {
            CollectionAdminResponse resp = (CollectionAdminResponse)createSnap.process((SolrClient)this.solrClient);
            Preconditions.checkState((resp.getStatus() == 0 ? 1 : 0) != 0, (Object)("The CREATESNAPSHOT request failed. The status code is " + resp.getStatus()));
            System.out.println("Successfully created snapshot with name " + snapshotName + " for collection " + collectionName);
        }
        catch (Exception e) {
            log.error("Failed to create a snapshot with name {} for collection {}", new Object[]{snapshotName, collectionName, e});
            System.out.println("Failed to create a snapshot with name " + snapshotName + " for collection " + collectionName + " due to following error : " + e.getLocalizedMessage());
        }
    }

    public void deleteSnapshot(String collectionName, String snapshotName) {
        CollectionAdminRequest.DeleteSnapshot deleteSnap = new CollectionAdminRequest.DeleteSnapshot(collectionName, snapshotName);
        try {
            CollectionAdminResponse resp = (CollectionAdminResponse)deleteSnap.process((SolrClient)this.solrClient);
            Preconditions.checkState((resp.getStatus() == 0 ? 1 : 0) != 0, (Object)("The DELETESNAPSHOT request failed. The status code is " + resp.getStatus()));
            System.out.println("Successfully deleted snapshot with name " + snapshotName + " for collection " + collectionName);
        }
        catch (Exception e) {
            log.error("Failed to delete a snapshot with name {} for collection {}", new Object[]{snapshotName, collectionName, e});
            System.out.println("Failed to delete a snapshot with name " + snapshotName + " for collection " + collectionName + " due to following error : " + e.getLocalizedMessage());
        }
    }

    public void listSnapshots(String collectionName) {
        CollectionAdminRequest.ListSnapshots listSnaps = new CollectionAdminRequest.ListSnapshots(collectionName);
        try {
            CollectionAdminResponse resp = (CollectionAdminResponse)listSnaps.process((SolrClient)this.solrClient);
            Preconditions.checkState((resp.getStatus() == 0 ? 1 : 0) != 0, (Object)("The LISTSNAPSHOTS request failed. The status code is " + resp.getStatus()));
            NamedList apiResult = (NamedList)resp.getResponse().get("snapshots");
            for (int i = 0; i < apiResult.size(); ++i) {
                System.out.println(apiResult.getName(i));
            }
        }
        catch (Exception e) {
            log.error("Failed to list snapshots for collection {}", (Object)collectionName, (Object)e);
            System.out.println("Failed to list snapshots for collection " + collectionName + " due to following error : " + e.getLocalizedMessage());
        }
    }

    public void describeSnapshot(String collectionName, String snapshotName) {
        try {
            Collection<CollectionSnapshotMetaData> snaps = this.listCollectionSnapshots(collectionName);
            for (CollectionSnapshotMetaData m : snaps) {
                if (!snapshotName.equals(m.getName())) continue;
                System.out.println("Name: " + m.getName());
                System.out.println("Status: " + (Object)((Object)m.getStatus()));
                System.out.println("Time of creation: " + dateFormat.format(m.getCreationDate()));
                System.out.println("Total number of cores with snapshot: " + m.getReplicaSnapshots().size());
                System.out.println("-----------------------------------");
                for (CollectionSnapshotMetaData.CoreSnapshotMetaData n : m.getReplicaSnapshots()) {
                    StringBuilder builder = new StringBuilder();
                    builder.append("Core [name=");
                    builder.append(n.getCoreName());
                    builder.append(", leader=");
                    builder.append(n.isLeader());
                    builder.append(", generation=");
                    builder.append(n.getGenerationNumber());
                    builder.append(", indexDirPath=");
                    builder.append(n.getIndexDirPath());
                    builder.append("]\n");
                    System.out.println(builder.toString());
                }
            }
        }
        catch (Exception e) {
            log.error("Failed to fetch snapshot details", (Throwable)e);
            System.out.println("Failed to fetch snapshot details due to following error : " + e.getLocalizedMessage());
        }
    }

    /*
     * WARNING - void declaration
     */
    public Map<String, List<String>> getIndexFilesPathForSnapshot(String collectionName, String snapshotName, Optional<String> pathPrefix) throws SolrServerException, IOException {
        HashMap<String, List<String>> result = new HashMap<String, List<String>>();
        Collection<CollectionSnapshotMetaData> snaps = this.listCollectionSnapshots(collectionName);
        Optional<Object> meta = Optional.empty();
        for (CollectionSnapshotMetaData m : snaps) {
            if (!snapshotName.equals(m.getName())) continue;
            meta = Optional.of(m);
        }
        if (!meta.isPresent()) {
            throw new IllegalArgumentException("The snapshot named " + snapshotName + " is not found for collection " + collectionName);
        }
        DocCollection collectionState = this.solrClient.getZkStateReader().getClusterState().getCollection(collectionName);
        for (Slice s : collectionState.getSlices()) {
            List<CollectionSnapshotMetaData.CoreSnapshotMetaData> replicaSnaps = ((CollectionSnapshotMetaData)meta.get()).getReplicaSnapshotsForShard(s.getName());
            ArrayList<CollectionSnapshotMetaData.CoreSnapshotMetaData> availableReplicas = new ArrayList<CollectionSnapshotMetaData.CoreSnapshotMetaData>();
            for (CollectionSnapshotMetaData.CoreSnapshotMetaData coreSnapshotMetaData : replicaSnaps) {
                if (!SolrSnapshotsTool.isReplicaAvailable(s, coreSnapshotMetaData.getCoreName())) continue;
                availableReplicas.add(coreSnapshotMetaData);
            }
            if (availableReplicas.isEmpty()) {
                throw new IllegalArgumentException("The snapshot named " + snapshotName + " not found for shard " + s.getName() + " of collection " + collectionName);
            }
            CollectionSnapshotMetaData.CoreSnapshotMetaData coreSnap = (CollectionSnapshotMetaData.CoreSnapshotMetaData)availableReplicas.get(0);
            for (CollectionSnapshotMetaData.CoreSnapshotMetaData m3 : availableReplicas) {
                if (!m3.isLeader()) continue;
                coreSnap = m3;
            }
            String string = coreSnap.getIndexDirPath();
            if (pathPrefix.isPresent()) {
                Path t = new Path(coreSnap.getIndexDirPath());
                String string2 = new Path(pathPrefix.get(), t.toUri().getPath()).toString();
            }
            ArrayList<String> paths = new ArrayList<String>();
            for (String fileName : coreSnap.getFiles()) {
                void var13_13;
                Path p = new Path((String)var13_13, fileName);
                paths.add(p.toString());
            }
            result.put(s.getName(), paths);
        }
        return result;
    }

    public void buildCopyListings(String collectionName, String snapshotName, String localFsPath, Optional<String> pathPrefix) throws SolrServerException, IOException {
        Map<String, List<String>> paths = this.getIndexFilesPathForSnapshot(collectionName, snapshotName, pathPrefix);
        for (Map.Entry<String, List<String>> entry : paths.entrySet()) {
            StringBuilder filesBuilder = new StringBuilder();
            for (String filePath : entry.getValue()) {
                filesBuilder.append(filePath);
                filesBuilder.append("\n");
            }
            String files = filesBuilder.toString().trim();
            try (OutputStreamWriter w = new OutputStreamWriter((OutputStream)new FileOutputStream(new File(localFsPath, entry.getKey())), StandardCharsets.UTF_8);){
                w.write(files);
            }
        }
    }

    public void backupCollectionMetaData(String collectionName, String snapshotName, String backupLoc) throws SolrServerException, IOException {
        CollectionAdminRequest.Backup backup = new CollectionAdminRequest.Backup(collectionName, snapshotName);
        backup.setIndexBackupStrategy("none");
        backup.setLocation(backupLoc);
        CollectionAdminResponse resp = (CollectionAdminResponse)backup.process((SolrClient)this.solrClient);
        Preconditions.checkState((resp.getStatus() == 0 ? 1 : 0) != 0, (Object)("The request failed. The status code is " + resp.getStatus()));
    }

    public void prepareForExport(String collectionName, String snapshotName, String localFsPath, Optional<String> pathPrefix, String destPath) {
        try {
            this.buildCopyListings(collectionName, snapshotName, localFsPath, pathPrefix);
            System.out.println("Successfully prepared copylisting for the snapshot export.");
        }
        catch (Exception e) {
            log.error("Failed to prepare a copylisting for snapshot with name {} for collection {}", new Object[]{snapshotName, collectionName, e});
            System.out.println("Failed to prepare a copylisting for snapshot with name " + snapshotName + " for collection " + collectionName + " due to following error : " + e.getLocalizedMessage());
            System.exit(1);
        }
        try {
            this.backupCollectionMetaData(collectionName, snapshotName, destPath);
            System.out.println("Successfully backed up collection meta-data");
        }
        catch (Exception e) {
            log.error("Failed to backup collection meta-data for collection {}", (Object)collectionName, (Object)e);
            System.out.println("Failed to backup collection meta-data for collection " + collectionName + " due to following error : " + e.getLocalizedMessage());
            System.exit(1);
        }
    }

    public void exportSnapshot(String collectionName, String snapshotName, String destPath, Optional<String> backupRepo, Optional<String> asyncReqId) {
        try {
            CollectionAdminRequest.Backup backup = new CollectionAdminRequest.Backup(collectionName, snapshotName);
            backup.setCommitName(snapshotName);
            backup.setIndexBackupStrategy("copy-files");
            backup.setLocation(destPath);
            if (backupRepo.isPresent()) {
                backup.setRepositoryName(backupRepo.get());
            }
            backup.processAsync((String)asyncReqId.orElse(null), (SolrClient)this.solrClient);
        }
        catch (Exception e) {
            log.error("Failed to backup collection meta-data for collection {}", (Object)collectionName, (Object)e);
            System.out.println("Failed to backup collection meta-data for collection " + collectionName + " due to following error : " + e.getLocalizedMessage());
            System.exit(1);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static void main(String[] args) throws IOException {
        PosixParser parser = new PosixParser();
        Options options = new Options();
        options.addOption(null, CREATE, true, "This command will create a snapshot with the specified name");
        options.addOption(null, DELETE, true, "This command will delete a snapshot with the specified name");
        options.addOption(null, LIST, false, "This command will list all the named snapshots for the specified collection.");
        options.addOption(null, DESCRIBE, true, "This command will print details for a named snapshot for the specified collection.");
        options.addOption(null, PREPARE_FOR_EXPORT, true, "This command will prepare copylistings for the specified snapshot. This command should only be used only if Solr is deployed with Hadoop and collection index files are stored on a shared file-system e.g. HDFS");
        options.addOption(null, EXPORT_SNAPSHOT, true, "This command will create a backup for the specified snapshot.");
        options.addOption(null, HELP, false, "This command will print the help message for the snapshots related commands.");
        options.addOption(TEMP_DIR, true, "This parameter specifies the path of a temporary directory on local filesystem during prepare-snapshot-export command.");
        options.addOption(DEST_DIR, true, "This parameter specifies the path on shared file-system (e.g. HDFS) where the snapshot related information should be stored.");
        options.addOption(COLLECTION, true, "This parameter specifies the name of the collection to be used during snapshot operation");
        options.addOption(SOLR_ZK_ENSEMBLE, true, "This parameter specifies the Solr Zookeeper ensemble address");
        options.addOption(HDFS_PATH_PREFIX, true, "This parameter specifies the HDFS URI prefix to be used during snapshot export preparation. This is applicable only if the Solr collection index files are stored on HDFS.");
        options.addOption(BACKUP_REPO_NAME, true, "This parameter specifies the name of the backup repository to be used during snapshot export preparation");
        options.addOption(ASYNC_REQ_ID, true, "This parameter specifies the async request identifier to be used during snapshot export preparation");
        CommandLine cmd = null;
        try {
            cmd = parser.parse(options, args);
        }
        catch (ParseException e) {
            System.out.println(e.getLocalizedMessage());
            SolrSnapshotsTool.printHelp(options);
            System.exit(1);
        }
        if (cmd.hasOption(CREATE) || cmd.hasOption(DELETE) || cmd.hasOption(LIST) || cmd.hasOption(DESCRIBE) || cmd.hasOption(PREPARE_FOR_EXPORT) || cmd.hasOption(EXPORT_SNAPSHOT)) {
            try (SolrSnapshotsTool tool = new SolrSnapshotsTool(SolrSnapshotsTool.requiredArg(options, cmd, SOLR_ZK_ENSEMBLE));){
                if (cmd.hasOption(CREATE)) {
                    String snapshotName = cmd.getOptionValue(CREATE);
                    String collectionName = SolrSnapshotsTool.requiredArg(options, cmd, COLLECTION);
                    tool.createSnapshot(collectionName, snapshotName);
                    return;
                }
                if (cmd.hasOption(DELETE)) {
                    String snapshotName = cmd.getOptionValue(DELETE);
                    String collectionName = SolrSnapshotsTool.requiredArg(options, cmd, COLLECTION);
                    tool.deleteSnapshot(collectionName, snapshotName);
                    return;
                }
                if (cmd.hasOption(LIST)) {
                    String collectionName = SolrSnapshotsTool.requiredArg(options, cmd, COLLECTION);
                    tool.listSnapshots(collectionName);
                    return;
                }
                if (cmd.hasOption(DESCRIBE)) {
                    String snapshotName = cmd.getOptionValue(DESCRIBE);
                    String collectionName = SolrSnapshotsTool.requiredArg(options, cmd, COLLECTION);
                    tool.describeSnapshot(collectionName, snapshotName);
                    return;
                }
                if (cmd.hasOption(PREPARE_FOR_EXPORT)) {
                    String snapshotName = cmd.getOptionValue(PREPARE_FOR_EXPORT);
                    String collectionName = SolrSnapshotsTool.requiredArg(options, cmd, COLLECTION);
                    String localFsDir = SolrSnapshotsTool.requiredArg(options, cmd, TEMP_DIR);
                    String hdfsOpDir = SolrSnapshotsTool.requiredArg(options, cmd, DEST_DIR);
                    Optional<String> pathPrefix = Optional.ofNullable(cmd.getOptionValue(HDFS_PATH_PREFIX));
                    if (pathPrefix.isPresent()) {
                        try {
                            new URI(pathPrefix.get());
                        }
                        catch (URISyntaxException e) {
                            System.out.println("The specified File system path prefix " + pathPrefix.get() + " is invalid. The error is " + e.getLocalizedMessage());
                            System.exit(1);
                        }
                    }
                    tool.prepareForExport(collectionName, snapshotName, localFsDir, pathPrefix, hdfsOpDir);
                    return;
                }
                if (!cmd.hasOption(EXPORT_SNAPSHOT)) return;
                String snapshotName = cmd.getOptionValue(EXPORT_SNAPSHOT);
                String collectionName = SolrSnapshotsTool.requiredArg(options, cmd, COLLECTION);
                String destDir = SolrSnapshotsTool.requiredArg(options, cmd, DEST_DIR);
                Optional<String> backupRepo = Optional.ofNullable(cmd.getOptionValue(BACKUP_REPO_NAME));
                Optional<String> asyncReqId = Optional.ofNullable(cmd.getOptionValue(ASYNC_REQ_ID));
                tool.exportSnapshot(collectionName, snapshotName, destDir, backupRepo, asyncReqId);
                return;
            }
        }
        if (cmd.hasOption(HELP)) {
            SolrSnapshotsTool.printHelp(options);
            return;
        } else {
            System.out.println("Unknown command specified.");
            SolrSnapshotsTool.printHelp(options);
        }
    }

    private static String requiredArg(Options options, CommandLine cmd, String optVal) {
        if (!cmd.hasOption(optVal)) {
            System.out.println("Please specify the value for option " + optVal);
            SolrSnapshotsTool.printHelp(options);
            System.exit(1);
        }
        return cmd.getOptionValue(optVal);
    }

    private static boolean isReplicaAvailable(Slice s, String coreName) {
        for (Replica r : s.getReplicas()) {
            if (!coreName.equals(r.getCoreName())) continue;
            return true;
        }
        return false;
    }

    private Collection<CollectionSnapshotMetaData> listCollectionSnapshots(String collectionName) throws SolrServerException, IOException {
        CollectionAdminRequest.ListSnapshots listSnapshots = new CollectionAdminRequest.ListSnapshots(collectionName);
        CollectionAdminResponse resp = (CollectionAdminResponse)listSnapshots.process((SolrClient)this.solrClient);
        Preconditions.checkState((resp.getStatus() == 0 ? 1 : 0) != 0);
        NamedList apiResult = (NamedList)resp.getResponse().get("snapshots");
        ArrayList<CollectionSnapshotMetaData> result = new ArrayList<CollectionSnapshotMetaData>();
        for (int i = 0; i < apiResult.size(); ++i) {
            result.add(new CollectionSnapshotMetaData((NamedList<Object>)((NamedList)apiResult.getVal(i))));
        }
        return result;
    }

    private static void printHelp(Options options) {
        StringBuilder helpFooter = new StringBuilder();
        helpFooter.append("Examples: \n");
        helpFooter.append("snapshotscli.sh --create snapshot-1 -c books -z localhost:2181 \n");
        helpFooter.append("snapshotscli.sh --list -c books -z localhost:2181 \n");
        helpFooter.append("snapshotscli.sh --describe snapshot-1 -c books -z localhost:2181 \n");
        helpFooter.append("snapshotscli.sh --export snapshot-1 -c books -z localhost:2181 -b repo -l backupPath -i req_0 \n");
        helpFooter.append("snapshotscli.sh --delete snapshot-1 -c books -z localhost:2181 \n");
        HelpFormatter formatter = new HelpFormatter();
        formatter.setOptionComparator(new OptionComarator());
        formatter.printHelp("SolrSnapshotsTool", null, options, helpFooter.toString(), false);
    }

    private static class OptionComarator<T extends Option>
    implements Comparator<T> {
        private OptionComarator() {
        }

        @Override
        public int compare(T o1, T o2) {
            String s1 = o1.hasLongOpt() ? o1.getLongOpt() : o1.getOpt();
            String s2 = o2.hasLongOpt() ? o2.getLongOpt() : o2.getOpt();
            return OPTION_HELP_ORDER.indexOf(s1) - OPTION_HELP_ORDER.indexOf(s2);
        }
    }
}

