"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processTaskError = exports.isRealTimeTask = exports.isIndexNotFoundError = exports.getTaskState = exports.getTaskInitProgress = exports.getResultAggregationQuery = exports.getLatestTaskForDetectorQuery = exports.getLatestDetectorTasksQuery = exports.getFinalDetectorStates = exports.getFiltersFromEntityList = exports.getErrorMessage = exports.getDetectorsWithJob = exports.getDetectorTasks = exports.getDetectorResults = exports.convertTaskAndJobFieldsToCamelCase = exports.convertStaticFieldsToCamelCase = exports.convertPreviewInputKeysToSnakeCase = exports.convertDetectorKeysToSnakeCase = exports.convertDetectorKeysToCamelCase = exports.appendTaskInfo = exports.anomalyResultMapper = void 0;

var _lodash = require("lodash");

var _helpers = require("../../utils/helpers");

var _constants = require("../../utils/constants");

/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
const convertDetectorKeysToSnakeCase = payload => {
  return { ...(0, _helpers.mapKeysDeep)({ ...(0, _lodash.omit)(payload, ['filterQuery', 'featureAttributes']) // Exclude the filterQuery,

    }, _helpers.toSnake),
    filter_query: (0, _lodash.get)(payload, 'filterQuery', {}),
    ui_metadata: (0, _lodash.get)(payload, 'uiMetadata', {}),
    feature_attributes: (0, _lodash.get)(payload, 'featureAttributes', []).map(feature => ({ ...(0, _helpers.mapKeysDeep)({ ...(0, _lodash.omit)(feature, ['aggregationQuery'])
      }, _helpers.toSnake),
      aggregation_query: feature.aggregationQuery
    }))
  };
};

exports.convertDetectorKeysToSnakeCase = convertDetectorKeysToSnakeCase;

const convertPreviewInputKeysToSnakeCase = payload => {
  return { ...(0, _helpers.mapKeysDeep)({ ...(0, _lodash.omit)(payload, ['detector']) // Exclude the detector,

    }, _helpers.toSnake),
    detector: convertDetectorKeysToSnakeCase((0, _lodash.get)(payload, 'detector', {}))
  };
};

exports.convertPreviewInputKeysToSnakeCase = convertPreviewInputKeysToSnakeCase;

const convertDetectorKeysToCamelCase = response => {
  let camelCaseResponse = { ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'ui_metadata', 'feature_query', 'feature_attributes', 'adJob', 'historicalTask']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({ ...(0, _helpers.mapKeysDeep)({ ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    enabled: (0, _lodash.get)(response, 'adJob.enabled', false),
    enabledTime: (0, _lodash.get)(response, 'adJob.enabled_time'),
    disabledTime: (0, _lodash.get)(response, 'adJob.disabled_time'),
    categoryField: (0, _lodash.get)(response, 'category_field')
  };

  if (!(0, _lodash.isEmpty)((0, _lodash.get)(response, 'historicalTask', {}))) {
    camelCaseResponse = { ...camelCaseResponse,
      //@ts-ignore
      taskId: (0, _lodash.get)(response, 'historicalTask.task_id'),
      taskState: getTaskState((0, _lodash.get)(response, 'historicalTask', {})),
      taskProgress: (0, _lodash.get)(response, 'historicalTask.task_progress'),
      taskError: processTaskError((0, _lodash.get)(response, 'historicalTask.error', '')),
      detectionDateRange: {
        startTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.start_time'),
        endTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.end_time')
      }
    };
  }

  return camelCaseResponse;
}; // Converts the static detector fields into camelcase. Ignores any job or task-related fields


exports.convertDetectorKeysToCamelCase = convertDetectorKeysToCamelCase;

const convertStaticFieldsToCamelCase = response => {
  return { ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'feature_query', 'feature_attributes', 'ui_metadata', 'anomaly_detector_job', 'anomaly_detection_task', 'realtime_detection_task', 'historical_analysis_task']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({ ...(0, _helpers.mapKeysDeep)({ ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {})
  };
}; // Converts the task-related detector fields into camelcase


exports.convertStaticFieldsToCamelCase = convertStaticFieldsToCamelCase;

const convertTaskAndJobFieldsToCamelCase = (realtimeTask, historicalTask, detectorJob) => {
  let response = {}; // Populate AD job fields

  response = { ...response,
    enabled: (0, _lodash.get)(detectorJob, 'enabled', false),
    enabledTime: (0, _lodash.get)(detectorJob, 'enabled_time'),
    disabledTime: (0, _lodash.get)(detectorJob, 'disabled_time')
  }; // Populate RT-task-related fields

  response = realtimeTask !== undefined ? { ...response,
    curState: getTaskState(realtimeTask),
    stateError: processTaskError((0, _lodash.get)(realtimeTask, 'error', '')),
    initProgress: getTaskInitProgress(realtimeTask)
  } : { ...response,
    curState: (0, _lodash.get)(detectorJob, 'enabled', false) ? _constants.DETECTOR_STATE.RUNNING : _constants.DETECTOR_STATE.DISABLED
  }; // Detection date range field is stored under the 'detector' field in legacy historical tasks.
  // To handle this, need to add a check to fetch the date range from the correct place

  const isLegacyHistorical = (0, _lodash.get)(historicalTask, 'detection_date_range') === undefined && (0, _lodash.get)(historicalTask, 'detector.detection_date_range') !== undefined;
  const legacyDateRangePrefix = isLegacyHistorical ? 'detector.' : ''; // Populate historical-task-related fields

  response = { ...response,
    taskId: (0, _lodash.get)(historicalTask, 'id'),
    taskState: getTaskState(historicalTask),
    taskProgress: (0, _lodash.get)(historicalTask, 'task_progress'),
    taskError: processTaskError((0, _lodash.get)(historicalTask, 'error', '')),
    detectionDateRange: {
      startTime: (0, _lodash.get)(historicalTask, `${legacyDateRangePrefix}detection_date_range.start_time`),
      endTime: (0, _lodash.get)(historicalTask, `${legacyDateRangePrefix}detection_date_range.end_time`)
    }
  };

  if ((0, _lodash.isEmpty)(historicalTask)) {
    //@ts-ignore
    delete response.detectionDateRange;
  }

  return response;
};

exports.convertTaskAndJobFieldsToCamelCase = convertTaskAndJobFieldsToCamelCase;

const getResultAggregationQuery = (detectors, queryParams) => {
  const aggregationSort = {
    totalAnomalies: 'total_anomalies_in_24hr',
    latestAnomalyTime: 'latest_anomaly_time'
  };
  let aggsSortingOrder = {};

  if (aggregationSort[queryParams.sortField]) {
    aggsSortingOrder = {
      order: {
        [aggregationSort[queryParams.sortField]]: queryParams.sortDirection
      }
    };
  }

  return {
    size: 0,
    query: {
      bool: {
        must: [{
          terms: {
            detector_id: detectors
          }
        }, {
          range: {
            anomaly_grade: {
              gt: 0
            }
          }
        }],
        must_not: {
          exists: {
            field: 'task_id'
          }
        }
      }
    },
    aggs: {
      unique_detectors: {
        terms: {
          field: 'detector_id',
          size: queryParams.from + queryParams.size,
          ...aggsSortingOrder
        },
        aggs: {
          total_anomalies_in_24hr: {
            filter: {
              range: {
                data_start_time: {
                  gte: 'now-24h',
                  lte: 'now'
                }
              }
            }
          },
          latest_anomaly_time: {
            max: {
              field: 'data_start_time'
            }
          }
        }
      }
    }
  };
};

exports.getResultAggregationQuery = getResultAggregationQuery;

const anomalyResultMapper = anomalyResults => {
  let resultData = {
    anomalies: [],
    featureData: {}
  };
  if (anomalyResults.length === 0) return resultData; //initialize features list.

  const firstAnomaly = anomalyResults[0];
  Object.values(firstAnomaly.featureData).forEach(feature => {
    resultData.featureData[feature.featureId] = [];
  });
  anomalyResults.forEach(({
    featureData,
    ...rest
  }) => {
    const {
      dataStartTime,
      dataEndTime,
      ...others
    } = rest;
    resultData.anomalies.push({ ...others,
      anomalyGrade: rest.anomalyGrade != null && rest.anomalyGrade > 0 ? Number.parseFloat(rest.anomalyGrade).toFixed(2) : 0,
      confidence: rest.anomalyGrade != null && rest.anomalyGrade > 0 ? Number.parseFloat(rest.confidence).toFixed(2) : 0,
      startTime: rest.dataStartTime,
      endTime: rest.dataEndTime,
      plotTime: rest.dataEndTime,
      ...(rest.entity !== undefined ? {
        entity: rest.entity
      } : {})
    });
    featureData.forEach(feature => {
      resultData.featureData[feature.featureId].push({
        startTime: rest.dataStartTime,
        endTime: rest.dataEndTime,
        plotTime: rest.dataEndTime,
        data: feature.data
      });
    });
  });
  return resultData;
};

exports.anomalyResultMapper = anomalyResultMapper;

const getTaskInitProgress = task => {
  if ((task === null || task === void 0 ? void 0 : task.init_progress) !== undefined) {
    return {
      percentageStr: `${((0, _lodash.get)(task, 'init_progress', 0) * 100).toFixed(0)}%`,
      estimatedMinutesLeft: task.estimated_minutes_left
    };
  }

  return undefined;
};

exports.getTaskInitProgress = getTaskInitProgress;

const getFinalDetectorStates = (detectorStateResponses, finalDetectors) => {
  let finalDetectorStates = (0, _lodash.cloneDeep)(detectorStateResponses);
  finalDetectorStates.forEach(detectorState => {
    //@ts-ignore
    detectorState.state = _constants.DETECTOR_STATE[detectorState.state];
  }); // check if there was any failures / detectors that are unable to start

  finalDetectorStates.forEach((detectorState, i) => {
    /*
      If the error starts with 'Stopped detector', then an EndRunException was thrown.
      All EndRunExceptions are related to initialization failures except for the
      unknown prediction error which contains the message "We might have bugs".
    */
    if (detectorState.state === _constants.DETECTOR_STATE.DISABLED && detectorState.error !== undefined && detectorState.error.includes('Stopped detector')) {
      detectorState.state = detectorState.error.includes('We might have bugs') ? _constants.DETECTOR_STATE.UNEXPECTED_FAILURE : _constants.DETECTOR_STATE.INIT_FAILURE;
    }
    /*
      If a detector is disabled and has no features, set to
      a feature required state
    */


    if (detectorState.state === _constants.DETECTOR_STATE.DISABLED && finalDetectors[i].featureAttributes.length === 0) {
      detectorState.state = _constants.DETECTOR_STATE.FEATURE_REQUIRED;
    }
  });
  return finalDetectorStates;
};

exports.getFinalDetectorStates = getFinalDetectorStates;

const getDetectorsWithJob = detectorsWithJobResponses => {
  const finalDetectorsWithJobResponses = (0, _lodash.cloneDeep)(detectorsWithJobResponses);
  const resultDetectorWithJobs = [];
  finalDetectorsWithJobResponses.forEach(detectorWithJobResponse => {
    const resp = { ...detectorWithJobResponse.anomaly_detector,
      id: detectorWithJobResponse._id,
      primaryTerm: detectorWithJobResponse._primary_term,
      seqNo: detectorWithJobResponse._seq_no,
      adJob: { ...detectorWithJobResponse.anomaly_detector_job
      },
      historicalTask: { ...detectorWithJobResponse.anomaly_detection_task
      }
    };
    resultDetectorWithJobs.push(convertDetectorKeysToCamelCase(resp));
  });
  return resultDetectorWithJobs;
};

exports.getDetectorsWithJob = getDetectorsWithJob;

const isIndexNotFoundError = err => {
  return err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception';
};

exports.isIndexNotFoundError = isIndexNotFoundError;

const getErrorMessage = err => {
  return !(0, _lodash.isEmpty)((0, _lodash.get)(err, 'body.error.reason')) ? (0, _lodash.get)(err, 'body.error.reason') : (0, _lodash.get)(err, 'message');
};

exports.getErrorMessage = getErrorMessage;

const getDetectorTasks = detectorTaskResponses => {
  const detectorToTaskMap = {};
  detectorTaskResponses.forEach(response => {
    const detectorId = (0, _lodash.get)(response, '_id', '');
    const detectorTask = (0, _lodash.get)(response, 'anomaly_detection_task', null);

    if (detectorTask !== null) {
      detectorToTaskMap[detectorId] = detectorTask;
    }
  });
  return detectorToTaskMap;
};

exports.getDetectorTasks = getDetectorTasks;

const getDetectorResults = detectorResultsResponses => {
  const detectorToResultsMap = {};
  detectorResultsResponses.forEach(response => {
    const detectorId = (0, _lodash.get)(response, 'hits.hits.0._source.detector_id', null);

    if (detectorId !== null) {
      detectorToResultsMap[detectorId] = response;
    }
  });
  return detectorToResultsMap;
}; // Append task-related info - task state & anomaly results of the task.
// If there is no related task info for a detector: set to default values of DISABLED state and 0 anomalies


exports.getDetectorResults = getDetectorResults;

const appendTaskInfo = (detectorMap, detectorToTaskMap, detectorToResultsMap) => {
  const detectorMapWithTaskInfo = {};
  const detectorsWithTasks = Object.keys(detectorToTaskMap);
  Object.keys(detectorMap).forEach((detectorId, index) => {
    if (!detectorsWithTasks.includes(detectorId)) {
      detectorMapWithTaskInfo[detectorId] = { ...detectorMap[detectorId],
        curState: _constants.DETECTOR_STATE.DISABLED,
        totalAnomalies: 0
      };
    } else {
      const task = detectorToTaskMap[detectorId];
      const state = getTaskState(task);
      const totalAnomalies = (0, _lodash.get)(detectorToResultsMap[detectorId], 'hits.total.value', 0);
      detectorMapWithTaskInfo[detectorId] = { ...detectorMap[detectorId],
        curState: state,
        totalAnomalies: totalAnomalies
      };
    }
  });
  return detectorMapWithTaskInfo;
}; // Following checks/transformations need to be made here:
// - set to DISABLED if there is no existing task for this detector
// - set to UNEXPECTED_FAILURE if the task is in a FAILED state & the error message is unreadable / is a stack trace
// - set to INIT if the task is in a CREATED state
// - set to DISABLED if the task is in a STOPPED state


exports.appendTaskInfo = appendTaskInfo;

const getTaskState = task => {
  const state = (0, _lodash.get)(task, 'state', 'DISABLED');
  const errorMessage = processTaskError((0, _lodash.get)(task, 'error', ''));
  const updatedState = state === 'FAILED' && errorMessage.includes(_constants.STACK_TRACE_PATTERN) ? 'UNEXPECTED_FAILURE' : state === 'CREATED' ? 'INIT' : state === 'STOPPED' ? 'DISABLED' : state; //@ts-ignore

  return _constants.DETECTOR_STATE[updatedState];
};

exports.getTaskState = getTaskState;

const processTaskError = error => {
  const errorWithPrefixRemoved = error.replace(_constants.OPENSEARCH_EXCEPTION_PREFIX, '');
  return (0, _lodash.isEmpty)(errorWithPrefixRemoved) || errorWithPrefixRemoved.endsWith('.') ? errorWithPrefixRemoved : errorWithPrefixRemoved + '.';
}; // Filtering by 'is_latest=true' is not enough here. During backfilling of legacy
// realtime detectors on the backend, it is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.


exports.processTaskError = processTaskError;

const getLatestDetectorTasksQuery = realtime => {
  const taskTypes = realtime ? _constants.REALTIME_TASK_TYPES : _constants.HISTORICAL_TASK_TYPES;
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    },
    aggs: {
      detectors: {
        terms: {
          field: 'detector_id',
          size: _constants.MAX_DETECTORS
        },
        aggs: {
          latest_tasks: {
            top_hits: {
              size: 1,
              sort: {
                execution_start_time: _constants.SORT_DIRECTION.DESC
              }
            }
          }
        }
      }
    }
  };
};

exports.getLatestDetectorTasksQuery = getLatestDetectorTasksQuery;

const isRealTimeTask = task => {
  return (0, _lodash.get)(task, 'task_type', '').includes(_constants.REALTIME_TASK_TYPE_PREFIX);
};

exports.isRealTimeTask = isRealTimeTask;

const getFiltersFromEntityList = entityListAsObj => {
  let filters = [];
  Object.values(entityListAsObj).forEach(entity => {
    filters.push({
      nested: {
        path: _constants.ENTITY_FIELD,
        query: {
          term: {
            [_constants.ENTITY_NAME_PATH_FIELD]: {
              value: entity.name
            }
          }
        }
      }
    });
    filters.push({
      nested: {
        path: _constants.ENTITY_FIELD,
        query: {
          term: {
            [_constants.ENTITY_VALUE_PATH_FIELD]: {
              value: entity.value
            }
          }
        }
      }
    });
  });
  return filters;
}; // Filtering by 'is_latest=true' is not enough here. During backfilling of legacy
// realtime detectors on the backend, it is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.


exports.getFiltersFromEntityList = getFiltersFromEntityList;

const getLatestTaskForDetectorQuery = (detectorId, realtime) => {
  const taskTypes = realtime ? _constants.REALTIME_TASK_TYPES : _constants.HISTORICAL_TASK_TYPES;
  return {
    size: 1,
    sort: {
      execution_start_time: _constants.SORT_DIRECTION.DESC
    },
    query: {
      bool: {
        filter: [{
          term: {
            detector_id: detectorId
          }
        }, {
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    }
  };
};

exports.getLatestTaskForDetectorQuery = getLatestTaskForDetectorQuery;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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