import { Client, ClientEvents } from "matrix-org-irc";
import { IrcServer } from "./IrcServer";
import { IrcPoolClient } from "../pool-service/IrcPoolClient";
import { RedisIrcConnection } from "../pool-service/RedisIrcConnection";
export interface ConnectionOpts {
    localAddress?: string;
    password?: string;
    realname: string;
    username?: string;
    nick: string;
    secure?: {
        ca?: string;
    };
    encodingFallback?: string;
    useRedisPool?: IrcPoolClient;
}
export type InstanceDisconnectReason = "throttled" | "irc_error" | "net_error" | "timeout" | "raw_error" | "toomanyconns" | "banned" | "killed" | "idle" | "limit_reached" | "iwanttoreconnect";
export declare enum IRCConnectionErrorCode {
    Unknown = 0,
    Banned = 1,
    ILine = 2
}
export declare class IRCConnectionError extends Error {
    readonly code: IRCConnectionErrorCode;
    constructor(code: IRCConnectionErrorCode, message: string);
}
export declare class ConnectionInstance {
    readonly client: Client;
    private readonly domain;
    private nick;
    private pingOpts;
    private readonly homeserverDomain;
    private readonly redisConn?;
    dead: boolean;
    private state;
    private pingRateTimerId;
    private clientSidePingTimeoutTimerId;
    private connectDefer;
    onDisconnect?: (reason: string) => void;
    /**
     * Create an IRC connection instance. Wraps the matrix-org-irc library to handle
     * connections correctly.
     * @constructor
     * @param client The new IRC client.
     * @param domain The domain (for logging purposes)
     * @param nick The nick (for logging purposes)
     * @param pingOpts Options for automatic pings to the IRCd.
     * @param homeserverDomain The homeserver's domain, for the CTCP version string.
     */
    constructor(client: Client, domain: string, nick: string, pingOpts: {
        pingRateMs: number;
        pingTimeoutMs: number;
    }, homeserverDomain: string, redisConn?: RedisIrcConnection | undefined);
    /**
     * Connect this client to the server. There are zero guarantees this will ever
     * connect.
     * @return {Promise} Resolves if connected; rejects if failed to connect.
     */
    connect(): Promise<ConnectionInstance>;
    /**
     * Blow away the connection. You MUST destroy this object afterwards.
     * @param {string} reason - Reason to reject with. One of:
     * throttled|irc_error|net_error|timeout|raw_error|toomanyconns|banned
     */
    disconnect(reason: InstanceDisconnectReason, ircReason?: string): Promise<void>;
    addListener<T extends keyof ClientEvents>(eventName: T, fn: ClientEvents[T]): void;
    private listenForErrors;
    private listenForPings;
    private listenForCTCPVersions;
    private resetPingSendTimer;
    /**
     * Create an IRC client connection and connect to it.
     * @param {IrcServer} server The server to connect to.
     * @param {Object} opts Options for this connection.
     * @param {string} opts.nick The nick to use.
     * @param {string} opts.username The username to use.
     * @param {string} opts.realname The real name of the user.
     * @param {string} opts.password The password to give NickServ.
     * @param {string} opts.localAddress The local address to bind to when connecting.
     * @param {string} homeserverDomain Domain of the homeserver bridging requests.
     * @param {Function} onCreatedCallback Called with the client when created.
     * @return {Promise} Resolves to an ConnectionInstance or rejects.
     */
    static create(server: IrcServer, opts: ConnectionOpts, homeserverDomain: string, ident: string, onCreatedCallback?: (inst: ConnectionInstance) => void): Promise<ConnectionInstance>;
}
