package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.36.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.36.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````Y!@``#0```"\H0````0`!30`(``)
M`"@`'``;``$``'!TC0``=(T``'2-``!P````<`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````#HC0``Z(T```4```````$`
M`0```*R=``"LG0$`K)T!`)P"``"P`@``!@```````0`"````M)T``+2=`0"T
MG0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DK)T`
M`*R=`0"LG0$`5`(``%0"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`.*M0<78;7/5[ANUZ[[\=-AS-C",$
M````$`````$```!'3E4```````,````"`````````#L```!.````"`````@`
M```!D!`$)4`$@"B```6@AE"$@($``@LH!(PX@!)!B@``"`````!.````4```
M`%$```````````````````!3`````````%0```!5````5@``````````````
M5P```%@`````````````````````````60```%H```!;`````````%P`````
M``````````````!=````7@```%\```````````````````!@````80``````
M``!C`````````&0`````````90``````````````````````````````````
M`&8```````````````````!G`````````&@```````````````````!J````
M?JF<FUGOEWEK"4/6=E0\E"'_Y`X-.I<+P^4_+(D'<-/9<5@<K4OCP&L:GE#5
M9-1\F4_#FR_%F[8S8MOM:W^:?`&O-.B[XY)\LP&+;T+ZUX=#1=7L9Z_,9A6,
M)<RCX/M)S]Y'+>-'Q@'@IP(*;\7?'[[:5QO[H6Q@````````````````````
M```````T$P````````,`"P```````*`!```````#`!<`Y0(`````````````
M$@```#\!`````````````!(```!=`@`````````````2````CP(`````````
M````$@```$0"`````````````!(```"````````````````2````L```````
M````````$@```!D#`````````````!(```!\`0`````````````2`````0``
M````````````(````+$!`````````````!(```"G```````````````2````
M4P``````````````$@````,"`````````````!(```"[```````````````2
M````?P(`````````````$@```!```````````````"`````?`0``````````
M```2````$@(`````````````$@```$0!`````````````!(```"L`0``````
M```````2````W@(`````````````$@```"P``````````````"````"(`@``
M```````````2````CP$`````````````$@````P!`````````````!8```"7
M```````````````2````;P``````````````$@```&L!`````````````!(`
M``#8```````````````2````-@(`````````````$@```/4!````````````
M`!(```!X`@`````````````2````_@,`````````````$@```,D`````````
M`````!(```#M`@`````````````1````[@$`````````````$@````<#````
M`````````!$```"?`@`````````````2````*0(`````````````$@```.<`
M`````````````!(````@`@`````````````2````]0``````````````$@``
M`#T"`````````````!(```!C```````````````2````EP(`````````````
M$@```%4!`````````````!(```#-`@`````````````2````V0$`````````
M````$@```/P!`````````````!(````P`@`````````````2````AP(`````
M````````$@```+\!`````````````!(```#U`@`````````````2````[`(`
M````````````(0```"H!`````````````!(```"X`0`````````````2````
M20$`````````````$@```,0"`````````````!(```#K`@`````````````1
M````<0(`````````````$0```$8``````````````!$```!3`@``````````
M```2````8P(`````````````$@````H"`````````````!(```",````````
M```````2````2P(`````````````$@```%P!`````````````!(```#'`0``
M```````````2````,`,`````````````$@```!D"`````````````!(```#G
M`0`````````````2````OP(`````````````$@```*X"`````````````!(`
M``"=`0`````````````B````[`,``)0O```D````$@`-`%`$``!<H`$`````
M`!``&``B!````*`!```````0`!<`+P0``$B@`0``````$``8`*0$```((P``
MD`(``!(`#0`D!````*`!```````@`!<`2`0``)PA``!4````$@`-`+$$``!$
MH`$`!````!$`%P!D!```2*`!```````0`!@`$P0``%A+```$````$0`/`.(#
M``#020``;`$``!(`#0"<`P``'#@``-0"```2``T`NP0``%PM``!T````$@`-
M`'X#```$(P``!````!(`#0`H!```Y!@````````2``T`U`$``#`6``"T`@``
M$@`-`%D$``!<H`$``````!``&`!Y!```7*`!```````0`!@`L@,``+PL``"@
M````$@`-`#T$``"X2```&`$``!(`#0"+!```2*`!```````0`!<`U@,``,@E
M``"P`0``$@`-`'`$``!D2```5````!(`#0##!```>"<``$0%```2``T`?@0`
M`"PQ``#P!@``$@`-`(\#``"8)0``,````!(`#0#``P``\"$``!0!```2``T`
M!P0``/`Z``!``0``$@`-`%@$``!<H`$``````!``&`"2!```N"\``'0!```2
M``T``%]?9VUO;E]S=&%R=%]?`%])5$U?9&5R96=I<W1E<E1-0VQO;F5486)L
M90!?251-7W)E9VES=&5R5$U#;&]N951A8FQE`%!,7V1O7W5N9'5M<`!097)L
M7V=V7V9E=&-H<'8`4&5R;%]M9U]S970`4&5R;%]A=&9O<FM?;&]C:P!097)L
M7V=E=%]S=@!097)L7VYE=UA3`&)O;W1?1'EN84QO861E<@!P97)L7W)U;@!P
M97)L7W!A<G-E`%!E<FQ?<W9?<V5T<'8`4&5R;%]S=E]S971P=FX`<&5R;%]C
M;VYS=')U8W0`<&5R;%]D97-T<G5C=`!097)L7V-R;V%K7VUE;6]R>5]W<F%P
M`%!,7V-U<G)E;G1?8V]N=&5X=`!P97)L7V%L;&]C`%!E<FQ?8W)O86M?;F]C
M;VYT97AT`'!E<FQ?9G)E90!097)L7W-A9F5S>7-M86QL;V,`4&5R;%]S>7-?
M:6YI=#,`4&5R;%]N97=84U]F;&%G<P!097)L7V%T9F]R:U]U;FQO8VL`4&5R
M;%]S>7-?=&5R;0!?7V-X85]F:6YA;&EZ90!R96%D`'-T<F-H<@!G971P:60`
M;65M;6]V90!?7VQI8F-?<W1A<G1?;6%I;@!?7V9P<FEN=&9?8VAK`'-T<F-M
M<`!O<&5N-C0`=6YL:6YK`'-T<G1O:P!M96US970`<W1R;F-M<`!G971U:60`
M;65M8VAR`&-L;W-E9&ER`'-T<G-T<@!R;61I<@!S=')L96X`<W1R9'5P`&UE
M;6-M<`!R96%L;&]C`')E861D:7(V-`!A8F]R=`!?7W-P<FEN=&9?8VAK`'-T
M9&5R<@!M96UC<'D`<W1R<F-H<@!L<W1A=#8T`&]P96YD:7(`;'-E96LV-`!?
M7W9F<')I;G1F7V-H:P!?7V5R<FYO7VQO8V%T:6]N`&5X:70`<F5A9&QI;FL`
M7U]S=&%C:U]C:&M?9F%I;`!A8V-E<W,`;6MD:7(`7U]E;G9I<F]N`%]?<F5G
M:7-T97)?871F;W)K`%]?<W1A8VM?8VAK7V=U87)D`%]?865A8FE?=6YW:6YD
M7V-P<%]P<C``7U]A96%B:5]U;G=I;F1?8W!P7W!R,0!L:6)P97)L+G-O`&QI
M8F,N<V\N-@!L9"UL:6YU>"UA<FUH9BYS;RXS`&QI8F=C8U]S+G-O+C$`<&%R
M7V-U<G)E;G1?97AE8P!P87)?8F%S96YA;64`6%-?26YT97)N86QS7U!!4E]"
M3T]4`'!A<E]E;G9?8VQE86X`<&%R7V-U<G)E;G1?97AE8U]P<F]C`'!A<E]D
M:7)N86UE`'-H85]F:6YA;`!G971?=7-E<FYA;65?9G)O;5]G971P=W5I9`!P
M87)?8VQE86YU<`!?24]?<W1D:6Y?=7-E9`!?7V1A=&%?<W1A<G0`7U]B<W-?
M<W1A<G1?7P!S:&%?=7!D871E`'AS7VEN:70`7U]E;F1?7P!?7V)S<U]E;F1?
M7P!?7V)S<U]S=&%R=`!S:&%?:6YI=`!?96YD`'!A<E]M:W1M<&1I<@!?961A
M=&$`<&%R7W-E='5P7VQI8G!A=&@`<&%R7V9I;F1P<F]G`&UY7W!A<E]P;`!P
M87)?9&EE`'!A<E]I;FET7V5N=@!'3$E"0U\R+C0`1T-#7S,N-0!'3$E"0U\R
M+C,T`$=,24)#7S(N,S,`+W5S<B]L:6(O<&5R;#4O-2XS-B]C;W)E7W!E<FPO
M0T]210```````````@`!``(``@`"``$``0`#``$``0`"``$``0`"``$``@`!
M``$``@`"``(``@`!``0``0`!``$``0`!``$``@`"``(``@`!``(``@`%``(`
M`@`!``(``0`"``$``@`"``(``@`"``(`!``"``(``@`!``(``0`"``(``@`!
M``(``@`"``$``@`!``8``P`"``(``@`"``(``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!
M`%P#```0````(````!1I:0T```4`T`0````````!``$`<`,``!`````@````
M529Y"P```P#:!`````````$``P!2`P``$`````````"TD98&```&`.($```0
M````LY&6!@``!`#M!```$````!1I:0T```(`T`0```````"LG0$`%P```+"=
M`0`7````_)\!`!<````$H`$`%P```!2@`0`7````&*`!`!<````@H`$`%P``
M`"2@`0`7````**`!`!<````LH`$`%P```#"@`0`7````.*`!`!<````\H`$`
M%P```$2@`0`7````S)\!`!4+``#0GP$`%0P``-2?`0`5$P``V)\!`!49``#<
MGP$`$QP``."?`0`5'0``Y)\!`!4>``#HGP$`%28``.R?`0`5*```\)\!`!4_
M``#TGP$`%4```/B?`0`530``V)X!`!8#``#<G@$`%@0``.">`0`6!0``Y)X!
M`!8&``#HG@$`%@<``.R>`0`6"```\)X!`!8)``#TG@$`%@P``/B>`0`6#0``
M_)X!`!8.````GP$`%@\```2?`0`6$```")\!`!81```,GP$`%A(``!"?`0`6
M%```%)\!`!85```8GP$`%A8``!R?`0`6%P``()\!`!88```DGP$`%AH``"B?
M`0`6&P``+)\!`!8?```PGP$`%B```#2?`0`6(0``.)\!`!8B```\GP$`%B,`
M`$"?`0`6)```1)\!`!8E``!(GP$`%B<``$R?`0`6*0``4)\!`!8J``!4GP$`
M%BL``%B?`0`6+```7)\!`!8M``!@GP$`%BX``&2?`0`6+P``:)\!`!8P``!L
MGP$`%C$``'"?`0`6,@``=)\!`!8S``!XGP$`%C0``'R?`0`6-0``@)\!`!8V
M``"$GP$`%C<``(B?`0`6.```C)\!`!8Z``"0GP$`%CL``)2?`0`6/```F)\!
M`!8]``"<GP$`%D$``*"?`0`60@``I)\!`!9#``"HGP$`%D0``*R?`0`610``
ML)\!`!9&``"TGP$`%D<``+B?`0`620``O)\!`!9*``#`GP$`%DL``,2?`0`6
M3```R)\!`!9-```(0"WI>@$`ZPB`O>@$X"WE!."?Y0[@C^`(\+[E?(L!``#&
MC^(8RHSB?/N\Y0#&C^(8RHSB=/N\Y0#&C^(8RHSB;/N\Y0#&C^(8RHSB9/N\
MY0#&C^(8RHSB7/N\Y0#&C^(8RHSB5/N\Y0#&C^(8RHSB3/N\Y0#&C^(8RHSB
M1/N\Y0#&C^(8RHSB//N\Y0#&C^(8RHSB-/N\Y0#&C^(8RHSB+/N\Y0#&C^(8
MRHSB)/N\Y0#&C^(8RHSB'/N\Y0#&C^(8RHSB%/N\Y0#&C^(8RHSB#/N\Y0#&
MC^(8RHSB!/N\Y0#&C^(8RHSB_/J\Y0#&C^(8RHSB]/J\Y0#&C^(8RHSB[/J\
MY0#&C^(8RHSBY/J\Y0#&C^(8RHSBW/J\Y0#&C^(8RHSBU/J\Y0#&C^(8RHSB
MS/J\Y0#&C^(8RHSBQ/J\Y0#&C^(8RHSBO/J\Y0#&C^(8RHSBM/J\Y0#&C^(8
MRHSBK/J\Y0#&C^(8RHSBI/J\Y0#&C^(8RHSBG/J\Y0#&C^(8RHSBE/J\Y0#&
MC^(8RHSBC/J\Y0#&C^(8RHSBA/J\Y0#&C^(8RHSB?/J\Y0#&C^(8RHSB=/J\
MY0#&C^(8RHSB;/J\Y0#&C^(8RHSB9/J\Y0#&C^(8RHSB7/J\Y0#&C^(8RHSB
M5/J\Y0#&C^(8RHSB3/J\Y0#&C^(8RHSB1/J\Y0#&C^(8RHSB//J\Y0#&C^(8
MRHSB-/J\Y0#&C^(8RHSB+/J\Y0#&C^(8RHSB)/J\Y0#&C^(8RHSB'/J\Y0#&
MC^(8RHSB%/J\Y0#&C^(8RHSB#/J\Y0#&C^(8RHSB!/J\Y0#&C^(8RHSB_/F\
MY0#&C^(8RHSB]/F\Y0#&C^(8RHSB[/F\Y0#&C^(8RHSBY/F\Y0#&C^(8RHSB
MW/F\Y0#&C^(8RHSBU/F\Y0#&C^(8RHSBS/F\Y0#&C^(8RHSBQ/F\Y0#&C^(8
MRHSBO/F\Y0#&C^(8RHSBM/F\Y0#&C^(8RHSBK/F\Y0#&C^(8RHSBI/F\Y0#&
MC^(8RHSBG/F\Y7!`+>D!RJ#C#,!-X,@/C.5,PI_E*-!-XD@RG^4,P(_@1$*?
MY1P`C>4<`(WB&!"-Y1@0C>(4((WE%""-X@1`C^`#,)SG`#"3Y20PC>4`,*#C
MU?__ZQ0BG^44,I_E`B"4YPP@C>4"$*#A`S"4YQ`PC>4#`*#A)`T`Z_@QG^4#
M,)3G`$#3Y0``5.-2```*<"\=[N0QG^4<4)WE!`"%X@-0A>(#,)_G!P%PXP,@
MDN>>,]+E`S"#XYXSPN5C``"*``&@X:3__^NT(9_E&$"=Y0`PH.$"((_@J!&?
MY0$0C^`$`(+E`""4Y0`@@^68(9_E'`"=Y0(@C^`,((/EC"&?Y0$`4.,`$)'E
M`B"/X`@0@^4$((/E3@``V@00A.(,P(/B`$&$X`3@D>0$X*SE`0!4X?O__QH#
M`(#B5$&?Y0`0H.,`$(WE!2"@X01`C^``$8/G0!&?Y0P`E.4!$(_@`___ZP``
M4.,I```*+%&?Y2!@C>(%4(_@#`"5Y4?__^L`0*#A&`&?Y080H.$``(_@I@``
MZP``4.,@```*!#&?Y0,PC^`,`)/EXO[_ZQK__^OT()_EM#"?Y0(@C^`#,)+G
M`""3Y20PG>4"(#/@`#"@XR0``!H$`*#A*-"-XG"`O>C[_O_KQ""?Y0``4.,"
M((_@#`""Y1D```H-___KL#"?Y7`O'>X#,)_G`S"2YT='P^6@___J#`"4Y=[^
M_^O2___J"`"5Y0``4.,!```*I0@`Z]G__^I\`)_E!A"@X0``C^!Z``#K``!0
MXP@`A>72__\*]?__ZA?__^L$`*#CMO__Z@$`H.->___K(?__ZWB(`0`@`0``
M6(@!```!```8`0``*`$```2)`0!$B0$`-(D!`.@W``#0-P``T(@!`&0)``"H
MB`$`]#0``'R(`0#DA@$`,(@!`*B'`0#@,P```+"@XP#@H.,$$)WD#2"@X00@
M+>4$`"WE'*"?Y1@PC^(#H(K@`#"@XP0P+>4,`)_E``":YS+__^N2_O_KK(4!
M`#`!```4,)_E%""?Y0,PC^`"()/G``!2XQ[_+P&8_O_JE(4!``0!```L`)_E
M+#"?Y0``C^`H()_E`S"/X```4^$"((_@'O\O`1@PG^4#,)+G``!3XQ[_+P$3
M_R_A[(8!`.2&`0!@A0$`"`$``#@`G^4X,)_E``"/X#0@G^4#,(_@`#!#X`(@
MC^"C'Z#A0Q&!X,$0L.$>_R\!&#"?Y0,PDN<``%/C'O\O`1/_+^&HA@$`H(8!
M`!R%`0`,`0``6#"?Y5@@G^4#,(_@`B"/X``PT^4``%/C'O\O$1!`+>D!RJ#C
M#,!-X/@/C.4T,)_E`S"2YP``4^,"```**#"?Y0,`G^?^_O_KQ___ZQPPG^4!
M(*#C`S"/X``@P^40@+WH6(8!`-B$`0`L`0``W(4!``R&`0#-___J\$<MZ0'*
MH.,,P$W@X`^,Y<@PG^4`@%#B`S"/X"L```J\()_E`C"3YP"0D^4``%GC*```
M"@!0V.4!<*#A``!5XST`51,!4*`3`%"@`Q$``!H`0)GE``!4XPP```H)8*#A
M!`"@X04@H.$($*#AN_[_ZP``4.,"```:!3#4YST`4^,+```*!$"VY0``5./S
M__\:!`"@X?"'O>@(4*#A`3#UY0``4^,]`%,3^___&@A01>#F___J"6!&X`%0
MA>(%0(3@1F&@X00`H.$`8(?E\(>]Z`A`H.'M___J"4"@X>O__^I4A`$`'`$`
M`!!`+>D!RJ#C#,!-X(@/C.6,()_E<-!-XH@PG^4-$*#A`B"/X`!`H.$#,)+G
M`#"3Y6PPC>4`,*#C<_[_ZP``4.,``*`3!```&A`PG>4/.@/B"@I3XP$)4Q,*
M```*2""?Y4`PG^4"((_@`S"2YP`@D^5L,)WE`B`SX``PH.,'```:<-"-XA"`
MO>@"$*#C!`"@X17^_^L0#V_AH`*@X>[__^I-_O_K9(,!`"`!```@@P$`\$`M
MZ0'*H.,,P$W@<`^,Y5PAG^5\T$WB6#&?Y0!04.("((_@`S"2YP`PD^5T,(WE
M`#"@XP(```H`,-7E``!3XPH``!HP(9_E*#&?Y0(@C^`#,)+G`""3Y70PG>4"
M(#/@`#"@XT$``!I\T(WB\("]Z,3]_^L`0%#B\?__"@0`H.%*_O_K``!0XRH`
M``H3,-#E$V"`XBX`4^,"```:`3#6Y0``4^/T__\*$S#0Y2X`4^,E```*!0"@
MX>[]_^L`<*#A!@"@X>O]_^L``(?@`@"`XA+^_^ND,)_E`'"@X0$0H.,#,(_@
M`"#@XP1@C>4`4(WE,?[_ZP@0C>('`*#AT?W_ZP$`<.,'`*#A`P``"A@PG>4/
M.@/B`0E3XQ(```K8_?_K!P"@X;[]_^L$`*#A'_[_ZP``4./4__\:!`"@X>C]
M_^L%`*#A`?[_Z[W__^H!,-;E+@!3X];__QH",-;E``!3X\7__PK2___JI/__
MZ^O__^KM_?_KM((!`"`!``"(@@$`%"\``/!!+>D!RJ#C#,!-X```C.4@``SE
M`7"@X0`1G^4`,9_E`=I-X@$0C^`(T$WB`&"@X0$*C>($`(#B!("-X@,PD>?_
M+P#C"!"@X0`PD^4`,(#E`#"@X\P`G^4``(_@[_W_ZP!`4.('``#:"%"-X@$P
M1.(#P(7@!"!<Y0``4N,7```:``!3XQ0``!J<,)_E<`\=[@<@H.$&$*#A`S"?
MYP,`D.=R_?_KA""?Y0$:C>)P,)_E!!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@
M`#"@XQ(``!H!VHWB"-"-XO"!O>@#0*#A!""@X2\0H.,(`*#AX_W_ZP``4./C
M__\*-`"?Y7#/'>X$,*#A"""@X080H.$``)_G``"<YX+]_^OA___JH?W_ZRR!
M`0`@`0``("X``,B!`0"<@`$`5($!`/!'+>D!RJ#C#,!-X-@/C.44(I_E"-!-
MXA`RG^4!0*#A`B"/X`!0H.$$@I_E`S"2YPB`C^``,)/E!#"-Y0`PH.,`,-'E
M/0!3XP%`@0($`*#A6?W_ZPT0H.$`8*#A!0"@X='^_^L`D%#B0```"E+]_^L`
M`%;A-P``FK@QG^4`<)WE`X"8YP=QH.$`,-7E``!3XST`4Q,!,*`3`#"@`P4`
M``H%,*#A`2#SY0``4N,]`%(3^___&@4P0^`"`(;B`&"8Y0,`@.!F_?_K``!0
MXP<`AN<5```*`#"8Y04`H.$'$)/G`3#0Y``PP>4``%/C/0!3$P8```H!(('B
M`3#0Y`(0H.$!,,+D/0!3XP``4Q/Y__\:`2"@X3TPH.,`,,'E`3#4Y`$PXN4`
M`%/C^___&@@AG^7X,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C-0``&@C0
MC>+PA[WH`9!)X@$PU.0!,.GE``!3X_O__QKN___JP#"?Y0.`F.<`<)CE`#"7
MY0``4^,D```*!S"@X00@L^4)$*#A`9")X@``4N/Z__\:`Q"!X@$1H.&0,)_E
M`S"/X``@D^4``%+C#P``&@$`H.$!(*#C`""#Y27]_^L`H%#BU?__"@<0H.$)
M<:#A!R"@X3']_^L`H(CE`)"-Y0&0B>(`,*#C"3&*YZ+__^H'`*#A1_W_ZP"@
MH.$``%#C``"(Y<7__PH)<:#A\O__Z@.0H.$($*#CW___ZA#]_^OX?P$`(`$`
M`.A_`0`<`0``Y'X!`.Q_`0`00"WI`<J@XPS`3>#H#XSE<!"?Y7!/'>YL()_E
M$-!-XF@`G^4"((_@9#"?Y0$0G^<#,(_@`""2YP$`E.=4$)_E#""-Y0$0C^`A
M_?_K2!"?Y4@PG^4`P*#C1""?Y0,PC^`!$)_G`B"/X`$`E.<T$)_E!,"-Y0$0
MC^``,(WEM_S_ZQ#0C>(0@+WHN'X!`+!]`0`4`0``A&L``,@J``"$?@$`M"T`
M`,`6``"T*@``.`"?Y0C03>(T()_E<,\=[C`0G^4"((_@+#"?Y0``G^<#,(_@
M`2"2YR`0G^4``)SG`1"/X`0@C>4(T(WB^?S_ZAQ^`0`4?0$`%`$``.AJ```P
M*@``<$`MZ0'*H.,,P$W@``",Y0'*3.(``(SE*``,Y=P@G^4"VDWBV#"?Y1C0
M3>("((_@`AJ-XA00@>(,0(WB`6J-X@,PDN<08(;B`#"3Y0`P@>4`,*#CS/S_
MZZ@0G^4!(`'CI#"?Y0$0C^#P`,WA`S"/X`$0H.,$`*#AS_S_ZP0`H.$!*J#C
M!A"@X<7\_^L`0%#B&```N@$`A.*@_/_K`%!0XA0```H$(*#A!A"@X7?\_^L`
M,*#C!##%YU`@G^4"&HWB/#"?Y100@>("((_@`S"2YP`@D^4`,)'E`B`SX``P
MH.,#```:!0"@X0+:C>(8T(WB<("]Z(S\_^L`4*#C[?__ZJA\`0`@`0``Z"D`
M`-0I```,?`$`N?__ZO!/+>D!RJ#C#,!-X```C.6H``SE1"*?Y4+=3>)`,I_E
M!-!-X@(@C^``8*#A-`*?Y4'-C>(0@(WB/,",X@,PDN<!0*#A``"/X`@0H.$`
M,)/E`#",Y0`PH..\_?_K+Q"@XP!0H.$&`*#A#_S_ZP``4.,2```*\`&?Y080
MH.$``(_@Q_[_ZP8`H.'@(9_E01V-XLPQG^4\$('B`B"/X`,PDN<`()/E`#"1
MY0(@,^``,*#C:0``&D+=C>($T(WB\(^]Z`0`H.%&_/_KI!&?Y0$0C^!5_/_K
M`$!0XN7__PJ4D9_E@*"-X@`PU.4$<$KB"9"/X```4^,[```*``!5XP0```H$
M$*#A!0"@X7K\_^L``%#C*P``"@0`H.$0_/_K`3!`X@,PA.``L*#A`P!4X0`P
MA#``$*`S`0"$,@,``#H(``#J`P!0X0`0P^4"```*`2!SY2\`4N/Y__\*!`"@
MX?_[_^L`L*#A!@"@X?S[_^L+`(#@`0"`X@$*4..]__\J^`"?Y0DPH.$!*J#C
M`1"@XP``C^``0(WE00"-Z0<`H.%`_/_K"!"@X0<`H.'@^__K``!0XP,``!I@
M,!KE#SH#X@()4^,5```*M!"?Y0``H.,!$(_@%?S_ZP!`4.*E__\*`##4Y0``
M4^/#__\:``!5XQ8```H`,-7E+@!3XP,```J`0)_E`;"@XP1`C^#5___J`3#5
MY0``4^/J__\*]___Z@$0H.,'`*#AOOO_ZP``4./D__\:4`"?Y0<0H.$``(_@
M6/[_ZP<`H.'F^__KCO__ZCA`G^4!L*#C!$"/X,'__^KL^__KF'L!`"`!``#X
M*```S"@``"1[`0",*```<"@``-@G``",)P``-"<``!`G``#D)@``$$`MZ0'*
MH.,,P$W@^`^,Y2\0H.,`0*#ACOO_ZP``4.,!0(`2!`"@X1!`O>C(^__J\$`M
MZ0'*H.,,P$W@``",Y2``#.5X(9_E`=I-XG0QG^4,T$WB`B"/X`$:C>($$('B
M`%!0X@,PDN<`,)/E`#"!Y0`PH.,Q```*`##5Y0``4^,N```*D?O_ZP$@@.(`
M0*#A`0I2XSH``(H$<(WB!1"@X0<`H.&/^__K`3!$X@,PA^`#`%?A"0``*@0`
MA^`%((WB`!"@XP(``.H"`%#A`!#`Y0(```H!,'#E+P!3X_G__PHO$*#C!P"@
MX5K[_^L``%#C)0``"@<`4.$`,*"#`##`A0<`H.&2^__KO""?Y0$:C>*P,)_E
M!!"!X@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XR(``!H!VHWB#-"-XO"`O>B(
M()_E`1J-XG@PG^4$$('B`B"/X`,PDN<`()/E`#"1Y0(@,^``,*#C%```&F``
MG^4``(_@`=J-X@S0C>+P0+WH<_O_Z@``H./>___J1""?Y0$:C>(L,)_E!!"!
MX@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XR``GP4``(\`[/__"F_[_^O8>`$`
M(`$``!1X`0#<=P$`."4``)!W`0#P)```<$`MZ0'*H.,,P$W@Z`^,Y<0DG^4(
MT$WBP#2?Y0U`H.$"((_@N&2?Y;A4G^4&8(_@`S"2YP50C^``,)/E!#"-Y0`P
MH.,$$*#A!@"@X:/\_^L``%#C#@``"HPTG^4``)WE`S"5YP`0D^4`,8'@!""3
MY0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZE1DG^4&8(_@
M!!"@X08`H.&-_/_K``!0XPX```HT-)_E``"=Y0,PE><`$)/E`#&!X`0@D^4`
M(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^H`9)_E!F"/X`00
MH.$&`*#A=_S_ZP``4.,.```*W#.?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!
MYP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JK&.?Y09@C^`$$*#A
M!@"@X6'\_^L``%#C#@``"H0SG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><`
M`%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZEACG^4&8(_@!!"@X08`
MH.%+_/_K``!0XPX```HL,Y_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2
MX_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^H$8Y_E!F"/X`00H.$&`*#A
M-?S_ZP``4.,.```*U#*?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q
M__\*!#"#X@0@D^4$((/D``!2X_O__QKK___JL&*?Y09@C^`$$*#A!@"@X1_\
M_^L``%#C#@``"GPRG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__
M"@0P@^($()/E!""#Y```4N/[__\:Z___ZEQBG^4&8(_@!!"@X08`H.$)_/_K
M``!0XPX```HD,I_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$
M,(/B!""3Y00@@^0``%+C^___&NO__^H(8I_E!F"/X`00H.$&`*#A\_O_ZP``
M4.,.```*S#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#
MX@0@D^4$((/D``!2X_O__QKK___JM&&?Y09@C^`$$*#A!@"@X=W[_^L``%#C
M#@``"G0QG^4``)WE`S"5YP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($
M()/E!""#Y```4N/[__\:Z___ZF!AG^4&8(_@!!"@X08`H.''^__K``!0XPX`
M``H<,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3
MY00@@^0``%+C^___&NO__^H,`9_E!!"@X0``C^"R^__K`!!0X@(```KX`)_E
M``"/X,+\_^OP`)_E!!"@X0``C^"I^__K`!!0X@(```K<`)_E``"/X+G\_^O4
M`)_E!!"@X0``C^"@^__K`!!0XA(```K``)_E``"/X+#\_^NX()_E9#"?Y0(@
MC^`#,)+G`""3Y00PG>4"(#/@`#"@XQ```!J8$)_EF`"?Y0$0C^```(_@"-"-
MXG!`O>B@_/_JA`"?Y000H.$``(_@A_O_ZP`04.+I__\*<`"?Y0``C^"7_/_K
MY?__ZC'Z_^LL=P$`(`$``,PD```8=P$`'`$``&PD```<)```T",``(`C```X
M(P``?"(``)0B``!((@``_"$``"@A``!4(0``*"$``$0A``!$(0``0"$``&0@
M``#4<@$`'"$``(`@``#L(```<"````3@+>4!RJ#C#,!-X/`/C.5X()_E#-!-
MXG0PG^4-$*#A`B"/X&P`G^4``(_@`S"2YP`PD^4$,(WE`#"@XU;[_^L``%#C
M!0``"@``T.4``%#C,`!0$P$`H`,``*`3`0`@XC0@G^4H,)_E`B"/X`,PDN<`
M()/E!#"=Y0(@,^``,*#C`0``&@S0C>($\)WD\_G_Z^AQ`0`@`0``T!\``*!Q
M`0`/`"WI`<J@XP3@+>4,P$W@\`^,Y4@0G^4,T$WB1""?Y10PC>(!$(_@/`"?
MY0`PC>4XP)_E``"/X`(@D><!$*#C`""2Y00@C>4`(*#C$""=Y0S`D.<``)SE
MO?G_Z_\`H.,2^O_K1'$!`"`!```T<0$`)`$``/!'+>D`P*#C`>B@XPS`3>`.
MX$S@`<I,X@``C.4.`%SA^___&C@`#N4!V$WB&-!-XG3!G^4"$*#C`>B-X@S`
MC^`4X([B`!"-Y0`@H.-<$9_E`#"@XP"`H.$!$)SG`!"1Y0`0CN4`$*#CL?G_
MZP$`4.(`<,'BQR^@X0(P(.`",%/@<S#_Y@(P(^`",%/@`B#"X`-@4.`"<,?@
M``!7XS@``+H(D*#C`9!`XPQ0C>(`$*#C!B"@X0<PH.$(`*#A`!"-Y9OY_^L!
M`''C`0!P`S(```H)(*#A!1"@X0@`H.%;^?_K`0!PXRD```H'`%#C'P``F@@`
M0.(`0)7@'```*K"@G^4*H(_@`@``Z@%`1.(%`%3A%@``.@@@H.,*$*#A!`"@
MX2/Y_^L``%#C]O__&@5`1.`&`)3@Q!^GX'@@G^4!R(WB:#"?Y13`C.("((_@
M`S"2YP`@D^4`,)SE`B`SX``PH.,0```:`=B-XAC0C>+PA[WH`6A6X@!PQ^(!
M`'?C`0AV`\G__QH``.#C`!"@X>G__^HD`)_E``"/X'O__^L<`)_E``"/X'C_
M_^MG^?_KN'`!`"`!```P<0$`K&\!`.`=``#$'0``$$`MZ0'*H.,,P$W@^`^,
MY1?Y_^LW^?_K``!0XP``D!40@+WH\$`MZ0'*H.,,P$W@T`^,Y3@AG^4<T$WB
M-#&?Y1`0C>("((_@+&&?Y0!0H.$&8(_@`S"2YP8`H.$`,)/E%#"-Y0`PH..5
M^O_K`$!0X@(```H`,-3E``!3XPX``!KX()_E[#"?Y0(@C^`#,)+G`""3Y10P
MG>4"(#/@`#"@XS(``!K8`)_E!1"@X0``C^`<T(WB\$"]Z);[_^H%$*#A&/G_
MZP``4.,*```*M""?Y:`PG^4"((_@`S"2YP`@D^44,)WE`B`SX``PH.,?```:
M'-"-XO"`O>@%`*#A]/C_ZP!PH.$$`*#A\?C_ZP``A^`"`(#B&/G_ZVPPG^5L
M()_E`1"@XP,PC^`(0(WE`B"/X`!0C>4$((WE`"#@XP!`H.$T^?_K2""?Y2@P
MG^4"((_@`S"2YP`@D^44,)WE`B`SX``PH.,$$*`!!@"@`<___PH$^?_K[&X!
M`"`!``!L'0``J&X!`!`=``!<;@$`K!L``*`;``#D;0$`\$\MZ0'*H.,,P$W@
MZ`Z,Y50FG^7TT$WB4#:?Y8A@C>("((_@2,:?Y0"@H.&@0(WB#,"/X`3@H.$#
M,)+G&'",XI10C>(LMI_E`#"3Y>PPC>4`,*#C##",XC#`C.(@@(WB"["/X`<`
MD^@'`(;H#P"WZ`\`KN@#`)?H`P".Z`<`G.@'`(7H\`6?Y0@0H.$``(_@)?K_
MZP!P4.("```*`##7Y0``4^,7`0`:B_C_ZZOX_^L``%#C>@``"@!PD.4``%?C
M=P``"@<`H.&;^/_K@`"@X0$`@.+"^/_K`##7Y0!0H.$<`(WE``!3XPL```J,
ME9_E"9"/X`4`H.$`,(WE`1"@XPDPH.$`(.#C`E"%XMOX_^L!,/?E``!3X_7_
M_QI@!9_E``"/X`(``.H$`+3E``!0XPT```H($*#A^_G_ZP!04.+X__\*`##5
MY0``4^/U__\*+_K_ZP``4./R__\*!0"@X9;X_^L`0%#B#0``&@1`EN0``%3C
M"@``"@`PU.4``%/CSP``"@0`H.$A^O_K``!0X_7__PH$`*#AB/C_ZP!`4.+Q
M__\*!`"@X6/X_^L<4)WE`&"@X04`H.%?^/_K`&"&X`%KAN($8(;B!@"@X83X
M_^NP-)_E`'"@X0$0H.,#,(_@"#"-Y:`TG^4`(.#C#%"-Y0,PC^`$,(WED#2?
MY0!`C>4#,(_@G?C_ZP<=H.,'`*#A!/C_ZP$`<.,#```:LOC_ZP`0D.41`%'C
MJP``&@@0H.$'`*#A>?C_ZP!04.(M```*3!2?Y0`PFN5())_E`0";YP(@C^``
M<(WE`1"@XP``D.5E^/_K`&"@XRPDG^7\,Y_E`B"/X`,PDN<`()/E[#"=Y0(@
M,^``,*#C\P``&@8`H.'TT(WB\(^]Z$CX_^L`<%#BB0``&@0`E>0``%#C"0``
M"@@0H.&<^?_K`#!0XOC__PH`,-/E``!3X_+__QH$`)7D``!0X_7__QK`<Y_E
M#0"@XS_X_^M3,*#C`%"@X0=PC^`<`(WE?/__ZC`PG>4/.@/B`0E3X\W__QHX
M0)WE\O?_ZP``5.')__\:,#"=Y5,PZ.<'#5/CQ?__&@8`H.$L^/_K`&"@X6@#
MG^4($*#A``"/X'GY_^L`$%#B?```"@``FN6A^__K`)!0XG@```H)`*#A]??_
MZP!`H.$#`%#C#```FC`3G^4$`$#B``")X`$0C^!3^/_K``!0XP4``!H$`%3C
MH@``"@1`B>`%,%3E+P!3XYX```H``Y_E"!"@X0``C^!=^?_K``!0XP,```H`
M,-#E,`!3XP``4Q,$```:`!"@XPD`H.'G]__K`$!0XED``!K($I_ER`*?Y0$0
MC^#$HI_E``"/X,"2G^5@^O_KO(*?Y1+X_^NX,I_E"`"-Y0$0H.,#,(_@##"-
MY:@RG^4&`*#A`'"-Y0`@X.,#,(_@!#"-Y90RG^4*H(_@"9"/X`B`C^`#,(_@
M#?C_ZPX``.HF^/_K`#"0Y1$`4^,0```:^_?_ZP%0A>(`P*#A"C"@X00@H.$!
M$*#C!@"@X1"0C>4,4(WE@!&-Z/WW_^L'':#C!@"@X63W_^L`0*#A`0!PX^K_
M_PH'`*#AC_?_ZR`"G^4&$*#A``"/X#'Z_^L&`*#A<O[_ZVC__^H#0*#A-?__
MZIKW_^N``*#A`0"`XOW^_^JP(9_E`#":Y0(`F^<$$(WE`1"@XP!PC>78(9_E
M``"0Y0(@C^"]]__K5O__ZE_^_^O$(9_E6#&?Y0(@C^`#,)+G`""3Y>PPG>4"
M(#/@`#"@XTH``!H'`*#A]-"-XO!/O>B?]__J`)":Y83__^K5_?_K`("@X0"0
M4>*A__^Z"B!0X@"@H.,`,,GB!`"@X;BPC>(`H(WEF/?_ZP8@H.,+$*#A!`"@
MX5OW_^L&`%#C-```&K@@G>5#/*#C03-$XP,`4N&/__\:M"#;X4@U!.,#`%+A
MB___&C(@6.($`*#A`##)XL!0C>(`H(WE@_?_ZP0`H.$H(*#C!1"@X4;W_^LH
M`%#C'P``&O0PG^4!$*#C\""?Y08`H.$#,(_@!#"-Y>0PG^4"((_@"%"-Y0P@
MC>4#,(_@`'"-Y0`@X./HH,WEF/?_ZY___^H$0)KE``!4XUW__PH$`*#A0??_
MZP,`4.-9__^:I!"?Y00`0.(``(3@`1"/X*#W_^L``%#C!)"@`5'__^IC]__K
MA`"?Y0``C^!P_?_K>&T!`"`!``"D;@$`.&T!`(P:``!,&P``M!H``'0:``!0
M&0``1!H``"0!``!$&@``*&L!`$@9``"L&0``=!D``-P7```<&```F!<``/@8
M``"<&0``!!<``,09```8%P``W!@``&@6``!<%P``A&@!`%`5``#8%P``!!<`
M`+`6``"<%@``\$`MZ0'*H.,,P$W@X`^,Y5P2G^4,T$WB6"*?Y7!/'>X!$(_@
M4#*?Y5`"G^4-8*#A`B"1YP``C^`&$*#A`""2Y00@C>4`(*#C`S"?YP-0E.=D
M,-7E93#%Y7?X_^L``%#C`@``"@`PT.4Q`%/C9@``"KG[_^L(,I_E`S"?YP-0
ME.<`$I_E!0"@X0,PH.,!(*#C`1"/X,7V_^L`4%#B#```"N0QG^4,()7E`S"?
MYP`0DN78(9_E`P"4YP(@C^#!]O_K##"5Y0`0D^4(,)'E`043XUD``!JX$9_E
M`S"@XP$@H.,!$)_G`0"4YZ@1G^4!$(_@KO;_ZP!04.(+```*F`&?Y080H.$`
M`(_@2_C_ZP`@4.)!```*A#&?Y0P0E>4#,)_G`!"1Y0,`E.>F]O_K<#&?Y0$@
MH.-L$9_E;'&?Y0$0C^`#,)_G!W"/X`-0E.<%`*#AA_;_ZU0AG^4`$*#A!0"@
MX0(@C^"7]O_K1#&?Y40!G^4`(*#C!A"@X0``C^`#,)_G`S"4YV4@P^4K^/_K
M`$"@X0``4.,(`(?E$@``"@<=H.,$`*#A8_;_ZP$@@.(`,*#A$0!0XP$`4A,E
M``"*^""?Y;`PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,;```:#-"-XO"`
MO>@$`)?EQ_W_ZP!`H.$``%#C"`"'Y>;__QJX`)_E``"/X-'V_^L!,-#E``!3
MXYG__PJ4___JH#"?Y0P@E>4#,(_@``"2Y00PD^4`$)/EQ_C_Z[O__^J$,)_E
M`S"?YP,`E.>C]O_KH/__ZJKV_^MP$)_E!""@X6P`G^4!$(_@``"/X`00D>4`
M$)'EM_;_ZXAF`0`@`0``<&<!`$@4``!`9P$`P!4``!!G`0"<%0``V&8!`'`5
M```P$P``G&8!`'AF`0`D%0``Y&8!`!@5```\9@$`L!(``.1D`0!\%```!&8!
M`'AE`0#(90$`0!0``#!`+>D!RJ#C#,!-X.@/C.40(9_E#-!-X@PQG^4`4*#A
M`B"/X`,PDN<`,)/E!#"-Y0`PH..G^O_K+Q"@XP!`H.$N]O_K``!0XP%`@!($
M`*#A:?;_ZP!`H.'0`)_E#1"@X0``C^"_]__K`#!0X@H```H`,-/E,`!3XP``
M4Q,!,*`#`#"@$P$P(^(``%7C`#"@`P$P`Q(``%/C"@``&I`@G^6$,)_E`B"/
MX`,PDN<`()/E!#"=Y0(@,^``,*#C&```&@S0C>(P@+WH`##5Y0``4^/Q__\*
M7!"?Y00@H.,$`*#A`1"/X'?V_^L``%#CZO__&D0@G^4P,)_E`B"/X`,PDN<`
M()/E!#"=Y0(@,^``,*#C`P``&@4`H.$,T(WB,$"]Z/;W_^I`]O_KM&,!`"`!
M``!D$0``,&,!`*P1``#<8@$`\$\MZ0'*H.,,P$W@@`Z,Y1!,G^57WTWB#.R?
MY1`PC>($0(_@`Q"@X1P@@.)<P(#B#N"4YP#@GN54X8WE`."@XXT*8O3@`/#S
M`@!<X=T*0?3Z__\:!DO=[0@[W>T1'HWB"AO=[0```.JR$6+R""O3[30@@^(`
M"]/M"#"#X@$`4^&/5V+TL`%"\[4!0/.T`4#SLT%C\K$Q8?*@&&#R,`#A\[`!
M8?(."\/M[O__&B!`D.F9.0?C@CI%XPQ`D.40P)WE`&"0Y000+N`0<)#E`\",
MX`40`>!ED:#A!\",X`00(>`4()WEYLV,X`'`C.`)$"[@`R""X`80`>!F@:#A
M!"""X!B@G>4.$"'@`A"!X`@@*>`#H(K@#"`"X`Z@BN`)("+@"B""X!R@G>7L
M'8'@;,&@X0.@BN`)H(K@#)`HX`&0">#A+8+@")`IX&$1H.$*D(G@(*"=Y>*=
MB>`#H(K@"*"*X`&`+.`"@`C@8B&@X0R`*.`"L"'@"H"(X"2@G>4)L`O@Z8V(
MX`.@BN`,H(K@:<&@X0&0*^`,L"+@"I")X"B@G>4(L`O@Z)V)X`.@BN`!H(K@
M:!&@X0*`*^`!L"S@"H"(X"R@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"
ML"'@"I")X#"@G>4(L`O@Z)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@"H"(X#2@
MG>4)L`O@Z8V(X`.@BN`!H(K@:1&@X0*0*^`!L"S@"I")X#B@G>4(L`O@Z)V)
MX`.@BN`"H(K@:"&@X0R`*^`"L"'@"H"(X#R@G>4)L`O@Z8V(X`.@BN`,H(K@
M:<&@X0&0*^`,L"+@"I")X$"@G>4(L`O@Z)V)X`.@BN`!H(K@:!&@X0*`*^`!
ML"S@"H"(X$2@G>4)L`O@Z8V(X`.@BN`"H(K@:2&@X0R0*^`"L"'@"I")X$B@
MG>4(L`O@Z)V)X`.@BN`,H(K@:,&@X0&`*^`,L"+@"H"(X$R@G>4)L`O@Z8V(
MX`.@BN`!H(K@:1&@X0*0*^`!L"S@"I")X%"@G>4(L`O@Z)V)X`.@BN`"H(K@
M:"&@X0R`*^`*@(C@5*"=Y>F-B.`#H(K@#,"*X`*@(>`)H`K@:9&@X0&@*N`,
MH(K@6,"=Y>BMBN`#P(S@`<",X`D0(N`($`'@:(&@X0(0(>`,$('@7,"=Y>H=
M@>`#,(S@",`IX`K`#.`",(/@"<`LX&`@G>4#P(S@H3L.X]D^1N-JH:#A`R""
MX.'-C.`)D(+@9""=Y0,@@N`(((+@"H`HX`&`*.`(@(G@89&@X6@0G>7LC8C@
M`Q"!X`H0@>`)H"K@#*`JX&S!H.$*((+@Z*V"X&P@G>4#((+@"2""X`R0*>`(
MD"G@:(&@X0D0@>#JG8'@<!"=Y0,0@>`,$('@",`LX`K`+.!JH:#A#"""X.G-
M@N!T()WE`R""X`@@@N`*@"C@"8`HX&F1H.$($('@[(V!X'@0G>4#$('@"A"!
MX`F@*N`,H"K@;,&@X0H@@N#HK8+@?""=Y0,@@N`)((+@#)`IX`B0*>!H@:#A
M"1"!X.J=@>"`$)WE`Q"!X`P0@>`(P"S@"L`LX&JAH.$,((+@Z<V"X(0@G>4#
M((+@""""X`J`*.`)@"C@:9&@X0@0@>#LC8'@B!"=Y0,0@>`*$('@":`JX`R@
M*N!LP:#A"B""X.BM@N",()WE`R""X&JQH.$)((+@#)`IX`B0*>!H@:#A"1"!
MX.J=@>"0$)WE`Q"!X`P0@>`(P"S@"L`LX`P@@N#IK8+@E""=Y0,@@N`(((+@
M"X`HX`G`*.!ID:#A#!"!X)C`G>7JC8'@"1`KX`/`C.`*$"'@"\",X)RPG>5J
MH:#A`2""X`.PB^#H'8+@";"+X`J0*>`(("G@:(&@X0+`C."@()WEX9V,X`,@
M@N`*((+@"*`JX`'`*N"DH)WE81&@X0S`B^`#H(K@Z<V,X`B@BN`!@"C@"8`H
MX&F1H.$(((+@L+"=Y>R-@N"H()WE`R""X`$@@N`)$"'@#!`AX&S!H.$!H(K@
M#!`IX.BMBN`($"'@`2""X&B!H.'J'8+@K""=Y0,@@N`(,"S@"C`CX`D@@N`#
M((+@W#P+XQL_2.-JD:#A`["+X`F@B.$,L(O@M,"=Y>$M@N`!H`K@`\",X&$1
MH.$(P(S@"8`(X`B@BN$!@(GA"Z"*X`*`".`!L`G@XJV*X`N`B.&\L)WE#("(
MX+C`G>5B(:#A`["+X`/`C.`!L(O@"9",X`+`@>$*P`S@`A`!X`'`C.'JC8C@
M"<",X,"0G>5JH:#AZ,V,X`H0@N$#D(G@`I")X`@0`>`*(`+@:(&@X0(0@>$(
M(`K@"["!X`@0BN$,$`'@[+V+X`(0@>%LP:#A"1"!X,20G>4,((CAZQV!X`.0
MB>`+(`+@"I")X,B@G>5KL:#A`Z"*X`B@BN`,@`C@""""X6&!H.$)D(+@"R",
MX>&=B>`!(`+@S!"=Y0,0@>`,$('@"\`,X`S`@N$(((OA"L",X`D@`N`(H`O@
MZ<V,X`H@@N'0H)WE:9&@X0$0@N`#H(K@"2"(X0N@BN#4L)WE#"`"X.P=@>`#
ML(O@;,&@X0BPB^`)@`C@""""X6&!H.$*H(+@#"")X>&MBN`!(`+@V!"=Y0,0
M@>`)$('@#)`)X`F0@N$(((SA"Y")X`H@`N`(L`S@ZIV)X`L@@N'@L)WE`1""
MX-P@G>5JH:#A`["+X`,@@N`(L(O@#,""X`H@B.$)(`+@"H`(X.D=@>`(@(+A
M:9&@X0R`B.`)((KA8<&@X>&-B.`!(`+@Y!"=Y0,0@>`*$('@":`*X`J@@N$,
M((GA"Z"*X`@@`N`,L`G@Z*V*X`L@@N'LL)WE`1""X.@@G>5H@:#A`["+X`,@
M@N`,L(O@"9""X`@@C.$*(`+@",`,X.H=@>`,((+A:J&@X0F0@N`*((CA8<&@
MX>&=B>`!(`+@\!"=Y0,0@>`($('@"H`(X`B`@N$,((KA"X"(X`D@`N`,L`K@
MZ8V(X`L@@N'XL)WE`2""X/00G>5ID:#A`["+X`,0@>`,L(O@"J"!X`D0C.$(
M$`'@"<`,X`S`@>'H+8+@"L",X/R@G>5H@:#AXLV,X`@0B>$#,(K@"3"#X`(0
M`>`(D`G@8B&@X0*@B.$)$('A"Q"!X`R@"N`"L`C@[!V!X&R1H.$+H(KA`,&=
MY0.@BN#6,0SC8CI,XP/`C.#AK8K@"(",X`3!G>5AL:#A`\",X`+`C.`)("+@
M`2`BX`@1G>4"@(C@"R`IX.J-B.`*("+@`Q"!X&JAH.$)$('@`L",X`J0*^#H
MS8S@")`IX`PAG>4)$('@:(&@X>R=@>`0$9WE`R""X`,0@>`+((+@"A"!X`B@
M*N`,H"K@;,&@X0H@@N#IK8+@%"&=Y0,@@N`(((+@#(`HX`F`*.!ID:#A"!"!
MX.J-@>`8$9WE`Q"!X`P0@>`)P"S@"L`LX&JAH.$,((+@Z,V"X!PAG>4#((+@
M"2""X`J0*>`(D"G@:(&@X0D0@>#LG8'@(!&=Y0,0@>`*$('@"*`JX`R@*N!L
MP:#A"B""X.FM@N`D(9WE`R""X`@@@N`,@"C@"8`HX&F1H.$($('@ZHV!X"@1
MG>4#$('@#!"!X`G`+.`*P"S@:J&@X0P@@N#HS8+@+"&=Y0,@@N`)((+@"I`I
MX`B0*>!H@:#A"1"!X.R=@>`P$9WE`Q"!X`H0@>`(H"K@#*`JX&S!H.$*((+@
MZ:V"X#0AG>4#((+@""""X`R`*.`)@"C@:9&@X0@0@>#JC8'@.!&=Y0,0@>`,
M$('@"<`LX`K`+.!JH:#A#"""X.C-@N`\(9WE`R""X`D@@N`*D"G@")`IX&B!
MH.$)$('@0)&=Y>P=@>`#D(G@"I")X`B@*N`,H"K@;,&@X0H@@N!$H9WEX2V"
MX`.@BN`(H(K@#(`HX`&`*.!A$:#A")")X$B!G>7BG8G@`X"(X`R`B.`!P"S@
M`L`LX&(AH.$,P(K@`G"'X.G-C.`0<(#E;.&.X`C@C>5,X9WE`S".X`8P@^`!
M,(/@`A`AX`D0(>!ID:#A`1"(X`D@(N#L'8'@#"`BX`(P@^`)0(3@X3V#X`Q`
MC>4`,(WE`5"%X#P@G^4$4(WEWPIM]`(@C^`H,)_ECPI`]`,PDN<`()/E5#&=
MY0(@,^``,*#C`0``&E??C>+PC[WH,//_ZW1B`0`@`0``G%8!`!!`+>D!RJ#C
M#,!-X/@/C.5@`*#C)//_ZP<+W^T(*]_M"3O?[0`@H.,$"\#M&""`Y5P@@.7?
M*D#T$("]Z/#ATL,``````2-%9XFKS>_^W+J8=E0R$/!'+>D!RJ#C#,!-X.`/
MC.44,)#E`F"@X8(AH.$`4*#A`C"3X!@@D.44,(#E`4"@X5P`D.4!(((BIBZ"
MX!R`A>(``%#C&""%Y2```!H_`%;C+@``VD"01N(I8Z#A`7"&X@=SA.`$$*#A
M"#"@X4!`A.(`X)'E$!"!X@S`$>40,(/B"``1Y00@$>4$`%'A$.`#Y0S``^4(
M``/E!"`#Y?/__QH!0*#A!0"@X;+\_^L'`%3AZ___&@9C2>`&(*#A!Q"@X0@`
MH.&_\O_K7&"%Y?"'O>A`<&#B``"(X`8`5^$&<*"A!R"@X;?R_^M<,)7E`S"'
MX%PPA>5``%/C\(>]&`4`H.$'8$;@!T"$X)K\_^O.___J!'"@X>?__^IP0"WI
M`<J@XP*++>T,P$W@Z`^,Y0%`H.$4$)'E`%"@X1Q@A.(Q/[_F%!"4Y1@`E.71
M$>7G,"^_YA@K0^P!(('B?S#@XS@`4N,"`(;@`3#&YT$``-I`(&+B`!"@XVGR
M_^L$`*#A??S_ZRPPA.)0`,#R#PI&]`\*0_0\,(3B#PI#]$PPA.(/!T/T5#"$
MX@0`H.$/AP/T<?S_ZP,PU.4`,,7E!`"@X;(PU.$!,,7E`#"4Y2,TH.$",,7E
M`#"4Y0,PQ>4',-3E!##%Y;8PU.$%,,7E!#"4Y2,TH.$&,,7E!#"4Y0<PQ>4+
M,-3E"##%Y;HPU.$),,7E"#"4Y2,TH.$*,,7E"#"4Y0LPQ>4/,-3E###%Y;XP
MU.$-,,7E##"4Y2,TH.$.,,7E##"4Y0\PQ>43,-3E$##%Y;(QU.$1,,7E$#"4
MY2,TH.$2,,7E$#"4Y1,PQ>4"B[WL<$"]Z#GR_^HX(&+B`!"@XR?R_^O&___J
M"#"?Y0,PC^``,)/EA?+_ZKQ4`0`(0"WI"("]Z`$``@`]````0`````L```!`
M````$````$`````F````0````#4```!``````P```$`````&````0````!0`
M``!`````#@```$`````-````0````"4```!`````$@```$`````H````0```
M``(```!`````&````$`````)````0`````H```!````````````````E<R\E
M<P```"]P<F]C+W-E;&8O97AE``!$>6YA3&]A9&5R.CIB;V]T7T1Y;F%,;V%D
M97(`26YT97)N86QS.CI005(Z.D)/3U0`````+W!R;V,O)6DO)7,`97AE`"X`
M``!005)?5$5-4`````!005)?4%)/1TY!344`````.@```"5S)7,E<P``+P``
M`%!%4DPU3$E"`````%!%4DQ,24(`4$523#5/4%0`````4$523$E/``!005)?
M24Y)5$E!3$E:140`4$%27U-005=.140`4$%27T-,14%.````4$%27T1%0E5'
M````4$%27T-!0TA%````4$%27T=,3T)!3%]$14)51P````!005)?1TQ/0D%,
M7U1-4$1)4@```%!!4E]435!$25(``%!!4E]'3$]"04Q?5$5-4`!005)?1TQ/
M0D%,7T-,14%.`````#$```!L<V5E:R!F86EL960`````<F5A9"!F86EL960`
M3$1?3$E"4D%265]0051(`%-94U1%30``)3`R>``````E<R5S)7,E<P````!P
M87(M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U8F1I<F5C=&]R>2`E
M<R!F86EL960@*&5R<FYO/25I*0H``"5S.B!P<FEV871E('-U8F1I<F5C=&]R
M>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@86YD(')E=')Y('EO
M=7(@;W!E<F%T:6]N*0H``%!!5$@`````<&%R;``````N<&%R`````'-H;W)T
M(')E860``"5S)7-C86-H92TE<R5S```E<R5S=&5M<"TE=25S````)7,E<W1E
M;7`M)74M)74E<P`````8````<&5R;``````P````4$%2.CI086-K97(Z.E9%
M4E-)3TX`````,2XP-3D```!5;F%B;&4@=&\@8W)E871E(&-A8VAE(&1I<F5C
M=&]R>0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C86-H92!S=6)D:7)E
M8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`````"UE```M+0``55-%4@``
M``!54T523D%-10````!414U01$E2`%1-4``O=&UP`````&5V86P@)V5X96,@
M+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII9B`P.R`C(&YO="!R
M=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?7W!A<E]P;#L*;7D@
M*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D8V%C:&5?;F%M95]S
M:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2+G!M7&XB.PHD1DE,
M15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*)&-A8VAE7VYA;65?
M<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S7#`P-"(["GT*<W5B
M(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*;7D@)&-H=6YK7W-I
M>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE(#T@+7,@)&9H.PIM
M>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@)&-H=6YK7W-I>F4[
M("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@;V8@)&-H=6YK7W-I
M>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@)'!O<RP@,#L*<F5A
M9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H*"1005)?34%'24,I
M.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?34%'24,I*2`^/2`P
M*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D8VAU;FM?<VEZ93L*
M?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R;V=N86UE+"!`=&UP
M9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%3E9[4$%27T-,14%.
M?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL93HZ0F%S96YA
M;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I<B`]($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H<7%;4F5M;W9I;F<@
M9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN9#HZ9FEN9&1E<'1H
M*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL("1P87)?=&5M<"D[
M"G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M<"`F)B`D7D\@;F4@
M)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ5&5M<"@*5$5-4$Q!
M5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A<V5N86UE.CID:7)N
M86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I53DQ)3DL@/3X@,"P*
M*3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64["G!R:6YT("1T;7`@
M/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@73L@9&\*<FT@+7)F
M("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M<"<@73L@=&AE;@IB
M<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*9&]N90IR;2`G)&9I
M;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P,"P@)&9I;&5N86UE
M.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL(#(^)C$@)B(["G-Y
M<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG<F]U;F0@<')O8V5S
M<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI.PI]"GT@?0I"14=)
M3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F:6YE9"`F26YT97)N
M86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?96YV*"D["FUY("1Q
M=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@/2`H"G!A=&A?<V5P
M("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G.B<I+`I?97AE("`@
M("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y9W=I;BDO(#\@)RYE
M>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^("]>35-7:6Y\3U,R
M+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE*"D["F]U=',H<7%;
M7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?<&%R7W1E;7`H*3L*
M;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM005)?5$5-4'TB72D[
M"FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*3L*34%'
M24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@>WT["G5N;&5S<R`H
M;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO=71S*'%Q6T-A;B=T
M(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C(&1O;B=T('5S92`D
M(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA<W0@34%'24,["GT*
M;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?1D@I.PII9B`H)&UA
M9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD(&UA9VEC('-T<FEN
M9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE(ETI.PIL87-T($U!
M1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P;W-I=&EO;B`D;6%G
M:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T
M7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A9"!?1D@L("1B=68L
M(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B7#!#04-(12(@)B8@
M)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C:&4@;6%R:V5R(&9O
M=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H92!M87)K97(@(B1B
M=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F
M<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,15]O9F9S971?<VEZ
M93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I.PIO=71S*")/9F9S
M970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T(BD["G-E96L@7T9(
M+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@+2`D;V9F<V5T+"`P
M.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?1D@["FUY("5R97%U
M:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@("`@("`@("`@("`@
M("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B"G=H:6QE("@D8G5F
M(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*<F5A9"!?1D@L("1B
M=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA;64@/2`D8G5F.PIO
M=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B+BXN72D["FUY("1C
M<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX?2DO?'P@*2`_("0Q
M(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@*"1B=68@/7X@;7PH
M/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B=68L(#0["G)E860@
M7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@*&1E9FEN960H)&5X
M="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI)"]I*2!["FUY("1F
M:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B=68L(#`W-34I.PHD
M4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@/2`D9FEL96YA;64[
M"B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE?2`]("1F=6QL;F%M
M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M97T@/2!["F)U9B`]
M/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L;&YA;64L"GT["B1R
M97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@7T9(+"`D8G5F+"`T
M.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF+"`D;6]D=6QE*2`]
M($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M;V1U;&4["FUY("1I
M;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U<FX@=6YL97-S("1I
M;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE?3L*)$E.0WLD;6]D
M=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII9B`H)$5.5GM005)?
M0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W*2D@>PIM>2`D9F@@
M/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#86XG="!C<F5A=&4@
M=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD9F@M/G!R:6YT*"1I
M;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T=7)N("1F:#L*?0IE
M;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I;F9O+3Y[8W)C?2YP
M;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@/2`D9FEL96YA;64[
M"F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N("1F:#L*?0ID:64@
M(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD(&UO9'5L92`D;6]D
M=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E(%!!
M4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17AP;W)T
M97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["G)E
M<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA;64@/2`H<V]R="!K
M97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H)$E.0WLD9FEL96YA
M;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII9B`H)&9I;&5N86UE
M(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA;64["GT*96QS92![
M"F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^("]S:71E8W5S=&]M
M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI<W1[)&9I;&5N86UE
M?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*;W5T<RAQ<5M.;R!Z
M:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I;&4@(B1P<F]G;F%M
M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]("AT96QL(%]&2"D@
M+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II<`I]"FUY($!P87)?
M87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@)&-A8VAE7VYA;64I
M.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S86YI=&EZ92`H4D55
M4T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@*"$D<W1A<G1?<&]S
M(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@)B8@<VAI9G0I*2![
M"FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P87(*:2`@(&EN<W1A
M;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N7W!A<@IV("`@=F5R
M:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=(&5Q("<M+7)E=7-E
M)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%55-%?2`]('-H:69T
M($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E:&%V:6]U<@IM>2!`
M861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=66S!=(#U^("]>+2A;
M04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF("@D,2!E<2`G22<I
M('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@*"0Q(&5Q("=-)RD@
M>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G02<I('L*=6YS:&EF
M="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G3R<I('L*)&]U="`]
M("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L92`]("=S:71E)SL*
M?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G86QL)SL*?0IE;'-I
M9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS:68@*"0Q(&5Q("=,
M)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE('%Q6T-A;B=T(&]P
M96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D,2!E<2`G5"<I('L*
M)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD["FEF("AM>2`D8VUD
M(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG4$%27U1%35`G?3L*
M:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[(E!!4CHZ1&ES=#HZ
M)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@D
M7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`24Y#+"!`861D7W1O
M7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q
M=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T.CI32$$["GT*;7D@
M)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@*&1E9FEN960@)'!A
M<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O<B!D:64@<7%;0V%N
M)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE<W,H)&9H+"`G24\Z
M.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH("1Z:7`M/G)E
M861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R8VAI=F4Z.EII<#HZ
M05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E
M("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII9B`H)'II<"!A;F0@
M;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY;6PG*2D@>PHD;65T
M82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>7%,N*B\O;7,["B1M
M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I].PII9B`H9&5F:6YE
M9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D<&%R(&]R(&1I92!Q
M<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=.PIM>2`D8G5F.PIR
M96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I<R!N;W0@82!P87(@
M9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#.PIC;&]S92`D<&@[
M"GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/.CI&:6QE+3YN97<H
M"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ1FEL93HZ3U]21%=2
M*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI(&]R(&1I92!Q<5M#
M86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H+3YB:6YM;V1E*"D[
M"G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H9&5F:6YE9"`D<W1A
M<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D;&]A9&5R+"`D<W1A
M<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A;"`D+R`]('5N9&5F
M.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!4E]615)"051)37T@
M86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2!["G)E<75I<F4@4$%2
M.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P
M<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D;&]A9&5R*0IO<B!D
M:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O=70B.B`D(5T["FEF
M("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*4$%2.CI(96%V>3HZ
M7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR97%U:7)E7VUO9'5L
M97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@?0IG<F5P('L**"1B
M=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I9SHZ0V]N9FEG>V%R
M8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF:6=[<')I=FQI8F5X
M<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R96%C:"!`:6YC.PIM
M>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,;V%D97(Z.F1L7W-H
M87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI8E]E>'0@/2`D0V]N
M9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!86%@@;&EB7V5X="!V
M<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY("1K97D@*'-O<G0@
M:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D["FEF("AD969I;F5D
M*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&ME>7TI*2!["B@D
M9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS92!["F9O<F5A8V@@
M;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<421D:7)<12\H+BHI
M)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q*3L*;&%S=#L*?0II
M9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD?"D@>PII9B`H;7D@
M)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE+"`D;F%M92D@/2`H
M)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D,2(I('L**"1F:6QE
M+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0["GT*?0I]"GT*;F5X
M="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D=W)I='1E;GLD;F%M
M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I;&4@/7X@+UPN7%$D
M;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C:VEN9R!&24Q%("(E
M<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65](#H@)&9I;&4I*3L*
M;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C;VYT96YT(#T@)&9I
M;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N9&5F.PIO<&5N(&UY
M("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A;B=T(')E860@(B1F
M:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S92`D=&@["E!!4CHZ
M1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT+"`B/&5M8F5D9&5D
M/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@86YD("1N86UE(#U^
M("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z.E!A=&-H0V]N=&5N
M="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D;F%M92D["GT*)&9H
M+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H)&YA;64I("L@.2DL
M"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z.F-O;7!U=&5#4D,S
M,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE;F=T:"@D8V]N=&5N
M="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@96UB961D
M960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7<FET=&5N(&%S("(D
M;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R:71E5&]&:6QE2&%N
M9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO<B!D:64@<7%;17)R
M;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]"FEF("@D;65T85]P
M87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB*3L*?0IE;'-E('L*
M:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C='@@/2!$:6=E<W0Z
M.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T>"T^861D9FEL92@D
M9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@/2`D8W1X+3YH97AD
M:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A8VAE7VYA;64L(#`L
M("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X](")<,"(@>"`H)&-A
M8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M92D["B1F:"T^<')I
M;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@)&]F9G-E="`]("1F
M:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP<FEN="AP86-K*"=.
M)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L;W-E(&]R(&1I92!Q
M<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B.B`D(5T["F-H;6]D
M(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S<R!D969I;F5D("1S
M=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER92!005(["E!!4CHZ
M2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U:7)E($9I;&4Z.D9I
M;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F:"`]($E/.CI&:6QE
M+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@("`C($%R8VAI=F4Z
M.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F:"T^9F1O<&5N*&9I
M;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH*2!F86EL960Z("0A
M73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@7T9(*3L*;7D@)'II
M<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A9$9R;VU&:6QE2&%N
M9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI::7`Z.D%:7T]+*"DI
M"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H:79E("(D<')O9VYA
M;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H-C0@*B`Q,#(T*3L*
M<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ3&EB0V%C:&5[)'!R
M;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@>FEP+BXN(BD["FEF
M("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO=6QD(&)E('-E="!A
M="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@*"`D>FEP+3YM96UB
M97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T;W)Y.PIM>2`D;65M
M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X="!U;FQE<W,@)&UE
M;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O;F9I9SHZ0V]N9FEG
M>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E>'1R86-T7VYA;64@
M/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M/F-A=&9I;&4H)$5.
M5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H+68@)&1E<W1?;F%M
M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E9%-I>F4H*2D@>PIO
M=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I;F-E(&ET(&%L<F5A
M9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L<V4@>PIO=71S*'%Q
M6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D97-T7VYA;64B+BXN
M72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H)&1E<W1?;F%M92D@
M/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@97AT
M<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE(ET["F-H;6]D*#`U
M-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"(["GT*?0I]"GT*=6YL
M97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2!["F1I92`\/"`B+B(@
M=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP87(@72!;("U)9&ER
M(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O9W)A;2YP;"!="B0P
M(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*)$5.5GM005)?4%)/
M1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`05)'5BD["GT*<W5B
M($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E<75I<F4@1FEL93HZ
M0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D97AT*2`]($9I;&4Z
M.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN*B<I*3L*<F5Q=6ER
M92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H)'!A=&@I('5N;&5S
M<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E<G)O<@I]"G-U8B!R
M97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q=6ER92!$>6YA3&]A
M9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R:6-T.PIR97%U:7)E
M('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@0V%R<#L*<F5Q=6ER
M92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q=6ER92!%>'!O<G1E
M<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I<F4@1F-N=&P["G)E
M<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3<&5C.PIR97%U:7)E
M(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER92!)3SHZ2&%N9&QE
M.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R97-S.CI:;&EB.PIR
M97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E<W0Z.E-(03L*<F5Q
M=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q=6ER92!005(Z.D1I
M<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PIR97%U:7)E(%!!
M4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@871T<FEB=71E<SL*
M979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U:7)E(%=I;C,R('T[
M"F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE=F%L('L@<F5Q=6ER
M92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![(')E<75I<F4@5&EE
M.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E<75I<F4@4&5R;$E/
M.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@>R!R97%U:7)E('5T
M9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD969I;F5D("1%3E9[
M4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O*"XK*2\I('L*)'!A
M<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY("1P871H("@**&UA
M<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2(%1%35!$25(@5$5-
M4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI('L*;F5X="!U;FQE
M<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD("UW("1P871H.PIM
M>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P=W5I9"D@/2!G971P
M=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E9FEN960H)E=I;C,R
M.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN,S(Z.DQO9VEN3F%M
M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S97)N86UE(#T@)'!W
M=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U5315).04U%?2!\?"`D
M14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP9&ER(#T@(B1P871H
M)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U<V5R;F%M92D["FUK
M9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER.PII9B`H)$5.5GM0
M05)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["GT*96QS92![
M"F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM>2`D;6%G:6-?<&]S
M(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS965K("1F:"P@)&UA
M9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N9W1H*")<,$-!0TA%
M(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L96YG=&@H(EPP0T%#
M2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E96L@)&9H+"`D;6%G
M:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG=&@H(EPP0T%#2$4B
M*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H+"`D8G5F+"`D8V%C
M:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C86-H95]D:7(@/2`B
M8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@*"$D8V%C:&5?9&ER
M*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[4$%27T-,14%.?2`]
M(#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C86-H95]D:7(B.PIM
M:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-4'T@/2`D<W1M<&1I
M<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6>U!!4E]414U0?2!A
M;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U8B!?<V%V95]A<R![
M"FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`7SL*;7D@)&9U;&QN
M86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H+64@)&9U;&QN86UE
M*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B.PIM>2`D9F@["BAO
M<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H<')I;G0@)&9H("1C
M;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q6T5R<F]R('=R:71I
M;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L("1T96UP;F%M92!I
M9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE+"`D9G5L;&YA;64I
M(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D9G5L;&YA;64["GT*
M<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D14Y6>U!!4E]04D]'
M3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O*"XK*2\I('L*)'!R
M;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*:68@*"1%3E9[4$%2
M7U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[4$%27U1%35!]*2`^
M/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA;64L(')I;F1E>"@D
M<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II9B`H(21%3E9[4$%2
M7U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@
M/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R;V=N86UE*2!["G)E
M='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA;64D<WES>U]E>&5]
M(BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T=7)N.PI]"GT*9F]R
M96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?<V5P?5Q%+RP@)$5.
M5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!4E]414U0?2!A;F0@
M)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S+UQ1)'-Y<WM?9&5L
M:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O
M9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D9&ER)'-Y<WM?9&5L
M:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N86UE(#T@(B1D:7(D
M<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@+7,@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?<')O9VYA;64@>PHD
M,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!345].PII9B`H:6YD
M97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@>PHD<')O9VYA;64@
M/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@)'!W9"`]("AD969I
M;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I"CH@*"AD969I;F5D
M("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H*2`Z(&!P=V1@*3L*
M8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<+EPN/UQ1)'-Y<WM?
M9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6>U!!4E]04D]'3D%-
M17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E;G8@>PII9B`H("1%
M3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E='5R;CL*?2!E;'-E
M('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF;W(@*'%W*"!34$%7
M3D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!344@*2`I('L*9&5L
M971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-4$1)4B!414U0($-,
M14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D14Y6>R)005)?1TQ/
M0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"04Q?)%\B?3L*?0II
M9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[4$%27T-,14%.?3L*
M?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*:68@*"1L;V=F:"D@
M>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP<FEN="`B0%]<;B([
M"GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I9SL*<'5S:"!`24Y#
M+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF:6=[)%]]+"!Q=R@*
M87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP('-I=&5L:6)E>'`*
M=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP86-K86=E(&UA:6X[
M"G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2.CIF:6YD7W!A<CL*
M4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A<BYP;#H@0V%N)W0@
M;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@<W5C:"!F:6QE(&]R
M(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE.PID;R`D<')O9VYA
M;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+7T58251?7"@H7&0K
M*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/4B`]("1`(&EF("1`
M.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2(#U^+UY?5$M?15A)
M5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@)#HZ7U]%4E)/4CL*
M,3L*7U]%3D1?7PH`<&5R;'AS:2YC`````$-!0TA%```]````0`````L```!`
M````$````$`````F````0````#4```!``````P```$`````&````0````!0`
M``!`````#@```$`````-````0````"4```!`````$@```$`````H````0```
M``(```!`````&````$`````)````0`````H```!```````````````"!L@&!
ML+"L!P````"$`@&!L`^Q``````"%L@&!L+"N?P````"\B/]_L*H)@&B+_W\!
M````;([_?["K'H#HC_]_P/__?QB1_W\!````5)/_?["H`X#XD_]_`0```+"?
M_W^L__]_'*#_?[#__W_8H?]_L+"H@/2A_W\!````8*/_?["O/(!(JO]_L*L"
M@!2M_W\!````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````!4
M&@``X!D```$```!'`P```0```%(#```!````7`,```$```!P`P``#P```/@$
M```,````-!,```T```!02P``&0```*R=`0`;````!````!H```"PG0$`'```
M``0```#U_O]OM`$```4````("@``!@```$@#```*````&P4```L````0````
M%0`````````#````S)X!``(```#H`0``%````!$````7````3!$``!$```!\
M$```$@```-`````3````"````!@`````````^___;P$```C^__]O_`\``/__
M_V\#````\/__;R0/``#Z__]O#@``````````````````````````````````
M``````````````````````"TG0$```````````!`$P``0!,``$`3``!`$P``
M0!,``$`3``!`$P``0!,``$`3``!`$P``0!,``$`3``!`$P``0!,``$`3``!`
M$P``0!,``$`3``!`$P``0!,``$`3``!`$P``0!,``$`3``!`$P``0!,``$`3
M``!`$P``0!,``$`3``!`$P``0!,``$`3``!`$P``0!,``$`3``!`$P``0!,`
M`$`3``!`$P``0!,``$`3``!`$P``0!,``$`3``!`$P``0!,``$`3``!`$P``
M0!,``$`3``!`$P``0!,``$`3``!`$P``0!,``$`3``!`$P``0!,``$`3``!`
M$P``````````````````````````````````````````````````````````
M````````,!8````````$H`$`"E!!4BYP;0H`````-$\``$A,````````"$T`
M``Q-```H3P``4$P``#!/````````%$\``!Q/````````/$\``$=#0SH@*$=.
M52D@,3(N,2XP`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(U+C`N,CA?8C@X9#,S
M."D@,3(N,2XP`$$T````865A8FD``2H````%-RU!``8*!T$(`0D""@,,`1($
M$P$4`14!%P,8`1D!&@(<`2(!`"YS:'-T<G1A8@`N:6YT97)P`"YN;W1E+F=N
M=2YB=6EL9"UI9``N;F]T92Y!0DDM=&%G`"YG;G4N:&%S:``N9'EN<WEM`"YD
M>6YS='(`+F=N=2YV97)S:6]N`"YG;G4N=F5R<VEO;E]R`"YR96PN9'EN`"YR
M96PN<&QT`"YI;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+D%232YE>'1A8@`N
M05)-+F5X:61X`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A<G)A>0`N
M9'EN86UI8P`N9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+D%232YA='1R:6)U
M=&5S```````````````````````````````````````````````````````+
M`````0````(```!4`0``5`$``!D```````````````$`````````$P````<`
M```"````<`$``'`!```D```````````````$`````````"8````'`````@``
M`)0!``"4`0``(```````````````!``````````T````]O__;P(```"T`0``
MM`$``)0!```%``````````0````$````/@````L````"````2`,``$@#``#`
M!@``!@````,````$````$````$8````#`````@````@*```("@``&P4`````
M`````````0````````!.````____;P(````D#P``)`\``-@````%````````
M``(````"````6P```/[__V\"````_`\``/P/``"`````!@````,````$````
M`````&H````)`````@```'P0``!\$```T`````4`````````!`````@```!S
M````"0```$(```!,$0``3!$``.@!```%````%@````0````(````?`````$`
M```&````-!,``#03```,```````````````$`````````'<````!````!@``
M`$`3``!`$P``\`(`````````````!`````0```""`````0````8````P%@``
M,!8``"`U``````````````@`````````B`````$````&````4$L``%!+```(
M```````````````$`````````(X````!`````@```%A+``!82P``^$$`````
M````````!`````````"6`````0````(```!0C0``4(T``"0`````````````
M``0`````````H0````$``'""````=(T``'2-``!P````#0`````````$````
M`````*P````!`````@```.2-``#DC0``!```````````````!`````````"V
M````#@````,```"LG0$`K)T```0```````````````0````$````P@````\`
M```#````L)T!`+"=```$```````````````$````!````,X````&`````P``
M`+2=`0"TG0``&`$```8`````````!`````@```#7`````0````,```#,G@$`
MS)X``#0!``````````````0````$````W`````$````#`````*`!``"@``!(
M```````````````$`````````.(````(`````P```$B@`0!(H```%```````
M````````!`````````#G`````0```#``````````2*```#\`````````````
M``$````!````\`````,``'```````````(>@```U```````````````!````
L``````$````#``````````````"\H`````$``````````````0``````````
