package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.38.0 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.38.0) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````]!@``#0```"\H0````0`!30`(``)
M`"@`'``;``$``'"$C0``A(T``(2-``!P````<`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````#XC0``^(T```4```````$`
M`0```*R=``"LG0$`K)T!`)P"``"P`@``!@```````0`"````M)T``+2=`0"T
MG0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DK)T`
M`*R=`0"LG0$`5`(``%0"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`"60[_0LVSM8;D^OT]Z,7-F/MBWX$
M````$`````$```!'3E4```````,````"`````````#L```!.````"`````@`
M```!D!`$)4`$@"B```6@AE"$@($``@LH!(PX@!)!B@``"`````!.````4```
M`%$```````````````````!3`````````%0```!5````5@``````````````
M5P```%@`````````````````````````60```%H```!;`````````%P`````
M``````````````!=````7@```%\```````````````````!@````80``````
M``!C`````````&0`````````90``````````````````````````````````
M`&8```````````````````!G`````````&@```````````````````!J````
M?JF<FUGOEWEK"4/6=E0\E"'_Y`X-.I<+P^4_+(D'<-/9<5@<K4OCP&L:GE#5
M9-1\F4_#FR_%F[8S8MOM:W^:?`&O-.B[XY)\LP&+;T+ZUX=#1=7L9Z_,9A6,
M)<RCX/M)S]Y'+>-'Q@'@IP(*;\7?'[[:5QO[H6Q@````````````````````
M```````L$P````````,`"P```````*`!```````#`!<`W0(`````````````
M$@```!D!`````````````!(```!5`@`````````````2````AP(`````````
M````$@```#P"`````````````!(```!C`0`````````````2````X@``````
M````````$@```!$#`````````````!(```#X```````````````2`````0``
M````````````(````*D!`````````````!(```":```````````````2````
MPP``````````````$@```/L!`````````````!(````R`0`````````````2
M````=P(`````````````$@```!```````````````"````#M````````````
M```2````"@(`````````````$@```!X!`````````````!(```"D`0``````
M```````2````U@(`````````````$@```"P``````````````"````"``@``
M```````````2````0`$`````````````$@```((!`````````````!8```"*
M```````````````2````1@``````````````$@```+(``````````````!(`
M``"C```````````````2````+@(`````````````$@```.T!````````````
M`!(```!P`@`````````````2````]@,`````````````$@```",!````````
M`````!(```#E`@`````````````1````Y@$`````````````$@```/\"````
M`````````!$```"7`@`````````````2````(0(`````````````$@````L!
M`````````````!(````8`@`````````````2````-0(`````````````$@``
M`%<``````````````!(```"/`@`````````````2````>P$`````````````
M$@```,4"`````````````!(```#1`0`````````````2````]`$`````````
M````$@```"@"`````````````!(```!_`@`````````````2````MP$`````
M````````$@```.T"`````````````!(```#D`@`````````````A````3@$`
M````````````$@```+`!`````````````!(```!O`0`````````````2````
MO`(`````````````$@```.,"`````````````!$```!I`@`````````````1
M````;@``````````````$0```$L"`````````````!(```#3````````````
M```1````6P(`````````````$@````("`````````````!(```!C````````
M```````2````0P(`````````````$@```'L``````````````!(```"_`0``
M```````````2````*`,`````````````$@```!$"`````````````!(```#?
M`0`````````````2````MP(`````````````$@```*8"`````````````!(`
M``"5`0`````````````B````Y`,``*0O```D````$@`-`$@$``!<H`$`````
M`!``&``:!````*`!```````0`!<`)P0``$B@`0``````$``8`)P$```8(P``
MD`(``!(`#0`<!````*`!```````@`!<`0`0``*PA``!4````$@`-`*D$``!$
MH`$`!````!$`%P!<!```2*`!```````0`!@`"P0``&A+```$````$0`/`-H#
M``#@20``;`$``!(`#0"4`P``+#@``-0"```2``T`LP0``&PM``!T````$@`-
M`'8#```4(P``!````!(`#0`@!```]!@````````2``T`S`$``"`6``#4`@``
M$@`-`%$$``!<H`$``````!``&`!Q!```7*`!```````0`!@`J@,``,PL``"@
M````$@`-`#4$``#(2```&`$``!(`#0"#!```2*`!```````0`!<`S@,``-@E
M``"P`0``$@`-`&@$``!T2```5````!(`#0"[!```B"<``$0%```2``T`=@0`
M`#PQ``#P!@``$@`-`(<#``"H)0``,````!(`#0"X`P```"(``!0!```2``T`
M_P,````[``!``0``$@`-`%`$``!<H`$``````!``&`"*!```R"\``'0!```2
M``T``%]?9VUO;E]S=&%R=%]?`%])5$U?9&5R96=I<W1E<E1-0VQO;F5486)L
M90!?251-7W)E9VES=&5R5$U#;&]N951A8FQE`%!E<FQ?871F;W)K7VQO8VL`
M4&5R;%]M9U]S970`4&5R;%]N97=84P!03%]D;U]U;F1U;7``4&5R;%]S>7-?
M:6YI=#,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?8V]N<W1R=6-T
M`%!E<FQ?;F5W6%-?9FQA9W,`4&5R;%]G=E]F971C:'!V`%!,7VUE;6]R>5]W
M<F%P`'!E<FQ?<&%R<V4`<&5R;%]A;&QO8P!097)L7V%T9F]R:U]U;FQO8VL`
M<&5R;%]D97-T<G5C=`!P97)L7V9R964`4&5R;%]S=E]S971P=FX`4&5R;%]S
M=E]S971P=@!097)L7W-Y<U]T97)M`%!E<FQ?8W)O86M?;F]C;VYT97AT`%!E
M<FQ?9V5T7W-V`%!E<FQ?<V%F97-Y<VUA;&QO8P!03%]C=7)R96YT7V-O;G1E
M>'0`7U]C>&%?9FEN86QI>F4`<F5A9`!S=')C:'(`9V5T<&ED`&UE;6UO=F4`
M7U]L:6)C7W-T87)T7VUA:6X`7U]F<')I;G1F7V-H:P!S=')C;7``;W!E;C8T
M`'5N;&EN:P!S=')T;VL`;65M<V5T`'-T<FYC;7``9V5T=6ED`&UE;6-H<@!C
M;&]S961I<@!S=')S='(`<FUD:7(`<W1R;&5N`'-T<F1U<`!M96UC;7``<F5A
M;&QO8P!R96%D9&ER-C0`86)O<G0`7U]S<')I;G1F7V-H:P!S=&1E<G(`;65M
M8W!Y`'-T<G)C:'(`;'-T870V-`!O<&5N9&ER`&QS965K-C0`7U]V9G!R:6YT
M9E]C:&L`7U]E<G)N;U]L;V-A=&EO;@!E>&ET`')E861L:6YK`%]?<W1A8VM?
M8VAK7V9A:6P`86-C97-S`&UK9&ER`%]?96YV:7)O;@!?7W)E9VES=&5R7V%T
M9F]R:P!?7W-T86-K7V-H:U]G=6%R9`!?7V%E86)I7W5N=VEN9%]C<'!?<'(P
M`%]?865A8FE?=6YW:6YD7V-P<%]P<C$`;&EB<&5R;"YS;P!L:6)C+G-O+C8`
M;&0M;&EN=7@M87)M:&8N<V\N,P!L:6)G8V-?<RYS;RXQ`'!A<E]C=7)R96YT
M7V5X96,`<&%R7V)A<V5N86UE`%A37TEN=&5R;F%L<U]005)?0D]/5`!P87)?
M96YV7V-L96%N`'!A<E]C=7)R96YT7V5X96-?<')O8P!P87)?9&ER;F%M90!S
M:&%?9FEN86P`9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N
M=7``7TE/7W-T9&EN7W5S960`7U]D871A7W-T87)T`%]?8G-S7W-T87)T7U\`
M<VAA7W5P9&%T90!X<U]I;FET`%]?96YD7U\`7U]B<W-?96YD7U\`7U]B<W-?
M<W1A<G0`<VAA7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S
M971U<%]L:6)P871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7V1I90!P
M87)?:6YI=%]E;G8`1TQ)0D-?,BXT`$=#0U\S+C4`1TQ)0D-?,BXS-`!'3$E"
M0U\R+C,S`"]U<W(O;&EB+W!E<FPU+S4N,S@O8V]R95]P97)L+T-/4D4`````
M``````(``0`"``(``@`!``$``P`!``$``@`!``$``@`!``(``0`!``(``@`"
M``(``0`$``$``0`!``$``0`!``(``@`"``(``0`"``(`!0`"``(``0`"``(`
M`0`"``(``@`"``(``@`$``(``@`"``$``@`!``(``@`"``$``@`!``(``@`!
M``(``0`&``,``@`"``(``@`"``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0!4`P``$```
M`"`````4:6D-```%`,@$`````````0`!`&@#```0````(````%4F>0L```,`
MT@0````````!``,`2@,``!``````````M)&6!@``!@#:!```$````+.1E@8`
M``0`Y00``!`````4:6D-```"`,@$````````K)T!`!<```"PG0$`%P```/R?
M`0`7````!*`!`!<````4H`$`%P```!B@`0`7````(*`!`!<````DH`$`%P``
M`"B@`0`7````+*`!`!<````PH`$`%P```#B@`0`7````/*`!`!<```!$H`$`
M%P```,B?`0`5"P``S)\!`!4,``#0GP$`%1,``-2?`0`5&0``V)\!`!,<``#<
MGP$`%1T``."?`0`5'@``Y)\!`!4F``#HGP$`%2@``.R?`0`5/@``\)\!`!4_
M``#TGP$`%4$``/B?`0`530``V)X!`!8#``#<G@$`%@0``.">`0`6!0``Y)X!
M`!8&``#HG@$`%@<``.R>`0`6"```\)X!`!8)``#TG@$`%@P``/B>`0`6#0``
M_)X!`!8.````GP$`%@\```2?`0`6$```")\!`!81```,GP$`%A(``!"?`0`6
M%```%)\!`!85```8GP$`%A8``!R?`0`6%P``()\!`!88```DGP$`%AH``"B?
M`0`6&P``+)\!`!8?```PGP$`%B```#2?`0`6(0``.)\!`!8B```\GP$`%B,`
M`$"?`0`6)```1)\!`!8E``!(GP$`%B<``$R?`0`6*0``4)\!`!8J``!4GP$`
M%BL``%B?`0`6+```7)\!`!8M``!@GP$`%BX``&2?`0`6+P``:)\!`!8P``!L
MGP$`%C$``'"?`0`6,@``=)\!`!8S``!XGP$`%C0``'R?`0`6-0``@)\!`!8V
M``"$GP$`%C<``(B?`0`6.0``C)\!`!8Z``"0GP$`%CL``)2?`0`6/```F)\!
M`!9```"<GP$`%D(``*"?`0`60P``I)\!`!9$``"HGP$`%D4``*R?`0`61@``
ML)\!`!9'``"TGP$`%DD``+B?`0`62@``O)\!`!9+``#`GP$`%DP``,2?`0`6
M30``"$`MZ8`!`.L(@+WH!.`MY03@G^4.X(_@"/"^Y82+`0``QH_B&,J,XH3[
MO.4`QH_B&,J,XGS[O.4`QH_B&,J,XG3[O.4`QH_B&,J,XFS[O.4`QH_B&,J,
MXF3[O.4`QH_B&,J,XES[O.4`QH_B&,J,XE3[O.4`QH_B&,J,XDS[O.4`QH_B
M&,J,XD3[O.4`QH_B&,J,XCS[O.4`QH_B&,J,XC3[O.4`QH_B&,J,XBS[O.4`
MQH_B&,J,XB3[O.4`QH_B&,J,XAS[O.4`QH_B&,J,XA3[O.4`QH_B&,J,X@S[
MO.4`QH_B&,J,X@3[O.4`QH_B&,J,XOSZO.4`QH_B&,J,XO3ZO.4`QH_B&,J,
MXNSZO.4`QH_B&,J,XN3ZO.4`QH_B&,J,XMSZO.4`QH_B&,J,XM3ZO.4`QH_B
M&,J,XLSZO.4`QH_B&,J,XL3ZO.4`QH_B&,J,XKSZO.4`QH_B&,J,XK3ZO.4`
MQH_B&,J,XJSZO.4`QH_B&,J,XJ3ZO.4`QH_B&,J,XISZO.4`QH_B&,J,XI3Z
MO.4`QH_B&,J,XHSZO.4`QH_B&,J,XH3ZO.4`QH_B&,J,XGSZO.4`QH_B&,J,
MXG3ZO.4`QH_B&,J,XFSZO.4`QH_B&,J,XF3ZO.4`QH_B&,J,XESZO.4`QH_B
M&,J,XE3ZO.4`QH_B&,J,XDSZO.4`QH_B&,J,XD3ZO.4`QH_B&,J,XCSZO.4`
MQH_B&,J,XC3ZO.4`QH_B&,J,XBSZO.4`QH_B&,J,XB3ZO.4`QH_B&,J,XASZ
MO.4`QH_B&,J,XA3ZO.4`QH_B&,J,X@SZO.4`QH_B&,J,X@3ZO.4`QH_B&,J,
MXOSYO.4`QH_B&,J,XO3YO.4`QH_B&,J,XNSYO.4`QH_B&,J,XN3YO.4`QH_B
M&,J,XMSYO.4`QH_B&,J,XM3YO.4`QH_B&,J,XLSYO.4`QH_B&,J,XL3YO.4`
MQH_B&,J,XKSYO.4`QH_B&,J,XK3YO.4`QH_B&,J,XJSYO.4`````<$`MZ0'*
MH.,,P$W@R`^,Y63"G^4HT$WB8#*?Y0S`C^!<0I_E'`"-Y1P`C>(8$(WE&!"-
MXA0@C>44((WB!$"/X`,PG.<`,)/E)#"-Y0`PH./4___K+"*?Y2PRG^4"()3G
M#""-Y0(0H.$#,)3G$#"-Y0,`H.$L#0#K$#*?Y0,PE.<`4-/E``!5XU(```IP
M+QWN_#&?Y1P`G>4#4(#B!`"`X@,PG^<'`7#C`R"2YZ8STN4#,(/CIC/"Y6,`
M`(H``:#AH___Z\P1G^7,(9_E`#"@X0$0C^`80)WE`B"/X``0D>4$`(+E``"4
MY0@0@^6L$9_E'""=Y0$0C^`,$(/EH!&?Y0$`4N,``(/E`1"/X`00@^54``#:
M!!"$X@S`@^("083@!."1Y`3@K.4!`%3A^___&@,0@N)L09_E``"@XP``C>4%
M(*#A!$"/X`$!@^=8$9_E#`"4Y0$0C^`%___K``!0XRD```I$49_E(&"-X@50
MC^`,`)7E2?__ZP!`H.$P`9_E!A"@X0``C^"N``#K``!0XR````H<,9_E`S"/
MX`P`D^7D_O_K'/__ZPPAG^7,,)_E`B"/X`,PDN<`()/E)#"=Y0(@,^``,*#C
M*@``&@0`H.$HT(WB<("]Z/W^_^O<()_E``!0XP(@C^`,`(+E'P``"@___^O(
M,)_E<"\=[@,PG^<#,)+G=E;#Y:#__^H,`)3EX/[_Z]+__^H(`)7E``!0XP$`
M``JM"`#KV?__ZI0`G^4&$*#A``"/X((``.L``%#C"`"%Y=+__PKU___J>#"?
MY7@`G^4``(_@`S"4YPPPC>4#$*#A-/__ZP00H..P___J`0"@XU?__^L:___K
MB(@!`!P!``!HB`$`_````!0!```D`0``$(D!`$R)`0!,B0$`!#@``.PW``#@
MB`$`A`D``+B(`0`4-0``C(@!`/2&`0!`B`$`M(<!```T```H`0``X#4```"P
MH.,`X*#C!!"=Y`T@H.$$("WE!``MY1R@G^48,(_B`Z"*X``PH.,$,"WE#`"?
MY0``FN<I___KC/[_ZYR%`0`P`0``%#"?Y10@G^4#,(_@`B"3YP``4N,>_R\!
MDO[_ZH2%`0```0``+`"?Y2PPG^4``(_@*""?Y0,PC^```%/A`B"/X![_+P$8
M,)_E`S"2YP``4^,>_R\!$_\OX=R&`0#4A@$`4(4!``0!```X`)_E.#"?Y0``
MC^`T()_E`S"/X``P0^`"((_@HQ^@X4,1@>#!$+#A'O\O`1@PG^4#,)+G``!3
MXQ[_+P$3_R_AF(8!`)"&`0`,A0$`"`$``%@PG^58()_E`S"/X`(@C^``,-/E
M``!3XQ[_+Q$00"WI`<J@XPS`3>#X#XSE-#"?Y0,PDN<``%/C`@``"B@PG^4#
M`)_G]?[_Z\?__^L<,)_E`2"@XP,PC^``(,/E$("]Z$B&`0#(A`$`+`$``,R%
M`0#\A0$`S?__ZO!'+>D!RJ#C#,!-X.`/C.7(,)_E`(!0X@,PC^`K```*O""?
MY0(PD^<`D)/E``!9XR@```H`4-CE`7"@X0``5>,]`%43`5"@$P!0H`,1```:
M`$"9Y0``5.,,```*"6"@X00`H.$%(*#A"!"@X;+^_^L``%#C`@``&@4PU.<]
M`%/C"P``"@1`MN4``%3C\___&@0`H.'PA[WH"%"@X0$P]>4``%/C/0!3$_O_
M_QH(4$7@YO__Z@E@1N`!4(7B!4"$X$9AH.$$`*#A`&"'Y?"'O>@(0*#A[?__
MZ@E`H.'K___J1(0!`!@!```00"WI`<J@XPS`3>"(#XSEC""?Y7#03>*(,)_E
M#1"@X0(@C^``0*#A`S"2YP`PD^5L,(WE`#"@XVK^_^L``%#C``"@$P0``!H0
M,)WE#SH#X@H*4^,!"5,3"@``"D@@G^5`,)_E`B"/X`,PDN<`()/E;#"=Y0(@
M,^``,*#C!P``&G#0C>(0@+WH`A"@XP0`H.$/_O_K$`]OX:`"H.'N___J1/[_
MZU2#`0`<`0``$(,!`/!`+>D!RJ#C#,!-X'`/C.5<(9_E?-!-XE@QG^4`4%#B
M`B"/X`,PDN<`,)/E=#"-Y0`PH.,"```*`##5Y0``4^,*```:,"&?Y2@QG^4"
M((_@`S"2YP`@D^5T,)WE`B`SX``PH.-!```:?-"-XO"`O>B^_?_K`$!0XO'_
M_PH$`*#A0?[_ZP``4.,J```*$S#0Y1-@@.(N`%/C`@``&@$PUN4``%/C]/__
M"A,PT.4N`%/C)0``"@4`H.'H_?_K`'"@X08`H.'E_?_K``"'X`(`@.()_O_K
MI#"?Y0!PH.$!$*#C`S"/X``@X.,$8(WE`%"-Y2C^_^L($(WB!P"@X<O]_^L!
M`'#C!P"@X0,```H8,)WE#SH#X@$)4^,2```*TOW_ZP<`H.&X_?_K!`"@X1;^
M_^L``%#CU/__&@0`H.'B_?_K!0"@X?C]_^N]___J`3#6Y2X`4^/6__\:`C#6
MY0``4^/%__\*TO__ZJ3__^OK___JY/W_ZZ2"`0`<`0``>((!`!0O``#P02WI
M`<J@XPS`3>```(SE(``,Y0%PH.$`$9_E`#&?Y0':3>(!$(_@"-!-X@!@H.$!
M"HWB!`"`X@2`C>(#,)'G_R\`XP@0H.$`,)/E`#"`Y0`PH./,`)_E``"/X.;]
M_^L`0%#B!P``V@A0C>(!,$3B`\"%X`0@7.4``%+C%P``&@``4^,4```:G#"?
MY7`/'>X'(*#A!A"@X0,PG^<#`)#G;/W_ZX0@G^4!&HWB<#"?Y000@>("((_@
M`S"2YP`@D^4`,)'E`B`SX``PH.,2```:`=J-X@C0C>+P@;WH`T"@X00@H.$O
M$*#C"`"@X=K]_^L``%#CX___"C0`G^5PSQWN!#"@X0@@H.$&$*#A``"?YP``
MG.=\_?_KX?__ZIC]_^L<@0$`'`$``"`N``"T@0$`C(`!`$"!`0#P1RWI`<J@
MXPS`3>#8#XSE%"*?Y0C03>(0,I_E`4"@X0(@C^``4*#A!(*?Y0,PDN<(@(_@
M`#"3Y00PC>4`,*#C`##1Y3T`4^,!0($"!`"@X5/]_^L-$*#A`&"@X04`H.'1
M_O_K`)!0XD````I,_?_K``!6X3<``)JX,9_E`'"=Y0.`F.<'<:#A`##5Y0``
M4^,]`%,3`3"@$P`PH`,%```*!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@`@"&
MX@!@F.4#`(#@7?W_ZP``4.,'`(;G%0``"@`PF.4%`*#A!Q"3YP$PT.0`,,'E
M``!3XST`4Q,&```*`2"!X@$PT.0"$*#A`3#"Y#T`4^,``%,3^?__&@$@H.$]
M,*#C`##!Y0$PU.0!,.+E``!3X_O__QH((9_E^#"?Y0(@C^`#,)+G`""3Y00P
MG>4"(#/@`#"@XS4``!H(T(WB\(>]Z`&02>(!,-3D`3#IY0``4^/[__\:[O__
MZL`PG^4#@)CG`'"8Y0`PE^4``%/C)```"@<PH.$$(+/E"1"@X0&0B>(``%+C
M^O__&@,0@>(!$:#AD#"?Y0,PC^``()/E``!2XP\``!H!`*#A`2"@XP`@@^4<
M_?_K`*!0XM7__PH'$*#A"7&@X0<@H.$H_?_K`*"(Y0"0C>4!D(GB`#"@XPDQ
MBN>B___J!P"@X3[]_^L`H*#A``!0XP``B.7%__\*"7&@X?+__^H#D*#A"!"@
MX]___^H'_?_KZ'\!`!P!``#8?P$`&`$``-1^`0#<?P$`$$`MZ0'*H.,,P$W@
MZ`^,Y7`0G^5P3QWN;""?Y1#03>)H`)_E`B"/X&0PG^4!$)_G`S"/X``@DN<!
M`)3G5!"?Y0P@C>4!$(_@&/W_ZT@0G^5(,)_E`,"@XT0@G^4#,(_@`1"?YP(@
MC^`!`)3G-!"?Y03`C>4!$(_@`#"-Y;'\_^L0T(WB$("]Z*1^`0"@?0$`$`$`
M`(1K``#(*@``<'X!`+0M``#`%@``M"H``#@`G^4(T$WB-""?Y7#/'>XP$)_E
M`B"/X"PPG^4``)_G`S"/X`$@DN<@$)_E``"<YP$0C^`$((WE"-"-XO#\_^H(
M?@$`!'T!`!`!``#H:@``,"H``'!`+>D!RJ#C#,!-X```C.4!RDSB``",Y2@`
M#.7<()_E`MI-XM@PG^48T$WB`B"/X`(:C>(4$('B#$"-X@%JC>(#,)+G$&"&
MX@`PD^4`,('E`#"@X\/\_^NH$)_E`2`!XZ0PG^4!$(_@\`#-X0,PC^`!$*#C
M!`"@X<;\_^L$`*#A`2J@XP80H.&\_/_K`$!0XA@``+H!`(3BE_S_ZP!04.(4
M```*!""@X080H.%Q_/_K`#"@XP0PQ>=0()_E`AJ-XCPPG^44$('B`B"/X`,P
MDN<`()/E`#"1Y0(@,^``,*#C`P``&@4`H.$"VHWB&-"-XG"`O>B#_/_K`%"@
MX^W__^J8?`$`'`$``.@I``#4*0``_'L!`+G__^KP3RWI`<J@XPS`3>```(SE
MJ``,Y40BG^5"W4WB0#*?Y0303>("((_@`&"@X30"G^5!S8WB$("-XCS`C.(#
M,)+G`4"@X0``C^`($*#A`#"3Y0`PC.4`,*#CO/W_ZR\0H.,`4*#A!@"@X0G\
M_^L``%#C$@``"O`!G^4&$*#A``"/X,?^_^L&`*#AX"&?Y4$=C>+,,9_E/!"!
MX@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XVD``!I"W8WB!-"-XO"/O>@$`*#A
M/?S_ZZ01G^4!$(_@3/S_ZP!`4.+E__\*E)&?Y8"@C>(`,-3E!'!*X@F0C^``
M`%/C.P``"@``5>,$```*!!"@X04`H.%Q_/_K``!0XRL```H$`*#A"OS_ZP$P
M0.(#,(3@`+"@X0,`5.$`,(0P`!"@,P$`A#(#```Z"```Z@,`4.$`$,/E`@``
M"@$@<^4O`%+C^?__"@0`H.'Y^__K`+"@X08`H.'V^__K"P"`X`$`@.(!"E#C
MO?__*O@`G^4),*#A`2J@XP$0H.,``(_@`$"-Y4$`C>D'`*#A-_S_ZP@0H.$'
M`*#AVOO_ZP``4.,#```:8#`:Y0\Z`^(""5/C%0``"K00G^4``*#C`1"/X`S\
M_^L`0%#BI?__"@`PU.4``%/CP___&@``5>,6```*`##5Y2X`4^,#```*@$"?
MY0&PH.,$0(_@U?__Z@$PU>4``%/CZO__"O?__^H!$*#C!P"@X;C[_^L``%#C
MY/__&E``G^4'$*#A``"/X%C^_^L'`*#AW?O_ZX[__^HX0)_E`;"@XP1`C^#!
M___JX_O_ZXA[`0`<`0``^"@``,PH```4>P$`C"@``'`H``#8)P``C"<``#0G
M```0)P``Y"8``!!`+>D!RJ#C#,!-X/@/C.4O$*#C`$"@X8C[_^L``%#C`4"`
M$@0`H.$00+WHO_O_ZO!`+>D!RJ#C#,!-X```C.4@``SE>"&?Y0':3>)T,9_E
M#-!-X@(@C^`!&HWB!!"!X@!04.(#,)+G`#"3Y0`P@>4`,*#C,0``"@`PU>4`
M`%/C+@``"HO[_^L!((#B`$"@X0$*4N,Z``"*!'"-X@40H.$'`*#AB?O_ZP$P
M1.(#,(?@`P!7X0D``"H$`(?@!2"-X@`0H.,"``#J`@!0X0`0P.4"```*`3!P
MY2\`4^/Y__\*+Q"@XP<`H.%4^__K``!0XR4```H'`%#A`#"@@P`PP(4'`*#A
MB?O_Z[P@G^4!&HWBL#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,B
M```:`=J-X@S0C>+P@+WHB""?Y0$:C>)X,)_E!!"!X@(@C^`#,)+G`""3Y0`P
MD>4"(#/@`#"@XQ0``!I@`)_E``"/X`':C>(,T(WB\$"]Z&K[_^H``*#CWO__
MZD0@G^4!&HWB+#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,@`)\%
M``"/`.S__PIF^__KR'@!`!P!```$>`$`S'<!`#@E``"`=P$`\"0``'!`+>D!
MRJ#C#,!-X.@/C.7$))_E"-!-XL`TG^4-0*#A`B"/X+ADG^6X5)_E!F"/X`,P
MDN<%4(_@`#"3Y00PC>4`,*#C!!"@X08`H.&C_/_K``!0XPX```J,-)_E``"=
MY0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C
M^___&NO__^I49)_E!F"/X`00H.$&`*#AC?S_ZP``4.,.```*-#2?Y0``G>4#
M,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O_
M_QKK___J`&2?Y09@C^`$$*#A!@"@X7?\_^L``%#C#@``"MPSG^4``)WE`S"5
MYP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:
MZ___ZJQCG^4&8(_@!!"@X08`H.%A_/_K``!0XPX```J$,Y_E``"=Y0,PE><`
M$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO_
M_^I88Y_E!F"/X`00H.$&`*#A2_S_ZP``4.,.```*+#.?Y0``G>4#,)7G`!"3
MY0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J
M!&.?Y09@C^`$$*#A!@"@X37\_^L``%#C#@``"M0RG^4``)WE`S"5YP`0D^4`
M,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK!B
MG^4&8(_@!!"@X08`H.$?_/_K``!0XPX```I\,I_E``"=Y0,PE><`$)/E`#&!
MX`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I<8I_E
M!F"/X`00H.$&`*#A"?S_ZP``4.,.```*)#*?Y0``G>4#,)7G`!"3Y0`Q@>`$
M()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J"&*?Y09@
MC^`$$*#A!@"@X?/[_^L``%#C#@``"LPQG^4``)WE`S"5YP`0D^4`,8'@!""3
MY0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK1AG^4&8(_@
M!!"@X08`H.'=^__K``!0XPX```IT,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`
M(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I@89_E!F"/X`00
MH.$&`*#AQ_O_ZP``4.,.```*'#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!
MYP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J#`&?Y000H.$``(_@
MLOO_ZP`04.("```*^`"?Y0``C^#"_/_K\`"?Y000H.$``(_@J?O_ZP`04.("
M```*W`"?Y0``C^"Y_/_KU`"?Y000H.$``(_@H/O_ZP`04.(2```*P`"?Y0``
MC^"P_/_KN""?Y60PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,0```:F!"?
MY9@`G^4!$(_@``"/X`C0C>)P0+WHH/S_ZH0`G^4$$*#A``"/X(?[_^L`$%#B
MZ?__"G``G^4``(_@E_S_Z^7__^HH^O_K''<!`!P!``#,)```"'<!`!@!``!L
M)```'"0``-`C``"`(P``.",``'PB``"4(@``2"(``/PA```H(0``5"$``"@A
M``!$(0``1"$``$`A``!D(```Q'(!`!PA``"`(```["```'`@```$X"WE`<J@
MXPS`3>#P#XSE>""?Y0S03>)T,)_E#1"@X0(@C^!L`)_E``"/X`,PDN<`,)/E
M!#"-Y0`PH.-6^__K``!0XP4```H``-#E``!0XS``4!,!`*`#``"@$P$`(.(T
M()_E*#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XP$``!H,T(WB!/"=Y.KY
M_^O8<0$`'`$``-`?``"0<0$`#P`MZ0'*H.,$X"WE#,!-X/`/C.5($)_E#-!-
MXD0@G^44,(WB`1"/X#P`G^4`,(WE.,"?Y0``C^`"()'G`1"@XP`@DN4$((WE
M`""@XQ`@G>4,P)#G``"<Y;?Y_^O_`*#C"?K_ZS1Q`0`<`0``)'$!`"`!``#P
M1RWI`,"@XP'HH.,,P$W@#N!,X`'*3.(``(SE#@!<X?O__QHX``[E`=A-XAC0
M3>)TP9_E`A"@XP'HC>(,P(_@%.".X@`0C>4`(*#C7!&?Y0`PH.,`@*#A`1"<
MYP`0D>4`$([E`!"@XZCY_^L!`%#B`'#!XL<OH.$","#@`C!3X',P_^8","/@
M`C!3X`(@PN`#8%#@`G#'X```5^,X``"Z")"@XP&00.,,4(WB`!"@XP8@H.$'
M,*#A"`"@X0`0C>62^?_K`0!QXP$`<`,R```*"2"@X040H.$(`*#A5?G_ZP$`
M<.,I```*!P!0XQ\``)H(`$#B`$"5X!P``"JPH)_E"J"/X`(``.H!0$3B!0!4
MX18``#H((*#C"A"@X00`H.$=^?_K``!0X_;__QH%0$3@!@"4X,0?I^!X()_E
M`<B-XF@PG^44P(SB`B"/X`,PDN<`()/E`#"<Y0(@,^``,*#C$```&@'8C>(8
MT(WB\(>]Z`%H5N(`<,?B`0!WXP$(=@/)__\:``#@XP`0H.'I___J)`"?Y0``
MC^![___K'`"?Y0``C^!X___K7OG_ZZAP`0`<`0``('$!`)QO`0#@'0``Q!T`
M`!!`+>D!RJ#C#,!-X/@/C.41^?_K,?G_ZP``4.,``)`5$("]Z/!`+>D!RJ#C
M#,!-X-`/C.4X(9_E'-!-XC0QG^40$(WB`B"/X"QAG^4`4*#A!F"/X`,PDN<&
M`*#A`#"3Y10PC>4`,*#CE?K_ZP!`4.("```*`##4Y0``4^,.```:^""?Y>PP
MG^4"((_@`S"2YP`@D^44,)WE`B`SX``PH.,R```:V`"?Y040H.$``(_@'-"-
MXO!`O>B6^__J!1"@X1+Y_^L``%#C"@``"K0@G^6@,)_E`B"/X`,PDN<`()/E
M%#"=Y0(@,^``,*#C'P``&AS0C>+P@+WH!0"@X>[X_^L`<*#A!`"@X>OX_^L`
M`(?@`@"`X@_Y_^ML,)_E;""?Y0$0H.,#,(_@"$"-Y0(@C^``4(WE!""-Y0`@
MX.,`0*#A*_G_ZT@@G^4H,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^``,*#C!!"@
M`08`H`'/__\*^_C_Z]QN`0`<`0``;!T``)AN`0`0'0``3&X!`*P;``"@&P``
MU&T!`/!/+>D!RJ#C#,!-X.@.C.54)I_E]-!-XE`VG^6(8(WB`B"/X$C&G^4`
MH*#AH$"-X@S`C^`$X*#A`S"2YQAPC.*44(WB++:?Y0`PD^7L,(WE`#"@XPPP
MC.(PP(SB(("-X@NPC^`'`)/H!P"&Z`\`M^@/`*[H`P"7Z`,`CN@'`)SH!P"%
MZ/`%G^4($*#A``"/X"7Z_^L`<%#B`@``"@`PU^4``%/C%P$`&H7X_^NE^/_K
M``!0XWH```H`<)#E``!7XW<```H'`*#AE?C_ZX``H.$!`(#BN?C_ZP`PU^4`
M4*#A'`"-Y0``4^,+```*C)6?Y0F0C^`%`*#A`#"-Y0$0H.,),*#A`"#@XP)0
MA>+2^/_K`3#WY0``4^/U__\:8`6?Y0``C^`"``#J!`"TY0``4.,-```*"!"@
MX?OY_^L`4%#B^/__"@`PU>4``%/C]?__"B_Z_^L``%#C\O__"@4`H.&-^/_K
M`$!0X@T``!H$0);D``!4XPH```H`,-3E``!3X\\```H$`*#A(?K_ZP``4./U
M__\*!`"@X7_X_^L`0%#B\?__"@0`H.%=^/_K'%"=Y0!@H.$%`*#A6?C_ZP!@
MAN`!:X;B!&"&X@8`H.%[^/_KL#2?Y0!PH.$!$*#C`S"/X`@PC>6@-)_E`"#@
MXPQ0C>4#,(_@!#"-Y9`TG^4`0(WE`S"/X)3X_^L'':#C!P"@X?[W_^L!`'#C
M`P``&JGX_^L`$)#E$0!1XZL``!H($*#A!P"@X7#X_^L`4%#B+0``"DP4G^4`
M,)KE2"2?Y0$`F^<"((_@`'"-Y0$0H.,``)#E7/C_ZP!@H.,L))_E_#.?Y0(@
MC^`#,)+G`""3Y>PPG>4"(#/@`#"@X_,``!H&`*#A]-"-XO"/O>@_^/_K`'!0
MXHD``!H$`)7D``!0XPD```H($*#AG/G_ZP`P4.+X__\*`##3Y0``4^/R__\:
M!`"5Y```4./U__\:P'.?Y0T`H.,V^/_K4S"@XP!0H.$'<(_@'`"-Y7S__^HP
M,)WE#SH#X@$)4^/-__\:.$"=Y>SW_^L``%3AR?__&C`PG>53,.CG!PU3X\7_
M_QH&`*#A(_C_ZP!@H.%H`Y_E"!"@X0``C^!Y^?_K`!!0XGP```H``)KEH?O_
MZP"04.)X```*"0"@X>_W_^L`0*#A`P!0XPP``)HP$Y_E!`!`X@``B>`!$(_@
M2OC_ZP``4.,%```:!`!4XZ(```H$0(G@!3!4Y2\`4^.>```*``.?Y0@0H.$`
M`(_@7?G_ZP``4.,#```*`##0Y3``4^,``%,3!```&@`0H.,)`*#AX??_ZP!`
M4.)9```:R!*?Y<@"G^4!$(_@Q**?Y0``C^#`DI_E8/K_Z[R"G^4)^/_KN#*?
MY0@`C>4!$*#C`S"/X`PPC>6H,I_E!@"@X0!PC>4`(.#C`S"/X`0PC>64,I_E
M"J"/X`F0C^`(@(_@`S"/X`3X_^L.``#J'?C_ZP`PD.41`%/C$```&O+W_^L!
M4(7B`,"@X0HPH.$$(*#A`1"@XP8`H.$0D(WE#%"-Y8`1C>CT]__K!QV@XP8`
MH.%>]__K`$"@X0$`<./J__\*!P"@X8GW_^L@`I_E!A"@X0``C^`Q^O_K!@"@
MX7+^_^MH___J`T"@X37__^J4]__K@`"@X0$`@.+]_O_JL"&?Y0`PFN4"`)OG
M!!"-Y0$0H.,`<(WEV"&?Y0``D.4"((_@M/?_ZU;__^I?_O_KQ"&?Y5@QG^4"
M((_@`S"2YP`@D^7L,)WE`B`SX``PH.-*```:!P"@X?30C>+P3[WHEO?_Z@"0
MFN6$___JU?W_ZP"`H.$`D%'BH?__N@H@4.(`H*#C`##)X@0`H.&XL(WB`*"-
MY8_W_^L&(*#C"Q"@X00`H.%5]__K!@!0XS0``!JX()WE0SR@XT$S1.,#`%+A
MC___&K0@V^%(-03C`P!2X8O__QHR(%CB!`"@X0`PR>+`4(WB`*"-Y7KW_^L$
M`*#A*""@XP40H.%`]__K*`!0XQ\``!KT,)_E`1"@X_`@G^4&`*#A`S"/X`0P
MC>7D,)_E`B"/X`A0C>4,((WE`S"/X`!PC>4`(.#CZ*#-Y8_W_^N?___J!$":
MY0``5.-=__\*!`"@X3OW_^L#`%#C6?__FJ00G^4$`$#B``"$X`$0C^"7]__K
M``!0XP20H`%1___J6O?_ZX0`G^4``(_@</W_ZVAM`0`<`0``E&X!`"AM`0",
M&@``3!L``+0:``!T&@``4!D``$0:```@`0``1!H``!AK`0!(&0``K!D``'09
M``#<%P``'!@``)@7``#X&```G!D```07``#$&0``&!<``-P8``!H%@``7!<`
M`'1H`0!0%0``V!<```07``"P%@``G!8``/!`+>D!RJ#C#,!-X.`/C.5<$I_E
M#-!-XE@BG^5P3QWN`1"/X%`RG^50`I_E#6"@X0(@D><``(_@!A"@X0`@DN4$
M((WE`""@XP,PG^<#4)3G9##5Y64PQ>5W^/_K``!0XP(```H`,-#E,0!3XV8`
M``JY^__K"#*?Y0,PG^<#4)3G`!*?Y04`H.$#,*#C`2"@XP$0C^"_]O_K`%!0
MX@P```KD,9_E#""5Y0,PG^<`$)+EV"&?Y0,`E.<"((_@N_;_ZPPPE>4`$)/E
M"#"1Y0$%$^-9```:N!&?Y0,PH.,!(*#C`1"?YP$`E.>H$9_E`1"/X*CV_^L`
M4%#B"P``"I@!G^4&$*#A``"/X$OX_^L`(%#B00``"H0QG^4,$)7E`S"?YP`0
MD>4#`)3GH/;_ZW`QG^4!(*#C;!&?Y6QQG^4!$(_@`S"?YP=PC^`#4)3G!0"@
MX8'V_^M4(9_E`!"@X04`H.$"((_@D?;_ZT0QG^5$`9_E`""@XP80H.$``(_@
M`S"?YP,PE.=E(,/E*_C_ZP!`H.$``%#C"`"'Y1(```H'':#C!`"@X5WV_^L!
M((#B`#"@X1$`4.,!`%(3)0``BO@@G^6P,)_E`B"/X`,PDN<`()/E!#"=Y0(@
M,^``,*#C&P``&@S0C>+P@+WH!`"7Y<?]_^L`0*#A``!0XP@`A^7F__\:N`"?
MY0``C^#(]O_K`3#0Y0``4^.9__\*E/__ZJ`PG^4,()7E`S"/X```DN4$,)/E
M`!"3Y<?X_^N[___JA#"?Y0,PG^<#`)3GFO;_ZZ#__^JA]O_K<!"?Y00@H.%L
M`)_E`1"/X```C^`$$)'E`!"1Y:[V_^MX9@$`'`$``%QG`0!(%```+&<!`,`5
M``#\9@$`G!4``,1F`0!P%0``,!,``(AF`0!D9@$`)!4``-1F`0`8%0``*&8!
M`+`2``#49`$`?!0``/1E`0!D90$`N&4!`$`4```P0"WI`<J@XPS`3>#H#XSE
M$"&?Y0S03>(,,9_E`%"@X0(@C^`#,)+G`#"3Y00PC>4`,*#CI_K_ZR\0H.,`
M0*#A*/;_ZP``4.,!0(`2!`"@X6#V_^L`0*#AT`"?Y0T0H.$``(_@O_?_ZP`P
M4.(*```*`##3Y3``4^,``%,3`3"@`P`PH!,!,"/B``!5XP`PH`,!,`,2``!3
MXPH``!J0()_EA#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XQ@``!H,T(WB
M,("]Z``PU>4``%/C\?__"EP0G^4$(*#C!`"@X0$0C^!N]O_K``!0X^K__QI$
M()_E,#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XP,``!H%`*#A#-"-XC!`
MO>CV]__J-_;_ZZ1C`0`<`0``9!$``"!C`0"L$0``S&(!`/!/+>D!RJ#C#,!-
MX(`.C.403)_E5]]-X@SLG^40,(WB!$"/X`,0H.$<((#B7,"`X@[@E.<`X)[E
M5.&-Y0#@H..-"F+TX`#P\P(`7.'="D'T^O__&@9+W>T(.]WM$1Z-X@H;W>T`
M``#JLA%B\@@KT^TT((/B``O3[0@P@^(!`%/ACU=B]+`!0O.U`4#SM`%`\[-!
M8_*Q,6'RH!A@\C``X?.P`6'R#@O#[>[__QH@0)#IF3D'XX(Z1>,,0)#E$,"=
MY0!@D.4$$"[@$'"0Y0/`C.`%$`'@99&@X0?`C.`$$"'@%""=Y>;-C.`!P(S@
M"1`NX`,@@N`&$`'@9H&@X00@@N`8H)WE#A`AX`(0@>`(("G@`Z"*X`P@`N`.
MH(K@"2`BX`H@@N`<H)WE[!V!X&S!H.$#H(K@":"*X`R0*.`!D`G@X2V"X`B0
M*>!A$:#A"I")X""@G>7BG8G@`Z"*X`B@BN`!@"S@`H`(X&(AH.$,@"C@`K`A
MX`J`B.`DH)WE";`+X.F-B.`#H(K@#*"*X&G!H.$!D"O@#+`BX`J0B>`HH)WE
M"+`+X.B=B>`#H(K@`:"*X&@1H.$"@"O@`;`LX`J`B.`LH)WE";`+X.F-B.`#
MH(K@`J"*X&DAH.$,D"O@`K`AX`J0B>`PH)WE"+`+X.B=B>`#H(K@#*"*X&C!
MH.$!@"O@#+`BX`J`B.`TH)WE";`+X.F-B.`#H(K@`:"*X&D1H.$"D"O@`;`L
MX`J0B>`XH)WE"+`+X.B=B>`#H(K@`J"*X&@AH.$,@"O@`K`AX`J`B.`\H)WE
M";`+X.F-B.`#H(K@#*"*X&G!H.$!D"O@#+`BX`J0B>!`H)WE"+`+X.B=B>`#
MH(K@`:"*X&@1H.$"@"O@`;`LX`J`B.!$H)WE";`+X.F-B.`#H(K@`J"*X&DA
MH.$,D"O@`K`AX`J0B>!(H)WE"+`+X.B=B>`#H(K@#*"*X&C!H.$!@"O@#+`B
MX`J`B.!,H)WE";`+X.F-B.`#H(K@`:"*X&D1H.$"D"O@`;`LX`J0B>!0H)WE
M"+`+X.B=B>`#H(K@`J"*X&@AH.$,@"O@"H"(X%2@G>7IC8C@`Z"*X`S`BN`"
MH"'@":`*X&F1H.$!H"K@#*"*X%C`G>7HK8K@`\",X`'`C.`)$"+@"!`!X&B!
MH.$"$"'@#!"!X%S`G>7J'8'@`S",X`C`*>`*P`S@`C"#X`G`+.!@()WE`\",
MX*$[#N/9/D;C:J&@X0,@@N#AS8S@"9""X&0@G>4#((+@""""X`J`*.`!@"C@
M"(")X&&1H.%H$)WE[(V(X`,0@>`*$('@":`JX`R@*N!LP:#A"B""X.BM@N!L
M()WE`R""X`D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X'`0G>4#$('@#!"!X`C`
M+.`*P"S@:J&@X0P@@N#IS8+@=""=Y0,@@N`(((+@"H`HX`F`*.!ID:#A"!"!
MX.R-@>!X$)WE`Q"!X`H0@>`)H"K@#*`JX&S!H.$*((+@Z*V"X'P@G>4#((+@
M"2""X`R0*>`(D"G@:(&@X0D0@>#JG8'@@!"=Y0,0@>`,$('@",`LX`K`+.!J
MH:#A#"""X.G-@N"$()WE`R""X`@@@N`*@"C@"8`HX&F1H.$($('@[(V!X(@0
MG>4#$('@"A"!X`F@*N`,H"K@;,&@X0H@@N#HK8+@C""=Y0,@@N!JL:#A"2""
MX`R0*>`(D"G@:(&@X0D0@>#JG8'@D!"=Y0,0@>`,$('@",`LX`K`+.`,((+@
MZ:V"X)0@G>4#((+@""""X`N`*.`)P"C@:9&@X0P0@>"8P)WEZHV!X`D0*^`#
MP(S@"A`AX`O`C."<L)WE:J&@X0$@@N`#L(O@Z!V"X`FPB^`*D"G@""`IX&B!
MH.$"P(S@H""=Y>&=C.`#((+@"B""X`B@*N`!P"K@I*"=Y6$1H.$,P(O@`Z"*
MX.G-C.`(H(K@`8`HX`F`*.!ID:#A""""X+"PG>7LC8+@J""=Y0,@@N`!((+@
M"1`AX`P0(>!LP:#A`:"*X`P0*>#HK8K@"!`AX`$@@N!H@:#AZAV"X*P@G>4#
M((+@"#`LX`HP(^`)((+@`R""X-P\"^,;/TCC:I&@X0.PB^`)H(CA#+"+X+3`
MG>7A+8+@`:`*X`/`C.!A$:#A",",X`F`".`(H(KA`8")X0N@BN`"@`C@`;`)
MX.*MBN`+@(CAO+"=Y0R`B."XP)WE8B&@X0.PB^`#P(S@`;"+X`F0C.`"P('A
M"L`,X`(0`>`!P(SAZHV(X`G`C.#`D)WE:J&@X>C-C.`*$(+A`Y")X`*0B>`(
M$`'@"B`"X&B!H.$"$('A""`*X`NP@>`($(KA#!`!X.R]B^`"$('A;,&@X0D0
M@>#$D)WE#""(X>L=@>`#D(G@"R`"X`J0B>#(H)WE:[&@X0.@BN`(H(K@#(`(
MX`@@@N%A@:#A"9""X`L@C.'AG8G@`2`"X,P0G>4#$('@#!"!X`O`#.`,P(+A
M"""+X0K`C.`)(`+@"*`+X.G-C.`*((+AT*"=Y6F1H.$!$(+@`Z"*X`D@B.$+
MH(K@U+"=Y0P@`N#L'8'@`["+X&S!H.$(L(O@"8`(X`@@@N%A@:#A"J""X`P@
MB>'AK8K@`2`"X-@0G>4#$('@"1"!X`R0">`)D(+A""",X0N0B>`*(`+@"+`,
MX.J=B>`+((+AX+"=Y0$0@N#<()WE:J&@X0.PB^`#((+@"+"+X`S`@N`*((CA
M"2`"X`J`".#I'8'@"(""X6F1H.$,@(C@"2"*X6'!H.'AC8C@`2`"X.00G>4#
M$('@"A"!X`F@"N`*H(+A#"")X0N@BN`((`+@#+`)X.BMBN`+((+A[+"=Y0$0
M@N#H()WE:(&@X0.PB^`#((+@#+"+X`F0@N`(((SA"B`"X`C`#.#J'8'@#"""
MX6JAH.$)D(+@"B"(X6'!H.'AG8G@`2`"X/`0G>4#$('@"!"!X`J`".`(@(+A
M#""*X0N`B.`)(`+@#+`*X.F-B.`+((+A^+"=Y0$@@N#T$)WE:9&@X0.PB^`#
M$('@#+"+X`J@@>`)$(SA"!`!X`G`#.`,P('AZ"V"X`K`C.#\H)WE:(&@X>+-
MC.`($(GA`S"*X`DP@^`"$`'@")`)X&(AH.$"H(CA"1"!X0L0@>`,H`K@`K`(
MX.P=@>!LD:#A"Z"*X0#!G>4#H(K@UC$,XV(Z3.,#P(S@X:V*X`B`C.`$P9WE
M8;&@X0/`C.`"P(S@"2`BX`$@(N`($9WE`H"(X`L@*>#JC8C@"B`BX`,0@>!J
MH:#A"1"!X`+`C.`*D"O@Z,V,X`B0*>`,(9WE"1"!X&B!H.'LG8'@$!&=Y0,@
M@N`#$('@"R""X`H0@>`(H"K@#*`JX&S!H.$*((+@Z:V"X!0AG>4#((+@""""
MX`R`*.`)@"C@:9&@X0@0@>#JC8'@&!&=Y0,0@>`,$('@"<`LX`K`+.!JH:#A
M#"""X.C-@N`<(9WE`R""X`D@@N`*D"G@")`IX&B!H.$)$('@[)V!X"`1G>4#
M$('@"A"!X`B@*N`,H"K@;,&@X0H@@N#IK8+@)"&=Y0,@@N`(((+@#(`HX`F`
M*.!ID:#A"!"!X.J-@>`H$9WE`Q"!X`P0@>`)P"S@"L`LX&JAH.$,((+@Z,V"
MX"PAG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0@>#LG8'@,!&=Y0,0@>`*$('@
M"*`JX`R@*N!LP:#A"B""X.FM@N`T(9WE`R""X`@@@N`,@"C@"8`HX&F1H.$(
M$('@ZHV!X#@1G>4#$('@#!"!X`G`+.`*P"S@:J&@X0P@@N#HS8+@/"&=Y0,@
M@N`)((+@"I`IX`B0*>!H@:#A"1"!X$"1G>7L'8'@`Y")X`J0B>`(H"K@#*`J
MX&S!H.$*((+@1*&=Y>$M@N`#H(K@"*"*X`R`*.`!@"C@81&@X0B0B>!(@9WE
MXIV)X`.`B.`,@(C@`<`LX`+`+.!B(:#A#,"*X`)PA^#IS8S@$'"`Y6SACN`(
MX(WE3.&=Y0,PCN`&,(/@`3"#X`(0(>`)$"'@:9&@X0$0B.`)("+@[!V!X`P@
M(N`",(/@"4"$X.$]@^`,0(WE`#"-Y0%0A>`\()_E!%"-Y=\*;?0"((_@*#"?
MY8\*0/0#,)+G`""3Y50QG>4"(#/@`#"@XP$``!I7WXWB\(^]Z"?S_^MD8@$`
M'`$``(Q6`0`00"WI`<J@XPS`3>#X#XSE8`"@XQOS_^L'"]_M""O?[0D[W^T`
M(*#C!`O`[1@@@.5<((#EWRI`]!"`O>CPX=+#``````$C16>)J\WO_MRZF'94
M,A#P1RWI`<J@XPS`3>#@#XSE%#"0Y0)@H.&"(:#A`%"@X0(PD^`8()#E%#"`
MY0%`H.%<`)#E`2""(J8N@N`<@(7B``!0XQ@@A>4@```:/P!6XRX``-I`D$;B
M*6.@X0%PAN('<X3@!!"@X0@PH.%`0(3B`."1Y1`0@>(,P!'E$#"#X@@`$>4$
M(!'E!`!1X1#@`^4,P`/E"``#Y00@`^7S__\:`4"@X04`H.&R_/_K!P!4X>O_
M_QH&8TG@!B"@X0<0H.$(`*#AN?+_ZUQ@A>7PA[WH0'!@X@``B.`&`%?A!G"@
MH0<@H.&Q\O_K7#"5Y0,PA^!<,(7E0`!3X_"'O1@%`*#A!V!&X`=`A.":_/_K
MSO__Z@1PH.'G___J<$`MZ0'*H.,"BRWM#,!-X.@/C.4!0*#A%!"1Y0!0H.$<
M8(3B,3^_YA00E.48`)3ET1'EYS`OO^88*T/L`2"!XG\PX.,X`%+C`@"&X`$P
MQN=!``#:0"!BX@`0H.-C\O_K!`"@X7W\_^LL,(3B4`#`\@\*1O0/"D/T/#"$
MX@\*0_1,,(3B#P=#]%0PA.($`*#A#X<#]''\_^L#,-3E`##%Y00`H.&R,-3A
M`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00PQ>6V,-3A!3#%Y00PE.4C
M-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%Y0@PE.4C-*#A"C#%Y0@P
ME.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A#C#%Y0PPE.4/,,7E$S#4
MY1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43,,7E`HN]['!`O>@S\O_J
M."!BX@`0H.,A\O_KQO__Z@@PG^4#,(_@`#"3Y7SR_^JL5`$`"$`MZ0B`O>@!
M``(`/0```$`````+````0````!````!`````)@```$`````U````0`````,`
M``!`````!@```$`````4````0`````X```!`````#0```$`````E````0```
M`!(```!`````*````$`````"````0````!@```!`````"0```$`````*````
M0```````````````)7,O)7,````O<')O8R]S96QF+V5X90``1'EN84QO861E
M<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````"]P
M<F]C+R5I+R5S`&5X90`N````4$%27U1%35``````4$%27U!23T=.04U%````
M`#H````E<R5S)7,``"\```!015),-4Q)0@````!015),3$E"`%!%4DPU3U!4
M`````%!%4DQ)3P``4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#
M3$5!3@```%!!4E]$14)51P```%!!4E]#04-(10```%!!4E]'3$]"04Q?1$5"
M54<`````4$%27T=,3T)!3%]435!$25(```!005)?5$U01$E2``!005)?1TQ/
M0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@`````Q````;'-E96L@9F%I;&5D
M`````')E860@9F%I;&5D`$Q$7TQ)0E)!4EE?4$%42`!365-414T``"4P,G@`
M````)7,E<R5S)7,`````<&%R+0`````E<SH@8W)E871I;VX@;V8@<')I=F%T
M92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*```E<SH@<')I
M=F%T92!S=6)D:7)E8W1O<GD@)7,@:7,@=6YS869E("AP;&5A<V4@<F5M;W9E
M(&ET(&%N9"!R971R>2!Y;W5R(&]P97)A=&EO;BD*``!0051(`````'!A<FP`
M````+G!A<@````!S:&]R="!R96%D```E<R5S8V%C:&4M)7,E<P``)7,E<W1E
M;7`M)74E<P```"5S)7-T96UP+25U+25U)7,`````&````'!E<FP`````,```
M`%!!4CHZ4&%C:V5R.CI615)324].`````#$N,#4Y````56YA8FQE('1O(&-R
M96%T92!C86-H92!D:7)E8W1O<GD`````)7,Z(&-R96%T:6]N(&]F('!R:79A
M=&4@8V%C:&4@<W5B9&ER96-T;W)Y("5S(&9A:6QE9"`H97)R;F\])6DI"@``
M```M90``+2T``%5315(`````55-%4DY!344`````5$5-4$1)4@!435``+W1M
M<`````!E=F%L("=E>&5C("]U<W(O8FEN+W!E<FP@("U3("0P("1[,2LB)$`B
M?2<*:68@,#L@(R!N;W0@<G5N;FEN9R!U;F1E<B!S;VUE('-H96QL"G!A8VMA
M9V4@7U]P87)?<&P["FUY("@D4$%27TU!1TE#+"`D1DE,15]O9F9S971?<VEZ
M92P@)&-A8VAE7VYA;65?<VEZ92D["D)%1TE.('L*)%!!4E]-04=)0R`](")<
M;E!!4BYP;5QN(CL*)$9)3$5?;V9F<V5T7W-I>F4@/2`T.R`@(",@<&%C:R@B
M3B(I"B1C86-H95]N86UE7W-I>F4@/2`T,#L*)%!+6DE07TU!1TE#(#T@(E!+
M7#`P,UPP,#0B.PI]"G-U8B!F:6YD7W!A<E]M86=I8PI["FUY("@D9F@I(#T@
M0%\["FUY("1C:'5N:U]S:7IE(#T@-C0@*B`Q,#(T.PIM>2`D8G5F.PIM>2`D
M<VEZ92`]("US("1F:#L*;7D@)'!O<R`]("@D<VEZ92TQ*2`M("@D<VEZ92TQ
M*2`E("1C:'5N:U]S:7IE.R`@("`@(",@3D]413H@)'!O<R!I<R!A(&UU;'1I
M<&QE(&]F("1C:'5N:U]S:7IE"G=H:6QE("@D<&]S(#X](#`I('L*<V5E:R`D
M9F@L("1P;W,L(#`["G)E860@)&9H+"`D8G5F+"`D8VAU;FM?<VEZ92`K(&QE
M;F=T:"@D4$%27TU!1TE#*3L*:68@*"AM>2`D:2`](')I;F1E>"@D8G5F+"`D
M4$%27TU!1TE#*2D@/CT@,"D@>PIR971U<FX@)'!O<R`K("1I.PI]"B1P;W,@
M+3T@)&-H=6YK7W-I>F4["GT*<F5T=7)N("TQ.PI]"FUY("@D<&%R7W1E;7`L
M("1P<F]G;F%M92P@0'1M<&9I;&4L("5-;V1U;&5#86-H92D["D5.1"![(&EF
M("@D14Y6>U!!4E]#3$5!3GTI('L*<F5Q=6ER92!&:6QE.CI496UP.PIR97%U
M:7)E($9I;&4Z.D)A<V5N86UE.PIR97%U:7)E($9I;&4Z.E-P96,["FUY("1T
M;W!D:7(@/2!&:6QE.CI"87-E;F%M93HZ9&ER;F%M92@D<&%R7W1E;7`I.PIO
M=71S*'%Q6U)E;6]V:6YG(&9I;&5S(&EN("(D<&%R7W1E;7`B72D["D9I;&4Z
M.D9I;F0Z.F9I;F1D97!T:"AS=6(@>R`H("UD("D@/R!R;61I<B`Z('5N;&EN
M:R!]+"`D<&%R7W1E;7`I.PIR;61I<B`D<&%R7W1E;7`["FEF("@M9"`D<&%R
M7W1E;7`@)B8@)%Y/(&YE("=-4U=I;C,R)RD@>PIM>2`D=&UP(#T@;F5W($9I
M;&4Z.E1E;7`H"E1%35!,051%(#T^("=T;7!86%A86"<L"D1)4B`]/B!&:6QE
M.CI"87-E;F%M93HZ9&ER;F%M92@D=&]P9&ER*2P*4U5&1DE8(#T^("<N8VUD
M)RP*54Y,24Y+(#T^(#`L"BD["FUY("1F:6QE;F%M92`]("1T;7`M/F9I;&5N
M86UE.PIP<FEN="`D=&UP(#P\(BXN+B(["G@],3L@=VAI;&4@6R!<)'@@+6QT
M(#$P(%T[(&1O"G)M("UR9B`G)'!A<E]T96UP)PII9B!;(%PA("UD("<D<&%R
M7W1E;7`G(%T[('1H96X*8G)E86L*9FD*<VQE97`@,0IX/6!E>'!R(%PD>"`K
M(#%@"F1O;F4*<FT@)R1F:6QE;F%M92<*+BXN"F-L;W-E("1T;7`["F-H;6]D
M(#`W,#`L("1F:6QE;F%M93L*;7D@)&-M9"`]("(D9FEL96YA;64@/B]D978O
M;G5L;"`R/B8Q("8B.PIS>7-T96TH)&-M9"D["F]U=',H<7%;4W!A=VYE9"!B
M86-K9W)O=6YD('!R;V-E<W,@=&\@<&5R9F]R;2!C;&5A;G5P.B`D9FEL96YA
M;65=*3L*?0I]('T*0D5'24X@>PI);G1E<FYA;',Z.E!!4CHZ0D]/5"@I(&EF
M(&1E9FEN960@)DEN=&5R;F%L<SHZ4$%2.CI"3T]4.PIE=F%L('L*7W!A<E]I
M;FET7V5N=B@I.PIM>2`D<75I970@/2`A)$5.5GM005)?1$5"54=].PIM>2`E
M<WES(#T@*`IP871H7W-E<"`@("`]/B`H)%Y/(#U^("]>35-7:6XO(#\@)SLG
M(#H@)SHG*2P*7V5X92`@("`@("`@/3X@*"1>3R`]?B`O7B@_.DU35VEN?$]3
M,GQC>6=W:6XI+R`_("<N97AE)R`Z("<G*2P*7V1E;&EM("`@("`@/3X@*"1>
M3R`]?B`O7DU35VEN?$]3,B\@/R`G7%PG(#H@)R\G*2P**3L*7W-E=%]P<F]G
M;F%M92@I.PIO=71S*'%Q6UPD<')O9VYA;64@/2`B)'!R;V=N86UE(ETI.PI?
M<V5T7W!A<E]T96UP*"D["F]U=',H<7%;7"1%3E9[4$%27U1%35!](#T@(B1%
M3E9[4$%27U1%35!](ETI.PIM>2`H)'-T87)T7W!O<RP@)'-T87)T7V]F7T9)
M3$5?<V5C=&EO;BD["DU!1TE#.B!["FQO8V%L("1324=[7U]705).7U]](#T@
M<W5B('M].PIU;FQE<W,@*&]P96X@7T9(+"`G/#IR87<G+"`D<')O9VYA;64I
M('L*;W5T<RAQ<5M#86XG="!R96%D(&9R;VT@9FEL92`B)'!R;V=N86UE(ETI
M.R`@(R!D;VXG="!U<V4@)"$@:&5R92!A<R!I="!R97%U:7)E<R!%<G)N;RYP
M;0IL87-T($U!1TE#.PI]"FUY("1M86=I8U]P;W,@/2!F:6YD7W!A<E]M86=I
M8R@J7T9(*3L*:68@*"1M86=I8U]P;W,@/"`P*2!["F]U=',H<7%;0V%N)W0@
M9FEN9"!M86=I8R!S=')I;F<@(B1005)?34%'24,B(&EN(&9I;&4@(B1P<F]G
M;F%M92)=*3L*;&%S="!-04=)0SL*?0IO=71S*")&;W5N9"!005(@;6%G:6,@
M870@<&]S:71I;VX@)&UA9VEC7W!O<R(I.PIS965K(%]&2"P@)&UA9VEC7W!O
M<R`M("1&24Q%7V]F9G-E=%]S:7IE("T@;&5N9W1H*")<,$-!0TA%(BDL(#`[
M"G)E860@7T9(+"`D8G5F+"!L96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@
M;F4@(EPP0T%#2$4B("8F("1B=68@;F4@(EPP0TQ%04XB*2!["F]U=',H(DYO
M(&-A8VAE(&UA<FME<B!F;W5N9"(I.PIL87-T($U!1TE#.PI]"F]U=',H<7%;
M0V%C:&4@;6%R:V5R("(D8G5F(B!F;W5N9%TI.PIS965K(%]&2"P@)&UA9VEC
M7W!O<R`M("1&24Q%7V]F9G-E=%]S:7IE+"`P.PIR96%D(%]&2"P@)&)U9BP@
M)$9)3$5?;V9F<V5T7W-I>F4["FUY("1O9F9S970@/2!U;G!A8VLH(DXB+"`D
M8G5F*3L*;W5T<R@B3V9F<V5T(&9R;VT@<W1A<G0@;V8@1DE,17,@:7,@)&]F
M9G-E="(I.PIS965K(%]&2"P@)&UA9VEC7W!O<R`M("1&24Q%7V]F9G-E=%]S
M:7IE("T@)&]F9G-E="P@,#L*)'-T87)T7V]F7T9)3$5?<V5C=&EO;B`]('1E
M;&P@7T9(.PIM>2`E<F5Q=6ER95]L:7-T.PIR96%D(%]&2"P@)&)U9BP@-#L@
M("`@("`@("`@("`@("`@("`@("`@("`@("`C(')E860@=&AE(&9I<G-T(")&
M24Q%(@IW:&EL92`H)&)U9B!E<2`B1DE,12(I('L*<F5A9"!?1D@L("1B=68L
M(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*;7D@)&9U
M;&QN86UE(#T@)&)U9CL*;W5T<RAQ<5M5;G!A8VMI;F<@1DE,12`B)&9U;&QN
M86UE(BXN+ETI.PIM>2`D8W)C(#T@*"`D9G5L;&YA;64@/7X@<WQ>*%MA+69<
M9%U[.'TI+WQ\("D@/R`D,2`Z('5N9&5F.PIM>2`H)&)A<V5N86UE+"`D97AT
M*2`]("@D8G5F(#U^(&U\*#\Z+BHO*3\H+BHI*%PN+BHI?"D["G)E860@7T9(
M+"`D8G5F+"`T.PIR96%D(%]&2"P@)&)U9BP@=6YP86-K*").(BP@)&)U9BD[
M"FEF("AD969I;F5D*"1E>'0I(&%N9"`D97AT("%^("]<+B@_.G!M?'!L?&EX
M?&%L*20O:2D@>PIM>2`D9FEL96YA;64@/2!?<V%V95]A<R@B)&-R8R1E>'0B
M+"`D8G5F+"`P-S4U*3L*)%!!4CHZ2&5A=GDZ.D9U;&Q#86-H97LD9G5L;&YA
M;65](#T@)&9I;&5N86UE.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F:6QE
M;F%M97T@/2`D9G5L;&YA;64["GT*96QS92!["B1-;V1U;&5#86-H97LD9G5L
M;&YA;65](#T@>PIB=68@/3X@)&)U9BP*8W)C(#T^("1C<F,L"FYA;64@/3X@
M)&9U;&QN86UE+`I].PHD<F5Q=6ER95]L:7-T>R1F=6QL;F%M97TK*SL*?0IR
M96%D(%]&2"P@)&)U9BP@-#L*?0IL;V-A;"!`24Y#(#T@*'-U8B!["FUY("@D
M<V5L9BP@)&UO9'5L92D@/2!`7SL*<F5T=7)N(&EF(')E9B`D;6]D=6QE(&]R
M("$D;6]D=6QE.PIM>2`D:6YF;R`]("1-;V1U;&5#86-H97LD;6]D=6QE?3L*
M<F5T=7)N('5N;&5S<R`D:6YF;SL*9&5L971E("1R97%U:7)E7VQI<W1[)&UO
M9'5L97T["B1)3D-[)&UO9'5L97T@/2`B+VQO861E<B\D:6YF;R\D;6]D=6QE
M(CL*:68@*"1%3E9[4$%27T-,14%.?2!A;F0@9&5F:6YE9"@F24\Z.D9I;&4Z
M.FYE=RDI('L*;7D@)&9H(#T@24\Z.D9I;&4M/FYE=U]T;7!F:6QE(&]R(&1I
M92`B0V%N)W0@8W)E871E('1E;7`@9FEL93H@)"$B.PHD9F@M/F)I;FUO9&4H
M*3L*)&9H+3YP<FEN="@D:6YF;RT^>V)U9GTI.PHD9F@M/G-E96LH,"P@,"D[
M"G)E='5R;B`D9F@["GT*96QS92!["FUY("1F:6QE;F%M92`](%]S879E7V%S
M*"(D:6YF;RT^>V-R8WTN<&TB+"`D:6YF;RT^>V)U9GTI.PHD:6YF;RT^>V9I
M;&5](#T@)&9I;&5N86UE.PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)&9I;&5N
M86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D9FEL96YA;64B.B`D(5T["G)E
M='5R;B`D9F@["GT*9&EE(")";V]T<W1R87!P:6YG(&9A:6QE9#H@8V%N)W0@
M9FEN9"!M;V1U;&4@)&UO9'5L92$B.PI]+"!`24Y#*3L*<F5Q=6ER92!84TQO
M861E<CL*<F5Q=6ER92!005(Z.DAE879Y.PIR97%U:7)E($-A<G`Z.DAE879Y
M.PIR97%U:7)E($5X<&]R=&5R.CI(96%V>3L*4$%2.CI(96%V>3HZ7VEN:71?
M9'EN86QO861E<B@I.PIR97%U:7)E($E/.CI&:6QE.PIW:&EL92`H;7D@)&9I
M;&5N86UE(#T@*'-O<G0@:V5Y<R`E<F5Q=6ER95]L:7-T*5LP72D@>PIU;FQE
M<W,@*"1)3D-[)&9I;&5N86UE?2!O<B`D9FEL96YA;64@/7X@+T)31%!!3B\I
M('L*:68@*"1F:6QE;F%M92`]?B`O7"YP;6,_)"]I*2!["G)E<75I<F4@)&9I
M;&5N86UE.PI]"F5L<V4@>PID;R`D9FEL96YA;64@=6YL97-S("1F:6QE;F%M
M92`]?B`O<VET96-U<W1O;6EZ95PN<&PD+SL*?0I]"F1E;&5T92`D<F5Q=6ER
M95]L:7-T>R1F:6QE;F%M97T["GT*:68@*"1B=68@;F4@)%!+6DE07TU!1TE#
M*2!["F]U=',H<7%;3F\@>FEP(&9O=6YD(&%F=&5R($9)3$4@<V5C=&EO;B!I
M;B!F:6QE("(D<')O9VYA;64B72D["FQA<W0@34%'24,@.PI]"B1S=&%R=%]P
M;W,@/2`H=&5L;"!?1D@I("T@-#L@("`@("`@("`@("`@("`@(R!S=&%R="!O
M9B!Z:7`*?0IM>2!`<&%R7V%R9W,["FUY("@D;W5T+"`D8G5N9&QE+"`D;&]G
M9F@L("1C86-H95]N86UE*3L*9&5L971E("1%3E9[4$%27T%04%]2155317T[
M(",@<V%N:71I>F4@*%)%55-%(&UA>2!B92!A('-E8W5R:71Y('!R;V)L96TI
M"FEF("@A)'-T87)T7W!O<R!O<B`H)$%21U9;,%T@97$@)RTM<&%R+6]P=&EO
M;G,G("8F('-H:69T*2D@>PIM>2`E9&ES=%]C;60@/2!Q=R@*<"`@(&)L:6)?
M=&]?<&%R"FD@("!I;G-T86QL7W!A<@IU("`@=6YI;G-T86QL7W!A<@IS("`@
M<VEG;E]P87(*=B`@('9E<FEF>5]P87(**3L*:68@*$!!4D=6(&%N9"`D05)'
M5ELP72!E<2`G+2UR975S92<I('L*<VAI9G0@0$%21U8["B1%3E9[4$%27T%0
M4%]2155317T@/2!S:&EF="!`05)'5CL*?0IE;'-E('L@(R!N;W)M86P@<&%R
M;"!B96AA=FEO=7(*;7D@0&%D9%]T;U]I;F,["G=H:6QE("A`05)'5BD@>PHD
M05)'5ELP72`]?B`O7BTH6T%)34]"3&)Q<&EU<U1V72DH+BHI+R!O<B!L87-T
M.PII9B`H)#$@97$@)TDG*2!["G!U<V@@0&%D9%]T;U]I;F,L("0R.PI]"F5L
M<VEF("@D,2!E<2`G32<I('L*979A;"`B=7-E("0R(CL*?0IE;'-I9B`H)#$@
M97$@)T$G*2!["G5N<VAI9G0@0'!A<E]A<F=S+"`D,CL*?0IE;'-I9B`H)#$@
M97$@)T\G*2!["B1O=70@/2`D,CL*?0IE;'-I9B`H)#$@97$@)V(G*2!["B1B
M=6YD;&4@/2`G<VET92<["GT*96QS:68@*"0Q(&5Q("=")RD@>PHD8G5N9&QE
M(#T@)V%L;"<["GT*96QS:68@*"0Q(&5Q("=Q)RD@>PHD<75I970@/2`Q.PI]
M"F5L<VEF("@D,2!E<2`G3"<I('L*;W!E;B`D;&]G9F@L("(^/B(L("0R(&]R
M(&1I92!Q<5M#86XG="!O<&5N(&QO9R!F:6QE("(D,B(Z("0A73L*?0IE;'-I
M9B`H)#$@97$@)U0G*2!["B1C86-H95]N86UE(#T@)#(["GT*<VAI9G0H0$%2
M1U8I.PII9B`H;7D@)&-M9"`]("1D:7-T7V-M9'LD,7TI('L*9&5L971E("1%
M3E9[)U!!4E]414U0)WT["FEN:71?:6YC*"D["G)E<75I<F4@4$%2.CI$:7-T
M.PHF>R)005(Z.D1I<W0Z.B1C;60B?2@I('5N;&5S<R!`05)'5CL*)GLB4$%2
M.CI$:7-T.CHD8VUD(GTH)%\I(&9O<B!`05)'5CL*97AI=#L*?0I]"G5N<VAI
M9G0@0$E.0RP@0&%D9%]T;U]I;F,["GT*?0II9B`H)&]U="D@>PI["G)E<75I
M<F4@24\Z.D9I;&4["G)E<75I<F4@07)C:&EV93HZ6FEP.PIR97%U:7)E($1I
M9V5S=#HZ4TA!.PI]"FUY("1P87(@/2!S:&EF="A`05)'5BD["FUY("1Z:7`[
M"FEF("AD969I;F5D("1P87(I('L*;W!E;B!M>2`D9F@L("<\.G)A=R<L("1P
M87(@;W(@9&EE('%Q6T-A;B=T(&9I;F0@<&%R(&9I;&4@(B1P87(B.B`D(5T[
M"F)L97-S*"1F:"P@)TE/.CI&:6QE)RD["B1Z:7`@/2!!<F-H:79E.CI::7`M
M/FYE=SL**"`D>FEP+3YR96%D1G)O;49I;&5(86YD;&4H)&9H+"`D<&%R*2`]
M/2!!<F-H:79E.CI::7`Z.D%:7T]+*"D@*0IO<B!D:64@<7%;17)R;W(@<F5A
M9&EN9R!Z:7`@87)C:&EV92`B)'!A<B)=.PI]"FUY("5M971A7W!A<B`](&1O
M('L*:68@*"1Z:7`@86YD(&UY("1M971A(#T@)'II<"T^8V]N=&5N=',H)TU%
M5$$N>6UL)RDI('L*)&UE=&$@/7X@<R\N*EYP87(Z)"\O;7,["B1M971A(#U^
M(',O7EQ3+BHO+VUS.PHD;65T82`]?B`O7B`@*%M>.ETK*3H@*"XK*20O;6<[
M"GT*?3L*:68@*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1P:"P@)SPZ<F%W
M)RP@)'!A<B!O<B!D:64@<7%;0V%N)W0@<F5A9"!P87(@9FEL92`B)'!A<B(Z
M("0A73L*;7D@)&)U9CL*<F5A9"`D<&@L("1B=68L(#0["F1I92!Q<5LB)'!A
M<B(@:7,@;F]T(&$@<&%R(&9I;&5=('5N;&5S<R`D8G5F(&5Q("102UI)4%]-
M04=)0SL*8VQO<V4@)'!H.PI]"D-R96%T95!A=&@H)&]U="D@.PIM>2`D9F@@
M/2!)3SHZ1FEL92T^;F5W*`HD;W5T+`I)3SHZ1FEL93HZ3U]#4D5!5"@I('P@
M24\Z.D9I;&4Z.D]?4D174B@I('P@24\Z.D9I;&4Z.D]?5%)53D,H*2P*,#<W
M-RP**2!O<B!D:64@<7%;0V%N)W0@8W)E871E(&9I;&4@(B1O=70B.B`D(5T[
M"B1F:"T^8FEN;6]D92@I.PIS965K(%]&2"P@,"P@,#L*;7D@)&QO861E<CL*
M:68@*&1E9FEN960@)'-T87)T7V]F7T9)3$5?<V5C=&EO;BD@>PIR96%D(%]&
M2"P@)&QO861E<BP@)'-T87)T7V]F7T9)3$5?<V5C=&EO;CL*?2!E;'-E('L*
M;&]C86P@)"\@/2!U;F1E9CL*)&QO861E<B`](#Q?1D@^.PI]"FEF("@A)$5.
M5GM005)?5D520D%424U](&%N9"`D;&]A9&5R(#U^("]>*#\Z(R%\7$!R96TI
M+RD@>PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13=')I<#L*4$%2.CI&:6QT
M97(Z.E!O9%-T<FEP+3YA<'!L>2A<)&QO861E<BP@)#`I.PI]"B1F:"T^<')I
M;G0H)&QO861E<BD*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@;&]A9&5R('1O
M("(D;W5T(CH@)"%=.PII9B`H)&)U;F1L92D@>PIR97%U:7)E(%!!4CHZ2&5A
M=GD["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*:6YI=%]I;F,H
M*3L*<F5Q=6ER95]M;V1U;&5S*"D["FUY($!I;F,@/2!G<F5P('L@(2]"4T10
M04XO('T*9W)E<"!["B@D8G5N9&QE(&YE("=S:71E)RD@;W(**"1?(&YE("1#
M;VYF:6<Z.D-O;F9I9WMA<F-H;&EB97AP?2!A;F0*)%\@;F4@)$-O;F9I9SHZ
M0V]N9FEG>W!R:79L:6)E>'!]*3L*?2!`24Y#.PIS?"\K?"]\9RP@<WPO)'Q\
M(&9O<F5A8V@@0&EN8SL*;7D@)69I;&5S.PHD9FEL97-[)%]]*RL@9F]R($!$
M>6YA3&]A9&5R.CID;%]S:&%R961?;V)J96-T<RP@=F%L=65S("5)3D,["FUY
M("1L:6)?97AT(#T@)$-O;F9I9SHZ0V]N9FEG>VQI8E]E>'1].R`@("`@("`@
M(",@6%A8(&QI8E]E>'0@=G,@9&QE>'0@/PIM>2`E=W)I='1E;CL*9F]R96%C
M:"!M>2`D:V5Y("AS;W)T(&ME>7,@)69I;&5S*2!["FUY("@D9FEL92P@)&YA
M;64I.PII9B`H9&5F:6YE9"AM>2`D9F,@/2`D4$%2.CI(96%V>3HZ1G5L;$-A
M8VAE>R1K97E]*2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y+"`D9F,I.PI]
M"F5L<V4@>PIF;W)E86-H(&UY("1D:7(@*$!I;F,I('L*:68@*"1K97D@/7X@
M;7Q>7%$D9&ER7$4O*"XJ*21\:2D@>PHH)&9I;&4L("1N86UE*2`]("@D:V5Y
M+"`D,2D["FQA<W0["GT*:68@*"1K97D@/7X@;7Q>+VQO861E<B];7B]=*R\H
M+BHI)'PI('L*:68@*&UY("1R968@/2`D36]D=6QE0V%C:&5[)#%]*2!["B@D
M9FEL92P@)&YA;64I(#T@*"1R968L("0Q*3L*;&%S=#L*?0II9B`H+68@(B1D
M:7(O)#$B*2!["B@D9FEL92P@)&YA;64I(#T@*"(D9&ER+R0Q(BP@)#$I.PIL
M87-T.PI]"GT*?0I]"FYE>'0@=6YL97-S(&1E9FEN960@)&YA;64["FYE>'0@
M:68@)'=R:71T96Y[)&YA;65]*RL["FYE>'0@:68@(7)E9B@D9FEL92D@86YD
M("1F:6QE(#U^("]<+EQ1)&QI8E]E>'1<120O:3L*;W5T<RAS<')I;G1F*'%Q
M6U!A8VMI;F<@1DE,12`B)7,B+BXN72P@<F5F("1F:6QE(#\@)&9I;&4M/GMN
M86UE?2`Z("1F:6QE*2D["FUY("1C;VYT96YT.PII9B`H<F5F*"1F:6QE*2D@
M>PHD8V]N=&5N="`]("1F:6QE+3Y[8G5F?3L*?0IE;'-E('L*;&]C86P@)"\@
M/2!U;F1E9CL*;W!E;B!M>2`D=&@L("<\.G)A=R<L("1F:6QE(&]R(&1I92!Q
M<5M#86XG="!R96%D("(D9FEL92(Z("0A73L*)&-O;G1E;G0@/2`\)'1H/CL*
M8VQO<V4@)'1H.PI005(Z.D9I;'1E<CHZ4&]D4W1R:7`M/F%P<&QY*%PD8V]N
M=&5N="P@(CQE;6)E9&1E9#XO)&YA;64B*0II9B`A)$5.5GM005)?5D520D%4
M24U](&%N9"`D;F%M92`]?B`O7"XH/SIP;7QI>'QA;"DD+VD["E!!4CHZ1FEL
M=&5R.CI0871C:$-O;G1E;G0M/FYE=RT^87!P;'DH7"1C;VYT96YT+"`D9FEL
M92P@)&YA;64I.PI]"B1F:"T^<')I;G0H(D9)3$4B+`IP86-K*"=.)RP@;&5N
M9W1H*"1N86UE*2`K(#DI+`IS<')I;G1F*"(E,#AX+R5S(BP@07)C:&EV93HZ
M6FEP.CIC;VUP=71E0U)#,S(H)&-O;G1E;G0I+"`D;F%M92DL"G!A8VLH)TXG
M+"!L96YG=&@H)&-O;G1E;G0I*2P*)&-O;G1E;G0I"F]R(&1I92!Q<5M%<G)O
M<B!W<FET:6YG(&5M8F5D9&5D($9)3$4@=&\@(B1O=70B.B`D(5T["F]U=',H
M<7%;5W)I='1E;B!A<R`B)&YA;64B72D["GT*?0II9B`H)'II<"D@>PHD>FEP
M+3YW<FET951O1FEL94AA;F1L92@D9F@I(#T]($%R8VAI=F4Z.EII<#HZ05I?
M3TL*;W(@9&EE('%Q6T5R<F]R('=R:71I;F<@>FEP('!A<G0@;V8@(B1O=70B
M73L*?0II9B`H)&UE=&%?<&%R>V-L96%N?2D@>PHD9F@M/G!R:6YT*")<,$-,
M14%.(BD["GT*96QS92!["FEF("@A9&5F:6YE9"`D8V%C:&5?;F%M92D@>PIM
M>2`D8W1X(#T@1&EG97-T.CI32$$M/FYE=R@Q*3L*<V5E:R`D9F@L(#`L(#`[
M"B1C='@M/F%D9&9I;&4H)&9H*3L*<V5E:R`D9F@L(#`L(#(["B1C86-H95]N
M86UE(#T@)&-T>"T^:&5X9&EG97-T.PI]"B1C86-H95]N86UE(#T@<W5B<W1R
M("1C86-H95]N86UE+"`P+"`D8V%C:&5?;F%M95]S:7IE.PHD8V%C:&5?;F%M
M92`N/2`B7#`B('@@*"1C86-H95]N86UE7W-I>F4@+2!L96YG=&@@)&-A8VAE
M7VYA;64I.PHD9F@M/G!R:6YT*"1C86-H95]N86UE+`HB7#!#04-(12(I.PI]
M"FUY("1O9F9S970@/2`D9F@M/G1E;&P@+2!L96YG=&@H)&QO861E<BD["B1F
M:"T^<')I;G0H<&%C:R@G3B<L("1O9F9S970I+`HD4$%27TU!1TE#*3L*)&9H
M+3YC;&]S92!O<B!D:64@<7%;17)R;W(@=W)I=&EN9R!T<F%I;&5R(&]F("(D
M;W5T(CH@)"%=.PIC:&UO9"`P-S4U+"`D;W5T.PIE>&ET.PI]"GL*;&%S="!U
M;FQE<W,@9&5F:6YE9"`D<W1A<G1?<&]S.PI?9FEX7W!R;V=N86UE*"D["G)E
M<75I<F4@4$%2.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA;&]A9&5R*"D["GL*
M<F5Q=6ER92!&:6QE.CI&:6YD.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*?0IM
M>2`D9F@@/2!)3SHZ1FEL92T^;F5W.R`@("`@("`@("`@("`@("`@("`@("`@
M("`@("`@(R!!<F-H:79E.CI::7`@;W!E<F%T97,@;VX@86X@24\Z.DAA;F1L
M90HD9F@M/F9D;W!E;BAF:6QE;F\H7T9(*2P@)W(G*2!O<B!D:64@<7%;9F1O
M<&5N*"D@9F%I;&5D.B`D(5T["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3:7IE
M*"US(%]&2"D["FUY("1Z:7`@/2!!<F-H:79E.CI::7`M/FYE=SL**"1Z:7`M
M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P<F]G;F%M92D@/3T@07)C:&EV
M93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@<7%;17)R;W(@<F5A9&EN9R!Z:7`@
M87)C:&EV92`B)'!R;V=N86UE(ET["D%R8VAI=F4Z.EII<#HZ<V5T0VAU;FM3
M:7IE*#8T("H@,3`R-"D["G!U<V@@0%!!4CHZ3&EB0V%C:&4L("1Z:7`["B10
M05(Z.DQI8D-A8VAE>R1P<F]G;F%M97T@/2`D>FEP.PIO=71S*")%>'1R86-T
M:6YG('II<"XN+B(I.PII9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2D@>R`C
M('-H;W5L9"!B92!S970@870@=&AI<R!P;VEN="$*9F]R96%C:"!M>2`D;65M
M8F5R("@@)'II<"T^;65M8F5R<R`I('L*;F5X="!I9B`D;65M8F5R+3YI<T1I
M<F5C=&]R>3L*;7D@)&UE;6)E<E]N86UE(#T@)&UE;6)E<BT^9FEL94YA;64[
M"FYE>'0@=6YL97-S("1M96UB97)?;F%M92`]?B!M>PI>"B\_<VAL:6(O"B@_
M.B1#;VYF:6<Z.D-O;F9I9WMA<F-H;F%M97TO*3\**%M>+UTK*0HD"GUX.PIM
M>2`D97AT<F%C=%]N86UE(#T@)#$["FUY("1D97-T7VYA;64@/2!&:6QE.CI3
M<&5C+3YC871F:6QE*"1%3E9[4$%27U1%35!]+"`D97AT<F%C=%]N86UE*3L*
M:68@*"UF("1D97-T7VYA;64@)B8@+7,@7R`]/2`D;65M8F5R+3YU;F-O;7!R
M97-S9613:7IE*"DI('L*;W5T<RAQ<5M3:VEP<&EN9R`B)&UE;6)E<E]N86UE
M(B!S:6YC92!I="!A;')E861Y(&5X:7-T<R!A="`B)&1E<W1?;F%M92)=*3L*
M?2!E;'-E('L*;W5T<RAQ<5M%>'1R86-T:6YG("(D;65M8F5R7VYA;64B('1O
M("(D9&5S=%]N86UE(BXN+ETI.PHH)&UE;6)E<BT^97AT<F%C=%1O1FEL94YA
M;65D*"1D97-T7VYA;64I(#T]($%R8VAI=F4Z.EII<#HZ05I?3TLH*2D*;W(@
M9&EE('%Q6T5R<F]R(&5X=')A8W1I;F<@>FEP(&UE;6)E<B!T;R`B)&1E<W1?
M;F%M92)=.PIC:&UO9"@P-34U+"`D9&5S=%]N86UE*2!I9B`D7D\@97$@(FAP
M=7@B.PI]"GT*?0I]"G5N;&5S<R`H)%!!4CHZ3&EB0V%C:&5[)'!R;V=N86UE
M?2D@>PID:64@/#P@(BXB('5N;&5S<R!`05)'5CL*57-A9V4Z("0P(%L@+4%L
M:6(N<&%R(%T@6R`M261I<B!=(%L@+4UM;V1U;&4@72!;('-R8RYP87(@72!;
M('!R;V=R86TN<&P@70HD,"!;("U"?"UB(%T@6RU/;W5T9FEL95T@<W)C+G!A
M<@HN"B1%3E9[4$%27U!23T=.04U%?2`]("1P<F]G;F%M92`]("0P(#T@<VAI
M9G0H0$%21U8I.PI]"G-U8B!#<F5A=&50871H('L*;7D@*"1N86UE*2`]($!?
M.PIR97%U:7)E($9I;&4Z.D)A<V5N86UE.PIM>2`H)&)A<V5N86UE+"`D<&%T
M:"P@)&5X="D@/2!&:6QE.CI"87-E;F%M93HZ9FEL97!A<G-E*"1N86UE+"`H
M)UPN+BHG*2D["G)E<75I<F4@1FEL93HZ4&%T:#L*1FEL93HZ4&%T:#HZ;6MP
M871H*"1P871H*2!U;FQE<W,H+64@)'!A=&@I.R`C(&UK<&%T:"!D:65S('=I
M=&@@97)R;W(*?0IS=6(@<F5Q=6ER95]M;V1U;&5S('L*<F5Q=6ER92!L:6([
M"G)E<75I<F4@1'EN84QO861E<CL*<F5Q=6ER92!I;G1E9V5R.PIR97%U:7)E
M('-T<FEC=#L*<F5Q=6ER92!W87)N:6YG<SL*<F5Q=6ER92!V87)S.PIR97%U
M:7)E($-A<G`["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E<75I<F4@17)R;F\[
M"G)E<75I<F4@17AP;W)T97(Z.DAE879Y.PIR97%U:7)E($5X<&]R=&5R.PIR
M97%U:7)E($9C;G1L.PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@1FEL
M93HZ4W!E8SL*<F5Q=6ER92!84TQO861E<CL*<F5Q=6ER92!#;VYF:6<["G)E
M<75I<F4@24\Z.DAA;F1L93L*<F5Q=6ER92!)3SHZ1FEL93L*<F5Q=6ER92!#
M;VUP<F5S<SHZ6FQI8CL*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@
M1&EG97-T.CI32$$["G)E<75I<F4@4$%2.PIR97%U:7)E(%!!4CHZ2&5A=GD[
M"G)E<75I<F4@4$%2.CI$:7-T.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0;V13
M=')I<#L*<F5Q=6ER92!005(Z.D9I;'1E<CHZ4&%T8VA#;VYT96YT.PIR97%U
M:7)E(&%T=')I8G5T97,["F5V86P@>R!R97%U:7)E($-W9"!].PIE=F%L('L@
M<F5Q=6ER92!7:6XS,B!].PIE=F%L('L@<F5Q=6ER92!38V%L87(Z.E5T:6P@
M?3L*979A;"![(')E<75I<F4@07)C:&EV93HZ56YZ:7`Z.D)U<G-T('T["F5V
M86P@>R!R97%U:7)E(%1I93HZ2&%S:#HZ3F%M961#87!T=7)E('T["F5V86P@
M>R!R97%U:7)E(%!E<FQ)3SL@<F5Q=6ER92!097)L24\Z.G-C86QA<B!].PIE
M=F%L('L@<F5Q=6ER92!U=&8X('T["GT*<W5B(%]S971?<&%R7W1E;7`@>PII
M9B`H9&5F:6YE9"`D14Y6>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@
M/7X@+R@N*RDO*2!["B1P87)?=&5M<"`]("0Q.PIR971U<FX["GT*9F]R96%C
M:"!M>2`D<&%T:"`H"BAM87`@)$5.5GLD7WTL('%W*"!005)?5$U01$E2(%1-
M4$1)4B!414U01$E2(%1%35`@5$U0("DI+`IQ=R@@0SI<7%1%35`@+W1M<"`N
M("D**2!["FYE>'0@=6YL97-S(&1E9FEN960@)'!A=&@@86YD("UD("1P871H
M(&%N9"`M=R`D<&%T:#L*;7D@)'5S97)N86UE.PIM>2`D<'=U:60["F5V86P@
M>R@D<'=U:60I(#T@9V5T<'=U:60H)#XI(&EF(&1E9FEN960@)#X[?3L*:68@
M*"!D969I;F5D*"97:6XS,CHZ3&]G:6Y.86UE*2`I('L*)'5S97)N86UE(#T@
M)E=I;C,R.CI,;V=I;DYA;64["GT*96QS:68@*&1E9FEN960@)'!W=6ED*2![
M"B1U<V5R;F%M92`]("1P=W5I9#L*?0IE;'-E('L*)'5S97)N86UE(#T@)$5.
M5GM54T523D%-17T@?'P@)$5.5GM54T52?2!\?"`G4UE35$5-)SL*?0IM>2`D
M<W1M<&1I<B`]("(D<&%T:"1S>7-[7V1E;&EM?7!A<BTB+G5N<&%C:R@B2"HB
M+"`D=7-E<FYA;64I.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*;7D@)&-A8VAE
M7V1I<CL*:68@*"1%3E9[4$%27T-,14%.?2D@>PHD8V%C:&5?9&ER(#T@(G1E
M;7`M)"0B.PI]"F5L<V4@>PIO<&5N(&UY("1F:"P@(CPZ<F%W(BP@)'!R;V=N
M86UE(&]R(&1I92!Q<5M#86XG="!R96%D("(D<')O9VYA;64B.B`D(5T["FEF
M("@H;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"1F:"DI(#X](#`I
M('L*<V5E:R`D9F@L("1M86=I8U]P;W,*+2`D1DE,15]O9F9S971?<VEZ90HM
M(&QE;F=T:"@B7#!#04-(12(I+"`P.PIM>2`D8G5F.PIR96%D("1F:"P@)&)U
M9BP@;&5N9W1H*")<,$-!0TA%(BD["FEF("@D8G5F(&5Q(")<,$-!0TA%(BD@
M>PIS965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@
M;&5N9W1H*")<,$-!0TA%(BD*+2`D8V%C:&5?;F%M95]S:7IE+"`P.PIR96%D
M("1F:"P@)&)U9BP@)&-A8VAE7VYA;65?<VEZ93L*)&)U9B`]?B!S+UPP+R]G
M.PHD8V%C:&5?9&ER(#T@(F-A8VAE+21B=68B.PI]"GT*8VQO<V4@)&9H.PI]
M"FEF("@A)&-A8VAE7V1I<BD@>PHD8V%C:&5?9&ER(#T@(G1E;7`M)"0B.PHD
M14Y6>U!!4E]#3$5!3GT@/2`Q.PI]"B1S=&UP9&ER("X]("(D<WES>U]D96QI
M;7TD8V%C:&5?9&ER(CL*;6MD:7(@)'-T;7!D:7(L(#`W-34["B1%3E9[4$%2
M7U1%35!](#T@)'-T;7!D:7(["FQA<W0["GT*)'!A<E]T96UP(#T@)#$@:68@
M)$5.5GM005)?5$5-4'T@86YD("1%3E9[4$%27U1%35!](#U^("\H+BLI+SL*
M?0IS=6(@7W-A=F5?87,@>PIM>2`H)&YA;64L("1C;VYT96YT<RP@)&UO9&4I
M(#T@0%\["FUY("1F=6QL;F%M92`]("(D<&%R7W1E;7`O)&YA;64B.PIU;FQE
M<W,@*"UE("1F=6QL;F%M92D@>PIM>2`D=&5M<&YA;64@/2`B)&9U;&QN86UE
M+B0D(CL*;7D@)&9H.PHH;W!E;B`D9F@L("<^.G)A=R<L("1T96UP;F%M92D*
M)B8@*'!R:6YT("1F:"`D8V]N=&5N=',I"B8F("AC;&]S92`D9F@I"F]R(&1I
M92!Q<5M%<G)O<B!W<FET:6YG("(D=&5M<&YA;64B.B`D(5T["F-H;6]D("1M
M;V1E+"`D=&5M<&YA;64@:68@9&5F:6YE9"`D;6]D93L*<F5N86UE*"1T96UP
M;F%M92P@)&9U;&QN86UE*2!O<B!U;FQI;FLH)'1E;7!N86UE*3L*?0IR971U
M<FX@)&9U;&QN86UE.PI]"G-U8B!?<V5T7W!R;V=N86UE('L*:68@*&1E9FEN
M960@)$5.5GM005)?4%)/1TY!345](&%N9"`D14Y6>U!!4E]04D]'3D%-17T@
M/7X@+R@N*RDO*2!["B1P<F]G;F%M92`]("0Q.PI]"B1P<F]G;F%M92!\?#T@
M)#`["FEF("@D14Y6>U!!4E]414U0?2!A;F0@:6YD97@H)'!R;V=N86UE+"`D
M14Y6>U!!4E]414U0?2D@/CT@,"D@>PHD<')O9VYA;64@/2!S=6)S='(H)'!R
M;V=N86UE+"!R:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI("L@,2D[
M"GT*:68@*"$D14Y6>U!!4E]04D]'3D%-17T@;W(@:6YD97@H)'!R;V=N86UE
M+"`D<WES>U]D96QI;7TI(#X](#`I('L*:68@*&]P96X@;7D@)&9H+"`G/"<L
M("1P<F]G;F%M92D@>PIR971U<FX@:68@+7,@)&9H.PI]"FEF("@M<R`B)'!R
M;V=N86UE)'-Y<WM?97AE?2(I('L*)'!R;V=N86UE("X]("1S>7-[7V5X97T[
M"G)E='5R;CL*?0I]"F9O<F5A8V@@;7D@)&1I<B`H<W!L:70@+UQ1)'-Y<WMP
M871H7W-E<'U<12\L("1%3E9[4$%42'TI('L*;F5X="!I9B!E>&ES=',@)$5.
M5GM005)?5$5-4'T@86YD("1D:7(@97$@)$5.5GM005)?5$5-4'T["B1D:7(@
M/7X@<R]<421S>7-[7V1E;&EM?5Q%)"\O.PHH*"1P<F]G;F%M92`]("(D9&ER
M)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(I+"!L87-T*0II9B`M
M<R`B)&1I<B1S>7-[7V1E;&EM?21P<F]G;F%M921S>7-[7V5X97TB.PHH*"1P
M<F]G;F%M92`]("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE(BDL(&QA<W0I
M"FEF("US("(D9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE(CL*?0I]"G-U8B!?
M9FEX7W!R;V=N86UE('L*)#`@/2`D<')O9VYA;64@?'P]("1%3E9[4$%27U!2
M3T=.04U%?3L*:68@*&EN9&5X*"1P<F]G;F%M92P@)'-Y<WM?9&5L:6U]*2`\
M(#`I('L*)'!R;V=N86UE(#T@(BXD<WES>U]D96QI;7TD<')O9VYA;64B.PI]
M"FUY("1P=V0@/2`H9&5F:6YE9"`F0W=D.CIG971C=V0I(#\@0W=D.CIG971C
M=V0H*0HZ("@H9&5F:6YE9"`F5VEN,S(Z.D=E=$-W9"D@/R!7:6XS,CHZ1V5T
M0W=D*"D@.B!@<'=D8"D["F-H;VUP*"1P=V0I.PHD<')O9VYA;64@/7X@<R]>
M*#\]7"Y<+C]<421S>7-[7V1E;&EM?5Q%*2\D<'=D)'-Y<WM?9&5L:6U]+SL*
M)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE.PI]"G-U8B!?<&%R7VEN
M:71?96YV('L*:68@*"`D14Y6>U!!4E])3DE424%,25I%1'TK*R`]/2`Q("D@
M>PIR971U<FX["GT@96QS92!["B1%3E9[4$%27TE.251)04Q)6D5$?2`](#([
M"GT*9F]R("AQ=R@@4U!!5TY%1"!414U0($-,14%.($1%0E5'($-!0TA%(%!2
M3T=.04U%("D@*2!["F1E;&5T92`D14Y6>R)005)?)%\B?3L*?0IF;W(@*'%W
M+R!435!$25(@5$5-4"!#3$5!3B!$14)51R`O*2!["B1%3E9[(E!!4E\D7R)]
M(#T@)$5.5GLB4$%27T=,3T)!3%\D7R)](&EF(&5X:7-T<R`D14Y6>R)005)?
M1TQ/0D%,7R1?(GT["GT*:68@*"1%3E9[4$%27U1%35!]*2!["F1E;&5T92`D
M14Y6>U!!4E]#3$5!3GT["GT*?0IS=6(@;W5T<R!["G)E='5R;B!I9B`D<75I
M970["FEF("@D;&]G9F@I('L*<')I;G0@)&QO9V9H(")`7UQN(CL*?0IE;'-E
M('L*<')I;G0@(D!?7&XB.PI]"GT*<W5B(&EN:71?:6YC('L*<F5Q=6ER92!#
M;VYF:6<["G!U<V@@0$E.0RP@9W)E<"!D969I;F5D+"!M87`@)$-O;F9I9SHZ
M0V]N9FEG>R1??2P@<7<H"F%R8VAL:6)E>'`@<')I=FQI8F5X<"!S:71E87)C
M:&5X<"!S:71E;&EB97AP"G9E;F1O<F%R8VAE>'`@=F5N9&]R;&EB97AP"BD[
M"GT*<&%C:V%G92!M86EN.PIR97%U:7)E(%!!4CL*=6YS:&EF="!`24Y#+"!<
M)E!!4CHZ9FEN9%]P87(["E!!4BT^:6UP;W)T*$!P87)?87)G<RD["F1I92!Q
M<5MP87(N<&PZ($-A;B=T(&]P96X@<&5R;"!S8W)I<'0@(B1P<F]G;F%M92(Z
M($YO('-U8V@@9FEL92!O<B!D:7)E8W1O<GE<;ET*=6YL97-S("UE("1P<F]G
M;F%M93L*9&\@)'!R;V=N86UE.PI#3U)%.CIE>&ET*"0Q*2!I9B`H)$`@/7XO
M7E]42U]%6$E47UPH*%QD*RE<*2\I.PID:64@)$`@:68@)$`["GT["B0Z.E]?
M15)23U(@/2`D0"!I9B`D0#L*?0I#3U)%.CIE>&ET*"0Q*2!I9B`H)#HZ7U]%
M4E)/4B`]?B]>7U1+7T58251?7"@H7&0K*5PI+RD["F1I92`D.CI?7T524D]2
M(&EF("0Z.E]?15)23U(["C$["E]?14Y$7U\*`'!E<FQX<VDN8P````!#04-(
M10``/0```$`````+````0````!````!`````)@```$`````U````0`````,`
M``!`````!@```$`````4````0`````X```!`````#0```$`````E````0```
M`!(```!`````*````$`````"````0````!@```!`````"0```$`````*````
M0```````````````@;(!@;"PK`<`````A`(!@;`/L0``````A;(!@;"PKG\`
M````G(C_?["J"8!HB_]_`0```&R._W^PJQZ`Z(__?\#__W\8D?]_`0```%23
M_W^PJ`.`^)/_?P$```"PG_]_K/__?QR@_W^P__]_V*'_?["PJ(#TH?]_`0``
M`&"C_W^PKSR`2*K_?["K`H`4K?]_`0``````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````!D
M&@``\!D```$````_`P```0```$H#```!````5`,```$```!H`P``#P```/`$
M```,````+!,```T```!@2P``&0```*R=`0`;````!````!H```"PG0$`'```
M``0```#U_O]OM`$```4````("@``!@```$@#```*````$P4```L````0````
M%0`````````#````S)X!``(```#@`0``%````!$````7````3!$``!$```!T
M$```$@```-@````3````"````!@`````````^___;P$```C^__]O]`\``/__
M_V\#````\/__;QP/``#Z__]O#@``````````````````````````````````
M``````````````````````"TG0$````````````X$P``.!,``#@3```X$P``
M.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X
M$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3
M```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,`
M`#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``
M.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3````
M````````````````````````````````````````````````````````````
M````````(!8````````$H`$`"E!!4BYP;0H`````1$\``%A,````````&$T`
M`!Q-```X3P``8$P``$!/````````)$\``"Q/````````3$\``$=#0SH@*$=.
M52D@,3(N,2XP`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(U+C`N,CA?8C@X9#,S
M."D@,3(N,2XP`$$T````865A8FD``2H````%-RU!``8*!T$(`0D""@,,`1($
M$P$4`14!%P,8`1D!&@(<`2(!`"YS:'-T<G1A8@`N:6YT97)P`"YN;W1E+F=N
M=2YB=6EL9"UI9``N;F]T92Y!0DDM=&%G`"YG;G4N:&%S:``N9'EN<WEM`"YD
M>6YS='(`+F=N=2YV97)S:6]N`"YG;G4N=F5R<VEO;E]R`"YR96PN9'EN`"YR
M96PN<&QT`"YI;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+D%232YE>'1A8@`N
M05)-+F5X:61X`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A<G)A>0`N
M9'EN86UI8P`N9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+D%232YA='1R:6)U
M=&5S```````````````````````````````````````````````````````+
M`````0````(```!4`0``5`$``!D```````````````$`````````$P````<`
M```"````<`$``'`!```D```````````````$`````````"8````'`````@``
M`)0!``"4`0``(```````````````!``````````T````]O__;P(```"T`0``
MM`$``)0!```%``````````0````$````/@````L````"````2`,``$@#``#`
M!@``!@````,````$````$````$8````#`````@````@*```("@``$P4`````
M`````````0````````!.````____;P(````<#P``'`\``-@````%````````
M``(````"````6P```/[__V\"````]`\``/0/``"`````!@````,````$````
M`````&H````)`````@```'00``!T$```V`````4`````````!`````@```!S
M````"0```$(```!,$0``3!$``.`!```%````%@````0````(````?`````$`
M```&````+!,``"P3```,```````````````$`````````'<````!````!@``
M`#@3```X$P``Y`(`````````````!`````0```""`````0````8````@%@``
M(!8``$`U``````````````@`````````B`````$````&````8$L``&!+```(
M```````````````$`````````(X````!`````@```&A+``!H2P``^$$`````
M````````!`````````"6`````0````(```!@C0``8(T``"0`````````````
M``0`````````H0````$``'""````A(T``(2-``!P````#0`````````$````
M`````*P````!`````@```/2-``#TC0``!```````````````!`````````"V
M````#@````,```"LG0$`K)T```0```````````````0````$````P@````\`
M```#````L)T!`+"=```$```````````````$````!````,X````&`````P``
M`+2=`0"TG0``&`$```8`````````!`````@```#7`````0````,```#,G@$`
MS)X``#0!``````````````0````$````W`````$````#`````*`!``"@``!(
M```````````````$`````````.(````(`````P```$B@`0!(H```%```````
M````````!`````````#G`````0```#``````````2*```#\`````````````
M``$````!````\`````,``'```````````(>@```U```````````````!````
L``````$````#``````````````"\H`````$``````````````0``````````
