"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.serializeMessageHeaderAuth = exports.serializeFactory = void 0;
/*
 * This public interface for serializing the AWS Encryption SDK Message Header Format
 * is provided for the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other than the
 * Encryption SDK for JavaScript.
 *
 * See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-structure
 */
const concat_buffers_1 = require("./concat_buffers");
const material_management_1 = require("@aws-crypto/material-management");
const identifiers_1 = require("./identifiers");
const uint_util_1 = require("./uint_util");
function serializeFactory(fromUtf8) {
    return {
        frameIv,
        nonFramedBodyIv,
        headerAuthIv,
        frameHeader,
        finalFrameHeader,
        encodeEncryptionContext,
        serializeEncryptionContext,
        serializeEncryptedDataKeys,
        serializeEncryptedDataKey,
        serializeMessageHeader,
        buildMessageHeader,
    };
    function frameIv(ivLength, sequenceNumber) {
        /* Precondition: sequenceNumber must conform to the specification. i.e. 1 - (2^32 - 1)
         * The sequence number starts at 1
         * https://github.com/awslabs/aws-encryption-sdk-specification/blob/master/data-format/message-body.md#sequence-number
         */
        (0, material_management_1.needs)(sequenceNumber > 0 &&
            identifiers_1.SequenceIdentifier.SEQUENCE_NUMBER_END >= sequenceNumber, 'sequenceNumber out of bounds');
        const buff = new Uint8Array(ivLength);
        const view = new DataView(buff.buffer, buff.byteOffset, buff.byteLength);
        view.setUint32(ivLength - 4, sequenceNumber, false); // big-endian
        return buff;
    }
    function nonFramedBodyIv(ivLength) {
        return frameIv(ivLength, 1);
    }
    function headerAuthIv(ivLength) {
        return new Uint8Array(ivLength); // new Uint8Array is 0 filled by default
    }
    function frameHeader(sequenceNumber, iv) {
        return (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt32BE)(sequenceNumber), iv);
    }
    function finalFrameHeader(sequenceNumber, iv, contentLength) {
        return (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt32BE)(identifiers_1.SequenceIdentifier.SEQUENCE_NUMBER_END), // Final Frame identifier
        (0, uint_util_1.uInt32BE)(sequenceNumber), iv, (0, uint_util_1.uInt32BE)(contentLength));
    }
    function encodeEncryptionContext(encryptionContext) {
        return (Object.entries(encryptionContext)
            /* Precondition: The serialized encryption context entries must be sorted by UTF-8 key value. */
            .sort(([aKey], [bKey]) => aKey.localeCompare(bKey))
            .map((entries) => entries.map(fromUtf8))
            .map(([key, value]) => (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt16BE)(key.byteLength), key, (0, uint_util_1.uInt16BE)(value.byteLength), value)));
    }
    function serializeEncryptionContext(encryptionContext) {
        const encryptionContextElements = encodeEncryptionContext(encryptionContext);
        /* Check for early return (Postcondition): If there is no context then the length of the _whole_ serialized portion is 0.
         * This is part of the specification of the AWS Encryption SDK Message Format.
         * It is not 0 for length and 0 for count.  The count element is omitted.
         */
        if (!encryptionContextElements.length)
            return (0, uint_util_1.uInt16BE)(0);
        const aadData = (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt16BE)(encryptionContextElements.length), ...encryptionContextElements);
        const aadLength = (0, uint_util_1.uInt16BE)(aadData.byteLength);
        return (0, concat_buffers_1.concatBuffers)(aadLength, aadData);
    }
    function serializeEncryptedDataKeys(encryptedDataKeys) {
        const encryptedKeyInfo = encryptedDataKeys.map(serializeEncryptedDataKey);
        return (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt16BE)(encryptedDataKeys.length), ...encryptedKeyInfo);
    }
    function serializeEncryptedDataKey(edk) {
        const { providerId, providerInfo, encryptedDataKey, rawInfo } = edk;
        const providerIdBytes = fromUtf8(providerId);
        // The providerInfo is technically a binary field, so I prefer rawInfo
        const providerInfoBytes = rawInfo || fromUtf8(providerInfo);
        return (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt16BE)(providerIdBytes.byteLength), providerIdBytes, (0, uint_util_1.uInt16BE)(providerInfoBytes.byteLength), providerInfoBytes, (0, uint_util_1.uInt16BE)(encryptedDataKey.byteLength), encryptedDataKey);
    }
    function serializeMessageHeader(messageHeader) {
        /* Precondition: Must be a version that can be serialized. */
        (0, material_management_1.needs)(identifiers_1.SerializationVersion[messageHeader.version], 'Unsupported version.');
        if (messageHeader.version === 1) {
            return serializeMessageHeaderV1(messageHeader);
        }
        else {
            return serializeMessageHeaderV2(messageHeader);
        }
    }
    function serializeMessageHeaderV1(messageHeader) {
        return (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt8)(messageHeader.version), (0, uint_util_1.uInt8)(messageHeader.type), (0, uint_util_1.uInt16BE)(messageHeader.suiteId), messageHeader.messageId, serializeEncryptionContext(messageHeader.encryptionContext), serializeEncryptedDataKeys(messageHeader.encryptedDataKeys), new Uint8Array([messageHeader.contentType]), new Uint8Array([0, 0, 0, 0]), (0, uint_util_1.uInt8)(messageHeader.headerIvLength), (0, uint_util_1.uInt32BE)(messageHeader.frameLength));
    }
    function serializeMessageHeaderV2(messageHeader) {
        return (0, concat_buffers_1.concatBuffers)((0, uint_util_1.uInt8)(messageHeader.version), (0, uint_util_1.uInt16BE)(messageHeader.suiteId), messageHeader.messageId, serializeEncryptionContext(messageHeader.encryptionContext), serializeEncryptedDataKeys(messageHeader.encryptedDataKeys), new Uint8Array([messageHeader.contentType]), (0, uint_util_1.uInt32BE)(messageHeader.frameLength), messageHeader.suiteData);
    }
    /* This _could_ take the material directly.
     * But I don't do that on purpose.
     * It may be overly paranoid,
     * but this way once the material is created,
     * it has a minimum of egress.
     */
    function buildMessageHeader({ encryptionContext, encryptedDataKeys, suite, messageId, frameLength, suiteData, }) {
        const { messageFormat: version, id: suiteId } = suite;
        const contentType = identifiers_1.ContentType.FRAMED_DATA;
        if (version === material_management_1.MessageFormat.V1) {
            const type = identifiers_1.ObjectType.CUSTOMER_AE_DATA;
            const { ivLength: headerIvLength } = suite;
            return {
                version,
                type,
                suiteId,
                messageId,
                encryptionContext,
                encryptedDataKeys,
                contentType,
                headerIvLength,
                frameLength,
            };
        }
        else if (version === material_management_1.MessageFormat.V2) {
            return {
                version,
                suiteId,
                messageId,
                encryptionContext: encryptionContext,
                encryptedDataKeys: encryptedDataKeys,
                contentType,
                frameLength,
                suiteData,
            };
        }
        (0, material_management_1.needs)(false, 'Unsupported message format version.');
    }
}
exports.serializeFactory = serializeFactory;
function serializeMessageHeaderAuth({ headerIv, headerAuthTag, messageHeader, }) {
    if (messageHeader.version === material_management_1.MessageFormat.V1) {
        return (0, concat_buffers_1.concatBuffers)(headerIv, headerAuthTag);
    }
    return headerAuthTag;
}
exports.serializeMessageHeaderAuth = serializeMessageHeaderAuth;
//# sourceMappingURL=data:application/json;base64,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