/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

'use strict';

const { stringify } = require('querystring');
const debug = require('debug')('opensearch');
const sjson = require('secure-json-parse');
const { SerializationError, DeserializationError } = require('./errors');
const kJsonOptions = Symbol('secure json parse options');

class Serializer {
  constructor(opts = {}) {
    const disable = opts.disablePrototypePoisoningProtection;
    this[kJsonOptions] = {
      protoAction: disable === true || disable === 'proto' ? 'ignore' : 'error',
      constructorAction: disable === true || disable === 'constructor' ? 'ignore' : 'error',
    };
  }

  serialize(object) {
    debug('Serializing', object);
    let json;
    try {
      json = JSON.stringify(object);
    } catch (err) {
      throw new SerializationError(err.message, object);
    }
    return json;
  }

  deserialize(json) {
    debug('Deserializing', json);
    let object;
    try {
      object = sjson.parse(json, this[kJsonOptions]);
    } catch (err) {
      throw new DeserializationError(err.message, json);
    }
    return object;
  }

  ndserialize(array) {
    debug('ndserialize', array);
    if (Array.isArray(array) === false) {
      throw new SerializationError('The argument provided is not an array', array);
    }
    let ndjson = '';
    for (let i = 0, len = array.length; i < len; i++) {
      if (typeof array[i] === 'string') {
        ndjson += array[i] + '\n';
      } else {
        ndjson += this.serialize(array[i]) + '\n';
      }
    }
    return ndjson;
  }

  qserialize(object) {
    debug('qserialize', object);
    if (object == null) return '';
    if (typeof object === 'string') return object;
    // arrays should be serialized as comma separated list
    const keys = Object.keys(object);
    for (let i = 0, len = keys.length; i < len; i++) {
      const key = keys[i];
      // OpenSearch will complain about keys without a value
      if (object[key] === undefined) {
        delete object[key];
      } else if (Array.isArray(object[key]) === true) {
        object[key] = object[key].join(',');
      }
    }
    return stringify(object);
  }
}

module.exports = Serializer;
