import { KeyValueAny, Tz, Zh, KeyValue, Logger } from './types';
export declare class ColorRGB {
    red: number;
    green: number;
    blue: number;
    /**
     * Create RGB color
     * @param {number} red
     * @param {number} green
     * @param {number} blue
     */
    constructor(red: number, green: number, blue: number);
    /**
     * Create RGB color from object
     * @param {ColorRGBT} rgb object with properties red, green and blue
     * @return {ColorRGB} new ColoRGB object
     */
    static fromObject(rgb: {
        red: number;
        green: number;
        blue: number;
    }): ColorRGB;
    /**
     * Create RGB color from hex string
     * @param {string} hex hex encoded RGB color
     * @return {ColorRGB} new ColoRGB object
     */
    static fromHex(hex: string): ColorRGB;
    /**
     * Return this color with values rounded to given precision
     * @param {number} precision decimal places to round to
     * @return {ColorRGB}
     */
    rounded(precision: number): ColorRGB;
    /**
     * Convert to Object
     * @return {ColorRGBT} object with properties red, green and blue
     */
    toObject(): {
        red: number;
        green: number;
        blue: number;
    };
    /**
     * Convert to HSV
     * @return {ColorHSV} color in HSV space
     */
    toHSV(): ColorHSV;
    /**
     * Convert to CIE
     * @return {ColorXY} color in CIE space
     */
    toXY(): ColorXY;
    /**
     * Returns color after sRGB gamma correction
     * @return {ColorRGB} corrected RGB
     */
    gammaCorrected(): ColorRGB;
    /**
     * Returns color after reverse sRGB gamma correction
     * @return {ColorRGB} raw RGB
     */
    gammaUncorrected(): ColorRGB;
    /**
     * Create hex string from RGB color
     * @return {ColorRGB} hex hex encoded RGB color
     */
    toHEX(): string;
}
/**
 *  Class representing color in CIE space
 */
export declare class ColorXY {
    x: number;
    y: number;
    /**
     * Create CIE color
     * @param {number} x
     * @param {number} y
     */
    constructor(x: number, y: number);
    /**
     * Create CIE color from object
     * @param {ColorXYT} xy object with properties x and y
     * @return {ColorXY} new ColorXY object
     */
    static fromObject(xy: {
        x: number;
        y: number;
    }): ColorXY;
    /**
     * Create XY object from color temp in mireds
     * @param {number} mireds color temp in mireds
     * @return {ColorXY} color in XY space
     */
    static fromMireds(mireds: number): ColorXY;
    /**
     * Converts color in XY space to temperature in mireds
     * @return {number} color temp in mireds
     */
    toMireds(): number;
    /**
     * Converts CIE color space to RGB color space
     * From: https://github.com/usolved/cie-rgb-converter/blob/master/cie_rgb_converter.js
     * @return {ColorRGB}
     */
    toRGB(): ColorRGB;
    /**
     * Convert to HSV
     * @return {ColorHSV} color in HSV space
     */
    toHSV(): ColorHSV;
    /**
     * Return this color with value rounded to given precision
     * @param {number} precision decimal places to round to
     * @return {ColorXY}
     */
    rounded(precision: number): ColorXY;
    /**
     * Convert to object
     * @return {ColorXYT} object with properties x and y
     */
    toObject(): {
        x: number;
        y: number;
    };
}
/**
 * Class representing color in HSV space
 */
declare class ColorHSV {
    hue: number;
    saturation: number;
    value: number;
    brightness: number;
    /**
     * Create color in HSV space
     * @param {?number} hue
     * @param {?number} [saturation=null]
     * @param {?number} [value=null]
     */
    constructor(hue: number, saturation?: number, value?: number);
    /**
     * Create HSV color from object
     * @param {object} hsv
     * @param {number} [hsv.hue]
     * @param {number} [hsv.saturation]
     * @param {number} [hsv.value]
     * @return {ColorHSV}
     */
    static fromObject(hsv: {
        hue?: number;
        saturation?: number;
        value: number;
    }): ColorHSV;
    /**
     * Create HSV color from HSL
     * @param {ColorHSL} hsl color in HSL space
     * @return {ColorHSV} color in HSV space
     */
    static fromHSL(hsl: {
        hue: number;
        saturation: number;
        lightness: number;
    }): ColorHSV;
    /**
     * Return this color with value rounded to given precision
     * @param {number} precision decimal places to round to
     * @return {ColorHSV}
     */
    rounded(precision: number): ColorHSV;
    /**
     * Convert to object
     * @param {boolean} short return h, s, v instead of hue, saturation, value
     * @param {boolean} includeValue omit v(alue) from return
     * @return {ColorHSVT}
     */
    toObject(short?: boolean, includeValue?: boolean): {
        h?: number;
        hue?: number;
        s?: number;
        saturation?: number;
        v?: number;
        value?: number;
    };
    /**
     * Convert RGB color
     * @return {ColorRGB}
     */
    toRGB(): ColorRGB;
    /**
     * Create CIE color from HSV
     * @return {ColorXY}
     */
    toXY(): ColorXY;
    /**
     * Create Mireds from HSV
     * @return {number} color temp in mireds
     */
    toMireds(): number;
    /**
     * Returns color with missing properties set to defaults
     * @return {ColorHSV} HSV color
     */
    complete(): ColorHSV;
    /**
     * Interpolates hue value based on correction map through ranged linear interpolation
     * @param {Nnmber} hue hue to be corrected
     * @param {Array} correctionMap array of hueIn -> hueOut mappings; example: [ {"in": 20, "out": 25}, {"in": 109, "out": 104}]
     * @return {number} corrected hue value
     */
    static interpolateHue(hue: number, correctionMap: KeyValueAny[]): number;
    /**
     * Applies hue interpolation if entity has hue correction data
     * @param {number} hue hue component of HSV color
     * @param {Object} meta entity meta object
     * @param {Object} meta.options meta object's options property
     * @param {Object} [meta.options.hue_correction] hue correction data
     * @return {number} corrected hue component of HSV color
     */
    static correctHue(hue: number, meta: Tz.Meta): number;
    /**
     * Returns HSV color after hue correction
     * @param {Object} meta entity meta object
     * @return {ColorHSV} hue corrected color
     */
    hueCorrected(meta: Tz.Meta): ColorHSV;
    /**
     * Returns HSV color after gamma and hue corrections
     * @param {Object} meta entity meta object
     * @return {ColorHSV} corrected color in HSV space
     */
    colorCorrected(meta: Tz.Meta): ColorHSV;
}
export declare class Color {
    hsv: ColorHSV;
    xy: ColorXY;
    rgb: ColorRGB;
    /**
     * Create Color object
     * @param {?ColorHSV} hsv ColorHSV instance
     * @param {?ColorRGB} rgb ColorRGB instance
     * @param {?ColorXY} xy ColorXY instance
     */
    constructor(hsv: ColorHSV, rgb: ColorRGB, xy: ColorXY);
    /**
     * Create Color object from converter's value argument
     * @param {Object} value converter value argument
     * @return {Color} Color object
     */
    static fromConverterArg(value: any): Color;
    /**
     * Returns true if color is HSV
     * @return {boolean} is HSV color
     */
    isHSV(): boolean;
    /**
     * Returns true if color is RGB
     * @return {boolean} is RGB color
     */
    isRGB(): boolean;
    /**
     * Returns true if color is XY
     * @return {boolean} is XY color
     */
    isXY(): boolean;
}
export declare function syncColorState(newState: KeyValueAny, oldState: KeyValueAny, endpoint: Zh.Endpoint | Zh.Group, options: KeyValue, logger: Logger): KeyValueAny;
export {};
//# sourceMappingURL=color.d.ts.map