"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var fs = require("fs-extra");
var path = require("path");
var _ = require("lodash");
var file_engine_1 = require("./file.engine");
var marked = require('8fold-marked');
var MarkdownEngine = /** @class */ (function () {
    function MarkdownEngine(fileEngine) {
        if (fileEngine === void 0) { fileEngine = new file_engine_1.FileEngine(); }
        var _this = this;
        this.fileEngine = fileEngine;
        /**
         * List of markdown files without .md extension
         */
        this.markdownFiles = [
            'README',
            'CHANGELOG',
            'LICENSE',
            'CONTRIBUTING',
            'TODO'
        ];
        var renderer = new marked.Renderer();
        renderer.code = function (code, language) {
            var highlighted = code;
            if (!language) {
                language = 'none';
            }
            highlighted = _this.escape(code);
            return "<pre class=\"line-numbers\"><code class=\"language-" + language + "\">" + highlighted + "</code></pre>";
        };
        renderer.table = function (header, body) {
            return '<table class="table table-bordered compodoc-table">\n'
                + '<thead>\n'
                + header
                + '</thead>\n'
                + '<tbody>\n'
                + body
                + '</tbody>\n'
                + '</table>\n';
        };
        var self = this;
        renderer.image = function (href, title, text) {
            var out = '<img src="' + href + '" alt="' + text + '" class="img-responsive"';
            if (title) {
                out += ' title="' + title + '"';
            }
            return out;
        };
        marked.setOptions({
            renderer: renderer,
            breaks: false
        });
    }
    MarkdownEngine.prototype.getTraditionalMarkdown = function (filepath) {
        var _this = this;
        return this.fileEngine.get(process.cwd() + path.sep + filepath + '.md')
            .catch(function (err) { return _this.fileEngine.get(process.cwd() + path.sep + filepath).then(); })
            .then(function (data) { return marked(data); });
    };
    MarkdownEngine.prototype.getReadmeFile = function () {
        return this.fileEngine.get(process.cwd() + path.sep + 'README.md').then(function (data) { return marked(data); });
    };
    MarkdownEngine.prototype.readNeighbourReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        return fs.readFileSync(readmeFile, 'utf8');
    };
    MarkdownEngine.prototype.hasNeighbourReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        return this.fileEngine.existsSync(readmeFile);
    };
    MarkdownEngine.prototype.componentReadmeFile = function (file) {
        var dirname = path.dirname(file);
        var readmeFile = dirname + path.sep + 'README.md';
        var readmeAlternativeFile = dirname + path.sep + path.basename(file, '.ts') + '.md';
        var finalPath = '';
        if (this.fileEngine.existsSync(readmeFile)) {
            finalPath = readmeFile;
        }
        else {
            finalPath = readmeAlternativeFile;
        }
        return finalPath;
    };
    /**
     * Checks if any of the markdown files is exists with or without endings
     */
    MarkdownEngine.prototype.hasRootMarkdowns = function () {
        var _this = this;
        return this.addEndings(this.markdownFiles)
            .some(function (x) { return _this.fileEngine.existsSync(process.cwd() + path.sep + x); });
    };
    MarkdownEngine.prototype.listRootMarkdowns = function () {
        var _this = this;
        var foundFiles = this.markdownFiles
            .filter(function (x) {
            return _this.fileEngine.existsSync(process.cwd() + path.sep + x + '.md') ||
                _this.fileEngine.existsSync(process.cwd() + path.sep + x);
        });
        return this.addEndings(foundFiles);
    };
    MarkdownEngine.prototype.escape = function (html) {
        return html
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;')
            .replace(/@/g, '&#64;');
    };
    /**
     * ['README'] => ['README', 'README.md']
     */
    MarkdownEngine.prototype.addEndings = function (files) {
        return _.flatMap(files, function (x) { return [x, x + '.md']; });
    };
    return MarkdownEngine;
}());
exports.MarkdownEngine = MarkdownEngine;
//# sourceMappingURL=markdown.engine.js.map