/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.cluster.decommission;

import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.OpenSearchTimeoutException;
import org.opensearch.action.ActionListener;
import org.opensearch.action.admin.cluster.configuration.VotingConfigExclusionsHelper;
import org.opensearch.action.admin.cluster.node.stats.NodeStats;
import org.opensearch.action.admin.cluster.node.stats.NodesStatsRequest;
import org.opensearch.action.admin.cluster.node.stats.NodesStatsResponse;
import org.opensearch.cluster.ClusterState;
import org.opensearch.cluster.ClusterStateObserver;
import org.opensearch.cluster.ClusterStateTaskConfig;
import org.opensearch.cluster.ClusterStateUpdateTask;
import org.opensearch.cluster.coordination.NodeRemovalClusterStateTaskExecutor;
import org.opensearch.cluster.decommission.DecommissionAttributeMetadata;
import org.opensearch.cluster.decommission.DecommissionStatus;
import org.opensearch.cluster.metadata.Metadata;
import org.opensearch.cluster.node.DiscoveryNode;
import org.opensearch.cluster.routing.allocation.AllocationService;
import org.opensearch.cluster.service.ClusterService;
import org.opensearch.common.Priority;
import org.opensearch.common.io.stream.StreamInput;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.http.HttpStats;
import org.opensearch.threadpool.ThreadPool;
import org.opensearch.transport.TransportException;
import org.opensearch.transport.TransportResponseHandler;
import org.opensearch.transport.TransportService;

public class DecommissionController {
    private static final Logger logger = LogManager.getLogger(DecommissionController.class);
    private final NodeRemovalClusterStateTaskExecutor nodeRemovalExecutor;
    private final ClusterService clusterService;
    private final TransportService transportService;
    private final ThreadPool threadPool;

    DecommissionController(ClusterService clusterService, TransportService transportService, AllocationService allocationService, ThreadPool threadPool) {
        this.clusterService = clusterService;
        this.transportService = transportService;
        this.nodeRemovalExecutor = new NodeRemovalClusterStateTaskExecutor(allocationService, logger);
        this.threadPool = threadPool;
    }

    public synchronized void removeDecommissionedNodes(final Set<DiscoveryNode> nodesToBeDecommissioned, String reason, TimeValue timeout, final ActionListener<Void> nodesRemovedListener) {
        LinkedHashMap nodesDecommissionTasks = new LinkedHashMap(nodesToBeDecommissioned.size());
        nodesToBeDecommissioned.forEach(discoveryNode -> {
            NodeRemovalClusterStateTaskExecutor.Task task = new NodeRemovalClusterStateTaskExecutor.Task((DiscoveryNode)discoveryNode, reason);
            nodesDecommissionTasks.put(task, this.nodeRemovalExecutor);
        });
        logger.info("submitting state update task to remove [{}] nodes due to decommissioning", (Object)nodesToBeDecommissioned.toString());
        this.clusterService.submitStateUpdateTasks("node-decommissioned", nodesDecommissionTasks, ClusterStateTaskConfig.build(Priority.URGENT), this.nodeRemovalExecutor);
        Predicate<ClusterState> allDecommissionedNodesRemovedPredicate = clusterState -> {
            Set intersection = Arrays.stream(clusterState.nodes().getNodes().values().toArray(new DiscoveryNode[0])).collect(Collectors.toSet());
            intersection.retainAll(nodesToBeDecommissioned);
            return intersection.size() == 0;
        };
        ClusterStateObserver observer = new ClusterStateObserver(this.clusterService, timeout, logger, this.threadPool.getThreadContext());
        ClusterStateObserver.Listener removalListener = new ClusterStateObserver.Listener(){

            @Override
            public void onNewClusterState(ClusterState state) {
                logger.info("successfully removed all decommissioned nodes [{}] from the cluster", (Object)nodesToBeDecommissioned.toString());
                nodesRemovedListener.onResponse(null);
            }

            @Override
            public void onClusterServiceClose() {
                logger.warn("cluster service closed while waiting for removal of decommissioned nodes [{}]", (Object)nodesToBeDecommissioned.toString());
            }

            @Override
            public void onTimeout(TimeValue timeout) {
                logger.info("timed out [{}] while waiting for removal of decommissioned nodes [{}]", (Object)timeout.toString(), (Object)nodesToBeDecommissioned.toString());
                nodesRemovedListener.onFailure((Exception)((Object)new OpenSearchTimeoutException("timed out [{}] while waiting for removal of decommissioned nodes [{}]", timeout.toString(), nodesToBeDecommissioned.toString())));
            }
        };
        if (allDecommissionedNodesRemovedPredicate.test(this.clusterService.getClusterApplierService().state())) {
            removalListener.onNewClusterState(this.clusterService.getClusterApplierService().state());
        } else {
            observer.waitForNextChange(removalListener, allDecommissionedNodesRemovedPredicate);
        }
    }

    public void updateMetadataWithDecommissionStatus(final DecommissionStatus decommissionStatus, final ActionListener<DecommissionStatus> listener) {
        this.clusterService.submitStateUpdateTask("update-decommission-status", new ClusterStateUpdateTask(Priority.URGENT){

            @Override
            public ClusterState execute(ClusterState currentState) {
                DecommissionAttributeMetadata decommissionAttributeMetadata = currentState.metadata().decommissionAttributeMetadata();
                assert (decommissionAttributeMetadata != null && decommissionAttributeMetadata.decommissionAttribute() != null);
                logger.info("attempting to update current decommission status [{}] with expected status [{}]", (Object)decommissionAttributeMetadata.status(), (Object)decommissionStatus);
                decommissionAttributeMetadata.validateNewStatus(decommissionStatus);
                decommissionAttributeMetadata = new DecommissionAttributeMetadata(decommissionAttributeMetadata.decommissionAttribute(), decommissionStatus, decommissionAttributeMetadata.requestID());
                ClusterState newState = ClusterState.builder(currentState).metadata(Metadata.builder(currentState.metadata()).decommissionAttributeMetadata(decommissionAttributeMetadata)).build();
                if (decommissionStatus.equals((Object)DecommissionStatus.SUCCESSFUL) || decommissionStatus.equals((Object)DecommissionStatus.FAILED)) {
                    newState = VotingConfigExclusionsHelper.clearExclusionsAndGetState(newState);
                }
                return newState;
            }

            @Override
            public void onFailure(String source, Exception e) {
                listener.onFailure(e);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                DecommissionAttributeMetadata decommissionAttributeMetadata = newState.metadata().decommissionAttributeMetadata();
                assert (decommissionAttributeMetadata != null);
                assert (decommissionAttributeMetadata.status().equals((Object)decommissionStatus));
                listener.onResponse(decommissionAttributeMetadata.status());
            }
        });
    }

    private void logActiveConnections(NodesStatsResponse nodesStatsResponse) {
        if (nodesStatsResponse == null || nodesStatsResponse.getNodes() == null) {
            logger.info("Node stats response received is null/empty.");
            return;
        }
        HashMap<String, Long> nodeActiveConnectionMap = new HashMap<String, Long>();
        List responseNodes = nodesStatsResponse.getNodes();
        for (int i = 0; i < responseNodes.size(); ++i) {
            HttpStats httpStats = ((NodeStats)responseNodes.get(i)).getHttp();
            DiscoveryNode node = ((NodeStats)responseNodes.get(i)).getNode();
            nodeActiveConnectionMap.put(node.getId(), httpStats.getServerOpen());
        }
        logger.info("Decommissioning node with connections : [{}]", nodeActiveConnectionMap);
    }

    void getActiveRequestCountOnDecommissionedNodes(Set<DiscoveryNode> decommissionedNodes) {
        if (decommissionedNodes == null || decommissionedNodes.isEmpty()) {
            return;
        }
        String[] nodes = (String[])decommissionedNodes.stream().map(DiscoveryNode::getId).toArray(String[]::new);
        if (nodes.length == 0) {
            return;
        }
        NodesStatsRequest nodesStatsRequest = new NodesStatsRequest(nodes);
        nodesStatsRequest.clear();
        nodesStatsRequest.addMetric(NodesStatsRequest.Metric.HTTP.metricName());
        this.transportService.sendRequest(this.transportService.getLocalNode(), "cluster:monitor/nodes/stats", nodesStatsRequest, new TransportResponseHandler<NodesStatsResponse>(){

            @Override
            public void handleResponse(NodesStatsResponse response) {
                DecommissionController.this.logActiveConnections(response);
            }

            @Override
            public void handleException(TransportException exp) {
                logger.error("Failure occurred while dumping connection for decommission nodes - ", exp.unwrapCause());
            }

            @Override
            public String executor() {
                return "same";
            }

            public NodesStatsResponse read(StreamInput in) throws IOException {
                return new NodesStatsResponse(in);
            }
        });
    }
}

