/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.repositories.s3;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.NoSuchFileException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.opensearch.common.collect.Tuple;
import org.opensearch.common.util.io.IOUtils;
import org.opensearch.repositories.s3.AmazonS3Reference;
import org.opensearch.repositories.s3.S3BlobStore;
import org.opensearch.repositories.s3.SocketAccess;
import org.opensearch.repositories.s3.utils.HttpRangeUtils;
import software.amazon.awssdk.core.ResponseInputStream;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.GetObjectResponse;
import software.amazon.awssdk.services.s3.model.S3Exception;

class S3RetryingInputStream
extends InputStream {
    private static final Logger logger = LogManager.getLogger(S3RetryingInputStream.class);
    static final int MAX_SUPPRESSED_EXCEPTIONS = 10;
    private final S3BlobStore blobStore;
    private final String blobKey;
    private final long start;
    private final long end;
    private final int maxAttempts;
    private final List<IOException> failures;
    private ResponseInputStream<GetObjectResponse> currentStream;
    private final AtomicBoolean isStreamAborted = new AtomicBoolean();
    private long currentStreamLastOffset;
    private int attempt = 1;
    private long currentOffset;
    private boolean closed;
    private boolean eof;

    S3RetryingInputStream(S3BlobStore blobStore, String blobKey) throws IOException {
        this(blobStore, blobKey, 0L, 0x7FFFFFFFFFFFFFFEL);
    }

    S3RetryingInputStream(S3BlobStore blobStore, String blobKey, long start, long end) throws IOException {
        if (start < 0L) {
            throw new IllegalArgumentException("start must be non-negative");
        }
        if (end < start || end == Long.MAX_VALUE) {
            throw new IllegalArgumentException("end must be >= start and not Long.MAX_VALUE");
        }
        this.blobStore = blobStore;
        this.blobKey = blobKey;
        this.maxAttempts = blobStore.getMaxRetries() + 1;
        this.failures = new ArrayList<IOException>(10);
        this.start = start;
        this.end = end;
        this.openStream();
    }

    private void openStream() throws IOException {
        try (AmazonS3Reference clientReference = this.blobStore.clientReference();){
            GetObjectRequest.Builder getObjectRequest = GetObjectRequest.builder().bucket(this.blobStore.bucket()).key(this.blobKey).overrideConfiguration(o -> o.addMetricPublisher(this.blobStore.getStatsMetricPublisher().getObjectMetricPublisher));
            if (this.currentOffset > 0L || this.start > 0L || this.end < 0x7FFFFFFFFFFFFFFEL) {
                assert (this.start + this.currentOffset <= this.end) : "requesting beyond end, start = " + this.start + " offset=" + this.currentOffset + " end=" + this.end;
                getObjectRequest.range(HttpRangeUtils.toHttpRangeHeader(Math.addExact(this.start, this.currentOffset), this.end));
            }
            ResponseInputStream getObjectResponseInputStream = SocketAccess.doPrivileged(() -> ((S3Client)clientReference.get()).getObject((GetObjectRequest)getObjectRequest.build()));
            this.currentStreamLastOffset = Math.addExact(Math.addExact(this.start, this.currentOffset), this.getStreamLength((GetObjectResponse)getObjectResponseInputStream.response()));
            this.currentStream = getObjectResponseInputStream;
            this.isStreamAborted.set(false);
        }
        catch (SdkException e) {
            if (e instanceof S3Exception && 404 == ((S3Exception)e).statusCode()) {
                throw this.addSuppressedExceptions(new NoSuchFileException("Blob object [" + this.blobKey + "] not found: " + e.getMessage()));
            }
            throw this.addSuppressedExceptions(e);
        }
    }

    private long getStreamLength(GetObjectResponse getObjectResponse) {
        try {
            if (getObjectResponse.contentRange() != null) {
                Tuple<Long, Long> s3ResponseRange = HttpRangeUtils.fromHttpRangeHeader(getObjectResponse.contentRange());
                assert ((Long)s3ResponseRange.v2() >= (Long)s3ResponseRange.v1()) : s3ResponseRange.v2() + " vs " + s3ResponseRange.v1();
                assert ((Long)s3ResponseRange.v1() == this.start + this.currentOffset) : "Content-Range start value [" + s3ResponseRange.v1() + "] exceeds start [" + this.start + "] + current offset [" + this.currentOffset + "]";
                assert ((Long)s3ResponseRange.v2() == this.end) : "Content-Range end value [" + s3ResponseRange.v2() + "] exceeds end [" + this.end + "]";
                return (Long)s3ResponseRange.v2() - (Long)s3ResponseRange.v1() + 1L;
            }
            return getObjectResponse.contentLength();
        }
        catch (Exception e) {
            assert (false) : e;
            return 0x7FFFFFFFFFFFFFFEL;
        }
    }

    @Override
    public int read() throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int result = this.currentStream.read();
                if (result == -1) {
                    this.eof = true;
                    return -1;
                }
                ++this.currentOffset;
                return result;
            }
            catch (IOException e) {
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        this.ensureOpen();
        while (true) {
            try {
                int bytesRead = this.currentStream.read(b, off, len);
                if (bytesRead == -1) {
                    this.eof = true;
                    return -1;
                }
                this.currentOffset += (long)bytesRead;
                return bytesRead;
            }
            catch (IOException e) {
                this.reopenStreamOrFail(e);
                continue;
            }
            break;
        }
    }

    private void ensureOpen() {
        if (this.closed) {
            assert (false) : "using S3RetryingInputStream after close";
            throw new IllegalStateException("using S3RetryingInputStream after close");
        }
    }

    private void reopenStreamOrFail(IOException e) throws IOException {
        if (this.attempt >= this.maxAttempts) {
            logger.debug((Message)new ParameterizedMessage("failed reading [{}/{}] at offset [{}], attempt [{}] of [{}], giving up", new Object[]{this.blobStore.bucket(), this.blobKey, this.start + this.currentOffset, this.attempt, this.maxAttempts}), (Throwable)e);
            throw this.addSuppressedExceptions(e);
        }
        logger.debug((Message)new ParameterizedMessage("failed reading [{}/{}] at offset [{}], attempt [{}] of [{}], retrying", new Object[]{this.blobStore.bucket(), this.blobKey, this.start + this.currentOffset, this.attempt, this.maxAttempts}), (Throwable)e);
        ++this.attempt;
        if (this.failures.size() < 10) {
            this.failures.add(e);
        }
        this.maybeAbort(this.currentStream);
        IOUtils.closeWhileHandlingException(this.currentStream);
        this.openStream();
    }

    @Override
    public void close() throws IOException {
        this.maybeAbort(this.currentStream);
        try {
            this.currentStream.close();
        }
        finally {
            this.closed = true;
        }
    }

    private void maybeAbort(ResponseInputStream<GetObjectResponse> stream) {
        if (this.isEof() || this.isAborted()) {
            return;
        }
        try {
            if (this.start + this.currentOffset < this.currentStreamLastOffset) {
                stream.abort();
                this.isStreamAborted.compareAndSet(false, true);
            }
        }
        catch (Exception e) {
            logger.warn("Failed to abort stream before closing", (Throwable)e);
        }
    }

    @Override
    public long skip(long n) {
        throw new UnsupportedOperationException("S3RetryingInputStream does not support seeking");
    }

    @Override
    public void reset() {
        throw new UnsupportedOperationException("S3RetryingInputStream does not support seeking");
    }

    private <T extends Exception> T addSuppressedExceptions(T e) {
        for (IOException failure : this.failures) {
            e.addSuppressed(failure);
        }
        return e;
    }

    boolean isEof() {
        return this.eof || this.start + this.currentOffset == this.currentStreamLastOffset;
    }

    boolean isAborted() {
        return this.isStreamAborted.get();
    }
}

