import { __awaiter, __extends, __generator } from "tslib";
import { QueryRunnerAlreadyReleasedError } from "../../error/QueryRunnerAlreadyReleasedError";
import { QueryFailedError } from "../../error/QueryFailedError";
import { AbstractSqliteQueryRunner } from "../sqlite-abstract/AbstractSqliteQueryRunner";
import { Broadcaster } from "../../subscriber/Broadcaster";
import { ConnectionIsNotSetError } from '../../error/ConnectionIsNotSetError';
import { QueryResult } from "../../query-runner/QueryResult";
/**
 * Runs queries on a single sqlite database connection.
 *
 * Does not support compose primary keys with autoincrement field.
 * todo: need to throw exception for this case.
 */
var SqliteQueryRunner = /** @class */ (function (_super) {
    __extends(SqliteQueryRunner, _super);
    // -------------------------------------------------------------------------
    // Constructor
    // -------------------------------------------------------------------------
    function SqliteQueryRunner(driver) {
        var _this = _super.call(this) || this;
        _this.driver = driver;
        _this.connection = driver.connection;
        _this.broadcaster = new Broadcaster(_this);
        return _this;
    }
    /**
     * Executes a given SQL query.
     */
    SqliteQueryRunner.prototype.query = function (query, parameters, useStructuredResult) {
        var _this = this;
        if (useStructuredResult === void 0) { useStructuredResult = false; }
        if (this.isReleased)
            throw new QueryRunnerAlreadyReleasedError();
        var connection = this.driver.connection;
        var options = connection.options;
        var maxQueryExecutionTime = this.driver.options.maxQueryExecutionTime;
        if (!connection.isConnected) {
            throw new ConnectionIsNotSetError('sqlite');
        }
        return new Promise(function (ok, fail) { return __awaiter(_this, void 0, void 0, function () {
            var databaseConnection, queryStartTime, isInsertQuery, isDeleteQuery, isUpdateQuery, execute, handler;
            var _this = this;
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0: return [4 /*yield*/, this.connect()];
                    case 1:
                        databaseConnection = _a.sent();
                        this.driver.connection.logger.logQuery(query, parameters, this);
                        queryStartTime = +new Date();
                        isInsertQuery = query.startsWith("INSERT ");
                        isDeleteQuery = query.startsWith("DELETE ");
                        isUpdateQuery = query.startsWith("UPDATE ");
                        execute = function () { return __awaiter(_this, void 0, void 0, function () {
                            return __generator(this, function (_a) {
                                switch (_a.label) {
                                    case 0:
                                        if (!(isInsertQuery || isDeleteQuery || isUpdateQuery)) return [3 /*break*/, 2];
                                        return [4 /*yield*/, databaseConnection.run(query, parameters, handler)];
                                    case 1:
                                        _a.sent();
                                        return [3 /*break*/, 4];
                                    case 2: return [4 /*yield*/, databaseConnection.all(query, parameters, handler)];
                                    case 3:
                                        _a.sent();
                                        _a.label = 4;
                                    case 4: return [2 /*return*/];
                                }
                            });
                        }); };
                        handler = function (err, rows) {
                            if (err && err.toString().indexOf("SQLITE_BUSY:") !== -1) {
                                if (typeof options.busyErrorRetry === "number" && options.busyErrorRetry > 0) {
                                    setTimeout(execute, options.busyErrorRetry);
                                    return;
                                }
                            }
                            // log slow queries if maxQueryExecution time is set
                            var queryEndTime = +new Date();
                            var queryExecutionTime = queryEndTime - queryStartTime;
                            if (maxQueryExecutionTime && queryExecutionTime > maxQueryExecutionTime)
                                connection.logger.logQuerySlow(queryExecutionTime, query, parameters, this);
                            if (err) {
                                connection.logger.logQueryError(err, query, parameters, this);
                                fail(new QueryFailedError(query, parameters, err));
                            }
                            else {
                                var result = new QueryResult();
                                if (isInsertQuery) {
                                    result.raw = this["lastID"];
                                }
                                else {
                                    result.raw = rows;
                                }
                                if (Array.isArray(rows)) {
                                    result.records = rows;
                                }
                                result.affected = this["changes"];
                                if (useStructuredResult) {
                                    ok(result);
                                }
                                else {
                                    ok(result.raw);
                                }
                            }
                        };
                        return [4 /*yield*/, execute()];
                    case 2:
                        _a.sent();
                        return [2 /*return*/];
                }
            });
        }); });
    };
    return SqliteQueryRunner;
}(AbstractSqliteQueryRunner));
export { SqliteQueryRunner };

//# sourceMappingURL=SqliteQueryRunner.js.map
