// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/*
 * This public interface for serializing the AWS Encryption SDK Message Footer Format
 * is provided for the use of the Encryption SDK for JavaScript only.  It can be used
 * as a reference but is not intended to be use by any packages other than the
 * Encryption SDK for JavaScript.
 *
 * See: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#footer-structure
 */
import { concatBuffers } from './concat_buffers';
import { uInt16BE } from './uint_util';
import { needs } from '@aws-crypto/material-management';
export function serializeSignatureInfo(signature) {
    return concatBuffers(uInt16BE(signature.byteLength), signature);
}
export function deserializeSignature({ buffer, byteOffset, byteLength, }) {
    /* Precondition: There must be information for a signature. */
    needs(byteLength && byteLength > 2, 'Invalid Signature');
    /* Uint8Array is a view on top of the underlying ArrayBuffer.
     * This means that raw underlying memory stored in the ArrayBuffer
     * may be larger than the Uint8Array.  This is especially true of
     * the Node.js Buffer object.  The offset and length *must* be
     * passed to the DataView otherwise I will get unexpected results.
     */
    const dataView = new DataView(buffer, byteOffset, byteLength);
    const signatureLength = dataView.getUint16(0, false); // big endian
    /* Precondition: The signature length must be positive. */
    needs(signatureLength > 0, 'Invalid Signature');
    /* Precondition: The data must match the serialized length. */
    needs(byteLength === signatureLength + 2, 'Invalid Signature');
    return new Uint8Array(buffer, byteOffset + 2, signatureLength);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic2lnbmF0dXJlX2luZm8uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvc2lnbmF0dXJlX2luZm8udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsb0VBQW9FO0FBQ3BFLHNDQUFzQztBQUV0Qzs7Ozs7OztHQU9HO0FBRUgsT0FBTyxFQUFFLGFBQWEsRUFBRSxNQUFNLGtCQUFrQixDQUFBO0FBQ2hELE9BQU8sRUFBRSxRQUFRLEVBQUUsTUFBTSxhQUFhLENBQUE7QUFDdEMsT0FBTyxFQUFFLEtBQUssRUFBRSxNQUFNLGlDQUFpQyxDQUFBO0FBRXZELE1BQU0sVUFBVSxzQkFBc0IsQ0FBQyxTQUFxQjtJQUMxRCxPQUFPLGFBQWEsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLFVBQVUsQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFBO0FBQ2pFLENBQUM7QUFFRCxNQUFNLFVBQVUsb0JBQW9CLENBQUMsRUFDbkMsTUFBTSxFQUNOLFVBQVUsRUFDVixVQUFVLEdBQ0M7SUFDWCw4REFBOEQ7SUFDOUQsS0FBSyxDQUFDLFVBQVUsSUFBSSxVQUFVLEdBQUcsQ0FBQyxFQUFFLG1CQUFtQixDQUFDLENBQUE7SUFDeEQ7Ozs7O09BS0c7SUFDSCxNQUFNLFFBQVEsR0FBRyxJQUFJLFFBQVEsQ0FBQyxNQUFNLEVBQUUsVUFBVSxFQUFFLFVBQVUsQ0FBQyxDQUFBO0lBQzdELE1BQU0sZUFBZSxHQUFHLFFBQVEsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLEtBQUssQ0FBQyxDQUFBLENBQUMsYUFBYTtJQUNsRSwwREFBMEQ7SUFDMUQsS0FBSyxDQUFDLGVBQWUsR0FBRyxDQUFDLEVBQUUsbUJBQW1CLENBQUMsQ0FBQTtJQUMvQyw4REFBOEQ7SUFDOUQsS0FBSyxDQUFDLFVBQVUsS0FBSyxlQUFlLEdBQUcsQ0FBQyxFQUFFLG1CQUFtQixDQUFDLENBQUE7SUFDOUQsT0FBTyxJQUFJLFVBQVUsQ0FBQyxNQUFNLEVBQUUsVUFBVSxHQUFHLENBQUMsRUFBRSxlQUFlLENBQUMsQ0FBQTtBQUNoRSxDQUFDIn0=