/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.env;

import java.io.Closeable;
import java.io.IOException;
import java.io.PrintStream;
import java.io.UncheckedIOException;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.apache.logging.log4j.util.Strings;
import org.apache.lucene.index.SegmentInfos;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockFactory;
import org.apache.lucene.store.LockObtainFailedException;
import org.apache.lucene.store.NIOFSDirectory;
import org.apache.lucene.store.NativeFSLockFactory;
import org.opensearch.Version;
import org.opensearch.cluster.node.DiscoveryNode;
import org.opensearch.cluster.node.DiscoveryNodeRole;
import org.opensearch.common.CheckedFunction;
import org.opensearch.common.Randomness;
import org.opensearch.common.SuppressForbidden;
import org.opensearch.common.UUIDs;
import org.opensearch.common.annotation.PublicApi;
import org.opensearch.common.collect.Tuple;
import org.opensearch.common.lease.Releasable;
import org.opensearch.common.settings.Setting;
import org.opensearch.common.settings.Settings;
import org.opensearch.common.unit.TimeValue;
import org.opensearch.common.util.io.IOUtils;
import org.opensearch.core.common.unit.ByteSizeValue;
import org.opensearch.core.index.Index;
import org.opensearch.core.index.shard.ShardId;
import org.opensearch.core.util.FileSystemUtils;
import org.opensearch.core.xcontent.NamedXContentRegistry;
import org.opensearch.env.Environment;
import org.opensearch.env.NodeMetadata;
import org.opensearch.env.ShardLock;
import org.opensearch.env.ShardLockObtainFailedException;
import org.opensearch.gateway.PersistedClusterStateService;
import org.opensearch.index.IndexSettings;
import org.opensearch.index.store.FsDirectoryFactory;
import org.opensearch.monitor.fs.FsInfo;
import org.opensearch.monitor.fs.FsProbe;
import org.opensearch.monitor.jvm.JvmInfo;

@PublicApi(since="1.0.0")
public final class NodeEnvironment
implements Closeable {
    private final Logger logger = LogManager.getLogger(NodeEnvironment.class);
    private final NodePath[] nodePaths;
    private final NodePath fileCacheNodePath;
    private final Path sharedDataPath;
    private final Lock[] locks;
    private final int nodeLockId;
    private final AtomicBoolean closed = new AtomicBoolean(false);
    private final Map<ShardId, InternalShardLock> shardLocks = new HashMap<ShardId, InternalShardLock>();
    private final NodeMetadata nodeMetadata;
    public static final Setting<Integer> MAX_LOCAL_STORAGE_NODES_SETTING = Setting.intSetting("node.max_local_storage_nodes", 1, 1, Setting.Property.NodeScope, Setting.Property.Deprecated);
    public static final Setting<Long> NODE_ID_SEED_SETTING = Setting.longSetting("node.id.seed", 0L, Long.MIN_VALUE, Setting.Property.NodeScope);
    public static final Setting<Boolean> ENABLE_LUCENE_SEGMENT_INFOS_TRACE_SETTING = Setting.boolSetting("node.enable_lucene_segment_infos_trace", false, Setting.Property.NodeScope);
    public static final String NODES_FOLDER = "nodes";
    public static final String INDICES_FOLDER = "indices";
    public static final String CACHE_FOLDER = "cache";
    public static final String NODE_LOCK_FILENAME = "node.lock";
    static final String TEMP_FILE_NAME = ".opensearch_temp_file";

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public NodeEnvironment(Settings settings, Environment environment) throws IOException {
        if (!DiscoveryNode.nodeRequiresLocalStorage(settings)) {
            this.nodePaths = null;
            this.fileCacheNodePath = null;
            this.sharedDataPath = null;
            this.locks = null;
            this.nodeLockId = -1;
            this.nodeMetadata = new NodeMetadata(NodeEnvironment.generateNodeId(settings), Version.CURRENT);
            return;
        }
        boolean success = false;
        NodeLock nodeLock = null;
        try {
            this.sharedDataPath = environment.sharedDataFile();
            IOException lastException = null;
            int maxLocalStorageNodes = MAX_LOCAL_STORAGE_NODES_SETTING.get(settings);
            AtomicReference onCreateDirectoriesException = new AtomicReference();
            for (int possibleLockId = 0; possibleLockId < maxLocalStorageNodes; ++possibleLockId) {
                try {
                    nodeLock = new NodeLock(possibleLockId, this.logger, environment, (CheckedFunction<Path, Boolean, IOException>)((CheckedFunction)dir -> {
                        try {
                            Files.createDirectories(dir, new FileAttribute[0]);
                        }
                        catch (IOException e) {
                            onCreateDirectoriesException.set(e);
                            throw e;
                        }
                        return true;
                    }));
                    break;
                }
                catch (LockObtainFailedException lockObtainFailedException) {
                    continue;
                }
                catch (IOException e) {
                    if (onCreateDirectoriesException.get() != null) {
                        throw (IOException)onCreateDirectoriesException.get();
                    }
                    lastException = e;
                }
            }
            if (nodeLock == null) {
                String message = String.format(Locale.ROOT, "failed to obtain node locks, tried [%s] with lock id%s; maybe these locations are not writable or multiple nodes were started without increasing [%s] (was [%d])?", Arrays.toString(environment.dataFiles()), maxLocalStorageNodes == 1 ? " [0]" : "s [0--" + (maxLocalStorageNodes - 1) + "]", MAX_LOCAL_STORAGE_NODES_SETTING.getKey(), maxLocalStorageNodes);
                throw new IllegalStateException(message, lastException);
            }
            this.locks = nodeLock.locks;
            this.nodePaths = nodeLock.nodePaths;
            this.fileCacheNodePath = this.nodePaths[0];
            this.nodeLockId = nodeLock.nodeId;
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("using node location [{}], local_lock_id [{}]", (Object)this.nodePaths, (Object)this.nodeLockId);
            }
            this.maybeLogPathDetails();
            this.maybeLogHeapDetails();
            NodeEnvironment.applySegmentInfosTrace(settings);
            this.assertCanWrite();
            if (DiscoveryNode.isClusterManagerNode(settings) || DiscoveryNode.isDataNode(settings)) {
                NodeEnvironment.ensureAtomicMoveSupported(this.nodePaths);
            }
            if (!DiscoveryNode.isDataNode(settings)) {
                if (!DiscoveryNode.isClusterManagerNode(settings)) {
                    this.ensureNoIndexMetadata(this.nodePaths);
                }
                this.ensureNoShardData(this.nodePaths);
            }
            if (!DiscoveryNode.isSearchNode(settings)) {
                this.ensureNoFileCacheData(this.fileCacheNodePath);
            }
            this.nodeMetadata = NodeEnvironment.loadNodeMetadata(settings, this.logger, this.nodePaths);
            success = true;
        }
        finally {
            if (!success) {
                this.close();
            }
        }
    }

    public static Path resolveNodePath(Path path, int nodeLockId) {
        return path.resolve(NODES_FOLDER).resolve(Integer.toString(nodeLockId));
    }

    private void maybeLogPathDetails() throws IOException {
        if (this.logger.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder();
            for (NodePath nodePath : this.nodePaths) {
                sb.append('\n').append(" -> ").append(nodePath.path.toAbsolutePath());
                FsInfo.Path fsPath = FsProbe.getFSInfo(nodePath);
                sb.append(", free_space [").append(fsPath.getFree()).append("], usable_space [").append(fsPath.getAvailable()).append("], total_space [").append(fsPath.getTotal()).append("], mount [").append(fsPath.getMount()).append("], type [").append(fsPath.getType()).append(']');
            }
            this.logger.debug("node data locations details:{}", (Object)sb);
        } else if (this.logger.isInfoEnabled()) {
            FsInfo.Path totFSPath = new FsInfo.Path();
            HashSet<String> allTypes = new HashSet<String>();
            HashSet<String> allMounts = new HashSet<String>();
            for (NodePath nodePath : this.nodePaths) {
                FsInfo.Path fsPath = FsProbe.getFSInfo(nodePath);
                String mount = fsPath.getMount();
                if (allMounts.contains(mount)) continue;
                allMounts.add(mount);
                String type = fsPath.getType();
                if (type != null) {
                    allTypes.add(type);
                }
                totFSPath.add(fsPath);
            }
            this.logger.info("using [{}] data paths, mounts [{}], net usable_space [{}], net total_space [{}], types [{}]", (Object)this.nodePaths.length, allMounts, (Object)totFSPath.getAvailable(), (Object)totFSPath.getTotal(), (Object)NodeEnvironment.toString(allTypes));
        }
    }

    private void maybeLogHeapDetails() {
        JvmInfo jvmInfo = JvmInfo.jvmInfo();
        ByteSizeValue maxHeapSize = jvmInfo.getMem().getHeapMax();
        String useCompressedOops = jvmInfo.useCompressedOops();
        this.logger.info("heap size [{}], compressed ordinary object pointers [{}]", (Object)maxHeapSize, (Object)useCompressedOops);
    }

    private static NodeMetadata loadNodeMetadata(Settings settings, Logger logger, NodePath ... nodePaths) throws IOException {
        Object[] paths = (Path[])Arrays.stream(nodePaths).map(np -> np.path).toArray(Path[]::new);
        NodeMetadata metadata = PersistedClusterStateService.nodeMetadata((Path[])paths);
        if (metadata == null) {
            HashSet<String> nodeIds = new HashSet<String>();
            for (Path path : paths) {
                NodeMetadata oldStyleMetadata = NodeMetadata.FORMAT.loadLatestState(logger, NamedXContentRegistry.EMPTY, path);
                if (oldStyleMetadata == null) continue;
                nodeIds.add(oldStyleMetadata.nodeId());
            }
            if (nodeIds.size() > 1) {
                throw new IllegalStateException("data paths " + Arrays.toString(paths) + " belong to multiple nodes with IDs " + nodeIds);
            }
            NodeMetadata legacyMetadata = NodeMetadata.FORMAT.loadLatestState(logger, NamedXContentRegistry.EMPTY, (Path[])paths);
            if (legacyMetadata == null) {
                assert (nodeIds.isEmpty()) : nodeIds;
                metadata = new NodeMetadata(NodeEnvironment.generateNodeId(settings), Version.CURRENT);
            } else {
                assert (nodeIds.equals(Collections.singleton(legacyMetadata.nodeId()))) : nodeIds + " doesn't match " + legacyMetadata;
                metadata = legacyMetadata;
            }
        }
        metadata = metadata.upgradeToCurrentVersion();
        assert (metadata.nodeVersion().equals((Object)Version.CURRENT)) : metadata.nodeVersion() + " != " + Version.CURRENT;
        return metadata;
    }

    public static String generateNodeId(Settings settings) {
        Random random = Randomness.get(settings, NODE_ID_SEED_SETTING);
        return UUIDs.randomBase64UUID((Random)random);
    }

    @SuppressForbidden(reason="System.out.*")
    static void applySegmentInfosTrace(Settings settings) {
        if (ENABLE_LUCENE_SEGMENT_INFOS_TRACE_SETTING.get(settings).booleanValue()) {
            SegmentInfos.setInfoStream((PrintStream)System.out);
        }
    }

    private static String toString(Collection<String> items) {
        StringBuilder b = new StringBuilder();
        for (String item : items) {
            if (b.length() > 0) {
                b.append(", ");
            }
            b.append(item);
        }
        return b.toString();
    }

    public void deleteShardDirectorySafe(ShardId shardId, IndexSettings indexSettings) throws IOException, ShardLockObtainFailedException {
        Path[] paths = this.availableShardPaths(shardId);
        this.logger.trace("deleting shard {} directory, paths: [{}]", (Object)shardId, (Object)paths);
        try (ShardLock lock = this.shardLock(shardId, "shard deletion under lock");){
            this.deleteShardDirectoryUnderLock(lock, indexSettings);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void acquireFSLockForPaths(IndexSettings indexSettings, Path ... shardPaths) throws IOException {
        Lock[] locks = new Lock[shardPaths.length];
        Directory[] dirs = new Directory[shardPaths.length];
        try {
            for (int i = 0; i < shardPaths.length; ++i) {
                Path p = shardPaths[i].resolve("index");
                dirs[i] = new NIOFSDirectory(p, indexSettings.getValue(FsDirectoryFactory.INDEX_LOCK_FACTOR_SETTING));
                try {
                    locks[i] = dirs[i].obtainLock("write.lock");
                    continue;
                }
                catch (IOException ex) {
                    throw new LockObtainFailedException("unable to acquire write.lock for " + p, (Throwable)ex);
                }
            }
        }
        finally {
            IOUtils.closeWhileHandlingException((Closeable[])locks);
            IOUtils.closeWhileHandlingException((Closeable[])dirs);
        }
    }

    public void deleteShardDirectoryUnderLock(ShardLock lock, IndexSettings indexSettings) throws IOException {
        ShardId shardId = lock.getShardId();
        assert (this.isShardLocked(shardId)) : "shard " + shardId + " is not locked";
        Path[] paths = this.availableShardPaths(shardId);
        this.logger.trace("acquiring locks for {}, paths: [{}]", (Object)shardId, (Object)paths);
        NodeEnvironment.acquireFSLockForPaths(indexSettings, paths);
        IOUtils.rm((Path[])paths);
        if (indexSettings.hasCustomDataPath()) {
            Path customLocation = this.resolveCustomLocation(indexSettings.customDataPath(), shardId);
            this.logger.trace("acquiring lock for {}, custom path: [{}]", (Object)shardId, (Object)customLocation);
            NodeEnvironment.acquireFSLockForPaths(indexSettings, customLocation);
            this.logger.trace("deleting custom shard {} directory [{}]", (Object)shardId, (Object)customLocation);
            IOUtils.rm((Path[])new Path[]{customLocation});
        }
        this.logger.trace("deleted shard {} directory, paths: [{}]", (Object)shardId, (Object)paths);
        assert (NodeEnvironment.assertPathsDoNotExist(paths));
    }

    private static boolean assertPathsDoNotExist(Path[] paths) {
        Set existingPaths = Stream.of(paths).filter(xva$0 -> FileSystemUtils.exists((Path[])new Path[]{xva$0})).filter(leftOver -> {
            try (DirectoryStream<Path> children = Files.newDirectoryStream(leftOver);){
                boolean bl;
                block18: {
                    Iterator<Path> iter = children.iterator();
                    if (!iter.hasNext()) {
                        boolean bl2 = true;
                        return bl2;
                    }
                    Path maybeState = iter.next();
                    if (iter.hasNext() || !maybeState.equals(leftOver.resolve("_state"))) {
                        boolean bl3 = true;
                        return bl3;
                    }
                    DirectoryStream<Path> stateChildren = Files.newDirectoryStream(maybeState);
                    try {
                        bl = stateChildren.iterator().hasNext();
                        if (stateChildren == null) break block18;
                    }
                    catch (Throwable throwable) {
                        if (stateChildren != null) {
                            try {
                                stateChildren.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    stateChildren.close();
                }
                return bl;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }).collect(Collectors.toSet());
        assert (existingPaths.size() == 0) : "Paths exist that should have been deleted: " + existingPaths;
        return existingPaths.size() == 0;
    }

    private boolean isShardLocked(ShardId id) {
        try {
            this.shardLock(id, "checking if shard is locked").close();
            return false;
        }
        catch (ShardLockObtainFailedException ex) {
            return true;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void deleteIndexDirectorySafe(Index index, long lockTimeoutMS, IndexSettings indexSettings) throws IOException, ShardLockObtainFailedException {
        List<ShardLock> locks = this.lockAllForIndex(index, indexSettings, "deleting index directory", lockTimeoutMS);
        try {
            this.deleteIndexDirectoryUnderLock(index, indexSettings);
        }
        finally {
            IOUtils.closeWhileHandlingException(locks);
        }
    }

    public void deleteIndexDirectoryUnderLock(Index index, IndexSettings indexSettings) throws IOException {
        Path[] indexPaths = this.indexPaths(index);
        this.logger.trace("deleting index {} directory, paths({}): [{}]", (Object)index, (Object)indexPaths.length, (Object)indexPaths);
        IOUtils.rm((Path[])indexPaths);
        if (indexSettings.hasCustomDataPath()) {
            Path customLocation = this.resolveIndexCustomLocation(indexSettings.customDataPath(), index.getUUID());
            this.logger.trace("deleting custom index {} directory [{}]", (Object)index, (Object)customLocation);
            IOUtils.rm((Path[])new Path[]{customLocation});
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public List<ShardLock> lockAllForIndex(Index index, IndexSettings settings, String lockDetails, long lockTimeoutMS) throws ShardLockObtainFailedException {
        int numShards = settings.getNumberOfShards();
        if (numShards <= 0) {
            throw new IllegalArgumentException("settings must contain a non-null > 0 number of shards");
        }
        this.logger.trace("locking all shards for index {} - [{}]", (Object)index, (Object)numShards);
        ArrayList<ShardLock> allLocks = new ArrayList<ShardLock>(numShards);
        boolean success = false;
        long startTimeNS = System.nanoTime();
        try {
            for (int i = 0; i < numShards; ++i) {
                long timeoutLeftMS = Math.max(0L, lockTimeoutMS - TimeValue.nsecToMSec((long)(System.nanoTime() - startTimeNS)));
                allLocks.add(this.shardLock(new ShardId(index, i), lockDetails, timeoutLeftMS));
            }
            success = true;
        }
        finally {
            if (!success) {
                this.logger.trace("unable to lock all shards for index {}", (Object)index);
                IOUtils.closeWhileHandlingException(allLocks);
            }
        }
        return allLocks;
    }

    public ShardLock shardLock(ShardId id, String details) throws ShardLockObtainFailedException {
        return this.shardLock(id, details, 0L);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public ShardLock shardLock(final ShardId shardId, String details, long lockTimeoutMS) throws ShardLockObtainFailedException {
        boolean acquired;
        InternalShardLock shardLock;
        this.logger.trace("acquiring node shardlock on [{}], timeout [{}], details [{}]", (Object)shardId, (Object)lockTimeoutMS, (Object)details);
        Map<ShardId, InternalShardLock> map = this.shardLocks;
        synchronized (map) {
            if (this.shardLocks.containsKey(shardId)) {
                shardLock = this.shardLocks.get(shardId);
                shardLock.incWaitCount();
                acquired = false;
            } else {
                shardLock = new InternalShardLock(shardId, details);
                this.shardLocks.put(shardId, shardLock);
                acquired = true;
            }
        }
        if (!acquired) {
            boolean success = false;
            try {
                shardLock.acquire(lockTimeoutMS, details);
                success = true;
            }
            finally {
                if (!success) {
                    shardLock.decWaitCount();
                }
            }
        }
        this.logger.trace("successfully acquired shardlock for [{}]", (Object)shardId);
        return new ShardLock(shardId){

            @Override
            protected void closeInternal() {
                shardLock.release();
                NodeEnvironment.this.logger.trace("released shard lock for [{}]", (Object)shardId);
            }

            @Override
            public void setDetails(String details) {
                shardLock.setDetails(details);
            }
        };
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Set<ShardId> lockedShards() {
        Map<ShardId, InternalShardLock> map = this.shardLocks;
        synchronized (map) {
            return Collections.unmodifiableSet(new HashSet<ShardId>(this.shardLocks.keySet()));
        }
    }

    public boolean hasNodeFile() {
        return this.nodePaths != null && this.locks != null;
    }

    public Path[] nodeDataPaths() {
        this.assertEnvIsLocked();
        Path[] paths = new Path[this.nodePaths.length];
        for (int i = 0; i < paths.length; ++i) {
            paths[i] = this.nodePaths[i].path;
        }
        return paths;
    }

    public Path sharedDataPath() {
        return this.sharedDataPath;
    }

    public String nodeId() {
        return this.nodeMetadata.nodeId();
    }

    public NodePath[] nodePaths() {
        this.assertEnvIsLocked();
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        return this.nodePaths;
    }

    public NodePath fileCacheNodePath() {
        this.assertEnvIsLocked();
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        return this.fileCacheNodePath;
    }

    public int getNodeLockId() {
        this.assertEnvIsLocked();
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        return this.nodeLockId;
    }

    public Path[] indexPaths(Index index) {
        this.assertEnvIsLocked();
        Path[] indexPaths = new Path[this.nodePaths.length];
        for (int i = 0; i < this.nodePaths.length; ++i) {
            indexPaths[i] = this.nodePaths[i].resolve(index);
        }
        return indexPaths;
    }

    public Path[] availableShardPaths(ShardId shardId) {
        this.assertEnvIsLocked();
        NodePath[] nodePaths = this.nodePaths();
        Path[] shardLocations = new Path[nodePaths.length];
        for (int i = 0; i < nodePaths.length; ++i) {
            shardLocations[i] = nodePaths[i].resolve(shardId);
        }
        return shardLocations;
    }

    public Set<String> availableIndexFolders() throws IOException {
        return this.availableIndexFolders(p -> false);
    }

    public Set<String> availableIndexFolders(Predicate<String> excludeIndexPathIdsPredicate) throws IOException {
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<String> indexFolders = new HashSet<String>();
        for (NodePath nodePath : this.nodePaths) {
            indexFolders.addAll(this.availableIndexFoldersForPath(nodePath, excludeIndexPathIdsPredicate));
        }
        return indexFolders;
    }

    public Set<String> availableIndexFoldersForPath(NodePath nodePath) throws IOException {
        return this.availableIndexFoldersForPath(nodePath, p -> false);
    }

    public Set<String> availableIndexFoldersForPath(NodePath nodePath, Predicate<String> excludeIndexPathIdsPredicate) throws IOException {
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<String> indexFolders = new HashSet<String>();
        Path indicesLocation = nodePath.indicesPath;
        if (Files.isDirectory(indicesLocation, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indicesLocation);){
                for (Path index : stream) {
                    String fileName = index.getFileName().toString();
                    if (excludeIndexPathIdsPredicate.test(fileName) || !Files.isDirectory(index, new LinkOption[0])) continue;
                    indexFolders.add(fileName);
                }
            }
        }
        return indexFolders;
    }

    public Path[] resolveIndexFolder(String indexFolderName) {
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        ArrayList<Path> paths = new ArrayList<Path>(this.nodePaths.length);
        for (NodePath nodePath : this.nodePaths) {
            Path indexFolder = nodePath.indicesPath.resolve(indexFolderName);
            if (!Files.exists(indexFolder, new LinkOption[0])) continue;
            paths.add(indexFolder);
        }
        return paths.toArray(new Path[paths.size()]);
    }

    public Set<ShardId> findAllShardIds(Index index) throws IOException {
        assert (index != null);
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashSet<ShardId> shardIds = new HashSet<ShardId>();
        String indexUniquePathId = index.getUUID();
        for (NodePath nodePath : this.nodePaths) {
            shardIds.addAll(NodeEnvironment.findAllShardsForIndex(nodePath.indicesPath.resolve(indexUniquePathId), index));
        }
        return shardIds;
    }

    public Map<NodePath, Long> shardCountPerPath(Index index) throws IOException {
        assert (index != null);
        if (this.nodePaths == null || this.locks == null) {
            throw new IllegalStateException("node is not configured to store local location");
        }
        this.assertEnvIsLocked();
        HashMap<NodePath, Long> shardCountPerPath = new HashMap<NodePath, Long>();
        String indexUniquePathId = index.getUUID();
        for (NodePath nodePath : this.nodePaths) {
            Path indexLocation = nodePath.indicesPath.resolve(indexUniquePathId);
            if (!Files.isDirectory(indexLocation, new LinkOption[0])) continue;
            shardCountPerPath.put(nodePath, Long.valueOf(NodeEnvironment.findAllShardsForIndex(indexLocation, index).size()));
        }
        return shardCountPerPath;
    }

    private static Set<ShardId> findAllShardsForIndex(Path indexPath, Index index) throws IOException {
        assert (indexPath.getFileName().toString().equals(index.getUUID()));
        HashSet<ShardId> shardIds = new HashSet<ShardId>();
        if (Files.isDirectory(indexPath, new LinkOption[0])) {
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(indexPath);){
                for (Path shardPath : stream) {
                    String fileName = shardPath.getFileName().toString();
                    if (!Files.isDirectory(shardPath, new LinkOption[0]) || !fileName.chars().allMatch(Character::isDigit)) continue;
                    int shardId = Integer.parseInt(fileName);
                    ShardId id = new ShardId(index, shardId);
                    shardIds.add(id);
                }
            }
        }
        return shardIds;
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true) && this.locks != null) {
            for (Lock lock : this.locks) {
                try {
                    this.logger.trace("releasing lock [{}]", (Object)lock);
                    lock.close();
                }
                catch (IOException e) {
                    this.logger.trace(() -> new ParameterizedMessage("failed to release lock [{}]", (Object)lock), (Throwable)e);
                }
            }
        }
    }

    private void assertEnvIsLocked() {
        if (!this.closed.get() && this.locks != null) {
            for (Lock lock : this.locks) {
                try {
                    lock.ensureValid();
                }
                catch (IOException e) {
                    this.logger.warn("lock assertion failed", (Throwable)e);
                    throw new IllegalStateException("environment is not locked", e);
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void ensureAtomicMoveSupported(NodePath[] nodePaths) throws IOException {
        for (NodePath nodePath : nodePaths) {
            assert (Files.isDirectory(nodePath.path, new LinkOption[0])) : nodePath.path + " is not a directory";
            Path src = nodePath.path.resolve(".opensearch_temp_file.tmp");
            Path target = nodePath.path.resolve(".opensearch_temp_file.final");
            try {
                Files.deleteIfExists(src);
                Files.createFile(src, new FileAttribute[0]);
                Files.move(src, target, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
            }
            catch (AtomicMoveNotSupportedException ex) {
                throw new IllegalStateException("atomic_move is not supported by the filesystem on path [" + nodePath.path + "] atomic_move is required for opensearch to work correctly.", ex);
            }
            finally {
                try {
                    Files.deleteIfExists(src);
                }
                finally {
                    Files.deleteIfExists(target);
                }
            }
        }
    }

    private void ensureNoShardData(NodePath[] nodePaths) throws IOException {
        List<Path> shardDataPaths = NodeEnvironment.collectShardDataPaths(nodePaths);
        if (!shardDataPaths.isEmpty()) {
            String message = String.format(Locale.ROOT, "node does not have the %s role but has shard data: %s. Use 'opensearch-node repurpose' tool to clean up", DiscoveryNodeRole.DATA_ROLE.roleName(), shardDataPaths);
            throw new IllegalStateException(message);
        }
    }

    private void ensureNoFileCacheData(NodePath fileCacheNodePath) throws IOException {
        List<Path> cacheDataPaths = NodeEnvironment.collectFileCacheDataPath(fileCacheNodePath);
        if (!cacheDataPaths.isEmpty()) {
            String message = String.format(Locale.ROOT, "node does not have the %s role but has data within node search cache: %s. Use 'opensearch-node repurpose' tool to clean up", DiscoveryNodeRole.SEARCH_ROLE.roleName(), cacheDataPaths);
            throw new IllegalStateException(message);
        }
    }

    private void ensureNoIndexMetadata(NodePath[] nodePaths) throws IOException {
        List<Path> indexMetadataPaths = NodeEnvironment.collectIndexMetadataPaths(nodePaths);
        if (!indexMetadataPaths.isEmpty()) {
            String message = String.format(Locale.ROOT, "node does not have the %s and %s roles but has index metadata: %s. Use 'opensearch-node repurpose' tool to clean up", DiscoveryNodeRole.DATA_ROLE.roleName(), DiscoveryNodeRole.CLUSTER_MANAGER_ROLE.roleName(), indexMetadataPaths);
            throw new IllegalStateException(message);
        }
    }

    static List<Path> collectShardDataPaths(NodePath[] nodePaths) throws IOException {
        return NodeEnvironment.collectIndexSubPaths(nodePaths, NodeEnvironment::isShardPath);
    }

    static List<Path> collectIndexMetadataPaths(NodePath[] nodePaths) throws IOException {
        return NodeEnvironment.collectIndexSubPaths(nodePaths, NodeEnvironment::isIndexMetadataPath);
    }

    private static List<Path> collectIndexSubPaths(NodePath[] nodePaths, Predicate<Path> subPathPredicate) throws IOException {
        ArrayList<Path> indexSubPaths = new ArrayList<Path>();
        for (NodePath nodePath : nodePaths) {
            Path indicesPath = nodePath.indicesPath;
            if (!Files.isDirectory(indicesPath, new LinkOption[0])) continue;
            try (DirectoryStream<Path> indexStream = Files.newDirectoryStream(indicesPath);){
                for (Path indexPath : indexStream) {
                    if (!Files.isDirectory(indexPath, new LinkOption[0])) continue;
                    Stream<Path> shardStream = Files.list(indexPath);
                    try {
                        shardStream.filter(subPathPredicate).map(Path::toAbsolutePath).forEach(indexSubPaths::add);
                    }
                    finally {
                        if (shardStream == null) continue;
                        shardStream.close();
                    }
                }
            }
        }
        return indexSubPaths;
    }

    private static boolean isShardPath(Path path) {
        return Files.isDirectory(path, new LinkOption[0]) && path.getFileName().toString().chars().allMatch(Character::isDigit);
    }

    private static boolean isIndexMetadataPath(Path path) {
        return Files.isDirectory(path, new LinkOption[0]) && path.getFileName().toString().equals("_state");
    }

    public static List<Path> collectFileCacheDataPath(NodePath fileCacheNodePath) throws IOException {
        ArrayList<Path> indexSubPaths = new ArrayList<Path>();
        Path fileCachePath = fileCacheNodePath.fileCachePath;
        if (Files.isDirectory(fileCachePath, new LinkOption[0])) {
            try (DirectoryStream<Path> indexStream = Files.newDirectoryStream(fileCachePath);){
                for (Path indexPath : indexStream) {
                    if (!Files.isDirectory(indexPath, new LinkOption[0])) continue;
                    Stream<Path> shardStream = Files.list(indexPath);
                    try {
                        shardStream.filter(NodeEnvironment::isShardPath).map(Path::toAbsolutePath).forEach(indexSubPaths::add);
                    }
                    finally {
                        if (shardStream == null) continue;
                        shardStream.close();
                    }
                }
            }
        }
        return indexSubPaths;
    }

    public static Path resolveBaseCustomLocation(String customDataPath, Path sharedDataPath, int nodeLockId) {
        if (Strings.isNotEmpty((CharSequence)customDataPath)) {
            assert (sharedDataPath != null);
            return sharedDataPath.resolve(customDataPath).resolve(Integer.toString(nodeLockId));
        }
        throw new IllegalArgumentException("no custom index.data_path setting available");
    }

    private Path resolveIndexCustomLocation(String customDataPath, String indexUUID) {
        return NodeEnvironment.resolveIndexCustomLocation(customDataPath, indexUUID, this.sharedDataPath, this.nodeLockId);
    }

    private static Path resolveIndexCustomLocation(String customDataPath, String indexUUID, Path sharedDataPath, int nodeLockId) {
        return NodeEnvironment.resolveBaseCustomLocation(customDataPath, sharedDataPath, nodeLockId).resolve(indexUUID);
    }

    public Path resolveFileCacheLocation(Path fileCachePath, ShardId shardId) {
        return fileCachePath.resolve(shardId.getIndex().getUUID()).resolve(Integer.toString(shardId.id()));
    }

    public Path resolveCustomLocation(String customDataPath, ShardId shardId) {
        return NodeEnvironment.resolveCustomLocation(customDataPath, shardId, this.sharedDataPath, this.nodeLockId);
    }

    public static Path resolveCustomLocation(String customDataPath, ShardId shardId, Path sharedDataPath, int nodeLockId) {
        return NodeEnvironment.resolveIndexCustomLocation(customDataPath, shardId.getIndex().getUUID(), sharedDataPath, nodeLockId).resolve(Integer.toString(shardId.id()));
    }

    public static Path shardStatePathToDataPath(Path shardPath) {
        int count = shardPath.getNameCount();
        assert (Integer.parseInt(shardPath.getName(count - 1).toString()) >= 0);
        assert (INDICES_FOLDER.equals(shardPath.getName(count - 3).toString()));
        return shardPath.getParent().getParent().getParent();
    }

    private void assertCanWrite() throws IOException {
        for (Path path : this.nodeDataPaths()) {
            NodeEnvironment.tryWriteTempFile(path);
        }
        for (String indexFolderName : this.availableIndexFolders()) {
            for (Path indexPath : this.resolveIndexFolder(indexFolderName)) {
                Path indexStatePath = indexPath.resolve("_state");
                NodeEnvironment.tryWriteTempFile(indexStatePath);
                NodeEnvironment.tryWriteTempFile(indexPath);
                try (DirectoryStream<Path> stream = Files.newDirectoryStream(indexPath);){
                    for (Path shardPath : stream) {
                        String fileName = shardPath.getFileName().toString();
                        if (!Files.isDirectory(shardPath, new LinkOption[0]) || !fileName.chars().allMatch(Character::isDigit)) continue;
                        Path indexDir = shardPath.resolve("index");
                        Path statePath = shardPath.resolve("_state");
                        Path translogDir = shardPath.resolve("translog");
                        NodeEnvironment.tryWriteTempFile(indexDir);
                        NodeEnvironment.tryWriteTempFile(translogDir);
                        NodeEnvironment.tryWriteTempFile(statePath);
                        NodeEnvironment.tryWriteTempFile(shardPath);
                    }
                }
            }
        }
    }

    private static void tryWriteTempFile(Path path) throws IOException {
        if (Files.exists(path, new LinkOption[0])) {
            Path resolve = path.resolve(TEMP_FILE_NAME);
            try {
                Files.deleteIfExists(resolve);
                Files.createFile(resolve, new FileAttribute[0]);
                Files.delete(resolve);
            }
            catch (IOException ex) {
                throw new IOException("failed to test writes in data directory [" + path + "] write permission is required", ex);
            }
        }
    }

    private final class InternalShardLock {
        private final Semaphore mutex = new Semaphore(1);
        private int waitCount = 1;
        private final ShardId shardId;
        private volatile Tuple<Long, String> lockDetails;

        InternalShardLock(ShardId shardId, String details) {
            this.shardId = shardId;
            this.mutex.acquireUninterruptibly();
            this.lockDetails = Tuple.tuple((Object)System.nanoTime(), (Object)details);
        }

        protected void release() {
            this.mutex.release();
            this.decWaitCount();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void incWaitCount() {
            Map<ShardId, InternalShardLock> map = NodeEnvironment.this.shardLocks;
            synchronized (map) {
                assert (this.waitCount > 0) : "waitCount is " + this.waitCount + " but should be > 0";
                ++this.waitCount;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void decWaitCount() {
            Map<ShardId, InternalShardLock> map = NodeEnvironment.this.shardLocks;
            synchronized (map) {
                assert (this.waitCount > 0) : "waitCount is " + this.waitCount + " but should be > 0";
                --this.waitCount;
                NodeEnvironment.this.logger.trace("shard lock wait count for {} is now [{}]", (Object)this.shardId, (Object)this.waitCount);
                if (this.waitCount == 0) {
                    NodeEnvironment.this.logger.trace("last shard lock wait decremented, removing lock for {}", (Object)this.shardId);
                    InternalShardLock remove = NodeEnvironment.this.shardLocks.remove(this.shardId);
                    assert (remove != null) : "Removed lock was null";
                }
            }
        }

        void acquire(long timeoutInMillis, String details) throws ShardLockObtainFailedException {
            try {
                if (!this.mutex.tryAcquire(timeoutInMillis, TimeUnit.MILLISECONDS)) {
                    Tuple<Long, String> lockDetails = this.lockDetails;
                    throw new ShardLockObtainFailedException(this.shardId, "obtaining shard lock for [" + details + "] timed out after [" + timeoutInMillis + "ms], lock already held for [" + (String)lockDetails.v2() + "] with age [" + TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - (Long)lockDetails.v1()) + "ms]");
                }
                this.setDetails(details);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new ShardLockObtainFailedException(this.shardId, "thread interrupted while trying to obtain shard lock", e);
            }
        }

        public void setDetails(String details) {
            this.lockDetails = Tuple.tuple((Object)System.nanoTime(), (Object)details);
        }
    }

    @FunctionalInterface
    public static interface ShardLocker {
        public ShardLock lock(ShardId var1, String var2, long var3) throws ShardLockObtainFailedException;
    }

    public static class NodeLock
    implements Releasable {
        private final int nodeId;
        private final Lock[] locks;
        private final NodePath[] nodePaths;

        public NodeLock(int nodeId, Logger logger, Environment environment, CheckedFunction<Path, Boolean, IOException> pathFunction) throws IOException {
            this.nodeId = nodeId;
            this.nodePaths = new NodePath[environment.dataFiles().length];
            this.locks = new Lock[this.nodePaths.length];
            try {
                Path[] dataPaths = environment.dataFiles();
                for (int dirIndex = 0; dirIndex < dataPaths.length; ++dirIndex) {
                    Path dataDir = dataPaths[dirIndex];
                    Path dir = NodeEnvironment.resolveNodePath(dataDir, nodeId);
                    if (!((Boolean)pathFunction.apply((Object)dir)).booleanValue()) continue;
                    try (FSDirectory luceneDir = FSDirectory.open((Path)dir, (LockFactory)NativeFSLockFactory.INSTANCE);){
                        logger.trace("obtaining node lock on {} ...", (Object)dir.toAbsolutePath());
                        this.locks[dirIndex] = luceneDir.obtainLock(NodeEnvironment.NODE_LOCK_FILENAME);
                        this.nodePaths[dirIndex] = new NodePath(dir);
                        continue;
                    }
                    catch (IOException e) {
                        logger.trace(() -> new ParameterizedMessage("failed to obtain node lock on {}", (Object)dir.toAbsolutePath()), (Throwable)e);
                        throw e instanceof LockObtainFailedException ? e : new IOException("failed to obtain lock on " + dir.toAbsolutePath(), e);
                    }
                }
            }
            catch (IOException e) {
                this.close();
                throw e;
            }
        }

        public NodePath[] getNodePaths() {
            return this.nodePaths;
        }

        public void close() {
            for (int i = 0; i < this.locks.length; ++i) {
                if (this.locks[i] != null) {
                    IOUtils.closeWhileHandlingException((Closeable)this.locks[i]);
                }
                this.locks[i] = null;
            }
        }
    }

    @PublicApi(since="1.0.0")
    public static class NodePath {
        public final Path path;
        public final Path indicesPath;
        public final FileStore fileStore;
        public final Path fileCachePath;
        public ByteSizeValue fileCacheReservedSize;
        public final int majorDeviceNumber;
        public final int minorDeviceNumber;

        public NodePath(Path path) throws IOException {
            this.path = path;
            this.indicesPath = path.resolve(NodeEnvironment.INDICES_FOLDER);
            this.fileCachePath = path.resolve(NodeEnvironment.CACHE_FOLDER);
            this.fileStore = Environment.getFileStore(path);
            this.fileCacheReservedSize = ByteSizeValue.ZERO;
            if (this.fileStore.supportsFileAttributeView("lucene")) {
                this.majorDeviceNumber = (Integer)this.fileStore.getAttribute("lucene:major_device_number");
                this.minorDeviceNumber = (Integer)this.fileStore.getAttribute("lucene:minor_device_number");
            } else {
                this.majorDeviceNumber = -1;
                this.minorDeviceNumber = -1;
            }
        }

        public Path resolve(ShardId shardId) {
            return this.resolve(shardId.getIndex()).resolve(Integer.toString(shardId.id()));
        }

        public Path resolve(Index index) {
            return this.resolve(index.getUUID());
        }

        Path resolve(String uuid) {
            return this.indicesPath.resolve(uuid);
        }

        public String toString() {
            return "NodePath{path=" + this.path + ", indicesPath=" + this.indicesPath + ", fileStore=" + this.fileStore + ", majorDeviceNumber=" + this.majorDeviceNumber + ", minorDeviceNumber=" + this.minorDeviceNumber + "}";
        }
    }
}

