"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.register = void 0;
const shared_1 = require("../shared");
function register(ctx) {
    const { ts } = ctx;
    return (uri) => {
        const document = ctx.getTextDocument(uri);
        if (!document)
            return [];
        const fileName = ctx.env.uriToFileName(document.uri);
        const outliningSpans = (0, shared_1.safeCall)(() => ctx.typescript.languageService.getOutliningSpans(fileName));
        if (!outliningSpans)
            return [];
        const foldingRanges = [];
        for (const outliningSpan of outliningSpans) {
            const start = document.positionAt(outliningSpan.textSpan.start);
            const end = adjustFoldingEnd(start, document.positionAt(outliningSpan.textSpan.start + outliningSpan.textSpan.length), document);
            const foldingRange = {
                startLine: start.line,
                endLine: end.line,
                startCharacter: start.character,
                endCharacter: end.character,
                kind: transformFoldingRangeKind(outliningSpan.kind),
            };
            foldingRanges.push(foldingRange);
        }
        return foldingRanges;
    };
    function transformFoldingRangeKind(tsKind) {
        switch (tsKind) {
            case ts.OutliningSpanKind.Comment: return 'comment';
            case ts.OutliningSpanKind.Imports: return 'imports';
            case ts.OutliningSpanKind.Region: return 'region';
        }
    }
}
exports.register = register;
const foldEndPairCharacters = ['}', ']', ')', '`'];
// https://github.com/microsoft/vscode/blob/bed61166fb604e519e82e4d1d1ed839bc45d65f8/extensions/typescript-language-features/src/languageFeatures/folding.ts#L61-L73
function adjustFoldingEnd(start, end, document) {
    // workaround for #47240
    if (end.character > 0) {
        const foldEndCharacter = document.getText({
            start: { line: end.line, character: end.character - 1 },
            end,
        });
        if (foldEndPairCharacters.includes(foldEndCharacter)) {
            const endOffset = Math.max(document.offsetAt({ line: end.line, character: 0 }) - 1, document.offsetAt(start));
            return document.positionAt(endOffset);
        }
    }
    return end;
}
//# sourceMappingURL=foldingRanges.js.map