"""Data containing all current emoji
   Extracted from https://unicode.org/Public/emoji/latest/emoji-test.txt
   and https://www.unicode.org/Public/UCD/latest/ucd/emoji/emoji-variation-sequences.txt
   See utils/get_codes_from_unicode_emoji_data_files.py

   +----------------+-------------+------------------+-------------------+
   | Emoji Version  |    Date     | Unicode Version  | Data File Comment |
   +----------------+-------------+------------------+-------------------+
   | N/A            | 2010-10-11  | Unicode 6.0      | E0.6              |
   | N/A            | 2014-06-16  | Unicode 7.0      | E0.7              |
   | Emoji 1.0      | 2015-06-09  | Unicode 8.0      | E1.0              |
   | Emoji 2.0      | 2015-11-12  | Unicode 8.0      | E2.0              |
   | Emoji 3.0      | 2016-06-03  | Unicode 9.0      | E3.0              |
   | Emoji 4.0      | 2016-11-22  | Unicode 9.0      | E4.0              |
   | Emoji 5.0      | 2017-06-20  | Unicode 10.0     | E5.0              |
   | Emoji 11.0     | 2018-05-21  | Unicode 11.0     | E11.0             |
   | Emoji 12.0     | 2019-03-05  | Unicode 12.0     | E12.0             |
   | Emoji 12.1     | 2019-10-21  | Unicode 12.1     | E12.1             |
   | Emoji 13.0     | 2020-03-10  | Unicode 13.0     | E13.0             |
   | Emoji 13.1     | 2020-09-15  | Unicode 13.0     | E13.1             |
   | Emoji 14.0     | 2021-09-14  | Unicode 14.0     | E14.0             |
   | Emoji 15.0     | 2022-09-13  | Unicode 15.0     | E15.0             |

                          http://www.unicode.org/reports/tr51/#Versioning

"""

__all__ = [
    'EMOJI_DATA', 'STATUS', 'LANGUAGES'
]

component = 1
fully_qualified = 2
minimally_qualified = 3
unqualified = 4

STATUS = {
    "component": component,
    "fully_qualified": fully_qualified,
    "minimally_qualified": minimally_qualified,
    "unqualified": unqualified
}

LANGUAGES = ['en', 'es', 'ja', 'ko', 'pt', 'it', 'fr', 'de', 'fa', 'id', 'zh', 'ru']

EMOJI_DATA = {
    '\U0001F947': {  # 🥇
        'en': ':1st_place_medal:',
        'status': fully_qualified,
        'E': 3,
        'de': ':goldmedaille:',
        'es': ':medalla_de_oro:',
        'fr': ':médaille_d’or:',
        'ja': ':金メダル:',
        'ko': ':금메달:',
        'pt': ':medalha_de_ouro:',
        'it': ':medaglia_d’oro:',
        'fa': ':مدال_طلا:',
        'id': ':medali_emas:',
        'zh': ':金牌:',
        'ru': ':золотая_медаль:'
    },
    '\U0001F948': {  # 🥈
        'en': ':2nd_place_medal:',
        'status': fully_qualified,
        'E': 3,
        'de': ':silbermedaille:',
        'es': ':medalla_de_plata:',
        'fr': ':médaille_d’argent:',
        'ja': ':銀メダル:',
        'ko': ':은메달:',
        'pt': ':medalha_de_prata:',
        'it': ':medaglia_d’argento:',
        'fa': ':مدال_نقره:',
        'id': ':medali_perak:',
        'zh': ':银牌:',
        'ru': ':серебряная_медаль:'
    },
    '\U0001F949': {  # 🥉
        'en': ':3rd_place_medal:',
        'status': fully_qualified,
        'E': 3,
        'de': ':bronzemedaille:',
        'es': ':medalla_de_bronce:',
        'fr': ':médaille_de_bronze:',
        'ja': ':銅メダル:',
        'ko': ':동메달:',
        'pt': ':medalha_de_bronze:',
        'it': ':medaglia_di_bronzo:',
        'fa': ':مدال_برنز:',
        'id': ':medali_perunggu:',
        'zh': ':铜牌:',
        'ru': ':бронзовая_медаль:'
    },
    '\U0001F18E': {  # 🆎
        'en': ':AB_button_(blood_type):',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':ab:', ':ab_button_blood_type:'],
        'de': ':großbuchstaben_ab_in_rotem_quadrat:',
        'es': ':grupo_sanguíneo_ab:',
        'fr': ':groupe_sanguin_ab:',
        'ja': ':血液型ab型:',
        'ko': ':에이비형:',
        'pt': ':botão_ab_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_ab:',
        'fa': ':دکمه_آ_ب_(گروه_خونی):',
        'id': ':tombol_ab_(golongan_darah):',
        'zh': ':AB型血:',
        'ru': ':IV_группа_крови:'
    },
    '\U0001F3E7': {  # 🏧
        'en': ':ATM_sign:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':atm:', ':atm_sign:'],
        'de': ':symbol_geldautomat:',
        'es': ':señal_de_cajero_automático:',
        'fr': ':distributeur_de_billets:',
        'ja': ':atm:',
        'ko': ':에이티엠:',
        'pt': ':símbolo_de_caixa_automático:',
        'it': ':simbolo_dello_sportello_bancomat:',
        'fa': ':نشان_عابربانک:',
        'id': ':tanda_atm:',
        'zh': ':取款机:',
        'ru': ':значок_банкомата:'
    },
    '\U0001F170\U0000FE0F': {  # 🅰️
        'en': ':A_button_(blood_type):',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':a:', ':a_button_blood_type:'],
        'variant': True,
        'de': ':großbuchstabe_a_in_rotem_quadrat:',
        'es': ':grupo_sanguíneo_a:',
        'fr': ':groupe_sanguin_a:',
        'ja': ':血液型a型:',
        'ko': ':에이형:',
        'pt': ':botão_a_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_a:',
        'fa': ':دکمه_آ_(گروه_خونی):',
        'id': ':tombol_a_(golongan_darah):',
        'zh': ':A型血:',
        'ru': ':ii_группа_крови:'
    },
    '\U0001F170': {  # 🅰
        'en': ':A_button_(blood_type):',
        'status': unqualified,
        'E': 0.6,
        'alias': [':a:', ':a_button_blood_type:'],
        'variant': True,
        'de': ':großbuchstabe_a_in_rotem_quadrat:',
        'es': ':grupo_sanguíneo_a:',
        'fr': ':groupe_sanguin_a:',
        'ja': ':血液型a型:',
        'ko': ':에이형:',
        'pt': ':botão_a_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_a:',
        'fa': ':دکمه_آ_(گروه_خونی):',
        'id': ':tombol_a_(golongan_darah):',
        'zh': ':A型血:',
        'ru': ':II_группа_крови:'
    },
    '\U0001F1E6\U0001F1EB': {  # 🇦🇫
        'en': ':Afghanistan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Afghanistan:', ':afghanistan:'],
        'de': ':flagge_afghanistan:',
        'es': ':bandera_afganistán:',
        'fr': ':drapeau_afghanistan:',
        'ja': ':旗_アフガニスタン:',
        'ko': ':깃발_아프가니스탄:',
        'pt': ':bandeira_afeganistão:',
        'it': ':bandiera_afghanistan:',
        'fa': ':پرچم_افغانستان:',
        'id': ':bendera_afganistan:',
        'zh': ':阿富汗:',
        'ru': ':флаг_Афганистан:'
    },
    '\U0001F1E6\U0001F1F1': {  # 🇦🇱
        'en': ':Albania:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Albania:', ':albania:'],
        'de': ':flagge_albanien:',
        'es': ':bandera_albania:',
        'fr': ':drapeau_albanie:',
        'ja': ':旗_アルバニア:',
        'ko': ':깃발_알바니아:',
        'pt': ':bandeira_albânia:',
        'it': ':bandiera_albania:',
        'fa': ':پرچم_آلبانی:',
        'id': ':bendera_albania:',
        'zh': ':阿尔巴尼亚:',
        'ru': ':флаг_Албания:'
    },
    '\U0001F1E9\U0001F1FF': {  # 🇩🇿
        'en': ':Algeria:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Algeria:', ':algeria:'],
        'de': ':flagge_algerien:',
        'es': ':bandera_argelia:',
        'fr': ':drapeau_algérie:',
        'ja': ':旗_アルジェリア:',
        'ko': ':깃발_알제리:',
        'pt': ':bandeira_argélia:',
        'it': ':bandiera_algeria:',
        'fa': ':پرچم_الجزایر:',
        'id': ':bendera_aljazair:',
        'zh': ':阿尔及利亚:',
        'ru': ':флаг_Алжир:'
    },
    '\U0001F1E6\U0001F1F8': {  # 🇦🇸
        'en': ':American_Samoa:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_American_Samoa:', ':american_samoa:'],
        'de': ':flagge_amerikanisch-samoa:',
        'es': ':bandera_samoa_americana:',
        'fr': ':drapeau_samoa_américaines:',
        'ja': ':旗_米領サモア:',
        'ko': ':깃발_아메리칸_사모아:',
        'pt': ':bandeira_samoa_americana:',
        'it': ':bandiera_samoa_americane:',
        'fa': ':پرچم_ساموآی_امریکا:',
        'id': ':bendera_samoa_amerika:',
        'zh': ':美属萨摩亚:',
        'ru': ':флаг_Американское_Самоа:'
    },
    '\U0001F1E6\U0001F1E9': {  # 🇦🇩
        'en': ':Andorra:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Andorra:', ':andorra:'],
        'de': ':flagge_andorra:',
        'es': ':bandera_andorra:',
        'fr': ':drapeau_andorre:',
        'ja': ':旗_アンドラ:',
        'ko': ':깃발_안도라:',
        'pt': ':bandeira_andorra:',
        'it': ':bandiera_andorra:',
        'fa': ':پرچم_آندورا:',
        'id': ':bendera_andorra:',
        'zh': ':安道尔:',
        'ru': ':флаг_Андорра:'
    },
    '\U0001F1E6\U0001F1F4': {  # 🇦🇴
        'en': ':Angola:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Angola:', ':angola:'],
        'de': ':flagge_angola:',
        'es': ':bandera_angola:',
        'fr': ':drapeau_angola:',
        'ja': ':旗_アンゴラ:',
        'ko': ':깃발_앙골라:',
        'pt': ':bandeira_angola:',
        'it': ':bandiera_angola:',
        'fa': ':پرچم_آنگولا:',
        'id': ':bendera_angola:',
        'zh': ':安哥拉:',
        'ru': ':флаг_Ангола:'
    },
    '\U0001F1E6\U0001F1EE': {  # 🇦🇮
        'en': ':Anguilla:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Anguilla:', ':anguilla:'],
        'de': ':flagge_anguilla:',
        'es': ':bandera_anguila:',
        'fr': ':drapeau_anguilla:',
        'ja': ':旗_アンギラ:',
        'ko': ':깃발_앵귈라:',
        'pt': ':bandeira_anguila:',
        'it': ':bandiera_anguilla:',
        'fa': ':پرچم_آنگویلا:',
        'id': ':bendera_anguilla:',
        'zh': ':安圭拉:',
        'ru': ':флаг_Ангилья:'
    },
    '\U0001F1E6\U0001F1F6': {  # 🇦🇶
        'en': ':Antarctica:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Antarctica:', ':antarctica:'],
        'de': ':flagge_antarktis:',
        'es': ':bandera_antártida:',
        'fr': ':drapeau_antarctique:',
        'ja': ':旗_南極:',
        'ko': ':깃발_남극_대륙:',
        'pt': ':bandeira_antártida:',
        'it': ':bandiera_antartide:',
        'fa': ':پرچم_جنوبگان:',
        'id': ':bendera_antarktika:',
        'zh': ':南极洲:',
        'ru': ':флаг_Антарктида:'
    },
    '\U0001F1E6\U0001F1EC': {  # 🇦🇬
        'en': ':Antigua_&_Barbuda:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Antigua_&_Barbuda:', ':antigua_barbuda:'],
        'de': ':flagge_antigua_und_barbuda:',
        'es': ':bandera_antigua_y_barbuda:',
        'fr': ':drapeau_antigua-et-barbuda:',
        'ja': ':旗_アンティグア・バーブーダ:',
        'ko': ':깃발_앤티가_바부다:',
        'pt': ':bandeira_antígua_e_barbuda:',
        'it': ':bandiera_antigua_e_barbuda:',
        'fa': ':پرچم_آنتیگوا_و_باربودا:',
        'id': ':bendera_antigua_dan_barbuda:',
        'zh': ':安提瓜和巴布达:',
        'ru': ':флаг_Антигуа_и_Барбуда:'
    },
    '\U00002652': {  # ♒
        'en': ':Aquarius:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':aquarius:'],
        'variant': True,
        'de': ':wassermann_(sternzeichen):',
        'es': ':acuario:',
        'fr': ':verseau:',
        'ja': ':みずがめ座:',
        'ko': ':물병자리:',
        'pt': ':signo_de_aquário:',
        'it': ':segno_zodiacale_dell’acquario:',
        'fa': ':برج_دلو:',
        'id': ':aquarius:',
        'zh': ':水瓶座:',
        'ru': ':знак_зодиака_Водолей:'
    },
    '\U0001F1E6\U0001F1F7': {  # 🇦🇷
        'en': ':Argentina:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Argentina:', ':argentina:'],
        'de': ':flagge_argentinien:',
        'es': ':bandera_argentina:',
        'fr': ':drapeau_argentine:',
        'ja': ':旗_アルゼンチン:',
        'ko': ':깃발_아르헨티나:',
        'pt': ':bandeira_argentina:',
        'it': ':bandiera_argentina:',
        'fa': ':پرچم_آرژانتین:',
        'id': ':bendera_argentina:',
        'zh': ':阿根廷:',
        'ru': ':флаг_Аргентина:'
    },
    '\U00002648': {  # ♈
        'en': ':Aries:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':aries:'],
        'variant': True,
        'de': ':widder_(sternzeichen):',
        'es': ':aries:',
        'fr': ':bélier_zodiaque:',
        'ja': ':おひつじ座:',
        'ko': ':양자리:',
        'pt': ':signo_de_áries:',
        'it': ':segno_zodiacale_dell’ariete:',
        'fa': ':برج_حمل:',
        'id': ':aries:',
        'zh': ':白羊座:',
        'ru': ':знак_зодиака_Овен:'
    },
    '\U0001F1E6\U0001F1F2': {  # 🇦🇲
        'en': ':Armenia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Armenia:', ':armenia:'],
        'de': ':flagge_armenien:',
        'es': ':bandera_armenia:',
        'fr': ':drapeau_arménie:',
        'ja': ':旗_アルメニア:',
        'ko': ':깃발_아르메니아:',
        'pt': ':bandeira_armênia:',
        'it': ':bandiera_armenia:',
        'fa': ':پرچم_ارمنستان:',
        'id': ':bendera_armenia:',
        'zh': ':亚美尼亚:',
        'ru': ':флаг_Армения:'
    },
    '\U0001F1E6\U0001F1FC': {  # 🇦🇼
        'en': ':Aruba:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Aruba:', ':aruba:'],
        'de': ':flagge_aruba:',
        'es': ':bandera_aruba:',
        'fr': ':drapeau_aruba:',
        'ja': ':旗_アルバ:',
        'ko': ':깃발_아루바:',
        'pt': ':bandeira_aruba:',
        'it': ':bandiera_aruba:',
        'fa': ':پرچم_آروبا:',
        'id': ':bendera_aruba:',
        'zh': ':阿鲁巴:',
        'ru': ':флаг_Аруба:'
    },
    '\U0001F1E6\U0001F1E8': {  # 🇦🇨
        'en': ':Ascension_Island:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Ascension_Island:', ':ascension_island:'],
        'de': ':flagge_ascension:',
        'es': ':bandera_isla_de_la_ascensión:',
        'fr': ':drapeau_île_de_l’ascension:',
        'ja': ':旗_アセンション島:',
        'ko': ':깃발_어센션섬:',
        'pt': ':bandeira_ilha_de_ascensão:',
        'it': ':bandiera_isola_ascensione:',
        'fa': ':پرچم_جزایر_آسنسیون:',
        'id': ':bendera_pulau_ascension:',
        'zh': ':阿森松岛:',
        'ru': ':флаг_о_в_Вознесения:'
    },
    '\U0001F1E6\U0001F1FA': {  # 🇦🇺
        'en': ':Australia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Australia:', ':australia:'],
        'de': ':flagge_australien:',
        'es': ':bandera_australia:',
        'fr': ':drapeau_australie:',
        'ja': ':旗_オーストラリア:',
        'ko': ':깃발_오스트레일리아:',
        'pt': ':bandeira_austrália:',
        'it': ':bandiera_australia:',
        'fa': ':پرچم_استرالیا:',
        'id': ':bendera_australia:',
        'zh': ':澳大利亚:',
        'ru': ':флаг_Австралия:'
    },
    '\U0001F1E6\U0001F1F9': {  # 🇦🇹
        'en': ':Austria:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Austria:', ':austria:'],
        'de': ':flagge_österreich:',
        'es': ':bandera_austria:',
        'fr': ':drapeau_autriche:',
        'ja': ':旗_オーストリア:',
        'ko': ':깃발_오스트리아:',
        'pt': ':bandeira_áustria:',
        'it': ':bandiera_austria:',
        'fa': ':پرچم_اتریش:',
        'id': ':bendera_austria:',
        'zh': ':奥地利:',
        'ru': ':флаг_Австрия:'
    },
    '\U0001F1E6\U0001F1FF': {  # 🇦🇿
        'en': ':Azerbaijan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Azerbaijan:', ':azerbaijan:'],
        'de': ':flagge_aserbaidschan:',
        'es': ':bandera_azerbaiyán:',
        'fr': ':drapeau_azerbaïdjan:',
        'ja': ':旗_アゼルバイジャン:',
        'ko': ':깃발_아제르바이잔:',
        'pt': ':bandeira_azerbaijão:',
        'it': ':bandiera_azerbaigian:',
        'fa': ':پرچم_جمهوری_آذربایجان:',
        'id': ':bendera_azerbaijan:',
        'zh': ':阿塞拜疆:',
        'ru': ':флаг_Азербайджан:'
    },
    '\U0001F519': {  # 🔙
        'en': ':BACK_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':back:', ':back_arrow:'],
        'de': ':back-pfeil:',
        'es': ':flecha_back:',
        'fr': ':flèche_retour:',
        'ja': ':back矢印:',
        'ko': ':뒤로:',
        'pt': ':seta_back:',
        'it': ':freccia_back:',
        'fa': ':پیکان_عقب:',
        'id': ':tanda_panah_back:',
        'zh': ':返回箭头:',
        'ru': ':стрелка_«назад»:'
    },
    '\U0001F171\U0000FE0F': {  # 🅱️
        'en': ':B_button_(blood_type):',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':b:', ':b_button_blood_type:'],
        'variant': True,
        'de': ':großbuchstabe_b_in_rotem_quadrat:',
        'es': ':grupo_sanguíneo_b:',
        'fr': ':groupe_sanguin_b:',
        'ja': ':血液型b型:',
        'ko': ':비형:',
        'pt': ':botão_b_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_b:',
        'fa': ':دکمه_ب_(گروه_خونی):',
        'id': ':tombol_b_(golongan_darah):',
        'zh': ':B型血:',
        'ru': ':iii_группа_крови:'
    },
    '\U0001F171': {  # 🅱
        'en': ':B_button_(blood_type):',
        'status': unqualified,
        'E': 0.6,
        'alias': [':b:', ':b_button_blood_type:'],
        'variant': True,
        'de': ':großbuchstabe_b_in_rotem_quadrat:',
        'es': ':grupo_sanguíneo_b:',
        'fr': ':groupe_sanguin_b:',
        'ja': ':血液型b型:',
        'ko': ':비형:',
        'pt': ':botão_b_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_b:',
        'fa': ':دکمه_ب_(گروه_خونی):',
        'id': ':tombol_b_(golongan_darah):',
        'zh': ':B型血:',
        'ru': ':III_группа_крови:'
    },
    '\U0001F1E7\U0001F1F8': {  # 🇧🇸
        'en': ':Bahamas:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bahamas:', ':bahamas:'],
        'de': ':flagge_bahamas:',
        'es': ':bandera_bahamas:',
        'fr': ':drapeau_bahamas:',
        'ja': ':旗_バハマ:',
        'ko': ':깃발_바하마:',
        'pt': ':bandeira_bahamas:',
        'it': ':bandiera_bahamas:',
        'fa': ':پرچم_باهاما:',
        'id': ':bendera_bahama:',
        'zh': ':巴哈马:',
        'ru': ':флаг_Багамы:'
    },
    '\U0001F1E7\U0001F1ED': {  # 🇧🇭
        'en': ':Bahrain:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bahrain:', ':bahrain:'],
        'de': ':flagge_bahrain:',
        'es': ':bandera_baréin:',
        'fr': ':drapeau_bahreïn:',
        'ja': ':旗_バーレーン:',
        'ko': ':깃발_바레인:',
        'pt': ':bandeira_barein:',
        'it': ':bandiera_bahrein:',
        'fa': ':پرچم_بحرین:',
        'id': ':bendera_bahrain:',
        'zh': ':巴林:',
        'ru': ':флаг_Бахрейн:'
    },
    '\U0001F1E7\U0001F1E9': {  # 🇧🇩
        'en': ':Bangladesh:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bangladesh:', ':bangladesh:'],
        'de': ':flagge_bangladesch:',
        'es': ':bandera_bangladés:',
        'fr': ':drapeau_bangladesh:',
        'ja': ':旗_バングラデシュ:',
        'ko': ':깃발_방글라데시:',
        'pt': ':bandeira_bangladesh:',
        'it': ':bandiera_bangladesh:',
        'fa': ':پرچم_بنگلادش:',
        'id': ':bendera_bangladesh:',
        'zh': ':孟加拉国:',
        'ru': ':флаг_Бангладеш:'
    },
    '\U0001F1E7\U0001F1E7': {  # 🇧🇧
        'en': ':Barbados:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Barbados:', ':barbados:'],
        'de': ':flagge_barbados:',
        'es': ':bandera_barbados:',
        'fr': ':drapeau_barbade:',
        'ja': ':旗_バルバドス:',
        'ko': ':깃발_바베이도스:',
        'pt': ':bandeira_barbados:',
        'it': ':bandiera_barbados:',
        'fa': ':پرچم_باربادوس:',
        'id': ':bendera_barbados:',
        'zh': ':巴巴多斯:',
        'ru': ':флаг_Барбадос:'
    },
    '\U0001F1E7\U0001F1FE': {  # 🇧🇾
        'en': ':Belarus:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Belarus:', ':belarus:'],
        'de': ':flagge_belarus:',
        'es': ':bandera_bielorrusia:',
        'fr': ':drapeau_biélorussie:',
        'ja': ':旗_ベラルーシ:',
        'ko': ':깃발_벨라루스:',
        'pt': ':bandeira_bielorrússia:',
        'it': ':bandiera_bielorussia:',
        'fa': ':پرچم_بلاروس:',
        'id': ':bendera_belarus:',
        'zh': ':白俄罗斯:',
        'ru': ':флаг_Беларусь:'
    },
    '\U0001F1E7\U0001F1EA': {  # 🇧🇪
        'en': ':Belgium:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Belgium:', ':belgium:'],
        'de': ':flagge_belgien:',
        'es': ':bandera_bélgica:',
        'fr': ':drapeau_belgique:',
        'ja': ':旗_ベルギー:',
        'ko': ':깃발_벨기에:',
        'pt': ':bandeira_bélgica:',
        'it': ':bandiera_belgio:',
        'fa': ':پرچم_بلژیک:',
        'id': ':bendera_belgia:',
        'zh': ':比利时:',
        'ru': ':флаг_Бельгия:'
    },
    '\U0001F1E7\U0001F1FF': {  # 🇧🇿
        'en': ':Belize:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Belize:', ':belize:'],
        'de': ':flagge_belize:',
        'es': ':bandera_belice:',
        'fr': ':drapeau_belize:',
        'ja': ':旗_ベリーズ:',
        'ko': ':깃발_벨리즈:',
        'pt': ':bandeira_belize:',
        'it': ':bandiera_belize:',
        'fa': ':پرچم_بلیز:',
        'id': ':bendera_belize:',
        'zh': ':伯利兹:',
        'ru': ':флаг_Белиз:'
    },
    '\U0001F1E7\U0001F1EF': {  # 🇧🇯
        'en': ':Benin:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Benin:', ':benin:'],
        'de': ':flagge_benin:',
        'es': ':bandera_benín:',
        'fr': ':drapeau_bénin:',
        'ja': ':旗_ベナン:',
        'ko': ':깃발_베냉:',
        'pt': ':bandeira_benin:',
        'it': ':bandiera_benin:',
        'fa': ':پرچم_بنین:',
        'id': ':bendera_benin:',
        'zh': ':贝宁:',
        'ru': ':флаг_Бенин:'
    },
    '\U0001F1E7\U0001F1F2': {  # 🇧🇲
        'en': ':Bermuda:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bermuda:', ':bermuda:'],
        'de': ':flagge_bermuda:',
        'es': ':bandera_bermudas:',
        'fr': ':drapeau_bermudes:',
        'ja': ':旗_バミューダ:',
        'ko': ':깃발_버뮤다:',
        'pt': ':bandeira_bermudas:',
        'it': ':bandiera_bermuda:',
        'fa': ':پرچم_برمودا:',
        'id': ':bendera_bermuda:',
        'zh': ':百慕大:',
        'ru': ':флаг_Бермудские_о_ва:'
    },
    '\U0001F1E7\U0001F1F9': {  # 🇧🇹
        'en': ':Bhutan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bhutan:', ':bhutan:'],
        'de': ':flagge_bhutan:',
        'es': ':bandera_bután:',
        'fr': ':drapeau_bhoutan:',
        'ja': ':旗_ブータン:',
        'ko': ':깃발_부탄:',
        'pt': ':bandeira_butão:',
        'it': ':bandiera_bhutan:',
        'fa': ':پرچم_بوتان:',
        'id': ':bendera_bhutan:',
        'zh': ':不丹:',
        'ru': ':флаг_Бутан:'
    },
    '\U0001F1E7\U0001F1F4': {  # 🇧🇴
        'en': ':Bolivia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bolivia:', ':bolivia:'],
        'de': ':flagge_bolivien:',
        'es': ':bandera_bolivia:',
        'fr': ':drapeau_bolivie:',
        'ja': ':旗_ボリビア:',
        'ko': ':깃발_볼리비아:',
        'pt': ':bandeira_bolívia:',
        'it': ':bandiera_bolivia:',
        'fa': ':پرچم_بولیوی:',
        'id': ':bendera_bolivia:',
        'zh': ':玻利维亚:',
        'ru': ':флаг_Боливия:'
    },
    '\U0001F1E7\U0001F1E6': {  # 🇧🇦
        'en': ':Bosnia_&_Herzegovina:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bosnia_&_Herzegovina:', ':bosnia_herzegovina:'],
        'de': ':flagge_bosnien_und_herzegowina:',
        'es': ':bandera_bosnia_y_herzegovina:',
        'fr': ':drapeau_bosnie-herzégovine:',
        'ja': ':旗_ボスニア・ヘルツェゴビナ:',
        'ko': ':깃발_보스니아_헤르체고비나:',
        'pt': ':bandeira_bósnia_e_herzegovina:',
        'it': ':bandiera_bosnia_ed_erzegovina:',
        'fa': ':پرچم_بوسنی_و_هرزگوین:',
        'id': ':bendera_bosnia_dan_herzegovina:',
        'zh': ':波斯尼亚和黑塞哥维那:',
        'ru': ':флаг_Босния_и_Герцеговина:'
    },
    '\U0001F1E7\U0001F1FC': {  # 🇧🇼
        'en': ':Botswana:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Botswana:', ':botswana:'],
        'de': ':flagge_botsuana:',
        'es': ':bandera_botsuana:',
        'fr': ':drapeau_botswana:',
        'ja': ':旗_ボツワナ:',
        'ko': ':깃발_보츠와나:',
        'pt': ':bandeira_botsuana:',
        'it': ':bandiera_botswana:',
        'fa': ':پرچم_بوتسوانا:',
        'id': ':bendera_botswana:',
        'zh': ':博茨瓦纳:',
        'ru': ':флаг_Ботсвана:'
    },
    '\U0001F1E7\U0001F1FB': {  # 🇧🇻
        'en': ':Bouvet_Island:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bouvet_Island:', ':bouvet_island:'],
        'de': ':flagge_bouvetinsel:',
        'es': ':bandera_isla_bouvet:',
        'fr': ':drapeau_île_bouvet:',
        'ja': ':旗_ブーベ島:',
        'ko': ':깃발_부베섬:',
        'pt': ':bandeira_ilha_bouvet:',
        'it': ':bandiera_isola_bouvet:',
        'fa': ':پرچم_جزیرهٔ_بووه:',
        'id': ':bendera_pulau_bouvet:',
        'zh': ':布韦岛:',
        'ru': ':флаг_о_в_Буве:'
    },
    '\U0001F1E7\U0001F1F7': {  # 🇧🇷
        'en': ':Brazil:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Brazil:', ':brazil:'],
        'de': ':flagge_brasilien:',
        'es': ':bandera_brasil:',
        'fr': ':drapeau_brésil:',
        'ja': ':旗_ブラジル:',
        'ko': ':깃발_브라질:',
        'pt': ':bandeira_brasil:',
        'it': ':bandiera_brasile:',
        'fa': ':پرچم_برزیل:',
        'id': ':bendera_brasil:',
        'zh': ':巴西:',
        'ru': ':флаг_Бразилия:'
    },
    '\U0001F1EE\U0001F1F4': {  # 🇮🇴
        'en': ':British_Indian_Ocean_Territory:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_British_Indian_Ocean_Territory:', ':british_indian_ocean_territory:'],
        'de': ':flagge_britisches_territorium_im_indischen_ozean:',
        'es': ':bandera_territorio_británico_del_océano_índico:',
        'fr': ':drapeau_territoire_britannique_de_l’océan_indien:',
        'ja': ':旗_英領インド洋地域:',
        'ko': ':깃발_영국령_인도양_식민지:',
        'pt': ':bandeira_território_britânico_do_oceano_índico:',
        'it': ':bandiera_territorio_britannico_dell’oceano_indiano:',
        'fa': ':پرچم_قلمرو_بریتانیا_در_اقیانوس_هند:',
        'id': ':bendera_wilayah_inggris_di_samudra_hindia:',
        'zh': ':英属印度洋领地:',
        'ru': ':флаг_Британская_территория_в_Индийском_океане:'
    },
    '\U0001F1FB\U0001F1EC': {  # 🇻🇬
        'en': ':British_Virgin_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_British_Virgin_Islands:', ':british_virgin_islands:'],
        'de': ':flagge_britische_jungferninseln:',
        'es': ':bandera_islas_vírgenes_británicas:',
        'fr': ':drapeau_îles_vierges_britanniques:',
        'ja': ':旗_英領ヴァージン諸島:',
        'ko': ':깃발_영국령_버진아일랜드:',
        'pt': ':bandeira_ilhas_virgens_britânicas:',
        'it': ':bandiera_isole_vergini_britanniche:',
        'fa': ':پرچم_جزایر_ویرجین_بریتانیا:',
        'id': ':bendera_kepulauan_virgin_britania_raya:',
        'zh': ':英属维尔京群岛:',
        'ru': ':флаг_Виргинские_о_ва_(Великобритания):'
    },
    '\U0001F1E7\U0001F1F3': {  # 🇧🇳
        'en': ':Brunei:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Brunei:', ':brunei:'],
        'de': ':flagge_brunei_darussalam:',
        'es': ':bandera_brunéi:',
        'fr': ':drapeau_brunei:',
        'ja': ':旗_ブルネイ:',
        'ko': ':깃발_브루나이:',
        'pt': ':bandeira_brunei:',
        'it': ':bandiera_brunei:',
        'fa': ':پرچم_برونئی:',
        'id': ':bendera_brunei:',
        'zh': ':文莱:',
        'ru': ':флаг_Бруней:'
    },
    '\U0001F1E7\U0001F1EC': {  # 🇧🇬
        'en': ':Bulgaria:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Bulgaria:', ':bulgaria:'],
        'de': ':flagge_bulgarien:',
        'es': ':bandera_bulgaria:',
        'fr': ':drapeau_bulgarie:',
        'ja': ':旗_ブルガリア:',
        'ko': ':깃발_불가리아:',
        'pt': ':bandeira_bulgária:',
        'it': ':bandiera_bulgaria:',
        'fa': ':پرچم_بلغارستان:',
        'id': ':bendera_bulgaria:',
        'zh': ':保加利亚:',
        'ru': ':флаг_Болгария:'
    },
    '\U0001F1E7\U0001F1EB': {  # 🇧🇫
        'en': ':Burkina_Faso:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Burkina_Faso:', ':burkina_faso:'],
        'de': ':flagge_burkina_faso:',
        'es': ':bandera_burkina_faso:',
        'fr': ':drapeau_burkina_faso:',
        'ja': ':旗_ブルキナファソ:',
        'ko': ':깃발_부르키나파소:',
        'pt': ':bandeira_burquina_faso:',
        'it': ':bandiera_burkina_faso:',
        'fa': ':پرچم_بورکینافاسو:',
        'id': ':bendera_burkina_faso:',
        'zh': ':布基纳法索:',
        'ru': ':флаг_Буркина_Фасо:'
    },
    '\U0001F1E7\U0001F1EE': {  # 🇧🇮
        'en': ':Burundi:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Burundi:', ':burundi:'],
        'de': ':flagge_burundi:',
        'es': ':bandera_burundi:',
        'fr': ':drapeau_burundi:',
        'ja': ':旗_ブルンジ:',
        'ko': ':깃발_부룬디:',
        'pt': ':bandeira_burundi:',
        'it': ':bandiera_burundi:',
        'fa': ':پرچم_بوروندی:',
        'id': ':bendera_burundi:',
        'zh': ':布隆迪:',
        'ru': ':флаг_Бурунди:'
    },
    '\U0001F191': {  # 🆑
        'en': ':CL_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cl:', ':cl_button:'],
        'de': ':großbuchstaben_cl_in_rotem_quadrat:',
        'es': ':borrar:',
        'fr': ':bouton_effacer:',
        'ja': ':clマーク:',
        'ko': ':씨엘:',
        'pt': ':botão_cl:',
        'it': ':pulsante_cl:',
        'fa': ':دکمهٔ_cl:',
        'id': ':tombol_cl:',
        'zh': ':CL按钮:',
        'ru': ':кнопка_«очистить»:'
    },
    '\U0001F192': {  # 🆒
        'en': ':COOL_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cool:', ':cool_button:'],
        'de': ':wort_cool_in_blauem_quadrat:',
        'es': ':botón_cool:',
        'fr': ':bouton_cool:',
        'ja': ':coolマーク:',
        'ko': ':쿨:',
        'pt': ':botão_cool:',
        'it': ':pulsante_cool:',
        'fa': ':دکمهٔ_cool:',
        'id': ':tombol_cool:',
        'zh': ':cool按钮:',
        'ru': ':значок_«круто»:'
    },
    '\U0001F1F0\U0001F1ED': {  # 🇰🇭
        'en': ':Cambodia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Cambodia:', ':cambodia:'],
        'de': ':flagge_kambodscha:',
        'es': ':bandera_camboya:',
        'fr': ':drapeau_cambodge:',
        'ja': ':旗_カンボジア:',
        'ko': ':깃발_캄보디아:',
        'pt': ':bandeira_camboja:',
        'it': ':bandiera_cambogia:',
        'fa': ':پرچم_کامبوج:',
        'id': ':bendera_kamboja:',
        'zh': ':柬埔寨:',
        'ru': ':флаг_Камбоджа:'
    },
    '\U0001F1E8\U0001F1F2': {  # 🇨🇲
        'en': ':Cameroon:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Cameroon:', ':cameroon:'],
        'de': ':flagge_kamerun:',
        'es': ':bandera_camerún:',
        'fr': ':drapeau_cameroun:',
        'ja': ':旗_カメルーン:',
        'ko': ':깃발_카메룬:',
        'pt': ':bandeira_camarões:',
        'it': ':bandiera_camerun:',
        'fa': ':پرچم_کامرون:',
        'id': ':bendera_kamerun:',
        'zh': ':喀麦隆:',
        'ru': ':флаг_Камерун:'
    },
    '\U0001F1E8\U0001F1E6': {  # 🇨🇦
        'en': ':Canada:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Canada:', ':canada:'],
        'de': ':flagge_kanada:',
        'es': ':bandera_canadá:',
        'fr': ':drapeau_canada:',
        'ja': ':旗_カナダ:',
        'ko': ':깃발_캐나다:',
        'pt': ':bandeira_canadá:',
        'it': ':bandiera_canada:',
        'fa': ':پرچم_کانادا:',
        'id': ':bendera_kanada:',
        'zh': ':加拿大:',
        'ru': ':флаг_Канада:'
    },
    '\U0001F1EE\U0001F1E8': {  # 🇮🇨
        'en': ':Canary_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Canary_Islands:', ':canary_islands:'],
        'de': ':flagge_kanarische_inseln:',
        'es': ':bandera_canarias:',
        'fr': ':drapeau_îles_canaries:',
        'ja': ':旗_カナリア諸島:',
        'ko': ':깃발_카나리아_제도:',
        'pt': ':bandeira_ilhas_canárias:',
        'it': ':bandiera_isole_canarie:',
        'fa': ':پرچم_جزایر_قناری:',
        'id': ':bendera_kepulauan_canaria:',
        'zh': ':加纳利群岛:',
        'ru': ':флаг_Канарские_о_ва:'
    },
    '\U0000264B': {  # ♋
        'en': ':Cancer:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cancer:'],
        'variant': True,
        'de': ':krebs_(sternzeichen):',
        'es': ':cáncer:',
        'fr': ':cancer:',
        'ja': ':かに座:',
        'ko': ':게자리:',
        'pt': ':signo_de_câncer:',
        'it': ':segno_zodiacale_del_cancro:',
        'fa': ':برج_سرطان:',
        'id': ':cancer:',
        'zh': ':巨蟹座:',
        'ru': ':знак_зодиака_Рак:'
    },
    '\U0001F1E8\U0001F1FB': {  # 🇨🇻
        'en': ':Cape_Verde:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Cape_Verde:', ':cape_verde:'],
        'de': ':flagge_cabo_verde:',
        'es': ':bandera_cabo_verde:',
        'fr': ':drapeau_cap-vert:',
        'ja': ':旗_カーボベルデ:',
        'ko': ':깃발_카보베르데:',
        'pt': ':bandeira_cabo_verde:',
        'it': ':bandiera_capo_verde:',
        'fa': ':پرچم_کیپ_ورد:',
        'id': ':bendera_tanjung_verde:',
        'zh': ':佛得角:',
        'ru': ':флаг_Кабо_Верде:'
    },
    '\U00002651': {  # ♑
        'en': ':Capricorn:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':capricorn:'],
        'variant': True,
        'de': ':steinbock_(sternzeichen):',
        'es': ':capricornio:',
        'fr': ':capricorne:',
        'ja': ':やぎ座:',
        'ko': ':염소자리:',
        'pt': ':signo_de_capricórnio:',
        'it': ':segno_zodiacale_del_capricorno:',
        'fa': ':برج_جدی:',
        'id': ':capricorn:',
        'zh': ':摩羯座:',
        'ru': ':знак_зодиака_Козерог:'
    },
    '\U0001F1E7\U0001F1F6': {  # 🇧🇶
        'en': ':Caribbean_Netherlands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Caribbean_Netherlands:', ':caribbean_netherlands:'],
        'de': ':flagge_karibische_niederlande:',
        'es': ':bandera_caribe_neerlandés:',
        'fr': ':drapeau_pays-bas_caribéens:',
        'ja': ':旗_オランダ領カリブ:',
        'ko': ':깃발_네덜란드령_카리브:',
        'pt': ':bandeira_países_baixos_caribenhos:',
        'it': ':bandiera_caraibi_olandesi:',
        'fa': ':پرچم_جزایر_کارائیب_هلند:',
        'id': ':bendera_belanda_karibia:',
        'zh': ':荷属加勒比区:',
        'ru': ':флаг_Бонэйр_Синт_Эстатиус_и_Саба:'
    },
    '\U0001F1F0\U0001F1FE': {  # 🇰🇾
        'en': ':Cayman_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Cayman_Islands:', ':cayman_islands:'],
        'de': ':flagge_kaimaninseln:',
        'es': ':bandera_islas_caimán:',
        'fr': ':drapeau_îles_caïmans:',
        'ja': ':旗_ケイマン諸島:',
        'ko': ':깃발_케이맨_제도:',
        'pt': ':bandeira_ilhas_cayman:',
        'it': ':bandiera_isole_cayman:',
        'fa': ':پرچم_جزایر_کِیمن:',
        'id': ':bendera_kepulauan_cayman:',
        'zh': ':开曼群岛:',
        'ru': ':флаг_о_ва_Кайман:'
    },
    '\U0001F1E8\U0001F1EB': {  # 🇨🇫
        'en': ':Central_African_Republic:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Central_African_Republic:', ':central_african_republic:'],
        'de': ':flagge_zentralafrikanische_republik:',
        'es': ':bandera_república_centroafricana:',
        'fr': ':drapeau_république_centrafricaine:',
        'ja': ':旗_中央アフリカ共和国:',
        'ko': ':깃발_중앙_아프리카_공화국:',
        'pt': ':bandeira_república_centro-africana:',
        'it': ':bandiera_repubblica_centrafricana:',
        'fa': ':پرچم_جمهوری_افریقای_مرکزی:',
        'id': ':bendera_republik_afrika_tengah:',
        'zh': ':中非共和国:',
        'ru': ':флаг_Центрально_Африканская_Республика:'
    },
    '\U0001F1EA\U0001F1E6': {  # 🇪🇦
        'en': ':Ceuta_&_Melilla:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Ceuta_&_Melilla:', ':ceuta_melilla:'],
        'de': ':flagge_ceuta_und_melilla:',
        'es': ':bandera_ceuta_y_melilla:',
        'fr': ':drapeau_ceuta_et_melilla:',
        'ja': ':旗_セウタ・メリリャ:',
        'ko': ':깃발_세우타_및_멜리야:',
        'pt': ':bandeira_ceuta_e_melilla:',
        'it': ':bandiera_ceuta_e_melilla:',
        'fa': ':پرچم_سبته_و_ملیله:',
        'id': ':bendera_ceuta_dan_melilla:',
        'zh': ':休达及梅利利亚:',
        'ru': ':флаг_Сеута_и_Мелилья:'
    },
    '\U0001F1F9\U0001F1E9': {  # 🇹🇩
        'en': ':Chad:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Chad:', ':chad:'],
        'de': ':flagge_tschad:',
        'es': ':bandera_chad:',
        'fr': ':drapeau_tchad:',
        'ja': ':旗_チャド:',
        'ko': ':깃발_차드:',
        'pt': ':bandeira_chade:',
        'it': ':bandiera_ciad:',
        'fa': ':پرچم_چاد:',
        'id': ':bendera_chad:',
        'zh': ':乍得:',
        'ru': ':флаг_Чад:'
    },
    '\U0001F1E8\U0001F1F1': {  # 🇨🇱
        'en': ':Chile:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Chile:', ':chile:'],
        'de': ':flagge_chile:',
        'es': ':bandera_chile:',
        'fr': ':drapeau_chili:',
        'ja': ':旗_チリ:',
        'ko': ':깃발_칠레:',
        'pt': ':bandeira_chile:',
        'it': ':bandiera_cile:',
        'fa': ':پرچم_شیلی:',
        'id': ':bendera_cile:',
        'zh': ':智利:',
        'ru': ':флаг_Чили:'
    },
    '\U0001F1E8\U0001F1F3': {  # 🇨🇳
        'en': ':China:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_China:', ':cn:'],
        'de': ':flagge_china:',
        'es': ':bandera_china:',
        'fr': ':drapeau_chine:',
        'ja': ':旗_中国:',
        'ko': ':깃발_중국:',
        'pt': ':bandeira_china:',
        'it': ':bandiera_cina:',
        'fa': ':پرچم_چین:',
        'id': ':bendera_tiongkok:',
        'zh': ':中国:',
        'ru': ':флаг_Китай:'
    },
    '\U0001F1E8\U0001F1FD': {  # 🇨🇽
        'en': ':Christmas_Island:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Christmas_Island:', ':christmas_island:'],
        'de': ':flagge_weihnachtsinsel:',
        'es': ':bandera_isla_de_navidad:',
        'fr': ':drapeau_île_christmas:',
        'ja': ':旗_クリスマス島:',
        'ko': ':깃발_크리스마스섬:',
        'pt': ':bandeira_ilha_christmas:',
        'it': ':bandiera_isola_christmas:',
        'fa': ':پرچم_جزیرهٔ_کریسمس:',
        'id': ':bendera_pulau_natal:',
        'zh': ':圣诞岛:',
        'ru': ':флаг_о_в_Рождества:'
    },
    '\U0001F384': {  # 🎄
        'en': ':Christmas_tree:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':christmas_tree:'],
        'de': ':weihnachtsbaum:',
        'es': ':árbol_de_navidad:',
        'fr': ':sapin_de_noël:',
        'ja': ':クリスマスツリー:',
        'ko': ':크리스마스_트리:',
        'pt': ':árvore_de_natal:',
        'it': ':albero_di_natale:',
        'fa': ':درخت_کریسمس:',
        'id': ':pohon_natal:',
        'zh': ':圣诞树:',
        'ru': ':новогодняя_елка:'
    },
    '\U0001F1E8\U0001F1F5': {  # 🇨🇵
        'en': ':Clipperton_Island:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Clipperton_Island:', ':clipperton_island:'],
        'de': ':flagge_clipperton-insel:',
        'es': ':bandera_isla_clipperton:',
        'fr': ':drapeau_île_clipperton:',
        'ja': ':旗_クリッパートン島:',
        'ko': ':깃발_클리퍼턴섬:',
        'pt': ':bandeira_ilha_de_clipperton:',
        'it': ':bandiera_isola_di_clipperton:',
        'fa': ':پرچم_جزایر_کلیپرتون:',
        'id': ':bendera_pulau_clipperton:',
        'zh': ':克利珀顿岛:',
        'ru': ':флаг_о_в_Клиппертон:'
    },
    '\U0001F1E8\U0001F1E8': {  # 🇨🇨
        'en': ':Cocos_(Keeling)_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Cocos_Islands:', ':cocos_islands:'],
        'de': ':flagge_kokosinseln:',
        'es': ':bandera_islas_cocos:',
        'fr': ':drapeau_îles_cocos:',
        'ja': ':旗_ココス(キーリング)諸島:',
        'ko': ':깃발_코코스_제도:',
        'pt': ':bandeira_ilhas_cocos_(keeling):',
        'it': ':bandiera_isole_cocos_(keeling):',
        'fa': ':پرچم_جزایر_کوکوس:',
        'id': ':bendera_kepulauan_cocos_(keeling):',
        'zh': ':科科斯(基林)群岛:',
        'ru': ':флаг_Кокосовые_о_ва:'
    },
    '\U0001F1E8\U0001F1F4': {  # 🇨🇴
        'en': ':Colombia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Colombia:', ':colombia:'],
        'de': ':flagge_kolumbien:',
        'es': ':bandera_colombia:',
        'fr': ':drapeau_colombie:',
        'ja': ':旗_コロンビア:',
        'ko': ':깃발_콜롬비아:',
        'pt': ':bandeira_colômbia:',
        'it': ':bandiera_colombia:',
        'fa': ':پرچم_کلمبیا:',
        'id': ':bendera_kolombia:',
        'zh': ':哥伦比亚:',
        'ru': ':флаг_Колумбия:'
    },
    '\U0001F1F0\U0001F1F2': {  # 🇰🇲
        'en': ':Comoros:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Comoros:', ':comoros:'],
        'de': ':flagge_komoren:',
        'es': ':bandera_comoras:',
        'fr': ':drapeau_comores:',
        'ja': ':旗_コモロ:',
        'ko': ':깃발_코모로:',
        'pt': ':bandeira_comores:',
        'it': ':bandiera_comore:',
        'fa': ':پرچم_کومور:',
        'id': ':bendera_komoro:',
        'zh': ':科摩罗:',
        'ru': ':флаг_Коморы:'
    },
    '\U0001F1E8\U0001F1EC': {  # 🇨🇬
        'en': ':Congo-Brazzaville:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Congo_Brazzaville:', ':congo_brazzaville:'],
        'de': ':flagge_kongo-brazzaville:',
        'es': ':bandera_congo:',
        'fr': ':drapeau_congo-brazzaville:',
        'ja': ':旗_コンゴ共和国(ブラザビル):',
        'ko': ':깃발_콩고-브라자빌:',
        'pt': ':bandeira_república_do_congo:',
        'it': ':bandiera_congo-brazzaville:',
        'fa': ':پرچم_کنگو-برازویل:',
        'id': ':bendera_kongo-brazzaville:',
        'zh': ':刚果_布:',
        'ru': ':флаг_Конго_Браззавиль:'
    },
    '\U0001F1E8\U0001F1E9': {  # 🇨🇩
        'en': ':Congo-Kinshasa:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Congo_Kinshasa:', ':congo_kinshasa:'],
        'de': ':flagge_kongo-kinshasa:',
        'es': ':bandera_república_democrática_del_congo:',
        'fr': ':drapeau_congo-kinshasa:',
        'ja': ':旗_コンゴ民主共和国(キンシャサ):',
        'ko': ':깃발_콩고-킨샤사:',
        'pt': ':bandeira_congo-kinshasa:',
        'it': ':bandiera_congo-kinshasa:',
        'fa': ':پرچم_کنگو-کینشاسا:',
        'id': ':bendera_kongo-kinshasa:',
        'zh': ':刚果_金:',
        'ru': ':флаг_Конго_Киншаса:'
    },
    '\U0001F1E8\U0001F1F0': {  # 🇨🇰
        'en': ':Cook_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Cook_Islands:', ':cook_islands:'],
        'de': ':flagge_cookinseln:',
        'es': ':bandera_islas_cook:',
        'fr': ':drapeau_îles_cook:',
        'ja': ':旗_クック諸島:',
        'ko': ':깃발_쿡_제도:',
        'pt': ':bandeira_ilhas_cook:',
        'it': ':bandiera_isole_cook:',
        'fa': ':پرچم_جزایر_کوک:',
        'id': ':bendera_kepulauan_cook:',
        'zh': ':库克群岛:',
        'ru': ':флаг_о_ва_Кука:'
    },
    '\U0001F1E8\U0001F1F7': {  # 🇨🇷
        'en': ':Costa_Rica:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Costa_Rica:', ':costa_rica:'],
        'de': ':flagge_costa_rica:',
        'es': ':bandera_costa_rica:',
        'fr': ':drapeau_costa_rica:',
        'ja': ':旗_コスタリカ:',
        'ko': ':깃발_코스타리카:',
        'pt': ':bandeira_costa_rica:',
        'it': ':bandiera_costa_rica:',
        'fa': ':پرچم_کاستاریکا:',
        'id': ':bendera_kosta_rika:',
        'zh': ':哥斯达黎加:',
        'ru': ':флаг_Коста_Рика:'
    },
    '\U0001F1ED\U0001F1F7': {  # 🇭🇷
        'en': ':Croatia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Croatia:', ':croatia:'],
        'de': ':flagge_kroatien:',
        'es': ':bandera_croacia:',
        'fr': ':drapeau_croatie:',
        'ja': ':旗_クロアチア:',
        'ko': ':깃발_크로아티아:',
        'pt': ':bandeira_croácia:',
        'it': ':bandiera_croazia:',
        'fa': ':پرچم_کرواسی:',
        'id': ':bendera_kroasia:',
        'zh': ':克罗地亚:',
        'ru': ':флаг_Хорватия:'
    },
    '\U0001F1E8\U0001F1FA': {  # 🇨🇺
        'en': ':Cuba:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Cuba:', ':cuba:'],
        'de': ':flagge_kuba:',
        'es': ':bandera_cuba:',
        'fr': ':drapeau_cuba:',
        'ja': ':旗_キューバ:',
        'ko': ':깃발_쿠바:',
        'pt': ':bandeira_cuba:',
        'it': ':bandiera_cuba:',
        'fa': ':پرچم_کوبا:',
        'id': ':bendera_kuba:',
        'zh': ':古巴:',
        'ru': ':флаг_Куба:'
    },
    '\U0001F1E8\U0001F1FC': {  # 🇨🇼
        'en': ':Curaçao:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Curaçao:', ':curacao:'],
        'de': ':flagge_curaçao:',
        'es': ':bandera_curazao:',
        'fr': ':drapeau_curaçao:',
        'ja': ':旗_キュラソー:',
        'ko': ':깃발_퀴라소:',
        'pt': ':bandeira_curaçao:',
        'it': ':bandiera_curaçao:',
        'fa': ':پرچم_کوراسائو:',
        'id': ':bendera_curaçao:',
        'zh': ':库拉索:',
        'ru': ':флаг_Кюрасао:'
    },
    '\U0001F1E8\U0001F1FE': {  # 🇨🇾
        'en': ':Cyprus:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Cyprus:', ':cyprus:'],
        'de': ':flagge_zypern:',
        'es': ':bandera_chipre:',
        'fr': ':drapeau_chypre:',
        'ja': ':旗_キプロス:',
        'ko': ':깃발_키프로스:',
        'pt': ':bandeira_chipre:',
        'it': ':bandiera_cipro:',
        'fa': ':پرچم_قبرس:',
        'id': ':bendera_siprus:',
        'zh': ':塞浦路斯:',
        'ru': ':флаг_Кипр:'
    },
    '\U0001F1E8\U0001F1FF': {  # 🇨🇿
        'en': ':Czechia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Czech_Republic:', ':czech_republic:'],
        'de': ':flagge_tschechien:',
        'es': ':bandera_chequia:',
        'fr': ':drapeau_tchéquie:',
        'ja': ':旗_チェコ:',
        'ko': ':깃발_체코:',
        'pt': ':bandeira_tchéquia:',
        'it': ':bandiera_cechia:',
        'fa': ':پرچم_چک:',
        'id': ':bendera_ceko:',
        'zh': ':捷克:',
        'ru': ':флаг_Чехия:'
    },
    '\U0001F1E8\U0001F1EE': {  # 🇨🇮
        'en': ':Côte_d’Ivoire:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Côte_d’Ivoire:', ':cote_divoire:'],
        'de': ':flagge_côte_d’ivoire:',
        'es': ':bandera_côte_d’ivoire:',
        'fr': ':drapeau_côte_d’ivoire:',
        'ja': ':旗_コートジボワール:',
        'ko': ':깃발_코트디부아르:',
        'pt': ':bandeira_costa_do_marfim:',
        'it': ':bandiera_costa_d’avorio:',
        'fa': ':پرچم_ساحل_عاج:',
        'id': ':bendera_côte_d’ivoire:',
        'zh': ':科特迪瓦:',
        'ru': ':флаг_Кот_д’Ивуар:'
    },
    '\U0001F1E9\U0001F1F0': {  # 🇩🇰
        'en': ':Denmark:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Denmark:', ':denmark:'],
        'de': ':flagge_dänemark:',
        'es': ':bandera_dinamarca:',
        'fr': ':drapeau_danemark:',
        'ja': ':旗_デンマーク:',
        'ko': ':깃발_덴마크:',
        'pt': ':bandeira_dinamarca:',
        'it': ':bandiera_danimarca:',
        'fa': ':پرچم_دانمارک:',
        'id': ':bendera_denmark:',
        'zh': ':丹麦:',
        'ru': ':флаг_Дания:'
    },
    '\U0001F1E9\U0001F1EC': {  # 🇩🇬
        'en': ':Diego_Garcia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Diego_Garcia:', ':diego_garcia:'],
        'de': ':flagge_diego_garcia:',
        'es': ':bandera_diego_garcía:',
        'fr': ':drapeau_diego_garcia:',
        'ja': ':旗_ディエゴガルシア島:',
        'ko': ':깃발_디에고_가르시아:',
        'pt': ':bandeira_diego_garcia:',
        'it': ':bandiera_diego_garcia:',
        'fa': ':پرچم_دیه_گو_گارسیا:',
        'id': ':bendera_diego_garcia:',
        'zh': ':迪戈加西亚岛:',
        'ru': ':флаг_Диего_Гарсия:'
    },
    '\U0001F1E9\U0001F1EF': {  # 🇩🇯
        'en': ':Djibouti:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Djibouti:', ':djibouti:'],
        'de': ':flagge_dschibuti:',
        'es': ':bandera_yibuti:',
        'fr': ':drapeau_djibouti:',
        'ja': ':旗_ジブチ:',
        'ko': ':깃발_지부티:',
        'pt': ':bandeira_djibuti:',
        'it': ':bandiera_gibuti:',
        'fa': ':پرچم_جیبوتی:',
        'id': ':bendera_jibuti:',
        'zh': ':吉布提:',
        'ru': ':флаг_Джибути:'
    },
    '\U0001F1E9\U0001F1F2': {  # 🇩🇲
        'en': ':Dominica:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Dominica:', ':dominica:'],
        'de': ':flagge_dominica:',
        'es': ':bandera_dominica:',
        'fr': ':drapeau_dominique:',
        'ja': ':旗_ドミニカ国:',
        'ko': ':깃발_도미니카:',
        'pt': ':bandeira_dominica:',
        'it': ':bandiera_dominica:',
        'fa': ':پرچم_دومینیکا:',
        'id': ':bendera_dominika:',
        'zh': ':多米尼克:',
        'ru': ':флаг_Доминика:'
    },
    '\U0001F1E9\U0001F1F4': {  # 🇩🇴
        'en': ':Dominican_Republic:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Dominican_Republic:', ':dominican_republic:'],
        'de': ':flagge_dominikanische_republik:',
        'es': ':bandera_república_dominicana:',
        'fr': ':drapeau_république_dominicaine:',
        'ja': ':旗_ドミニカ共和国:',
        'ko': ':깃발_도미니카_공화국:',
        'pt': ':bandeira_república_dominicana:',
        'it': ':bandiera_repubblica_dominicana:',
        'fa': ':پرچم_جمهوری_دومینیکن:',
        'id': ':bendera_republik_dominika:',
        'zh': ':多米尼加共和国:',
        'ru': ':флаг_Доминиканская_Республика:'
    },
    '\U0001F51A': {  # 🔚
        'en': ':END_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':end:', ':end_arrow:'],
        'de': ':end-pfeil:',
        'es': ':flecha_end:',
        'fr': ':flèche_fin:',
        'ja': ':end矢印:',
        'ko': ':종료:',
        'pt': ':seta_end:',
        'it': ':freccia_end:',
        'fa': ':پیکان_پایان:',
        'id': ':tanda_panah_end:',
        'zh': ':结束箭头:',
        'ru': ':стрелка_«конец»:'
    },
    '\U0001F1EA\U0001F1E8': {  # 🇪🇨
        'en': ':Ecuador:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Ecuador:', ':ecuador:'],
        'de': ':flagge_ecuador:',
        'es': ':bandera_ecuador:',
        'fr': ':drapeau_équateur:',
        'ja': ':旗_エクアドル:',
        'ko': ':깃발_에콰도르:',
        'pt': ':bandeira_equador:',
        'it': ':bandiera_ecuador:',
        'fa': ':پرچم_اکوادور:',
        'id': ':bendera_ekuador:',
        'zh': ':厄瓜多尔:',
        'ru': ':флаг_Эквадор:'
    },
    '\U0001F1EA\U0001F1EC': {  # 🇪🇬
        'en': ':Egypt:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Egypt:', ':egypt:'],
        'de': ':flagge_ägypten:',
        'es': ':bandera_egipto:',
        'fr': ':drapeau_égypte:',
        'ja': ':旗_エジプト:',
        'ko': ':깃발_이집트:',
        'pt': ':bandeira_egito:',
        'it': ':bandiera_egitto:',
        'fa': ':پرچم_مصر:',
        'id': ':bendera_mesir:',
        'zh': ':埃及:',
        'ru': ':флаг_Египет:'
    },
    '\U0001F1F8\U0001F1FB': {  # 🇸🇻
        'en': ':El_Salvador:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_El_Salvador:', ':el_salvador:'],
        'de': ':flagge_el_salvador:',
        'es': ':bandera_el_salvador:',
        'fr': ':drapeau_salvador:',
        'ja': ':旗_エルサルバドル:',
        'ko': ':깃발_엘살바도르:',
        'pt': ':bandeira_el_salvador:',
        'it': ':bandiera_el_salvador:',
        'fa': ':پرچم_السالوادور:',
        'id': ':bendera_el_salvador:',
        'zh': ':萨尔瓦多:',
        'ru': ':флаг_Сальвадор:'
    },
    '\U0001F3F4\U000E0067\U000E0062\U000E0065\U000E006E\U000E0067\U000E007F': {  # 🏴󠁧󠁢󠁥󠁮󠁧󠁿
        'en': ':England:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':england:'],
        'de': ':flagge_england:',
        'es': ':bandera_inglaterra:',
        'fr': ':drapeau_angleterre:',
        'ja': ':旗_イングランド:',
        'ko': ':깃발_잉글랜드:',
        'pt': ':bandeira_inglaterra:',
        'it': ':bandiera_inghilterra:',
        'fa': ':پرچم_انگلستان:',
        'id': ':bendera_inggris:',
        'zh': ':英格兰:',
        'ru': ':флаг_Англия:'
    },
    '\U0001F1EC\U0001F1F6': {  # 🇬🇶
        'en': ':Equatorial_Guinea:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Equatorial_Guinea:', ':equatorial_guinea:'],
        'de': ':flagge_äquatorialguinea:',
        'es': ':bandera_guinea_ecuatorial:',
        'fr': ':drapeau_guinée_équatoriale:',
        'ja': ':旗_赤道ギニア:',
        'ko': ':깃발_적도_기니:',
        'pt': ':bandeira_guiné_equatorial:',
        'it': ':bandiera_guinea_equatoriale:',
        'fa': ':پرچم_گینهٔ_استوایی:',
        'id': ':bendera_guinea_ekuatorial:',
        'zh': ':赤道几内亚:',
        'ru': ':флаг_Экваториальная_Гвинея:'
    },
    '\U0001F1EA\U0001F1F7': {  # 🇪🇷
        'en': ':Eritrea:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Eritrea:', ':eritrea:'],
        'de': ':flagge_eritrea:',
        'es': ':bandera_eritrea:',
        'fr': ':drapeau_érythrée:',
        'ja': ':旗_エリトリア:',
        'ko': ':깃발_에리트리아:',
        'pt': ':bandeira_eritreia:',
        'it': ':bandiera_eritrea:',
        'fa': ':پرچم_اریتره:',
        'id': ':bendera_eritrea:',
        'zh': ':厄立特里亚:',
        'ru': ':флаг_Эритрея:'
    },
    '\U0001F1EA\U0001F1EA': {  # 🇪🇪
        'en': ':Estonia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Estonia:', ':estonia:'],
        'de': ':flagge_estland:',
        'es': ':bandera_estonia:',
        'fr': ':drapeau_estonie:',
        'ja': ':旗_エストニア:',
        'ko': ':깃발_에스토니아:',
        'pt': ':bandeira_estônia:',
        'it': ':bandiera_estonia:',
        'fa': ':پرچم_استونی:',
        'id': ':bendera_estonia:',
        'zh': ':爱沙尼亚:',
        'ru': ':флаг_Эстония:'
    },
    '\U0001F1F8\U0001F1FF': {  # 🇸🇿
        'en': ':Eswatini:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Swaziland:', ':swaziland:'],
        'de': ':flagge_eswatini:',
        'es': ':bandera_esuatini:',
        'fr': ':drapeau_eswatini:',
        'ja': ':旗_エスワティニ:',
        'ko': ':깃발_에스와티니:',
        'pt': ':bandeira_essuatíni:',
        'it': ':bandiera_eswatini:',
        'fa': ':پرچم_اسواتینی:',
        'id': ':bendera_eswatini:',
        'zh': ':斯威士兰:',
        'ru': ':флаг_Эсватини:'
    },
    '\U0001F1EA\U0001F1F9': {  # 🇪🇹
        'en': ':Ethiopia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Ethiopia:', ':ethiopia:'],
        'de': ':flagge_äthiopien:',
        'es': ':bandera_etiopía:',
        'fr': ':drapeau_éthiopie:',
        'ja': ':旗_エチオピア:',
        'ko': ':깃발_에티오피아:',
        'pt': ':bandeira_etiópia:',
        'it': ':bandiera_etiopia:',
        'fa': ':پرچم_اتیوپی:',
        'id': ':bendera_etiopia:',
        'zh': ':埃塞俄比亚:',
        'ru': ':флаг_Эфиопия:'
    },
    '\U0001F1EA\U0001F1FA': {  # 🇪🇺
        'en': ':European_Union:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_European_Union:', ':eu:', ':european_union:'],
        'de': ':flagge_europäische_union:',
        'es': ':bandera_unión_europea:',
        'fr': ':drapeau_union_européenne:',
        'ja': ':旗_欧州連合:',
        'ko': ':깃발_유럽_연합:',
        'pt': ':bandeira_união_europeia:',
        'it': ':bandiera_unione_europea:',
        'fa': ':پرچم_اتحادیهٔ_اروپا:',
        'id': ':bendera_uni_eropa:',
        'zh': ':欧盟:',
        'ru': ':флаг_Европейский_союз:'
    },
    '\U0001F193': {  # 🆓
        'en': ':FREE_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':free:', ':free_button:'],
        'de': ':wort_free_in_blauem_quadrat:',
        'es': ':botón_free:',
        'fr': ':bouton_gratuit:',
        'ja': ':freeマーク:',
        'ko': ':프리:',
        'pt': ':botão_free:',
        'it': ':pulsante_free:',
        'fa': ':دکمهٔ_free:',
        'id': ':tombol_free:',
        'zh': ':FREE按钮:',
        'ru': ':значок_«бесплатно»:'
    },
    '\U0001F1EB\U0001F1F0': {  # 🇫🇰
        'en': ':Falkland_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Falkland_Islands:', ':falkland_islands:'],
        'de': ':flagge_falklandinseln:',
        'es': ':bandera_islas_malvinas:',
        'fr': ':drapeau_îles_malouines:',
        'ja': ':旗_フォークランド諸島:',
        'ko': ':깃발_포클랜드_제도:',
        'pt': ':bandeira_ilhas_malvinas:',
        'it': ':bandiera_isole_falkland:',
        'fa': ':پرچم_جزایر_فالکلند:',
        'id': ':bendera_kepulauan_falkland:',
        'zh': ':福克兰群岛:',
        'ru': ':флаг_Фолклендские_о_ва:'
    },
    '\U0001F1EB\U0001F1F4': {  # 🇫🇴
        'en': ':Faroe_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Faroe_Islands:', ':faroe_islands:'],
        'de': ':flagge_färöer:',
        'es': ':bandera_islas_feroe:',
        'fr': ':drapeau_îles_féroé:',
        'ja': ':旗_フェロー諸島:',
        'ko': ':깃발_페로_제도:',
        'pt': ':bandeira_ilhas_faroé:',
        'it': ':bandiera_isole_fær_øer:',
        'fa': ':پرچم_جزایر_فارو:',
        'id': ':bendera_kepulauan_faroe:',
        'zh': ':法罗群岛:',
        'ru': ':флаг_Фарерские_о_ва:'
    },
    '\U0001F1EB\U0001F1EF': {  # 🇫🇯
        'en': ':Fiji:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Fiji:', ':fiji:'],
        'de': ':flagge_fidschi:',
        'es': ':bandera_fiyi:',
        'fr': ':drapeau_fidji:',
        'ja': ':旗_フィジー:',
        'ko': ':깃발_피지:',
        'pt': ':bandeira_fiji:',
        'it': ':bandiera_figi:',
        'fa': ':پرچم_فیجی:',
        'id': ':bendera_fiji:',
        'zh': ':斐济:',
        'ru': ':флаг_Фиджи:'
    },
    '\U0001F1EB\U0001F1EE': {  # 🇫🇮
        'en': ':Finland:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Finland:', ':finland:'],
        'de': ':flagge_finnland:',
        'es': ':bandera_finlandia:',
        'fr': ':drapeau_finlande:',
        'ja': ':旗_フィンランド:',
        'ko': ':깃발_핀란드:',
        'pt': ':bandeira_finlândia:',
        'it': ':bandiera_finlandia:',
        'fa': ':پرچم_فنلاند:',
        'id': ':bendera_finlandia:',
        'zh': ':芬兰:',
        'ru': ':флаг_Финляндия:'
    },
    '\U0001F1EB\U0001F1F7': {  # 🇫🇷
        'en': ':France:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_France:', ':fr:'],
        'de': ':flagge_frankreich:',
        'es': ':bandera_francia:',
        'fr': ':drapeau_france:',
        'ja': ':旗_フランス:',
        'ko': ':깃발_프랑스:',
        'pt': ':bandeira_frança:',
        'it': ':bandiera_francia:',
        'fa': ':پرچم_فرانسه:',
        'id': ':bendera_prancis:',
        'zh': ':法国:',
        'ru': ':флаг_Франция:'
    },
    '\U0001F1EC\U0001F1EB': {  # 🇬🇫
        'en': ':French_Guiana:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_French_Guiana:', ':french_guiana:'],
        'de': ':flagge_französisch-guayana:',
        'es': ':bandera_guayana_francesa:',
        'fr': ':drapeau_guyane_française:',
        'ja': ':旗_仏領ギアナ:',
        'ko': ':깃발_프랑스령_기아나:',
        'pt': ':bandeira_guiana_francesa:',
        'it': ':bandiera_guyana_francese:',
        'fa': ':پرچم_گویان_فرانسه:',
        'id': ':bendera_guyana_prancis:',
        'zh': ':法属圭亚那:',
        'ru': ':флаг_Французская_Гвиана:'
    },
    '\U0001F1F5\U0001F1EB': {  # 🇵🇫
        'en': ':French_Polynesia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_French_Polynesia:', ':french_polynesia:'],
        'de': ':flagge_französisch-polynesien:',
        'es': ':bandera_polinesia_francesa:',
        'fr': ':drapeau_polynésie_française:',
        'ja': ':旗_仏領ポリネシア:',
        'ko': ':깃발_프랑스령_폴리네시아:',
        'pt': ':bandeira_polinésia_francesa:',
        'it': ':bandiera_polinesia_francese:',
        'fa': ':پرچم_پلی_نزی_فرانسه:',
        'id': ':bendera_polinesia_prancis:',
        'zh': ':法属波利尼西亚:',
        'ru': ':флаг_Французская_Полинезия:'
    },
    '\U0001F1F9\U0001F1EB': {  # 🇹🇫
        'en': ':French_Southern_Territories:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_French_Southern_Territories:', ':french_southern_territories:'],
        'de': ':flagge_französische_süd-_und_antarktisgebiete:',
        'es': ':bandera_territorios_australes_franceses:',
        'fr': ':drapeau_terres_australes_françaises:',
        'ja': ':旗_仏領極南諸島:',
        'ko': ':깃발_프랑스령_남방_지역:',
        'pt': ':bandeira_territórios_franceses_do_sul:',
        'it': ':bandiera_terre_australi_francesi:',
        'fa': ':پرچم_سرزمین_های_جنوبی_فرانسه:',
        'id': ':bendera_wilayah_selatan_prancis:',
        'zh': ':法属南部领地:',
        'ru': ':флаг_Французские_Южные_территории:'
    },
    '\U0001F1EC\U0001F1E6': {  # 🇬🇦
        'en': ':Gabon:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Gabon:', ':gabon:'],
        'de': ':flagge_gabun:',
        'es': ':bandera_gabón:',
        'fr': ':drapeau_gabon:',
        'ja': ':旗_ガボン:',
        'ko': ':깃발_가봉:',
        'pt': ':bandeira_gabão:',
        'it': ':bandiera_gabon:',
        'fa': ':پرچم_گابن:',
        'id': ':bendera_gabon:',
        'zh': ':加蓬:',
        'ru': ':флаг_Габон:'
    },
    '\U0001F1EC\U0001F1F2': {  # 🇬🇲
        'en': ':Gambia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Gambia:', ':gambia:'],
        'de': ':flagge_gambia:',
        'es': ':bandera_gambia:',
        'fr': ':drapeau_gambie:',
        'ja': ':旗_ガンビア:',
        'ko': ':깃발_감비아:',
        'pt': ':bandeira_gâmbia:',
        'it': ':bandiera_gambia:',
        'fa': ':پرچم_گامبیا:',
        'id': ':bendera_gambia:',
        'zh': ':冈比亚:',
        'ru': ':флаг_Гамбия:'
    },
    '\U0000264A': {  # ♊
        'en': ':Gemini:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':gemini:'],
        'variant': True,
        'de': ':zwillinge_(sternzeichen):',
        'es': ':géminis:',
        'fr': ':gémeaux:',
        'ja': ':ふたご座:',
        'ko': ':쌍둥이_자리:',
        'pt': ':signo_de_gêmeos:',
        'it': ':segno_zodiacale_dei_gemelli:',
        'fa': ':برج_جوزا:',
        'id': ':gemini:',
        'zh': ':双子座:',
        'ru': ':знак_зодиака_Близнецы:'
    },
    '\U0001F1EC\U0001F1EA': {  # 🇬🇪
        'en': ':Georgia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Georgia:', ':georgia:'],
        'de': ':flagge_georgien:',
        'es': ':bandera_georgia:',
        'fr': ':drapeau_géorgie:',
        'ja': ':旗_ジョージア:',
        'ko': ':깃발_조지아:',
        'pt': ':bandeira_geórgia:',
        'it': ':bandiera_georgia:',
        'fa': ':پرچم_گرجستان:',
        'id': ':bendera_georgia:',
        'zh': ':格鲁吉亚:',
        'ru': ':флаг_Грузия:'
    },
    '\U0001F1E9\U0001F1EA': {  # 🇩🇪
        'en': ':Germany:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_Germany:', ':de:'],
        'de': ':flagge_deutschland:',
        'es': ':bandera_alemania:',
        'fr': ':drapeau_allemagne:',
        'ja': ':旗_ドイツ:',
        'ko': ':깃발_독일:',
        'pt': ':bandeira_alemanha:',
        'it': ':bandiera_germania:',
        'fa': ':پرچم_آلمان:',
        'id': ':bendera_jerman:',
        'zh': ':德国:',
        'ru': ':флаг_Германия:'
    },
    '\U0001F1EC\U0001F1ED': {  # 🇬🇭
        'en': ':Ghana:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Ghana:', ':ghana:'],
        'de': ':flagge_ghana:',
        'es': ':bandera_ghana:',
        'fr': ':drapeau_ghana:',
        'ja': ':旗_ガーナ:',
        'ko': ':깃발_가나:',
        'pt': ':bandeira_gana:',
        'it': ':bandiera_ghana:',
        'fa': ':پرچم_غنا:',
        'id': ':bendera_ghana:',
        'zh': ':加纳:',
        'ru': ':флаг_Гана:'
    },
    '\U0001F1EC\U0001F1EE': {  # 🇬🇮
        'en': ':Gibraltar:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Gibraltar:', ':gibraltar:'],
        'de': ':flagge_gibraltar:',
        'es': ':bandera_gibraltar:',
        'fr': ':drapeau_gibraltar:',
        'ja': ':旗_ジブラルタル:',
        'ko': ':깃발_지브롤터:',
        'pt': ':bandeira_gibraltar:',
        'it': ':bandiera_gibilterra:',
        'fa': ':پرچم_جبل_الطارق:',
        'id': ':bendera_gibraltar:',
        'zh': ':直布罗陀:',
        'ru': ':флаг_Гибралтар:'
    },
    '\U0001F1EC\U0001F1F7': {  # 🇬🇷
        'en': ':Greece:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Greece:', ':greece:'],
        'de': ':flagge_griechenland:',
        'es': ':bandera_grecia:',
        'fr': ':drapeau_grèce:',
        'ja': ':旗_ギリシャ:',
        'ko': ':깃발_그리스:',
        'pt': ':bandeira_grécia:',
        'it': ':bandiera_grecia:',
        'fa': ':پرچم_یونان:',
        'id': ':bendera_yunani:',
        'zh': ':希腊:',
        'ru': ':флаг_Греция:'
    },
    '\U0001F1EC\U0001F1F1': {  # 🇬🇱
        'en': ':Greenland:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Greenland:', ':greenland:'],
        'de': ':flagge_grönland:',
        'es': ':bandera_groenlandia:',
        'fr': ':drapeau_groenland:',
        'ja': ':旗_グリーンランド:',
        'ko': ':깃발_그린란드:',
        'pt': ':bandeira_groenlândia:',
        'it': ':bandiera_groenlandia:',
        'fa': ':پرچم_گرینلند:',
        'id': ':bendera_greenland:',
        'zh': ':格陵兰:',
        'ru': ':флаг_Гренландия:'
    },
    '\U0001F1EC\U0001F1E9': {  # 🇬🇩
        'en': ':Grenada:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Grenada:', ':grenada:'],
        'de': ':flagge_grenada:',
        'es': ':bandera_granada:',
        'fr': ':drapeau_grenade:',
        'ja': ':旗_グレナダ:',
        'ko': ':깃발_그레나다:',
        'pt': ':bandeira_granada:',
        'it': ':bandiera_grenada:',
        'fa': ':پرچم_گرنادا:',
        'id': ':bendera_grenada:',
        'zh': ':格林纳达:',
        'ru': ':флаг_Гренада:'
    },
    '\U0001F1EC\U0001F1F5': {  # 🇬🇵
        'en': ':Guadeloupe:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Guadeloupe:', ':guadeloupe:'],
        'de': ':flagge_guadeloupe:',
        'es': ':bandera_guadalupe:',
        'fr': ':drapeau_guadeloupe:',
        'ja': ':旗_グアドループ:',
        'ko': ':깃발_과들루프:',
        'pt': ':bandeira_guadalupe:',
        'it': ':bandiera_guadalupa:',
        'fa': ':پرچم_گوادلوپ:',
        'id': ':bendera_guadeloupe:',
        'zh': ':瓜德罗普:',
        'ru': ':флаг_Гваделупа:'
    },
    '\U0001F1EC\U0001F1FA': {  # 🇬🇺
        'en': ':Guam:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Guam:', ':guam:'],
        'de': ':flagge_guam:',
        'es': ':bandera_guam:',
        'fr': ':drapeau_guam:',
        'ja': ':旗_グアム:',
        'ko': ':깃발_괌:',
        'pt': ':bandeira_guam:',
        'it': ':bandiera_guam:',
        'fa': ':پرچم_گوام:',
        'id': ':bendera_guam:',
        'zh': ':关岛:',
        'ru': ':флаг_Гуам:'
    },
    '\U0001F1EC\U0001F1F9': {  # 🇬🇹
        'en': ':Guatemala:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Guatemala:', ':guatemala:'],
        'de': ':flagge_guatemala:',
        'es': ':bandera_guatemala:',
        'fr': ':drapeau_guatemala:',
        'ja': ':旗_グアテマラ:',
        'ko': ':깃발_과테말라:',
        'pt': ':bandeira_guatemala:',
        'it': ':bandiera_guatemala:',
        'fa': ':پرچم_گواتمالا:',
        'id': ':bendera_guatemala:',
        'zh': ':危地马拉:',
        'ru': ':флаг_Гватемала:'
    },
    '\U0001F1EC\U0001F1EC': {  # 🇬🇬
        'en': ':Guernsey:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Guernsey:', ':guernsey:'],
        'de': ':flagge_guernsey:',
        'es': ':bandera_guernesey:',
        'fr': ':drapeau_guernesey:',
        'ja': ':旗_ガーンジー:',
        'ko': ':깃발_건지:',
        'pt': ':bandeira_guernsey:',
        'it': ':bandiera_guernsey:',
        'fa': ':پرچم_گرنزی:',
        'id': ':bendera_guernsey:',
        'zh': ':根西岛:',
        'ru': ':флаг_Гернси:'
    },
    '\U0001F1EC\U0001F1F3': {  # 🇬🇳
        'en': ':Guinea:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Guinea:', ':guinea:'],
        'de': ':flagge_guinea:',
        'es': ':bandera_guinea:',
        'fr': ':drapeau_guinée:',
        'ja': ':旗_ギニア:',
        'ko': ':깃발_기니:',
        'pt': ':bandeira_guiné:',
        'it': ':bandiera_guinea:',
        'fa': ':پرچم_گینه:',
        'id': ':bendera_guinea:',
        'zh': ':几内亚:',
        'ru': ':флаг_Гвинея:'
    },
    '\U0001F1EC\U0001F1FC': {  # 🇬🇼
        'en': ':Guinea-Bissau:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Guinea_Bissau:', ':guinea_bissau:'],
        'de': ':flagge_guinea-bissau:',
        'es': ':bandera_guinea-bisáu:',
        'fr': ':drapeau_guinée-bissau:',
        'ja': ':旗_ギニアビサウ:',
        'ko': ':깃발_기니비사우:',
        'pt': ':bandeira_guiné-bissau:',
        'it': ':bandiera_guinea-bissau:',
        'fa': ':پرچم_گینهٔ_بیسائو:',
        'id': ':bendera_guinea-bissau:',
        'zh': ':几内亚比绍:',
        'ru': ':флаг_Гвинея_Бисау:'
    },
    '\U0001F1EC\U0001F1FE': {  # 🇬🇾
        'en': ':Guyana:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Guyana:', ':guyana:'],
        'de': ':flagge_guyana:',
        'es': ':bandera_guyana:',
        'fr': ':drapeau_guyana:',
        'ja': ':旗_ガイアナ:',
        'ko': ':깃발_가이아나:',
        'pt': ':bandeira_guiana:',
        'it': ':bandiera_guyana:',
        'fa': ':پرچم_گویان:',
        'id': ':bendera_guyana:',
        'zh': ':圭亚那:',
        'ru': ':флаг_Гайана:'
    },
    '\U0001F1ED\U0001F1F9': {  # 🇭🇹
        'en': ':Haiti:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Haiti:', ':haiti:'],
        'de': ':flagge_haiti:',
        'es': ':bandera_haití:',
        'fr': ':drapeau_haïti:',
        'ja': ':旗_ハイチ:',
        'ko': ':깃발_아이티:',
        'pt': ':bandeira_haiti:',
        'it': ':bandiera_haiti:',
        'fa': ':پرچم_هائیتی:',
        'id': ':bendera_haiti:',
        'zh': ':海地:',
        'ru': ':флаг_Гаити:'
    },
    '\U0001F1ED\U0001F1F2': {  # 🇭🇲
        'en': ':Heard_&_McDonald_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Heard_&_McDonald_Islands:', ':heard_mcdonald_islands:'],
        'de': ':flagge_heard_und_mcdonaldinseln:',
        'es': ':bandera_islas_heard_y_mcdonald:',
        'fr': ':drapeau_îles_heard-et-macdonald:',
        'ja': ':旗_ハード島・マクドナルド諸島:',
        'ko': ':깃발_허드_맥도널드_제도:',
        'pt': ':bandeira_ilhas_heard_e_mcdonald:',
        'it': ':bandiera_isole_heard_e_mcdonald:',
        'fa': ':پرچم_هرد_و_جزایر_مک_دونالد:',
        'id': ':bendera_pulau_heard_dan_kepulauan_mcdonald:',
        'zh': ':赫德岛和麦克唐纳群岛:',
        'ru': ':флаг_о_ва_Херд_и_Макдональд:'
    },
    '\U0001F1ED\U0001F1F3': {  # 🇭🇳
        'en': ':Honduras:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Honduras:', ':honduras:'],
        'de': ':flagge_honduras:',
        'es': ':bandera_honduras:',
        'fr': ':drapeau_honduras:',
        'ja': ':旗_ホンジュラス:',
        'ko': ':깃발_온두라스:',
        'pt': ':bandeira_honduras:',
        'it': ':bandiera_honduras:',
        'fa': ':پرچم_هندوراس:',
        'id': ':bendera_honduras:',
        'zh': ':洪都拉斯:',
        'ru': ':флаг_Гондурас:'
    },
    '\U0001F1ED\U0001F1F0': {  # 🇭🇰
        'en': ':Hong_Kong_SAR_China:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Hong_Kong:', ':hong_kong:'],
        'de': ':flagge_sonderverwaltungsregion_hongkong:',
        'es': ':bandera_rae_de_hong_kong_(china):',
        'fr': ':drapeau_r.a.s._chinoise_de_hong_kong:',
        'ja': ':旗_中華人民共和国香港特別行政区:',
        'ko': ':깃발_홍콩(중국_특별행정구):',
        'pt': ':bandeira_hong_kong_rae_da_china:',
        'it': ':bandiera_ras_di_hong_kong:',
        'fa': ':پرچم_هنگ_کنگ_منطقهٔ_ویژهٔ_اداری_چین:',
        'id': ':bendera_hong_kong_dak_tiongkok:',
        'zh': ':香港:',
        'ru': ':флаг_Гонконг_(САР):'
    },
    '\U0001F1ED\U0001F1FA': {  # 🇭🇺
        'en': ':Hungary:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Hungary:', ':hungary:'],
        'de': ':flagge_ungarn:',
        'es': ':bandera_hungría:',
        'fr': ':drapeau_hongrie:',
        'ja': ':旗_ハンガリー:',
        'ko': ':깃발_헝가리:',
        'pt': ':bandeira_hungria:',
        'it': ':bandiera_ungheria:',
        'fa': ':پرچم_مجارستان:',
        'id': ':bendera_hungaria:',
        'zh': ':匈牙利:',
        'ru': ':флаг_Венгрия:'
    },
    '\U0001F194': {  # 🆔
        'en': ':ID_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':id:', ':id_button:'],
        'de': ':großbuchstaben_id_in_lila_quadrat:',
        'es': ':símbolo_de_identificación:',
        'fr': ':bouton_identifiant:',
        'ja': ':idマーク:',
        'ko': ':아이디:',
        'pt': ':botão_id:',
        'it': ':pulsante_id:',
        'fa': ':دکمهٔ_شناسایی:',
        'id': ':tombol_id:',
        'zh': ':ID按钮:',
        'ru': ':значок_«идентификация»:'
    },
    '\U0001F1EE\U0001F1F8': {  # 🇮🇸
        'en': ':Iceland:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Iceland:', ':iceland:'],
        'de': ':flagge_island:',
        'es': ':bandera_islandia:',
        'fr': ':drapeau_islande:',
        'ja': ':旗_アイスランド:',
        'ko': ':깃발_아이슬란드:',
        'pt': ':bandeira_islândia:',
        'it': ':bandiera_islanda:',
        'fa': ':پرچم_ایسلند:',
        'id': ':bendera_islandia:',
        'zh': ':冰岛:',
        'ru': ':флаг_Исландия:'
    },
    '\U0001F1EE\U0001F1F3': {  # 🇮🇳
        'en': ':India:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_India:', ':india:'],
        'de': ':flagge_indien:',
        'es': ':bandera_india:',
        'fr': ':drapeau_inde:',
        'ja': ':旗_インド:',
        'ko': ':깃발_인도:',
        'pt': ':bandeira_índia:',
        'it': ':bandiera_india:',
        'fa': ':پرچم_هند:',
        'id': ':bendera_india:',
        'zh': ':印度:',
        'ru': ':флаг_Индия:'
    },
    '\U0001F1EE\U0001F1E9': {  # 🇮🇩
        'en': ':Indonesia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Indonesia:', ':indonesia:'],
        'de': ':flagge_indonesien:',
        'es': ':bandera_indonesia:',
        'fr': ':drapeau_indonésie:',
        'ja': ':旗_インドネシア:',
        'ko': ':깃발_인도네시아:',
        'pt': ':bandeira_indonésia:',
        'it': ':bandiera_indonesia:',
        'fa': ':پرچم_اندونزی:',
        'id': ':bendera_indonesia:',
        'zh': ':印度尼西亞:',
        'ru': ':флаг_Индонезия:'
    },
    '\U0001F1EE\U0001F1F7': {  # 🇮🇷
        'en': ':Iran:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Iran:', ':iran:'],
        'de': ':flagge_iran:',
        'es': ':bandera_irán:',
        'fr': ':drapeau_iran:',
        'ja': ':旗_イラン:',
        'ko': ':깃발_이란:',
        'pt': ':bandeira_irã:',
        'it': ':bandiera_iran:',
        'fa': ':پرچم_ایران:',
        'id': ':bendera_iran:',
        'zh': ':伊朗:',
        'ru': ':флаг_Иран:'
    },
    '\U0001F1EE\U0001F1F6': {  # 🇮🇶
        'en': ':Iraq:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Iraq:', ':iraq:'],
        'de': ':flagge_irak:',
        'es': ':bandera_irak:',
        'fr': ':drapeau_irak:',
        'ja': ':旗_イラク:',
        'ko': ':깃발_이라크:',
        'pt': ':bandeira_iraque:',
        'it': ':bandiera_iraq:',
        'fa': ':پرچم_عراق:',
        'id': ':bendera_irak:',
        'zh': ':伊拉克:',
        'ru': ':флаг_Ирак:'
    },
    '\U0001F1EE\U0001F1EA': {  # 🇮🇪
        'en': ':Ireland:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Ireland:', ':ireland:'],
        'de': ':flagge_irland:',
        'es': ':bandera_irlanda:',
        'fr': ':drapeau_irlande:',
        'ja': ':旗_アイルランド:',
        'ko': ':깃발_아일랜드:',
        'pt': ':bandeira_irlanda:',
        'it': ':bandiera_irlanda:',
        'fa': ':پرچم_ایرلند:',
        'id': ':bendera_irlandia:',
        'zh': ':爱尔兰:',
        'ru': ':флаг_Ирландия:'
    },
    '\U0001F1EE\U0001F1F2': {  # 🇮🇲
        'en': ':Isle_of_Man:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Isle_of_Man:', ':isle_of_man:'],
        'de': ':flagge_isle_of_man:',
        'es': ':bandera_isla_de_man:',
        'fr': ':drapeau_île_de_man:',
        'ja': ':旗_マン島:',
        'ko': ':깃발_맨섬:',
        'pt': ':bandeira_ilha_de_man:',
        'it': ':bandiera_isola_di_man:',
        'fa': ':پرچم_جزیرهٔ_من:',
        'id': ':bendera_pulau_man:',
        'zh': ':马恩岛:',
        'ru': ':флаг_о_в_Мэн:'
    },
    '\U0001F1EE\U0001F1F1': {  # 🇮🇱
        'en': ':Israel:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Israel:', ':israel:'],
        'de': ':flagge_israel:',
        'es': ':bandera_israel:',
        'fr': ':drapeau_israël:',
        'ja': ':旗_イスラエル:',
        'ko': ':깃발_이스라엘:',
        'pt': ':bandeira_israel:',
        'it': ':bandiera_israele:',
        'fa': ':پرچم_اسرائیل:',
        'id': ':bendera_israel:',
        'zh': ':以色列:',
        'ru': ':флаг_Израиль:'
    },
    '\U0001F1EE\U0001F1F9': {  # 🇮🇹
        'en': ':Italy:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_Italy:', ':it:'],
        'de': ':flagge_italien:',
        'es': ':bandera_italia:',
        'fr': ':drapeau_italie:',
        'ja': ':旗_イタリア:',
        'ko': ':깃발_이탈리아:',
        'pt': ':bandeira_itália:',
        'it': ':bandiera_italia:',
        'fa': ':پرچم_ایتالیا:',
        'id': ':bendera_italia:',
        'zh': ':意大利:',
        'ru': ':флаг_Италия:'
    },
    '\U0001F1EF\U0001F1F2': {  # 🇯🇲
        'en': ':Jamaica:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Jamaica:', ':jamaica:'],
        'de': ':flagge_jamaika:',
        'es': ':bandera_jamaica:',
        'fr': ':drapeau_jamaïque:',
        'ja': ':旗_ジャマイカ:',
        'ko': ':깃발_자메이카:',
        'pt': ':bandeira_jamaica:',
        'it': ':bandiera_giamaica:',
        'fa': ':پرچم_جامائیکا:',
        'id': ':bendera_jamaika:',
        'zh': ':牙买加:',
        'ru': ':флаг_Ямайка:'
    },
    '\U0001F1EF\U0001F1F5': {  # 🇯🇵
        'en': ':Japan:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_Japan:', ':jp:'],
        'de': ':flagge_japan:',
        'es': ':bandera_japón:',
        'fr': ':drapeau_japon:',
        'ja': ':旗_日本:',
        'ko': ':깃발_일본:',
        'pt': ':bandeira_japão:',
        'it': ':bandiera_giappone:',
        'fa': ':پرچم_ژاپن:',
        'id': ':bendera_jepang:',
        'zh': ':日本:',
        'ru': ':флаг_Япония:'
    },
    '\U0001F251': {  # 🉑
        'en': ':Japanese_acceptable_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':accept:', ':japanese_acceptable_button:'],
        'de': ':schriftzeichen_für_akzeptieren:',
        'es': ':ideograma_japonés_para_aceptable:',
        'fr': ':bouton_accepter_en_japonais:',
        'ja': ':可マーク:',
        'ko': ':옳을_가:',
        'pt': ':botão_japonês_de_“aceitável”:',
        'it': ':ideogramma_giapponese_di_“accettabile”:',
        'fa': ':دکمۀ_«قابل_قبول»_به_ژاپنی:',
        'id': ':tombol_jepang_dapat_diterima:',
        'zh': ':可:',
        'ru': ':допустимость:'
    },
    '\U0001F238': {  # 🈸
        'en': ':Japanese_application_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u7533:', ':japanese_application_button:'],
        'de': ':schriftzeichen_für_anwenden:',
        'es': ':ideograma_japonés_para_aplicación:',
        'fr': ':bouton_application_en_japonais:',
        'ja': ':申マーク:',
        'ko': ':거듭_신:',
        'pt': ':botão_japonês_de_“aplicação”:',
        'it': ':ideogramma_giapponese_di_“candidatura”:',
        'fa': ':دکمۀ_«تقاضانامه»_به_ژاپنی:',
        'id': ':tombol_jepang_lamaran_kerja:',
        'zh': ':申:',
        'ru': ':заявка:'
    },
    '\U0001F250': {  # 🉐
        'en': ':Japanese_bargain_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':ideograph_advantage:', ':japanese_bargain_button:'],
        'de': ':schriftzeichen_für_schnäppchen:',
        'es': ':ideograma_japonés_para_ganga:',
        'fr': ':bouton_bonne_affaire_en_japonais:',
        'ja': ':マル得:',
        'ko': ':얻을_득:',
        'pt': ':botão_japonês_de_“barganha”:',
        'it': ':ideogramma_giapponese_di_“occasione”:',
        'fa': ':دکمهٔ_«قیمت_مناسب»_به_ژاپنی:',
        'id': ':karakter_keunggulan_dalam_lingkaran:',
        'zh': ':得:',
        'ru': ':удачная_сделка:'
    },
    '\U0001F3EF': {  # 🏯
        'en': ':Japanese_castle:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':japanese_castle:'],
        'de': ':japanisches_schloss:',
        'es': ':castillo_japonés:',
        'fr': ':château_japonais:',
        'ja': ':城:',
        'ko': ':일본_성:',
        'pt': ':castelo_japonês:',
        'it': ':castello_giapponese:',
        'fa': ':قلعهٔ_ژاپنی:',
        'id': ':kastel_jepang:',
        'zh': ':日本城堡:',
        'ru': ':японский_замок:'
    },
    '\U00003297\U0000FE0F': {  # ㊗️
        'en': ':Japanese_congratulations_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':congratulations:', ':japanese_congratulations_button:'],
        'variant': True,
        'de': ':schriftzeichen_für_gratulation:',
        'es': ':ideograma_japonés_para_enhorabuena:',
        'fr': ':bouton_félicitations_en_japonais:',
        'ja': ':祝マーク:',
        'ko': ':원_안의_축하_표의문자:',
        'pt': ':botão_japonês_de_“parabéns”:',
        'it': ':ideogramma_giapponese_di_“congratulazioni”:',
        'fa': ':دکمه_تبریک_به_ژاپنی:',
        'id': ':tombol_jepang_selamat:',
        'zh': ':祝:',
        'ru': ':поздравление:'
    },
    '\U00003297': {  # ㊗
        'en': ':Japanese_congratulations_button:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':congratulations:', ':japanese_congratulations_button:'],
        'variant': True,
        'de': ':schriftzeichen_für_gratulation:',
        'es': ':ideograma_japonés_para_enhorabuena:',
        'fr': ':bouton_félicitations_en_japonais:',
        'ja': ':祝マーク:',
        'ko': ':원_안의_축하_표의문자:',
        'pt': ':botão_japonês_de_“parabéns”:',
        'it': ':ideogramma_giapponese_di_“congratulazioni”:',
        'fa': ':دکمهٔ_«تبریک»_به_ژاپنی:',
        'id': ':tombol_jepang_selamat:',
        'zh': ':祝:',
        'ru': ':поздравление:'
    },
    '\U0001F239': {  # 🈹
        'en': ':Japanese_discount_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u5272:', ':japanese_discount_button:'],
        'de': ':schriftzeichen_für_rabatt:',
        'es': ':ideograma_japonés_para_descuento:',
        'fr': ':bouton_réduction_en_japonais:',
        'ja': ':割マーク:',
        'ko': ':나눌_할:',
        'pt': ':botão_japonês_de_“desconto”:',
        'it': ':ideogramma_giapponese_di_“sconto”:',
        'fa': ':دکمۀ_«تخفیف»_به_ژاپنی:',
        'id': ':tombol_jepang_diskon:',
        'zh': ':割:',
        'ru': ':скидка:'
    },
    '\U0001F38E': {  # 🎎
        'en': ':Japanese_dolls:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':dolls:', ':japanese_dolls:'],
        'de': ':japanische_puppen:',
        'es': ':muñecas_japonesas:',
        'fr': ':poupées_japonaises:',
        'ja': ':ひな祭り:',
        'ko': ':일본_인형:',
        'pt': ':bonecas_japonesas:',
        'it': ':bambole_giapponesi:',
        'fa': ':عروسک_های_ژاپنی:',
        'id': ':boneka_jepang:',
        'zh': ':日本人形:',
        'ru': ':японские_куклы:'
    },
    '\U0001F21A': {  # 🈚
        'en': ':Japanese_free_of_charge_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u7121:', ':japanese_free_of_charge_button:'],
        'variant': True,
        'de': ':schriftzeichen_für_gratis:',
        'es': ':ideograma_japonés_para_gratis:',
        'fr': ':bouton_gratuit_en_japonais:',
        'ja': ':無マーク:',
        'ko': ':없을_무:',
        'pt': ':botão_japonês_de_“gratuito”:',
        'it': ':ideogramma_giapponese_di_“gratis”:',
        'fa': ':دکمۀ_«رایگان»_به_ژاپنی:',
        'id': ':tombol_jepang_gratis:',
        'zh': ':无:',
        'ru': ':бесплатно:'
    },
    '\U0001F201': {  # 🈁
        'en': ':Japanese_here_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':koko:', ':japanese_here_button:'],
        'de': ':schriftzeichen_koko:',
        'es': ':ideograma_japonés_para_aquí:',
        'fr': ':bouton_ici_en_japonais:',
        'ja': ':ココのマーク:',
        'ko': ':일본어_여기_버튼:',
        'pt': ':botão_japonês_de_“aqui”:',
        'it': ':ideogramma_giapponese_per_“qui”:',
        'fa': ':دکمهٔ_«اینجا»_به_ژاپنی:',
        'id': ':tombol_jepang_di_sini:',
        'zh': ':这里:',
        'ru': ':значок_«здесь»:'
    },
    '\U0001F237\U0000FE0F': {  # 🈷️
        'en': ':Japanese_monthly_amount_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u6708:', ':japanese_monthly_amount_button:'],
        'variant': True,
        'de': ':schriftzeichen_für_monatsbetrag:',
        'es': ':ideograma_japonés_para_cantidad_mensual:',
        'fr': ':bouton_montant_mensuel_en_japonais:',
        'ja': ':月マーク:',
        'ko': ':달_월:',
        'pt': ':botão_japonês_de_“quantidade_mensal”:',
        'it': ':ideogramma_giapponese_di_“importo_mensile”:',
        'fa': ':دکمه_مقدار_ماهانه_به_ژاپنی:',
        'id': ':tombol_jepang_jumlah_bulanan:',
        'zh': ':月:',
        'ru': ':месяц:'
    },
    '\U0001F237': {  # 🈷
        'en': ':Japanese_monthly_amount_button:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':u6708:', ':japanese_monthly_amount_button:'],
        'variant': True,
        'de': ':schriftzeichen_für_monatsbetrag:',
        'es': ':ideograma_japonés_para_cantidad_mensual:',
        'fr': ':bouton_montant_mensuel_en_japonais:',
        'ja': ':月マーク:',
        'ko': ':달_월:',
        'pt': ':botão_japonês_de_“quantidade_mensal”:',
        'it': ':ideogramma_giapponese_di_“importo_mensile”:',
        'fa': ':دکمهٔ_«مقدار_ماهانه»_به_ژاپنی:',
        'id': ':tombol_jepang_jumlah_bulanan:',
        'zh': ':月:',
        'ru': ':месяц:'
    },
    '\U0001F235': {  # 🈵
        'en': ':Japanese_no_vacancy_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u6e80:', ':japanese_no_vacancy_button:'],
        'de': ':schriftzeichen_für_kein_zimmer_frei:',
        'es': ':ideograma_japonés_para_completo:',
        'fr': ':bouton_complet_en_japonais:',
        'ja': ':満マーク:',
        'ko': ':찰_만:',
        'pt': ':botão_japonês_de_“sem_vagas”:',
        'it': ':ideogramma_giapponese_di_“nessun_posto_libero”:',
        'fa': ':دکمهٔ_«پُر»_به_ژاپنی:',
        'id': ':tombol_jepang_tidak_ada_lowongan:',
        'zh': ':满:',
        'ru': ':нет_мест:'
    },
    '\U0001F236': {  # 🈶
        'en': ':Japanese_not_free_of_charge_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u6709:', ':japanese_not_free_of_charge_button:'],
        'de': ':schriftzeichen_für_nicht_gratis:',
        'es': ':ideograma_japonés_para_de_pago:',
        'fr': ':bouton_pas_gratuit_en_japonais:',
        'ja': ':有マーク:',
        'ko': ':있을_유:',
        'pt': ':botão_japonês_de_“não_gratuito”:',
        'it': ':ideogramma_giapponese_di_“a_pagamento”:',
        'fa': ':دکمهٔ_غیرمجانی_به_ژاپنی:',
        'id': ':tombol_jepang_tidak_gratis:',
        'zh': ':有:',
        'ru': ':платно:'
    },
    '\U0001F23A': {  # 🈺
        'en': ':Japanese_open_for_business_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u55b6:', ':japanese_open_for_business_button:'],
        'de': ':schriftzeichen_für_geöffnet:',
        'es': ':ideograma_japonés_para_abierto:',
        'fr': ':bouton_ouvert_pour_affaires_en_japonais:',
        'ja': ':営マーク:',
        'ko': ':벼슬_관:',
        'pt': ':botão_japonês_de_“aberto_para_negócios”:',
        'it': ':ideogramma_giapponese_di_“aperto_al_pubblico”:',
        'fa': ':دکمۀ_«آماده_برای_کسب_وکار»_به_ژاپنی:',
        'id': ':tombol_jepang_siap_berbisnis:',
        'zh': ':营:',
        'ru': ':открыто:'
    },
    '\U0001F234': {  # 🈴
        'en': ':Japanese_passing_grade_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u5408:', ':japanese_passing_grade_button:'],
        'de': ':schriftzeichen_für_note_zum_bestehen:',
        'es': ':ideograma_japonés_para_aprobado:',
        'fr': ':bouton_note_pour_réussir_en_japonais:',
        'ja': ':合マーク:',
        'ko': ':합할_합:',
        'pt': ':botão_japonês_de_“nota_mínima”:',
        'it': ':ideogramma_giapponese_di_“voto_di_sufficienza”:',
        'fa': ':دکمۀ_«نمرهٔ_قبولی»_به_ژاپنی:',
        'id': ':tombol_jepang_nilai_kelulusan:',
        'zh': ':合:',
        'ru': ':проходной_балл:'
    },
    '\U0001F3E3': {  # 🏣
        'en': ':Japanese_post_office:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':post_office:', ':japanese_post_office:'],
        'de': ':japanisches_postgebäude:',
        'es': ':oficina_de_correos_japonesa:',
        'fr': ':bureau_de_poste_japonais:',
        'ja': ':郵便局:',
        'ko': ':일본_우체국:',
        'pt': ':correio_japonês:',
        'it': ':ufficio_postale_giapponese:',
        'fa': ':پست_خانهٔ_ژاپنی:',
        'id': ':kantor_pos_jepang:',
        'zh': ':日本邮局:',
        'ru': ':японская_почта:'
    },
    '\U0001F232': {  # 🈲
        'en': ':Japanese_prohibited_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u7981:', ':japanese_prohibited_button:'],
        'de': ':schriftzeichen_für_verbieten:',
        'es': ':ideograma_japonés_para_prohibido:',
        'fr': ':bouton_interdit_en_japonais:',
        'ja': ':禁マーク:',
        'ko': ':금할_금:',
        'pt': ':botão_japonês_de_“proibido”:',
        'it': ':ideogramma_giapponese_di_“proibito”:',
        'fa': ':دکمۀ_«ممنوع»_به_ژاپنی:',
        'id': ':tombol_jepang_dilarang:',
        'zh': ':禁:',
        'ru': ':запрет:'
    },
    '\U0001F22F': {  # 🈯
        'en': ':Japanese_reserved_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u6307:', ':japanese_reserved_button:'],
        'variant': True,
        'de': ':schriftzeichen_für_reserviert:',
        'es': ':ideograma_japonés_para_reservado:',
        'fr': ':bouton_réservé_en_japonais:',
        'ja': ':指マーク:',
        'ko': ':가리킬_지:',
        'pt': ':botão_japonês_de_“reservado”:',
        'it': ':ideogramma_giapponese_di_“riservato”:',
        'fa': ':دکمۀ_«رزروشده»_به_ژاپنی:',
        'id': ':tombol_jepang_sudah_dipesan:',
        'zh': ':指:',
        'ru': ':забронировано:'
    },
    '\U00003299\U0000FE0F': {  # ㊙️
        'en': ':Japanese_secret_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':secret:', ':japanese_secret_button:'],
        'variant': True,
        'de': ':schriftzeichen_für_geheimnis:',
        'es': ':ideograma_japonés_para_secreto:',
        'fr': ':bouton_secret_en_japonais:',
        'ja': ':マル秘:',
        'ko': ':원_표시된_비밀_표의문자:',
        'pt': ':botão_japonês_de_“segredo”:',
        'it': ':ideogramma_giapponese_di_“segreto”:',
        'fa': ':دکمه_راز_به_ژاپنی:',
        'id': ':tombol_jepang_rahasia:',
        'zh': ':秘:',
        'ru': ':секрет:'
    },
    '\U00003299': {  # ㊙
        'en': ':Japanese_secret_button:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':secret:', ':japanese_secret_button:'],
        'variant': True,
        'de': ':schriftzeichen_für_geheimnis:',
        'es': ':ideograma_japonés_para_secreto:',
        'fr': ':bouton_secret_en_japonais:',
        'ja': ':マル秘:',
        'ko': ':원_표시된_비밀_표의문자:',
        'pt': ':botão_japonês_de_“segredo”:',
        'it': ':ideogramma_giapponese_di_“segreto”:',
        'fa': ':دکمهٔ_«راز»_به_ژاپنی:',
        'id': ':tombol_jepang_rahasia:',
        'zh': ':秘:',
        'ru': ':секрет:'
    },
    '\U0001F202\U0000FE0F': {  # 🈂️
        'en': ':Japanese_service_charge_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':sa:', ':japanese_service_charge_button:'],
        'variant': True,
        'de': ':schriftzeichen_sa:',
        'es': ':ideograma_japonés_para_de_cortesía:',
        'fr': ':bouton_frais_de_service_en_japonais:',
        'ja': ':サのマーク:',
        'ko': ':일본어_봉사료_버튼:',
        'pt': ':botão_japonês_de_“taxa_de_serviço”:',
        'it': ':ideogramma_giapponese_di_“costo_del_servizio”:',
        'fa': ':دکمه_هزینه_خدمات_به_ژاپنی:',
        'id': ':tombol_jepang_biaya_layanan:',
        'zh': ':服务费:',
        'ru': ':цена_услуги:'
    },
    '\U0001F202': {  # 🈂
        'en': ':Japanese_service_charge_button:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':sa:', ':japanese_service_charge_button:'],
        'variant': True,
        'de': ':schriftzeichen_sa:',
        'es': ':ideograma_japonés_para_de_cortesía:',
        'fr': ':bouton_frais_de_service_en_japonais:',
        'ja': ':サのマーク:',
        'ko': ':일본어_봉사료_버튼:',
        'pt': ':botão_japonês_de_“taxa_de_serviço”:',
        'it': ':ideogramma_giapponese_di_“costo_del_servizio”:',
        'fa': ':دکمهٔ_«هزینهٔ_خدمات»_به_ژاپنی:',
        'id': ':tombol_jepang_biaya_layanan:',
        'zh': ':服务:',
        'ru': ':цена_услуги:'
    },
    '\U0001F530': {  # 🔰
        'en': ':Japanese_symbol_for_beginner:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':beginner:', ':japanese_symbol_for_beginner:'],
        'de': ':japanisches_anfänger-zeichen:',
        'es': ':símbolo_japonés_para_principiante:',
        'fr': ':symbole_japonais_de_débutant:',
        'ja': ':初心者マーク:',
        'ko': ':노란색_초록색_이파리_모양:',
        'pt': ':símbolo_japonês_de_principiante:',
        'it': ':ideogramma_giapponese_di_“principiante”:',
        'fa': ':نماد_ژاپنی_سطح_مبتدی:',
        'id': ':simbol_jepang_untuk_pemula:',
        'zh': ':日本新手驾驶标志:',
        'ru': ':символ_новичка:'
    },
    '\U0001F233': {  # 🈳
        'en': ':Japanese_vacancy_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':u7a7a:', ':japanese_vacancy_button:'],
        'de': ':schriftzeichen_für_zimmer_frei:',
        'es': ':ideograma_japonés_para_vacante:',
        'fr': ':bouton_chambres_disponibles_en_japonais:',
        'ja': ':空マーク:',
        'ko': ':빌_공:',
        'pt': ':botão_japonês_de_“vago”:',
        'it': ':ideogramma_giapponese_di_“posto_libero”:',
        'fa': ':دکمۀ_«پست_خالی»_به_ژاپنی:',
        'id': ':tombol_jepang_ada_lowongan:',
        'zh': ':空:',
        'ru': ':есть_места:'
    },
    '\U0001F1EF\U0001F1EA': {  # 🇯🇪
        'en': ':Jersey:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Jersey:', ':jersey:'],
        'de': ':flagge_jersey:',
        'es': ':bandera_jersey:',
        'fr': ':drapeau_jersey:',
        'ja': ':旗_ジャージー:',
        'ko': ':깃발_저지:',
        'pt': ':bandeira_jersey:',
        'it': ':bandiera_jersey:',
        'fa': ':پرچم_جرزی:',
        'id': ':bendera_jersey:',
        'zh': ':泽西岛:',
        'ru': ':флаг_Джерси:'
    },
    '\U0001F1EF\U0001F1F4': {  # 🇯🇴
        'en': ':Jordan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Jordan:', ':jordan:'],
        'de': ':flagge_jordanien:',
        'es': ':bandera_jordania:',
        'fr': ':drapeau_jordanie:',
        'ja': ':旗_ヨルダン:',
        'ko': ':깃발_요르단:',
        'pt': ':bandeira_jordânia:',
        'it': ':bandiera_giordania:',
        'fa': ':پرچم_اردن:',
        'id': ':bendera_yordania:',
        'zh': ':约旦:',
        'ru': ':флаг_Иордания:'
    },
    '\U0001F1F0\U0001F1FF': {  # 🇰🇿
        'en': ':Kazakhstan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Kazakhstan:', ':kazakhstan:'],
        'de': ':flagge_kasachstan:',
        'es': ':bandera_kazajistán:',
        'fr': ':drapeau_kazakhstan:',
        'ja': ':旗_カザフスタン:',
        'ko': ':깃발_카자흐스탄:',
        'pt': ':bandeira_cazaquistão:',
        'it': ':bandiera_kazakistan:',
        'fa': ':پرچم_قزاقستان:',
        'id': ':bendera_kazakhstan:',
        'zh': ':哈薩克:',
        'ru': ':флаг_Казахстан:'
    },
    '\U0001F1F0\U0001F1EA': {  # 🇰🇪
        'en': ':Kenya:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Kenya:', ':kenya:'],
        'de': ':flagge_kenia:',
        'es': ':bandera_kenia:',
        'fr': ':drapeau_kenya:',
        'ja': ':旗_ケニア:',
        'ko': ':깃발_케냐:',
        'pt': ':bandeira_quênia:',
        'it': ':bandiera_kenya:',
        'fa': ':پرچم_کنیا:',
        'id': ':bendera_kenya:',
        'zh': ':肯尼亚:',
        'ru': ':флаг_Кения:'
    },
    '\U0001F1F0\U0001F1EE': {  # 🇰🇮
        'en': ':Kiribati:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Kiribati:', ':kiribati:'],
        'de': ':flagge_kiribati:',
        'es': ':bandera_kiribati:',
        'fr': ':drapeau_kiribati:',
        'ja': ':旗_キリバス:',
        'ko': ':깃발_키리바시:',
        'pt': ':bandeira_quiribati:',
        'it': ':bandiera_kiribati:',
        'fa': ':پرچم_کیریباتی:',
        'id': ':bendera_kiribati:',
        'zh': ':基里巴斯:',
        'ru': ':флаг_Кирибати:'
    },
    '\U0001F1FD\U0001F1F0': {  # 🇽🇰
        'en': ':Kosovo:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Kosovo:', ':kosovo:'],
        'de': ':flagge_kosovo:',
        'es': ':bandera_kosovo:',
        'fr': ':drapeau_kosovo:',
        'ja': ':旗_コソボ:',
        'ko': ':깃발_코소보:',
        'pt': ':bandeira_kosovo:',
        'it': ':bandiera_kosovo:',
        'fa': ':پرچم_کوزوو:',
        'id': ':bendera_kosovo:',
        'zh': ':科索沃:',
        'ru': ':флаг_Косово:'
    },
    '\U0001F1F0\U0001F1FC': {  # 🇰🇼
        'en': ':Kuwait:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Kuwait:', ':kuwait:'],
        'de': ':flagge_kuwait:',
        'es': ':bandera_kuwait:',
        'fr': ':drapeau_koweït:',
        'ja': ':旗_クウェート:',
        'ko': ':깃발_쿠웨이트:',
        'pt': ':bandeira_kuwait:',
        'it': ':bandiera_kuwait:',
        'fa': ':پرچم_کویت:',
        'id': ':bendera_kuwait:',
        'zh': ':科威特:',
        'ru': ':флаг_Кувейт:'
    },
    '\U0001F1F0\U0001F1EC': {  # 🇰🇬
        'en': ':Kyrgyzstan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Kyrgyzstan:', ':kyrgyzstan:'],
        'de': ':flagge_kirgisistan:',
        'es': ':bandera_kirguistán:',
        'fr': ':drapeau_kirghizstan:',
        'ja': ':旗_キルギス:',
        'ko': ':깃발_키르기스스탄:',
        'pt': ':bandeira_quirguistão:',
        'it': ':bandiera_kirghizistan:',
        'fa': ':پرچم_قرقیزستان:',
        'id': ':bendera_kirgizstan:',
        'zh': ':吉尔吉斯斯坦:',
        'ru': ':флаг_Киргизия:'
    },
    '\U0001F1F1\U0001F1E6': {  # 🇱🇦
        'en': ':Laos:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Laos:', ':laos:'],
        'de': ':flagge_laos:',
        'es': ':bandera_laos:',
        'fr': ':drapeau_laos:',
        'ja': ':旗_ラオス:',
        'ko': ':깃발_라오스:',
        'pt': ':bandeira_laos:',
        'it': ':bandiera_laos:',
        'fa': ':پرچم_لائوس:',
        'id': ':bendera_laos:',
        'zh': ':老挝:',
        'ru': ':флаг_Лаос:'
    },
    '\U0001F1F1\U0001F1FB': {  # 🇱🇻
        'en': ':Latvia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Latvia:', ':latvia:'],
        'de': ':flagge_lettland:',
        'es': ':bandera_letonia:',
        'fr': ':drapeau_lettonie:',
        'ja': ':旗_ラトビア:',
        'ko': ':깃발_라트비아:',
        'pt': ':bandeira_letônia:',
        'it': ':bandiera_lettonia:',
        'fa': ':پرچم_لتونی:',
        'id': ':bendera_latvia:',
        'zh': ':拉脱维亚:',
        'ru': ':флаг_Латвия:'
    },
    '\U0001F1F1\U0001F1E7': {  # 🇱🇧
        'en': ':Lebanon:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Lebanon:', ':lebanon:'],
        'de': ':flagge_libanon:',
        'es': ':bandera_líbano:',
        'fr': ':drapeau_liban:',
        'ja': ':旗_レバノン:',
        'ko': ':깃발_레바논:',
        'pt': ':bandeira_líbano:',
        'it': ':bandiera_libano:',
        'fa': ':پرچم_لبنان:',
        'id': ':bendera_lebanon:',
        'zh': ':黎巴嫩:',
        'ru': ':флаг_Ливан:'
    },
    '\U0000264C': {  # ♌
        'en': ':Leo:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':leo:'],
        'variant': True,
        'de': ':löwe_(sternzeichen):',
        'es': ':leo:',
        'fr': ':lion:',
        'ja': ':しし座:',
        'ko': ':사자자리:',
        'pt': ':signo_de_leão:',
        'it': ':segno_zodiacale_del_leone:',
        'fa': ':برج_اسد:',
        'id': ':leo:',
        'zh': ':狮子座:',
        'ru': ':знак_зодиака_Лев:'
    },
    '\U0001F1F1\U0001F1F8': {  # 🇱🇸
        'en': ':Lesotho:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Lesotho:', ':lesotho:'],
        'de': ':flagge_lesotho:',
        'es': ':bandera_lesoto:',
        'fr': ':drapeau_lesotho:',
        'ja': ':旗_レソト:',
        'ko': ':깃발_레소토:',
        'pt': ':bandeira_lesoto:',
        'it': ':bandiera_lesotho:',
        'fa': ':پرچم_لسوتو:',
        'id': ':bendera_lesotho:',
        'zh': ':莱索托:',
        'ru': ':флаг_Лесото:'
    },
    '\U0001F1F1\U0001F1F7': {  # 🇱🇷
        'en': ':Liberia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Liberia:', ':liberia:'],
        'de': ':flagge_liberia:',
        'es': ':bandera_liberia:',
        'fr': ':drapeau_liberia:',
        'ja': ':旗_リベリア:',
        'ko': ':깃발_라이베리아:',
        'pt': ':bandeira_libéria:',
        'it': ':bandiera_liberia:',
        'fa': ':پرچم_لیبریا:',
        'id': ':bendera_liberia:',
        'zh': ':利比里亚:',
        'ru': ':флаг_Либерия:'
    },
    '\U0000264E': {  # ♎
        'en': ':Libra:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':libra:'],
        'variant': True,
        'de': ':waage_(sternzeichen):',
        'es': ':libra:',
        'fr': ':balance:',
        'ja': ':てんびん座:',
        'ko': ':천칭자리:',
        'pt': ':signo_de_libra:',
        'it': ':segno_zodiacale_della_bilancia:',
        'fa': ':برج_میزان:',
        'id': ':libra:',
        'zh': ':天秤座:',
        'ru': ':знак_зодиака_Весы:'
    },
    '\U0001F1F1\U0001F1FE': {  # 🇱🇾
        'en': ':Libya:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Libya:', ':libya:'],
        'de': ':flagge_libyen:',
        'es': ':bandera_libia:',
        'fr': ':drapeau_libye:',
        'ja': ':旗_リビア:',
        'ko': ':깃발_리비아:',
        'pt': ':bandeira_líbia:',
        'it': ':bandiera_libia:',
        'fa': ':پرچم_لیبی:',
        'id': ':bendera_libya:',
        'zh': ':利比亚:',
        'ru': ':флаг_Ливия:'
    },
    '\U0001F1F1\U0001F1EE': {  # 🇱🇮
        'en': ':Liechtenstein:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Liechtenstein:', ':liechtenstein:'],
        'de': ':flagge_liechtenstein:',
        'es': ':bandera_liechtenstein:',
        'fr': ':drapeau_liechtenstein:',
        'ja': ':旗_リヒテンシュタイン:',
        'ko': ':깃발_리히텐슈타인:',
        'pt': ':bandeira_liechtenstein:',
        'it': ':bandiera_liechtenstein:',
        'fa': ':پرچم_لیختن_اشتاین:',
        'id': ':bendera_liechtenstein:',
        'zh': ':列支敦士登:',
        'ru': ':флаг_Лихтенштейн:'
    },
    '\U0001F1F1\U0001F1F9': {  # 🇱🇹
        'en': ':Lithuania:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Lithuania:', ':lithuania:'],
        'de': ':flagge_litauen:',
        'es': ':bandera_lituania:',
        'fr': ':drapeau_lituanie:',
        'ja': ':旗_リトアニア:',
        'ko': ':깃발_리투아니아:',
        'pt': ':bandeira_lituânia:',
        'it': ':bandiera_lituania:',
        'fa': ':پرچم_لیتوانی:',
        'id': ':bendera_lituania:',
        'zh': ':立陶宛:',
        'ru': ':флаг_Литва:'
    },
    '\U0001F1F1\U0001F1FA': {  # 🇱🇺
        'en': ':Luxembourg:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Luxembourg:', ':luxembourg:'],
        'de': ':flagge_luxemburg:',
        'es': ':bandera_luxemburgo:',
        'fr': ':drapeau_luxembourg:',
        'ja': ':旗_ルクセンブルク:',
        'ko': ':깃발_룩셈부르크:',
        'pt': ':bandeira_luxemburgo:',
        'it': ':bandiera_lussemburgo:',
        'fa': ':پرچم_لوکزامبورگ:',
        'id': ':bendera_luksemburg:',
        'zh': ':卢森堡:',
        'ru': ':флаг_Люксембург:'
    },
    '\U0001F1F2\U0001F1F4': {  # 🇲🇴
        'en': ':Macao_SAR_China:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Macau:', ':macau:'],
        'de': ':flagge_sonderverwaltungsregion_macau:',
        'es': ':bandera_rae_de_macao_(china):',
        'fr': ':drapeau_r.a.s._chinoise_de_macao:',
        'ja': ':旗_中華人民共和国マカオ特別行政区:',
        'ko': ':깃발_마카오(중국_특별행정구):',
        'pt': ':bandeira_macau_rae_da_china:',
        'it': ':bandiera_ras_di_macao:',
        'fa': ':پرچم_ماکائو_منطقهٔ_ویژهٔ_اداری_چین:',
        'id': ':bendera_makau_dak_tiongkok:',
        'zh': ':澳門:',
        'ru': ':флаг_Макао_(САР):'
    },
    '\U0001F1F2\U0001F1EC': {  # 🇲🇬
        'en': ':Madagascar:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Madagascar:', ':madagascar:'],
        'de': ':flagge_madagaskar:',
        'es': ':bandera_madagascar:',
        'fr': ':drapeau_madagascar:',
        'ja': ':旗_マダガスカル:',
        'ko': ':깃발_마다가스카르:',
        'pt': ':bandeira_madagascar:',
        'it': ':bandiera_madagascar:',
        'fa': ':پرچم_ماداگاسکار:',
        'id': ':bendera_madagaskar:',
        'zh': ':马达加斯加:',
        'ru': ':флаг_Мадагаскар:'
    },
    '\U0001F1F2\U0001F1FC': {  # 🇲🇼
        'en': ':Malawi:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Malawi:', ':malawi:'],
        'de': ':flagge_malawi:',
        'es': ':bandera_malaui:',
        'fr': ':drapeau_malawi:',
        'ja': ':旗_マラウイ:',
        'ko': ':깃발_말라위:',
        'pt': ':bandeira_malaui:',
        'it': ':bandiera_malawi:',
        'fa': ':پرچم_مالاوی:',
        'id': ':bendera_malawi:',
        'zh': ':马拉维:',
        'ru': ':флаг_Малави:'
    },
    '\U0001F1F2\U0001F1FE': {  # 🇲🇾
        'en': ':Malaysia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Malaysia:', ':malaysia:'],
        'de': ':flagge_malaysia:',
        'es': ':bandera_malasia:',
        'fr': ':drapeau_malaisie:',
        'ja': ':旗_マレーシア:',
        'ko': ':깃발_말레이시아:',
        'pt': ':bandeira_malásia:',
        'it': ':bandiera_malaysia:',
        'fa': ':پرچم_مالزی:',
        'id': ':bendera_malaysia:',
        'zh': ':马来西亚:',
        'ru': ':флаг_Малайзия:'
    },
    '\U0001F1F2\U0001F1FB': {  # 🇲🇻
        'en': ':Maldives:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Maldives:', ':maldives:'],
        'de': ':flagge_malediven:',
        'es': ':bandera_maldivas:',
        'fr': ':drapeau_maldives:',
        'ja': ':旗_モルディブ:',
        'ko': ':깃발_몰디브:',
        'pt': ':bandeira_maldivas:',
        'it': ':bandiera_maldive:',
        'fa': ':پرچم_مالدیو:',
        'id': ':bendera_maladewa:',
        'zh': ':马尔代夫:',
        'ru': ':флаг_Мальдивы:'
    },
    '\U0001F1F2\U0001F1F1': {  # 🇲🇱
        'en': ':Mali:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Mali:', ':mali:'],
        'de': ':flagge_mali:',
        'es': ':bandera_mali:',
        'fr': ':drapeau_mali:',
        'ja': ':旗_マリ:',
        'ko': ':깃발_말리:',
        'pt': ':bandeira_mali:',
        'it': ':bandiera_mali:',
        'fa': ':پرچم_مالی:',
        'id': ':bendera_mali:',
        'zh': ':马里:',
        'ru': ':флаг_Мали:'
    },
    '\U0001F1F2\U0001F1F9': {  # 🇲🇹
        'en': ':Malta:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Malta:', ':malta:'],
        'de': ':flagge_malta:',
        'es': ':bandera_malta:',
        'fr': ':drapeau_malte:',
        'ja': ':旗_マルタ:',
        'ko': ':깃발_몰타:',
        'pt': ':bandeira_malta:',
        'it': ':bandiera_malta:',
        'fa': ':پرچم_مالت:',
        'id': ':bendera_malta:',
        'zh': ':马耳他:',
        'ru': ':флаг_Мальта:'
    },
    '\U0001F1F2\U0001F1ED': {  # 🇲🇭
        'en': ':Marshall_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Marshall_Islands:', ':marshall_islands:'],
        'de': ':flagge_marshallinseln:',
        'es': ':bandera_islas_marshall:',
        'fr': ':drapeau_îles_marshall:',
        'ja': ':旗_マーシャル諸島:',
        'ko': ':깃발_마셜_제도:',
        'pt': ':bandeira_ilhas_marshall:',
        'it': ':bandiera_isole_marshall:',
        'fa': ':پرچم_جزایر_مارشال:',
        'id': ':bendera_kepulauan_marshall:',
        'zh': ':马绍尔群岛:',
        'ru': ':флаг_Маршалловы_о_ва:'
    },
    '\U0001F1F2\U0001F1F6': {  # 🇲🇶
        'en': ':Martinique:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Martinique:', ':martinique:'],
        'de': ':flagge_martinique:',
        'es': ':bandera_martinica:',
        'fr': ':drapeau_martinique:',
        'ja': ':旗_マルティニーク:',
        'ko': ':깃발_마르티니크:',
        'pt': ':bandeira_martinica:',
        'it': ':bandiera_martinica:',
        'fa': ':پرچم_مارتینیک:',
        'id': ':bendera_martinik:',
        'zh': ':马提尼克:',
        'ru': ':флаг_Мартиника:'
    },
    '\U0001F1F2\U0001F1F7': {  # 🇲🇷
        'en': ':Mauritania:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Mauritania:', ':mauritania:'],
        'de': ':flagge_mauretanien:',
        'es': ':bandera_mauritania:',
        'fr': ':drapeau_mauritanie:',
        'ja': ':旗_モーリタニア:',
        'ko': ':깃발_모리타니:',
        'pt': ':bandeira_mauritânia:',
        'it': ':bandiera_mauritania:',
        'fa': ':پرچم_موریتانی:',
        'id': ':bendera_mauritania:',
        'zh': ':毛里塔尼亚:',
        'ru': ':флаг_Мавритания:'
    },
    '\U0001F1F2\U0001F1FA': {  # 🇲🇺
        'en': ':Mauritius:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Mauritius:', ':mauritius:'],
        'de': ':flagge_mauritius:',
        'es': ':bandera_mauricio:',
        'fr': ':drapeau_maurice:',
        'ja': ':旗_モーリシャス:',
        'ko': ':깃발_모리셔스:',
        'pt': ':bandeira_maurício:',
        'it': ':bandiera_mauritius:',
        'fa': ':پرچم_موریس:',
        'id': ':bendera_mauritius:',
        'zh': ':毛里求斯:',
        'ru': ':флаг_Маврикий:'
    },
    '\U0001F1FE\U0001F1F9': {  # 🇾🇹
        'en': ':Mayotte:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Mayotte:', ':mayotte:'],
        'de': ':flagge_mayotte:',
        'es': ':bandera_mayotte:',
        'fr': ':drapeau_mayotte:',
        'ja': ':旗_マヨット:',
        'ko': ':깃발_마요트:',
        'pt': ':bandeira_mayotte:',
        'it': ':bandiera_mayotte:',
        'fa': ':پرچم_مایوت:',
        'id': ':bendera_mayotte:',
        'zh': ':马约特:',
        'ru': ':флаг_Майотта:'
    },
    '\U0001F1F2\U0001F1FD': {  # 🇲🇽
        'en': ':Mexico:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Mexico:', ':mexico:'],
        'de': ':flagge_mexiko:',
        'es': ':bandera_méxico:',
        'fr': ':drapeau_mexique:',
        'ja': ':旗_メキシコ:',
        'ko': ':깃발_멕시코:',
        'pt': ':bandeira_méxico:',
        'it': ':bandiera_messico:',
        'fa': ':پرچم_مکزیک:',
        'id': ':bendera_meksiko:',
        'zh': ':墨西哥:',
        'ru': ':флаг_Мексика:'
    },
    '\U0001F1EB\U0001F1F2': {  # 🇫🇲
        'en': ':Micronesia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Micronesia:', ':micronesia:'],
        'de': ':flagge_mikronesien:',
        'es': ':bandera_micronesia:',
        'fr': ':drapeau_micronésie:',
        'ja': ':旗_ミクロネシア連邦:',
        'ko': ':깃발_미크로네시아:',
        'pt': ':bandeira_micronésia:',
        'it': ':bandiera_micronesia:',
        'fa': ':پرچم_میکرونزی:',
        'id': ':bendera_mikronesia:',
        'zh': ':密克罗尼西亚:',
        'ru': ':флаг_Федеративные_Штаты_Микронезии:'
    },
    '\U0001F1F2\U0001F1E9': {  # 🇲🇩
        'en': ':Moldova:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Moldova:', ':moldova:'],
        'de': ':flagge_republik_moldau:',
        'es': ':bandera_moldavia:',
        'fr': ':drapeau_moldavie:',
        'ja': ':旗_モルドバ:',
        'ko': ':깃발_몰도바:',
        'pt': ':bandeira_moldávia:',
        'it': ':bandiera_moldavia:',
        'fa': ':پرچم_مولداوی:',
        'id': ':bendera_moldova:',
        'zh': ':摩尔多瓦:',
        'ru': ':флаг_Молдова:'
    },
    '\U0001F1F2\U0001F1E8': {  # 🇲🇨
        'en': ':Monaco:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Monaco:', ':monaco:'],
        'de': ':flagge_monaco:',
        'es': ':bandera_mónaco:',
        'fr': ':drapeau_monaco:',
        'ja': ':旗_モナコ:',
        'ko': ':깃발_모나코:',
        'pt': ':bandeira_mônaco:',
        'it': ':bandiera_monaco:',
        'fa': ':پرچم_موناکو:',
        'id': ':bendera_monako:',
        'zh': ':摩纳哥:',
        'ru': ':флаг_Монако:'
    },
    '\U0001F1F2\U0001F1F3': {  # 🇲🇳
        'en': ':Mongolia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Mongolia:', ':mongolia:'],
        'de': ':flagge_mongolei:',
        'es': ':bandera_mongolia:',
        'fr': ':drapeau_mongolie:',
        'ja': ':旗_モンゴル:',
        'ko': ':깃발_몽골:',
        'pt': ':bandeira_mongólia:',
        'it': ':bandiera_mongolia:',
        'fa': ':پرچم_مغولستان:',
        'id': ':bendera_mongolia:',
        'zh': ':蒙古:',
        'ru': ':флаг_Монголия:'
    },
    '\U0001F1F2\U0001F1EA': {  # 🇲🇪
        'en': ':Montenegro:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Montenegro:', ':montenegro:'],
        'de': ':flagge_montenegro:',
        'es': ':bandera_montenegro:',
        'fr': ':drapeau_monténégro:',
        'ja': ':旗_モンテネグロ:',
        'ko': ':깃발_몬테네그로:',
        'pt': ':bandeira_montenegro:',
        'it': ':bandiera_montenegro:',
        'fa': ':پرچم_مونته_نگرو:',
        'id': ':bendera_montenegro:',
        'zh': ':黑山:',
        'ru': ':флаг_Черногория:'
    },
    '\U0001F1F2\U0001F1F8': {  # 🇲🇸
        'en': ':Montserrat:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Montserrat:', ':montserrat:'],
        'de': ':flagge_montserrat:',
        'es': ':bandera_montserrat:',
        'fr': ':drapeau_montserrat:',
        'ja': ':旗_モントセラト:',
        'ko': ':깃발_몬트세라트:',
        'pt': ':bandeira_montserrat:',
        'it': ':bandiera_montserrat:',
        'fa': ':پرچم_مونت_سرات:',
        'id': ':bendera_montserrat:',
        'zh': ':蒙特塞拉特:',
        'ru': ':флаг_Монтсеррат:'
    },
    '\U0001F1F2\U0001F1E6': {  # 🇲🇦
        'en': ':Morocco:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Morocco:', ':morocco:'],
        'de': ':flagge_marokko:',
        'es': ':bandera_marruecos:',
        'fr': ':drapeau_maroc:',
        'ja': ':旗_モロッコ:',
        'ko': ':깃발_모로코:',
        'pt': ':bandeira_marrocos:',
        'it': ':bandiera_marocco:',
        'fa': ':پرچم_مراکش:',
        'id': ':bendera_maroko:',
        'zh': ':摩洛哥:',
        'ru': ':флаг_Марокко:'
    },
    '\U0001F1F2\U0001F1FF': {  # 🇲🇿
        'en': ':Mozambique:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Mozambique:', ':mozambique:'],
        'de': ':flagge_mosambik:',
        'es': ':bandera_mozambique:',
        'fr': ':drapeau_mozambique:',
        'ja': ':旗_モザンビーク:',
        'ko': ':깃발_모잠비크:',
        'pt': ':bandeira_moçambique:',
        'it': ':bandiera_mozambico:',
        'fa': ':پرچم_موزامبیک:',
        'id': ':bendera_mozambik:',
        'zh': ':莫桑比克:',
        'ru': ':флаг_Мозамбик:'
    },
    '\U0001F936': {  # 🤶
        'en': ':Mrs._Claus:',
        'status': fully_qualified,
        'E': 3,
        'alias': [':mrs_claus:'],
        'de': ':weihnachtsfrau:',
        'es': ':mamá_noel:',
        'fr': ':mère_noël:',
        'ja': ':女性のサンタ:',
        'ko': ':산타할머니:',
        'pt': ':mamãe_noel:',
        'it': ':mamma_natale:',
        'fa': ':مامان_نوئل:',
        'id': ':nyonya_sinterklas:',
        'zh': ':圣诞奶奶:',
        'ru': ':миссис_Клаус:'
    },
    '\U0001F936\U0001F3FF': {  # 🤶🏿
        'en': ':Mrs._Claus_dark_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':weihnachtsfrau_dunkle_hautfarbe:',
        'es': ':mamá_noel_tono_de_piel_oscuro:',
        'fr': ':mère_noël_peau_foncée:',
        'ja': ':女性のサンタ_濃い肌色:',
        'ko': ':산타할머니_검은색_피부:',
        'pt': ':mamãe_noel_pele_escura:',
        'it': ':mamma_natale_carnagione_scura:',
        'fa': ':مامان_نوئل_پوست_آبنوسی:',
        'id': ':nyonya_sinterklas_warna_kulit_gelap:',
        'zh': ':圣诞奶奶_较深肤色:',
        'ru': ':миссис_Клаус_очень_темный_тон_кожи:'
    },
    '\U0001F936\U0001F3FB': {  # 🤶🏻
        'en': ':Mrs._Claus_light_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':weihnachtsfrau_helle_hautfarbe:',
        'es': ':mamá_noel_tono_de_piel_claro:',
        'fr': ':mère_noël_peau_claire:',
        'ja': ':女性のサンタ_薄い肌色:',
        'ko': ':산타할머니_하얀_피부:',
        'pt': ':mamãe_noel_pele_clara:',
        'it': ':mamma_natale_carnagione_chiara:',
        'fa': ':مامان_نوئل_پوست_سفید:',
        'id': ':nyonya_sinterklas_warna_kulit_cerah:',
        'zh': ':圣诞奶奶_较浅肤色:',
        'ru': ':миссис_Клаус_очень_светлый_тон_кожи:'
    },
    '\U0001F936\U0001F3FE': {  # 🤶🏾
        'en': ':Mrs._Claus_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':weihnachtsfrau_mitteldunkle_hautfarbe:',
        'es': ':mamá_noel_tono_de_piel_oscuro_medio:',
        'fr': ':mère_noël_peau_mate:',
        'ja': ':女性のサンタ_やや濃い肌色:',
        'ko': ':산타할머니_진한_갈색_피부:',
        'pt': ':mamãe_noel_pele_morena_escura:',
        'it': ':mamma_natale_carnagione_abbastanza_scura:',
        'fa': ':مامان_نوئل_پوست_گندمی:',
        'id': ':nyonya_sinterklas_warna_kulit_gelap-sedang:',
        'zh': ':圣诞奶奶_中等深肤色:',
        'ru': ':миссис_Клаус_темный_тон_кожи:'
    },
    '\U0001F936\U0001F3FC': {  # 🤶🏼
        'en': ':Mrs._Claus_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':weihnachtsfrau_mittelhelle_hautfarbe:',
        'es': ':mamá_noel_tono_de_piel_claro_medio:',
        'fr': ':mère_noël_peau_moyennement_claire:',
        'ja': ':女性のサンタ_やや薄い肌色:',
        'ko': ':산타할머니_연한_갈색_피부:',
        'pt': ':mamãe_noel_pele_morena_clara:',
        'it': ':mamma_natale_carnagione_abbastanza_chiara:',
        'fa': ':مامان_نوئل_پوست_روشن:',
        'id': ':nyonya_sinterklas_warna_kulit_cerah-sedang:',
        'zh': ':圣诞奶奶_中等浅肤色:',
        'ru': ':миссис_Клаус_светлый_тон_кожи:'
    },
    '\U0001F936\U0001F3FD': {  # 🤶🏽
        'en': ':Mrs._Claus_medium_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':weihnachtsfrau_mittlere_hautfarbe:',
        'es': ':mamá_noel_tono_de_piel_medio:',
        'fr': ':mère_noël_peau_légèrement_mate:',
        'ja': ':女性のサンタ_中間の肌色:',
        'ko': ':산타할머니_갈색_피부:',
        'pt': ':mamãe_noel_pele_morena:',
        'it': ':mamma_natale_carnagione_olivastra:',
        'fa': ':مامان_نوئل_پوست_طلایی:',
        'id': ':nyonya_sinterklas_warna_kulit_sedang:',
        'zh': ':圣诞奶奶_中等肤色:',
        'ru': ':миссис_Клаус_средний_тон_кожи:'
    },
    '\U0001F1F2\U0001F1F2': {  # 🇲🇲
        'en': ':Myanmar_(Burma):',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Myanmar:', ':myanmar:'],
        'de': ':flagge_myanmar:',
        'es': ':bandera_myanmar_(birmania):',
        'fr': ':drapeau_myanmar_(birmanie):',
        'ja': ':旗_ミャンマー_(ビルマ):',
        'ko': ':깃발_미얀마:',
        'pt': ':bandeira_mianmar_(birmânia):',
        'it': ':bandiera_myanmar_(birmania):',
        'fa': ':پرچم_میانمار_(برمه):',
        'id': ':bendera_myanmar_(burma):',
        'zh': ':缅甸:',
        'ru': ':флаг_Мьянма_(Бирма):'
    },
    '\U0001F195': {  # 🆕
        'en': ':NEW_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':new:', ':new_button:'],
        'de': ':wort_new_in_blauem_quadrat:',
        'es': ':botón_new:',
        'fr': ':bouton_nouveau:',
        'ja': ':newマーク:',
        'ko': ':뉴:',
        'pt': ':botão_new:',
        'it': ':pulsante_new:',
        'fa': ':دکمهٔ_new:',
        'id': ':tombol_new:',
        'zh': ':new按钮:',
        'ru': ':значок_«новинка»:'
    },
    '\U0001F196': {  # 🆖
        'en': ':NG_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':ng:', ':ng_button:'],
        'de': ':großbuchstaben_ng_in_blauem_quadrat:',
        'es': ':botón_ng:',
        'fr': ':bouton_pas_bien:',
        'ja': ':ngマーク:',
        'ko': ':엔지:',
        'pt': ':botão_ng:',
        'it': ':pulsante_ng:',
        'fa': ':دکمهٔ_ng:',
        'id': ':tombol_ng:',
        'zh': ':NG按钮:',
        'ru': ':значок_«не_годится»:'
    },
    '\U0001F1F3\U0001F1E6': {  # 🇳🇦
        'en': ':Namibia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Namibia:', ':namibia:'],
        'de': ':flagge_namibia:',
        'es': ':bandera_namibia:',
        'fr': ':drapeau_namibie:',
        'ja': ':旗_ナミビア:',
        'ko': ':깃발_나미비아:',
        'pt': ':bandeira_namíbia:',
        'it': ':bandiera_namibia:',
        'fa': ':پرچم_نامیبیا:',
        'id': ':bendera_namibia:',
        'zh': ':纳米比亚:',
        'ru': ':флаг_Намибия:'
    },
    '\U0001F1F3\U0001F1F7': {  # 🇳🇷
        'en': ':Nauru:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Nauru:', ':nauru:'],
        'de': ':flagge_nauru:',
        'es': ':bandera_nauru:',
        'fr': ':drapeau_nauru:',
        'ja': ':旗_ナウル:',
        'ko': ':깃발_나우루:',
        'pt': ':bandeira_nauru:',
        'it': ':bandiera_nauru:',
        'fa': ':پرچم_نائورو:',
        'id': ':bendera_nauru:',
        'zh': ':瑙鲁:',
        'ru': ':флаг_Науру:'
    },
    '\U0001F1F3\U0001F1F5': {  # 🇳🇵
        'en': ':Nepal:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Nepal:', ':nepal:'],
        'de': ':flagge_nepal:',
        'es': ':bandera_nepal:',
        'fr': ':drapeau_népal:',
        'ja': ':旗_ネパール:',
        'ko': ':깃발_네팔:',
        'pt': ':bandeira_nepal:',
        'it': ':bandiera_nepal:',
        'fa': ':پرچم_نپال:',
        'id': ':bendera_nepal:',
        'zh': ':尼泊尔:',
        'ru': ':флаг_Непал:'
    },
    '\U0001F1F3\U0001F1F1': {  # 🇳🇱
        'en': ':Netherlands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Netherlands:', ':netherlands:'],
        'de': ':flagge_niederlande:',
        'es': ':bandera_países_bajos:',
        'fr': ':drapeau_pays-bas:',
        'ja': ':旗_オランダ:',
        'ko': ':깃발_네덜란드:',
        'pt': ':bandeira_países_baixos:',
        'it': ':bandiera_paesi_bassi:',
        'fa': ':پرچم_هلند:',
        'id': ':bendera_belanda:',
        'zh': ':荷兰:',
        'ru': ':флаг_Нидерланды:'
    },
    '\U0001F1F3\U0001F1E8': {  # 🇳🇨
        'en': ':New_Caledonia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_New_Caledonia:', ':new_caledonia:'],
        'de': ':flagge_neukaledonien:',
        'es': ':bandera_nueva_caledonia:',
        'fr': ':drapeau_nouvelle-calédonie:',
        'ja': ':旗_ニューカレドニア:',
        'ko': ':깃발_뉴칼레도니아:',
        'pt': ':bandeira_nova_caledônia:',
        'it': ':bandiera_nuova_caledonia:',
        'fa': ':پرچم_کالدونیای_جدید:',
        'id': ':bendera_kaledonia_baru:',
        'zh': ':新喀里多尼亚:',
        'ru': ':флаг_Новая_Каледония:'
    },
    '\U0001F1F3\U0001F1FF': {  # 🇳🇿
        'en': ':New_Zealand:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_New_Zealand:', ':new_zealand:'],
        'de': ':flagge_neuseeland:',
        'es': ':bandera_nueva_zelanda:',
        'fr': ':drapeau_nouvelle-zélande:',
        'ja': ':旗_ニュージーランド:',
        'ko': ':깃발_뉴질랜드:',
        'pt': ':bandeira_nova_zelândia:',
        'it': ':bandiera_nuova_zelanda:',
        'fa': ':پرچم_نیوزیلند:',
        'id': ':bendera_selandia_baru:',
        'zh': ':新西兰:',
        'ru': ':флаг_Новая_Зеландия:'
    },
    '\U0001F1F3\U0001F1EE': {  # 🇳🇮
        'en': ':Nicaragua:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Nicaragua:', ':nicaragua:'],
        'de': ':flagge_nicaragua:',
        'es': ':bandera_nicaragua:',
        'fr': ':drapeau_nicaragua:',
        'ja': ':旗_ニカラグア:',
        'ko': ':깃발_니카라과:',
        'pt': ':bandeira_nicarágua:',
        'it': ':bandiera_nicaragua:',
        'fa': ':پرچم_نیکاراگوئه:',
        'id': ':bendera_nikaragua:',
        'zh': ':尼加拉瓜:',
        'ru': ':флаг_Никарагуа:'
    },
    '\U0001F1F3\U0001F1EA': {  # 🇳🇪
        'en': ':Niger:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Niger:', ':niger:'],
        'de': ':flagge_niger:',
        'es': ':bandera_níger:',
        'fr': ':drapeau_niger:',
        'ja': ':旗_ニジェール:',
        'ko': ':깃발_니제르:',
        'pt': ':bandeira_níger:',
        'it': ':bandiera_niger:',
        'fa': ':پرچم_نیجر:',
        'id': ':bendera_niger:',
        'zh': ':尼日尔:',
        'ru': ':флаг_Нигер:'
    },
    '\U0001F1F3\U0001F1EC': {  # 🇳🇬
        'en': ':Nigeria:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Nigeria:', ':nigeria:'],
        'de': ':flagge_nigeria:',
        'es': ':bandera_nigeria:',
        'fr': ':drapeau_nigeria:',
        'ja': ':旗_ナイジェリア:',
        'ko': ':깃발_나이지리아:',
        'pt': ':bandeira_nigéria:',
        'it': ':bandiera_nigeria:',
        'fa': ':پرچم_نیجریه:',
        'id': ':bendera_nigeria:',
        'zh': ':尼日利亚:',
        'ru': ':флаг_Нигерия:'
    },
    '\U0001F1F3\U0001F1FA': {  # 🇳🇺
        'en': ':Niue:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Niue:', ':niue:'],
        'de': ':flagge_niue:',
        'es': ':bandera_niue:',
        'fr': ':drapeau_niue:',
        'ja': ':旗_ニウエ:',
        'ko': ':깃발_니우에:',
        'pt': ':bandeira_niue:',
        'it': ':bandiera_niue:',
        'fa': ':پرچم_نیوئه:',
        'id': ':bendera_niue:',
        'zh': ':纽埃:',
        'ru': ':флаг_Ниуэ:'
    },
    '\U0001F1F3\U0001F1EB': {  # 🇳🇫
        'en': ':Norfolk_Island:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Norfolk_Island:', ':norfolk_island:'],
        'de': ':flagge_norfolkinsel:',
        'es': ':bandera_isla_norfolk:',
        'fr': ':drapeau_île_norfolk:',
        'ja': ':旗_ノーフォーク島:',
        'ko': ':깃발_노퍽섬:',
        'pt': ':bandeira_ilha_norfolk:',
        'it': ':bandiera_isola_norfolk:',
        'fa': ':پرچم_جزیرهٔ_نورفولک:',
        'id': ':bendera_kepulauan_norfolk:',
        'zh': ':诺福克岛:',
        'ru': ':флаг_о_в_Норфолк:'
    },
    '\U0001F1F0\U0001F1F5': {  # 🇰🇵
        'en': ':North_Korea:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_North_Korea:', ':north_korea:'],
        'de': ':flagge_nordkorea:',
        'es': ':bandera_corea_del_norte:',
        'fr': ':drapeau_corée_du_nord:',
        'ja': ':旗_北朝鮮:',
        'ko': ':깃발_북한:',
        'pt': ':bandeira_coreia_do_norte:',
        'it': ':bandiera_corea_del_nord:',
        'fa': ':پرچم_کرهٔ_شمالی:',
        'id': ':bendera_korea_utara:',
        'zh': ':朝鲜:',
        'ru': ':флаг_КНДР:'
    },
    '\U0001F1F2\U0001F1F0': {  # 🇲🇰
        'en': ':North_Macedonia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Macedonia:', ':macedonia:'],
        'de': ':flagge_nordmazedonien:',
        'es': ':bandera_macedonia_del_norte:',
        'fr': ':drapeau_macédoine_du_nord:',
        'ja': ':旗_北マケドニア:',
        'ko': ':깃발_북마케도니아:',
        'pt': ':bandeira_macedônia_do_norte:',
        'it': ':bandiera_macedonia_del_nord:',
        'fa': ':پرچم_مقدونیهٔ_شمالی:',
        'id': ':bendera_makedonia_utara:',
        'zh': ':北马其顿:',
        'ru': ':флаг_Северная_Македония:'
    },
    '\U0001F1F2\U0001F1F5': {  # 🇲🇵
        'en': ':Northern_Mariana_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Northern_Mariana_Islands:', ':northern_mariana_islands:'],
        'de': ':flagge_nördliche_marianen:',
        'es': ':bandera_islas_marianas_del_norte:',
        'fr': ':drapeau_îles_mariannes_du_nord:',
        'ja': ':旗_北マリアナ諸島:',
        'ko': ':깃발_북마리아나제도:',
        'pt': ':bandeira_ilhas_marianas_do_norte:',
        'it': ':bandiera_isole_marianne_settentrionali:',
        'fa': ':پرچم_جزایر_ماریانای_شمالی:',
        'id': ':bendera_kepulauan_mariana_utara:',
        'zh': ':北马里亚纳群岛:',
        'ru': ':флаг_Северные_Марианские_о_ва:'
    },
    '\U0001F1F3\U0001F1F4': {  # 🇳🇴
        'en': ':Norway:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Norway:', ':norway:'],
        'de': ':flagge_norwegen:',
        'es': ':bandera_noruega:',
        'fr': ':drapeau_norvège:',
        'ja': ':旗_ノルウェー:',
        'ko': ':깃발_노르웨이:',
        'pt': ':bandeira_noruega:',
        'it': ':bandiera_norvegia:',
        'fa': ':پرچم_نروژ:',
        'id': ':bendera_norwegia:',
        'zh': ':挪威:',
        'ru': ':флаг_Норвегия:'
    },
    '\U0001F197': {  # 🆗
        'en': ':OK_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':ok:', ':ok_button:'],
        'de': ':großbuchstaben_ok_in_blauem_quadrat:',
        'es': ':botón_ok:',
        'fr': ':bouton_ok:',
        'ja': ':okマーク:',
        'ko': ':오케이:',
        'pt': ':botão_ok:',
        'it': ':pulsante_ok:',
        'fa': ':دکمهٔ_ok:',
        'id': ':tombol_ok:',
        'zh': ':OK按钮:',
        'ru': ':значок_«ОК»:'
    },
    '\U0001F44C': {  # 👌
        'en': ':OK_hand:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':ok_hand:'],
        'de': ':ok-zeichen:',
        'es': ':señal_de_aprobación_con_la_mano:',
        'fr': ':ok:',
        'ja': ':okの手:',
        'ko': ':오케이를_표시한_손:',
        'pt': ':sinal_de_ok:',
        'it': ':mano_che_fa_ok:',
        'fa': ':علامت_اوکی:',
        'id': ':oke:',
        'zh': ':OK:',
        'ru': ':жест_«все_хорошо»:'
    },
    '\U0001F44C\U0001F3FF': {  # 👌🏿
        'en': ':OK_hand_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ok-zeichen_dunkle_hautfarbe:',
        'es': ':señal_de_aprobación_con_la_mano_tono_de_piel_oscuro:',
        'fr': ':ok_peau_foncée:',
        'ja': ':okの手_濃い肌色:',
        'ko': ':오케이를_표시한_손_검은색_피부:',
        'pt': ':sinal_de_ok_pele_escura:',
        'it': ':mano_che_fa_ok_carnagione_scura:',
        'fa': ':علامت_اوکی_پوست_آبنوسی:',
        'id': ':oke_warna_kulit_gelap:',
        'zh': ':OK_较深肤色:',
        'ru': ':жест_«все_хорошо»_очень_темный_тон_кожи:'
    },
    '\U0001F44C\U0001F3FB': {  # 👌🏻
        'en': ':OK_hand_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ok-zeichen_helle_hautfarbe:',
        'es': ':señal_de_aprobación_con_la_mano_tono_de_piel_claro:',
        'fr': ':ok_peau_claire:',
        'ja': ':okの手_薄い肌色:',
        'ko': ':오케이를_표시한_손_하얀_피부:',
        'pt': ':sinal_de_ok_pele_clara:',
        'it': ':mano_che_fa_ok_carnagione_chiara:',
        'fa': ':علامت_اوکی_پوست_سفید:',
        'id': ':oke_warna_kulit_cerah:',
        'zh': ':OK_较浅肤色:',
        'ru': ':жест_«все_хорошо»_очень_светлый_тон_кожи:'
    },
    '\U0001F44C\U0001F3FE': {  # 👌🏾
        'en': ':OK_hand_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ok-zeichen_mitteldunkle_hautfarbe:',
        'es': ':señal_de_aprobación_con_la_mano_tono_de_piel_oscuro_medio:',
        'fr': ':ok_peau_mate:',
        'ja': ':okの手_やや濃い肌色:',
        'ko': ':오케이를_표시한_손_진한_갈색_피부:',
        'pt': ':sinal_de_ok_pele_morena_escura:',
        'it': ':mano_che_fa_ok_carnagione_abbastanza_scura:',
        'fa': ':علامت_اوکی_پوست_گندمی:',
        'id': ':oke_warna_kulit_gelap-sedang:',
        'zh': ':OK_中等深肤色:',
        'ru': ':жест_«все_хорошо»_темный_тон_кожи:'
    },
    '\U0001F44C\U0001F3FC': {  # 👌🏼
        'en': ':OK_hand_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ok-zeichen_mittelhelle_hautfarbe:',
        'es': ':señal_de_aprobación_con_la_mano_tono_de_piel_claro_medio:',
        'fr': ':ok_peau_moyennement_claire:',
        'ja': ':okの手_やや薄い肌色:',
        'ko': ':오케이를_표시한_손_연한_갈색_피부:',
        'pt': ':sinal_de_ok_pele_morena_clara:',
        'it': ':mano_che_fa_ok_carnagione_abbastanza_chiara:',
        'fa': ':علامت_اوکی_پوست_روشن:',
        'id': ':oke_warna_kulit_cerah-sedang:',
        'zh': ':OK_中等浅肤色:',
        'ru': ':жест_«все_хорошо»_светлый_тон_кожи:'
    },
    '\U0001F44C\U0001F3FD': {  # 👌🏽
        'en': ':OK_hand_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ok-zeichen_mittlere_hautfarbe:',
        'es': ':señal_de_aprobación_con_la_mano_tono_de_piel_medio:',
        'fr': ':ok_peau_légèrement_mate:',
        'ja': ':okの手_中間の肌色:',
        'ko': ':오케이를_표시한_손_갈색_피부:',
        'pt': ':sinal_de_ok_pele_morena:',
        'it': ':mano_che_fa_ok_carnagione_olivastra:',
        'fa': ':علامت_اوکی_پوست_طلایی:',
        'id': ':oke_warna_kulit_sedang:',
        'zh': ':OK_中等肤色:',
        'ru': ':жест_«все_хорошо»_средний_тон_кожи:'
    },
    '\U0001F51B': {  # 🔛
        'en': ':ON!_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':on:', ':on_arrow:', ':on!_arrow:'],
        'de': ':on!-pfeil:',
        'es': ':flecha_on!:',
        'fr': ':flèche_activé:',
        'ja': ':on矢印:',
        'ko': ':켜짐:',
        'pt': ':seta_on!:',
        'it': ':freccia_on:',
        'fa': ':پیکان_روشن:',
        'id': ':tanda_panah_on!:',
        'zh': ':ON箭头:',
        'ru': ':стрелка_«активно»:'
    },
    '\U0001F17E\U0000FE0F': {  # 🅾️
        'en': ':O_button_(blood_type):',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':o2:', ':o_button_blood_type:'],
        'variant': True,
        'de': ':großbuchstabe_o_in_rotem_quadrat:',
        'es': ':grupo_sanguíneo_tipo_o:',
        'fr': ':groupe_sanguin_o:',
        'ja': ':血液型o型:',
        'ko': ':오형:',
        'pt': ':botão_o_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_0:',
        'fa': ':دکمه_ا_(گروه_خونی):',
        'id': ':tombol_o_(golongan_darah):',
        'zh': ':O型血:',
        'ru': ':i_группа_крови:'
    },
    '\U0001F17E': {  # 🅾
        'en': ':O_button_(blood_type):',
        'status': unqualified,
        'E': 0.6,
        'alias': [':o2:', ':o_button_blood_type:'],
        'variant': True,
        'de': ':großbuchstabe_o_in_rotem_quadrat:',
        'es': ':grupo_sanguíneo_tipo_o:',
        'fr': ':groupe_sanguin_o:',
        'ja': ':血液型o型:',
        'ko': ':오형:',
        'pt': ':botão_o_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_0:',
        'fa': ':دکمه_اُ_(گروه_خونی):',
        'id': ':tombol_o_(golongan_darah):',
        'zh': ':O型血:',
        'ru': ':I_группа_крови:'
    },
    '\U0001F1F4\U0001F1F2': {  # 🇴🇲
        'en': ':Oman:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Oman:', ':oman:'],
        'de': ':flagge_oman:',
        'es': ':bandera_omán:',
        'fr': ':drapeau_oman:',
        'ja': ':旗_オマーン:',
        'ko': ':깃발_오만:',
        'pt': ':bandeira_omã:',
        'it': ':bandiera_oman:',
        'fa': ':پرچم_عمان:',
        'id': ':bendera_oman:',
        'zh': ':阿曼:',
        'ru': ':флаг_Оман:'
    },
    '\U000026CE': {  # ⛎
        'en': ':Ophiuchus:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':ophiuchus:'],
        'de': ':schlangenträger:',
        'es': ':ofiuco:',
        'fr': ':serpentaire:',
        'ja': ':へびつかい座:',
        'ko': ':뱀주인자리:',
        'pt': ':signo_de_ofiúco:',
        'it': ':segno_zodiacale_dell’ofiuco:',
        'fa': ':برج_حوا:',
        'id': ':ophiuchus:',
        'zh': ':蛇夫座:',
        'ru': ':знак_зодиака_Змееносец:'
    },
    '\U0001F17F\U0000FE0F': {  # 🅿️
        'en': ':P_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':parking:', ':p_button:'],
        'variant': True,
        'de': ':großbuchstabe_p_in_blauem_quadrat:',
        'es': ':aparcamiento:',
        'fr': ':bouton_p:',
        'ja': ':pマーク:',
        'ko': ':주차_표시:',
        'pt': ':botão_p:',
        'it': ':pulsante_p:',
        'fa': ':دکمه_پی:',
        'id': ':tombol_p:',
        'zh': ':停车按钮:',
        'ru': ':парковка:'
    },
    '\U0001F17F': {  # 🅿
        'en': ':P_button:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':parking:', ':p_button:'],
        'variant': True,
        'de': ':großbuchstabe_p_in_blauem_quadrat:',
        'es': ':aparcamiento:',
        'fr': ':bouton_p:',
        'ja': ':pマーク:',
        'ko': ':주차_표시:',
        'pt': ':botão_p:',
        'it': ':pulsante_p:',
        'fa': ':دکمه_پی:',
        'id': ':tombol_p:',
        'zh': ':停车按钮:',
        'ru': ':парковка:'
    },
    '\U0001F1F5\U0001F1F0': {  # 🇵🇰
        'en': ':Pakistan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Pakistan:', ':pakistan:'],
        'de': ':flagge_pakistan:',
        'es': ':bandera_pakistán:',
        'fr': ':drapeau_pakistan:',
        'ja': ':旗_パキスタン:',
        'ko': ':깃발_파키스탄:',
        'pt': ':bandeira_paquistão:',
        'it': ':bandiera_pakistan:',
        'fa': ':پرچم_پاکستان:',
        'id': ':bendera_pakistan:',
        'zh': ':巴基斯坦:',
        'ru': ':флаг_Пакистан:'
    },
    '\U0001F1F5\U0001F1FC': {  # 🇵🇼
        'en': ':Palau:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Palau:', ':palau:'],
        'de': ':flagge_palau:',
        'es': ':bandera_palaos:',
        'fr': ':drapeau_palaos:',
        'ja': ':旗_パラオ:',
        'ko': ':깃발_팔라우:',
        'pt': ':bandeira_palau:',
        'it': ':bandiera_palau:',
        'fa': ':پرچم_پالائو:',
        'id': ':bendera_palau:',
        'zh': ':帕劳:',
        'ru': ':флаг_Палау:'
    },
    '\U0001F1F5\U0001F1F8': {  # 🇵🇸
        'en': ':Palestinian_Territories:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Palestinian_Territories:', ':palestinian_territories:'],
        'de': ':flagge_palästinensische_autonomiegebiete:',
        'es': ':bandera_territorios_palestinos:',
        'fr': ':drapeau_territoires_palestiniens:',
        'ja': ':旗_パレスチナ自治区:',
        'ko': ':깃발_팔레스타인_지구:',
        'pt': ':bandeira_territórios_palestinos:',
        'it': ':bandiera_territori_palestinesi:',
        'fa': ':پرچم_سرزمین_های_فلسطینی:',
        'id': ':bendera_wilayah_palestina:',
        'zh': ':巴勒斯坦领土:',
        'ru': ':флаг_Палестинские_территории:'
    },
    '\U0001F1F5\U0001F1E6': {  # 🇵🇦
        'en': ':Panama:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Panama:', ':panama:'],
        'de': ':flagge_panama:',
        'es': ':bandera_panamá:',
        'fr': ':drapeau_panama:',
        'ja': ':旗_パナマ:',
        'ko': ':깃발_파나마:',
        'pt': ':bandeira_panamá:',
        'it': ':bandiera_panama:',
        'fa': ':پرچم_پاناما:',
        'id': ':bendera_panama:',
        'zh': ':巴拿马:',
        'ru': ':флаг_Панама:'
    },
    '\U0001F1F5\U0001F1EC': {  # 🇵🇬
        'en': ':Papua_New_Guinea:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Papua_New_Guinea:', ':papua_new_guinea:'],
        'de': ':flagge_papua-neuguinea:',
        'es': ':bandera_papúa_nueva_guinea:',
        'fr': ':drapeau_papouasie-nouvelle-guinée:',
        'ja': ':旗_パプアニューギニア:',
        'ko': ':깃발_파푸아뉴기니:',
        'pt': ':bandeira_papua-nova_guiné:',
        'it': ':bandiera_papua_nuova_guinea:',
        'fa': ':پرچم_پاپوا_گینهٔ_نو:',
        'id': ':bendera_papua_nugini:',
        'zh': ':巴布亚新几内亚:',
        'ru': ':флаг_Папуа_Новая_Гвинея:'
    },
    '\U0001F1F5\U0001F1FE': {  # 🇵🇾
        'en': ':Paraguay:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Paraguay:', ':paraguay:'],
        'de': ':flagge_paraguay:',
        'es': ':bandera_paraguay:',
        'fr': ':drapeau_paraguay:',
        'ja': ':旗_パラグアイ:',
        'ko': ':깃발_파라과이:',
        'pt': ':bandeira_paraguai:',
        'it': ':bandiera_paraguay:',
        'fa': ':پرچم_پاراگوئه:',
        'id': ':bendera_paraguay:',
        'zh': ':巴拉圭:',
        'ru': ':флаг_Парагвай:'
    },
    '\U0001F1F5\U0001F1EA': {  # 🇵🇪
        'en': ':Peru:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Peru:', ':peru:'],
        'de': ':flagge_peru:',
        'es': ':bandera_perú:',
        'fr': ':drapeau_pérou:',
        'ja': ':旗_ペルー:',
        'ko': ':깃발_페루:',
        'pt': ':bandeira_peru:',
        'it': ':bandiera_perù:',
        'fa': ':پرچم_پرو:',
        'id': ':bendera_peru:',
        'zh': ':秘鲁:',
        'ru': ':флаг_Перу:'
    },
    '\U0001F1F5\U0001F1ED': {  # 🇵🇭
        'en': ':Philippines:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Philippines:', ':philippines:'],
        'de': ':flagge_philippinen:',
        'es': ':bandera_filipinas:',
        'fr': ':drapeau_philippines:',
        'ja': ':旗_フィリピン:',
        'ko': ':깃발_필리핀:',
        'pt': ':bandeira_filipinas:',
        'it': ':bandiera_filippine:',
        'fa': ':پرچم_فیلیپین:',
        'id': ':bendera_filipina:',
        'zh': ':菲律宾:',
        'ru': ':флаг_Филиппины:'
    },
    '\U00002653': {  # ♓
        'en': ':Pisces:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':pisces:'],
        'variant': True,
        'de': ':fische_(sternzeichen):',
        'es': ':piscis:',
        'fr': ':poissons:',
        'ja': ':うお座:',
        'ko': ':물고기자리:',
        'pt': ':signo_de_peixes:',
        'it': ':segno_zodiacale_dei_pesci:',
        'fa': ':برج_حوت:',
        'id': ':pisces:',
        'zh': ':双鱼座:',
        'ru': ':знак_зодиака_Рыбы:'
    },
    '\U0001F1F5\U0001F1F3': {  # 🇵🇳
        'en': ':Pitcairn_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Pitcairn_Islands:', ':pitcairn_islands:'],
        'de': ':flagge_pitcairninseln:',
        'es': ':bandera_islas_pitcairn:',
        'fr': ':drapeau_îles_pitcairn:',
        'ja': ':旗_ピトケアン諸島:',
        'ko': ':깃발_핏케언_제도:',
        'pt': ':bandeira_ilhas_pitcairn:',
        'it': ':bandiera_isole_pitcairn:',
        'fa': ':پرچم_جزایر_پیت_کرن:',
        'id': ':bendera_kepulauan_pitcairn:',
        'zh': ':皮特凯恩群岛:',
        'ru': ':флаг_о_ва_Питкэрн:'
    },
    '\U0001F1F5\U0001F1F1': {  # 🇵🇱
        'en': ':Poland:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Poland:', ':poland:'],
        'de': ':flagge_polen:',
        'es': ':bandera_polonia:',
        'fr': ':drapeau_pologne:',
        'ja': ':旗_ポーランド:',
        'ko': ':깃발_폴란드:',
        'pt': ':bandeira_polônia:',
        'it': ':bandiera_polonia:',
        'fa': ':پرچم_لهستان:',
        'id': ':bendera_polandia:',
        'zh': ':波兰:',
        'ru': ':флаг_Польша:'
    },
    '\U0001F1F5\U0001F1F9': {  # 🇵🇹
        'en': ':Portugal:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Portugal:', ':portugal:'],
        'de': ':flagge_portugal:',
        'es': ':bandera_portugal:',
        'fr': ':drapeau_portugal:',
        'ja': ':旗_ポルトガル:',
        'ko': ':깃발_포르투갈:',
        'pt': ':bandeira_portugal:',
        'it': ':bandiera_portogallo:',
        'fa': ':پرچم_پرتغال:',
        'id': ':bendera_portugal:',
        'zh': ':葡萄牙:',
        'ru': ':флаг_Португалия:'
    },
    '\U0001F1F5\U0001F1F7': {  # 🇵🇷
        'en': ':Puerto_Rico:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Puerto_Rico:', ':puerto_rico:'],
        'de': ':flagge_puerto_rico:',
        'es': ':bandera_puerto_rico:',
        'fr': ':drapeau_porto_rico:',
        'ja': ':旗_プエルトリコ:',
        'ko': ':깃발_푸에르토리코:',
        'pt': ':bandeira_porto_rico:',
        'it': ':bandiera_portorico:',
        'fa': ':پرچم_پورتوریکو:',
        'id': ':bendera_puerto_riko:',
        'zh': ':波多黎各:',
        'ru': ':флаг_Пуэрто_Рико:'
    },
    '\U0001F1F6\U0001F1E6': {  # 🇶🇦
        'en': ':Qatar:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Qatar:', ':qatar:'],
        'de': ':flagge_katar:',
        'es': ':bandera_catar:',
        'fr': ':drapeau_qatar:',
        'ja': ':旗_カタール:',
        'ko': ':깃발_카타르:',
        'pt': ':bandeira_catar:',
        'it': ':bandiera_qatar:',
        'fa': ':پرچم_قطر:',
        'id': ':bendera_qatar:',
        'zh': ':卡塔尔:',
        'ru': ':флаг_Катар:'
    },
    '\U0001F1F7\U0001F1F4': {  # 🇷🇴
        'en': ':Romania:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Romania:', ':romania:'],
        'de': ':flagge_rumänien:',
        'es': ':bandera_rumanía:',
        'fr': ':drapeau_roumanie:',
        'ja': ':旗_ルーマニア:',
        'ko': ':깃발_루마니아:',
        'pt': ':bandeira_romênia:',
        'it': ':bandiera_romania:',
        'fa': ':پرچم_رومانی:',
        'id': ':bendera_rumania:',
        'zh': ':罗马尼亚:',
        'ru': ':флаг_Румыния:'
    },
    '\U0001F1F7\U0001F1FA': {  # 🇷🇺
        'en': ':Russia:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_Russia:', ':ru:'],
        'de': ':flagge_russland:',
        'es': ':bandera_rusia:',
        'fr': ':drapeau_russie:',
        'ja': ':旗_ロシア:',
        'ko': ':깃발_러시아:',
        'pt': ':bandeira_rússia:',
        'it': ':bandiera_russia:',
        'fa': ':پرچم_روسیه:',
        'id': ':bendera_rusia:',
        'zh': ':俄罗斯:',
        'ru': ':флаг_Россия:'
    },
    '\U0001F1F7\U0001F1FC': {  # 🇷🇼
        'en': ':Rwanda:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Rwanda:', ':rwanda:'],
        'de': ':flagge_ruanda:',
        'es': ':bandera_ruanda:',
        'fr': ':drapeau_rwanda:',
        'ja': ':旗_ルワンダ:',
        'ko': ':깃발_르완다:',
        'pt': ':bandeira_ruanda:',
        'it': ':bandiera_ruanda:',
        'fa': ':پرچم_رواندا:',
        'id': ':bendera_rwanda:',
        'zh': ':卢旺达:',
        'ru': ':флаг_Руанда:'
    },
    '\U0001F1F7\U0001F1EA': {  # 🇷🇪
        'en': ':Réunion:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Réunion:', ':reunion:'],
        'de': ':flagge_réunion:',
        'es': ':bandera_reunión:',
        'fr': ':drapeau_la_réunion:',
        'ja': ':旗_レユニオン:',
        'ko': ':깃발_레위니옹:',
        'pt': ':bandeira_reunião:',
        'it': ':bandiera_riunione:',
        'fa': ':پرچم_رئونیون:',
        'id': ':bendera_réunion:',
        'zh': ':留尼汪:',
        'ru': ':флаг_Реюньон:'
    },
    '\U0001F51C': {  # 🔜
        'en': ':SOON_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':soon:', ':soon_arrow:'],
        'de': ':soon-pfeil:',
        'es': ':flecha_soon:',
        'fr': ':flèche_bientôt:',
        'ja': ':soon矢印:',
        'ko': ':곧:',
        'pt': ':seta_soon:',
        'it': ':freccia_soon:',
        'fa': ':پیکان_به_زودی:',
        'id': ':tanda_panah_soon:',
        'zh': ':SOON箭头:',
        'ru': ':стрелка_«скоро»:'
    },
    '\U0001F198': {  # 🆘
        'en': ':SOS_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':sos:', ':sos_button:'],
        'de': ':sos-zeichen:',
        'es': ':símbolo_de_socorro:',
        'fr': ':bouton_sos:',
        'ja': ':sosマーク:',
        'ko': ':에스오에스:',
        'pt': ':botão_sos:',
        'it': ':pulsante_sos:',
        'fa': ':دکمه_اس_او_اس:',
        'id': ':tombol_sos:',
        'zh': ':SOS按钮:',
        'ru': ':значок_SOS:'
    },
    '\U00002650': {  # ♐
        'en': ':Sagittarius:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':sagittarius:'],
        'variant': True,
        'de': ':schütze_(sternzeichen):',
        'es': ':sagitario:',
        'fr': ':sagittaire:',
        'ja': ':いて座:',
        'ko': ':사수자리:',
        'pt': ':signo_de_sagitário:',
        'it': ':segno_zodiacale_del_saggitario:',
        'fa': ':برج_قوس:',
        'id': ':sagitarius:',
        'zh': ':射手座:',
        'ru': ':знак_зодиака_Стрелец:'
    },
    '\U0001F1FC\U0001F1F8': {  # 🇼🇸
        'en': ':Samoa:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Samoa:', ':samoa:'],
        'de': ':flagge_samoa:',
        'es': ':bandera_samoa:',
        'fr': ':drapeau_samoa:',
        'ja': ':旗_サモア:',
        'ko': ':깃발_사모아:',
        'pt': ':bandeira_samoa:',
        'it': ':bandiera_samoa:',
        'fa': ':پرچم_ساموآ:',
        'id': ':bendera_samoa:',
        'zh': ':萨摩亚:',
        'ru': ':флаг_Самоа:'
    },
    '\U0001F1F8\U0001F1F2': {  # 🇸🇲
        'en': ':San_Marino:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_San_Marino:', ':san_marino:'],
        'de': ':flagge_san_marino:',
        'es': ':bandera_san_marino:',
        'fr': ':drapeau_saint-marin:',
        'ja': ':旗_サンマリノ:',
        'ko': ':깃발_산마리노:',
        'pt': ':bandeira_san_marino:',
        'it': ':bandiera_san_marino:',
        'fa': ':پرچم_سان_مارینو:',
        'id': ':bendera_san_marino:',
        'zh': ':圣马力诺:',
        'ru': ':флаг_Сан_Марино:'
    },
    '\U0001F385': {  # 🎅
        'en': ':Santa_Claus:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':santa:', ':santa_claus:'],
        'de': ':weihnachtsmann:',
        'es': ':papá_noel:',
        'fr': ':père_noël:',
        'ja': ':サンタ:',
        'ko': ':산타클로스:',
        'pt': ':papai_noel:',
        'it': ':babbo_natale:',
        'fa': ':بابا_نوئل:',
        'id': ':sinterklas:',
        'zh': ':圣诞老人:',
        'ru': ':Санта_Клаус:'
    },
    '\U0001F385\U0001F3FF': {  # 🎅🏿
        'en': ':Santa_Claus_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':weihnachtsmann_dunkle_hautfarbe:',
        'es': ':papá_noel_tono_de_piel_oscuro:',
        'fr': ':père_noël_peau_foncée:',
        'ja': ':サンタ_濃い肌色:',
        'ko': ':산타클로스_검은색_피부:',
        'pt': ':papai_noel_pele_escura:',
        'it': ':babbo_natale_carnagione_scura:',
        'fa': ':بابا_نوئل_پوست_آبنوسی:',
        'id': ':sinterklas_warna_kulit_gelap:',
        'zh': ':圣诞老人_较深肤色:',
        'ru': ':Санта_Клаус_очень_темный_тон_кожи:'
    },
    '\U0001F385\U0001F3FB': {  # 🎅🏻
        'en': ':Santa_Claus_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':weihnachtsmann_helle_hautfarbe:',
        'es': ':papá_noel_tono_de_piel_claro:',
        'fr': ':père_noël_peau_claire:',
        'ja': ':サンタ_薄い肌色:',
        'ko': ':산타클로스_하얀_피부:',
        'pt': ':papai_noel_pele_clara:',
        'it': ':babbo_natale_carnagione_chiara:',
        'fa': ':بابا_نوئل_پوست_سفید:',
        'id': ':sinterklas_warna_kulit_cerah:',
        'zh': ':圣诞老人_较浅肤色:',
        'ru': ':Санта_Клаус_очень_светлый_тон_кожи:'
    },
    '\U0001F385\U0001F3FE': {  # 🎅🏾
        'en': ':Santa_Claus_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':weihnachtsmann_mitteldunkle_hautfarbe:',
        'es': ':papá_noel_tono_de_piel_oscuro_medio:',
        'fr': ':père_noël_peau_mate:',
        'ja': ':サンタ_やや濃い肌色:',
        'ko': ':산타클로스_진한_갈색_피부:',
        'pt': ':papai_noel_pele_morena_escura:',
        'it': ':babbo_natale_carnagione_abbastanza_scura:',
        'fa': ':بابا_نوئل_پوست_گندمی:',
        'id': ':sinterklas_warna_kulit_gelap-sedang:',
        'zh': ':圣诞老人_中等深肤色:',
        'ru': ':Санта_Клаус_темный_тон_кожи:'
    },
    '\U0001F385\U0001F3FC': {  # 🎅🏼
        'en': ':Santa_Claus_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':weihnachtsmann_mittelhelle_hautfarbe:',
        'es': ':papá_noel_tono_de_piel_claro_medio:',
        'fr': ':père_noël_peau_moyennement_claire:',
        'ja': ':サンタ_やや薄い肌色:',
        'ko': ':산타클로스_연한_갈색_피부:',
        'pt': ':papai_noel_pele_morena_clara:',
        'it': ':babbo_natale_carnagione_abbastanza_chiara:',
        'fa': ':بابا_نوئل_پوست_روشن:',
        'id': ':sinterklas_warna_kulit_cerah-sedang:',
        'zh': ':圣诞老人_中等浅肤色:',
        'ru': ':Санта_Клаус_светлый_тон_кожи:'
    },
    '\U0001F385\U0001F3FD': {  # 🎅🏽
        'en': ':Santa_Claus_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':weihnachtsmann_mittlere_hautfarbe:',
        'es': ':papá_noel_tono_de_piel_medio:',
        'fr': ':père_noël_peau_légèrement_mate:',
        'ja': ':サンタ_中間の肌色:',
        'ko': ':산타클로스_갈색_피부:',
        'pt': ':papai_noel_pele_morena:',
        'it': ':babbo_natale_carnagione_olivastra:',
        'fa': ':بابا_نوئل_پوست_طلایی:',
        'id': ':sinterklas_warna_kulit_sedang:',
        'zh': ':圣诞老人_中等肤色:',
        'ru': ':Санта_Клаус_средний_тон_кожи:'
    },
    '\U0001F1F8\U0001F1E6': {  # 🇸🇦
        'en': ':Saudi_Arabia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Saudi_Arabia:', ':saudi_arabia:'],
        'de': ':flagge_saudi-arabien:',
        'es': ':bandera_arabia_saudí:',
        'fr': ':drapeau_arabie_saoudite:',
        'ja': ':旗_サウジアラビア:',
        'ko': ':깃발_사우디아라비아:',
        'pt': ':bandeira_arábia_saudita:',
        'it': ':bandiera_arabia_saudita:',
        'fa': ':پرچم_عربستان_سعودی:',
        'id': ':bendera_arab_saudi:',
        'zh': ':沙特阿拉伯:',
        'ru': ':флаг_Саудовская_Аравия:'
    },
    '\U0000264F': {  # ♏
        'en': ':Scorpio:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':scorpius:', ':scorpio:'],
        'variant': True,
        'de': ':skorpion_(sternzeichen):',
        'es': ':escorpio:',
        'fr': ':scorpion_zodiaque:',
        'ja': ':さそり座:',
        'ko': ':전갈자리:',
        'pt': ':signo_de_escorpião:',
        'it': ':segno_zodiacale_dello_scorpione:',
        'fa': ':برج_عقرب:',
        'id': ':scorpio:',
        'zh': ':天蝎座:',
        'ru': ':знак_зодиака_Скорпион:'
    },
    '\U0001F3F4\U000E0067\U000E0062\U000E0073\U000E0063\U000E0074\U000E007F': {  # 🏴󠁧󠁢󠁳󠁣󠁴󠁿
        'en': ':Scotland:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':scotland:'],
        'de': ':flagge_schottland:',
        'es': ':bandera_escocia:',
        'fr': ':drapeau_écosse:',
        'ja': ':旗_スコットランド:',
        'ko': ':깃발_스코틀랜드:',
        'pt': ':bandeira_escócia:',
        'it': ':bandiera_scozia:',
        'fa': ':پرچم_اسکاتلند:',
        'id': ':bendera_skotlandia:',
        'zh': ':苏格兰:',
        'ru': ':флаг_Шотландия:'
    },
    '\U0001F1F8\U0001F1F3': {  # 🇸🇳
        'en': ':Senegal:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Senegal:', ':senegal:'],
        'de': ':flagge_senegal:',
        'es': ':bandera_senegal:',
        'fr': ':drapeau_sénégal:',
        'ja': ':旗_セネガル:',
        'ko': ':깃발_세네갈:',
        'pt': ':bandeira_senegal:',
        'it': ':bandiera_senegal:',
        'fa': ':پرچم_سنگال:',
        'id': ':bendera_senegal:',
        'zh': ':塞内加尔:',
        'ru': ':флаг_Сенегал:'
    },
    '\U0001F1F7\U0001F1F8': {  # 🇷🇸
        'en': ':Serbia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Serbia:', ':serbia:'],
        'de': ':flagge_serbien:',
        'es': ':bandera_serbia:',
        'fr': ':drapeau_serbie:',
        'ja': ':旗_セルビア:',
        'ko': ':깃발_세르비아:',
        'pt': ':bandeira_sérvia:',
        'it': ':bandiera_serbia:',
        'fa': ':پرچم_صربستان:',
        'id': ':bendera_serbia:',
        'zh': ':塞尔维亚:',
        'ru': ':флаг_Сербия:'
    },
    '\U0001F1F8\U0001F1E8': {  # 🇸🇨
        'en': ':Seychelles:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Seychelles:', ':seychelles:'],
        'de': ':flagge_seychellen:',
        'es': ':bandera_seychelles:',
        'fr': ':drapeau_seychelles:',
        'ja': ':旗_セーシェル:',
        'ko': ':깃발_세이셸:',
        'pt': ':bandeira_seicheles:',
        'it': ':bandiera_seychelles:',
        'fa': ':پرچم_سیشل:',
        'id': ':bendera_seychelles:',
        'zh': ':塞舌尔:',
        'ru': ':флаг_Сейшельские_о_ва:'
    },
    '\U0001F1F8\U0001F1F1': {  # 🇸🇱
        'en': ':Sierra_Leone:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Sierra_Leone:', ':sierra_leone:'],
        'de': ':flagge_sierra_leone:',
        'es': ':bandera_sierra_leona:',
        'fr': ':drapeau_sierra_leone:',
        'ja': ':旗_シエラレオネ:',
        'ko': ':깃발_시에라리온:',
        'pt': ':bandeira_serra_leoa:',
        'it': ':bandiera_sierra_leone:',
        'fa': ':پرچم_سیرالئون:',
        'id': ':bendera_sierra_leone:',
        'zh': ':塞拉利昂:',
        'ru': ':флаг_Сьерра_Леоне:'
    },
    '\U0001F1F8\U0001F1EC': {  # 🇸🇬
        'en': ':Singapore:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Singapore:', ':singapore:'],
        'de': ':flagge_singapur:',
        'es': ':bandera_singapur:',
        'fr': ':drapeau_singapour:',
        'ja': ':旗_シンガポール:',
        'ko': ':깃발_싱가포르:',
        'pt': ':bandeira_singapura:',
        'it': ':bandiera_singapore:',
        'fa': ':پرچم_سنگاپور:',
        'id': ':bendera_singapura:',
        'zh': ':新加坡:',
        'ru': ':флаг_Сингапур:'
    },
    '\U0001F1F8\U0001F1FD': {  # 🇸🇽
        'en': ':Sint_Maarten:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Sint_Maarten:', ':sint_maarten:'],
        'de': ':flagge_sint_maarten:',
        'es': ':bandera_sint_maarten:',
        'fr': ':drapeau_saint-martin_(partie_néerlandaise):',
        'ja': ':旗_シント・マールテン:',
        'ko': ':깃발_신트마르턴:',
        'pt': ':bandeira_sint_maarten:',
        'it': ':bandiera_sint_maarten:',
        'fa': ':پرچم_سنت_مارتن:',
        'id': ':bendera_sint_maarten:',
        'zh': ':荷属圣马丁:',
        'ru': ':флаг_Синт_Мартен:'
    },
    '\U0001F1F8\U0001F1F0': {  # 🇸🇰
        'en': ':Slovakia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Slovakia:', ':slovakia:'],
        'de': ':flagge_slowakei:',
        'es': ':bandera_eslovaquia:',
        'fr': ':drapeau_slovaquie:',
        'ja': ':旗_スロバキア:',
        'ko': ':깃발_슬로바키아:',
        'pt': ':bandeira_eslováquia:',
        'it': ':bandiera_slovacchia:',
        'fa': ':پرچم_اسلواکی:',
        'id': ':bendera_slovakia:',
        'zh': ':斯洛伐克:',
        'ru': ':флаг_Словакия:'
    },
    '\U0001F1F8\U0001F1EE': {  # 🇸🇮
        'en': ':Slovenia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Slovenia:', ':slovenia:'],
        'de': ':flagge_slowenien:',
        'es': ':bandera_eslovenia:',
        'fr': ':drapeau_slovénie:',
        'ja': ':旗_スロベニア:',
        'ko': ':깃발_슬로베니아:',
        'pt': ':bandeira_eslovênia:',
        'it': ':bandiera_slovenia:',
        'fa': ':پرچم_اسلوونی:',
        'id': ':bendera_slovenia:',
        'zh': ':斯洛文尼亚:',
        'ru': ':флаг_Словения:'
    },
    '\U0001F1F8\U0001F1E7': {  # 🇸🇧
        'en': ':Solomon_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Solomon_Islands:', ':solomon_islands:'],
        'de': ':flagge_salomonen:',
        'es': ':bandera_islas_salomón:',
        'fr': ':drapeau_îles_salomon:',
        'ja': ':旗_ソロモン諸島:',
        'ko': ':깃발_솔로몬_제도:',
        'pt': ':bandeira_ilhas_salomão:',
        'it': ':bandiera_isole_salomone:',
        'fa': ':پرچم_جزایر_سلیمان:',
        'id': ':bendera_kepulauan_solomon:',
        'zh': ':所罗门群岛:',
        'ru': ':флаг_Соломоновы_о_ва:'
    },
    '\U0001F1F8\U0001F1F4': {  # 🇸🇴
        'en': ':Somalia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Somalia:', ':somalia:'],
        'de': ':flagge_somalia:',
        'es': ':bandera_somalia:',
        'fr': ':drapeau_somalie:',
        'ja': ':旗_ソマリア:',
        'ko': ':깃발_소말리아:',
        'pt': ':bandeira_somália:',
        'it': ':bandiera_somalia:',
        'fa': ':پرچم_سومالی:',
        'id': ':bendera_somalia:',
        'zh': ':索马里:',
        'ru': ':флаг_Сомали:'
    },
    '\U0001F1FF\U0001F1E6': {  # 🇿🇦
        'en': ':South_Africa:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_South_Africa:', ':south_africa:'],
        'de': ':flagge_südafrika:',
        'es': ':bandera_sudáfrica:',
        'fr': ':drapeau_afrique_du_sud:',
        'ja': ':旗_南アフリカ:',
        'ko': ':깃발_남아프리카:',
        'pt': ':bandeira_áfrica_do_sul:',
        'it': ':bandiera_sudafrica:',
        'fa': ':پرچم_افریقای_جنوبی:',
        'id': ':bendera_afrika_selatan:',
        'zh': ':南非:',
        'ru': ':флаг_Южно_Африканская_Республика:'
    },
    '\U0001F1EC\U0001F1F8': {  # 🇬🇸
        'en': ':South_Georgia_&_South_Sandwich_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_South_Georgia_&_South_Sandwich_Islands:', ':south_georgia_south_sandwich_islands:'],
        'de': ':flagge_südgeorgien_und_die_südlichen_sandwichinseln:',
        'es': ':bandera_islas_georgia_del_sur_y_sandwich_del_sur:',
        'fr': ':drapeau_géorgie_du_sud-et-les_îles_sandwich_du_sud:',
        'ja': ':旗_サウスジョージア・サウスサンドウィッチ諸島:',
        'ko': ':깃발_사우스조지아_사우스샌드위치_제도:',
        'pt': ':bandeira_ilhas_geórgia_do_sul_e_sandwich_do_sul:',
        'it': ':bandiera_georgia_del_sud_e_sandwich_australi:',
        'fa': ':پرچم_جورجیای_جنوبی_و_جزایر_ساندویچ_جنوبی:',
        'id': ':bendera_georgia_selatan_&_kep._sandwich_selatan:',
        'zh': ':南乔治亚和南桑威奇群岛:',
        'ru': ':флаг_Южная_Георгия_и_Южные_Сандвичевы_о_ва:'
    },
    '\U0001F1F0\U0001F1F7': {  # 🇰🇷
        'en': ':South_Korea:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_South_Korea:', ':kr:'],
        'de': ':flagge_südkorea:',
        'es': ':bandera_corea_del_sur:',
        'fr': ':drapeau_corée_du_sud:',
        'ja': ':旗_韓国:',
        'ko': ':깃발_대한민국:',
        'pt': ':bandeira_coreia_do_sul:',
        'it': ':bandiera_corea_del_sud:',
        'fa': ':پرچم_کرهٔ_جنوبی:',
        'id': ':bendera_korea_selatan:',
        'zh': ':韩国:',
        'ru': ':флаг_Республика_Корея:'
    },
    '\U0001F1F8\U0001F1F8': {  # 🇸🇸
        'en': ':South_Sudan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_South_Sudan:', ':south_sudan:'],
        'de': ':flagge_südsudan:',
        'es': ':bandera_sudán_del_sur:',
        'fr': ':drapeau_soudan_du_sud:',
        'ja': ':旗_南スーダン:',
        'ko': ':깃발_남수단:',
        'pt': ':bandeira_sudão_do_sul:',
        'it': ':bandiera_sud_sudan:',
        'fa': ':پرچم_سودان_جنوبی:',
        'id': ':bendera_sudan_selatan:',
        'zh': ':南苏丹:',
        'ru': ':флаг_Южный_Судан:'
    },
    '\U0001F1EA\U0001F1F8': {  # 🇪🇸
        'en': ':Spain:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_Spain:', ':es:'],
        'de': ':flagge_spanien:',
        'es': ':bandera_españa:',
        'fr': ':drapeau_espagne:',
        'ja': ':旗_スペイン:',
        'ko': ':깃발_스페인:',
        'pt': ':bandeira_espanha:',
        'it': ':bandiera_spagna:',
        'fa': ':پرچم_اسپانیا:',
        'id': ':bendera_spanyol:',
        'zh': ':西班牙:',
        'ru': ':флаг_Испания:'
    },
    '\U0001F1F1\U0001F1F0': {  # 🇱🇰
        'en': ':Sri_Lanka:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Sri_Lanka:', ':sri_lanka:'],
        'de': ':flagge_sri_lanka:',
        'es': ':bandera_sri_lanka:',
        'fr': ':drapeau_sri_lanka:',
        'ja': ':旗_スリランカ:',
        'ko': ':깃발_스리랑카:',
        'pt': ':bandeira_sri_lanka:',
        'it': ':bandiera_sri_lanka:',
        'fa': ':پرچم_سری_لانکا:',
        'id': ':bendera_sri_lanka:',
        'zh': ':斯里兰卡:',
        'ru': ':флаг_Шри_Ланка:'
    },
    '\U0001F1E7\U0001F1F1': {  # 🇧🇱
        'en': ':St._Barthélemy:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_St._Barthélemy:', ':st_barthelemy:'],
        'de': ':flagge_st._barthélemy:',
        'es': ':bandera_san_bartolomé:',
        'fr': ':drapeau_saint-barthélemy:',
        'ja': ':旗_サン・バルテルミー:',
        'ko': ':깃발_생바르텔레미:',
        'pt': ':bandeira_são_bartolomeu:',
        'it': ':bandiera_saint-barthélemy:',
        'fa': ':پرچم_سن_بارتلمی:',
        'id': ':bendera_saint_barthélemy:',
        'zh': ':圣巴泰勒米:',
        'ru': ':флаг_Сен_Бартелеми:'
    },
    '\U0001F1F8\U0001F1ED': {  # 🇸🇭
        'en': ':St._Helena:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_St._Helena:', ':st_helena:'],
        'de': ':flagge_st._helena:',
        'es': ':bandera_santa_elena:',
        'fr': ':drapeau_sainte-hélène:',
        'ja': ':旗_セントヘレナ:',
        'ko': ':깃발_세인트헬레나:',
        'pt': ':bandeira_santa_helena:',
        'it': ':bandiera_sant’elena:',
        'fa': ':پرچم_سنت_هلن:',
        'id': ':bendera_saint_helena:',
        'zh': ':圣赫勒拿:',
        'ru': ':флаг_о_в_Св._Елены:'
    },
    '\U0001F1F0\U0001F1F3': {  # 🇰🇳
        'en': ':St._Kitts_&_Nevis:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_St._Kitts_&_Nevis:', ':st_kitts_nevis:'],
        'de': ':flagge_st._kitts_und_nevis:',
        'es': ':bandera_san_cristóbal_y_nieves:',
        'fr': ':drapeau_saint-christophe-et-niévès:',
        'ja': ':旗_セントクリストファー・ネーヴィス:',
        'ko': ':깃발_세인트키츠_네비스:',
        'pt': ':bandeira_são_cristóvão_e_névis:',
        'it': ':bandiera_saint_kitts_e_nevis:',
        'fa': ':پرچم_سنت_کیتس_و_نویس:',
        'id': ':bendera_saint_kitts_dan_nevis:',
        'zh': ':圣基茨和尼维斯:',
        'ru': ':флаг_Сент_Китс_и_Невис:'
    },
    '\U0001F1F1\U0001F1E8': {  # 🇱🇨
        'en': ':St._Lucia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_St._Lucia:', ':st_lucia:'],
        'de': ':flagge_st._lucia:',
        'es': ':bandera_santa_lucía:',
        'fr': ':drapeau_sainte-lucie:',
        'ja': ':旗_セントルシア:',
        'ko': ':깃발_세인트루시아:',
        'pt': ':bandeira_santa_lúcia:',
        'it': ':bandiera_saint_lucia:',
        'fa': ':پرچم_سنت_لوسیا:',
        'id': ':bendera_saint_lucia:',
        'zh': ':圣卢西亚:',
        'ru': ':флаг_Сент_Люсия:'
    },
    '\U0001F1F2\U0001F1EB': {  # 🇲🇫
        'en': ':St._Martin:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_St._Martin:', ':st_martin:'],
        'de': ':flagge_st._martin:',
        'es': ':bandera_san_martín:',
        'fr': ':drapeau_saint-martin:',
        'ja': ':旗_サン・マルタン:',
        'ko': ':깃발_생마르탱:',
        'pt': ':bandeira_são_martinho:',
        'it': ':bandiera_saint_martin:',
        'fa': ':پرچم_سنت_مارتین:',
        'id': ':bendera_saint_martin:',
        'zh': ':法属圣马丁:',
        'ru': ':флаг_Сен_Мартен:'
    },
    '\U0001F1F5\U0001F1F2': {  # 🇵🇲
        'en': ':St._Pierre_&_Miquelon:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_St._Pierre_&_Miquelon:', ':st_pierre_miquelon:'],
        'de': ':flagge_st._pierre_und_miquelon:',
        'es': ':bandera_san_pedro_y_miquelón:',
        'fr': ':drapeau_saint-pierre-et-miquelon:',
        'ja': ':旗_サンピエール島・ミクロン島:',
        'ko': ':깃발_생피에르_미클롱:',
        'pt': ':bandeira_são_pedro_e_miquelão:',
        'it': ':bandiera_saint-pierre_e_miquelon:',
        'fa': ':پرچم_سن_پیر_و_میکلن:',
        'id': ':bendera_saint_pierre_dan_miquelon:',
        'zh': ':圣皮埃尔和密克隆群岛:',
        'ru': ':флаг_Сен_Пьер_и_Микелон:'
    },
    '\U0001F1FB\U0001F1E8': {  # 🇻🇨
        'en': ':St._Vincent_&_Grenadines:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_St._Vincent_&_Grenadines:', ':st_vincent_grenadines:'],
        'de': ':flagge_st._vincent_und_die_grenadinen:',
        'es': ':bandera_san_vicente_y_las_granadinas:',
        'fr': ':drapeau_saint-vincent-et-les_grenadines:',
        'ja': ':旗_セントビンセント及びグレナディーン諸島:',
        'ko': ':깃발_세인트빈센트그레나딘:',
        'pt': ':bandeira_são_vicente_e_granadinas:',
        'it': ':bandiera_saint_vincent_e_grenadine:',
        'fa': ':پرچم_سنت_وینسنت_و_گرنادین:',
        'id': ':bendera_saint_vincent_dan_grenadine:',
        'zh': ':圣文森特和格林纳丁斯:',
        'ru': ':флаг_Сент_Винсент_и_Гренадины:'
    },
    '\U0001F5FD': {  # 🗽
        'en': ':Statue_of_Liberty:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':statue_of_liberty:'],
        'de': ':freiheitsstatue:',
        'es': ':estatua_de_la_libertad:',
        'fr': ':statue_de_la_liberté:',
        'ja': ':自由の女神:',
        'ko': ':자유의_여신상:',
        'pt': ':estátua_da_liberdade:',
        'it': ':statua_della_libertà:',
        'fa': ':مجسمهٔ_آزادی:',
        'id': ':patung_liberty:',
        'zh': ':自由女神像:',
        'ru': ':статуя_Свободы:'
    },
    '\U0001F1F8\U0001F1E9': {  # 🇸🇩
        'en': ':Sudan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Sudan:', ':sudan:'],
        'de': ':flagge_sudan:',
        'es': ':bandera_sudán:',
        'fr': ':drapeau_soudan:',
        'ja': ':旗_スーダン:',
        'ko': ':깃발_수단:',
        'pt': ':bandeira_sudão:',
        'it': ':bandiera_sudan:',
        'fa': ':پرچم_سودان:',
        'id': ':bendera_sudan:',
        'zh': ':苏丹:',
        'ru': ':флаг_Судан:'
    },
    '\U0001F1F8\U0001F1F7': {  # 🇸🇷
        'en': ':Suriname:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Suriname:', ':suriname:'],
        'de': ':flagge_suriname:',
        'es': ':bandera_surinam:',
        'fr': ':drapeau_suriname:',
        'ja': ':旗_スリナム:',
        'ko': ':깃발_수리남:',
        'pt': ':bandeira_suriname:',
        'it': ':bandiera_suriname:',
        'fa': ':پرچم_سورینام:',
        'id': ':bendera_suriname:',
        'zh': ':苏里南:',
        'ru': ':флаг_Суринам:'
    },
    '\U0001F1F8\U0001F1EF': {  # 🇸🇯
        'en': ':Svalbard_&_Jan_Mayen:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Svalbard_&_Jan_Mayen:', ':svalbard_jan_mayen:'],
        'de': ':flagge_spitzbergen_und_jan_mayen:',
        'es': ':bandera_svalbard_y_jan_mayen:',
        'fr': ':drapeau_svalbard_et_jan_mayen:',
        'ja': ':旗_スバールバル諸島・ヤンマイエン島:',
        'ko': ':깃발_스발바르제도-얀마웬섬:',
        'pt': ':bandeira_svalbard_e_jan_mayen:',
        'it': ':bandiera_svalbard_e_jan_mayen:',
        'fa': ':پرچم_سوالبارد_و_یان_ماین:',
        'id': ':bendera_kepulauan_svalbard_dan_jan_mayen:',
        'zh': ':斯瓦尔巴和扬马延:',
        'ru': ':флаг_Шпицберген_и_Ян_Майен:'
    },
    '\U0001F1F8\U0001F1EA': {  # 🇸🇪
        'en': ':Sweden:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Sweden:', ':sweden:'],
        'de': ':flagge_schweden:',
        'es': ':bandera_suecia:',
        'fr': ':drapeau_suède:',
        'ja': ':旗_スウェーデン:',
        'ko': ':깃발_스웨덴:',
        'pt': ':bandeira_suécia:',
        'it': ':bandiera_svezia:',
        'fa': ':پرچم_سوئد:',
        'id': ':bendera_swedia:',
        'zh': ':瑞典:',
        'ru': ':флаг_Швеция:'
    },
    '\U0001F1E8\U0001F1ED': {  # 🇨🇭
        'en': ':Switzerland:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Switzerland:', ':switzerland:'],
        'de': ':flagge_schweiz:',
        'es': ':bandera_suiza:',
        'fr': ':drapeau_suisse:',
        'ja': ':旗_スイス:',
        'ko': ':깃발_스위스:',
        'pt': ':bandeira_suíça:',
        'it': ':bandiera_svizzera:',
        'fa': ':پرچم_سوئیس:',
        'id': ':bendera_swiss:',
        'zh': ':瑞士:',
        'ru': ':флаг_Швейцария:'
    },
    '\U0001F1F8\U0001F1FE': {  # 🇸🇾
        'en': ':Syria:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Syria:', ':syria:'],
        'de': ':flagge_syrien:',
        'es': ':bandera_siria:',
        'fr': ':drapeau_syrie:',
        'ja': ':旗_シリア:',
        'ko': ':깃발_시리아:',
        'pt': ':bandeira_síria:',
        'it': ':bandiera_siria:',
        'fa': ':پرچم_سوریه:',
        'id': ':bendera_suriah:',
        'zh': ':叙利亚:',
        'ru': ':флаг_Сирия:'
    },
    '\U0001F1F8\U0001F1F9': {  # 🇸🇹
        'en': ':São_Tomé_&_Príncipe:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_São_Tomé_&_Príncipe:', ':sao_tome_principe:'],
        'de': ':flagge_são_tomé_und_príncipe:',
        'es': ':bandera_santo_tomé_y_príncipe:',
        'fr': ':drapeau_sao_tomé-et-principe:',
        'ja': ':旗_サントメ・プリンシペ:',
        'ko': ':깃발_상투메_프린시페:',
        'pt': ':bandeira_são_tomé_e_príncipe:',
        'it': ':bandiera_são_tomé_e_príncipe:',
        'fa': ':پرچم_سائوتومه_و_پرینسیپ:',
        'id': ':bendera_sao_tome_dan_principe:',
        'zh': ':圣多美和普林西比:',
        'ru': ':флаг_Сан_Томе_и_Принсипи:'
    },
    '\U0001F996': {  # 🦖
        'en': ':T-Rex:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':t-rex:', ':t_rex:'],
        'de': ':t-rex:',
        'es': ':t-rex:',
        'fr': ':t-rex:',
        'ja': ':ティラノサウルス:',
        'ko': ':티라노사우루스:',
        'pt': ':tiranossauro_rex:',
        'it': ':t-rex:',
        'fa': ':تی_رکس:',
        'id': ':t-rex:',
        'zh': ':霸王龙:',
        'ru': ':Ти_Рекс:'
    },
    '\U0001F51D': {  # 🔝
        'en': ':TOP_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':top:', ':top_arrow:'],
        'de': ':top-pfeil:',
        'es': ':flecha_top:',
        'fr': ':flèche_en_haut:',
        'ja': ':top矢印:',
        'ko': ':위로_화살표:',
        'pt': ':seta_top:',
        'it': ':freccia_top:',
        'fa': ':پیکان_رو:',
        'id': ':tanda_panah_top:',
        'zh': ':置顶:',
        'ru': ':стрелка_«верх»:'
    },
    '\U0001F1F9\U0001F1FC': {  # 🇹🇼
        'en': ':Taiwan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Taiwan:', ':taiwan:'],
        'de': ':flagge_taiwan:',
        'es': ':bandera_taiwán:',
        'fr': ':drapeau_taïwan:',
        'ja': ':旗_台湾:',
        'ko': ':깃발_대만:',
        'pt': ':bandeira_taiwan:',
        'it': ':bandiera_taiwan:',
        'fa': ':پرچم_تایوان:',
        'id': ':bendera_taiwan:',
        'zh': ':台湾:',
        'ru': ':флаг_Тайвань:'
    },
    '\U0001F1F9\U0001F1EF': {  # 🇹🇯
        'en': ':Tajikistan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Tajikistan:', ':tajikistan:'],
        'de': ':flagge_tadschikistan:',
        'es': ':bandera_tayikistán:',
        'fr': ':drapeau_tadjikistan:',
        'ja': ':旗_タジキスタン:',
        'ko': ':깃발_타지키스탄:',
        'pt': ':bandeira_tadjiquistão:',
        'it': ':bandiera_tagikistan:',
        'fa': ':پرچم_تاجیکستان:',
        'id': ':bendera_tajikistan:',
        'zh': ':塔吉克斯坦:',
        'ru': ':флаг_Таджикистан:'
    },
    '\U0001F1F9\U0001F1FF': {  # 🇹🇿
        'en': ':Tanzania:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Tanzania:', ':tanzania:'],
        'de': ':flagge_tansania:',
        'es': ':bandera_tanzania:',
        'fr': ':drapeau_tanzanie:',
        'ja': ':旗_タンザニア:',
        'ko': ':깃발_탄자니아:',
        'pt': ':bandeira_tanzânia:',
        'it': ':bandiera_tanzania:',
        'fa': ':پرچم_تانزانیا:',
        'id': ':bendera_tanzania:',
        'zh': ':坦桑尼亚:',
        'ru': ':флаг_Танзания:'
    },
    '\U00002649': {  # ♉
        'en': ':Taurus:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':taurus:'],
        'variant': True,
        'de': ':stier_(sternzeichen):',
        'es': ':tauro:',
        'fr': ':taureau:',
        'ja': ':おうし座:',
        'ko': ':황소자리:',
        'pt': ':signo_de_touro:',
        'it': ':segno_zodiacale_del_toro:',
        'fa': ':برج_ثور:',
        'id': ':taurus:',
        'zh': ':金牛座:',
        'ru': ':знак_зодиака_Телец:'
    },
    '\U0001F1F9\U0001F1ED': {  # 🇹🇭
        'en': ':Thailand:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Thailand:', ':thailand:'],
        'de': ':flagge_thailand:',
        'es': ':bandera_tailandia:',
        'fr': ':drapeau_thaïlande:',
        'ja': ':旗_タイ:',
        'ko': ':깃발_태국:',
        'pt': ':bandeira_tailândia:',
        'it': ':bandiera_thailandia:',
        'fa': ':پرچم_تایلند:',
        'id': ':bendera_thailand:',
        'zh': ':泰国:',
        'ru': ':флаг_Таиланд:'
    },
    '\U0001F1F9\U0001F1F1': {  # 🇹🇱
        'en': ':Timor-Leste:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Timor_Leste:', ':timor_leste:'],
        'de': ':flagge_timor-leste:',
        'es': ':bandera_timor-leste:',
        'fr': ':drapeau_timor_oriental:',
        'ja': ':旗_東ティモール:',
        'ko': ':깃발_동티모르:',
        'pt': ':bandeira_timor-leste:',
        'it': ':bandiera_timor_est:',
        'fa': ':پرچم_تیمور-لسته:',
        'id': ':bendera_timor_leste:',
        'zh': ':东帝汶:',
        'ru': ':флаг_Восточный_Тимор:'
    },
    '\U0001F1F9\U0001F1EC': {  # 🇹🇬
        'en': ':Togo:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Togo:', ':togo:'],
        'de': ':flagge_togo:',
        'es': ':bandera_togo:',
        'fr': ':drapeau_togo:',
        'ja': ':旗_トーゴ:',
        'ko': ':깃발_토고:',
        'pt': ':bandeira_togo:',
        'it': ':bandiera_togo:',
        'fa': ':پرچم_توگو:',
        'id': ':bendera_togo:',
        'zh': ':多哥:',
        'ru': ':флаг_Того:'
    },
    '\U0001F1F9\U0001F1F0': {  # 🇹🇰
        'en': ':Tokelau:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Tokelau:', ':tokelau:'],
        'de': ':flagge_tokelau:',
        'es': ':bandera_tokelau:',
        'fr': ':drapeau_tokelau:',
        'ja': ':旗_トケラウ:',
        'ko': ':깃발_토켈라우:',
        'pt': ':bandeira_tokelau:',
        'it': ':bandiera_tokelau:',
        'fa': ':پرچم_توکلائو:',
        'id': ':bendera_tokelau:',
        'zh': ':托克劳:',
        'ru': ':флаг_Токелау:'
    },
    '\U0001F5FC': {  # 🗼
        'en': ':Tokyo_tower:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':tokyo_tower:'],
        'de': ':tokyo_tower:',
        'es': ':torre_de_tokio:',
        'fr': ':tour_de_tokyo:',
        'ja': ':東京タワー:',
        'ko': ':도쿄_타워:',
        'pt': ':torre_de_tóquio:',
        'it': ':torre_di_tokyo:',
        'fa': ':برج_توکیو:',
        'id': ':menara_tokyo:',
        'zh': ':东京塔:',
        'ru': ':Токийская_башня:'
    },
    '\U0001F1F9\U0001F1F4': {  # 🇹🇴
        'en': ':Tonga:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Tonga:', ':tonga:'],
        'de': ':flagge_tonga:',
        'es': ':bandera_tonga:',
        'fr': ':drapeau_tonga:',
        'ja': ':旗_トンガ:',
        'ko': ':깃발_통가:',
        'pt': ':bandeira_tonga:',
        'it': ':bandiera_tonga:',
        'fa': ':پرچم_تونگا:',
        'id': ':bendera_tonga:',
        'zh': ':汤加:',
        'ru': ':флаг_Тонга:'
    },
    '\U0001F1F9\U0001F1F9': {  # 🇹🇹
        'en': ':Trinidad_&_Tobago:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Trinidad_&_Tobago:', ':trinidad_tobago:'],
        'de': ':flagge_trinidad_und_tobago:',
        'es': ':bandera_trinidad_y_tobago:',
        'fr': ':drapeau_trinité-et-tobago:',
        'ja': ':旗_トリニダード・トバゴ:',
        'ko': ':깃발_트리니다드_토바고:',
        'pt': ':bandeira_trinidad_e_tobago:',
        'it': ':bandiera_trinidad_e_tobago:',
        'fa': ':پرچم_ترینیداد_و_توباگو:',
        'id': ':bendera_trinidad_dan_tobago:',
        'zh': ':特立尼达和多巴哥:',
        'ru': ':флаг_Тринидад_и_Тобаго:'
    },
    '\U0001F1F9\U0001F1E6': {  # 🇹🇦
        'en': ':Tristan_da_Cunha:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Tristan_da_Cunha:', ':tristan_da_cunha:'],
        'de': ':flagge_tristan_da_cunha:',
        'es': ':bandera_tristán_de_acuña:',
        'fr': ':drapeau_tristan_da_cunha:',
        'ja': ':旗_トリスタン・ダ・クーニャ:',
        'ko': ':깃발_트리스탄다쿠나:',
        'pt': ':bandeira_tristão_da_cunha:',
        'it': ':bandiera_tristan_da_cunha:',
        'fa': ':پرچم_تریستان_دا_کونا:',
        'id': ':bendera_tristan_da_cunha:',
        'zh': ':特里斯坦达库尼亚群岛:',
        'ru': ':флаг_Тристан_да_Кунья:'
    },
    '\U0001F1F9\U0001F1F3': {  # 🇹🇳
        'en': ':Tunisia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Tunisia:', ':tunisia:'],
        'de': ':flagge_tunesien:',
        'es': ':bandera_túnez:',
        'fr': ':drapeau_tunisie:',
        'ja': ':旗_チュニジア:',
        'ko': ':깃발_튀니지:',
        'pt': ':bandeira_tunísia:',
        'it': ':bandiera_tunisia:',
        'fa': ':پرچم_تونس:',
        'id': ':bendera_tunisia:',
        'zh': ':突尼斯:',
        'ru': ':флаг_Тунис:'
    },
    '\U0001F1F9\U0001F1F7': {  # 🇹🇷
        'en': ':Turkey:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Turkey:', ':tr:'],
        'de': ':flagge_türkei:',
        'es': ':bandera_turquía:',
        'fr': ':drapeau_turquie:',
        'ja': ':旗_トルコ:',
        'ko': ':깃발_튀르키예:',
        'pt': ':bandeira_turquia:',
        'it': ':bandiera_turchia:',
        'fa': ':پرچم_ترکیه:',
        'id': ':bendera_turki:',
        'zh': ':土耳其:',
        'ru': ':флаг_Турция:'
    },
    '\U0001F1F9\U0001F1F2': {  # 🇹🇲
        'en': ':Turkmenistan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Turkmenistan:', ':turkmenistan:'],
        'de': ':flagge_turkmenistan:',
        'es': ':bandera_turkmenistán:',
        'fr': ':drapeau_turkménistan:',
        'ja': ':旗_トルクメニスタン:',
        'ko': ':깃발_투르크메니스탄:',
        'pt': ':bandeira_turcomenistão:',
        'it': ':bandiera_turkmenistan:',
        'fa': ':پرچم_ترکمنستان:',
        'id': ':bendera_turkmenistan:',
        'zh': ':土库曼斯坦:',
        'ru': ':флаг_Туркменистан:'
    },
    '\U0001F1F9\U0001F1E8': {  # 🇹🇨
        'en': ':Turks_&_Caicos_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Turks_&_Caicos_Islands:', ':turks_caicos_islands:'],
        'de': ':flagge_turks-_und_caicosinseln:',
        'es': ':bandera_islas_turcas_y_caicos:',
        'fr': ':drapeau_îles_turques-et-caïques:',
        'ja': ':旗_タークス・カイコス諸島:',
        'ko': ':깃발_터크스_케이커스_제도:',
        'pt': ':bandeira_ilhas_turcas_e_caicos:',
        'it': ':bandiera_isole_turks_e_caicos:',
        'fa': ':پرچم_جزایر_تورکس_و_کایکوس:',
        'id': ':bendera_kepulauan_turks_dan_caicos:',
        'zh': ':特克斯和凯科斯群岛:',
        'ru': ':флаг_Тёркс_и_Кайкос:'
    },
    '\U0001F1F9\U0001F1FB': {  # 🇹🇻
        'en': ':Tuvalu:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Tuvalu:', ':tuvalu:'],
        'de': ':flagge_tuvalu:',
        'es': ':bandera_tuvalu:',
        'fr': ':drapeau_tuvalu:',
        'ja': ':旗_ツバル:',
        'ko': ':깃발_투발루:',
        'pt': ':bandeira_tuvalu:',
        'it': ':bandiera_tuvalu:',
        'fa': ':پرچم_تووالو:',
        'id': ':bendera_tuvalu:',
        'zh': ':图瓦卢:',
        'ru': ':флаг_Тувалу:'
    },
    '\U0001F1FA\U0001F1F2': {  # 🇺🇲
        'en': ':U.S._Outlying_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_U.S._Outlying_Islands:', ':us_outlying_islands:'],
        'de': ':flagge_amerikanische_überseeinseln:',
        'es': ':bandera_islas_menores_alejadas_de_ee._uu.:',
        'fr': ':drapeau_îles_mineures_éloignées_des_états-unis:',
        'ja': ':旗_合衆国領有小離島:',
        'ko': ':깃발_미국령_해외_제도:',
        'pt': ':bandeira_ilhas_menores_distantes_dos_eua:',
        'it': ':bandiera_isole_minori_esterne_degli_stati_uniti:',
        'fa': ':پرچم_جزایر_دورافتادهٔ_ایالات_متحده:',
        'id': ':bendera_kepulauan_terluar_as:',
        'zh': ':美国本土外小岛屿:',
        'ru': ':флаг_Внешние_малые_о_ва_(США):'
    },
    '\U0001F1FB\U0001F1EE': {  # 🇻🇮
        'en': ':U.S._Virgin_Islands:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_U.S._Virgin_Islands:', ':us_virgin_islands:'],
        'de': ':flagge_amerikanische_jungferninseln:',
        'es': ':bandera_islas_vírgenes_de_ee._uu.:',
        'fr': ':drapeau_îles_vierges_des_états-unis:',
        'ja': ':旗_米領ヴァージン諸島:',
        'ko': ':깃발_미국령_버진아일랜드:',
        'pt': ':bandeira_ilhas_virgens_americanas:',
        'it': ':bandiera_isole_vergini_americane:',
        'fa': ':پرچم_جزایر_ویرجین_ایالات_متحده:',
        'id': ':bendera_kepulauan_virgin_amerika_serikat:',
        'zh': ':美属维尔京群岛:',
        'ru': ':флаг_Виргинские_о_ва_(США):'
    },
    '\U0001F199': {  # 🆙
        'en': ':UP!_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':up:', ':up_button:'],
        'de': ':schriftzug_up!_im_blauen_quadrat:',
        'es': ':botón_up!:',
        'fr': ':bouton_vers_le_haut:',
        'ja': ':upマーク:',
        'ko': ':위로_버튼:',
        'pt': ':botão_up!:',
        'it': ':pulsante_up!:',
        'fa': ':دکمهٔ_!up:',
        'id': ':tombol_up!:',
        'zh': ':up按钮:',
        'ru': ':значок_«обновление»:'
    },
    '\U0001F1FA\U0001F1EC': {  # 🇺🇬
        'en': ':Uganda:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Uganda:', ':uganda:'],
        'de': ':flagge_uganda:',
        'es': ':bandera_uganda:',
        'fr': ':drapeau_ouganda:',
        'ja': ':旗_ウガンダ:',
        'ko': ':깃발_우간다:',
        'pt': ':bandeira_uganda:',
        'it': ':bandiera_uganda:',
        'fa': ':پرچم_اوگاندا:',
        'id': ':bendera_uganda:',
        'zh': ':乌干达:',
        'ru': ':флаг_Уганда:'
    },
    '\U0001F1FA\U0001F1E6': {  # 🇺🇦
        'en': ':Ukraine:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Ukraine:', ':ukraine:'],
        'de': ':flagge_ukraine:',
        'es': ':bandera_ucrania:',
        'fr': ':drapeau_ukraine:',
        'ja': ':旗_ウクライナ:',
        'ko': ':깃발_우크라이나:',
        'pt': ':bandeira_ucrânia:',
        'it': ':bandiera_ucraina:',
        'fa': ':پرچم_اوکراین:',
        'id': ':bendera_ukraina:',
        'zh': ':乌克兰:',
        'ru': ':флаг_Украина:'
    },
    '\U0001F1E6\U0001F1EA': {  # 🇦🇪
        'en': ':United_Arab_Emirates:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_United_Arab_Emirates:', ':united_arab_emirates:'],
        'de': ':flagge_vereinigte_arabische_emirate:',
        'es': ':bandera_emiratos_árabes_unidos:',
        'fr': ':drapeau_émirats_arabes_unis:',
        'ja': ':旗_アラブ首長国連邦:',
        'ko': ':깃발_아랍에미리트:',
        'pt': ':bandeira_emirados_árabes_unidos:',
        'it': ':bandiera_emirati_arabi_uniti:',
        'fa': ':پرچم_امارات_متحدهٔ_عربی:',
        'id': ':bendera_uni_emirat_arab:',
        'zh': ':阿拉伯联合酋长国:',
        'ru': ':флаг_ОАЭ:'
    },
    '\U0001F1EC\U0001F1E7': {  # 🇬🇧
        'en': ':United_Kingdom:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_United_Kingdom:', ':gb:', ':uk:'],
        'de': ':flagge_vereinigtes_königreich:',
        'es': ':bandera_reino_unido:',
        'fr': ':drapeau_royaume-uni:',
        'ja': ':旗_イギリス:',
        'ko': ':깃발_영국:',
        'pt': ':bandeira_reino_unido:',
        'it': ':bandiera_regno_unito:',
        'fa': ':پرچم_بریتانیا:',
        'id': ':bendera_inggris_raya:',
        'zh': ':英国:',
        'ru': ':флаг_Великобритания:'
    },
    '\U0001F1FA\U0001F1F3': {  # 🇺🇳
        'en': ':United_Nations:',
        'status': fully_qualified,
        'E': 4,
        'alias': [':united_nations:'],
        'de': ':flagge_vereinte_nationen:',
        'es': ':bandera_naciones_unidas:',
        'fr': ':drapeau_nations_unies:',
        'ja': ':旗_国際連合:',
        'ko': ':깃발_국제연합:',
        'pt': ':bandeira_nações_unidas:',
        'it': ':bandiera_nazioni_unite:',
        'fa': ':پرچم_سازمان_ملل_متحد:',
        'id': ':bendera_perserikatan_bangsa-bangsa:',
        'zh': ':联合国:',
        'ru': ':флаг_Организация_Объединенных_Наций:'
    },
    '\U0001F1FA\U0001F1F8': {  # 🇺🇸
        'en': ':United_States:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flag_for_United_States:', ':us:'],
        'de': ':flagge_vereinigte_staaten:',
        'es': ':bandera_estados_unidos:',
        'fr': ':drapeau_états-unis:',
        'ja': ':旗_アメリカ合衆国:',
        'ko': ':깃발_미국:',
        'pt': ':bandeira_estados_unidos:',
        'it': ':bandiera_stati_uniti:',
        'fa': ':پرچم_ایالات_متحده:',
        'id': ':bendera_amerika_serikat:',
        'zh': ':美国:',
        'ru': ':флаг_Соединенные_Штаты:'
    },
    '\U0001F1FA\U0001F1FE': {  # 🇺🇾
        'en': ':Uruguay:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Uruguay:', ':uruguay:'],
        'de': ':flagge_uruguay:',
        'es': ':bandera_uruguay:',
        'fr': ':drapeau_uruguay:',
        'ja': ':旗_ウルグアイ:',
        'ko': ':깃발_우루과이:',
        'pt': ':bandeira_uruguai:',
        'it': ':bandiera_uruguay:',
        'fa': ':پرچم_اروگوئه:',
        'id': ':bendera_uruguay:',
        'zh': ':乌拉圭:',
        'ru': ':флаг_Уругвай:'
    },
    '\U0001F1FA\U0001F1FF': {  # 🇺🇿
        'en': ':Uzbekistan:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Uzbekistan:', ':uzbekistan:'],
        'de': ':flagge_usbekistan:',
        'es': ':bandera_uzbekistán:',
        'fr': ':drapeau_ouzbékistan:',
        'ja': ':旗_ウズベキスタン:',
        'ko': ':깃발_우즈베키스탄:',
        'pt': ':bandeira_uzbequistão:',
        'it': ':bandiera_uzbekistan:',
        'fa': ':پرچم_ازبکستان:',
        'id': ':bendera_uzbekistan:',
        'zh': ':乌兹别克斯坦:',
        'ru': ':флаг_Узбекистан:'
    },
    '\U0001F19A': {  # 🆚
        'en': ':VS_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':vs:', ':vs_button:'],
        'de': ':schriftzug_vs_in_orangem_quadrat:',
        'es': ':botón_vs:',
        'fr': ':bouton_vs:',
        'ja': ':vsマーク:',
        'ko': ':대:',
        'pt': ':botão_vs:',
        'it': ':pulsante_vs:',
        'fa': ':دکمهٔ_vs:',
        'id': ':tombol_vs:',
        'zh': ':VS按钮:',
        'ru': ':значок_«против»:'
    },
    '\U0001F1FB\U0001F1FA': {  # 🇻🇺
        'en': ':Vanuatu:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Vanuatu:', ':vanuatu:'],
        'de': ':flagge_vanuatu:',
        'es': ':bandera_vanuatu:',
        'fr': ':drapeau_vanuatu:',
        'ja': ':旗_バヌアツ:',
        'ko': ':깃발_바누아투:',
        'pt': ':bandeira_vanuatu:',
        'it': ':bandiera_vanuatu:',
        'fa': ':پرچم_وانواتو:',
        'id': ':bendera_vanuatu:',
        'zh': ':瓦努阿图:',
        'ru': ':флаг_Вануату:'
    },
    '\U0001F1FB\U0001F1E6': {  # 🇻🇦
        'en': ':Vatican_City:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Vatican_City:', ':vatican_city:'],
        'de': ':flagge_vatikanstadt:',
        'es': ':bandera_ciudad_del_vaticano:',
        'fr': ':drapeau_état_de_la_cité_du_vatican:',
        'ja': ':旗_バチカン市国:',
        'ko': ':깃발_바티칸_시국:',
        'pt': ':bandeira_cidade_do_vaticano:',
        'it': ':bandiera_città_del_vaticano:',
        'fa': ':پرچم_واتیکان:',
        'id': ':bendera_vatikan:',
        'zh': ':梵蒂冈:',
        'ru': ':флаг_Ватикан:'
    },
    '\U0001F1FB\U0001F1EA': {  # 🇻🇪
        'en': ':Venezuela:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Venezuela:', ':venezuela:'],
        'de': ':flagge_venezuela:',
        'es': ':bandera_venezuela:',
        'fr': ':drapeau_venezuela:',
        'ja': ':旗_ベネズエラ:',
        'ko': ':깃발_베네수엘라:',
        'pt': ':bandeira_venezuela:',
        'it': ':bandiera_venezuela:',
        'fa': ':پرچم_ونزوئلا:',
        'id': ':bendera_venezuela:',
        'zh': ':委内瑞拉:',
        'ru': ':флаг_Венесуэла:'
    },
    '\U0001F1FB\U0001F1F3': {  # 🇻🇳
        'en': ':Vietnam:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Vietnam:', ':vietnam:'],
        'de': ':flagge_vietnam:',
        'es': ':bandera_vietnam:',
        'fr': ':drapeau_viêt_nam:',
        'ja': ':旗_ベトナム:',
        'ko': ':깃발_베트남:',
        'pt': ':bandeira_vietnã:',
        'it': ':bandiera_vietnam:',
        'fa': ':پرچم_ویتنام:',
        'id': ':bendera_vietnam:',
        'zh': ':越南:',
        'ru': ':флаг_Вьетнам:'
    },
    '\U0000264D': {  # ♍
        'en': ':Virgo:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':virgo:'],
        'variant': True,
        'de': ':jungfrau_(sternzeichen):',
        'es': ':virgo:',
        'fr': ':vierge:',
        'ja': ':おとめ座:',
        'ko': ':처녀자리:',
        'pt': ':signo_de_virgem:',
        'it': ':segno_zodiacale_della_vergine:',
        'fa': ':برج_سنبله:',
        'id': ':virgo:',
        'zh': ':处女座:',
        'ru': ':знак_зодиака_Дева:'
    },
    '\U0001F3F4\U000E0067\U000E0062\U000E0077\U000E006C\U000E0073\U000E007F': {  # 🏴󠁧󠁢󠁷󠁬󠁳󠁿
        'en': ':Wales:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':wales:'],
        'de': ':flagge_wales:',
        'es': ':bandera_gales:',
        'fr': ':drapeau_pays_de_galles:',
        'ja': ':旗_ウェールズ:',
        'ko': ':깃발_웨일즈:',
        'pt': ':bandeira_país_de_gales:',
        'it': ':bandiera_galles:',
        'fa': ':پرچم_ویلز:',
        'id': ':bendera_wales:',
        'zh': ':威尔士:',
        'ru': ':флаг_Уэльс:'
    },
    '\U0001F1FC\U0001F1EB': {  # 🇼🇫
        'en': ':Wallis_&_Futuna:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Wallis_&_Futuna:', ':wallis_futuna:'],
        'de': ':flagge_wallis_und_futuna:',
        'es': ':bandera_wallis_y_futuna:',
        'fr': ':drapeau_wallis-et-futuna:',
        'ja': ':旗_ウォリス・フツナ:',
        'ko': ':깃발_왈리스-푸투나_제도:',
        'pt': ':bandeira_wallis_e_futuna:',
        'it': ':bandiera_wallis_e_futuna:',
        'fa': ':پرچم_والیس_و_فوتونا:',
        'id': ':bendera_kepulauan_wallis_dan_futuna:',
        'zh': ':瓦利斯和富图纳:',
        'ru': ':флаг_Уоллис_и_Футуна:'
    },
    '\U0001F1EA\U0001F1ED': {  # 🇪🇭
        'en': ':Western_Sahara:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Western_Sahara:', ':western_sahara:'],
        'de': ':flagge_westsahara:',
        'es': ':bandera_sáhara_occidental:',
        'fr': ':drapeau_sahara_occidental:',
        'ja': ':旗_西サハラ:',
        'ko': ':깃발_서사하라:',
        'pt': ':bandeira_saara_ocidental:',
        'it': ':bandiera_sahara_occidentale:',
        'fa': ':پرچم_صحرای_غربی:',
        'id': ':bendera_sahara_barat:',
        'zh': ':西撒哈拉:',
        'ru': ':флаг_Западная_Сахара:'
    },
    '\U0001F1FE\U0001F1EA': {  # 🇾🇪
        'en': ':Yemen:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Yemen:', ':yemen:'],
        'de': ':flagge_jemen:',
        'es': ':bandera_yemen:',
        'fr': ':drapeau_yémen:',
        'ja': ':旗_イエメン:',
        'ko': ':깃발_예멘:',
        'pt': ':bandeira_iêmen:',
        'it': ':bandiera_yemen:',
        'fa': ':پرچم_یمن:',
        'id': ':bendera_yaman:',
        'zh': ':也门:',
        'ru': ':флаг_Йемен:'
    },
    '\U0001F4A4': {  # 💤
        'en': ':ZZZ:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':zzz:'],
        'de': ':schlafen:',
        'es': ':símbolo_de_sueño:',
        'fr': ':endormi:',
        'ja': ':グーグー:',
        'ko': ':졸림:',
        'pt': ':zzz:',
        'it': ':sonno:',
        'fa': ':خروپف:',
        'id': ':zzz:',
        'zh': ':睡着:',
        'ru': ':сон:'
    },
    '\U0001F1FF\U0001F1F2': {  # 🇿🇲
        'en': ':Zambia:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Zambia:', ':zambia:'],
        'de': ':flagge_sambia:',
        'es': ':bandera_zambia:',
        'fr': ':drapeau_zambie:',
        'ja': ':旗_ザンビア:',
        'ko': ':깃발_잠비아:',
        'pt': ':bandeira_zâmbia:',
        'it': ':bandiera_zambia:',
        'fa': ':پرچم_زامبیا:',
        'id': ':bendera_zambia:',
        'zh': ':赞比亚:',
        'ru': ':флаг_Замбия:'
    },
    '\U0001F1FF\U0001F1FC': {  # 🇿🇼
        'en': ':Zimbabwe:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':flag_for_Zimbabwe:', ':zimbabwe:'],
        'de': ':flagge_simbabwe:',
        'es': ':bandera_zimbabue:',
        'fr': ':drapeau_zimbabwe:',
        'ja': ':旗_ジンバブエ:',
        'ko': ':깃발_짐바브웨:',
        'pt': ':bandeira_zimbábue:',
        'it': ':bandiera_zimbabwe:',
        'fa': ':پرچم_زیمبابوه:',
        'id': ':bendera_zimbabwe:',
        'zh': ':津巴布韦:',
        'ru': ':флаг_Зимбабве:'
    },
    '\U0001F9EE': {  # 🧮
        'en': ':abacus:',
        'status': fully_qualified,
        'E': 11,
        'de': ':abakus:',
        'es': ':ábaco:',
        'fr': ':abaque:',
        'ja': ':そろばん:',
        'ko': ':주판:',
        'pt': ':ábaco:',
        'it': ':abaco:',
        'fa': ':چرتکه:',
        'id': ':sempoa:',
        'zh': ':算盘:',
        'ru': ':счеты:'
    },
    '\U0001FA97': {  # 🪗
        'en': ':accordion:',
        'status': fully_qualified,
        'E': 13,
        'de': ':akkordeon:',
        'es': ':acordeón:',
        'fr': ':accordéon:',
        'ja': ':アコーディオン:',
        'ko': ':아코디언:',
        'pt': ':acordeão:',
        'it': ':fisarmonica:',
        'fa': ':آکوردئون:',
        'id': ':akordeon:',
        'zh': ':手风琴:',
        'ru': ':аккордеон:'
    },
    '\U0001FA79': {  # 🩹
        'en': ':adhesive_bandage:',
        'status': fully_qualified,
        'E': 12,
        'de': ':heftpflaster:',
        'es': ':tirita:',
        'fr': ':sparadrap:',
        'ja': ':絆創膏:',
        'ko': ':반창고:',
        'pt': ':atadura_adesiva:',
        'it': ':cerotto:',
        'fa': ':چسب_زخم:',
        'id': ':plester_luka:',
        'zh': ':创可贴:',
        'ru': ':пластырь:'
    },
    '\U0001F39F\U0000FE0F': {  # 🎟️
        'en': ':admission_tickets:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':tickets:'],
        'variant': True,
        'de': ':eintrittskarten:',
        'es': ':entradas:',
        'fr': ':billet_d’entrée:',
        'ja': ':入場券:',
        'ko': ':입장_티켓:',
        'pt': ':ingresso_de_cinema:',
        'it': ':biglietto_d’ingresso:',
        'fa': ':بلیت_ورود:',
        'id': ':tiket_masuk:',
        'zh': ':入场券:',
        'ru': ':билеты:'
    },
    '\U0001F39F': {  # 🎟
        'en': ':admission_tickets:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':tickets:'],
        'variant': True,
        'de': ':eintrittskarten:',
        'es': ':entradas:',
        'fr': ':billet_d’entrée:',
        'ja': ':入場券:',
        'ko': ':입장_티켓:',
        'pt': ':ingresso_de_cinema:',
        'it': ':biglietto_d’ingresso:',
        'fa': ':بلیت_ورود:',
        'id': ':tiket_masuk:',
        'zh': ':入场券:',
        'ru': ':билеты:'
    },
    '\U0001F6A1': {  # 🚡
        'en': ':aerial_tramway:',
        'status': fully_qualified,
        'E': 1,
        'de': ':bergseilbahn:',
        'es': ':teleférico:',
        'fr': ':tramway_aérien:',
        'ja': ':ロープウェイ:',
        'ko': ':케이블카:',
        'pt': ':teleférico:',
        'it': ':funivia_aerea:',
        'fa': ':تراموای_هوایی:',
        'id': ':trem_gantung:',
        'zh': ':索道:',
        'ru': ':канатная_дорога:'
    },
    '\U00002708\U0000FE0F': {  # ✈️
        'en': ':airplane:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':flugzeug:',
        'es': ':avión:',
        'fr': ':avion:',
        'ja': ':飛行機:',
        'ko': ':비행기:',
        'pt': ':avião:',
        'it': ':aeroplano:',
        'fa': ':هواپیما:',
        'id': ':pesawat:',
        'zh': ':飞机:',
        'ru': ':самолет:'
    },
    '\U00002708': {  # ✈
        'en': ':airplane:',
        'status': unqualified,
        'E': 0.6,
        'variant': True,
        'de': ':flugzeug:',
        'es': ':avión:',
        'fr': ':avion:',
        'ja': ':飛行機:',
        'ko': ':비행기:',
        'pt': ':avião:',
        'it': ':aeroplano:',
        'fa': ':هواپیما:',
        'id': ':pesawat:',
        'zh': ':飞机:',
        'ru': ':самолет:'
    },
    '\U0001F6EC': {  # 🛬
        'en': ':airplane_arrival:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':airplane_arriving:', ':flight_arrival:'],
        'de': ':landung_eines_flugzeugs:',
        'es': ':avión_aterrizando:',
        'fr': ':avion_à_l’atterrissage:',
        'ja': ':飛行機着陸:',
        'ko': ':비행기_착륙:',
        'pt': ':avião_aterrissando:',
        'it': ':atterraggio:',
        'fa': ':زمان_ورود_هواپیما:',
        'id': ':pesawat_mendarat:',
        'zh': ':航班降落:',
        'ru': ':посадка:'
    },
    '\U0001F6EB': {  # 🛫
        'en': ':airplane_departure:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':flight_departure:'],
        'de': ':abflug:',
        'es': ':avión_despegando:',
        'fr': ':avion_au_décollage:',
        'ja': ':飛行機離陸:',
        'ko': ':비행기_이륙:',
        'pt': ':avião_decolando:',
        'it': ':decollo:',
        'fa': ':زمان_حرکت_هواپیما:',
        'id': ':pesawat_lepas_landas:',
        'zh': ':航班起飞:',
        'ru': ':взлет:'
    },
    '\U000023F0': {  # ⏰
        'en': ':alarm_clock:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':wecker:',
        'es': ':reloj_despertador:',
        'fr': ':réveil:',
        'ja': ':目覚まし時計:',
        'ko': ':알람_시계:',
        'pt': ':despertador:',
        'it': ':sveglia:',
        'fa': ':ساعت_رومیزی:',
        'id': ':jam_weker:',
        'zh': ':闹钟:',
        'ru': ':будильник:'
    },
    '\U00002697\U0000FE0F': {  # ⚗️
        'en': ':alembic:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':destillierapparat:',
        'es': ':alambique:',
        'fr': ':alambic:',
        'ja': ':蒸留器:',
        'ko': ':증류기:',
        'pt': ':alambique:',
        'it': ':alambicco:',
        'fa': ':انبیق:',
        'id': ':alembic:',
        'zh': ':蒸馏器:',
        'ru': ':перегонныи_сосуд:'
    },
    '\U00002697': {  # ⚗
        'en': ':alembic:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':destillierapparat:',
        'es': ':alambique:',
        'fr': ':alambic:',
        'ja': ':蒸留器:',
        'ko': ':증류기:',
        'pt': ':alambique:',
        'it': ':alambicco:',
        'fa': ':انبیق:',
        'id': ':alembic:',
        'zh': ':蒸馏器:',
        'ru': ':перегонный_сосуд:'
    },
    '\U0001F47D': {  # 👽
        'en': ':alien:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':außerirdischer:',
        'es': ':alienígena:',
        'fr': ':alien:',
        'ja': ':エイリアン:',
        'ko': ':외계인:',
        'pt': ':alienígena:',
        'it': ':alieno:',
        'fa': ':موجود_فضایی:',
        'id': ':alien:',
        'zh': ':外星人:',
        'ru': ':инопланетянин:'
    },
    '\U0001F47E': {  # 👾
        'en': ':alien_monster:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':space_invader:'],
        'de': ':computerspiel-monster:',
        'es': ':monstruo_alienígena:',
        'fr': ':monstre_de_l’espace:',
        'ja': ':インベーダー:',
        'ko': ':에일리언:',
        'pt': ':monstro_alienígena:',
        'it': ':mostro_alieno:',
        'fa': ':هیولای_فضایی:',
        'id': ':monster_alien:',
        'zh': ':外星怪物:',
        'ru': ':космический_монстр:'
    },
    '\U0001F691': {  # 🚑
        'en': ':ambulance:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':krankenwagen:',
        'es': ':ambulancia:',
        'fr': ':ambulance:',
        'ja': ':救急車:',
        'ko': ':구급차:',
        'pt': ':ambulância:',
        'it': ':ambulanza:',
        'fa': ':آمبولانس:',
        'id': ':ambulans:',
        'zh': ':救护车:',
        'ru': ':машина_скорой_помощи:'
    },
    '\U0001F3C8': {  # 🏈
        'en': ':american_football:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':football:'],
        'de': ':football:',
        'es': ':balón_de_fútbol_americano:',
        'fr': ':football_américain:',
        'ja': ':アメフト:',
        'ko': ':미식축구공:',
        'pt': ':bola_de_futebol_americano:',
        'it': ':football_americano:',
        'fa': ':فوتبال_امریکایی:',
        'id': ':american_football:',
        'zh': ':美式橄榄球:',
        'ru': ':американский_футбол:'
    },
    '\U0001F3FA': {  # 🏺
        'en': ':amphora:',
        'status': fully_qualified,
        'E': 1,
        'de': ':amphore:',
        'es': ':ánfora:',
        'fr': ':amphore:',
        'ja': ':壺:',
        'ko': ':암포라:',
        'pt': ':ânfora:',
        'it': ':anfora:',
        'fa': ':سبو:',
        'id': ':amphora:',
        'zh': ':双耳瓶:',
        'ru': ':амфора:'
    },
    '\U0001FAC0': {  # 🫀
        'en': ':anatomical_heart:',
        'status': fully_qualified,
        'E': 13,
        'de': ':herz_(organ):',
        'es': ':corazón_humano:',
        'fr': ':cœur:',
        'ja': ':心臓:',
        'ko': ':심장:',
        'pt': ':coração_humano:',
        'it': ':organo_del_cuore:',
        'fa': ':قلب:',
        'id': ':organ_jantung:',
        'zh': ':心脏器官:',
        'ru': ':сердце_человека:'
    },
    '\U00002693': {  # ⚓
        'en': ':anchor:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':anker:',
        'es': ':ancla:',
        'fr': ':ancre:',
        'ja': ':錨:',
        'ko': ':닻:',
        'pt': ':âncora:',
        'it': ':ancora:',
        'fa': ':لنگر:',
        'id': ':jangkar:',
        'zh': ':锚:',
        'ru': ':якорь:'
    },
    '\U0001F4A2': {  # 💢
        'en': ':anger_symbol:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':anger:'],
        'de': ':ärger:',
        'es': ':símbolo_de_enfado:',
        'fr': ':symbole_de_colère:',
        'ja': ':むかっ:',
        'ko': ':화남:',
        'pt': ':símbolo_de_raiva:',
        'it': ':rabbia:',
        'fa': ':نماد_عصبانیت:',
        'id': ':marah:',
        'zh': ':怒:',
        'ru': ':значок_«гнев»:'
    },
    '\U0001F620': {  # 😠
        'en': ':angry_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':angry:'],
        'de': ':verärgertes_gesicht:',
        'es': ':cara_enfadada:',
        'fr': ':visage_en_colère:',
        'ja': ':ぷんぷん:',
        'ko': ':화난_얼굴:',
        'pt': ':rosto_zangado:',
        'it': ':faccina_arrabbiata:',
        'fa': ':عصبانی:',
        'id': ':wajah_marah:',
        'zh': ':生气:',
        'ru': ':сердится:'
    },
    '\U0001F47F': {  # 👿
        'en': ':angry_face_with_horns:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':imp:'],
        'de': ':wütendes_gesicht_mit_hörnern:',
        'es': ':cara_enfadada_con_cuernos:',
        'fr': ':visage_en_colère_avec_des_cornes:',
        'ja': ':怒った悪魔:',
        'ko': ':머리에_뿔이_달린_화난_얼굴:',
        'pt': ':rosto_zangado_com_chifres:',
        'it': ':faccina_arrabbiata_con_corna:',
        'fa': ':چهره_عصبانی_با_دو_شاخ:',
        'id': ':wajah_marah_dengan_tanduk:',
        'zh': ':生气的恶魔:',
        'ru': ':злой_чертенок:'
    },
    '\U0001F627': {  # 😧
        'en': ':anguished_face:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':anguished:'],
        'de': ':qualvolles_gesicht:',
        'es': ':cara_angustiada:',
        'fr': ':visage_angoissé:',
        'ja': ':苦悩:',
        'ko': ':괴로워하는_얼굴:',
        'pt': ':rosto_angustiado:',
        'it': ':faccina_angosciata:',
        'fa': ':دردمند:',
        'id': ':wajah_menderita:',
        'zh': ':极度痛苦:',
        'ru': ':страдает:'
    },
    '\U0001F41C': {  # 🐜
        'en': ':ant:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':ameise:',
        'es': ':hormiga:',
        'fr': ':fourmi:',
        'ja': ':アリ:',
        'ko': ':개미:',
        'pt': ':formiga:',
        'it': ':formica:',
        'fa': ':مورچه:',
        'id': ':semut:',
        'zh': ':蚂蚁:',
        'ru': ':муравей:'
    },
    '\U0001F4F6': {  # 📶
        'en': ':antenna_bars:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':signal_strength:'],
        'de': ':balkenförmige_signalstärkenanzeige:',
        'es': ':barras_de_cobertura:',
        'fr': ':barres_de_réseau:',
        'ja': ':アンテナマーク:',
        'ko': ':안테나_신호:',
        'pt': ':barras_de_sinal:',
        'it': ':segnale_cellulare:',
        'fa': ':نوار_سیگنال:',
        'id': ':batang_antena:',
        'zh': ':信号强度条:',
        'ru': ':сигнал_сети:'
    },
    '\U0001F630': {  # 😰
        'en': ':anxious_face_with_sweat:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cold_sweat:'],
        'de': ':besorgtes_gesicht_mit_schweißtropfen:',
        'es': ':cara_con_ansiedad_y_sudor:',
        'fr': ':visage_anxieux_avec_goutte_de_sueur:',
        'ja': ':冷や汗青ざめ:',
        'ko': ':땀_흘리며_불안해하는_얼굴:',
        'pt': ':rosto_ansioso_com_gota_de_suor:',
        'it': ':faccina_sudata_in_ansia:',
        'fa': ':عرق_ترس:',
        'id': ':wajah_ketakutan_berkeringat:',
        'zh': ':冷汗:',
        'ru': ':сильно_беспокоится:'
    },
    '\U0001F69B': {  # 🚛
        'en': ':articulated_lorry:',
        'status': fully_qualified,
        'E': 1,
        'de': ':sattelzug:',
        'es': ':camión_articulado:',
        'fr': ':semi-remorque:',
        'ja': ':トレーラー:',
        'ko': ':트레일러_트럭:',
        'pt': ':caminhão:',
        'it': ':autoarticolato:',
        'fa': ':کامیون_مفصلی:',
        'id': ':lori_gandeng:',
        'zh': ':铰接式货车:',
        'ru': ':фура:'
    },
    '\U0001F9D1\U0000200D\U0001F3A8': {  # 🧑‍🎨
        'en': ':artist:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':künstler(in):',
        'es': ':artista:',
        'fr': ':artiste:',
        'ja': ':芸術家:',
        'ko': ':화가:',
        'pt': ':artista:',
        'it': ':artista:',
        'fa': ':هنرمند:',
        'id': ':seniman:',
        'zh': ':艺术家:',
        'ru': ':художник:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U0001F3A8': {  # 🧑🏿‍🎨
        'en': ':artist_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':künstler(in)_dunkle_hautfarbe:',
        'es': ':artista_tono_de_piel_oscuro:',
        'fr': ':artiste_peau_foncée:',
        'ja': ':芸術家_濃い肌色:',
        'ko': ':화가_검은색_피부:',
        'pt': ':artista_pele_escura:',
        'it': ':artista_carnagione_scura:',
        'fa': ':هنرمند_پوست_آبنوسی:',
        'id': ':seniman_warna_kulit_gelap:',
        'zh': ':艺术家_较深肤色:',
        'ru': ':художник_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U0001F3A8': {  # 🧑🏻‍🎨
        'en': ':artist_light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':künstler(in)_helle_hautfarbe:',
        'es': ':artista_tono_de_piel_claro:',
        'fr': ':artiste_peau_claire:',
        'ja': ':芸術家_薄い肌色:',
        'ko': ':화가_하얀_피부:',
        'pt': ':artista_pele_clara:',
        'it': ':artista_carnagione_chiara:',
        'fa': ':هنرمند_پوست_سفید:',
        'id': ':seniman_warna_kulit_cerah:',
        'zh': ':艺术家_较浅肤色:',
        'ru': ':художник_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U0001F3A8': {  # 🧑🏾‍🎨
        'en': ':artist_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':künstler(in)_mitteldunkle_hautfarbe:',
        'es': ':artista_tono_de_piel_oscuro_medio:',
        'fr': ':artiste_peau_mate:',
        'ja': ':芸術家_やや濃い肌色:',
        'ko': ':화가_진한_갈색_피부:',
        'pt': ':artista_pele_morena_escura:',
        'it': ':artista_carnagione_abbastanza_scura:',
        'fa': ':هنرمند_پوست_گندمی:',
        'id': ':seniman_warna_kulit_gelap-sedang:',
        'zh': ':艺术家_中等深肤色:',
        'ru': ':художник_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U0001F3A8': {  # 🧑🏼‍🎨
        'en': ':artist_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':künstler(in)_mittelhelle_hautfarbe:',
        'es': ':artista_tono_de_piel_claro_medio:',
        'fr': ':artiste_peau_moyennement_claire:',
        'ja': ':芸術家_やや薄い肌色:',
        'ko': ':화가_연한_갈색_피부:',
        'pt': ':artista_pele_morena_clara:',
        'it': ':artista_carnagione_abbastanza_chiara:',
        'fa': ':هنرمند_پوست_روشن:',
        'id': ':seniman_warna_kulit_cerah-sedang:',
        'zh': ':艺术家_中等浅肤色:',
        'ru': ':художник_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U0001F3A8': {  # 🧑🏽‍🎨
        'en': ':artist_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':künstler(in)_mittlere_hautfarbe:',
        'es': ':artista_tono_de_piel_medio:',
        'fr': ':artiste_peau_légèrement_mate:',
        'ja': ':芸術家_中間の肌色:',
        'ko': ':화가_갈색_피부:',
        'pt': ':artista_pele_morena:',
        'it': ':artista_carnagione_olivastra:',
        'fa': ':هنرمند_پوست_طلایی:',
        'id': ':seniman_warna_kulit_sedang:',
        'zh': ':艺术家_中等肤色:',
        'ru': ':художник_средний_тон_кожи:'
    },
    '\U0001F3A8': {  # 🎨
        'en': ':artist_palette:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':art:'],
        'de': ':mischpalette:',
        'es': ':paleta_de_pintor:',
        'fr': ':palette_de_peinture:',
        'ja': ':絵の具パレット:',
        'ko': ':팔레트:',
        'pt': ':paleta_de_tintas:',
        'it': ':tavolozza:',
        'fa': ':پالت_نقاشی:',
        'id': ':palet_pelukis:',
        'zh': ':调色盘:',
        'ru': ':палитра_с_красками:'
    },
    '\U0001F632': {  # 😲
        'en': ':astonished_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':astonished:'],
        'de': ':erstauntes_gesicht:',
        'es': ':cara_asombrada:',
        'fr': ':visage_stupéfait:',
        'ja': ':びっくり:',
        'ko': ':깜짝_놀란_얼굴:',
        'pt': ':rosto_espantado:',
        'it': ':faccina_stupita:',
        'fa': ':بهت_زده:',
        'id': ':wajah_terpukau:',
        'zh': ':震惊:',
        'ru': ':в_шоке:'
    },
    '\U0001F9D1\U0000200D\U0001F680': {  # 🧑‍🚀
        'en': ':astronaut:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':astronaut(in):',
        'es': ':astronauta:',
        'fr': ':astronaute:',
        'ja': ':宇宙飛行士:',
        'ko': ':우주비행사:',
        'pt': ':astronauta:',
        'it': ':astronauta:',
        'fa': ':فضانورد:',
        'id': ':astronaut:',
        'zh': ':宇航员:',
        'ru': ':космонавт:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U0001F680': {  # 🧑🏿‍🚀
        'en': ':astronaut_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':astronaut(in)_dunkle_hautfarbe:',
        'es': ':astronauta_tono_de_piel_oscuro:',
        'fr': ':astronaute_peau_foncée:',
        'ja': ':宇宙飛行士_濃い肌色:',
        'ko': ':우주비행사_검은색_피부:',
        'pt': ':astronauta_pele_escura:',
        'it': ':astronauta_carnagione_scura:',
        'fa': ':فضانورد_پوست_آبنوسی:',
        'id': ':astronaut_warna_kulit_gelap:',
        'zh': ':宇航员_较深肤色:',
        'ru': ':космонавт_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U0001F680': {  # 🧑🏻‍🚀
        'en': ':astronaut_light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':astronaut(in)_helle_hautfarbe:',
        'es': ':astronauta_tono_de_piel_claro:',
        'fr': ':astronaute_peau_claire:',
        'ja': ':宇宙飛行士_薄い肌色:',
        'ko': ':우주비행사_하얀_피부:',
        'pt': ':astronauta_pele_clara:',
        'it': ':astronauta_carnagione_chiara:',
        'fa': ':فضانورد_پوست_سفید:',
        'id': ':astronaut_warna_kulit_cerah:',
        'zh': ':宇航员_较浅肤色:',
        'ru': ':космонавт_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U0001F680': {  # 🧑🏾‍🚀
        'en': ':astronaut_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':astronaut(in)_mitteldunkle_hautfarbe:',
        'es': ':astronauta_tono_de_piel_oscuro_medio:',
        'fr': ':astronaute_peau_mate:',
        'ja': ':宇宙飛行士_やや濃い肌色:',
        'ko': ':우주비행사_진한_갈색_피부:',
        'pt': ':astronauta_pele_morena_escura:',
        'it': ':astronauta_carnagione_abbastanza_scura:',
        'fa': ':فضانورد_پوست_گندمی:',
        'id': ':astronaut_warna_kulit_gelap-sedang:',
        'zh': ':宇航员_中等深肤色:',
        'ru': ':космонавт_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U0001F680': {  # 🧑🏼‍🚀
        'en': ':astronaut_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':astronaut(in)_mittelhelle_hautfarbe:',
        'es': ':astronauta_tono_de_piel_claro_medio:',
        'fr': ':astronaute_peau_moyennement_claire:',
        'ja': ':宇宙飛行士_やや薄い肌色:',
        'ko': ':우주비행사_연한_갈색_피부:',
        'pt': ':astronauta_pele_morena_clara:',
        'it': ':astronauta_carnagione_abbastanza_chiara:',
        'fa': ':فضانورد_پوست_روشن:',
        'id': ':astronaut_warna_kulit_cerah-sedang:',
        'zh': ':宇航员_中等浅肤色:',
        'ru': ':космонавт_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U0001F680': {  # 🧑🏽‍🚀
        'en': ':astronaut_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':astronaut(in)_mittlere_hautfarbe:',
        'es': ':astronauta_tono_de_piel_medio:',
        'fr': ':astronaute_peau_légèrement_mate:',
        'ja': ':宇宙飛行士_中間の肌色:',
        'ko': ':우주비행사_갈색_피부:',
        'pt': ':astronauta_pele_morena:',
        'it': ':astronauta_carnagione_olivastra:',
        'fa': ':فضانورد_پوست_طلایی:',
        'id': ':astronaut_warna_kulit_sedang:',
        'zh': ':宇航员_中等肤色:',
        'ru': ':космонавт_средний_тон_кожи:'
    },
    '\U0000269B\U0000FE0F': {  # ⚛️
        'en': ':atom_symbol:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':atomzeichen:',
        'es': ':símbolo_de_átomo:',
        'fr': ':symbole_de_l’atome:',
        'ja': ':原子のシンボル:',
        'ko': ':원자_기호:',
        'pt': ':símbolo_de_átomo:',
        'it': ':simbolo_dell’atomo:',
        'fa': ':نماد_اتم:',
        'id': ':simbol_atom:',
        'zh': ':原子符号:',
        'ru': ':символ_атома:'
    },
    '\U0000269B': {  # ⚛
        'en': ':atom_symbol:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':atomzeichen:',
        'es': ':símbolo_de_átomo:',
        'fr': ':symbole_de_l’atome:',
        'ja': ':原子のシンボル:',
        'ko': ':원자_기호:',
        'pt': ':símbolo_de_átomo:',
        'it': ':simbolo_dell’atomo:',
        'fa': ':نماد_اتم:',
        'id': ':simbol_atom:',
        'zh': ':原子符号:',
        'ru': ':символ_атома:'
    },
    '\U0001F6FA': {  # 🛺
        'en': ':auto_rickshaw:',
        'status': fully_qualified,
        'E': 12,
        'de': ':autorikscha:',
        'es': ':mototaxi:',
        'fr': ':tuk_tuk:',
        'ja': ':三輪タクシー:',
        'ko': ':경삼륜차:',
        'pt': ':automóvel_riquixá:',
        'it': ':risciò_a_motore:',
        'fa': ':اتوریکشا:',
        'id': ':bajaj:',
        'zh': ':三轮摩托车:',
        'ru': ':авторикша:'
    },
    '\U0001F697': {  # 🚗
        'en': ':automobile:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':car:', ':red_car:'],
        'de': ':auto:',
        'es': ':coche:',
        'fr': ':voiture:',
        'ja': ':自動車:',
        'ko': ':자동차:',
        'pt': ':carro:',
        'it': ':auto:',
        'fa': ':خودرو:',
        'id': ':mobil:',
        'zh': ':汽车:',
        'ru': ':автомобиль:'
    },
    '\U0001F951': {  # 🥑
        'en': ':avocado:',
        'status': fully_qualified,
        'E': 3,
        'de': ':avocado:',
        'es': ':aguacate:',
        'fr': ':avocat:',
        'ja': ':アボカド:',
        'ko': ':아보카도:',
        'pt': ':abacate:',
        'it': ':avocado:',
        'fa': ':آووکادو:',
        'id': ':alpukat:',
        'zh': ':鳄梨:',
        'ru': ':авокадо:'
    },
    '\U0001FA93': {  # 🪓
        'en': ':axe:',
        'status': fully_qualified,
        'E': 12,
        'de': ':axt:',
        'es': ':hacha:',
        'fr': ':hache:',
        'ja': ':斧:',
        'ko': ':도끼:',
        'pt': ':machado:',
        'it': ':ascia:',
        'fa': ':تبر:',
        'id': ':kapak:',
        'zh': ':斧头:',
        'ru': ':топор:'
    },
    '\U0001F476': {  # 👶
        'en': ':baby:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':baby:',
        'es': ':bebé:',
        'fr': ':bébé:',
        'ja': ':赤ん坊:',
        'ko': ':아기:',
        'pt': ':bebê:',
        'it': ':neonato:',
        'fa': ':نوزاد:',
        'id': ':bayi:',
        'zh': ':小宝贝:',
        'ru': ':младенец:'
    },
    '\U0001F47C': {  # 👼
        'en': ':baby_angel:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':angel:'],
        'de': ':putte:',
        'es': ':bebé_ángel:',
        'fr': ':bébé_ange:',
        'ja': ':天使:',
        'ko': ':아기_천사:',
        'pt': ':bebê_anjo:',
        'it': ':angioletto:',
        'fa': ':فرشته_کوچولو:',
        'id': ':malaikat_bayi:',
        'zh': ':小天使:',
        'ru': ':ангелочек:'
    },
    '\U0001F47C\U0001F3FF': {  # 👼🏿
        'en': ':baby_angel_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':putte_dunkle_hautfarbe:',
        'es': ':bebé_ángel_tono_de_piel_oscuro:',
        'fr': ':bébé_ange_peau_foncée:',
        'ja': ':天使_濃い肌色:',
        'ko': ':아기_천사_검은색_피부:',
        'pt': ':bebê_anjo_pele_escura:',
        'it': ':angioletto_carnagione_scura:',
        'fa': ':فرشته_کوچولو_پوست_آبنوسی:',
        'id': ':malaikat_bayi_warna_kulit_gelap:',
        'zh': ':小天使_较深肤色:',
        'ru': ':ангелочек_очень_темный_тон_кожи:'
    },
    '\U0001F47C\U0001F3FB': {  # 👼🏻
        'en': ':baby_angel_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':putte_helle_hautfarbe:',
        'es': ':bebé_ángel_tono_de_piel_claro:',
        'fr': ':bébé_ange_peau_claire:',
        'ja': ':天使_薄い肌色:',
        'ko': ':아기_천사_하얀_피부:',
        'pt': ':bebê_anjo_pele_clara:',
        'it': ':angioletto_carnagione_chiara:',
        'fa': ':فرشته_کوچولو_پوست_سفید:',
        'id': ':malaikat_bayi_warna_kulit_cerah:',
        'zh': ':小天使_较浅肤色:',
        'ru': ':ангелочек_очень_светлый_тон_кожи:'
    },
    '\U0001F47C\U0001F3FE': {  # 👼🏾
        'en': ':baby_angel_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':putte_mitteldunkle_hautfarbe:',
        'es': ':bebé_ángel_tono_de_piel_oscuro_medio:',
        'fr': ':bébé_ange_peau_mate:',
        'ja': ':天使_やや濃い肌色:',
        'ko': ':아기_천사_진한_갈색_피부:',
        'pt': ':bebê_anjo_pele_morena_escura:',
        'it': ':angioletto_carnagione_abbastanza_scura:',
        'fa': ':فرشته_کوچولو_پوست_گندمی:',
        'id': ':malaikat_bayi_warna_kulit_gelap-sedang:',
        'zh': ':小天使_中等深肤色:',
        'ru': ':ангелочек_темный_тон_кожи:'
    },
    '\U0001F47C\U0001F3FC': {  # 👼🏼
        'en': ':baby_angel_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':putte_mittelhelle_hautfarbe:',
        'es': ':bebé_ángel_tono_de_piel_claro_medio:',
        'fr': ':bébé_ange_peau_moyennement_claire:',
        'ja': ':天使_やや薄い肌色:',
        'ko': ':아기_천사_연한_갈색_피부:',
        'pt': ':bebê_anjo_pele_morena_clara:',
        'it': ':angioletto_carnagione_abbastanza_chiara:',
        'fa': ':فرشته_کوچولو_پوست_روشن:',
        'id': ':malaikat_bayi_warna_kulit_cerah-sedang:',
        'zh': ':小天使_中等浅肤色:',
        'ru': ':ангелочек_светлый_тон_кожи:'
    },
    '\U0001F47C\U0001F3FD': {  # 👼🏽
        'en': ':baby_angel_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':putte_mittlere_hautfarbe:',
        'es': ':bebé_ángel_tono_de_piel_medio:',
        'fr': ':bébé_ange_peau_légèrement_mate:',
        'ja': ':天使_中間の肌色:',
        'ko': ':아기_천사_갈색_피부:',
        'pt': ':bebê_anjo_pele_morena:',
        'it': ':angioletto_carnagione_olivastra:',
        'fa': ':فرشته_کوچولو_پوست_طلایی:',
        'id': ':malaikat_bayi_warna_kulit_sedang:',
        'zh': ':小天使_中等肤色:',
        'ru': ':ангелочек_средний_тон_кожи:'
    },
    '\U0001F37C': {  # 🍼
        'en': ':baby_bottle:',
        'status': fully_qualified,
        'E': 1,
        'de': ':babyflasche:',
        'es': ':biberón:',
        'fr': ':biberon:',
        'ja': ':ほにゅう瓶:',
        'ko': ':젖병:',
        'pt': ':mamadeira:',
        'it': ':biberon:',
        'fa': ':شیشه_شیر:',
        'id': ':botol_bayi:',
        'zh': ':奶瓶:',
        'ru': ':бутылочка_с_соской:'
    },
    '\U0001F424': {  # 🐤
        'en': ':baby_chick:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':küken:',
        'es': ':pollito:',
        'fr': ':poussin:',
        'ja': ':ひよこ:',
        'ko': ':병아리:',
        'pt': ':pintinho_de_perfil:',
        'it': ':pulcino:',
        'fa': ':جوجه:',
        'id': ':itik_bayi:',
        'zh': ':小鸡:',
        'ru': ':цыпленок_боком:'
    },
    '\U0001F476\U0001F3FF': {  # 👶🏿
        'en': ':baby_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':baby_dunkle_hautfarbe:',
        'es': ':bebé_tono_de_piel_oscuro:',
        'fr': ':bébé_peau_foncée:',
        'ja': ':赤ん坊_濃い肌色:',
        'ko': ':아기_검은색_피부:',
        'pt': ':bebê_pele_escura:',
        'it': ':neonato_carnagione_scura:',
        'fa': ':نوزاد_پوست_آبنوسی:',
        'id': ':bayi_warna_kulit_gelap:',
        'zh': ':小宝贝_较深肤色:',
        'ru': ':младенец_очень_темный_тон_кожи:'
    },
    '\U0001F476\U0001F3FB': {  # 👶🏻
        'en': ':baby_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':baby_helle_hautfarbe:',
        'es': ':bebé_tono_de_piel_claro:',
        'fr': ':bébé_peau_claire:',
        'ja': ':赤ん坊_薄い肌色:',
        'ko': ':아기_하얀_피부:',
        'pt': ':bebê_pele_clara:',
        'it': ':neonato_carnagione_chiara:',
        'fa': ':نوزاد_پوست_سفید:',
        'id': ':bayi_warna_kulit_cerah:',
        'zh': ':小宝贝_较浅肤色:',
        'ru': ':младенец_очень_светлый_тон_кожи:'
    },
    '\U0001F476\U0001F3FE': {  # 👶🏾
        'en': ':baby_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':baby_mitteldunkle_hautfarbe:',
        'es': ':bebé_tono_de_piel_oscuro_medio:',
        'fr': ':bébé_peau_mate:',
        'ja': ':赤ん坊_やや濃い肌色:',
        'ko': ':아기_진한_갈색_피부:',
        'pt': ':bebê_pele_morena_escura:',
        'it': ':neonato_carnagione_abbastanza_scura:',
        'fa': ':نوزاد_پوست_گندمی:',
        'id': ':bayi_warna_kulit_gelap-sedang:',
        'zh': ':小宝贝_中等深肤色:',
        'ru': ':младенец_темный_тон_кожи:'
    },
    '\U0001F476\U0001F3FC': {  # 👶🏼
        'en': ':baby_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':baby_mittelhelle_hautfarbe:',
        'es': ':bebé_tono_de_piel_claro_medio:',
        'fr': ':bébé_peau_moyennement_claire:',
        'ja': ':赤ん坊_やや薄い肌色:',
        'ko': ':아기_연한_갈색_피부:',
        'pt': ':bebê_pele_morena_clara:',
        'it': ':neonato_carnagione_abbastanza_chiara:',
        'fa': ':نوزاد_پوست_روشن:',
        'id': ':bayi_warna_kulit_cerah-sedang:',
        'zh': ':小宝贝_中等浅肤色:',
        'ru': ':младенец_светлый_тон_кожи:'
    },
    '\U0001F476\U0001F3FD': {  # 👶🏽
        'en': ':baby_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':baby_mittlere_hautfarbe:',
        'es': ':bebé_tono_de_piel_medio:',
        'fr': ':bébé_peau_légèrement_mate:',
        'ja': ':赤ん坊_中間の肌色:',
        'ko': ':아기_갈색_피부:',
        'pt': ':bebê_pele_morena:',
        'it': ':neonato_carnagione_olivastra:',
        'fa': ':نوزاد_پوست_طلایی:',
        'id': ':bayi_warna_kulit_sedang:',
        'zh': ':小宝贝_中等肤色:',
        'ru': ':младенец_средний_тон_кожи:'
    },
    '\U0001F6BC': {  # 🚼
        'en': ':baby_symbol:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':symbol_baby:',
        'es': ':señal_de_bebé:',
        'fr': ':symbole_bébé:',
        'ja': ':ベビーシート:',
        'ko': ':수유실:',
        'pt': ':símbolo_de_bebê:',
        'it': ':simbolo_con_immagine_di_bambino:',
        'fa': ':نماد_نوزاد:',
        'id': ':simbol_bayi:',
        'zh': ':宝宝:',
        'ru': ':пеленальная:'
    },
    '\U0001F447': {  # 👇
        'en': ':backhand_index_pointing_down:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':point_down:'],
        'variant': True,
        'de': ':nach_unten_weisender_zeigefinger:',
        'es': ':dorso_de_mano_con_índice_hacia_abajo:',
        'fr': ':main_avec_index_pointant_vers_le_bas:',
        'ja': ':下指差し:',
        'ko': ':아래쪽을_가리키는_손등:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_baixo:',
        'it': ':indice_abbassato:',
        'fa': ':اشاره_به_پایین:',
        'id': ':menunjuk_ke_bawah:',
        'zh': ':反手食指向下指:',
        'ru': ':указывающий_вниз_палец:'
    },
    '\U0001F447\U0001F3FF': {  # 👇🏿
        'en': ':backhand_index_pointing_down_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_unten_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_oscuro:',
        'fr': ':main_avec_index_pointant_vers_le_bas_peau_foncée:',
        'ja': ':下指差し_濃い肌色:',
        'ko': ':아래쪽을_가리키는_손등_검은색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_escura:',
        'it': ':indice_abbassato_carnagione_scura:',
        'fa': ':اشاره_به_پایین_پوست_آبنوسی:',
        'id': ':menunjuk_ke_bawah_warna_kulit_gelap:',
        'zh': ':反手食指向下指_较深肤色:',
        'ru': ':указывающий_вниз_палец_очень_темный_тон_кожи:'
    },
    '\U0001F447\U0001F3FB': {  # 👇🏻
        'en': ':backhand_index_pointing_down_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_unten_weisender_zeigefinger_helle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_claro:',
        'fr': ':main_avec_index_pointant_vers_le_bas_peau_claire:',
        'ja': ':下指差し_薄い肌色:',
        'ko': ':아래쪽을_가리키는_손등_하얀_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_clara:',
        'it': ':indice_abbassato_carnagione_chiara:',
        'fa': ':اشاره_به_پایین_پوست_سفید:',
        'id': ':menunjuk_ke_bawah_warna_kulit_cerah:',
        'zh': ':反手食指向下指_较浅肤色:',
        'ru': ':указывающий_вниз_палец_очень_светлый_тон_кожи:'
    },
    '\U0001F447\U0001F3FE': {  # 👇🏾
        'en': ':backhand_index_pointing_down_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_unten_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_oscuro_medio:',
        'fr': ':main_avec_index_pointant_vers_le_bas_peau_mate:',
        'ja': ':下指差し_やや濃い肌色:',
        'ko': ':아래쪽을_가리키는_손등_진한_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena_escura:',
        'it': ':indice_abbassato_carnagione_abbastanza_scura:',
        'fa': ':اشاره_به_پایین_پوست_گندمی:',
        'id': ':menunjuk_ke_bawah_warna_kulit_gelap-sedang:',
        'zh': ':反手食指向下指_中等深肤色:',
        'ru': ':указывающий_вниз_палец_темный_тон_кожи:'
    },
    '\U0001F447\U0001F3FC': {  # 👇🏼
        'en': ':backhand_index_pointing_down_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_unten_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_claro_medio:',
        'fr': ':main_avec_index_pointant_vers_le_bas_peau_moyennement_claire:',
        'ja': ':下指差し_やや薄い肌色:',
        'ko': ':아래쪽을_가리키는_손등_연한_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena_clara:',
        'it': ':indice_abbassato_carnagione_abbastanza_chiara:',
        'fa': ':اشاره_به_پایین_پوست_روشن:',
        'id': ':menunjuk_ke_bawah_warna_kulit_cerah-sedang:',
        'zh': ':反手食指向下指_中等浅肤色:',
        'ru': ':указывающий_вниз_палец_светлый_тон_кожи:'
    },
    '\U0001F447\U0001F3FD': {  # 👇🏽
        'en': ':backhand_index_pointing_down_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_unten_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_medio:',
        'fr': ':main_avec_index_pointant_vers_le_bas_peau_légèrement_mate:',
        'ja': ':下指差し_中間の肌色:',
        'ko': ':아래쪽을_가리키는_손등_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena:',
        'it': ':indice_abbassato_carnagione_olivastra:',
        'fa': ':اشاره_به_پایین_پوست_طلایی:',
        'id': ':menunjuk_ke_bawah_warna_kulit_sedang:',
        'zh': ':反手食指向下指_中等肤色:',
        'ru': ':указывающий_вниз_палец_средний_тон_кожи:'
    },
    '\U0001F448': {  # 👈
        'en': ':backhand_index_pointing_left:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':point_left:'],
        'variant': True,
        'de': ':nach_links_weisender_zeigefinger:',
        'es': ':dorso_de_mano_con_índice_a_la_izquierda:',
        'fr': ':main_avec_index_pointant_à_gauche:',
        'ja': ':左指差し:',
        'ko': ':왼쪽을_가리키는_손등:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda:',
        'it': ':indice_verso_sinistra:',
        'fa': ':اشاره_به_چپ:',
        'id': ':menunjuk_ke_arah_kiri:',
        'zh': ':反手食指向左指:',
        'ru': ':указывающий_налево_палец:'
    },
    '\U0001F448\U0001F3FF': {  # 👈🏿
        'en': ':backhand_index_pointing_left_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_links_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_oscuro:',
        'fr': ':main_avec_index_pointant_à_gauche_peau_foncée:',
        'ja': ':左指差し_濃い肌色:',
        'ko': ':왼쪽을_가리키는_손등_검은색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_escura:',
        'it': ':indice_verso_sinistra_carnagione_scura:',
        'fa': ':اشاره_به_چپ_پوست_آبنوسی:',
        'id': ':menunjuk_ke_arah_kiri_warna_kulit_gelap:',
        'zh': ':反手食指向左指_较深肤色:',
        'ru': ':указывающий_налево_палец_очень_темный_тон_кожи:'
    },
    '\U0001F448\U0001F3FB': {  # 👈🏻
        'en': ':backhand_index_pointing_left_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_links_weisender_zeigefinger_helle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_claro:',
        'fr': ':main_avec_index_pointant_à_gauche_peau_claire:',
        'ja': ':左指差し_薄い肌色:',
        'ko': ':왼쪽을_가리키는_손등_하얀_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_clara:',
        'it': ':indice_verso_sinistra_carnagione_chiara:',
        'fa': ':اشاره_به_چپ_پوست_سفید:',
        'id': ':menunjuk_ke_arah_kiri_warna_kulit_cerah:',
        'zh': ':反手食指向左指_较浅肤色:',
        'ru': ':указывающий_налево_палец_очень_светлый_тон_кожи:'
    },
    '\U0001F448\U0001F3FE': {  # 👈🏾
        'en': ':backhand_index_pointing_left_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_links_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_oscuro_medio:',
        'fr': ':main_avec_index_pointant_à_gauche_peau_mate:',
        'ja': ':左指差し_やや濃い肌色:',
        'ko': ':왼쪽을_가리키는_손등_진한_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena_escura:',
        'it': ':indice_verso_sinistra_carnagione_abbastanza_scura:',
        'fa': ':اشاره_به_چپ_پوست_گندمی:',
        'id': ':menunjuk_ke_arah_kiri_warna_kulit_gelap-sedang:',
        'zh': ':反手食指向左指_中等深肤色:',
        'ru': ':указывающий_налево_палец_темный_тон_кожи:'
    },
    '\U0001F448\U0001F3FC': {  # 👈🏼
        'en': ':backhand_index_pointing_left_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_links_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_claro_medio:',
        'fr': ':main_avec_index_pointant_à_gauche_peau_moyennement_claire:',
        'ja': ':左指差し_やや薄い肌色:',
        'ko': ':왼쪽을_가리키는_손등_연한_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena_clara:',
        'it': ':indice_verso_sinistra_carnagione_abbastanza_chiara:',
        'fa': ':اشاره_به_چپ_پوست_روشن:',
        'id': ':menunjuk_ke_arah_kiri_warna_kulit_cerah-sedang:',
        'zh': ':反手食指向左指_中等浅肤色:',
        'ru': ':указывающий_налево_палец_светлый_тон_кожи:'
    },
    '\U0001F448\U0001F3FD': {  # 👈🏽
        'en': ':backhand_index_pointing_left_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_links_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_medio:',
        'fr': ':main_avec_index_pointant_à_gauche_peau_légèrement_mate:',
        'ja': ':左指差し_中間の肌色:',
        'ko': ':왼쪽을_가리키는_손등_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena:',
        'it': ':indice_verso_sinistra_carnagione_olivastra:',
        'fa': ':اشاره_به_چپ_پوست_طلایی:',
        'id': ':menunjuk_ke_arah_kiri_warna_kulit_sedang:',
        'zh': ':反手食指向左指_中等肤色:',
        'ru': ':указывающий_налево_палец_средний_тон_кожи:'
    },
    '\U0001F449': {  # 👉
        'en': ':backhand_index_pointing_right:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':point_right:'],
        'variant': True,
        'de': ':nach_rechts_weisender_zeigefinger:',
        'es': ':dorso_de_mano_con_índice_a_la_derecha:',
        'fr': ':main_avec_index_pointant_à_droite:',
        'ja': ':右指差し:',
        'ko': ':오른쪽을_가리키는_손등:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita:',
        'it': ':indice_verso_destra:',
        'fa': ':اشاره_به_راست:',
        'id': ':menunjuk_ke_arah_kanan:',
        'zh': ':反手食指向右指:',
        'ru': ':указывающий_направо_палец:'
    },
    '\U0001F449\U0001F3FF': {  # 👉🏿
        'en': ':backhand_index_pointing_right_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_rechts_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_oscuro:',
        'fr': ':main_avec_index_pointant_à_droite_peau_foncée:',
        'ja': ':右指差し_濃い肌色:',
        'ko': ':오른쪽을_가리키는_손등_검은색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_escura:',
        'it': ':indice_verso_destra_carnagione_scura:',
        'fa': ':اشاره_به_راست_پوست_آبنوسی:',
        'id': ':menunjuk_ke_arah_kanan_warna_kulit_gelap:',
        'zh': ':反手食指向右指_较深肤色:',
        'ru': ':указывающий_направо_палец_очень_темный_тон_кожи:'
    },
    '\U0001F449\U0001F3FB': {  # 👉🏻
        'en': ':backhand_index_pointing_right_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_rechts_weisender_zeigefinger_helle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_claro:',
        'fr': ':main_avec_index_pointant_à_droite_peau_claire:',
        'ja': ':右指差し_薄い肌色:',
        'ko': ':오른쪽을_가리키는_손등_하얀_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_clara:',
        'it': ':indice_verso_destra_carnagione_chiara:',
        'fa': ':اشاره_به_راست_پوست_سفید:',
        'id': ':menunjuk_ke_arah_kanan_warna_kulit_cerah:',
        'zh': ':反手食指向右指_较浅肤色:',
        'ru': ':указывающий_направо_палец_очень_светлый_тон_кожи:'
    },
    '\U0001F449\U0001F3FE': {  # 👉🏾
        'en': ':backhand_index_pointing_right_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_rechts_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_oscuro_medio:',
        'fr': ':main_avec_index_pointant_à_droite_peau_mate:',
        'ja': ':右指差し_やや濃い肌色:',
        'ko': ':오른쪽을_가리키는_손등_진한_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena_escura:',
        'it': ':indice_verso_destra_carnagione_abbastanza_scura:',
        'fa': ':اشاره_به_راست_پوست_گندمی:',
        'id': ':menunjuk_ke_arah_kanan_warna_kulit_gelap-sedang:',
        'zh': ':反手食指向右指_中等深肤色:',
        'ru': ':указывающий_направо_палец_темный_тон_кожи:'
    },
    '\U0001F449\U0001F3FC': {  # 👉🏼
        'en': ':backhand_index_pointing_right_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_rechts_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_claro_medio:',
        'fr': ':main_avec_index_pointant_à_droite_peau_moyennement_claire:',
        'ja': ':右指差し_やや薄い肌色:',
        'ko': ':오른쪽을_가리키는_손등_연한_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena_clara:',
        'it': ':indice_verso_destra_carnagione_abbastanza_chiara:',
        'fa': ':اشاره_به_راست_پوست_روشن:',
        'id': ':menunjuk_ke_arah_kanan_warna_kulit_cerah-sedang:',
        'zh': ':反手食指向右指_中等浅肤色:',
        'ru': ':указывающий_направо_палец_светлый_тон_кожи:'
    },
    '\U0001F449\U0001F3FD': {  # 👉🏽
        'en': ':backhand_index_pointing_right_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_rechts_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_medio:',
        'fr': ':main_avec_index_pointant_à_droite_peau_légèrement_mate:',
        'ja': ':右指差し_中間の肌色:',
        'ko': ':오른쪽을_가리키는_손등_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena:',
        'it': ':indice_verso_destra_carnagione_olivastra:',
        'fa': ':اشاره_به_راست_پوست_طلایی:',
        'id': ':menunjuk_ke_arah_kanan_warna_kulit_sedang:',
        'zh': ':反手食指向右指_中等肤色:',
        'ru': ':указывающий_направо_палец_средний_тон_кожи:'
    },
    '\U0001F446': {  # 👆
        'en': ':backhand_index_pointing_up:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':point_up_2:'],
        'variant': True,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten:',
        'es': ':dorso_de_mano_con_índice_hacia_arriba:',
        'fr': ':main_avec_index_pointant_vers_le_haut:',
        'ja': ':手の甲上指差し:',
        'ko': ':위쪽을_가리키는_손등:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_cima:',
        'it': ':indice_alzato:',
        'fa': ':اشاره_به_بالا_با_پشت_دست:',
        'id': ':menunjuk_ke_atas_(dibalik):',
        'zh': ':反手食指向上指:',
        'ru': ':указывающий_вверх_палец:'
    },
    '\U0001F446\U0001F3FF': {  # 👆🏿
        'en': ':backhand_index_pointing_up_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_dunkle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_oscuro:',
        'fr': ':main_avec_index_pointant_vers_le_haut_peau_foncée:',
        'ja': ':手の甲上指差し_濃い肌色:',
        'ko': ':위쪽을_가리키는_손등_검은색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_escura:',
        'it': ':indice_alzato_carnagione_scura:',
        'fa': ':اشاره_به_بالا_با_پشت_دست_پوست_آبنوسی:',
        'id': ':menunjuk_ke_atas_(dibalik)_warna_kulit_gelap:',
        'zh': ':反手食指向上指_较深肤色:',
        'ru': ':указывающий_вверх_палец_очень_темный_тон_кожи:'
    },
    '\U0001F446\U0001F3FB': {  # 👆🏻
        'en': ':backhand_index_pointing_up_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_helle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_claro:',
        'fr': ':main_avec_index_pointant_vers_le_haut_peau_claire:',
        'ja': ':手の甲上指差し_薄い肌色:',
        'ko': ':위쪽을_가리키는_손등_하얀_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_clara:',
        'it': ':indice_alzato_carnagione_chiara:',
        'fa': ':اشاره_به_بالا_با_پشت_دست_پوست_سفید:',
        'id': ':menunjuk_ke_atas_(dibalik)_warna_kulit_cerah:',
        'zh': ':反手食指向上指_较浅肤色:',
        'ru': ':указывающий_вверх_палец_очень_светлый_тон_кожи:'
    },
    '\U0001F446\U0001F3FE': {  # 👆🏾
        'en': ':backhand_index_pointing_up_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_mitteldunkle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_oscuro_medio:',
        'fr': ':main_avec_index_pointant_vers_le_haut_peau_mate:',
        'ja': ':手の甲上指差し_やや濃い肌色:',
        'ko': ':위쪽을_가리키는_손등_진한_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena_escura:',
        'it': ':indice_alzato_carnagione_abbastanza_scura:',
        'fa': ':اشاره_به_بالا_با_پشت_دست_پوست_گندمی:',
        'id': ':menunjuk_ke_atas_(dibalik)_warna_kulit_gelap-sedang:',
        'zh': ':反手食指向上指_中等深肤色:',
        'ru': ':указывающий_вверх_палец_темный_тон_кожи:'
    },
    '\U0001F446\U0001F3FC': {  # 👆🏼
        'en': ':backhand_index_pointing_up_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_mittelhelle_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_claro_medio:',
        'fr': ':main_avec_index_pointant_vers_le_haut_peau_moyennement_claire:',
        'ja': ':手の甲上指差し_やや薄い肌色:',
        'ko': ':위쪽을_가리키는_손등_연한_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena_clara:',
        'it': ':indice_alzato_carnagione_abbastanza_chiara:',
        'fa': ':اشاره_به_بالا_با_پشت_دست_پوست_روشن:',
        'id': ':menunjuk_ke_atas_(dibalik)_warna_kulit_cerah-sedang:',
        'zh': ':反手食指向上指_中等浅肤色:',
        'ru': ':указывающий_вверх_палец_светлый_тон_кожи:'
    },
    '\U0001F446\U0001F3FD': {  # 👆🏽
        'en': ':backhand_index_pointing_up_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_mittlere_hautfarbe:',
        'es': ':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_medio:',
        'fr': ':main_avec_index_pointant_vers_le_haut_peau_légèrement_mate:',
        'ja': ':手の甲上指差し_中間の肌色:',
        'ko': ':위쪽을_가리키는_손등_갈색_피부:',
        'pt': ':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena:',
        'it': ':indice_alzato_carnagione_olivastra:',
        'fa': ':اشاره_به_بالا_با_پشت_دست_پوست_طلایی:',
        'id': ':menunjuk_ke_atas_(dibalik)_warna_kulit_sedang:',
        'zh': ':反手食指向上指_中等肤色:',
        'ru': ':указывающий_вверх_палец_средний_тон_кожи:'
    },
    '\U0001F392': {  # 🎒
        'en': ':backpack:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':school_satchel:'],
        'de': ':schulranzen:',
        'es': ':mochila_escolar:',
        'fr': ':cartable:',
        'ja': ':バックパック:',
        'ko': ':학교_가방:',
        'pt': ':mochila:',
        'it': ':zaino:',
        'fa': ':کوله_پشتی:',
        'id': ':tas_sekolah:',
        'zh': ':书包:',
        'ru': ':ранец:'
    },
    '\U0001F953': {  # 🥓
        'en': ':bacon:',
        'status': fully_qualified,
        'E': 3,
        'de': ':bacon:',
        'es': ':beicon:',
        'fr': ':lard:',
        'ja': ':ベーコン:',
        'ko': ':베이컨:',
        'pt': ':bacon:',
        'it': ':pancetta:',
        'fa': ':بیکن:',
        'id': ':babi_asap:',
        'zh': ':培根:',
        'ru': ':бекон:'
    },
    '\U0001F9A1': {  # 🦡
        'en': ':badger:',
        'status': fully_qualified,
        'E': 11,
        'de': ':dachs:',
        'es': ':tejón:',
        'fr': ':blaireau:',
        'ja': ':アナグマ:',
        'ko': ':오소리:',
        'pt': ':texugo:',
        'it': ':tasso:',
        'fa': ':گورکن:',
        'id': ':musang:',
        'zh': ':獾:',
        'ru': ':барсук:'
    },
    '\U0001F3F8': {  # 🏸
        'en': ':badminton:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':badminton_racquet_and_shuttlecock:'],
        'de': ':badminton:',
        'es': ':bádminton:',
        'fr': ':badminton:',
        'ja': ':バドミントン:',
        'ko': ':배드민턴:',
        'pt': ':badminton:',
        'it': ':badminton:',
        'fa': ':بدمینتون:',
        'id': ':bulu_tangkis:',
        'zh': ':羽毛球:',
        'ru': ':бадминтон:'
    },
    '\U0001F96F': {  # 🥯
        'en': ':bagel:',
        'status': fully_qualified,
        'E': 11,
        'de': ':bagel:',
        'es': ':bagel:',
        'fr': ':bagel:',
        'ja': ':ベーグル:',
        'ko': ':베이글:',
        'pt': ':rosca:',
        'it': ':bagel:',
        'fa': ':نان_بیگل:',
        'id': ':bagel:',
        'zh': ':面包圈:',
        'ru': ':бублик:'
    },
    '\U0001F6C4': {  # 🛄
        'en': ':baggage_claim:',
        'status': fully_qualified,
        'E': 1,
        'de': ':gepäckausgabe:',
        'es': ':recogida_de_equipajes:',
        'fr': ':retrait_des_bagages:',
        'ja': ':手荷物受取:',
        'ko': ':짐_찾는_곳:',
        'pt': ':restituição_de_bagagem:',
        'it': ':simbolo_del_ritiro_bagagli:',
        'fa': ':دریافت_بار:',
        'id': ':pengambilan_bagasi:',
        'zh': ':提取行李:',
        'ru': ':получение_багажа:'
    },
    '\U0001F956': {  # 🥖
        'en': ':baguette_bread:',
        'status': fully_qualified,
        'E': 3,
        'de': ':baguette:',
        'es': ':baguete:',
        'fr': ':baguette:',
        'ja': ':バゲット:',
        'ko': ':바게트:',
        'pt': ':baguete:',
        'it': ':baguette:',
        'fa': ':نان_باگت:',
        'id': ':roti_baguette:',
        'zh': ':法式长棍面包:',
        'ru': ':багет:'
    },
    '\U00002696\U0000FE0F': {  # ⚖️
        'en': ':balance_scale:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':scales:'],
        'variant': True,
        'de': ':waage:',
        'es': ':balanza:',
        'fr': ':balance_à_poids:',
        'ja': ':天秤:',
        'ko': ':접시저울:',
        'pt': ':balança:',
        'it': ':bilancia_a_doppio_piatto:',
        'fa': ':ترازو:',
        'id': ':timbangan_gantung:',
        'zh': ':天平:',
        'ru': ':весы:'
    },
    '\U00002696': {  # ⚖
        'en': ':balance_scale:',
        'status': unqualified,
        'E': 1,
        'alias': [':scales:'],
        'variant': True,
        'de': ':waage:',
        'es': ':balanza:',
        'fr': ':balance_à_poids:',
        'ja': ':天秤:',
        'ko': ':접시저울:',
        'pt': ':balança:',
        'it': ':bilancia_a_doppio_piatto:',
        'fa': ':ترازو:',
        'id': ':timbangan_gantung:',
        'zh': ':天平:',
        'ru': ':весы:'
    },
    '\U0001F9B2': {  # 🦲
        'en': ':bald:',
        'status': component,
        'E': 11,
        'de': ':glatze:',
        'es': ':sin_pelo:',
        'fr': ':chauve:',
        'ja': ':はげ頭:',
        'ko': ':대머리:',
        'pt': ':careca:',
        'it': ':calvo:',
        'fa': ':تاس:',
        'id': ':gundul:',
        'zh': ':秃顶:',
        'ru': ':без_волос:'
    },
    '\U0001FA70': {  # 🩰
        'en': ':ballet_shoes:',
        'status': fully_qualified,
        'E': 12,
        'de': ':ballettschuhe:',
        'es': ':zapatillas_de_ballet:',
        'fr': ':chaussons_de_danse:',
        'ja': ':トウシューズ:',
        'ko': ':발레_슈즈:',
        'pt': ':sapatilha_de_balé:',
        'it': ':scarpette_da_ballerina:',
        'fa': ':کفش_باله:',
        'id': ':sepatu_balet:',
        'zh': ':芭蕾舞鞋:',
        'ru': ':пуанты:'
    },
    '\U0001F388': {  # 🎈
        'en': ':balloon:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':luftballon:',
        'es': ':globo:',
        'fr': ':ballon_gonflable:',
        'ja': ':風船:',
        'ko': ':풍선:',
        'pt': ':balão:',
        'it': ':palloncino:',
        'fa': ':بادکنک:',
        'id': ':balon:',
        'zh': ':气球:',
        'ru': ':воздушный_шарик:'
    },
    '\U0001F5F3\U0000FE0F': {  # 🗳️
        'en': ':ballot_box_with_ballot:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':ballot_box:'],
        'variant': True,
        'de': ':urne_mit_wahlzettel:',
        'es': ':urna_con_papeleta:',
        'fr': ':urne_électorale:',
        'ja': ':投票箱:',
        'ko': ':투표_용지가_있는_투표_상자:',
        'pt': ':urna_eleitoral_com_cédula:',
        'it': ':urna_elettorale_con_scheda:',
        'fa': ':صندوق_رأی:',
        'id': ':kotak_dengan_surat_suara:',
        'zh': ':投票箱:',
        'ru': ':избирательная_урна:'
    },
    '\U0001F5F3': {  # 🗳
        'en': ':ballot_box_with_ballot:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':ballot_box:'],
        'variant': True,
        'de': ':urne_mit_wahlzettel:',
        'es': ':urna_con_papeleta:',
        'fr': ':urne_électorale:',
        'ja': ':投票箱:',
        'ko': ':투표_용지가_있는_투표_상자:',
        'pt': ':urna_eleitoral_com_cédula:',
        'it': ':urna_elettorale_con_scheda:',
        'fa': ':صندوق_رأی:',
        'id': ':kotak_dengan_surat_suara:',
        'zh': ':投票箱:',
        'ru': ':избирательная_урна:'
    },
    '\U0001F34C': {  # 🍌
        'en': ':banana:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':banane:',
        'es': ':plátano:',
        'fr': ':banane:',
        'ja': ':バナナ:',
        'ko': ':바나나:',
        'pt': ':banana:',
        'it': ':banana:',
        'fa': ':موز:',
        'id': ':pisang:',
        'zh': ':香蕉:',
        'ru': ':банан:'
    },
    '\U0001FA95': {  # 🪕
        'en': ':banjo:',
        'status': fully_qualified,
        'E': 12,
        'de': ':banjo:',
        'es': ':banjo:',
        'fr': ':banjo:',
        'ja': ':バンジョー:',
        'ko': ':밴조:',
        'pt': ':banjo:',
        'it': ':banjo:',
        'fa': ':بانجو:',
        'id': ':banjo:',
        'zh': ':班卓琴:',
        'ru': ':банджо:'
    },
    '\U0001F3E6': {  # 🏦
        'en': ':bank:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':bank:',
        'es': ':banco:',
        'fr': ':banque:',
        'ja': ':銀行:',
        'ko': ':은행:',
        'pt': ':banco:',
        'it': ':banca:',
        'fa': ':بانک:',
        'id': ':bank:',
        'zh': ':银行:',
        'ru': ':банк:'
    },
    '\U0001F4CA': {  # 📊
        'en': ':bar_chart:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':balkendiagramm:',
        'es': ':gráfico_de_barras:',
        'fr': ':graphiques_à_barres:',
        'ja': ':棒グラフ:',
        'ko': ':막대_그래프:',
        'pt': ':gráfico_de_barras:',
        'it': ':grafico_a_barre:',
        'fa': ':نمودار_ستونی:',
        'id': ':diagram_batang:',
        'zh': ':条形图:',
        'ru': ':гистограмма:'
    },
    '\U0001F488': {  # 💈
        'en': ':barber_pole:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':barber:'],
        'de': ':barbershop-säule:',
        'es': ':poste_de_barbero:',
        'fr': ':enseigne_de_barbier:',
        'ja': ':床屋:',
        'ko': ':이발소:',
        'pt': ':barbearia:',
        'it': ':barbiere:',
        'fa': ':چراغ_سلمانی:',
        'id': ':tanda_salon:',
        'zh': ':理发店:',
        'ru': ':парикмахерская:'
    },
    '\U000026BE': {  # ⚾
        'en': ':baseball:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':baseball:',
        'es': ':béisbol:',
        'fr': ':baseball:',
        'ja': ':野球:',
        'ko': ':야구공:',
        'pt': ':bola_de_beisebol:',
        'it': ':palla_da_baseball:',
        'fa': ':بیسبال:',
        'id': ':bisbol:',
        'zh': ':棒球:',
        'ru': ':бейсбол:'
    },
    '\U0001F9FA': {  # 🧺
        'en': ':basket:',
        'status': fully_qualified,
        'E': 11,
        'de': ':korb:',
        'es': ':cesta:',
        'fr': ':panier:',
        'ja': ':かご:',
        'ko': ':바구니:',
        'pt': ':cesta:',
        'it': ':cesto:',
        'fa': ':سبد:',
        'id': ':keranjang:',
        'zh': ':筐:',
        'ru': ':корзина:'
    },
    '\U0001F3C0': {  # 🏀
        'en': ':basketball:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':basketball:',
        'es': ':balón_de_baloncesto:',
        'fr': ':basket:',
        'ja': ':バスケットボール:',
        'ko': ':농구:',
        'pt': ':bola_de_basquete:',
        'it': ':palla_da_pallacanestro:',
        'fa': ':بسکتبال:',
        'id': ':bola_basket:',
        'zh': ':篮球:',
        'ru': ':баскетбол:'
    },
    '\U0001F987': {  # 🦇
        'en': ':bat:',
        'status': fully_qualified,
        'E': 3,
        'de': ':fledermaus:',
        'es': ':murciélago:',
        'fr': ':chauve-souris:',
        'ja': ':コウモリ:',
        'ko': ':박쥐:',
        'pt': ':morcego:',
        'it': ':pipistrello:',
        'fa': ':خفاش:',
        'id': ':kelelawar:',
        'zh': ':蝙蝠:',
        'ru': ':летучая_мышь:'
    },
    '\U0001F6C1': {  # 🛁
        'en': ':bathtub:',
        'status': fully_qualified,
        'E': 1,
        'de': ':badewanne:',
        'es': ':bañera:',
        'fr': ':baignoire:',
        'ja': ':バスタブ:',
        'ko': ':욕조:',
        'pt': ':banheira:',
        'it': ':vasca:',
        'fa': ':وان_حمام:',
        'id': ':bathtub:',
        'zh': ':浴缸:',
        'ru': ':ванна:'
    },
    '\U0001F50B': {  # 🔋
        'en': ':battery:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':batterie:',
        'es': ':pila:',
        'fr': ':pile:',
        'ja': ':電池:',
        'ko': ':배터리:',
        'pt': ':pilha:',
        'it': ':batteria:',
        'fa': ':باتری:',
        'id': ':baterai:',
        'zh': ':电池:',
        'ru': ':батарейка:'
    },
    '\U0001F3D6\U0000FE0F': {  # 🏖️
        'en': ':beach_with_umbrella:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':beach_umbrella:'],
        'variant': True,
        'de': ':strand_mit_sonnenschirm:',
        'es': ':playa_y_sombrilla:',
        'fr': ':plage_avec_parasol:',
        'ja': ':ビーチパラソル:',
        'ko': ':파라솔이_있는_해변:',
        'pt': ':praia_e_guarda-sol:',
        'it': ':spiaggia_con_ombrellone:',
        'fa': ':ساحل_و_سایبان:',
        'id': ':pantai_dan_payung:',
        'zh': ':沙滩伞:',
        'ru': ':пляж:'
    },
    '\U0001F3D6': {  # 🏖
        'en': ':beach_with_umbrella:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':beach_umbrella:'],
        'variant': True,
        'de': ':strand_mit_sonnenschirm:',
        'es': ':playa_y_sombrilla:',
        'fr': ':plage_avec_parasol:',
        'ja': ':ビーチパラソル:',
        'ko': ':파라솔이_있는_해변:',
        'pt': ':praia_e_guarda-sol:',
        'it': ':spiaggia_con_ombrellone:',
        'fa': ':ساحل_و_سایبان:',
        'id': ':pantai_dan_payung:',
        'zh': ':沙滩伞:',
        'ru': ':пляж:'
    },
    '\U0001F601': {  # 😁
        'en': ':beaming_face_with_smiling_eyes:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':grin:'],
        'de': ':strahlendes_gesicht_mit_lachenden_augen:',
        'es': ':cara_radiante_con_ojos_sonrientes:',
        'fr': ':visage_souriant_aux_yeux_rieurs:',
        'ja': ':にやっと笑う:',
        'ko': ':미소_짓는_눈으로_웃는_얼굴:',
        'pt': ':rosto_contente_com_olhos_sorridentes:',
        'it': ':faccina_raggiante_con_occhi_felici:',
        'fa': ':خندهٔ_دندانی:',
        'id': ':wajah_gembira_dengan_mata_bahagia:',
        'zh': ':嘻嘻:',
        'ru': ':светится_от_счастья:'
    },
    '\U0001FAD8': {  # 🫘
        'en': ':beans:',
        'status': fully_qualified,
        'E': 14,
        'de': ':bohnen:',
        'es': ':alubias:',
        'fr': ':haricots:',
        'ja': ':豆:',
        'ko': ':콩:',
        'pt': ':feijões:',
        'it': ':fagioli:',
        'fa': ':لوبیا:',
        'id': ':kacang_merah:',
        'zh': ':豆:',
        'ru': ':фасоль:'
    },
    '\U0001F43B': {  # 🐻
        'en': ':bear:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':bär:',
        'es': ':oso:',
        'fr': ':ours:',
        'ja': ':クマの顔:',
        'ko': ':곰_얼굴:',
        'pt': ':rosto_de_urso:',
        'it': ':orso:',
        'fa': ':خرس:',
        'id': ':beruang:',
        'zh': ':熊:',
        'ru': ':морда_медведя:'
    },
    '\U0001F493': {  # 💓
        'en': ':beating_heart:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':heartbeat:'],
        'de': ':schlagendes_herz:',
        'es': ':corazón_latiendo:',
        'fr': ':cœur_battant:',
        'ja': ':ドキドキするハート:',
        'ko': ':두근거리는_하트:',
        'pt': ':coração_pulsante:',
        'it': ':cuore_che_batte:',
        'fa': ':قلب_تپنده:',
        'id': ':hati_berdetak:',
        'zh': ':心跳:',
        'ru': ':пульсирующее_сердце:'
    },
    '\U0001F9AB': {  # 🦫
        'en': ':beaver:',
        'status': fully_qualified,
        'E': 13,
        'de': ':biber:',
        'es': ':castor:',
        'fr': ':castor:',
        'ja': ':ビーバー:',
        'ko': ':비버:',
        'pt': ':castor:',
        'it': ':castoro:',
        'fa': ':بیدستر:',
        'id': ':biwara:',
        'zh': ':海狸:',
        'ru': ':бобр:'
    },
    '\U0001F6CF\U0000FE0F': {  # 🛏️
        'en': ':bed:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':bett:',
        'es': ':cama:',
        'fr': ':lit:',
        'ja': ':ベッド:',
        'ko': ':침대:',
        'pt': ':cama:',
        'it': ':letto:',
        'fa': ':تختخواب:',
        'id': ':tempat_tidur:',
        'zh': ':床:',
        'ru': ':кровать:'
    },
    '\U0001F6CF': {  # 🛏
        'en': ':bed:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':bett:',
        'es': ':cama:',
        'fr': ':lit:',
        'ja': ':ベッド:',
        'ko': ':침대:',
        'pt': ':cama:',
        'it': ':letto:',
        'fa': ':تختخواب:',
        'id': ':tempat_tidur:',
        'zh': ':床:',
        'ru': ':кровать:'
    },
    '\U0001F37A': {  # 🍺
        'en': ':beer_mug:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':beer:'],
        'de': ':bierkrug:',
        'es': ':jarra_de_cerveza:',
        'fr': ':chope:',
        'ja': ':ビールジョッキ:',
        'ko': ':맥주잔:',
        'pt': ':cerveja:',
        'it': ':boccale_di_birra:',
        'fa': ':لیوان_آب_جو:',
        'id': ':mug_bir:',
        'zh': ':啤酒:',
        'ru': ':кружка_пива:'
    },
    '\U0001FAB2': {  # 🪲
        'en': ':beetle:',
        'status': fully_qualified,
        'E': 13,
        'de': ':käfer:',
        'es': ':escarabajo:',
        'fr': ':scarabée:',
        'ja': ':カブトムシ:',
        'ko': ':딱정벌레:',
        'pt': ':besouro:',
        'it': ':coleottero:',
        'fa': ':سوسک:',
        'id': ':kumbang:',
        'zh': ':甲虫:',
        'ru': ':жук:'
    },
    '\U0001F514': {  # 🔔
        'en': ':bell:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':glocke:',
        'es': ':campana:',
        'fr': ':cloche:',
        'ja': ':ベル:',
        'ko': ':종:',
        'pt': ':sino:',
        'it': ':campana:',
        'fa': ':زنگ:',
        'id': ':lonceng:',
        'zh': ':铃铛:',
        'ru': ':колокольчик:'
    },
    '\U0001FAD1': {  # 🫑
        'en': ':bell_pepper:',
        'status': fully_qualified,
        'E': 13,
        'de': ':paprika:',
        'es': ':pimiento:',
        'fr': ':poivron:',
        'ja': ':赤ピーマン:',
        'ko': ':피망:',
        'pt': ':pimentão:',
        'it': ':peperone:',
        'fa': ':فلفل_دلمه_ای:',
        'id': ':paprika:',
        'zh': ':灯笼椒:',
        'ru': ':болгарский_перец:'
    },
    '\U0001F515': {  # 🔕
        'en': ':bell_with_slash:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':no_bell:'],
        'de': ':durchgestrichene_glocke:',
        'es': ':campana_con_signo_de_cancelación:',
        'fr': ':alarme_désactivée:',
        'ja': ':ベル消音:',
        'ko': ':음소거:',
        'pt': ':sino_silenciado:',
        'it': ':campana_sbarrata:',
        'fa': ':خط_قرمز_روی_زنگوله:',
        'id': ':lonceng_dicoret:',
        'zh': ':禁止响铃:',
        'ru': ':перечеркнутый_колокольчик:'
    },
    '\U0001F6CE\U0000FE0F': {  # 🛎️
        'en': ':bellhop_bell:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':rezeptionsklingel:',
        'es': ':timbre_de_hotel:',
        'fr': ':cloche_de_comptoir:',
        'ja': ':ベルボーイベル:',
        'ko': ':호출_벨:',
        'pt': ':sineta:',
        'it': ':campanello_da_hotel:',
        'fa': ':زنگ_میز_پذیرش:',
        'id': ':bel_meja:',
        'zh': ':服务铃:',
        'ru': ':гостиничныи_звонок:'
    },
    '\U0001F6CE': {  # 🛎
        'en': ':bellhop_bell:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':rezeptionsklingel:',
        'es': ':timbre_de_hotel:',
        'fr': ':cloche_de_comptoir:',
        'ja': ':ベルボーイベル:',
        'ko': ':호출_벨:',
        'pt': ':sineta:',
        'it': ':campanello_da_hotel:',
        'fa': ':زنگ_میز_پذیرش:',
        'id': ':bel_meja:',
        'zh': ':服务铃:',
        'ru': ':гостиничный_звонок:'
    },
    '\U0001F371': {  # 🍱
        'en': ':bento_box:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':bento:'],
        'de': ':bento-box:',
        'es': ':caja_de_bento:',
        'fr': ':boîte_déjeuner:',
        'ja': ':弁当:',
        'ko': ':도시락:',
        'pt': ':bentô:',
        'it': ':bento_box:',
        'fa': ':جعبه_بنتو:',
        'id': ':kotak_bento:',
        'zh': ':盒饭:',
        'ru': ':бэнто:'
    },
    '\U0001F9C3': {  # 🧃
        'en': ':beverage_box:',
        'status': fully_qualified,
        'E': 12,
        'de': ':trinkpäckchen:',
        'es': ':tetrabrik:',
        'fr': ':briquette_de_jus:',
        'ja': ':紙パック飲料:',
        'ko': ':음료_팩:',
        'pt': ':suco_de_caixa:',
        'it': ':bevanda_monodose:',
        'fa': ':پاکت_نوشیدنی:',
        'id': ':kotak_minuman:',
        'zh': ':饮料盒:',
        'ru': ':пакетик_сока:'
    },
    '\U0001F6B2': {  # 🚲
        'en': ':bicycle:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':bike:'],
        'variant': True,
        'de': ':fahrrad:',
        'es': ':bicicleta:',
        'fr': ':vélo:',
        'ja': ':自転車:',
        'ko': ':자전거:',
        'pt': ':bicicleta:',
        'it': ':bicicletta:',
        'fa': ':دوچرخه:',
        'id': ':sepeda:',
        'zh': ':自行车:',
        'ru': ':велосипед:'
    },
    '\U0001F459': {  # 👙
        'en': ':bikini:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':bikini:',
        'es': ':bikini:',
        'fr': ':bikini:',
        'ja': ':ビキニ:',
        'ko': ':비키니:',
        'pt': ':biquíni:',
        'it': ':bikini:',
        'fa': ':بیکینی:',
        'id': ':bikini:',
        'zh': ':比基尼:',
        'ru': ':бикини:'
    },
    '\U0001F9E2': {  # 🧢
        'en': ':billed_cap:',
        'status': fully_qualified,
        'E': 5,
        'de': ':baseballmütze:',
        'es': ':gorra_con_visera:',
        'fr': ':casquette_américaine:',
        'ja': ':キャップ:',
        'ko': ':야구모자:',
        'pt': ':boné:',
        'it': ':cappello_con_visiera:',
        'fa': ':کلاه_لبه_دار:',
        'id': ':topi_berlidah:',
        'zh': ':鸭舌帽:',
        'ru': ':бейсболка:'
    },
    '\U00002623\U0000FE0F': {  # ☣️
        'en': ':biohazard:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':biohazard_sign:'],
        'variant': True,
        'de': ':biogefährdung:',
        'es': ':riesgo_biológico:',
        'fr': ':danger_biologique:',
        'ja': ':バイオハザード:',
        'ko': ':생물학적_위험:',
        'pt': ':risco_biológico:',
        'it': ':simbolo_del_rischio_biologico:',
        'fa': ':خطر_زیستی:',
        'id': ':biohazard:',
        'zh': ':生物危害:',
        'ru': ':биологическая_угроза:'
    },
    '\U00002623': {  # ☣
        'en': ':biohazard:',
        'status': unqualified,
        'E': 1,
        'alias': [':biohazard_sign:'],
        'variant': True,
        'de': ':biogefährdung:',
        'es': ':riesgo_biológico:',
        'fr': ':danger_biologique:',
        'ja': ':バイオハザード:',
        'ko': ':생물학적_위험:',
        'pt': ':risco_biológico:',
        'it': ':simbolo_del_rischio_biologico:',
        'fa': ':خطر_زیستی:',
        'id': ':biohazard:',
        'zh': ':生物危害:',
        'ru': ':биологическая_угроза:'
    },
    '\U0001F426': {  # 🐦
        'en': ':bird:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':vogel:',
        'es': ':pájaro:',
        'fr': ':oiseau:',
        'ja': ':鳥:',
        'ko': ':새:',
        'pt': ':pássaro:',
        'it': ':uccello:',
        'fa': ':پرنده:',
        'id': ':burung:',
        'zh': ':鸟:',
        'ru': ':птица:'
    },
    '\U0001F382': {  # 🎂
        'en': ':birthday_cake:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':birthday:'],
        'de': ':geburtstagskuchen:',
        'es': ':tarta_de_cumpleaños:',
        'fr': ':gâteau_d’anniversaire:',
        'ja': ':バースデーケーキ:',
        'ko': ':생일_케이크:',
        'pt': ':bolo_de_aniversário:',
        'it': ':torta_di_compleanno:',
        'fa': ':کیک_تولد:',
        'id': ':kue_ulang_tahun:',
        'zh': ':生日蛋糕:',
        'ru': ':торт_ко_дню_рождения:'
    },
    '\U0001F9AC': {  # 🦬
        'en': ':bison:',
        'status': fully_qualified,
        'E': 13,
        'de': ':bison:',
        'es': ':bisonte:',
        'fr': ':bison:',
        'ja': ':バイソン:',
        'ko': ':들소:',
        'pt': ':bisão:',
        'it': ':bisonte:',
        'fa': ':گاومیش:',
        'id': ':bison:',
        'zh': ':大野牛:',
        'ru': ':бизон:'
    },
    '\U0001FAE6': {  # 🫦
        'en': ':biting_lip:',
        'status': fully_qualified,
        'E': 14,
        'de': ':auf_lippe_beißen:',
        'es': ':labio_mordido:',
        'fr': ':lèvres_qui_se_mordent:',
        'ja': ':唇を咬んでいる口:',
        'ko': ':입술을_깨물고_있는_입:',
        'pt': ':mordendo_o_lábio:',
        'it': ':bocca_che_morde_il_labbro:',
        'fa': ':گازگرفتن_لب:',
        'id': ':menggigit_bibir:',
        'zh': ':咬住嘴唇:',
        'ru': ':кусать_губы:'
    },
    '\U0001F426\U0000200D\U00002B1B': {  # 🐦‍⬛
        'en': ':black_bird:',
        'status': fully_qualified,
        'E': 15,
        'alias': [':raven:', ':crow:', ':rook:'],
        'de': ':schwarzer_vogel:',
        'es': ':pájaro_negro:',
        'fr': ':oiseau_noir:',
        'ja': ':黒鳥:',
        'ko': ':검은_새:',
        'pt': ':pássaro_preto:',
        'it': ':uccello_nero:',
        'fa': ':پرنده_سیاه:',
        'id': ':burung_hitam:',
        'zh': ':黑鸟:',
        'ru': ':*black_bird:'
    },
    '\U0001F408\U0000200D\U00002B1B': {  # 🐈‍⬛
        'en': ':black_cat:',
        'status': fully_qualified,
        'E': 13,
        'de': ':schwarze_katze:',
        'es': ':gato_negro:',
        'fr': ':chat_noir:',
        'ja': ':黒猫:',
        'ko': ':검은_고양이:',
        'pt': ':gato_preto:',
        'it': ':gatto_nero:',
        'fa': ':گربه_سیاه:',
        'id': ':kucing_hitam:',
        'zh': ':黑猫:',
        'ru': ':черная_кошка:'
    },
    '\U000026AB': {  # ⚫
        'en': ':black_circle:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':schwarzer_punkt:',
        'es': ':círculo_negro:',
        'fr': ':disque_noir:',
        'ja': ':黒い丸:',
        'ko': ':검은색_원:',
        'pt': ':círculo_preto:',
        'it': ':cerchio_nero:',
        'fa': ':دایره_مشکی:',
        'id': ':lingkaran_hitam:',
        'zh': ':黑色圆:',
        'ru': ':черный_круг:'
    },
    '\U0001F3F4': {  # 🏴
        'en': ':black_flag:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':waving_black_flag:'],
        'de': ':schwarze_flagge:',
        'es': ':bandera_negra:',
        'fr': ':drapeau_noir:',
        'ja': ':黒旗:',
        'ko': ':펄럭이는_검은색_깃발:',
        'pt': ':bandeira_preta:',
        'it': ':bandiera_nera:',
        'fa': ':پرچم_سیاه:',
        'id': ':bendera_hitam:',
        'zh': ':举黑旗:',
        'ru': ':черный_флаг:'
    },
    '\U0001F5A4': {  # 🖤
        'en': ':black_heart:',
        'status': fully_qualified,
        'E': 3,
        'de': ':schwarzes_herz:',
        'es': ':corazón_negro:',
        'fr': ':cœur_noir:',
        'ja': ':黒いハート:',
        'ko': ':검은색_하트:',
        'pt': ':coração_preto:',
        'it': ':cuore_nero:',
        'fa': ':قلب_سیاه:',
        'id': ':hati_hitam:',
        'zh': ':黑心:',
        'ru': ':черное_сердце:'
    },
    '\U00002B1B': {  # ⬛
        'en': ':black_large_square:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':großes_schwarzes_quadrat:',
        'es': ':cuadrado_negro_grande:',
        'fr': ':grand_carré_noir:',
        'ja': ':黒四角特大:',
        'ko': ':큰_검은색_사각형:',
        'pt': ':quadrado_preto_grande:',
        'it': ':quadrato_nero_grande:',
        'fa': ':مربع_مشکی_بزرگ:',
        'id': ':persegi_besar_hitam:',
        'zh': ':黑线大方框:',
        'ru': ':большой_черный_квадрат:'
    },
    '\U000025FE': {  # ◾
        'en': ':black_medium-small_square:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':black_medium_small_square:'],
        'variant': True,
        'de': ':mittelkleines_schwarzes_quadrat:',
        'es': ':cuadrado_negro_mediano-pequeño:',
        'fr': ':carré_petit_moyen_noir:',
        'ja': ':黒四角中:',
        'ko': ':중각_작은_검은색_사각형:',
        'pt': ':quadrado_preto_médio_menor:',
        'it': ':quadrato_nero_medio-piccolo:',
        'fa': ':مربع_مشکی_نسبتاً_کوچک:',
        'id': ':persegi_kecil_medium_hitam:',
        'zh': ':黑色中小方块:',
        'ru': ':небольшой_черный_квадрат:'
    },
    '\U000025FC\U0000FE0F': {  # ◼️
        'en': ':black_medium_square:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':mittelgroßes_schwarzes_quadrat:',
        'es': ':cuadrado_negro_mediano:',
        'fr': ':carré_moyen_noir:',
        'ja': ':黒四角大:',
        'ko': ':중간_검은색_사각형:',
        'pt': ':quadrado_preto_médio:',
        'it': ':quadrato_nero_medio:',
        'fa': ':مربع_مشکی_متوسط:',
        'id': ':persegi_medium_hitam:',
        'zh': ':黑色中方块:',
        'ru': ':черныи_квадрат:'
    },
    '\U000025FC': {  # ◼
        'en': ':black_medium_square:',
        'status': unqualified,
        'E': 0.6,
        'variant': True,
        'de': ':mittelgroßes_schwarzes_quadrat:',
        'es': ':cuadrado_negro_mediano:',
        'fr': ':carré_moyen_noir:',
        'ja': ':黒四角大:',
        'ko': ':중간_검은색_사각형:',
        'pt': ':quadrado_preto_médio:',
        'it': ':quadrato_nero_medio:',
        'fa': ':مربع_مشکی_متوسط:',
        'id': ':persegi_medium_hitam:',
        'zh': ':黑色中方块:',
        'ru': ':черный_квадрат:'
    },
    '\U00002712\U0000FE0F': {  # ✒️
        'en': ':black_nib:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':schwarzer_federhalter:',
        'es': ':pluma_negra:',
        'fr': ':stylo_plume_noir:',
        'ja': ':ペン先:',
        'ko': ':검은색_펜촉:',
        'pt': ':ponta_de_caneta_tinteiro:',
        'it': ':pennino:',
        'fa': ':قلم_مشکی:',
        'id': ':ujung_pena:',
        'zh': ':钢笔尖:',
        'ru': ':перьевои_наконечник:'
    },
    '\U00002712': {  # ✒
        'en': ':black_nib:',
        'status': unqualified,
        'E': 0.6,
        'variant': True,
        'de': ':schwarzer_federhalter:',
        'es': ':pluma_negra:',
        'fr': ':stylo_plume_noir:',
        'ja': ':ペン先:',
        'ko': ':검은색_펜촉:',
        'pt': ':ponta_de_caneta_tinteiro:',
        'it': ':pennino:',
        'fa': ':قلم_مشکی:',
        'id': ':ujung_pena:',
        'zh': ':钢笔尖:',
        'ru': ':перьевой_наконечник:'
    },
    '\U000025AA\U0000FE0F': {  # ▪️
        'en': ':black_small_square:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':kleines_schwarzes_quadrat:',
        'es': ':cuadrado_negro_pequeño:',
        'fr': ':petit_carré_noir:',
        'ja': ':黒四角小:',
        'ko': ':작은_검은색_사각형:',
        'pt': ':quadrado_preto_pequeno:',
        'it': ':quadrato_nero_piccolo:',
        'fa': ':مربع_مشکی_کوچک:',
        'id': ':persegi_kecil_hitam:',
        'zh': ':黑色小方块:',
        'ru': ':черныи_квадратик:'
    },
    '\U000025AA': {  # ▪
        'en': ':black_small_square:',
        'status': unqualified,
        'E': 0.6,
        'variant': True,
        'de': ':kleines_schwarzes_quadrat:',
        'es': ':cuadrado_negro_pequeño:',
        'fr': ':petit_carré_noir:',
        'ja': ':黒四角小:',
        'ko': ':작은_검은색_사각형:',
        'pt': ':quadrado_preto_pequeno:',
        'it': ':quadrato_nero_piccolo:',
        'fa': ':مربع_مشکی_کوچک:',
        'id': ':persegi_kecil_hitam:',
        'zh': ':黑色小方块:',
        'ru': ':черный_квадратик:'
    },
    '\U0001F532': {  # 🔲
        'en': ':black_square_button:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':schwarze_quadratische_schaltfläche:',
        'es': ':botón_cuadrado_con_borde_negro:',
        'fr': ':carré_noir:',
        'ja': ':黒枠四角:',
        'ko': ':테두리가_검은_사각_버튼:',
        'pt': ':botão_quadrado_preto:',
        'it': ':tasto_quadrato_bianco_con_bordo_nero:',
        'fa': ':دکمۀ_مربع_مشکی:',
        'id': ':tombol_persegi_hitam:',
        'zh': ':黑色方形按钮:',
        'ru': ':кнопка_с_черной_рамкой:'
    },
    '\U0001F33C': {  # 🌼
        'en': ':blossom:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':gelbe_blüte:',
        'es': ':flor:',
        'fr': ':bourgeon:',
        'ja': ':開花:',
        'ko': ':꽃송이:',
        'pt': ':flor:',
        'it': ':fiore:',
        'fa': ':شکوفه:',
        'id': ':mekar:',
        'zh': ':开花:',
        'ru': ':цветок:'
    },
    '\U0001F421': {  # 🐡
        'en': ':blowfish:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kugelfisch:',
        'es': ':pez_globo:',
        'fr': ':poisson-lune:',
        'ja': ':フグ:',
        'ko': ':복어:',
        'pt': ':baiacu:',
        'it': ':pesce_palla:',
        'fa': ':ماهی_بادکنکی:',
        'id': ':ikan_gembung:',
        'zh': ':河豚:',
        'ru': ':рыба_еж:'
    },
    '\U0001F4D8': {  # 📘
        'en': ':blue_book:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':blaues_buch:',
        'es': ':libro_azul:',
        'fr': ':livre_bleu:',
        'ja': ':青の本:',
        'ko': ':파란색_책:',
        'pt': ':livro_azul:',
        'it': ':libro_blu:',
        'fa': ':کتاب_آبی:',
        'id': ':buku_biru:',
        'zh': ':蓝色书本:',
        'ru': ':синяя_книга:'
    },
    '\U0001F535': {  # 🔵
        'en': ':blue_circle:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':large_blue_circle:'],
        'de': ':blauer_punkt:',
        'es': ':círculo_azul:',
        'fr': ':disque_bleu:',
        'ja': ':青い丸:',
        'ko': ':파란_원:',
        'pt': ':círculo_azul:',
        'it': ':cerchio_blu:',
        'fa': ':دایره_آبی:',
        'id': ':lingkaran_biru:',
        'zh': ':蓝色圆:',
        'ru': ':голубой_круг:'
    },
    '\U0001F499': {  # 💙
        'en': ':blue_heart:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':blaues_herz:',
        'es': ':corazón_azul:',
        'fr': ':cœur_bleu:',
        'ja': ':青いハート:',
        'ko': ':파란색_하트:',
        'pt': ':coração_azul:',
        'it': ':cuore_blu:',
        'fa': ':قلب_آبی:',
        'id': ':hati_biru:',
        'zh': ':蓝心:',
        'ru': ':синее_сердце:'
    },
    '\U0001F7E6': {  # 🟦
        'en': ':blue_square:',
        'status': fully_qualified,
        'E': 12,
        'de': ':blaues_quadrat:',
        'es': ':cuadrado_azul:',
        'fr': ':carré_bleu:',
        'ja': ':青い四角:',
        'ko': ':파란색_사각형:',
        'pt': ':quadrado_azul:',
        'it': ':quadrato_blu:',
        'fa': ':مربع_آبی:',
        'id': ':persegi_biru:',
        'zh': ':蓝色方块:',
        'ru': ':синий_квадрат:'
    },
    '\U0001FAD0': {  # 🫐
        'en': ':blueberries:',
        'status': fully_qualified,
        'E': 13,
        'de': ':blaubeeren:',
        'es': ':arándanos:',
        'fr': ':myrtilles:',
        'ja': ':ブルーベリー:',
        'ko': ':블루베리:',
        'pt': ':mirtilos:',
        'it': ':mirtilli:',
        'fa': ':توت_آبی:',
        'id': ':blueberry:',
        'zh': ':蓝莓:',
        'ru': ':голубика:'
    },
    '\U0001F417': {  # 🐗
        'en': ':boar:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':wildschwein:',
        'es': ':jabalí:',
        'fr': ':sanglier:',
        'ja': ':イノシシ:',
        'ko': ':멧돼지:',
        'pt': ':javali:',
        'it': ':cinghiale:',
        'fa': ':گراز_نر:',
        'id': ':babi_hutan:',
        'zh': ':野猪:',
        'ru': ':кабан:'
    },
    '\U0001F4A3': {  # 💣
        'en': ':bomb:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':bombe:',
        'es': ':bomba:',
        'fr': ':bombe:',
        'ja': ':爆弾:',
        'ko': ':폭탄:',
        'pt': ':bomba:',
        'it': ':bomba:',
        'fa': ':بمب:',
        'id': ':bom:',
        'zh': ':炸弹:',
        'ru': ':бомба:'
    },
    '\U0001F9B4': {  # 🦴
        'en': ':bone:',
        'status': fully_qualified,
        'E': 11,
        'de': ':knochen:',
        'es': ':hueso:',
        'fr': ':os:',
        'ja': ':骨:',
        'ko': ':뼈다귀:',
        'pt': ':osso:',
        'it': ':osso:',
        'fa': ':استخوان:',
        'id': ':tulang:',
        'zh': ':骨头:',
        'ru': ':кость:'
    },
    '\U0001F516': {  # 🔖
        'en': ':bookmark:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':lesezeichen:',
        'es': ':marcapáginas:',
        'fr': ':marque-page:',
        'ja': ':しおり:',
        'ko': ':북마크:',
        'pt': ':marcador_de_página:',
        'it': ':segnalibro:',
        'fa': ':نشان_گذار_کتاب:',
        'id': ':pembatas_buku:',
        'zh': ':书签:',
        'ru': ':закладка:'
    },
    '\U0001F4D1': {  # 📑
        'en': ':bookmark_tabs:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':pagemarker:',
        'es': ':marcadores:',
        'fr': ':signets:',
        'ja': ':ページに付箋:',
        'ko': ':북마크_탭:',
        'pt': ':marcadores_de_página:',
        'it': ':etichette_segnalibro:',
        'fa': ':برچسب_های_نشانه_گذاری:',
        'id': ':penanda_halaman_buku:',
        'zh': ':标签页:',
        'ru': ':стикеры_закладки:'
    },
    '\U0001F4DA': {  # 📚
        'en': ':books:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':bücherstapel:',
        'es': ':libros:',
        'fr': ':livres:',
        'ja': ':本の山:',
        'ko': ':책_여러_권:',
        'pt': ':livros:',
        'it': ':libri:',
        'fa': ':چند_کتاب:',
        'id': ':buku:',
        'zh': ':书:',
        'ru': ':книги:'
    },
    '\U0001FA83': {  # 🪃
        'en': ':boomerang:',
        'status': fully_qualified,
        'E': 13,
        'de': ':bumerang:',
        'es': ':bumerán:',
        'fr': ':boomerang:',
        'ja': ':ブーメラン:',
        'ko': ':부메랑:',
        'pt': ':bumerangue:',
        'it': ':boomerang:',
        'fa': ':بومرنگ:',
        'id': ':bumerang:',
        'zh': ':回旋镖:',
        'ru': ':бумеранг:'
    },
    '\U0001F37E': {  # 🍾
        'en': ':bottle_with_popping_cork:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':champagne:'],
        'de': ':flasche_mit_knallendem_korken:',
        'es': ':botella_descorchada:',
        'fr': ':bouteille_de_champagne:',
        'ja': ':シャンパン:',
        'ko': ':코르크가_튀어나오고_있는_병:',
        'pt': ':garrafa_de_champanhe:',
        'it': ':bottiglia_stappata:',
        'fa': ':شامپاین_جوشان:',
        'id': ':botol_dengan_penyumbat_terlepas:',
        'zh': ':开香槟:',
        'ru': ':шампанское:'
    },
    '\U0001F490': {  # 💐
        'en': ':bouquet:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':blumenstrauß:',
        'es': ':ramo_de_flores:',
        'fr': ':bouquet:',
        'ja': ':花束:',
        'ko': ':꽃다발:',
        'pt': ':buquê:',
        'it': ':mazzo_di_fiori:',
        'fa': ':دسته_گل:',
        'id': ':karangan_bunga:',
        'zh': ':花束:',
        'ru': ':букет:'
    },
    '\U0001F3F9': {  # 🏹
        'en': ':bow_and_arrow:',
        'status': fully_qualified,
        'E': 1,
        'de': ':pfeil_und_bogen:',
        'es': ':arco_y_flecha:',
        'fr': ':arc_et_flèche:',
        'ja': ':弓矢:',
        'ko': ':활과_화살:',
        'pt': ':arco_e_flecha:',
        'it': ':arco_e_freccia:',
        'fa': ':تیر_و_کمان:',
        'id': ':busur_dan_panah:',
        'zh': ':弓和箭:',
        'ru': ':лук_со_стрелой:'
    },
    '\U0001F963': {  # 🥣
        'en': ':bowl_with_spoon:',
        'status': fully_qualified,
        'E': 5,
        'de': ':schüssel_mit_löffel:',
        'es': ':cuenco_con_cuchara:',
        'fr': ':bol_avec_cuillère:',
        'ja': ':ボウルとスプーン:',
        'ko': ':스푼과_사발:',
        'pt': ':tigela_com_colher:',
        'it': ':ciotola_con_cucchiaio:',
        'fa': ':کاسه_و_قاشق:',
        'id': ':mangkuk_dengan_sendok:',
        'zh': ':碗勺:',
        'ru': ':миска_с_ложкой:'
    },
    '\U0001F3B3': {  # 🎳
        'en': ':bowling:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':bowling:',
        'es': ':bolos:',
        'fr': ':bowling:',
        'ja': ':ボウリング:',
        'ko': ':볼링:',
        'pt': ':boliche:',
        'it': ':bowling:',
        'fa': ':بولینگ:',
        'id': ':bowling:',
        'zh': ':保龄球:',
        'ru': ':боулинг:'
    },
    '\U0001F94A': {  # 🥊
        'en': ':boxing_glove:',
        'status': fully_qualified,
        'E': 3,
        'de': ':boxhandschuh:',
        'es': ':guante_de_boxeo:',
        'fr': ':gant_de_boxe:',
        'ja': ':ボクシング:',
        'ko': ':권투_글러브:',
        'pt': ':luva_de_boxe:',
        'it': ':guantone_da_pugilato:',
        'fa': ':دست_کش_بوکس:',
        'id': ':sarung_tinju:',
        'zh': ':拳击手套:',
        'ru': ':боксерская_перчатка:'
    },
    '\U0001F466': {  # 👦
        'en': ':boy:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':junge:',
        'es': ':niño:',
        'fr': ':garçon:',
        'ja': ':男の子:',
        'ko': ':남자_아이:',
        'pt': ':menino:',
        'it': ':bambino:',
        'fa': ':پسر:',
        'id': ':anak_laki-laki:',
        'zh': ':男孩:',
        'ru': ':мальчик:'
    },
    '\U0001F466\U0001F3FF': {  # 👦🏿
        'en': ':boy_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':junge_dunkle_hautfarbe:',
        'es': ':niño_tono_de_piel_oscuro:',
        'fr': ':garçon_peau_foncée:',
        'ja': ':男の子_濃い肌色:',
        'ko': ':남자_아이_검은색_피부:',
        'pt': ':menino_pele_escura:',
        'it': ':bambino_carnagione_scura:',
        'fa': ':پسر_پوست_آبنوسی:',
        'id': ':anak_laki-laki_warna_kulit_gelap:',
        'zh': ':男孩_较深肤色:',
        'ru': ':мальчик_очень_темный_тон_кожи:'
    },
    '\U0001F466\U0001F3FB': {  # 👦🏻
        'en': ':boy_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':junge_helle_hautfarbe:',
        'es': ':niño_tono_de_piel_claro:',
        'fr': ':garçon_peau_claire:',
        'ja': ':男の子_薄い肌色:',
        'ko': ':남자_아이_하얀_피부:',
        'pt': ':menino_pele_clara:',
        'it': ':bambino_carnagione_chiara:',
        'fa': ':پسر_پوست_سفید:',
        'id': ':anak_laki-laki_warna_kulit_cerah:',
        'zh': ':男孩_较浅肤色:',
        'ru': ':мальчик_очень_светлый_тон_кожи:'
    },
    '\U0001F466\U0001F3FE': {  # 👦🏾
        'en': ':boy_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':junge_mitteldunkle_hautfarbe:',
        'es': ':niño_tono_de_piel_oscuro_medio:',
        'fr': ':garçon_peau_mate:',
        'ja': ':男の子_やや濃い肌色:',
        'ko': ':남자_아이_진한_갈색_피부:',
        'pt': ':menino_pele_morena_escura:',
        'it': ':bambino_carnagione_abbastanza_scura:',
        'fa': ':پسر_پوست_گندمی:',
        'id': ':anak_laki-laki_warna_kulit_gelap-sedang:',
        'zh': ':男孩_中等深肤色:',
        'ru': ':мальчик_темный_тон_кожи:'
    },
    '\U0001F466\U0001F3FC': {  # 👦🏼
        'en': ':boy_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':junge_mittelhelle_hautfarbe:',
        'es': ':niño_tono_de_piel_claro_medio:',
        'fr': ':garçon_peau_moyennement_claire:',
        'ja': ':男の子_やや薄い肌色:',
        'ko': ':남자_아이_연한_갈색_피부:',
        'pt': ':menino_pele_morena_clara:',
        'it': ':bambino_carnagione_abbastanza_chiara:',
        'fa': ':پسر_پوست_روشن:',
        'id': ':anak_laki-laki_warna_kulit_cerah-sedang:',
        'zh': ':男孩_中等浅肤色:',
        'ru': ':мальчик_светлый_тон_кожи:'
    },
    '\U0001F466\U0001F3FD': {  # 👦🏽
        'en': ':boy_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':junge_mittlere_hautfarbe:',
        'es': ':niño_tono_de_piel_medio:',
        'fr': ':garçon_peau_légèrement_mate:',
        'ja': ':男の子_中間の肌色:',
        'ko': ':남자_아이_갈색_피부:',
        'pt': ':menino_pele_morena:',
        'it': ':bambino_carnagione_olivastra:',
        'fa': ':پسر_پوست_طلایی:',
        'id': ':anak_laki-laki_warna_kulit_sedang:',
        'zh': ':男孩_中等肤色:',
        'ru': ':мальчик_средний_тон_кожи:'
    },
    '\U0001F9E0': {  # 🧠
        'en': ':brain:',
        'status': fully_qualified,
        'E': 5,
        'de': ':gehirn:',
        'es': ':cerebro:',
        'fr': ':cerveau:',
        'ja': ':脳:',
        'ko': ':뇌:',
        'pt': ':cérebro:',
        'it': ':cervello:',
        'fa': ':مغز:',
        'id': ':otak:',
        'zh': ':脑:',
        'ru': ':мозг:'
    },
    '\U0001F35E': {  # 🍞
        'en': ':bread:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':brot:',
        'es': ':pan_de_molde:',
        'fr': ':pain:',
        'ja': ':食パン:',
        'ko': ':빵:',
        'pt': ':pão:',
        'it': ':pane_in_cassetta:',
        'fa': ':نان:',
        'id': ':roti:',
        'zh': ':面包:',
        'ru': ':хлеб:'
    },
    '\U0001F931': {  # 🤱
        'en': ':breast-feeding:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':breast_feeding:'],
        'de': ':stillen:',
        'es': ':lactancia_materna:',
        'fr': ':allaitement:',
        'ja': ':授乳:',
        'ko': ':모유_수유:',
        'pt': ':amamentando:',
        'it': ':allattare:',
        'fa': ':تغذیه_با_شیر_مادر:',
        'id': ':menyusui:',
        'zh': ':母乳喂养:',
        'ru': ':кормление_грудью:'
    },
    '\U0001F931\U0001F3FF': {  # 🤱🏿
        'en': ':breast-feeding_dark_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':stillen_dunkle_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_oscuro:',
        'fr': ':allaitement_peau_foncée:',
        'ja': ':授乳_濃い肌色:',
        'ko': ':모유_수유_검은색_피부:',
        'pt': ':amamentando_pele_escura:',
        'it': ':allattare_carnagione_scura:',
        'fa': ':تغذیه_با_شیر_مادر_پوست_آبنوسی:',
        'id': ':menyusui_warna_kulit_gelap:',
        'zh': ':母乳喂养_较深肤色:',
        'ru': ':кормление_грудью_очень_темный_тон_кожи:'
    },
    '\U0001F931\U0001F3FB': {  # 🤱🏻
        'en': ':breast-feeding_light_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':stillen_helle_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_claro:',
        'fr': ':allaitement_peau_claire:',
        'ja': ':授乳_薄い肌色:',
        'ko': ':모유_수유_하얀_피부:',
        'pt': ':amamentando_pele_clara:',
        'it': ':allattare_carnagione_chiara:',
        'fa': ':تغذیه_با_شیر_مادر_پوست_سفید:',
        'id': ':menyusui_warna_kulit_cerah:',
        'zh': ':母乳喂养_较浅肤色:',
        'ru': ':кормление_грудью_очень_светлый_тон_кожи:'
    },
    '\U0001F931\U0001F3FE': {  # 🤱🏾
        'en': ':breast-feeding_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':stillen_mitteldunkle_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_oscuro_medio:',
        'fr': ':allaitement_peau_mate:',
        'ja': ':授乳_やや濃い肌色:',
        'ko': ':모유_수유_진한_갈색_피부:',
        'pt': ':amamentando_pele_morena_escura:',
        'it': ':allattare_carnagione_abbastanza_scura:',
        'fa': ':تغذیه_با_شیر_مادر_پوست_گندمی:',
        'id': ':menyusui_warna_kulit_gelap-sedang:',
        'zh': ':母乳喂养_中等深肤色:',
        'ru': ':кормление_грудью_темный_тон_кожи:'
    },
    '\U0001F931\U0001F3FC': {  # 🤱🏼
        'en': ':breast-feeding_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':stillen_mittelhelle_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_claro_medio:',
        'fr': ':allaitement_peau_moyennement_claire:',
        'ja': ':授乳_やや薄い肌色:',
        'ko': ':모유_수유_연한_갈색_피부:',
        'pt': ':amamentando_pele_morena_clara:',
        'it': ':allattare_carnagione_abbastanza_chiara:',
        'fa': ':تغذیه_با_شیر_مادر_پوست_روشن:',
        'id': ':menyusui_warna_kulit_cerah-sedang:',
        'zh': ':母乳喂养_中等浅肤色:',
        'ru': ':кормление_грудью_светлый_тон_кожи:'
    },
    '\U0001F931\U0001F3FD': {  # 🤱🏽
        'en': ':breast-feeding_medium_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':stillen_mittlere_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_medio:',
        'fr': ':allaitement_peau_légèrement_mate:',
        'ja': ':授乳_中間の肌色:',
        'ko': ':모유_수유_갈색_피부:',
        'pt': ':amamentando_pele_morena:',
        'it': ':allattare_carnagione_olivastra:',
        'fa': ':تغذیه_با_شیر_مادر_پوست_طلایی:',
        'id': ':menyusui_warna_kulit_sedang:',
        'zh': ':母乳喂养_中等肤色:',
        'ru': ':кормление_грудью_средний_тон_кожи:'
    },
    '\U0001F9F1': {  # 🧱
        'en': ':brick:',
        'status': fully_qualified,
        'E': 11,
        'alias': [':bricks:'],
        'de': ':ziegelstein:',
        'es': ':ladrillo:',
        'fr': ':brique:',
        'ja': ':れんが:',
        'ko': ':벽돌:',
        'pt': ':tijolo:',
        'it': ':mattoni:',
        'fa': ':دیوار_آجری:',
        'id': ':batu_bata:',
        'zh': ':砖:',
        'ru': ':кирпичи:'
    },
    '\U0001F309': {  # 🌉
        'en': ':bridge_at_night:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':brücke_vor_nachthimmel:',
        'es': ':puente_de_noche:',
        'fr': ':pont_de_nuit:',
        'ja': ':夜の橋:',
        'ko': ':밤하늘을_배경으로_하는_다리:',
        'pt': ':ponte_à_noite:',
        'it': ':ponte_di_notte:',
        'fa': ':پل_در_شب:',
        'id': ':jembatan_di_malam_hari:',
        'zh': ':夜幕下的桥:',
        'ru': ':мост_ночью:'
    },
    '\U0001F4BC': {  # 💼
        'en': ':briefcase:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':aktentasche:',
        'es': ':maletín:',
        'fr': ':porte-documents:',
        'ja': ':ブリーフケース:',
        'ko': ':서류_가방:',
        'pt': ':maleta:',
        'it': ':valigetta_24_ore:',
        'fa': ':کیف_اسناد:',
        'id': ':tas_kantor:',
        'zh': ':公文包:',
        'ru': ':портфель:'
    },
    '\U0001FA72': {  # 🩲
        'en': ':briefs:',
        'status': fully_qualified,
        'E': 12,
        'alias': [':swim_brief:'],
        'de': ':slip:',
        'es': ':ropa_interior:',
        'fr': ':slip:',
        'ja': ':ブリーフ:',
        'ko': ':삼각_수영복:',
        'pt': ':cueca:',
        'it': ':slip:',
        'fa': ':شورت_اسلیپ:',
        'id': ':celana_renang:',
        'zh': ':三角裤:',
        'ru': ':плавки:'
    },
    '\U0001F506': {  # 🔆
        'en': ':bright_button:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':high_brightness:'],
        'de': ':heller-taste:',
        'es': ':brillo_alto:',
        'fr': ':luminosité_élevée:',
        'ja': ':高輝度:',
        'ko': ':밝기_높음:',
        'pt': ':botão_de_aumentar_brilho:',
        'it': ':luminosità_elevata:',
        'fa': ':دکمهٔ_روشن_کردن_صفحه:',
        'id': ':tombol_cerahkan:',
        'zh': ':高亮度按钮:',
        'ru': ':высокая_яркость:'
    },
    '\U0001F966': {  # 🥦
        'en': ':broccoli:',
        'status': fully_qualified,
        'E': 5,
        'de': ':brokkoli:',
        'es': ':brócoli:',
        'fr': ':brocoli:',
        'ja': ':ブロッコリー:',
        'ko': ':브로콜리:',
        'pt': ':brócolis:',
        'it': ':broccoli:',
        'fa': ':بروکلی:',
        'id': ':brokoli:',
        'zh': ':西兰花:',
        'ru': ':брокколи:'
    },
    '\U0001F494': {  # 💔
        'en': ':broken_heart:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':gebrochenes_herz:',
        'es': ':corazón_roto:',
        'fr': ':cœur_brisé:',
        'ja': ':割れたハート:',
        'ko': ':깨진_하트:',
        'pt': ':coração_partido:',
        'it': ':cuore_infranto:',
        'fa': ':قلب_شکسته:',
        'id': ':patah_hati:',
        'zh': ':心碎:',
        'ru': ':разбитое_сердце:'
    },
    '\U0001F9F9': {  # 🧹
        'en': ':broom:',
        'status': fully_qualified,
        'E': 11,
        'de': ':besen:',
        'es': ':escoba:',
        'fr': ':balai:',
        'ja': ':ほうき:',
        'ko': ':빗자루:',
        'pt': ':vassoura:',
        'it': ':scopa:',
        'fa': ':جارو:',
        'id': ':sapu:',
        'zh': ':扫帚:',
        'ru': ':метла:'
    },
    '\U0001F7E4': {  # 🟤
        'en': ':brown_circle:',
        'status': fully_qualified,
        'E': 12,
        'de': ':brauner_punkt:',
        'es': ':círculo_marrón:',
        'fr': ':disque_marron:',
        'ja': ':茶色の丸:',
        'ko': ':갈색_원:',
        'pt': ':círculo_marrom:',
        'it': ':cerchio_marrone:',
        'fa': ':دایره_قهوه_ای:',
        'id': ':lingkaran_cokelat:',
        'zh': ':棕色圆:',
        'ru': ':коричневый_круг:'
    },
    '\U0001F90E': {  # 🤎
        'en': ':brown_heart:',
        'status': fully_qualified,
        'E': 12,
        'de': ':braunes_herz:',
        'es': ':corazón_marrón:',
        'fr': ':cœur_marron:',
        'ja': ':茶色いハート:',
        'ko': ':갈색_하트:',
        'pt': ':coração_marrom:',
        'it': ':cuore_marrone:',
        'fa': ':قلب_قهوه_ای:',
        'id': ':hati_cokelat:',
        'zh': ':棕心:',
        'ru': ':коричневое_сердце:'
    },
    '\U0001F7EB': {  # 🟫
        'en': ':brown_square:',
        'status': fully_qualified,
        'E': 12,
        'de': ':braunes_quadrat:',
        'es': ':cuadrado_marrón:',
        'fr': ':carré_marron:',
        'ja': ':茶色い四角:',
        'ko': ':갈색_사각형:',
        'pt': ':quadrado_marrom:',
        'it': ':quadrato_marrone:',
        'fa': ':مربع_قهوه_ای:',
        'id': ':persegi_cokelat:',
        'zh': ':棕色方块:',
        'ru': ':коричневый_квадрат:'
    },
    '\U0001F9CB': {  # 🧋
        'en': ':bubble_tea:',
        'status': fully_qualified,
        'E': 13,
        'de': ':bubble_tea:',
        'es': ':té_de_burbujas:',
        'fr': ':thé_aux_perles:',
        'ja': ':タピオカドリンク:',
        'ko': ':버블티:',
        'pt': ':chá_perolado:',
        'it': ':bubble_tea:',
        'fa': ':چای_حُبابی:',
        'id': ':teh_bubble:',
        'zh': ':珍珠奶茶:',
        'ru': ':чай_с_шариками:'
    },
    '\U0001FAE7': {  # 🫧
        'en': ':bubbles:',
        'status': fully_qualified,
        'E': 14,
        'de': ':blasen:',
        'es': ':burbujas:',
        'fr': ':bulles:',
        'ja': ':泡:',
        'ko': ':거품:',
        'pt': ':bolhas:',
        'it': ':bolle:',
        'fa': ':حباب:',
        'id': ':gelembung:',
        'zh': ':气泡:',
        'ru': ':пузыри:'
    },
    '\U0001FAA3': {  # 🪣
        'en': ':bucket:',
        'status': fully_qualified,
        'E': 13,
        'de': ':eimer:',
        'es': ':cubo:',
        'fr': ':seau:',
        'ja': ':バケツ:',
        'ko': ':양동이:',
        'pt': ':balde:',
        'it': ':secchio:',
        'fa': ':سطل:',
        'id': ':ember:',
        'zh': ':桶:',
        'ru': ':ведро:'
    },
    '\U0001F41B': {  # 🐛
        'en': ':bug:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':raupe:',
        'es': ':bicho:',
        'fr': ':chenille:',
        'ja': ':毛虫:',
        'ko': ':송충이:',
        'pt': ':inseto:',
        'it': ':insetto:',
        'fa': ':حشره:',
        'id': ':serangga_kecil:',
        'zh': ':毛毛虫:',
        'ru': ':гусеница:'
    },
    '\U0001F3D7\U0000FE0F': {  # 🏗️
        'en': ':building_construction:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':kran:',
        'es': ':construcción:',
        'fr': ':construction_d’un_bâtiment:',
        'ja': ':建設中:',
        'ko': ':건물_공사:',
        'pt': ':construção:',
        'it': ':edificio_in_costruzione:',
        'fa': ':ساخت_وساز_ساختمانی:',
        'id': ':konstruksi_bangunan:',
        'zh': ':施工:',
        'ru': ':строика:'
    },
    '\U0001F3D7': {  # 🏗
        'en': ':building_construction:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':kran:',
        'es': ':construcción:',
        'fr': ':construction_d’un_bâtiment:',
        'ja': ':建設中:',
        'ko': ':건물_공사:',
        'pt': ':construção:',
        'it': ':edificio_in_costruzione:',
        'fa': ':ساخت_وساز_ساختمانی:',
        'id': ':konstruksi_bangunan:',
        'zh': ':施工:',
        'ru': ':стройка:'
    },
    '\U0001F685': {  # 🚅
        'en': ':bullet_train:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':bullettrain_front:'],
        'de': ':hochgeschwindigkeitszug:',
        'es': ':tren_bala:',
        'fr': ':train_à_grande_vitesse:',
        'ja': ':0系新幹線:',
        'ko': ':고속철:',
        'pt': ':trem_de_alta_velocidade_japonês:',
        'it': ':treno_alta_velocità_punta_arrotondata:',
        'fa': ':قطار_تندرو:',
        'id': ':kereta_ekspres_peluru:',
        'zh': ':子弹头高速列车:',
        'ru': ':поезд_пуля:'
    },
    '\U0001F3AF': {  # 🎯
        'en': ':bullseye:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':dart:'],
        'de': ':darts:',
        'es': ':diana:',
        'fr': ':dans_le_mille:',
        'ja': ':的:',
        'ko': ':과녁_명중:',
        'pt': ':no_alvo:',
        'it': ':bersaglio:',
        'fa': ':پیکان_در_قلب_هدف:',
        'id': ':dart:',
        'zh': ':正中靶心的飞镖:',
        'ru': ':мишень:'
    },
    '\U0001F32F': {  # 🌯
        'en': ':burrito:',
        'status': fully_qualified,
        'E': 1,
        'de': ':burrito:',
        'es': ':burrito:',
        'fr': ':burrito:',
        'ja': ':ブリトー:',
        'ko': ':부리또:',
        'pt': ':burrito:',
        'it': ':burrito:',
        'fa': ':بوریتو:',
        'id': ':burrito:',
        'zh': ':墨西哥玉米煎饼:',
        'ru': ':буррито:'
    },
    '\U0001F68C': {  # 🚌
        'en': ':bus:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':bus:',
        'es': ':autobús:',
        'fr': ':bus:',
        'ja': ':バス:',
        'ko': ':버스:',
        'pt': ':ônibus:',
        'it': ':bus:',
        'fa': ':اتوبوس:',
        'id': ':bus:',
        'zh': ':公交车:',
        'ru': ':автобус:'
    },
    '\U0001F68F': {  # 🚏
        'en': ':bus_stop:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':busstop:'],
        'de': ':bushaltestelle:',
        'es': ':parada_de_autobús:',
        'fr': ':arrêt_de_bus:',
        'ja': ':バス停:',
        'ko': ':버스_정류장:',
        'pt': ':ponto_de_ônibus:',
        'it': ':fermata_dell’autobus:',
        'fa': ':ایستگاه_اتوبوس:',
        'id': ':halte_bus:',
        'zh': ':公交车站:',
        'ru': ':автобусная_остановка:'
    },
    '\U0001F464': {  # 👤
        'en': ':bust_in_silhouette:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':silhouette_einer_büste:',
        'es': ':silueta_de_busto:',
        'fr': ':silhouette_de_buste:',
        'ja': ':人のシルエット:',
        'ko': ':사람_그림자:',
        'pt': ':silhueta_de_busto:',
        'it': ':profilo_di_persona:',
        'fa': ':تندیس_سایه_نما:',
        'id': ':siluet_foto_setengah_badan:',
        'zh': ':人像:',
        'ru': ':силуэт_человека:'
    },
    '\U0001F465': {  # 👥
        'en': ':busts_in_silhouette:',
        'status': fully_qualified,
        'E': 1,
        'de': ':silhouette_mehrerer_büsten:',
        'es': ':dos_siluetas_de_bustos:',
        'fr': ':silhouettes_de_bustes:',
        'ja': ':2人のシルエット:',
        'ko': ':사람들_그림자:',
        'pt': ':silhueta_de_bustos:',
        'it': ':profilo_di_due_persone:',
        'fa': ':تندیس_های_سایه_نما:',
        'id': ':beberapa_siluet_foto_setengah_badan:',
        'zh': ':双人像:',
        'ru': ':два_силуэта:'
    },
    '\U0001F9C8': {  # 🧈
        'en': ':butter:',
        'status': fully_qualified,
        'E': 12,
        'de': ':butter:',
        'es': ':mantequilla:',
        'fr': ':beurre:',
        'ja': ':バター:',
        'ko': ':버터:',
        'pt': ':manteiga:',
        'it': ':burro:',
        'fa': ':کره:',
        'id': ':mentega:',
        'zh': ':黄油:',
        'ru': ':масло:'
    },
    '\U0001F98B': {  # 🦋
        'en': ':butterfly:',
        'status': fully_qualified,
        'E': 3,
        'de': ':schmetterling:',
        'es': ':mariposa:',
        'fr': ':papillon:',
        'ja': ':チョウ:',
        'ko': ':나비:',
        'pt': ':borboleta:',
        'it': ':farfalla:',
        'fa': ':پروانه:',
        'id': ':kupu-kupu:',
        'zh': ':蝴蝶:',
        'ru': ':бабочка:'
    },
    '\U0001F335': {  # 🌵
        'en': ':cactus:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kaktus:',
        'es': ':cactus:',
        'fr': ':cactus:',
        'ja': ':サボテン:',
        'ko': ':선인장:',
        'pt': ':cacto:',
        'it': ':cactus:',
        'fa': ':کاکتوس:',
        'id': ':kaktus:',
        'zh': ':仙人掌:',
        'ru': ':кактус:'
    },
    '\U0001F4C5': {  # 📅
        'en': ':calendar:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':date:'],
        'de': ':kalender:',
        'es': ':calendario:',
        'fr': ':calendrier:',
        'ja': ':カレンダー:',
        'ko': ':달력:',
        'pt': ':calendário:',
        'it': ':calendario:',
        'fa': ':تقویم:',
        'id': ':kalender:',
        'zh': ':日历:',
        'ru': ':календарь:'
    },
    '\U0001F919': {  # 🤙
        'en': ':call_me_hand:',
        'status': fully_qualified,
        'E': 3,
        'de': ':ruf-mich-an-handzeichen:',
        'es': ':mano_haciendo_el_gesto_de_llamar:',
        'fr': ':signe_appel_téléphonique_avec_les_doigts:',
        'ja': ':電話の合図:',
        'ko': ':전화를_걸라는_손_모양:',
        'pt': ':sinal_me_liga:',
        'it': ':mano_con_gesto_di_chiamata:',
        'fa': ':دست_به_شکل_تلفن:',
        'id': ':panggil_saya:',
        'zh': ':给我打电话:',
        'ru': ':жест_«позвони_мне»:'
    },
    '\U0001F919\U0001F3FF': {  # 🤙🏿
        'en': ':call_me_hand_dark_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':ruf-mich-an-handzeichen_dunkle_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_oscuro:',
        'fr': ':signe_appel_téléphonique_avec_les_doigts_peau_foncée:',
        'ja': ':電話の合図_濃い肌色:',
        'ko': ':전화를_걸라는_손_모양_검은색_피부:',
        'pt': ':sinal_me_liga_pele_escura:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_scura:',
        'fa': ':دست_به_شکل_تلفن_پوست_آبنوسی:',
        'id': ':panggil_saya_warna_kulit_gelap:',
        'zh': ':给我打电话_较深肤色:',
        'ru': ':жест_«позвони_мне»_очень_темный_тон_кожи:'
    },
    '\U0001F919\U0001F3FB': {  # 🤙🏻
        'en': ':call_me_hand_light_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':ruf-mich-an-handzeichen_helle_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_claro:',
        'fr': ':signe_appel_téléphonique_avec_les_doigts_peau_claire:',
        'ja': ':電話の合図_薄い肌色:',
        'ko': ':전화를_걸라는_손_모양_하얀_피부:',
        'pt': ':sinal_me_liga_pele_clara:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_chiara:',
        'fa': ':دست_به_شکل_تلفن_پوست_سفید:',
        'id': ':panggil_saya_warna_kulit_cerah:',
        'zh': ':给我打电话_较浅肤色:',
        'ru': ':жест_«позвони_мне»_очень_светлый_тон_кожи:'
    },
    '\U0001F919\U0001F3FE': {  # 🤙🏾
        'en': ':call_me_hand_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':ruf-mich-an-handzeichen_mitteldunkle_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_oscuro_medio:',
        'fr': ':signe_appel_téléphonique_avec_les_doigts_peau_mate:',
        'ja': ':電話の合図_やや濃い肌色:',
        'ko': ':전화를_걸라는_손_모양_진한_갈색_피부:',
        'pt': ':sinal_me_liga_pele_morena_escura:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_abbastanza_scura:',
        'fa': ':دست_به_شکل_تلفن_پوست_گندمی:',
        'id': ':panggil_saya_warna_kulit_gelap-sedang:',
        'zh': ':给我打电话_中等深肤色:',
        'ru': ':жест_«позвони_мне»_темный_тон_кожи:'
    },
    '\U0001F919\U0001F3FC': {  # 🤙🏼
        'en': ':call_me_hand_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':ruf-mich-an-handzeichen_mittelhelle_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_claro_medio:',
        'fr': ':signe_appel_téléphonique_avec_les_doigts_peau_moyennement_claire:',
        'ja': ':電話の合図_やや薄い肌色:',
        'ko': ':전화를_걸라는_손_모양_연한_갈색_피부:',
        'pt': ':sinal_me_liga_pele_morena_clara:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_abbastanza_chiara:',
        'fa': ':دست_به_شکل_تلفن_پوست_روشن:',
        'id': ':panggil_saya_warna_kulit_cerah-sedang:',
        'zh': ':给我打电话_中等浅肤色:',
        'ru': ':жест_«позвони_мне»_светлый_тон_кожи:'
    },
    '\U0001F919\U0001F3FD': {  # 🤙🏽
        'en': ':call_me_hand_medium_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':ruf-mich-an-handzeichen_mittlere_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_medio:',
        'fr': ':signe_appel_téléphonique_avec_les_doigts_peau_légèrement_mate:',
        'ja': ':電話の合図_中間の肌色:',
        'ko': ':전화를_걸라는_손_모양_갈색_피부:',
        'pt': ':sinal_me_liga_pele_morena:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_olivastra:',
        'fa': ':دست_به_شکل_تلفن_پوست_طلایی:',
        'id': ':panggil_saya_warna_kulit_sedang:',
        'zh': ':给我打电话_中等肤色:',
        'ru': ':жест_«позвони_мне»_средний_тон_кожи:'
    },
    '\U0001F42A': {  # 🐪
        'en': ':camel:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':dromedary_camel:'],
        'de': ':dromedar:',
        'es': ':dromedario:',
        'fr': ':dromadaire:',
        'ja': ':ラクダ:',
        'ko': ':낙타:',
        'pt': ':camelo:',
        'it': ':dromedario:',
        'fa': ':شتر:',
        'id': ':unta:',
        'zh': ':骆驼:',
        'ru': ':одногорбый_верблюд:'
    },
    '\U0001F4F7': {  # 📷
        'en': ':camera:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':fotoapparat:',
        'es': ':cámara_de_fotos:',
        'fr': ':appareil_photo:',
        'ja': ':カメラ:',
        'ko': ':카메라:',
        'pt': ':câmera:',
        'it': ':fotocamera:',
        'fa': ':دوربین:',
        'id': ':kamera:',
        'zh': ':相机:',
        'ru': ':фотоаппарат:'
    },
    '\U0001F4F8': {  # 📸
        'en': ':camera_with_flash:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':camera_flash:'],
        'de': ':fotoapparat_mit_blitz:',
        'es': ':cámara_con_flash:',
        'fr': ':appareil_photo_avec_flash:',
        'ja': ':フラッシュを焚いているカメラ:',
        'ko': ':플래시를_터트리고_있는_카메라:',
        'pt': ':câmera_com_flash:',
        'it': ':fotocamera_con_flash:',
        'fa': ':دوربین_با_فلاش:',
        'id': ':kamera_dengan_flash:',
        'zh': ':开闪光灯的相机:',
        'ru': ':фотоаппарат_со_вспышкой:'
    },
    '\U0001F3D5\U0000FE0F': {  # 🏕️
        'en': ':camping:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':camping:',
        'es': ':camping:',
        'fr': ':camping:',
        'ja': ':キャンプ:',
        'ko': ':캠핑:',
        'pt': ':acampamento:',
        'it': ':campeggio:',
        'fa': ':چادرزنی:',
        'id': ':berkemah:',
        'zh': ':露营:',
        'ru': ':кемпинг:'
    },
    '\U0001F3D5': {  # 🏕
        'en': ':camping:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':camping:',
        'es': ':camping:',
        'fr': ':camping:',
        'ja': ':キャンプ:',
        'ko': ':캠핑:',
        'pt': ':acampamento:',
        'it': ':campeggio:',
        'fa': ':چادرزنی:',
        'id': ':berkemah:',
        'zh': ':露营:',
        'ru': ':кемпинг:'
    },
    '\U0001F56F\U0000FE0F': {  # 🕯️
        'en': ':candle:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':kerze:',
        'es': ':vela:',
        'fr': ':bougie:',
        'ja': ':ろうそく:',
        'ko': ':양초:',
        'pt': ':vela:',
        'it': ':candela:',
        'fa': ':شمع:',
        'id': ':lilin:',
        'zh': ':蜡烛:',
        'ru': ':свеча:'
    },
    '\U0001F56F': {  # 🕯
        'en': ':candle:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':kerze:',
        'es': ':vela:',
        'fr': ':bougie:',
        'ja': ':ろうそく:',
        'ko': ':양초:',
        'pt': ':vela:',
        'it': ':candela:',
        'fa': ':شمع:',
        'id': ':lilin:',
        'zh': ':蜡烛:',
        'ru': ':свеча:'
    },
    '\U0001F36C': {  # 🍬
        'en': ':candy:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':bonbon:',
        'es': ':caramelo:',
        'fr': ':bonbon:',
        'ja': ':キャンディ:',
        'ko': ':사탕:',
        'pt': ':bala:',
        'it': ':caramella:',
        'fa': ':آب_نبات_:',
        'id': ':permen:',
        'zh': ':糖:',
        'ru': ':конфета:'
    },
    '\U0001F96B': {  # 🥫
        'en': ':canned_food:',
        'status': fully_qualified,
        'E': 5,
        'de': ':konserve:',
        'es': ':comida_enlatada:',
        'fr': ':aliments_en_conserve:',
        'ja': ':缶詰:',
        'ko': ':통조림:',
        'pt': ':comida_enlatada:',
        'it': ':cibo_in_scatola:',
        'fa': ':غذای_کنسروشده:',
        'id': ':makanan_kaleng:',
        'zh': ':罐头食品:',
        'ru': ':консервы:'
    },
    '\U0001F6F6': {  # 🛶
        'en': ':canoe:',
        'status': fully_qualified,
        'E': 3,
        'de': ':kanu:',
        'es': ':canoa:',
        'fr': ':canoë:',
        'ja': ':カヌー:',
        'ko': ':카누:',
        'pt': ':canoa:',
        'it': ':canoa:',
        'fa': ':بلم:',
        'id': ':kano:',
        'zh': ':独木舟:',
        'ru': ':каноэ:'
    },
    '\U0001F5C3\U0000FE0F': {  # 🗃️
        'en': ':card_file_box:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':karteikasten:',
        'es': ':archivador_de_tarjetas:',
        'fr': ':boîte_à_dossiers:',
        'ja': ':カードファイルボックス:',
        'ko': ':카드_파일_상자:',
        'pt': ':caixa_de_arquivos:',
        'it': ':schedario_da_tavolo:',
        'fa': ':جعبه_نگه_داری_کارت:',
        'id': ':kotak_file_kartu:',
        'zh': ':卡片盒:',
        'ru': ':картотека:'
    },
    '\U0001F5C3': {  # 🗃
        'en': ':card_file_box:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':karteikasten:',
        'es': ':archivador_de_tarjetas:',
        'fr': ':boîte_à_dossiers:',
        'ja': ':カードファイルボックス:',
        'ko': ':카드_파일_상자:',
        'pt': ':caixa_de_arquivos:',
        'it': ':schedario_da_tavolo:',
        'fa': ':جعبه_نگه_داری_کارت:',
        'id': ':kotak_file_kartu:',
        'zh': ':卡片盒:',
        'ru': ':картотека:'
    },
    '\U0001F4C7': {  # 📇
        'en': ':card_index:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':rotationskartei:',
        'es': ':organizador_de_fichas:',
        'fr': ':carnet_d’adresses:',
        'ja': ':カードインデックス:',
        'ko': ':카드_인덱스:',
        'pt': ':índice_de_cartões:',
        'it': ':schedario:',
        'fa': ':کارت_های_راهنما:',
        'id': ':indeks_kartu:',
        'zh': ':卡片索引:',
        'ru': ':органайзер:'
    },
    '\U0001F5C2\U0000FE0F': {  # 🗂️
        'en': ':card_index_dividers:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':karteibahnen:',
        'es': ':separador_de_fichas:',
        'fr': ':intercalaires:',
        'ja': ':カードフォルダー:',
        'ko': ':카드_색인_파일:',
        'pt': ':divisores_de_pastas:',
        'it': ':divisori_per_schedario:',
        'fa': ':جداکننده:',
        'id': ':pembagi_indeks_kartu:',
        'zh': ':索引分隔文件夹:',
        'ru': ':маркированные_папки:'
    },
    '\U0001F5C2': {  # 🗂
        'en': ':card_index_dividers:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':karteireiter:',
        'es': ':separador_de_fichas:',
        'fr': ':intercalaires:',
        'ja': ':カードフォルダー:',
        'ko': ':카드_색인_파일:',
        'pt': ':divisores_de_pastas:',
        'it': ':divisori_per_schedario:',
        'fa': ':جداکننده:',
        'id': ':pembagi_indeks_kartu:',
        'zh': ':索引分隔文件夹:',
        'ru': ':маркированные_папки:'
    },
    '\U0001F3A0': {  # 🎠
        'en': ':carousel_horse:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':karussellpferd:',
        'es': ':caballo_de_tiovivo:',
        'fr': ':cheval_de_manège:',
        'ja': ':メリーゴーランド:',
        'ko': ':회전_목마:',
        'pt': ':carrossel:',
        'it': ':cavallo_da_giostra:',
        'fa': ':چرخ_وفلک_اسبی:',
        'id': ':komidi_putar:',
        'zh': ':旋转木马:',
        'ru': ':лошадь_на_карусели:'
    },
    '\U0001F38F': {  # 🎏
        'en': ':carp_streamer:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flags:'],
        'de': ':traditionelle_japanische_windsäcke:',
        'es': ':banderín_de_carpas:',
        'fr': ':koinobori:',
        'ja': ':こいのぼり:',
        'ko': ':물고기_깃발:',
        'pt': ':bandeira_de_carpas:',
        'it': ':aquilone_a_forma_di_carpa:',
        'fa': ':پرچم_های_ماهی_در_باد:',
        'id': ':bendera_ikan_koi:',
        'zh': ':鲤鱼旗:',
        'ru': ':вымпелы_в_виде_карпов:'
    },
    '\U0001FA9A': {  # 🪚
        'en': ':carpentry_saw:',
        'status': fully_qualified,
        'E': 13,
        'de': ':handsäge:',
        'es': ':sierra_de_carpintería:',
        'fr': ':scie:',
        'ja': ':のこぎり:',
        'ko': ':목공_톱:',
        'pt': ':serrote:',
        'it': ':sega_da_falegname:',
        'fa': ':اره_نجاری:',
        'id': ':gergaji_kayu:',
        'zh': ':木工锯:',
        'ru': ':пила_по_дереву:'
    },
    '\U0001F955': {  # 🥕
        'en': ':carrot:',
        'status': fully_qualified,
        'E': 3,
        'de': ':karotte:',
        'es': ':zanahoria:',
        'fr': ':carotte:',
        'ja': ':人参:',
        'ko': ':당근:',
        'pt': ':cenoura:',
        'it': ':carota:',
        'fa': ':هویج:',
        'id': ':wortel:',
        'zh': ':胡萝卜:',
        'ru': ':морковь:'
    },
    '\U0001F3F0': {  # 🏰
        'en': ':castle:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':european_castle:'],
        'de': ':schloss:',
        'es': ':castillo_europeo:',
        'fr': ':château:',
        'ja': ':西洋の城:',
        'ko': ':유럽_성:',
        'pt': ':castelo:',
        'it': ':castello:',
        'fa': ':قلعه:',
        'id': ':kastel:',
        'zh': ':欧洲城堡:',
        'ru': ':замок:'
    },
    '\U0001F408': {  # 🐈
        'en': ':cat:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':cat2:'],
        'variant': True,
        'de': ':katze:',
        'es': ':gato:',
        'fr': ':chat:',
        'ja': ':ネコ:',
        'ko': ':고양이:',
        'pt': ':gato:',
        'it': ':gatto:',
        'fa': ':گربه:',
        'id': ':kucing:',
        'zh': ':猫:',
        'ru': ':кошка:'
    },
    '\U0001F431': {  # 🐱
        'en': ':cat_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cat:'],
        'de': ':katzengesicht:',
        'es': ':cara_de_gato:',
        'fr': ':tête_de_chat:',
        'ja': ':ネコの顔:',
        'ko': ':고양이_얼굴:',
        'pt': ':rosto_de_gato:',
        'it': ':muso_di_gatto:',
        'fa': ':صورت_گربه:',
        'id': ':wajah_kucing:',
        'zh': ':猫脸:',
        'ru': ':морда_кошки:'
    },
    '\U0001F639': {  # 😹
        'en': ':cat_with_tears_of_joy:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':joy_cat:'],
        'de': ':katze_mit_freudentränen:',
        'es': ':gato_llorando_de_risa:',
        'fr': ':chat_qui_pleure_de_joie:',
        'ja': ':嬉し泣きする猫:',
        'ko': ':기쁨의_눈물을_흘리는_고양이_얼굴:',
        'pt': ':rosto_de_gato_com_lágrimas_de_alegria:',
        'it': ':gatto_con_lacrime_di_gioia:',
        'fa': ':گربهٔ_خندان_با_اشک_شوق:',
        'id': ':wajah_kucing_gembira_berurai_air_mata:',
        'zh': ':笑出眼泪的猫:',
        'ru': ':смеющийся_до_слез_кот:'
    },
    '\U0001F63C': {  # 😼
        'en': ':cat_with_wry_smile:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':smirk_cat:'],
        'de': ':verwegen_lächelnde_katze:',
        'es': ':gato_haciendo_una_mueca:',
        'fr': ':chat_avec_sourire_en_coin:',
        'ja': ':にやりとする猫:',
        'ko': ':썩소_짓는_고양이:',
        'pt': ':rosto_de_gato_com_sorriso_irônico:',
        'it': ':gatto_con_sorriso_sarcastico:',
        'fa': ':گربه_با_پوزخند:',
        'id': ':wajah_kucing_tersenyum_sombong:',
        'zh': ':奸笑的猫:',
        'ru': ':ухмыляющийся_кот:'
    },
    '\U000026D3\U0000FE0F': {  # ⛓️
        'en': ':chains:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':ketten:',
        'es': ':cadenas:',
        'fr': ':chaînes:',
        'ja': ':鎖:',
        'ko': ':쇠사슬:',
        'pt': ':correntes:',
        'it': ':catene:',
        'fa': ':غل_و_زنجیر:',
        'id': ':rantai:',
        'zh': ':链条:',
        'ru': ':цепь:'
    },
    '\U000026D3': {  # ⛓
        'en': ':chains:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':ketten:',
        'es': ':cadenas:',
        'fr': ':chaînes:',
        'ja': ':鎖:',
        'ko': ':쇠사슬:',
        'pt': ':correntes:',
        'it': ':catene:',
        'fa': ':غل_و_زنجیر:',
        'id': ':rantai:',
        'zh': ':链条:',
        'ru': ':цепь:'
    },
    '\U0001FA91': {  # 🪑
        'en': ':chair:',
        'status': fully_qualified,
        'E': 12,
        'de': ':stuhl:',
        'es': ':silla:',
        'fr': ':chaise:',
        'ja': ':椅子:',
        'ko': ':의자:',
        'pt': ':cadeira:',
        'it': ':sedia:',
        'fa': ':صندلی:',
        'id': ':bangku:',
        'zh': ':椅子:',
        'ru': ':стул:'
    },
    '\U0001F4C9': {  # 📉
        'en': ':chart_decreasing:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':chart_with_downwards_trend:'],
        'de': ':abwärtstrend:',
        'es': ':gráfica_de_evolución_descendente:',
        'fr': ':graphique_en_baisse:',
        'ja': ':グラフ下向き:',
        'ko': ':내려가는_그래프:',
        'pt': ':gráfico_caindo:',
        'it': ':grafico_con_andamento_negativo:',
        'fa': ':نمودار_نزولی:',
        'id': ':bagan_menurun:',
        'zh': ':趋势向下的图表:',
        'ru': ':диаграмма_падения:'
    },
    '\U0001F4C8': {  # 📈
        'en': ':chart_increasing:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':chart_with_upwards_trend:'],
        'de': ':aufwärtstrend:',
        'es': ':gráfica_de_evolución_ascendente:',
        'fr': ':graphique_en_hausse:',
        'ja': ':グラフ上向き:',
        'ko': ':상승하는_그래프:',
        'pt': ':gráfico_subindo:',
        'it': ':grafico_con_andamento_positivo:',
        'fa': ':نمودار_صعودی:',
        'id': ':bagan_meningkat:',
        'zh': ':趋势向上的图表:',
        'ru': ':диаграмма_роста:'
    },
    '\U0001F4B9': {  # 💹
        'en': ':chart_increasing_with_yen:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':chart:'],
        'de': ':steigende_kurve_mit_yen-zeichen:',
        'es': ':mercado_al_alza:',
        'fr': ':courbe_avec_yen_en_hausse:',
        'ja': ':為替:',
        'ko': ':엔화_관련_상승하는_그래프:',
        'pt': ':gráfico_subindo_com_iene:',
        'it': ':grafico_ascendente:',
        'fa': ':نمودار_صعودی_با_ین:',
        'id': ':bagan_meningkat_dengan_yen:',
        'zh': ':趋势向上且带有日元符号的图表:',
        'ru': ':биржа:'
    },
    '\U00002611\U0000FE0F': {  # ☑️
        'en': ':check_box_with_check:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':ballot_box_with_check:'],
        'variant': True,
        'de': ':abstimmungsfeld_mit_häkchen:',
        'es': ':casilla_con_marca_de_verificación:',
        'fr': ':case_cochée:',
        'ja': ':チェックボックス:',
        'ko': ':체크박스:',
        'pt': ':caixa_de_seleção_marcada_com_tique:',
        'it': ':riquadro_con_spunta:',
        'fa': ':مربع_با_علامت_تیک:',
        'id': ':kotak_bercentang:',
        'zh': ':勾选框:',
        'ru': ':галочка_в_квадрате:'
    },
    '\U00002611': {  # ☑
        'en': ':check_box_with_check:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':ballot_box_with_check:'],
        'variant': True,
        'de': ':abgehaktes_kästchen:',
        'es': ':casilla_con_marca_de_verificación:',
        'fr': ':case_cochée:',
        'ja': ':チェックボックス:',
        'ko': ':체크박스:',
        'pt': ':caixa_de_seleção_marcada_com_tique:',
        'it': ':riquadro_con_spunta:',
        'fa': ':مربع_با_علامت_تیک:',
        'id': ':kotak_bercentang:',
        'zh': ':勾选框:',
        'ru': ':галочка_в_квадрате:'
    },
    '\U00002714\U0000FE0F': {  # ✔️
        'en': ':check_mark:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':heavy_check_mark:'],
        'variant': True,
        'de': ':kräftiges_häkchen:',
        'es': ':marca_de_verificación:',
        'fr': ':coche:',
        'ja': ':太字のチェック:',
        'ko': ':진한_체크_표시:',
        'pt': ':marca_de_seleção:',
        'it': ':segno_di_spunta_nero:',
        'fa': ':علامت_تیک:',
        'id': ':tanda_centang_tebal:',
        'zh': ':勾号:',
        'ru': ':галочка:'
    },
    '\U00002714': {  # ✔
        'en': ':check_mark:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':heavy_check_mark:'],
        'variant': True,
        'de': ':kräftiges_häkchen:',
        'es': ':marca_de_verificación:',
        'fr': ':coche:',
        'ja': ':太字のチェック:',
        'ko': ':진한_체크_표시:',
        'pt': ':marca_de_seleção:',
        'it': ':segno_di_spunta:',
        'fa': ':علامت_تیک:',
        'id': ':tanda_centang_tebal:',
        'zh': ':勾号:',
        'ru': ':галочка:'
    },
    '\U00002705': {  # ✅
        'en': ':check_mark_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':white_check_mark:'],
        'de': ':weißes_häkchen:',
        'es': ':botón_de_marca_de_verificación:',
        'fr': ':bouton_coché:',
        'ja': ':白抜きのチェック:',
        'ko': ':체크_표시:',
        'pt': ':marca_de_seleção_branca:',
        'it': ':pulsante_segno_di_spunta:',
        'fa': ':علامت_تأیید:',
        'id': ':tanda_centang_putih_tebal:',
        'zh': ':勾号按钮:',
        'ru': ':белая_галочка:'
    },
    '\U0001F9C0': {  # 🧀
        'en': ':cheese_wedge:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':cheese:'],
        'de': ':käsestück:',
        'es': ':cuña_de_queso:',
        'fr': ':part_de_fromage:',
        'ja': ':チーズ:',
        'ko': ':치즈_조각:',
        'pt': ':queijo:',
        'it': ':fetta_di_formaggio:',
        'fa': ':قاچ_پنیر:',
        'id': ':irisan_keju:',
        'zh': ':芝士:',
        'ru': ':сыр:'
    },
    '\U0001F3C1': {  # 🏁
        'en': ':chequered_flag:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':checkered_flag:'],
        'de': ':zielflagge:',
        'es': ':bandera_de_cuadros:',
        'fr': ':drapeau_à_damier:',
        'ja': ':チェッカーフラッグ:',
        'ko': ':격자_무늬_깃발:',
        'pt': ':bandeira_quadriculada:',
        'it': ':bandiera_a_scacchi:',
        'fa': ':پرچم_شطرنجی:',
        'id': ':bendera_kotak-kotak:',
        'zh': ':终点旗:',
        'ru': ':клетчатый_флаг:'
    },
    '\U0001F352': {  # 🍒
        'en': ':cherries:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kirschen:',
        'es': ':cerezas:',
        'fr': ':cerises:',
        'ja': ':さくらんぼ:',
        'ko': ':체리:',
        'pt': ':cereja:',
        'it': ':ciliegie:',
        'fa': ':گیلاس:',
        'id': ':ceri:',
        'zh': ':樱桃:',
        'ru': ':вишня:'
    },
    '\U0001F338': {  # 🌸
        'en': ':cherry_blossom:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kirschblüte:',
        'es': ':flor_de_cerezo:',
        'fr': ':fleur_de_cerisier:',
        'ja': ':桜:',
        'ko': ':벚꽃:',
        'pt': ':flor_de_cerejeira:',
        'it': ':fiore_di_ciliegio:',
        'fa': ':شکوفه_گیلاس:',
        'id': ':sakura:',
        'zh': ':樱花:',
        'ru': ':цветок_вишни:'
    },
    '\U0000265F\U0000FE0F': {  # ♟️
        'en': ':chess_pawn:',
        'status': fully_qualified,
        'E': 11,
        'variant': True,
        'de': ':bauer_schach:',
        'es': ':peón_de_ajedrez:',
        'fr': ':pion_d’échec:',
        'ja': ':チェスの駒:',
        'ko': ':체스_폰:',
        'pt': ':peão_de_xadrez:',
        'it': ':pedina_degli_scacchi:',
        'fa': ':پیاده_شطرنج:',
        'id': ':pion_catur:',
        'zh': ':兵:',
        'ru': ':пешка:'
    },
    '\U0000265F': {  # ♟
        'en': ':chess_pawn:',
        'status': unqualified,
        'E': 11,
        'variant': True,
        'de': ':bauer_schach:',
        'es': ':peón_de_ajedrez:',
        'fr': ':pion_d’échec:',
        'ja': ':チェスの駒:',
        'ko': ':체스_폰:',
        'pt': ':peão_de_xadrez:',
        'it': ':pedina_degli_scacchi:',
        'fa': ':پیاده_شطرنج:',
        'id': ':pion_catur:',
        'zh': ':兵:',
        'ru': ':пешка:'
    },
    '\U0001F330': {  # 🌰
        'en': ':chestnut:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kastanie:',
        'es': ':castaña:',
        'fr': ':châtaigne:',
        'ja': ':くり:',
        'ko': ':밤:',
        'pt': ':castanha:',
        'it': ':castagna:',
        'fa': ':فندق:',
        'id': ':kastanye:',
        'zh': ':栗子:',
        'ru': ':каштан:'
    },
    '\U0001F414': {  # 🐔
        'en': ':chicken:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':huhn:',
        'es': ':gallina:',
        'fr': ':poule:',
        'ja': ':にわとり:',
        'ko': ':닭:',
        'pt': ':galinha:',
        'it': ':gallina:',
        'fa': ':مرغ:',
        'id': ':ayam:',
        'zh': ':鸡:',
        'ru': ':курица:'
    },
    '\U0001F9D2': {  # 🧒
        'en': ':child:',
        'status': fully_qualified,
        'E': 5,
        'de': ':kind:',
        'es': ':infante:',
        'fr': ':enfant:',
        'ja': ':子供:',
        'ko': ':어린이:',
        'pt': ':criança:',
        'it': ':bimbo:',
        'fa': ':بچه:',
        'id': ':anak:',
        'zh': ':儿童:',
        'ru': ':ребенок:'
    },
    '\U0001F9D2\U0001F3FF': {  # 🧒🏿
        'en': ':child_dark_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':kind_dunkle_hautfarbe:',
        'es': ':infante_tono_de_piel_oscuro:',
        'fr': ':enfant_peau_foncée:',
        'ja': ':子供_濃い肌色:',
        'ko': ':어린이_검은색_피부:',
        'pt': ':criança_pele_escura:',
        'it': ':bimbo_carnagione_scura:',
        'fa': ':بچه_پوست_آبنوسی:',
        'id': ':anak_warna_kulit_gelap:',
        'zh': ':儿童_较深肤色:',
        'ru': ':ребенок_очень_темный_тон_кожи:'
    },
    '\U0001F9D2\U0001F3FB': {  # 🧒🏻
        'en': ':child_light_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':kind_helle_hautfarbe:',
        'es': ':infante_tono_de_piel_claro:',
        'fr': ':enfant_peau_claire:',
        'ja': ':子供_薄い肌色:',
        'ko': ':어린이_하얀_피부:',
        'pt': ':criança_pele_clara:',
        'it': ':bimbo_carnagione_chiara:',
        'fa': ':بچه_پوست_سفید:',
        'id': ':anak_warna_kulit_cerah:',
        'zh': ':儿童_较浅肤色:',
        'ru': ':ребенок_очень_светлый_тон_кожи:'
    },
    '\U0001F9D2\U0001F3FE': {  # 🧒🏾
        'en': ':child_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':kind_mitteldunkle_hautfarbe:',
        'es': ':infante_tono_de_piel_oscuro_medio:',
        'fr': ':enfant_peau_mate:',
        'ja': ':子供_やや濃い肌色:',
        'ko': ':어린이_진한_갈색_피부:',
        'pt': ':criança_pele_morena_escura:',
        'it': ':bimbo_carnagione_abbastanza_scura:',
        'fa': ':بچه_پوست_گندمی:',
        'id': ':anak_warna_kulit_gelap-sedang:',
        'zh': ':儿童_中等深肤色:',
        'ru': ':ребенок_темный_тон_кожи:'
    },
    '\U0001F9D2\U0001F3FC': {  # 🧒🏼
        'en': ':child_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':kind_mittelhelle_hautfarbe:',
        'es': ':infante_tono_de_piel_claro_medio:',
        'fr': ':enfant_peau_moyennement_claire:',
        'ja': ':子供_やや薄い肌色:',
        'ko': ':어린이_연한_갈색_피부:',
        'pt': ':criança_pele_morena_clara:',
        'it': ':bimbo_carnagione_abbastanza_chiara:',
        'fa': ':بچه_پوست_روشن:',
        'id': ':anak_warna_kulit_cerah-sedang:',
        'zh': ':儿童_中等浅肤色:',
        'ru': ':ребенок_светлый_тон_кожи:'
    },
    '\U0001F9D2\U0001F3FD': {  # 🧒🏽
        'en': ':child_medium_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':kind_mittlere_hautfarbe:',
        'es': ':infante_tono_de_piel_medio:',
        'fr': ':enfant_peau_légèrement_mate:',
        'ja': ':子供_中間の肌色:',
        'ko': ':어린이_갈색_피부:',
        'pt': ':criança_pele_morena:',
        'it': ':bimbo_carnagione_olivastra:',
        'fa': ':بچه_پوست_طلایی:',
        'id': ':anak_warna_kulit_sedang:',
        'zh': ':儿童_中等肤色:',
        'ru': ':ребенок_средний_тон_кожи:'
    },
    '\U0001F6B8': {  # 🚸
        'en': ':children_crossing:',
        'status': fully_qualified,
        'E': 1,
        'de': ':kinder_überqueren_die_straße:',
        'es': ':niños_cruzando:',
        'fr': ':traversée_d’enfants:',
        'ja': ':児童横断:',
        'ko': ':어린이_보호_구역:',
        'pt': ':crianças_atravessando:',
        'it': ':attraversamento_bambini:',
        'fa': ':عبور_کودکان:',
        'id': ':anak-anak_menyeberang:',
        'zh': ':儿童过街:',
        'ru': ':знак_«дети»:'
    },
    '\U0001F43F\U0000FE0F': {  # 🐿️
        'en': ':chipmunk:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':streifenhörnchen:',
        'es': ':ardilla:',
        'fr': ':écureuil:',
        'ja': ':リス:',
        'ko': ':얼룩다람쥐:',
        'pt': ':esquilo:',
        'it': ':scoiattolo:',
        'fa': ':سنجاب_راه_راه:',
        'id': ':tupai:',
        'zh': ':松鼠:',
        'ru': ':бурундук:'
    },
    '\U0001F43F': {  # 🐿
        'en': ':chipmunk:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':streifenhörnchen:',
        'es': ':ardilla:',
        'fr': ':écureuil:',
        'ja': ':リス:',
        'ko': ':얼룩다람쥐:',
        'pt': ':esquilo:',
        'it': ':scoiattolo:',
        'fa': ':سنجاب_راه_راه:',
        'id': ':tupai:',
        'zh': ':松鼠:',
        'ru': ':бурундук:'
    },
    '\U0001F36B': {  # 🍫
        'en': ':chocolate_bar:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':schokoladentafel:',
        'es': ':tableta_de_chocolate:',
        'fr': ':barre_chocolatée:',
        'ja': ':チョコレート:',
        'ko': ':초콜렛:',
        'pt': ':chocolate:',
        'it': ':cioccolato:',
        'fa': ':تخته_شکلات:',
        'id': ':sebatang_cokelat:',
        'zh': ':巧克力:',
        'ru': ':шоколад:'
    },
    '\U0001F962': {  # 🥢
        'en': ':chopsticks:',
        'status': fully_qualified,
        'E': 5,
        'de': ':essstäbchen:',
        'es': ':palillos:',
        'fr': ':baguettes:',
        'ja': ':はし:',
        'ko': ':젓가락:',
        'pt': ':hashi:',
        'it': ':bacchette:',
        'fa': ':چاپ_استیک:',
        'id': ':sumpit:',
        'zh': ':筷子:',
        'ru': ':палочки_для_еды:'
    },
    '\U000026EA': {  # ⛪
        'en': ':church:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':kirche:',
        'es': ':iglesia:',
        'fr': ':église:',
        'ja': ':教会:',
        'ko': ':교회:',
        'pt': ':igreja:',
        'it': ':chiesa:',
        'fa': ':کلیسا:',
        'id': ':gereja:',
        'zh': ':教堂:',
        'ru': ':церковь:'
    },
    '\U0001F6AC': {  # 🚬
        'en': ':cigarette:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':smoking:'],
        'de': ':zigarette:',
        'es': ':cigarrillo:',
        'fr': ':cigarette:',
        'ja': ':煙草:',
        'ko': ':흡연_구역_신호:',
        'pt': ':cigarro:',
        'it': ':sigaretta:',
        'fa': ':سیگار:',
        'id': ':rokok:',
        'zh': ':香烟:',
        'ru': ':сигарета:'
    },
    '\U0001F3A6': {  # 🎦
        'en': ':cinema:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kinosymbol:',
        'es': ':cine:',
        'fr': ':cinéma:',
        'ja': ':映画:',
        'ko': ':영화:',
        'pt': ':cinema:',
        'it': ':simbolo_del_cinema:',
        'fa': ':سینما:',
        'id': ':film:',
        'zh': ':电影院:',
        'ru': ':видеозапись:'
    },
    '\U000024C2\U0000FE0F': {  # Ⓜ️
        'en': ':circled_M:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':m:', ':circled_m:'],
        'variant': True,
        'de': ':buchstabe_m_in_kreis:',
        'es': ':m_en_círculo:',
        'fr': ':m_encerclé:',
        'ja': ':丸いmマーク:',
        'ko': ':원글자_엠:',
        'pt': ':círculo_com_a_letra_m:',
        'it': ':pulsante_m_cerchiata:',
        'fa': ':حرف_ام_در_دایره:',
        'id': ':huruf_m_dalam_lingkaran:',
        'zh': ':圆圈包围的M:',
        'ru': ':буква_«м»_в_кружке:'
    },
    '\U000024C2': {  # Ⓜ
        'en': ':circled_M:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':m:', ':circled_m:'],
        'variant': True,
        'de': ':buchstabe_m_in_kreis:',
        'es': ':m_en_círculo:',
        'fr': ':m_encerclé:',
        'ja': ':丸いmマーク:',
        'ko': ':원글자_엠:',
        'pt': ':círculo_com_a_letra_m:',
        'it': ':pulsante_m_cerchiata:',
        'fa': ':حرف_ام_در_دایره:',
        'id': ':huruf_m_dalam_lingkaran:',
        'zh': ':圆圈包围的M:',
        'ru': ':буква_«м»_в_кружке:'
    },
    '\U0001F3AA': {  # 🎪
        'en': ':circus_tent:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':zirkuszelt:',
        'es': ':carpa_de_circo:',
        'fr': ':chapiteau:',
        'ja': ':サーカス:',
        'ko': ':서커스:',
        'pt': ':circo:',
        'it': ':circo:',
        'fa': ':چادر_سیرک:',
        'id': ':tenda_sirkus:',
        'zh': ':马戏团帐篷:',
        'ru': ':цирковой_шатер:'
    },
    '\U0001F3D9\U0000FE0F': {  # 🏙️
        'en': ':cityscape:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':skyline:',
        'es': ':paisaje_urbano:',
        'fr': ':ville:',
        'ja': ':高層ビル:',
        'ko': ':도시_전경:',
        'pt': ':cidade:',
        'it': ':paesaggio_urbano:',
        'fa': ':نمای_شهر:',
        'id': ':panorama_kota:',
        'zh': ':城市风光:',
        'ru': ':ночнои_город:'
    },
    '\U0001F3D9': {  # 🏙
        'en': ':cityscape:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':skyline:',
        'es': ':paisaje_urbano:',
        'fr': ':ville:',
        'ja': ':高層ビル:',
        'ko': ':도시_전경:',
        'pt': ':cidade:',
        'it': ':paesaggio_urbano:',
        'fa': ':نمای_شهر:',
        'id': ':panorama_kota:',
        'zh': ':城市风光:',
        'ru': ':ночной_город:'
    },
    '\U0001F306': {  # 🌆
        'en': ':cityscape_at_dusk:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':city_sunset:'],
        'de': ':abendstimmung_in_der_stadt:',
        'es': ':ciudad_al_atardecer:',
        'fr': ':ville_au_crépuscule:',
        'ja': ':夕暮れの都会:',
        'ko': ':도시_야경:',
        'pt': ':cidade_ao_anoitecer:',
        'it': ':città_al_tramonto:',
        'fa': ':نمای_شهر_هنگام_غروب:',
        'id': ':panorama_kota_di_sore_hari:',
        'zh': ':城市黄昏:',
        'ru': ':закат_в_городе:'
    },
    '\U0001F5DC\U0000FE0F': {  # 🗜️
        'en': ':clamp:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':compression:'],
        'variant': True,
        'de': ':schraubzwinge:',
        'es': ':tornillo_de_banco:',
        'fr': ':serre-joint:',
        'ja': ':万力:',
        'ko': ':압축기:',
        'pt': ':braçadeira:',
        'it': ':morsetto:',
        'fa': ':پرس:',
        'id': ':klem:',
        'zh': ':夹钳:',
        'ru': ':компрессор:'
    },
    '\U0001F5DC': {  # 🗜
        'en': ':clamp:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':compression:'],
        'variant': True,
        'de': ':schraubzwinge:',
        'es': ':tornillo_de_banco:',
        'fr': ':serre-joint:',
        'ja': ':万力:',
        'ko': ':압축기:',
        'pt': ':braçadeira:',
        'it': ':morsetto:',
        'fa': ':پرس:',
        'id': ':klem:',
        'zh': ':夹钳:',
        'ru': ':компрессор:'
    },
    '\U0001F3AC': {  # 🎬
        'en': ':clapper_board:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':clapper:'],
        'variant': True,
        'de': ':filmklappe:',
        'es': ':claqueta:',
        'fr': ':clap:',
        'ja': ':カチンコ:',
        'ko': ':슬레이트:',
        'pt': ':claquete:',
        'it': ':ciak:',
        'fa': ':کلاکت:',
        'id': ':papan_sutradara:',
        'zh': ':场记板:',
        'ru': ':хлопушка_нумератор:'
    },
    '\U0001F44F': {  # 👏
        'en': ':clapping_hands:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':clap:'],
        'de': ':klatschende_hände:',
        'es': ':manos_aplaudiendo:',
        'fr': ':applaudissements:',
        'ja': ':拍手:',
        'ko': ':손뼉:',
        'pt': ':mãos_aplaudindo:',
        'it': ':mani_che_applaudono:',
        'fa': ':دست_زدن:',
        'id': ':tepuk_tangan:',
        'zh': ':鼓掌:',
        'ru': ':аплодисменты:'
    },
    '\U0001F44F\U0001F3FF': {  # 👏🏿
        'en': ':clapping_hands_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':klatschende_hände_dunkle_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_oscuro:',
        'fr': ':applaudissements_peau_foncée:',
        'ja': ':拍手_濃い肌色:',
        'ko': ':손뼉_검은색_피부:',
        'pt': ':mãos_aplaudindo_pele_escura:',
        'it': ':mani_che_applaudono_carnagione_scura:',
        'fa': ':دست_زدن_پوست_آبنوسی:',
        'id': ':tepuk_tangan_warna_kulit_gelap:',
        'zh': ':鼓掌_较深肤色:',
        'ru': ':аплодисменты_очень_темный_тон_кожи:'
    },
    '\U0001F44F\U0001F3FB': {  # 👏🏻
        'en': ':clapping_hands_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':klatschende_hände_helle_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_claro:',
        'fr': ':applaudissements_peau_claire:',
        'ja': ':拍手_薄い肌色:',
        'ko': ':손뼉_하얀_피부:',
        'pt': ':mãos_aplaudindo_pele_clara:',
        'it': ':mani_che_applaudono_carnagione_chiara:',
        'fa': ':دست_زدن_پوست_سفید:',
        'id': ':tepuk_tangan_warna_kulit_cerah:',
        'zh': ':鼓掌_较浅肤色:',
        'ru': ':аплодисменты_очень_светлый_тон_кожи:'
    },
    '\U0001F44F\U0001F3FE': {  # 👏🏾
        'en': ':clapping_hands_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':klatschende_hände_mitteldunkle_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_oscuro_medio:',
        'fr': ':applaudissements_peau_mate:',
        'ja': ':拍手_やや濃い肌色:',
        'ko': ':손뼉_진한_갈색_피부:',
        'pt': ':mãos_aplaudindo_pele_morena_escura:',
        'it': ':mani_che_applaudono_carnagione_abbastanza_scura:',
        'fa': ':دست_زدن_پوست_گندمی:',
        'id': ':tepuk_tangan_warna_kulit_gelap-sedang:',
        'zh': ':鼓掌_中等深肤色:',
        'ru': ':аплодисменты_темный_тон_кожи:'
    },
    '\U0001F44F\U0001F3FC': {  # 👏🏼
        'en': ':clapping_hands_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':klatschende_hände_mittelhelle_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_claro_medio:',
        'fr': ':applaudissements_peau_moyennement_claire:',
        'ja': ':拍手_やや薄い肌色:',
        'ko': ':손뼉_연한_갈색_피부:',
        'pt': ':mãos_aplaudindo_pele_morena_clara:',
        'it': ':mani_che_applaudono_carnagione_abbastanza_chiara:',
        'fa': ':دست_زدن_پوست_روشن:',
        'id': ':tepuk_tangan_warna_kulit_cerah-sedang:',
        'zh': ':鼓掌_中等浅肤色:',
        'ru': ':аплодисменты_светлый_тон_кожи:'
    },
    '\U0001F44F\U0001F3FD': {  # 👏🏽
        'en': ':clapping_hands_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':klatschende_hände_mittlere_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_medio:',
        'fr': ':applaudissements_peau_légèrement_mate:',
        'ja': ':拍手_中間の肌色:',
        'ko': ':손뼉_갈색_피부:',
        'pt': ':mãos_aplaudindo_pele_morena:',
        'it': ':mani_che_applaudono_carnagione_olivastra:',
        'fa': ':دست_زدن_پوست_طلایی:',
        'id': ':tepuk_tangan_warna_kulit_sedang:',
        'zh': ':鼓掌_中等肤色:',
        'ru': ':аплодисменты_средний_тон_кожи:'
    },
    '\U0001F3DB\U0000FE0F': {  # 🏛️
        'en': ':classical_building:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':antikes_gebäude:',
        'es': ':edificio_clásico:',
        'fr': ':monument_classique:',
        'ja': ':歴史的な建物:',
        'ko': ':고전_양식의_건축물:',
        'pt': ':prédio_grego:',
        'it': ':edificio_classico:',
        'fa': ':ساختمان_باستانی:',
        'id': ':bangunan_klasik:',
        'zh': ':古典建筑:',
        'ru': ':античное_здание:'
    },
    '\U0001F3DB': {  # 🏛
        'en': ':classical_building:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':antikes_gebäude:',
        'es': ':edificio_clásico:',
        'fr': ':monument_classique:',
        'ja': ':歴史的な建物:',
        'ko': ':고전_양식의_건축물:',
        'pt': ':prédio_grego:',
        'it': ':edificio_classico:',
        'fa': ':ساختمان_باستانی:',
        'id': ':bangunan_klasik:',
        'zh': ':古典建筑:',
        'ru': ':античное_здание:'
    },
    '\U0001F37B': {  # 🍻
        'en': ':clinking_beer_mugs:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':beers:'],
        'de': ':bierkrüge:',
        'es': ':jarras_de_cerveza_brindando:',
        'fr': ':chopes_de_bière:',
        'ja': ':ビールで乾杯:',
        'ko': ':맥주:',
        'pt': ':canecas_de_cerveja:',
        'it': ':boccali_di_birra:',
        'fa': ':لیوان_های_آبجو_را_به_هم_زدن:',
        'id': ':mug_bir_berdenting:',
        'zh': ':干杯:',
        'ru': ':чокающиеся_кружки:'
    },
    '\U0001F942': {  # 🥂
        'en': ':clinking_glasses:',
        'status': fully_qualified,
        'E': 3,
        'de': ':sektgläser:',
        'es': ':copas_brindando:',
        'fr': ':trinquer:',
        'ja': ':グラスで乾杯:',
        'ko': ':건배하는_샴페인_잔:',
        'pt': ':taças_brindando:',
        'it': ':brindisi:',
        'fa': ':صدای_به_هم_خوردن_پیاله_ها:',
        'id': ':mendentingkan_gelas:',
        'zh': ':碰杯:',
        'ru': ':чокающиеся_бокалы:'
    },
    '\U0001F4CB': {  # 📋
        'en': ':clipboard:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':klemmbrett:',
        'es': ':portapapeles:',
        'fr': ':porte-bloc:',
        'ja': ':クリップボード:',
        'ko': ':클립보드:',
        'pt': ':prancheta:',
        'it': ':portablocco:',
        'fa': ':تختهٔ_گیره_دار:',
        'id': ':papan_klip:',
        'zh': ':剪贴板:',
        'ru': ':планшет_с_зажимом:'
    },
    '\U0001F503': {  # 🔃
        'en': ':clockwise_vertical_arrows:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':arrows_clockwise:'],
        'de': ':kreisförmige_pfeile_im_uhrzeigersinn:',
        'es': ':flechas_verticales_en_sentido_horario:',
        'fr': ':flèches_dans_le_sens_horaire:',
        'ja': ':右回り縦矢印:',
        'ko': ':시계_방향_화살표:',
        'pt': ':setas_verticais_no_sentido_horário:',
        'it': ':frecce_verticali_che_ruotano_in_senso_orario:',
        'fa': ':جهت_عقربه_های_ساعت:',
        'id': ':tanda_panah_searah_jarum_jam:',
        'zh': ':顺时针垂直箭头:',
        'ru': ':по_часовой_стрелке:'
    },
    '\U0001F4D5': {  # 📕
        'en': ':closed_book:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':geschlossenes_buch:',
        'es': ':libro_cerrado:',
        'fr': ':livre_fermé:',
        'ja': ':閉じた本:',
        'ko': ':펼치지_않은_책:',
        'pt': ':livro_fechado:',
        'it': ':libro_chiuso:',
        'fa': ':کتاب_بسته:',
        'id': ':buku_tertutup:',
        'zh': ':合上的书本:',
        'ru': ':закрытая_книга:'
    },
    '\U0001F4EA': {  # 📪
        'en': ':closed_mailbox_with_lowered_flag:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':mailbox_closed:'],
        'variant': True,
        'de': ':geschlossener_briefkasten_ohne_post:',
        'es': ':buzón_cerrado_con_la_bandera_bajada:',
        'fr': ':boîte_aux_lettres_fermée_drapeau_baissé:',
        'ja': ':閉じた郵便受け(手紙なし):',
        'ko': ':표지가_내려간_닫힌_우편함:',
        'pt': ':caixa_de_correio_fechada_com_bandeira_abaixada:',
        'it': ':cassetta_postale_chiusa_bandierina_abbassata:',
        'fa': ':صندوق_پستی_بسته_با_پرچم_پایین:',
        'id': ':kotak_surat_tertutup_dengan_bendera_turun:',
        'zh': ':无待收信件:',
        'ru': ':закрытый_почтовый_ящик_с_опущенным_флажком:'
    },
    '\U0001F4EB': {  # 📫
        'en': ':closed_mailbox_with_raised_flag:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':mailbox:'],
        'variant': True,
        'de': ':geschlossener_briefkasten_mit_post:',
        'es': ':buzón_cerrado_con_la_bandera_levantada:',
        'fr': ':boîte_aux_lettres_fermée_drapeau_levé:',
        'ja': ':閉じた郵便受け(手紙あり):',
        'ko': ':표지가_올라간_닫힌_우편함:',
        'pt': ':caixa_de_correio_fechada_com_bandeira_levantada:',
        'it': ':cassetta_postale_chiusa_bandierina_alzata:',
        'fa': ':صندوق_پستی_بسته_با_پرچم_بالا:',
        'id': ':kotak_surat_tertutup_dengan_bendera_terangkat:',
        'zh': ':有待收信件:',
        'ru': ':закрытый_почтовый_ящик_с_поднятым_флажком:'
    },
    '\U0001F302': {  # 🌂
        'en': ':closed_umbrella:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':geschlossener_regenschirm:',
        'es': ':paraguas_cerrado:',
        'fr': ':parapluie_fermé:',
        'ja': ':閉じた傘:',
        'ko': ':접힌_우산:',
        'pt': ':guarda-chuva_fechado:',
        'it': ':ombrello_chiuso:',
        'fa': ':چتر_بسته:',
        'id': ':payung_tertutup:',
        'zh': ':收起的伞:',
        'ru': ':закрытый_зонт:'
    },
    '\U00002601\U0000FE0F': {  # ☁️
        'en': ':cloud:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':wolke:',
        'es': ':nube:',
        'fr': ':nuage:',
        'ja': ':雲:',
        'ko': ':구름:',
        'pt': ':nuvem:',
        'it': ':nuvola:',
        'fa': ':ابر:',
        'id': ':awan:',
        'zh': ':云:',
        'ru': ':облако:'
    },
    '\U00002601': {  # ☁
        'en': ':cloud:',
        'status': unqualified,
        'E': 0.6,
        'variant': True,
        'de': ':wolke:',
        'es': ':nube:',
        'fr': ':nuage:',
        'ja': ':雲:',
        'ko': ':구름:',
        'pt': ':nuvem:',
        'it': ':nuvola:',
        'fa': ':ابر:',
        'id': ':awan:',
        'zh': ':云:',
        'ru': ':облако:'
    },
    '\U0001F329\U0000FE0F': {  # 🌩️
        'en': ':cloud_with_lightning:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':wolke_mit_blitz:',
        'es': ':nube_con_rayo:',
        'fr': ':nuage_avec_éclair:',
        'ja': ':雷雲:',
        'ko': ':번개구름:',
        'pt': ':nuvem_com_trovão:',
        'it': ':fulmini:',
        'fa': ':ابر_با_برق:',
        'id': ':awan_petir:',
        'zh': ':打雷:',
        'ru': ':молния:'
    },
    '\U0001F329': {  # 🌩
        'en': ':cloud_with_lightning:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':wolke_mit_blitz:',
        'es': ':nube_con_rayo:',
        'fr': ':nuage_avec_éclair:',
        'ja': ':雷雲:',
        'ko': ':번개구름:',
        'pt': ':nuvem_com_trovão:',
        'it': ':fulmini:',
        'fa': ':ابر_با_برق:',
        'id': ':awan_petir:',
        'zh': ':打雷:',
        'ru': ':молния:'
    },
    '\U000026C8\U0000FE0F': {  # ⛈️
        'en': ':cloud_with_lightning_and_rain:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':thunder_cloud_and_rain:'],
        'variant': True,
        'de': ':wolke_mit_blitz_und_regen:',
        'es': ':nube_con_rayo_y_lluvia:',
        'fr': ':nuage_avec_éclair_et_pluie:',
        'ja': ':雷雨:',
        'ko': ':번개가_치는_비구름:',
        'pt': ':chuva_com_trovão:',
        'it': ':temporale:',
        'fa': ':ابر_با_رعدوبرق_و_باران:',
        'id': ':awan_dengan_petir_dan_hujan:',
        'zh': ':雷阵雨:',
        'ru': ':гроза:'
    },
    '\U000026C8': {  # ⛈
        'en': ':cloud_with_lightning_and_rain:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':thunder_cloud_and_rain:'],
        'variant': True,
        'de': ':wolke_mit_blitz_und_regen:',
        'es': ':nube_con_rayo_y_lluvia:',
        'fr': ':nuage_avec_éclair_et_pluie:',
        'ja': ':雷雨:',
        'ko': ':번개가_치는_비구름:',
        'pt': ':chuva_com_trovão:',
        'it': ':temporale:',
        'fa': ':ابر_با_رعدوبرق_و_باران:',
        'id': ':awan_dengan_petir_dan_hujan:',
        'zh': ':雷阵雨:',
        'ru': ':гроза:'
    },
    '\U0001F327\U0000FE0F': {  # 🌧️
        'en': ':cloud_with_rain:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':wolke_mit_regen:',
        'es': ':nube_con_lluvia:',
        'fr': ':nuage_avec_pluie:',
        'ja': ':雨雲:',
        'ko': ':비구름:',
        'pt': ':nuvem_com_chuva:',
        'it': ':pioggia:',
        'fa': ':ابر_بارانی:',
        'id': ':awan_hujan:',
        'zh': ':下雨:',
        'ru': ':дождь:'
    },
    '\U0001F327': {  # 🌧
        'en': ':cloud_with_rain:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':wolke_mit_regen:',
        'es': ':nube_con_lluvia:',
        'fr': ':nuage_avec_pluie:',
        'ja': ':雨雲:',
        'ko': ':비구름:',
        'pt': ':nuvem_com_chuva:',
        'it': ':pioggia:',
        'fa': ':ابر_بارانی:',
        'id': ':awan_hujan:',
        'zh': ':下雨:',
        'ru': ':дождь:'
    },
    '\U0001F328\U0000FE0F': {  # 🌨️
        'en': ':cloud_with_snow:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':wolke_mit_schnee:',
        'es': ':nube_con_nieve:',
        'fr': ':nuage_avec_neige:',
        'ja': ':雪雲:',
        'ko': ':눈구름:',
        'pt': ':nuvem_com_neve:',
        'it': ':neve:',
        'fa': ':ابر_برفی:',
        'id': ':awan_salju:',
        'zh': ':下雪:',
        'ru': ':снег:'
    },
    '\U0001F328': {  # 🌨
        'en': ':cloud_with_snow:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':wolke_mit_schnee:',
        'es': ':nube_con_nieve:',
        'fr': ':nuage_avec_neige:',
        'ja': ':雪雲:',
        'ko': ':눈구름:',
        'pt': ':nuvem_com_neve:',
        'it': ':neve:',
        'fa': ':ابر_برفی:',
        'id': ':awan_salju:',
        'zh': ':下雪:',
        'ru': ':снег:'
    },
    '\U0001F921': {  # 🤡
        'en': ':clown_face:',
        'status': fully_qualified,
        'E': 3,
        'de': ':clown-gesicht:',
        'es': ':cara_de_payaso:',
        'fr': ':visage_de_clown:',
        'ja': ':ピエロの顔:',
        'ko': ':어릿광대_얼굴:',
        'pt': ':rosto_de_palhaço:',
        'it': ':faccina_pagliaccio:',
        'fa': ':دلقک:',
        'id': ':badut:',
        'zh': ':小丑脸:',
        'ru': ':клоун:'
    },
    '\U00002663\U0000FE0F': {  # ♣️
        'en': ':club_suit:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':clubs:'],
        'variant': True,
        'de': ':kreuz:',
        'es': ':palo_de_tréboles:',
        'fr': ':trèfle_cartes:',
        'ja': ':クラブ:',
        'ko': ':클럽:',
        'pt': ':naipe_de_paus:',
        'it': ':fiori:',
        'fa': ':خال_گشنیز:',
        'id': ':keriting:',
        'zh': ':梅花:',
        'ru': ':трефы:'
    },
    '\U00002663': {  # ♣
        'en': ':club_suit:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':clubs:'],
        'variant': True,
        'de': ':kreuz:',
        'es': ':palo_de_tréboles:',
        'fr': ':trèfle_cartes:',
        'ja': ':クラブ:',
        'ko': ':클럽:',
        'pt': ':naipe_de_paus:',
        'it': ':fiori:',
        'fa': ':خال_گشنیز:',
        'id': ':keriting:',
        'zh': ':梅花:',
        'ru': ':трефы:'
    },
    '\U0001F45D': {  # 👝
        'en': ':clutch_bag:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':pouch:'],
        'de': ':clutch:',
        'es': ':bolso_de_mano:',
        'fr': ':pochette:',
        'ja': ':ポーチ:',
        'ko': ':파우치:',
        'pt': ':bolsa_pequena:',
        'it': ':pochette:',
        'fa': ':کیف_بدون_تسمه_زنانه:',
        'id': ':kantung:',
        'zh': ':手袋:',
        'ru': ':клатч:'
    },
    '\U0001F9E5': {  # 🧥
        'en': ':coat:',
        'status': fully_qualified,
        'E': 5,
        'de': ':mantel:',
        'es': ':abrigo:',
        'fr': ':manteau:',
        'ja': ':コート:',
        'ko': ':코트:',
        'pt': ':casaco:',
        'it': ':cappotto:',
        'fa': ':پالتو:',
        'id': ':mantel:',
        'zh': ':外套:',
        'ru': ':пальто:'
    },
    '\U0001FAB3': {  # 🪳
        'en': ':cockroach:',
        'status': fully_qualified,
        'E': 13,
        'de': ':kakerlake:',
        'es': ':cucaracha:',
        'fr': ':cafard:',
        'ja': ':ゴキブリ:',
        'ko': ':바퀴벌레:',
        'pt': ':barata:',
        'it': ':scarafaggio:',
        'fa': ':سوسک_حمام:',
        'id': ':kecoak:',
        'zh': ':蟑螂:',
        'ru': ':таракан:'
    },
    '\U0001F378': {  # 🍸
        'en': ':cocktail_glass:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cocktail:'],
        'variant': True,
        'de': ':cocktailglas:',
        'es': ':copa_de_cóctel:',
        'fr': ':cocktail:',
        'ja': ':カクテルグラス:',
        'ko': ':칵테일:',
        'pt': ':coquetel:',
        'it': ':cocktail:',
        'fa': ':جام_کوکتل:',
        'id': ':gelas_cocktail:',
        'zh': ':鸡尾酒:',
        'ru': ':коктейль:'
    },
    '\U0001F965': {  # 🥥
        'en': ':coconut:',
        'status': fully_qualified,
        'E': 5,
        'de': ':kokosnuss:',
        'es': ':coco:',
        'fr': ':noix_de_coco:',
        'ja': ':ココナツ:',
        'ko': ':코코넛:',
        'pt': ':coco:',
        'it': ':cocco:',
        'fa': ':نارگیل:',
        'id': ':kelapa:',
        'zh': ':椰子:',
        'ru': ':кокос:'
    },
    '\U000026B0\U0000FE0F': {  # ⚰️
        'en': ':coffin:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':sarg:',
        'es': ':ataúd:',
        'fr': ':cercueil:',
        'ja': ':棺桶:',
        'ko': ':관:',
        'pt': ':caixão:',
        'it': ':bara:',
        'fa': ':تابوت:',
        'id': ':peti_mati:',
        'zh': ':棺材:',
        'ru': ':гроб:'
    },
    '\U000026B0': {  # ⚰
        'en': ':coffin:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':sarg:',
        'es': ':ataúd:',
        'fr': ':cercueil:',
        'ja': ':棺桶:',
        'ko': ':관:',
        'pt': ':caixão:',
        'it': ':bara:',
        'fa': ':تابوت:',
        'id': ':peti_mati:',
        'zh': ':棺材:',
        'ru': ':гроб:'
    },
    '\U0001FA99': {  # 🪙
        'en': ':coin:',
        'status': fully_qualified,
        'E': 13,
        'de': ':münze:',
        'es': ':moneda:',
        'fr': ':pièce:',
        'ja': ':コイン:',
        'ko': ':동전:',
        'pt': ':moeda:',
        'it': ':moneta:',
        'fa': ':سکه:',
        'id': ':koin:',
        'zh': ':硬币:',
        'ru': ':монета:'
    },
    '\U0001F976': {  # 🥶
        'en': ':cold_face:',
        'status': fully_qualified,
        'E': 11,
        'de': ':frierendes_gesicht:',
        'es': ':cara_con_frío:',
        'fr': ':visage_bleu_et_froid:',
        'ja': ':寒い顔:',
        'ko': ':추워하는_얼굴:',
        'pt': ':rosto_gelado:',
        'it': ':faccina_congelata:',
        'fa': ':خیلی_سرد:',
        'id': ':wajah_kedinginan:',
        'zh': ':冷脸:',
        'ru': ':мерзнет:'
    },
    '\U0001F4A5': {  # 💥
        'en': ':collision:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':boom:'],
        'de': ':zusammenstoß:',
        'es': ':colisión:',
        'fr': ':explosion:',
        'ja': ':衝突:',
        'ko': ':충돌:',
        'pt': ':colisão:',
        'it': ':collisione:',
        'fa': ':برخورد:',
        'id': ':tabrakan:',
        'zh': ':爆炸:',
        'ru': ':взрыв:'
    },
    '\U00002604\U0000FE0F': {  # ☄️
        'en': ':comet:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':komet:',
        'es': ':meteorito:',
        'fr': ':comète:',
        'ja': ':彗星:',
        'ko': ':혜성:',
        'pt': ':cometa:',
        'it': ':cometa:',
        'fa': ':ستاره_دنباله_دار:',
        'id': ':komet:',
        'zh': ':彗星:',
        'ru': ':комета:'
    },
    '\U00002604': {  # ☄
        'en': ':comet:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':komet:',
        'es': ':meteorito:',
        'fr': ':comète:',
        'ja': ':彗星:',
        'ko': ':혜성:',
        'pt': ':cometa:',
        'it': ':cometa:',
        'fa': ':ستاره_دنباله_دار:',
        'id': ':komet:',
        'zh': ':彗星:',
        'ru': ':комета:'
    },
    '\U0001F9ED': {  # 🧭
        'en': ':compass:',
        'status': fully_qualified,
        'E': 11,
        'de': ':kompass:',
        'es': ':brújula:',
        'fr': ':boussole:',
        'ja': ':コンパス:',
        'ko': ':나침반:',
        'pt': ':bússola:',
        'it': ':bussola:',
        'fa': ':قطب_نما:',
        'id': ':kompas:',
        'zh': ':指南针:',
        'ru': ':компас:'
    },
    '\U0001F4BD': {  # 💽
        'en': ':computer_disk:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':minidisc:'],
        'de': ':minidisc:',
        'es': ':minidisc:',
        'fr': ':disque_d’ordinateur:',
        'ja': ':md:',
        'ko': ':엠디:',
        'pt': ':minidisc:',
        'it': ':minidisc:',
        'fa': ':مینی_دیسک:',
        'id': ':disk_komputer:',
        'zh': ':电脑光盘:',
        'ru': ':диск:'
    },
    '\U0001F5B1\U0000FE0F': {  # 🖱️
        'en': ':computer_mouse:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':three_button_mouse:'],
        'variant': True,
        'de': ':computermaus:',
        'es': ':ratón_de_ordenador:',
        'fr': ':souris_d’ordinateur:',
        'ja': ':マウス:',
        'ko': ':컴퓨터_마우스:',
        'pt': ':mouse:',
        'it': ':mouse:',
        'fa': ':موشواره:',
        'id': ':mouse_komputer:',
        'zh': ':电脑鼠标:',
        'ru': ':мышка:'
    },
    '\U0001F5B1': {  # 🖱
        'en': ':computer_mouse:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':three_button_mouse:'],
        'variant': True,
        'de': ':computermaus:',
        'es': ':ratón_de_ordenador:',
        'fr': ':souris_d’ordinateur:',
        'ja': ':マウス:',
        'ko': ':컴퓨터_마우스:',
        'pt': ':mouse:',
        'it': ':mouse:',
        'fa': ':موشواره:',
        'id': ':mouse_komputer:',
        'zh': ':电脑鼠标:',
        'ru': ':мышка:'
    },
    '\U0001F38A': {  # 🎊
        'en': ':confetti_ball:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':konfettiball:',
        'es': ':bola_de_confeti:',
        'fr': ':confettis:',
        'ja': ':くす玉:',
        'ko': ':박_터트리기:',
        'pt': ':confete:',
        'it': ':coriandoli:',
        'fa': ':توپ_پولکی:',
        'id': ':bola_konfeti:',
        'zh': ':五彩纸屑球:',
        'ru': ':конфетти:'
    },
    '\U0001F616': {  # 😖
        'en': ':confounded_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':confounded:'],
        'de': ':verwirrtes_gesicht:',
        'es': ':cara_de_frustración:',
        'fr': ':visage_déconcerté:',
        'ja': ':困惑した顔:',
        'ko': ':당혹한_얼굴:',
        'pt': ':rosto_perplexo:',
        'it': ':faccina_frustrata:',
        'fa': ':سردرگم:',
        'id': ':wajah_stres:',
        'zh': ':困惑:',
        'ru': ':в_растерянности:'
    },
    '\U0001F615': {  # 😕
        'en': ':confused_face:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':confused:'],
        'de': ':verwundertes_gesicht:',
        'es': ':cara_de_confusión:',
        'fr': ':visage_confus:',
        'ja': ':混乱:',
        'ko': ':혼란스러워하는_얼굴:',
        'pt': ':rosto_confuso:',
        'it': ':faccina_confusa:',
        'fa': ':گیج_شده:',
        'id': ':wajah_bingung:',
        'zh': ':困扰:',
        'ru': ':в_замешательстве:'
    },
    '\U0001F6A7': {  # 🚧
        'en': ':construction:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':baustellenabsperrung:',
        'es': ':obras:',
        'fr': ':travaux:',
        'ja': ':工事中:',
        'ko': ':공사_중:',
        'pt': ':em_construção:',
        'it': ':area_cantiere:',
        'fa': ':ساخت_وساز:',
        'id': ':konstruksi:',
        'zh': ':路障:',
        'ru': ':строительные_работы:'
    },
    '\U0001F477': {  # 👷
        'en': ':construction_worker:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':bauarbeiter(in):',
        'es': ':profesional_de_la_construcción:',
        'fr': ':personnel_du_bâtiment:',
        'ja': ':建設作業員:',
        'ko': ':건설_노동자:',
        'pt': ':trabalhador_de_construção_civil:',
        'it': ':operaio_edile:',
        'fa': ':کارگر:',
        'id': ':pekerja_konstruksi:',
        'zh': ':建筑工人:',
        'ru': ':строитель:'
    },
    '\U0001F477\U0001F3FF': {  # 👷🏿
        'en': ':construction_worker_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':bauarbeiter(in)_dunkle_hautfarbe:',
        'es': ':profesional_de_la_construcción_tono_de_piel_oscuro:',
        'fr': ':personnel_du_bâtiment_peau_foncée:',
        'ja': ':建設作業員_濃い肌色:',
        'ko': ':건설_노동자_검은색_피부:',
        'pt': ':trabalhador_de_construção_civil_pele_escura:',
        'it': ':operaio_edile_carnagione_scura:',
        'fa': ':کارگر_پوست_آبنوسی:',
        'id': ':pekerja_konstruksi_warna_kulit_gelap:',
        'zh': ':建筑工人_较深肤色:',
        'ru': ':строитель_очень_темный_тон_кожи:'
    },
    '\U0001F477\U0001F3FB': {  # 👷🏻
        'en': ':construction_worker_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':bauarbeiter(in)_helle_hautfarbe:',
        'es': ':profesional_de_la_construcción_tono_de_piel_claro:',
        'fr': ':personnel_du_bâtiment_peau_claire:',
        'ja': ':建設作業員_薄い肌色:',
        'ko': ':건설_노동자_하얀_피부:',
        'pt': ':trabalhador_de_construção_civil_pele_clara:',
        'it': ':operaio_edile_carnagione_chiara:',
        'fa': ':کارگر_پوست_سفید:',
        'id': ':pekerja_konstruksi_warna_kulit_cerah:',
        'zh': ':建筑工人_较浅肤色:',
        'ru': ':строитель_очень_светлый_тон_кожи:'
    },
    '\U0001F477\U0001F3FE': {  # 👷🏾
        'en': ':construction_worker_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':bauarbeiter(in)_mitteldunkle_hautfarbe:',
        'es': ':profesional_de_la_construcción_tono_de_piel_oscuro_medio:',
        'fr': ':personnel_du_bâtiment_peau_mate:',
        'ja': ':建設作業員_やや濃い肌色:',
        'ko': ':건설_노동자_진한_갈색_피부:',
        'pt': ':trabalhador_de_construção_civil_pele_morena_escura:',
        'it': ':operaio_edile_carnagione_abbastanza_scura:',
        'fa': ':کارگر_پوست_گندمی:',
        'id': ':pekerja_konstruksi_warna_kulit_gelap-sedang:',
        'zh': ':建筑工人_中等深肤色:',
        'ru': ':строитель_темный_тон_кожи:'
    },
    '\U0001F477\U0001F3FC': {  # 👷🏼
        'en': ':construction_worker_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':bauarbeiter(in)_mittelhelle_hautfarbe:',
        'es': ':profesional_de_la_construcción_tono_de_piel_claro_medio:',
        'fr': ':personnel_du_bâtiment_peau_moyennement_claire:',
        'ja': ':建設作業員_やや薄い肌色:',
        'ko': ':건설_노동자_연한_갈색_피부:',
        'pt': ':trabalhador_de_construção_civil_pele_morena_clara:',
        'it': ':operaio_edile_carnagione_abbastanza_chiara:',
        'fa': ':کارگر_پوست_روشن:',
        'id': ':pekerja_konstruksi_warna_kulit_cerah-sedang:',
        'zh': ':建筑工人_中等浅肤色:',
        'ru': ':строитель_светлый_тон_кожи:'
    },
    '\U0001F477\U0001F3FD': {  # 👷🏽
        'en': ':construction_worker_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':bauarbeiter(in)_mittlere_hautfarbe:',
        'es': ':profesional_de_la_construcción_tono_de_piel_medio:',
        'fr': ':personnel_du_bâtiment_peau_légèrement_mate:',
        'ja': ':建設作業員_中間の肌色:',
        'ko': ':건설_노동자_갈색_피부:',
        'pt': ':trabalhador_de_construção_civil_pele_morena:',
        'it': ':operaio_edile_carnagione_olivastra:',
        'fa': ':کارگر_پوست_طلایی:',
        'id': ':pekerja_konstruksi_warna_kulit_sedang:',
        'zh': ':建筑工人_中等肤色:',
        'ru': ':строитель_средний_тон_кожи:'
    },
    '\U0001F39B\U0000FE0F': {  # 🎛️
        'en': ':control_knobs:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':drehregler:',
        'es': ':ruedas_de_control:',
        'fr': ':boutons_de_réglage:',
        'ja': ':コントロールつまみ:',
        'ko': ':컨트롤_레버:',
        'pt': ':botões_giratórios:',
        'it': ':manopole_di_controllo:',
        'fa': ':دسته_های_کنترل:',
        'id': ':kenop_kontrol:',
        'zh': ':控制旋钮:',
        'ru': ':регуляторы:'
    },
    '\U0001F39B': {  # 🎛
        'en': ':control_knobs:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':drehregler:',
        'es': ':ruedas_de_control:',
        'fr': ':boutons_de_réglage:',
        'ja': ':コントロールつまみ:',
        'ko': ':컨트롤_레버:',
        'pt': ':botões_giratórios:',
        'it': ':manopole_di_controllo:',
        'fa': ':دسته_های_کنترل:',
        'id': ':kenop_kontrol:',
        'zh': ':控制旋钮:',
        'ru': ':регуляторы:'
    },
    '\U0001F3EA': {  # 🏪
        'en': ':convenience_store:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':minimarkt:',
        'es': ':tienda_24_horas:',
        'fr': ':supérette:',
        'ja': ':コンビニ:',
        'ko': ':편의점:',
        'pt': ':loja_de_conveniência:',
        'it': ':minimarket:',
        'fa': ':بقالی:',
        'id': ':minimarket:',
        'zh': ':便利店:',
        'ru': ':круглосуточный_магазин:'
    },
    '\U0001F9D1\U0000200D\U0001F373': {  # 🧑‍🍳
        'en': ':cook:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':koch/köchin:',
        'es': ':chef:',
        'fr': ':cuisinier_(tous_genres):',
        'ja': ':コック:',
        'ko': ':요리사:',
        'pt': ':chef_de_cozinha:',
        'it': ':persona_che_cucina:',
        'fa': ':آشپز:',
        'id': ':koki:',
        'zh': ':厨师:',
        'ru': ':повар:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U0001F373': {  # 🧑🏿‍🍳
        'en': ':cook_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':koch/köchin_dunkle_hautfarbe:',
        'es': ':chef_tono_de_piel_oscuro:',
        'fr': ':cuisinier_(tous_genres)_peau_foncée:',
        'ja': ':コック_濃い肌色:',
        'ko': ':요리사_검은색_피부:',
        'pt': ':chef_de_cozinha_pele_escura:',
        'it': ':persona_che_cucina_carnagione_scura:',
        'fa': ':آشپز_پوست_آبنوسی:',
        'id': ':koki_warna_kulit_gelap:',
        'zh': ':厨师_较深肤色:',
        'ru': ':повар_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U0001F373': {  # 🧑🏻‍🍳
        'en': ':cook_light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':koch/köchin_helle_hautfarbe:',
        'es': ':chef_tono_de_piel_claro:',
        'fr': ':cuisinier_(tous_genres)_peau_claire:',
        'ja': ':コック_薄い肌色:',
        'ko': ':요리사_하얀_피부:',
        'pt': ':chef_de_cozinha_pele_clara:',
        'it': ':persona_che_cucina_carnagione_chiara:',
        'fa': ':آشپز_پوست_سفید:',
        'id': ':koki_warna_kulit_cerah:',
        'zh': ':厨师_较浅肤色:',
        'ru': ':повар_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U0001F373': {  # 🧑🏾‍🍳
        'en': ':cook_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':koch/köchin_mitteldunkle_hautfarbe:',
        'es': ':chef_tono_de_piel_oscuro_medio:',
        'fr': ':cuisinier_(tous_genres)_peau_mate:',
        'ja': ':コック_やや濃い肌色:',
        'ko': ':요리사_진한_갈색_피부:',
        'pt': ':chef_de_cozinha_pele_morena_escura:',
        'it': ':persona_che_cucina_carnagione_abbastanza_scura:',
        'fa': ':آشپز_پوست_گندمی:',
        'id': ':koki_warna_kulit_gelap-sedang:',
        'zh': ':厨师_中等深肤色:',
        'ru': ':повар_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U0001F373': {  # 🧑🏼‍🍳
        'en': ':cook_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':koch/köchin_mittelhelle_hautfarbe:',
        'es': ':chef_tono_de_piel_claro_medio:',
        'fr': ':cuisinier_(tous_genres)_peau_moyennement_claire:',
        'ja': ':コック_やや薄い肌色:',
        'ko': ':요리사_연한_갈색_피부:',
        'pt': ':chef_de_cozinha_pele_morena_clara:',
        'it': ':persona_che_cucina_carnagione_abbastanza_chiara:',
        'fa': ':آشپز_پوست_روشن:',
        'id': ':koki_warna_kulit_cerah-sedang:',
        'zh': ':厨师_中等浅肤色:',
        'ru': ':повар_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U0001F373': {  # 🧑🏽‍🍳
        'en': ':cook_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':koch/köchin_mittlere_hautfarbe:',
        'es': ':chef_tono_de_piel_medio:',
        'fr': ':cuisinier_(tous_genres)_peau_légèrement_mate:',
        'ja': ':コック_中間の肌色:',
        'ko': ':요리사_갈색_피부:',
        'pt': ':chef_de_cozinha_pele_morena:',
        'it': ':persona_che_cucina_carnagione_olivastra:',
        'fa': ':آشپز_پوست_طلایی:',
        'id': ':koki_warna_kulit_sedang:',
        'zh': ':厨师_中等肤色:',
        'ru': ':повар_средний_тон_кожи:'
    },
    '\U0001F35A': {  # 🍚
        'en': ':cooked_rice:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':rice:'],
        'de': ':reis_in_schüssel:',
        'es': ':arroz_cocido:',
        'fr': ':bol_de_riz:',
        'ja': ':ごはん:',
        'ko': ':밥:',
        'pt': ':arroz_cozido:',
        'it': ':riso_bollito:',
        'fa': ':برنج_پخته:',
        'id': ':nasi_matang:',
        'zh': ':米饭:',
        'ru': ':рис:'
    },
    '\U0001F36A': {  # 🍪
        'en': ':cookie:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':keks:',
        'es': ':galleta:',
        'fr': ':cookie:',
        'ja': ':クッキー:',
        'ko': ':쿠키:',
        'pt': ':biscoito:',
        'it': ':biscotto:',
        'fa': ':کلوچه:',
        'id': ':biskuit:',
        'zh': ':饼干:',
        'ru': ':печенье:'
    },
    '\U0001F373': {  # 🍳
        'en': ':cooking:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':egg:', ':fried_egg:'],
        'de': ':spiegelei_in_bratpfanne:',
        'es': ':cocinar:',
        'fr': ':œuf_au_plat:',
        'ja': ':料理:',
        'ko': ':프라이팬:',
        'pt': ':ovo_frito:',
        'it': ':cucinare:',
        'fa': ':آشپزی:',
        'id': ':memasak:',
        'zh': ':煎蛋:',
        'ru': ':яичница_на_сковороде:'
    },
    '\U000000A9\U0000FE0F': {  # ©️
        'en': ':copyright:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':copyright:',
        'es': ':copyright:',
        'fr': ':symbole_copyright:',
        'ja': ':著作権マーク:',
        'ko': ':저작권:',
        'pt': ':símbolo_de_copyright:',
        'it': ':copyright:',
        'fa': ':حق_نسخه_برداری:',
        'id': ':hak_cipta:',
        'zh': ':版权:',
        'ru': ':значок_защиты_авторского_права:'
    },
    '\U000000A9': {  # ©
        'en': ':copyright:',
        'status': unqualified,
        'E': 0.6,
        'variant': True,
        'de': ':copyright:',
        'es': ':copyright:',
        'fr': ':symbole_copyright:',
        'ja': ':著作権マーク:',
        'ko': ':저작권:',
        'pt': ':símbolo_de_copyright:',
        'it': ':copyright:',
        'fa': ':حق_نسخه_برداری:',
        'id': ':hak_cipta:',
        'zh': ':版权:',
        'ru': ':значок_защиты_авторского_права:'
    },
    '\U0001FAB8': {  # 🪸
        'en': ':coral:',
        'status': fully_qualified,
        'E': 14,
        'de': ':koralle:',
        'es': ':coral:',
        'fr': ':corail:',
        'ja': ':サンゴ:',
        'ko': ':산호초:',
        'pt': ':coral:',
        'it': ':corallo:',
        'fa': ':مرجان:',
        'id': ':terumbu_karang:',
        'zh': ':珊瑚:',
        'ru': ':коралл:'
    },
    '\U0001F6CB\U0000FE0F': {  # 🛋️
        'en': ':couch_and_lamp:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':sofa_und_lampe:',
        'es': ':sofá_y_lámpara:',
        'fr': ':canapé_et_lampe:',
        'ja': ':ソファとランプ:',
        'ko': ':소파와_전등:',
        'pt': ':sofá_e_luminária:',
        'it': ':divano_e_lampada:',
        'fa': ':مبل_راحتی_و_چراغ:',
        'id': ':sofa_dan_lampu:',
        'zh': ':沙发和灯:',
        'ru': ':диван_и_торшер:'
    },
    '\U0001F6CB': {  # 🛋
        'en': ':couch_and_lamp:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':sofa_und_lampe:',
        'es': ':sofá_y_lámpara:',
        'fr': ':canapé_et_lampe:',
        'ja': ':ソファとランプ:',
        'ko': ':소파와_전등:',
        'pt': ':sofá_e_luminária:',
        'it': ':divano_e_lampada:',
        'fa': ':مبل_راحتی_و_چراغ:',
        'id': ':sofa_dan_lampu:',
        'zh': ':沙发和灯:',
        'ru': ':диван_и_торшер:'
    },
    '\U0001F504': {  # 🔄
        'en': ':counterclockwise_arrows_button:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':arrows_counterclockwise:'],
        'de': ':pfeile_gegen_den_uhrzeigersinn:',
        'es': ':flechas_en_sentido_antihorario:',
        'fr': ':flèches_dans_le_sens_antihoraire:',
        'ja': ':左回り矢印:',
        'ko': ':반시계_방향_화살표:',
        'pt': ':botão_de_setas_em_sentido_anti-horário:',
        'it': ':frecce_che_ruotano_in_senso_antiorario:',
        'fa': ':خلاف_جهت_عقربه_های_ساعت:',
        'id': ':tanda_panah_berlawanan_arah_jarum_jam:',
        'zh': ':逆时针箭头按钮:',
        'ru': ':против_часовой_стрелки:'
    },
    '\U0001F491': {  # 💑
        'en': ':couple_with_heart:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':liebespaar:',
        'es': ':pareja_enamorada:',
        'fr': ':couple_avec_cœur:',
        'ja': ':カップルとハート:',
        'ko': ':연인:',
        'pt': ':casal_apaixonado:',
        'it': ':coppia_con_cuore:',
        'fa': ':زوج_عاشق:',
        'id': ':pasangan_dengan_hati:',
        'zh': ':情侣:',
        'ru': ':влюбленная_пара:'
    },
    '\U0001F491\U0001F3FF': {  # 💑🏿
        'en': ':couple_with_heart_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_dunkle_hautfarbe:',
        'es': ':pareja_enamorada_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_peau_foncée:',
        'ja': ':カップルとハート_濃い肌色:',
        'ko': ':연인_검은색_피부:',
        'pt': ':casal_apaixonado_pele_escura:',
        'it': ':coppia_con_cuore_carnagione_scura:',
        'fa': ':زوج_عاشق_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_warna_kulit_gelap:',
        'zh': ':情侣_较深肤色:',
        'ru': ':влюбленная_пара_очень_темный_тон_кожи:'
    },
    '\U0001F491\U0001F3FB': {  # 💑🏻
        'en': ':couple_with_heart_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_helle_hautfarbe:',
        'es': ':pareja_enamorada_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_peau_claire:',
        'ja': ':カップルとハート_薄い肌色:',
        'ko': ':연인_하얀_피부:',
        'pt': ':casal_apaixonado_pele_clara:',
        'it': ':coppia_con_cuore_carnagione_chiara:',
        'fa': ':زوج_عاشق_پوست_سفید:',
        'id': ':pasangan_dengan_hati_warna_kulit_cerah:',
        'zh': ':情侣_较浅肤色:',
        'ru': ':влюбленная_пара_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468': {  # 👨‍❤️‍👨
        'en': ':couple_with_heart_man_man:',
        'status': fully_qualified,
        'E': 2,
        'de': ':liebespaar_mann,_mann:',
        'es': ':pareja_enamorada_hombre_y_hombre:',
        'fr': ':couple_avec_cœur_homme_et_homme:',
        'ja': ':カップルとハート_男性_男性:',
        'ko': ':연인_남자_남자:',
        'pt': ':casal_apaixonado_homem_e_homem:',
        'it': ':coppia_con_cuore_uomo_e_uomo:',
        'fa': ':زوج_عاشق_مرد_مرد:',
        'id': ':pasangan_dengan_hati_pria_pria:',
        'zh': ':情侣_男人男人:',
        'ru': ':влюбленная_пара_мужчина_мужчина:'
    },
    '\U0001F468\U0000200D\U00002764\U0000200D\U0001F468': {  # 👨‍❤‍👨
        'en': ':couple_with_heart_man_man:',
        'status': minimally_qualified,
        'E': 2,
        'de': ':liebespaar_mann_mann:',
        'es': ':pareja_enamorada_hombre_y_hombre:',
        'fr': ':couple_avec_cœur_homme_et_homme:',
        'ja': ':カップルとハート_男性_男性:',
        'ko': ':연인_남자_남자:',
        'pt': ':casal_apaixonado_homem_e_homem:',
        'it': ':coppia_con_cuore_uomo_e_uomo:',
        'fa': ':زوج_عاشق_مرد_مرد:',
        'id': ':pasangan_dengan_hati_pria_pria:',
        'zh': ':情侣_男人男人:',
        'ru': ':влюбленная_пара_мужчина_мужчина:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👨🏿‍❤️‍👨🏿
        'en': ':couple_with_heart_man_man_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_濃い肌色:',
        'ko': ':연인_남자_남자_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap:',
        'zh': ':情侣_男人男人较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👨🏿‍❤‍👨🏿
        'en': ':couple_with_heart_man_man_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_濃い肌色:',
        'ko': ':연인_남자_남자_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap:',
        'zh': ':情侣_男人男人较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👨🏿‍❤️‍👨🏻
        'en': ':couple_with_heart_man_man_dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_foncée_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_濃い肌色_薄い肌色:',
        'ko': ':연인_남자_남자_검은색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':情侣_男人男人较深肤色较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👨🏿‍❤‍👨🏻
        'en': ':couple_with_heart_man_man_dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_foncée_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_濃い肌色_薄い肌色:',
        'ko': ':연인_남자_남자_검은색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':情侣_男人男人较深肤色较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👨🏿‍❤️‍👨🏾
        'en': ':couple_with_heart_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_foncée_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_濃い肌色_やや濃い肌色:',
        'ko': ':연인_남자_남자_검은색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_escura_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人较深肤色中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темныи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👨🏿‍❤‍👨🏾
        'en': ':couple_with_heart_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_foncée_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_濃い肌色_やや濃い肌色:',
        'ko': ':연인_남자_남자_검은색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_escura_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人较深肤色中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темный_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👨🏿‍❤️‍👨🏼
        'en': ':couple_with_heart_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_濃い肌色_やや薄い肌色:',
        'ko': ':연인_남자_남자_검은색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人较深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👨🏿‍❤‍👨🏼
        'en': ':couple_with_heart_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_濃い肌色_やや薄い肌色:',
        'ko': ':연인_남자_남자_검은색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人较深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👨🏿‍❤️‍👨🏽
        'en': ':couple_with_heart_man_man_dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_濃い肌色_中間の肌色:',
        'ko': ':연인_남자_남자_검은색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':情侣_男人男人较深肤色中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👨🏿‍❤‍👨🏽
        'en': ':couple_with_heart_man_man_dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_濃い肌色_中間の肌色:',
        'ko': ':연인_남자_남자_검은색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':情侣_男人男人较深肤色中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👨🏻‍❤️‍👨🏻
        'en': ':couple_with_heart_man_man_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_薄い肌色:',
        'ko': ':연인_남자_남자_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah:',
        'zh': ':情侣_男人男人较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👨🏻‍❤‍👨🏻
        'en': ':couple_with_heart_man_man_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_薄い肌色:',
        'ko': ':연인_남자_남자_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah:',
        'zh': ':情侣_男人男人较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👨🏻‍❤️‍👨🏿
        'en': ':couple_with_heart_man_man_light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_claire_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_薄い肌色_濃い肌色:',
        'ko': ':연인_남자_남자_하얀_피부_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':情侣_男人男人较浅肤色较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👨🏻‍❤‍👨🏿
        'en': ':couple_with_heart_man_man_light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_claire_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_薄い肌色_濃い肌色:',
        'ko': ':연인_남자_남자_하얀_피부_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':情侣_男人男人较浅肤色较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👨🏻‍❤️‍👨🏾
        'en': ':couple_with_heart_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_claire_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_薄い肌色_やや濃い肌色:',
        'ko': ':연인_남자_남자_하얀_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_سفید_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人较浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👨🏻‍❤‍👨🏾
        'en': ':couple_with_heart_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_claire_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_薄い肌色_やや濃い肌色:',
        'ko': ':연인_남자_남자_하얀_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_سفید_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人较浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👨🏻‍❤️‍👨🏼
        'en': ':couple_with_heart_man_man_light_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_claire_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_薄い肌色_やや薄い肌色:',
        'ko': ':연인_남자_남자_하얀_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_clara_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_سفید_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人较浅肤色中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлыи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👨🏻‍❤‍👨🏼
        'en': ':couple_with_heart_man_man_light_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_claire_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_薄い肌色_やや薄い肌色:',
        'ko': ':연인_남자_남자_하얀_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_clara_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_سفید_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人较浅肤色中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлый_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👨🏻‍❤️‍👨🏽
        'en': ':couple_with_heart_man_man_light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_薄い肌色_中間の肌色:',
        'ko': ':연인_남자_남자_하얀_피부_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_سفید_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':情侣_男人男人较浅肤色中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👨🏻‍❤‍👨🏽
        'en': ':couple_with_heart_man_man_light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_薄い肌色_中間の肌色:',
        'ko': ':연인_남자_남자_하얀_피부_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_سفید_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':情侣_男人男人较浅肤色中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_очень_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👨🏾‍❤️‍👨🏾
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👨🏾‍❤‍👨🏾
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👨🏾‍❤️‍👨🏿
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_mate_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色_濃い肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_escura_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':情侣_男人男人中等深肤色较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темныи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👨🏾‍❤‍👨🏿
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_mate_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色_濃い肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_escura_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':情侣_男人男人中等深肤色较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темный_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👨🏾‍❤️‍👨🏻
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_mate_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色_薄い肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_گندمی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':情侣_男人男人中等深肤色较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👨🏾‍❤‍👨🏻
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_mate_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色_薄い肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_گندمی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':情侣_男人男人中等深肤色较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👨🏾‍❤️‍👨🏼
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色_やや薄い肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_گندمی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人中等深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👨🏾‍❤‍👨🏼
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色_やや薄い肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_گندمی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人中等深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👨🏾‍❤️‍👨🏽
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_mate_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色_中間の肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_گندمی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':情侣_男人男人中等深肤色中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👨🏾‍❤‍👨🏽
        'en': ':couple_with_heart_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_mate_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_やや濃い肌色_中間の肌色:',
        'ko': ':연인_남자_남자_진한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_گندمی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':情侣_男人男人中等深肤色中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👨🏼‍❤️‍👨🏼
        'en': ':couple_with_heart_man_man_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👨🏼‍❤‍👨🏼
        'en': ':couple_with_heart_man_man_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👨🏼‍❤️‍👨🏿
        'en': ':couple_with_heart_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色_濃い肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':情侣_男人男人中等浅肤色较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👨🏼‍❤‍👨🏿
        'en': ':couple_with_heart_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色_濃い肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':情侣_男人男人中等浅肤色较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👨🏼‍❤️‍👨🏻
        'en': ':couple_with_heart_man_man_medium-light_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_moyennement_claire_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色_薄い肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_clara_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_روشن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':情侣_男人男人中等浅肤色较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлыи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👨🏼‍❤‍👨🏻
        'en': ':couple_with_heart_man_man_medium-light_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_moyennement_claire_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色_薄い肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_clara_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_روشن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':情侣_男人男人中等浅肤色较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлый_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👨🏼‍❤️‍👨🏾
        'en': ':couple_with_heart_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_moyennement_claire_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色_やや濃い肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_روشن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人中等浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👨🏼‍❤‍👨🏾
        'en': ':couple_with_heart_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_moyennement_claire_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色_やや濃い肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_روشن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人中等浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👨🏼‍❤️‍👨🏽
        'en': ':couple_with_heart_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色_中間の肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_روشن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':情侣_男人男人中等浅肤色中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👨🏼‍❤‍👨🏽
        'en': ':couple_with_heart_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_やや薄い肌色_中間の肌色:',
        'ko': ':연인_남자_남자_연한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_روشن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':情侣_男人男人中等浅肤色中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👨🏽‍❤️‍👨🏽
        'en': ':couple_with_heart_man_man_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_中間の肌色:',
        'ko': ':연인_남자_남자_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang:',
        'zh': ':情侣_男人男人中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👨🏽‍❤‍👨🏽
        'en': ':couple_with_heart_man_man_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_und_mittlere_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_homme_homme_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_男性_男性_中間の肌色:',
        'ko': ':연인_남자_남자_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_e_pele_morena:',
        'it': ':coppia_con_cuore_uomo_uomo_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_مرد_مرد_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang:',
        'zh': ':情侣_男人男人中等肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👨🏽‍❤️‍👨🏿
        'en': ':couple_with_heart_man_man_medium_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_中間の肌色_濃い肌色:',
        'ko': ':연인_남자_남자_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':情侣_男人男人中等肤色较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_среднии_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👨🏽‍❤‍👨🏿
        'en': ':couple_with_heart_man_man_medium_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':カップルとハート_男性_男性_中間の肌色_濃い肌色:',
        'ko': ':연인_남자_남자_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_e_pele_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':情侣_男人男人中等肤色较深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_средний_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👨🏽‍❤️‍👨🏻
        'en': ':couple_with_heart_man_man_medium_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_légèrement_mate_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_中間の肌色_薄い肌色:',
        'ko': ':연인_남자_남자_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_طلایی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':情侣_男人男人中等肤色较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_среднии_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👨🏽‍❤‍👨🏻
        'en': ':couple_with_heart_man_man_medium_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_homme_homme_peau_légèrement_mate_et_peau_claire:',
        'ja': ':カップルとハート_男性_男性_中間の肌色_薄い肌色:',
        'ko': ':연인_남자_남자_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_e_pele_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_طلایی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':情侣_男人男人中等肤色较浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_средний_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👨🏽‍❤️‍👨🏾
        'en': ':couple_with_heart_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_légèrement_mate_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_中間の肌色_やや濃い肌色:',
        'ko': ':연인_남자_남자_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_طلایی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人中等肤色中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_среднии_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👨🏽‍❤‍👨🏾
        'en': ':couple_with_heart_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_légèrement_mate_et_peau_mate:',
        'ja': ':カップルとハート_男性_男性_中間の肌色_やや濃い肌色:',
        'ko': ':연인_남자_남자_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_طلایی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_男人男人中等肤色中等深肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_средний_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👨🏽‍❤️‍👨🏼
        'en': ':couple_with_heart_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_中間の肌色_やや薄い肌色:',
        'ko': ':연인_남자_남자_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_طلایی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人中等肤色中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_среднии_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👨🏽‍❤‍👨🏼
        'en': ':couple_with_heart_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_mann_mann_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_hombre_hombre_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_homme_homme_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_男性_男性_中間の肌色_やや薄い肌色:',
        'ko': ':연인_남자_남자_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_homem_homem_pele_morena_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_uomo_uomo_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_مرد_مرد_پوست_طلایی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_pria_pria_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_男人男人中等肤色中等浅肤色:',
        'ru': ':влюбленная_пара_мужчина_мужчина_средний_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F491\U0001F3FE': {  # 💑🏾
        'en': ':couple_with_heart_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_peau_mate:',
        'ja': ':カップルとハート_やや濃い肌色:',
        'ko': ':연인_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pele_morena_escura:',
        'it': ':coppia_con_cuore_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_warna_kulit_gelap-sedang:',
        'zh': ':情侣_中等深肤色:',
        'ru': ':влюбленная_пара_темный_тон_кожи:'
    },
    '\U0001F491\U0001F3FC': {  # 💑🏼
        'en': ':couple_with_heart_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_peau_moyennement_claire:',
        'ja': ':カップルとハート_やや薄い肌色:',
        'ko': ':연인_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pele_morena_clara:',
        'it': ':coppia_con_cuore_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_پوست_روشن:',
        'id': ':pasangan_dengan_hati_warna_kulit_cerah-sedang:',
        'zh': ':情侣_中等浅肤色:',
        'ru': ':влюбленная_пара_светлый_тон_кожи:'
    },
    '\U0001F491\U0001F3FD': {  # 💑🏽
        'en': ':couple_with_heart_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_mittlere_hautfarbe:',
        'es': ':pareja_enamorada_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_peau_légèrement_mate:',
        'ja': ':カップルとハート_中間の肌色:',
        'ko': ':연인_갈색_피부:',
        'pt': ':casal_apaixonado_pele_morena:',
        'it': ':coppia_con_cuore_carnagione_olivastra:',
        'fa': ':زوج_عاشق_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_warna_kulit_sedang:',
        'zh': ':情侣_中等肤色:',
        'ru': ':влюбленная_пара_средний_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏿‍❤️‍🧑🏻
        'en': ':couple_with_heart_person_person_dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_foncée_et_peau_claire:',
        'ja': ':カップルとハート_大人_大人_濃い肌色_薄い肌色:',
        'ko': ':연인_사람_사람_검은색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':情侣_成人成人较深肤色较浅肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_очень_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏿‍❤‍🧑🏻
        'en': ':couple_with_heart_person_person_dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_foncée_et_peau_claire:',
        'ja': ':カップルとハート_大人_大人_濃い肌色_薄い肌色:',
        'ko': ':연인_사람_사람_검은색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':情侣_成人成人较深肤色较浅肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_очень_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏿‍❤️‍🧑🏾
        'en': ':couple_with_heart_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_foncée_et_peau_mate:',
        'ja': ':カップルとハート_大人_大人_濃い肌色_やや濃い肌色:',
        'ko': ':연인_사람_사람_검은색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_escura_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':情侣_成人成人较深肤色中等深肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_очень_темныи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏿‍❤‍🧑🏾
        'en': ':couple_with_heart_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_foncée_et_peau_mate:',
        'ja': ':カップルとハート_大人_大人_濃い肌色_やや濃い肌色:',
        'ko': ':연인_사람_사람_검은색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_escura_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':情侣_成人成人较深肤色中等深肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_очень_темный_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏿‍❤️‍🧑🏼
        'en': ':couple_with_heart_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_大人_大人_濃い肌色_やや薄い肌色:',
        'ko': ':연인_사람_사람_검은색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':情侣_成人成人较深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_очень_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏿‍❤‍🧑🏼
        'en': ':couple_with_heart_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_大人_大人_濃い肌色_やや薄い肌色:',
        'ko': ':연인_사람_사람_검은색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':情侣_成人成人较深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_очень_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏿‍❤️‍🧑🏽
        'en': ':couple_with_heart_person_person_dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_大人_大人_濃い肌色_中間の肌色:',
        'ko': ':연인_사람_사람_검은색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':情侣_成人成人较深肤色中等肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_очень_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏿‍❤‍🧑🏽
        'en': ':couple_with_heart_person_person_dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_大人_大人_濃い肌色_中間の肌色:',
        'ko': ':연인_사람_사람_검은색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':情侣_成人成人较深肤色中等肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_очень_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏻‍❤️‍🧑🏿
        'en': ':couple_with_heart_person_person_light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_claire_et_peau_foncée:',
        'ja': ':カップルとハート_大人_大人_薄い肌色_濃い肌色:',
        'ko': ':연인_사람_사람_하얀_피부_검은색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':情侣_成人成人较浅肤色较深肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_очень_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏻‍❤‍🧑🏿
        'en': ':couple_with_heart_person_person_light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_claire_et_peau_foncée:',
        'ja': ':カップルとハート_大人_大人_薄い肌色_濃い肌色:',
        'ko': ':연인_사람_사람_하얀_피부_검은색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':情侣_成人成人较浅肤色较深肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_очень_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏻‍❤️‍🧑🏾
        'en': ':couple_with_heart_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_claire_et_peau_mate:',
        'ja': ':カップルとハート_大人_大人_薄い肌色_やや濃い肌色:',
        'ko': ':연인_사람_사람_하얀_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_سفید_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':情侣_成人成人较浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_очень_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏻‍❤‍🧑🏾
        'en': ':couple_with_heart_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_claire_et_peau_mate:',
        'ja': ':カップルとハート_大人_大人_薄い肌色_やや濃い肌色:',
        'ko': ':연인_사람_사람_하얀_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_سفید_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':情侣_成人成人较浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_очень_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏻‍❤️‍🧑🏼
        'en': ':couple_with_heart_person_person_light_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_claire_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_大人_大人_薄い肌色_やや薄い肌色:',
        'ko': ':연인_사람_사람_하얀_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_clara_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_سفید_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':情侣_成人成人较浅肤色中等浅肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_очень_светлыи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏻‍❤‍🧑🏼
        'en': ':couple_with_heart_person_person_light_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_claire_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_大人_大人_薄い肌色_やや薄い肌色:',
        'ko': ':연인_사람_사람_하얀_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_clara_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_سفید_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':情侣_成人成人较浅肤色中等浅肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_очень_светлый_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏻‍❤️‍🧑🏽
        'en': ':couple_with_heart_person_person_light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_大人_大人_薄い肌色_中間の肌色:',
        'ko': ':연인_사람_사람_하얀_피부_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_سفید_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':情侣_成人成人较浅肤色中等肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_очень_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏻‍❤‍🧑🏽
        'en': ':couple_with_heart_person_person_light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_大人_大人_薄い肌色_中間の肌色:',
        'ko': ':연인_사람_사람_하얀_피부_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_سفید_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':情侣_成人成人较浅肤色中等肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_очень_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏾‍❤️‍🧑🏿
        'en': ':couple_with_heart_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_mate_et_peau_foncée:',
        'ja': ':カップルとハート_大人_大人_やや濃い肌色_濃い肌色:',
        'ko': ':연인_사람_사람_진한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_escura_e_pele_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':情侣_成人成人中等深肤色较深肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_темныи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏾‍❤‍🧑🏿
        'en': ':couple_with_heart_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_mate_et_peau_foncée:',
        'ja': ':カップルとハート_大人_大人_やや濃い肌色_濃い肌色:',
        'ko': ':연인_사람_사람_진한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_escura_e_pele_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':情侣_成人成人中等深肤色较深肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_темный_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏾‍❤️‍🧑🏻
        'en': ':couple_with_heart_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_mate_et_peau_claire:',
        'ja': ':カップルとハート_大人_大人_やや濃い肌色_薄い肌色:',
        'ko': ':연인_사람_사람_진한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_گندمی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':情侣_成人成人中等深肤色较浅肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏾‍❤‍🧑🏻
        'en': ':couple_with_heart_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_mate_et_peau_claire:',
        'ja': ':カップルとハート_大人_大人_やや濃い肌色_薄い肌色:',
        'ko': ':연인_사람_사람_진한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_گندمی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':情侣_成人成人中等深肤色较浅肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏾‍❤️‍🧑🏼
        'en': ':couple_with_heart_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_大人_大人_やや濃い肌色_やや薄い肌色:',
        'ko': ':연인_사람_사람_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_گندمی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_成人成人中等深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏾‍❤‍🧑🏼
        'en': ':couple_with_heart_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_大人_大人_やや濃い肌色_やや薄い肌色:',
        'ko': ':연인_사람_사람_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_گندمی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_成人成人中等深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏾‍❤️‍🧑🏽
        'en': ':couple_with_heart_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_mate_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_大人_大人_やや濃い肌色_中間の肌色:',
        'ko': ':연인_사람_사람_진한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_گندمی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':情侣_成人成人中等深肤色中等肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏾‍❤‍🧑🏽
        'en': ':couple_with_heart_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_mate_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_大人_大人_やや濃い肌色_中間の肌色:',
        'ko': ':연인_사람_사람_진한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_گندمی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':情侣_成人成人中等深肤色中等肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏼‍❤️‍🧑🏿
        'en': ':couple_with_heart_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':カップルとハート_大人_大人_やや薄い肌色_濃い肌色:',
        'ko': ':연인_사람_사람_연한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':情侣_成人成人中等浅肤色较深肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏼‍❤‍🧑🏿
        'en': ':couple_with_heart_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':カップルとハート_大人_大人_やや薄い肌色_濃い肌色:',
        'ko': ':연인_사람_사람_연한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':情侣_成人成人中等浅肤色较深肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏼‍❤️‍🧑🏻
        'en': ':couple_with_heart_person_person_medium-light_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_moyennement_claire_et_peau_claire:',
        'ja': ':カップルとハート_大人_大人_やや薄い肌色_薄い肌色:',
        'ko': ':연인_사람_사람_연한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_clara_e_pele_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_روشن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':情侣_成人成人中等浅肤色较浅肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_светлыи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏼‍❤‍🧑🏻
        'en': ':couple_with_heart_person_person_medium-light_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_moyennement_claire_et_peau_claire:',
        'ja': ':カップルとハート_大人_大人_やや薄い肌色_薄い肌色:',
        'ko': ':연인_사람_사람_연한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_clara_e_pele_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_روشن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':情侣_成人成人中等浅肤色较浅肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_светлый_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏼‍❤️‍🧑🏾
        'en': ':couple_with_heart_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_moyennement_claire_et_peau_mate:',
        'ja': ':カップルとハート_大人_大人_やや薄い肌色_やや濃い肌色:',
        'ko': ':연인_사람_사람_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_روشن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_成人成人中等浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏼‍❤‍🧑🏾
        'en': ':couple_with_heart_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_moyennement_claire_et_peau_mate:',
        'ja': ':カップルとハート_大人_大人_やや薄い肌色_やや濃い肌色:',
        'ko': ':연인_사람_사람_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_روشن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_成人成人中等浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏼‍❤️‍🧑🏽
        'en': ':couple_with_heart_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_大人_大人_やや薄い肌色_中間の肌色:',
        'ko': ':연인_사람_사람_연한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_روشن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':情侣_成人成人中等浅肤色中等肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏼‍❤‍🧑🏽
        'en': ':couple_with_heart_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_大人_大人_やや薄い肌色_中間の肌色:',
        'ko': ':연인_사람_사람_연한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_روشن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':情侣_成人成人中等浅肤色中等肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏽‍❤️‍🧑🏿
        'en': ':couple_with_heart_person_person_medium_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':カップルとハート_大人_大人_中間の肌色_濃い肌色:',
        'ko': ':연인_사람_사람_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_e_pele_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':情侣_成人成人中等肤色较深肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_среднии_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏽‍❤‍🧑🏿
        'en': ':couple_with_heart_person_person_medium_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':カップルとハート_大人_大人_中間の肌色_濃い肌色:',
        'ko': ':연인_사람_사람_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_e_pele_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':情侣_成人成人中等肤色较深肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_средний_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏽‍❤️‍🧑🏻
        'en': ':couple_with_heart_person_person_medium_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_légèrement_mate_et_peau_claire:',
        'ja': ':カップルとハート_大人_大人_中間の肌色_薄い肌色:',
        'ko': ':연인_사람_사람_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_e_pele_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_طلایی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':情侣_成人成人中等肤色较浅肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_среднии_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏽‍❤‍🧑🏻
        'en': ':couple_with_heart_person_person_medium_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_légèrement_mate_et_peau_claire:',
        'ja': ':カップルとハート_大人_大人_中間の肌色_薄い肌色:',
        'ko': ':연인_사람_사람_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_e_pele_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_طلایی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':情侣_成人成人中等肤色较浅肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_средний_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏽‍❤️‍🧑🏾
        'en': ':couple_with_heart_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_légèrement_mate_et_peau_mate:',
        'ja': ':カップルとハート_大人_大人_中間の肌色_やや濃い肌色:',
        'ko': ':연인_사람_사람_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_طلایی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_成人成人中等肤色中等深肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_среднии_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏽‍❤‍🧑🏾
        'en': ':couple_with_heart_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_légèrement_mate_et_peau_mate:',
        'ja': ':カップルとハート_大人_大人_中間の肌色_やや濃い肌色:',
        'ko': ':연인_사람_사람_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_طلایی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_成人成人中等肤色中等深肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_средний_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏽‍❤️‍🧑🏼
        'en': ':couple_with_heart_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_erwachsener_erwachsener_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_大人_大人_中間の肌色_やや薄い肌色:',
        'ko': ':연인_사람_사람_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_بزرگسال_بزرگسال_پوست_طلایی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_成人成人中等肤色中等浅肤色:',
        'ru': ':влюбленная_пара_взрослыи_взрослыи_среднии_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏽‍❤‍🧑🏼
        'en': ':couple_with_heart_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_person_person_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_persona_adulta_persona_adulta_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_adulte_adulte_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_大人_大人_中間の肌色_やや薄い肌色:',
        'ko': ':연인_사람_사람_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_pessoa_pessoa_pele_morena_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_persona_persona_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_شخص_شخص_پوست_طلایی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_dewasa_dewasa_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_成人成人中等肤色中等浅肤色:',
        'ru': ':влюбленная_пара_взрослый_взрослый_средний_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468': {  # 👩‍❤️‍👨
        'en': ':couple_with_heart_woman_man:',
        'status': fully_qualified,
        'E': 2,
        'de': ':liebespaar_frau,_mann:',
        'es': ':pareja_enamorada_mujer_y_hombre:',
        'fr': ':couple_avec_cœur_femme_et_homme:',
        'ja': ':カップルとハート_女性_男性:',
        'ko': ':연인_여자_남자:',
        'pt': ':casal_apaixonado_mulher_e_homem:',
        'it': ':coppia_con_cuore_donna_e_uomo:',
        'fa': ':زوج_عاشق_زن_مرد:',
        'id': ':pasangan_dengan_hati_wanita_pria:',
        'zh': ':情侣_女人男人:',
        'ru': ':влюбленная_пара_женщина_мужчина:'
    },
    '\U0001F469\U0000200D\U00002764\U0000200D\U0001F468': {  # 👩‍❤‍👨
        'en': ':couple_with_heart_woman_man:',
        'status': minimally_qualified,
        'E': 2,
        'de': ':liebespaar_frau_mann:',
        'es': ':pareja_enamorada_mujer_y_hombre:',
        'fr': ':couple_avec_cœur_femme_et_homme:',
        'ja': ':カップルとハート_女性_男性:',
        'ko': ':연인_여자_남자:',
        'pt': ':casal_apaixonado_mulher_e_homem:',
        'it': ':coppia_con_cuore_donna_e_uomo:',
        'fa': ':زوج_عاشق_زن_مرد:',
        'id': ':pasangan_dengan_hati_wanita_pria:',
        'zh': ':情侣_女人男人:',
        'ru': ':влюбленная_пара_женщина_мужчина:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👩🏿‍❤️‍👨🏿
        'en': ':couple_with_heart_woman_man_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_濃い肌色:',
        'ko': ':연인_여자_남자_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap:',
        'zh': ':情侣_女人男人较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👩🏿‍❤‍👨🏿
        'en': ':couple_with_heart_woman_man_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_濃い肌色:',
        'ko': ':연인_여자_남자_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap:',
        'zh': ':情侣_女人男人较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👩🏿‍❤️‍👨🏻
        'en': ':couple_with_heart_woman_man_dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_foncée_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_濃い肌色_薄い肌色:',
        'ko': ':연인_여자_남자_검은색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':情侣_女人男人较深肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👩🏿‍❤‍👨🏻
        'en': ':couple_with_heart_woman_man_dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_foncée_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_濃い肌色_薄い肌色:',
        'ko': ':연인_여자_남자_검은색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':情侣_女人男人较深肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👩🏿‍❤️‍👨🏾
        'en': ':couple_with_heart_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_foncée_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_濃い肌色_やや濃い肌色:',
        'ko': ':연인_여자_남자_검은색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_escura_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人较深肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темныи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👩🏿‍❤‍👨🏾
        'en': ':couple_with_heart_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_foncée_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_濃い肌色_やや濃い肌色:',
        'ko': ':연인_여자_남자_검은색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_escura_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人较深肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темный_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👩🏿‍❤️‍👨🏼
        'en': ':couple_with_heart_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_濃い肌色_やや薄い肌色:',
        'ko': ':연인_여자_남자_검은색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人较深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👩🏿‍❤‍👨🏼
        'en': ':couple_with_heart_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_濃い肌色_やや薄い肌色:',
        'ko': ':연인_여자_남자_검은색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人较深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👩🏿‍❤️‍👨🏽
        'en': ':couple_with_heart_woman_man_dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_濃い肌色_中間の肌色:',
        'ko': ':연인_여자_남자_검은색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':情侣_女人男人较深肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👩🏿‍❤‍👨🏽
        'en': ':couple_with_heart_woman_man_dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_濃い肌色_中間の肌色:',
        'ko': ':연인_여자_남자_검은색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':情侣_女人男人较深肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👩🏻‍❤️‍👨🏻
        'en': ':couple_with_heart_woman_man_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_薄い肌色:',
        'ko': ':연인_여자_남자_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah:',
        'zh': ':情侣_女人男人较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👩🏻‍❤‍👨🏻
        'en': ':couple_with_heart_woman_man_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_薄い肌色:',
        'ko': ':연인_여자_남자_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah:',
        'zh': ':情侣_女人男人较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👩🏻‍❤️‍👨🏿
        'en': ':couple_with_heart_woman_man_light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_claire_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_薄い肌色_濃い肌色:',
        'ko': ':연인_여자_남자_하얀_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':情侣_女人男人较浅肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👩🏻‍❤‍👨🏿
        'en': ':couple_with_heart_woman_man_light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_claire_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_薄い肌色_濃い肌色:',
        'ko': ':연인_여자_남자_하얀_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':情侣_女人男人较浅肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👩🏻‍❤️‍👨🏾
        'en': ':couple_with_heart_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_claire_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_薄い肌色_やや濃い肌色:',
        'ko': ':연인_여자_남자_하얀_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_سفید_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人较浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👩🏻‍❤‍👨🏾
        'en': ':couple_with_heart_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_claire_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_薄い肌色_やや濃い肌色:',
        'ko': ':연인_여자_남자_하얀_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_سفید_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人较浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👩🏻‍❤️‍👨🏼
        'en': ':couple_with_heart_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_claire_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_薄い肌色_やや薄い肌色:',
        'ko': ':연인_여자_남자_하얀_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_clara_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_سفید_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人较浅肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлыи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👩🏻‍❤‍👨🏼
        'en': ':couple_with_heart_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_claire_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_薄い肌色_やや薄い肌色:',
        'ko': ':연인_여자_남자_하얀_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_clara_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_سفید_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人较浅肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлый_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👩🏻‍❤️‍👨🏽
        'en': ':couple_with_heart_woman_man_light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_薄い肌色_中間の肌色:',
        'ko': ':연인_여자_남자_하얀_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_سفید_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':情侣_女人男人较浅肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👩🏻‍❤‍👨🏽
        'en': ':couple_with_heart_woman_man_light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_薄い肌色_中間の肌色:',
        'ko': ':연인_여자_남자_하얀_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_سفید_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':情侣_女人男人较浅肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_очень_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👩🏾‍❤️‍👨🏾
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👩🏾‍❤‍👨🏾
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👩🏾‍❤️‍👨🏿
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_mate_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色_濃い肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_escura_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人男人中等深肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темныи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👩🏾‍❤‍👨🏿
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_mate_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色_濃い肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_escura_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人男人中等深肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темный_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👩🏾‍❤️‍👨🏻
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_mate_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色_薄い肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_گندمی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人男人中等深肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👩🏾‍❤‍👨🏻
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_mate_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色_薄い肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_گندمی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人男人中等深肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👩🏾‍❤️‍👨🏼
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色_やや薄い肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_گندمی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人中等深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👩🏾‍❤‍👨🏼
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色_やや薄い肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_گندمی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人中等深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👩🏾‍❤️‍👨🏽
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_mate_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色_中間の肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_گندمی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':情侣_女人男人中等深肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👩🏾‍❤‍👨🏽
        'en': ':couple_with_heart_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_mate_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_やや濃い肌色_中間の肌色:',
        'ko': ':연인_여자_남자_진한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_گندمی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':情侣_女人男人中等深肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👩🏼‍❤️‍👨🏼
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👩🏼‍❤‍👨🏼
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👩🏼‍❤️‍👨🏿
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色_濃い肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人男人中等浅肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👩🏼‍❤‍👨🏿
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色_濃い肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人男人中等浅肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👩🏼‍❤️‍👨🏻
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_moyennement_claire_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色_薄い肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_clara_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_روشن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人男人中等浅肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлыи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👩🏼‍❤‍👨🏻
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_moyennement_claire_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色_薄い肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_clara_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_روشن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人男人中等浅肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлый_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👩🏼‍❤️‍👨🏾
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_moyennement_claire_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色_やや濃い肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_روشن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人中等浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👩🏼‍❤‍👨🏾
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_moyennement_claire_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色_やや濃い肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_روشن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人中等浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👩🏼‍❤️‍👨🏽
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色_中間の肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_روشن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':情侣_女人男人中等浅肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👩🏼‍❤‍👨🏽
        'en': ':couple_with_heart_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_やや薄い肌色_中間の肌色:',
        'ko': ':연인_여자_남자_연한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_روشن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':情侣_女人男人中等浅肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': {  # 👩🏽‍❤️‍👨🏽
        'en': ':couple_with_heart_woman_man_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_中間の肌色:',
        'ko': ':연인_여자_남자_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang:',
        'zh': ':情侣_女人男人中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': {  # 👩🏽‍❤‍👨🏽
        'en': ':couple_with_heart_woman_man_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_und_mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_homme_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_男性_中間の肌色:',
        'ko': ':연인_여자_남자_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_uomo_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_مرد_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang:',
        'zh': ':情侣_女人男人中等肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': {  # 👩🏽‍❤️‍👨🏿
        'en': ':couple_with_heart_woman_man_medium_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_中間の肌色_濃い肌色:',
        'ko': ':연인_여자_남자_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人男人中等肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_среднии_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': {  # 👩🏽‍❤‍👨🏿
        'en': ':couple_with_heart_woman_man_medium_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':カップルとハート_女性_男性_中間の肌色_濃い肌色:',
        'ko': ':연인_여자_남자_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人男人中等肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_средний_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': {  # 👩🏽‍❤️‍👨🏻
        'en': ':couple_with_heart_woman_man_medium_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_légèrement_mate_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_中間の肌色_薄い肌色:',
        'ko': ':연인_여자_남자_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_طلایی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人男人中等肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_среднии_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': {  # 👩🏽‍❤‍👨🏻
        'en': ':couple_with_heart_woman_man_medium_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_homme_peau_légèrement_mate_et_peau_claire:',
        'ja': ':カップルとハート_女性_男性_中間の肌色_薄い肌色:',
        'ko': ':연인_여자_남자_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_طلایی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人男人中等肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_средний_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': {  # 👩🏽‍❤️‍👨🏾
        'en': ':couple_with_heart_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_légèrement_mate_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_中間の肌色_やや濃い肌色:',
        'ko': ':연인_여자_남자_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_طلایی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人中等肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_среднии_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': {  # 👩🏽‍❤‍👨🏾
        'en': ':couple_with_heart_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_légèrement_mate_et_peau_mate:',
        'ja': ':カップルとハート_女性_男性_中間の肌色_やや濃い肌色:',
        'ko': ':연인_여자_남자_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_طلایی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人男人中等肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_средний_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': {  # 👩🏽‍❤️‍👨🏼
        'en': ':couple_with_heart_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_中間の肌色_やや薄い肌色:',
        'ko': ':연인_여자_남자_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_طلایی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人中等肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_среднии_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': {  # 👩🏽‍❤‍👨🏼
        'en': ':couple_with_heart_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_mann_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_hombre_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_homme_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_男性_中間の肌色_やや薄い肌色:',
        'ko': ':연인_여자_남자_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_homem_pele_morena_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_uomo_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_مرد_پوست_طلایی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_pria_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人男人中等肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_мужчина_средний_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469': {  # 👩‍❤️‍👩
        'en': ':couple_with_heart_woman_woman:',
        'status': fully_qualified,
        'E': 2,
        'de': ':liebespaar_frau,_frau:',
        'es': ':pareja_enamorada_mujer_y_mujer:',
        'fr': ':couple_avec_cœur_femme_et_femme:',
        'ja': ':カップルとハート_女性_女性:',
        'ko': ':연인_여자_여자:',
        'pt': ':casal_apaixonado_mulher_e_mulher:',
        'it': ':coppia_con_cuore_donna_e_donna:',
        'fa': ':زوج_عاشق_زن_زن:',
        'id': ':pasangan_dengan_hati_wanita_wanita:',
        'zh': ':情侣_女人女人:',
        'ru': ':влюбленная_пара_женщина_женщина:'
    },
    '\U0001F469\U0000200D\U00002764\U0000200D\U0001F469': {  # 👩‍❤‍👩
        'en': ':couple_with_heart_woman_woman:',
        'status': minimally_qualified,
        'E': 2,
        'de': ':liebespaar_frau_frau:',
        'es': ':pareja_enamorada_mujer_y_mujer:',
        'fr': ':couple_avec_cœur_femme_et_femme:',
        'ja': ':カップルとハート_女性_女性:',
        'ko': ':연인_여자_여자:',
        'pt': ':casal_apaixonado_mulher_e_mulher:',
        'it': ':coppia_con_cuore_donna_e_donna:',
        'fa': ':زوج_عاشق_زن_زن:',
        'id': ':pasangan_dengan_hati_wanita_wanita:',
        'zh': ':情侣_女人女人:',
        'ru': ':влюбленная_пара_женщина_женщина:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {  # 👩🏿‍❤️‍👩🏿
        'en': ':couple_with_heart_woman_woman_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_濃い肌色:',
        'ko': ':연인_여자_여자_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap:',
        'zh': ':情侣_女人女人较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {  # 👩🏿‍❤‍👩🏿
        'en': ':couple_with_heart_woman_woman_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_濃い肌色:',
        'ko': ':연인_여자_여자_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap:',
        'zh': ':情侣_女人女人较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {  # 👩🏿‍❤️‍👩🏻
        'en': ':couple_with_heart_woman_woman_dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_foncée_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_濃い肌色_薄い肌色:',
        'ko': ':연인_여자_여자_검은색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':情侣_女人女人较深肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {  # 👩🏿‍❤‍👩🏻
        'en': ':couple_with_heart_woman_woman_dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_foncée_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_濃い肌色_薄い肌色:',
        'ko': ':연인_여자_여자_검은색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':情侣_女人女人较深肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {  # 👩🏿‍❤️‍👩🏾
        'en': ':couple_with_heart_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_foncée_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_濃い肌色_やや濃い肌色:',
        'ko': ':연인_여자_여자_검은색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_escura_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人较深肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темныи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {  # 👩🏿‍❤‍👩🏾
        'en': ':couple_with_heart_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_foncée_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_濃い肌色_やや濃い肌色:',
        'ko': ':연인_여자_여자_검은색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_escura_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人较深肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темный_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {  # 👩🏿‍❤️‍👩🏼
        'en': ':couple_with_heart_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_濃い肌色_やや薄い肌色:',
        'ko': ':연인_여자_여자_검은색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人较深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {  # 👩🏿‍❤‍👩🏼
        'en': ':couple_with_heart_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_濃い肌色_やや薄い肌色:',
        'ko': ':연인_여자_여자_검은색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人较深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {  # 👩🏿‍❤️‍👩🏽
        'en': ':couple_with_heart_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_濃い肌色_中間の肌色:',
        'ko': ':연인_여자_여자_검은색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':情侣_女人女人较深肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {  # 👩🏿‍❤‍👩🏽
        'en': ':couple_with_heart_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_濃い肌色_中間の肌色:',
        'ko': ':연인_여자_여자_검은색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':情侣_女人女人较深肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {  # 👩🏻‍❤️‍👩🏻
        'en': ':couple_with_heart_woman_woman_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_薄い肌色:',
        'ko': ':연인_여자_여자_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah:',
        'zh': ':情侣_女人女人较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {  # 👩🏻‍❤‍👩🏻
        'en': ':couple_with_heart_woman_woman_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_薄い肌色:',
        'ko': ':연인_여자_여자_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah:',
        'zh': ':情侣_女人女人较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {  # 👩🏻‍❤️‍👩🏿
        'en': ':couple_with_heart_woman_woman_light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_claire_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_薄い肌色_濃い肌色:',
        'ko': ':연인_여자_여자_하얀_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':情侣_女人女人较浅肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {  # 👩🏻‍❤‍👩🏿
        'en': ':couple_with_heart_woman_woman_light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_claire_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_薄い肌色_濃い肌色:',
        'ko': ':연인_여자_여자_하얀_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':情侣_女人女人较浅肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {  # 👩🏻‍❤️‍👩🏾
        'en': ':couple_with_heart_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_claire_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_薄い肌色_やや濃い肌色:',
        'ko': ':연인_여자_여자_하얀_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_سفید_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人较浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {  # 👩🏻‍❤‍👩🏾
        'en': ':couple_with_heart_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_claire_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_薄い肌色_やや濃い肌色:',
        'ko': ':연인_여자_여자_하얀_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_سفید_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人较浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {  # 👩🏻‍❤️‍👩🏼
        'en': ':couple_with_heart_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_claire_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_薄い肌色_やや薄い肌色:',
        'ko': ':연인_여자_여자_하얀_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_clara_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_سفید_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人较浅肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлыи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {  # 👩🏻‍❤‍👩🏼
        'en': ':couple_with_heart_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_claire_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_薄い肌色_やや薄い肌色:',
        'ko': ':연인_여자_여자_하얀_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_clara_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_سفید_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人较浅肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлый_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {  # 👩🏻‍❤️‍👩🏽
        'en': ':couple_with_heart_woman_woman_light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_薄い肌色_中間の肌色:',
        'ko': ':연인_여자_여자_하얀_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_پوست_سفید_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':情侣_女人女人较浅肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {  # 👩🏻‍❤‍👩🏽
        'en': ':couple_with_heart_woman_woman_light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_薄い肌色_中間の肌色:',
        'ko': ':연인_여자_여자_하얀_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_پوست_سفید_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':情侣_女人女人较浅肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_очень_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {  # 👩🏾‍❤️‍👩🏾
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {  # 👩🏾‍❤‍👩🏾
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {  # 👩🏾‍❤️‍👩🏿
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_mate_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色_濃い肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_escura_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人女人中等深肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темныи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {  # 👩🏾‍❤‍👩🏿
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_mate_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色_濃い肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_escura_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人女人中等深肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темный_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {  # 👩🏾‍❤️‍👩🏻
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_mate_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色_薄い肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_گندمی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人女人中等深肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {  # 👩🏾‍❤‍👩🏻
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_mate_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色_薄い肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_escura_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_گندمی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人女人中等深肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {  # 👩🏾‍❤️‍👩🏼
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色_やや薄い肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_گندمی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人中等深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {  # 👩🏾‍❤‍👩🏼
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色_やや薄い肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_escura_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_گندمی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人中等深肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {  # 👩🏾‍❤️‍👩🏽
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_mate_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色_中間の肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_پوست_گندمی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':情侣_女人女人中等深肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {  # 👩🏾‍❤‍👩🏽
        'en': ':couple_with_heart_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_mate_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_やや濃い肌色_中間の肌色:',
        'ko': ':연인_여자_여자_진한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_escura_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_پوست_گندمی_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':情侣_女人女人中等深肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {  # 👩🏼‍❤️‍👩🏼
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {  # 👩🏼‍❤‍👩🏼
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {  # 👩🏼‍❤️‍👩🏿
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色_濃い肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人女人中等浅肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {  # 👩🏼‍❤‍👩🏿
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色_濃い肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_clara_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人女人中等浅肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {  # 👩🏼‍❤️‍👩🏻
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_moyennement_claire_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色_薄い肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_clara_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_روشن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人女人中等浅肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлыи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {  # 👩🏼‍❤‍👩🏻
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_moyennement_claire_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色_薄い肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_clara_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_روشن_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人女人中等浅肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлый_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {  # 👩🏼‍❤️‍👩🏾
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_moyennement_claire_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色_やや濃い肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_روشن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人中等浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {  # 👩🏼‍❤‍👩🏾
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_moyennement_claire_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色_やや濃い肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_clara_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_روشن_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人中等浅肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {  # 👩🏼‍❤️‍👩🏽
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色_中間の肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_پوست_روشن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':情侣_女人女人中等浅肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {  # 👩🏼‍❤‍👩🏽
        'en': ':couple_with_heart_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_やや薄い肌色_中間の肌色:',
        'ko': ':연인_여자_여자_연한_갈색_피부_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_clara_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_پوست_روشن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':情侣_女人女人中等浅肤色中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': {  # 👩🏽‍❤️‍👩🏽
        'en': ':couple_with_heart_woman_woman_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_中間の肌色:',
        'ko': ':연인_여자_여자_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang:',
        'zh': ':情侣_女人女人中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_среднии_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': {  # 👩🏽‍❤‍👩🏽
        'en': ':couple_with_heart_woman_woman_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_und_mittlere_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio:',
        'fr': ':couple_avec_cœur_femme_femme_et_peau_légèrement_mate:',
        'ja': ':カップルとハート_女性_女性_中間の肌色:',
        'ko': ':연인_여자_여자_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_e_pele_morena:',
        'it': ':coppia_con_cuore_donna_donna_e_carnagione_olivastra:',
        'fa': ':زوج_عاشق_زن_زن_و_پوست_طلایی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang:',
        'zh': ':情侣_女人女人中等肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_средний_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': {  # 👩🏽‍❤️‍👩🏿
        'en': ':couple_with_heart_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_中間の肌色_濃い肌色:',
        'ko': ':연인_여자_여자_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人女人中等肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_среднии_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': {  # 👩🏽‍❤‍👩🏿
        'en': ':couple_with_heart_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':カップルとハート_女性_女性_中間の肌色_濃い肌色:',
        'ko': ':연인_여자_여자_갈색_피부_검은색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_e_pele_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':情侣_女人女人中等肤色较深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_средний_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': {  # 👩🏽‍❤️‍👩🏻
        'en': ':couple_with_heart_woman_woman_medium_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_légèrement_mate_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_中間の肌色_薄い肌色:',
        'ko': ':연인_여자_여자_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_طلایی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人女人中等肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_среднии_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': {  # 👩🏽‍❤‍👩🏻
        'en': ':couple_with_heart_woman_woman_medium_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':couple_avec_cœur_femme_femme_peau_légèrement_mate_et_peau_claire:',
        'ja': ':カップルとハート_女性_女性_中間の肌色_薄い肌色:',
        'ko': ':연인_여자_여자_갈색_피부_하얀_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_e_pele_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_طلایی_و_پوست_سفید:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':情侣_女人女人中等肤色较浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_средний_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': {  # 👩🏽‍❤️‍👩🏾
        'en': ':couple_with_heart_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_légèrement_mate_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_中間の肌色_やや濃い肌色:',
        'ko': ':연인_여자_여자_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_طلایی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人中等肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_среднии_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': {  # 👩🏽‍❤‍👩🏾
        'en': ':couple_with_heart_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_légèrement_mate_et_peau_mate:',
        'ja': ':カップルとハート_女性_女性_中間の肌色_やや濃い肌色:',
        'ko': ':연인_여자_여자_갈색_피부_진한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_e_pele_morena_escura:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':زوج_عاشق_زن_زن_پوست_طلایی_و_پوست_گندمی:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':情侣_女人女人中等肤色中等深肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_средний_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': {  # 👩🏽‍❤️‍👩🏼
        'en': ':couple_with_heart_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_中間の肌色_やや薄い肌色:',
        'ko': ':연인_여자_여자_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_طلایی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人中等肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_среднии_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': {  # 👩🏽‍❤‍👩🏼
        'en': ':couple_with_heart_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':liebespaar_frau_frau_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':pareja_enamorada_mujer_mujer_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':couple_avec_cœur_femme_femme_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':カップルとハート_女性_女性_中間の肌色_やや薄い肌色:',
        'ko': ':연인_여자_여자_갈색_피부_연한_갈색_피부:',
        'pt': ':casal_apaixonado_mulher_mulher_pele_morena_e_pele_morena_clara:',
        'it': ':coppia_con_cuore_donna_donna_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':زوج_عاشق_زن_زن_پوست_طلایی_و_پوست_روشن:',
        'id': ':pasangan_dengan_hati_wanita_wanita_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':情侣_女人女人中等肤色中等浅肤色:',
        'ru': ':влюбленная_пара_женщина_женщина_средний_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F404': {  # 🐄
        'en': ':cow:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':cow2:'],
        'de': ':kuh:',
        'es': ':vaca:',
        'fr': ':vache:',
        'ja': ':牝牛:',
        'ko': ':젖소:',
        'pt': ':vaca:',
        'it': ':mucca:',
        'fa': ':گاو_ماده:',
        'id': ':sapi:',
        'zh': ':奶牛:',
        'ru': ':корова:'
    },
    '\U0001F42E': {  # 🐮
        'en': ':cow_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cow:'],
        'de': ':kuhgesicht:',
        'es': ':cara_de_vaca:',
        'fr': ':tête_de_vache:',
        'ja': ':牛の顔:',
        'ko': ':소_얼굴:',
        'pt': ':rosto_de_vaca:',
        'it': ':muso_di_mucca:',
        'fa': ':صورت_گاو:',
        'id': ':wajah_sapi:',
        'zh': ':奶牛头:',
        'ru': ':морда_коровы:'
    },
    '\U0001F920': {  # 🤠
        'en': ':cowboy_hat_face:',
        'status': fully_qualified,
        'E': 3,
        'de': ':gesicht_mit_cowboyhut:',
        'es': ':cara_con_sombrero_de_vaquero:',
        'fr': ':visage_avec_chapeau_de_cowboy:',
        'ja': ':カウボーイの顔:',
        'ko': ':카우보이_모자_쓴_얼굴:',
        'pt': ':rosto_com_chapéu_de_caubói:',
        'it': ':faccina_con_cappello_da_cowboy:',
        'fa': ':کلاه_کابوی:',
        'id': ':wajah_koboi:',
        'zh': ':牛仔帽脸:',
        'ru': ':в_ковбойской_шляпе:'
    },
    '\U0001F980': {  # 🦀
        'en': ':crab:',
        'status': fully_qualified,
        'E': 1,
        'de': ':krebs:',
        'es': ':cangrejo:',
        'fr': ':crabe:',
        'ja': ':カニ:',
        'ko': ':꽃게:',
        'pt': ':caranguejo:',
        'it': ':granchio:',
        'fa': ':خرچنگ:',
        'id': ':kepiting:',
        'zh': ':蟹:',
        'ru': ':краб:'
    },
    '\U0001F58D\U0000FE0F': {  # 🖍️
        'en': ':crayon:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':lower_left_crayon:'],
        'variant': True,
        'de': ':wachsmalstift:',
        'es': ':lápiz_de_cera:',
        'fr': ':crayon_pastel:',
        'ja': ':クレヨン:',
        'ko': ':크레용:',
        'pt': ':giz_de_cera:',
        'it': ':pastello_a_cera:',
        'fa': ':مدادشمعی:',
        'id': ':krayon:',
        'zh': ':蜡笔:',
        'ru': ':восковои_мелок:'
    },
    '\U0001F58D': {  # 🖍
        'en': ':crayon:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':lower_left_crayon:'],
        'variant': True,
        'de': ':wachsmalstift:',
        'es': ':lápiz_de_cera:',
        'fr': ':crayon_pastel:',
        'ja': ':クレヨン:',
        'ko': ':크레용:',
        'pt': ':giz_de_cera:',
        'it': ':pastello_a_cera:',
        'fa': ':مدادشمعی:',
        'id': ':krayon:',
        'zh': ':蜡笔:',
        'ru': ':восковой_мелок:'
    },
    '\U0001F4B3': {  # 💳
        'en': ':credit_card:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':kreditkarte:',
        'es': ':tarjeta_de_crédito:',
        'fr': ':carte_bancaire:',
        'ja': ':クレジットカード:',
        'ko': ':신용카드:',
        'pt': ':cartão_de_crédito:',
        'it': ':carta_di_credito:',
        'fa': ':کارت_اعتباری:',
        'id': ':kartu_kredit:',
        'zh': ':信用卡:',
        'ru': ':кредитная_карта:'
    },
    '\U0001F319': {  # 🌙
        'en': ':crescent_moon:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':mondsichel:',
        'es': ':luna:',
        'fr': ':croissant_de_lune:',
        'ja': ':三日月:',
        'ko': ':초승달:',
        'pt': ':lua_crescente:',
        'it': ':spicchio_di_luna:',
        'fa': ':هلال_ماه:',
        'id': ':bulan_sabit:',
        'zh': ':弯月:',
        'ru': ':полумесяц:'
    },
    '\U0001F997': {  # 🦗
        'en': ':cricket:',
        'status': fully_qualified,
        'E': 5,
        'de': ':grille:',
        'es': ':grillo:',
        'fr': ':criquet:',
        'ja': ':バッタ:',
        'ko': ':귀뚜라미:',
        'pt': ':grilo:',
        'it': ':grillo:',
        'fa': ':جیرجیرک:',
        'id': ':jangkrik:',
        'zh': ':蟋蟀:',
        'ru': ':сверчок:'
    },
    '\U0001F3CF': {  # 🏏
        'en': ':cricket_game:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':cricket_bat_and_ball:'],
        'de': ':kricket:',
        'es': ':críquet:',
        'fr': ':cricket:',
        'ja': ':クリケット:',
        'ko': ':크리켓:',
        'pt': ':críquete:',
        'it': ':cricket:',
        'fa': ':کریکت:',
        'id': ':kriket:',
        'zh': ':板球:',
        'ru': ':крикет:'
    },
    '\U0001F40A': {  # 🐊
        'en': ':crocodile:',
        'status': fully_qualified,
        'E': 1,
        'de': ':krokodil:',
        'es': ':cocodrilo:',
        'fr': ':crocodile:',
        'ja': ':ワニ:',
        'ko': ':악어:',
        'pt': ':crocodilo:',
        'it': ':coccodrillo:',
        'fa': ':تمساح:',
        'id': ':buaya:',
        'zh': ':鳄鱼:',
        'ru': ':крокодил:'
    },
    '\U0001F950': {  # 🥐
        'en': ':croissant:',
        'status': fully_qualified,
        'E': 3,
        'de': ':croissant:',
        'es': ':cruasán:',
        'fr': ':croissant:',
        'ja': ':クロワッサン:',
        'ko': ':크루아상:',
        'pt': ':croissant:',
        'it': ':croissant:',
        'fa': ':کرواسان:',
        'id': ':croissant:',
        'zh': ':羊角面包:',
        'ru': ':круассан:'
    },
    '\U0000274C': {  # ❌
        'en': ':cross_mark:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':x:'],
        'de': ':kreuzzeichen:',
        'es': ':marca_de_cruz:',
        'fr': ':croix:',
        'ja': ':バツ:',
        'ko': ':엑스_표시:',
        'pt': ':xis:',
        'it': ':croce:',
        'fa': ':ضربدر:',
        'id': ':tanda_silang:',
        'zh': ':叉号:',
        'ru': ':крестик:'
    },
    '\U0000274E': {  # ❎
        'en': ':cross_mark_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':negative_squared_cross_mark:'],
        'de': ':angekreuztes_kästchen:',
        'es': ':botón_con_marca_de_cruz:',
        'fr': ':bouton_croix:',
        'ja': ':白抜きのバツ:',
        'ko': ':초록_상자_안_엑스표:',
        'pt': ':botão_de_xis:',
        'it': ':croce_con_quadrato:',
        'fa': ':دکمه_ایکس:',
        'id': ':tombol_tanda_silang:',
        'zh': ':叉号按钮:',
        'ru': ':кнопка_«крестик»:'
    },
    '\U0001F91E': {  # 🤞
        'en': ':crossed_fingers:',
        'status': fully_qualified,
        'E': 3,
        'de': ':hand_mit_gekreuzten_fingern:',
        'es': ':dedos_cruzados:',
        'fr': ':doigts_croisés:',
        'ja': ':指をクロス:',
        'ko': ':검지와_중지_크로스:',
        'pt': ':dedos_cruzados:',
        'it': ':dita_incrociate:',
        'fa': ':انگشتان_گره_خورده:',
        'id': ':jari_menyilang:',
        'zh': ':交叉的手指:',
        'ru': ':скрещенные_пальцы:'
    },
    '\U0001F91E\U0001F3FF': {  # 🤞🏿
        'en': ':crossed_fingers_dark_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':hand_mit_gekreuzten_fingern_dunkle_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_oscuro:',
        'fr': ':doigts_croisés_peau_foncée:',
        'ja': ':指をクロス_濃い肌色:',
        'ko': ':검지와_중지_크로스_검은색_피부:',
        'pt': ':dedos_cruzados_pele_escura:',
        'it': ':dita_incrociate_carnagione_scura:',
        'fa': ':انگشتان_گره_خورده_پوست_آبنوسی:',
        'id': ':jari_menyilang_warna_kulit_gelap:',
        'zh': ':交叉的手指_较深肤色:',
        'ru': ':скрещенные_пальцы_очень_темный_тон_кожи:'
    },
    '\U0001F91E\U0001F3FB': {  # 🤞🏻
        'en': ':crossed_fingers_light_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':hand_mit_gekreuzten_fingern_helle_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_claro:',
        'fr': ':doigts_croisés_peau_claire:',
        'ja': ':指をクロス_薄い肌色:',
        'ko': ':검지와_중지_크로스_하얀_피부:',
        'pt': ':dedos_cruzados_pele_clara:',
        'it': ':dita_incrociate_carnagione_chiara:',
        'fa': ':انگشتان_گره_خورده_پوست_سفید:',
        'id': ':jari_menyilang_warna_kulit_cerah:',
        'zh': ':交叉的手指_较浅肤色:',
        'ru': ':скрещенные_пальцы_очень_светлый_тон_кожи:'
    },
    '\U0001F91E\U0001F3FE': {  # 🤞🏾
        'en': ':crossed_fingers_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':hand_mit_gekreuzten_fingern_mitteldunkle_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_oscuro_medio:',
        'fr': ':doigts_croisés_peau_mate:',
        'ja': ':指をクロス_やや濃い肌色:',
        'ko': ':검지와_중지_크로스_진한_갈색_피부:',
        'pt': ':dedos_cruzados_pele_morena_escura:',
        'it': ':dita_incrociate_carnagione_abbastanza_scura:',
        'fa': ':انگشتان_گره_خورده_پوست_گندمی:',
        'id': ':jari_menyilang_warna_kulit_gelap-sedang:',
        'zh': ':交叉的手指_中等深肤色:',
        'ru': ':скрещенные_пальцы_темный_тон_кожи:'
    },
    '\U0001F91E\U0001F3FC': {  # 🤞🏼
        'en': ':crossed_fingers_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':hand_mit_gekreuzten_fingern_mittelhelle_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_claro_medio:',
        'fr': ':doigts_croisés_peau_moyennement_claire:',
        'ja': ':指をクロス_やや薄い肌色:',
        'ko': ':검지와_중지_크로스_연한_갈색_피부:',
        'pt': ':dedos_cruzados_pele_morena_clara:',
        'it': ':dita_incrociate_carnagione_abbastanza_chiara:',
        'fa': ':انگشتان_گره_خورده_پوست_روشن:',
        'id': ':jari_menyilang_warna_kulit_cerah-sedang:',
        'zh': ':交叉的手指_中等浅肤色:',
        'ru': ':скрещенные_пальцы_светлый_тон_кожи:'
    },
    '\U0001F91E\U0001F3FD': {  # 🤞🏽
        'en': ':crossed_fingers_medium_skin_tone:',
        'status': fully_qualified,
        'E': 3,
        'de': ':hand_mit_gekreuzten_fingern_mittlere_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_medio:',
        'fr': ':doigts_croisés_peau_légèrement_mate:',
        'ja': ':指をクロス_中間の肌色:',
        'ko': ':검지와_중지_크로스_갈색_피부:',
        'pt': ':dedos_cruzados_pele_morena:',
        'it': ':dita_incrociate_carnagione_olivastra:',
        'fa': ':انگشتان_گره_خورده_پوست_طلایی:',
        'id': ':jari_menyilang_warna_kulit_sedang:',
        'zh': ':交叉的手指_中等肤色:',
        'ru': ':скрещенные_пальцы_средний_тон_кожи:'
    },
    '\U0001F38C': {  # 🎌
        'en': ':crossed_flags:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':überkreuzte_flaggen:',
        'es': ':banderas_cruzadas:',
        'fr': ':drapeaux_croisés:',
        'ja': ':祝日の旗:',
        'ko': ':엇갈린_일본_국기:',
        'pt': ':bandeiras_cruzadas:',
        'it': ':bandiere_del_giappone_incrociate:',
        'fa': ':پرچم_های_ضرب_دری:',
        'id': ':bendera_menyilang:',
        'zh': ':交叉旗:',
        'ru': ':скрещенные_флаги:'
    },
    '\U00002694\U0000FE0F': {  # ⚔️
        'en': ':crossed_swords:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':gekreuzte_schwerter:',
        'es': ':espadas_cruzadas:',
        'fr': ':épées_croisées:',
        'ja': ':クロスした剣:',
        'ko': ':쌍검:',
        'pt': ':espadas_cruzadas:',
        'it': ':spade_incrociate:',
        'fa': ':شمشیرهای_ضرب_دری:',
        'id': ':pedang_bersilang:',
        'zh': ':交叉放置的剑:',
        'ru': ':скрещенные_мечи:'
    },
    '\U00002694': {  # ⚔
        'en': ':crossed_swords:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':gekreuzte_schwerter:',
        'es': ':espadas_cruzadas:',
        'fr': ':épées_croisées:',
        'ja': ':クロスした剣:',
        'ko': ':쌍검:',
        'pt': ':espadas_cruzadas:',
        'it': ':spade_incrociate:',
        'fa': ':شمشیرهای_ضرب_دری:',
        'id': ':pedang_bersilang:',
        'zh': ':交叉放置的剑:',
        'ru': ':скрещенные_мечи:'
    },
    '\U0001F451': {  # 👑
        'en': ':crown:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':krone:',
        'es': ':corona:',
        'fr': ':couronne:',
        'ja': ':王冠:',
        'ko': ':왕관:',
        'pt': ':coroa:',
        'it': ':corona:',
        'fa': ':تاج:',
        'id': ':mahkota:',
        'zh': ':皇冠:',
        'ru': ':корона:'
    },
    '\U0001FA7C': {  # 🩼
        'en': ':crutch:',
        'status': fully_qualified,
        'E': 14,
        'de': ':krücke:',
        'es': ':muleta:',
        'fr': ':béquille:',
        'ja': ':松葉杖:',
        'ko': ':목발:',
        'pt': ':muleta:',
        'it': ':stampella:',
        'fa': ':عصای_زیر_بغل:',
        'id': ':kruk:',
        'zh': ':拐杖:',
        'ru': ':костыль:'
    },
    '\U0001F63F': {  # 😿
        'en': ':crying_cat:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':crying_cat_face:'],
        'de': ':weinende_katze:',
        'es': ':gato_llorando:',
        'fr': ':chat_qui_pleure:',
        'ja': ':泣いている猫:',
        'ko': ':울고_있는_고양이_얼굴:',
        'pt': ':rosto_de_gato_chorando:',
        'it': ':gatto_che_piange:',
        'fa': ':گربهٔ_گریان:',
        'id': ':wajah_kucing_menangis:',
        'zh': ':哭泣的猫:',
        'ru': ':плачущий_кот:'
    },
    '\U0001F622': {  # 😢
        'en': ':crying_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cry:'],
        'de': ':weinendes_gesicht:',
        'es': ':cara_llorando:',
        'fr': ':visage_qui_pleure:',
        'ja': ':泣き顔:',
        'ko': ':우는_얼굴:',
        'pt': ':rosto_chorando:',
        'it': ':faccina_che_piange:',
        'fa': ':گریان:',
        'id': ':wajah_menangis:',
        'zh': ':哭:',
        'ru': ':плачет:'
    },
    '\U0001F52E': {  # 🔮
        'en': ':crystal_ball:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kristallkugel:',
        'es': ':bola_de_cristal:',
        'fr': ':boule_de_cristal:',
        'ja': ':水晶玉:',
        'ko': ':수정_구슬:',
        'pt': ':bola_de_cristal:',
        'it': ':sfera_di_cristallo:',
        'fa': ':گوی:',
        'id': ':bola_kristal:',
        'zh': ':水晶球:',
        'ru': ':хрустальный_шар:'
    },
    '\U0001F952': {  # 🥒
        'en': ':cucumber:',
        'status': fully_qualified,
        'E': 3,
        'de': ':gurke:',
        'es': ':pepino:',
        'fr': ':concombre:',
        'ja': ':キュウリ:',
        'ko': ':오이:',
        'pt': ':pepino:',
        'it': ':cetriolo:',
        'fa': ':خیار:',
        'id': ':ketimun:',
        'zh': ':黄瓜:',
        'ru': ':огурец:'
    },
    '\U0001F964': {  # 🥤
        'en': ':cup_with_straw:',
        'status': fully_qualified,
        'E': 5,
        'de': ':becher_mit_strohhalm:',
        'es': ':vaso_con_pajita:',
        'fr': ':gobelet_avec_paille:',
        'ja': ':ストローカップ:',
        'ko': ':빨대와_컵:',
        'pt': ':copo_com_canudo:',
        'it': ':bicchiere_con_cannuccia:',
        'fa': ':فنجان_با_نی:',
        'id': ':cangkir_dengan_sedotan:',
        'zh': ':带吸管杯:',
        'ru': ':стакан_с_соломинкой:'
    },
    '\U0001F9C1': {  # 🧁
        'en': ':cupcake:',
        'status': fully_qualified,
        'E': 11,
        'de': ':cupcake:',
        'es': ':magdalena:',
        'fr': ':cupcake:',
        'ja': ':カップケーキ:',
        'ko': ':컵케이크:',
        'pt': ':cupcake:',
        'it': ':cupcake:',
        'fa': ':کیک_فنجانی:',
        'id': ':kue_mangkuk:',
        'zh': ':纸杯蛋糕:',
        'ru': ':капкейк:'
    },
    '\U0001F94C': {  # 🥌
        'en': ':curling_stone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':curlingstein:',
        'es': ':piedra_de_curling:',
        'fr': ':pierre_de_curling:',
        'ja': ':カーリング:',
        'ko': ':컬링_스톤:',
        'pt': ':pedra_de_curling:',
        'it': ':stone_da_curling:',
        'fa': ':سنگ_کرلینگ:',
        'id': ':batu_curling:',
        'zh': ':冰壶:',
        'ru': ':керлинг:'
    },
    '\U0001F9B1': {  # 🦱
        'en': ':curly_hair:',
        'status': component,
        'E': 11,
        'de': ':lockiges_haar:',
        'es': ':pelo_rizado:',
        'fr': ':cheveux_bouclés:',
        'ja': ':巻き毛:',
        'ko': ':곱슬머리:',
        'pt': ':cabelo_cacheado:',
        'it': ':capelli_ricci:',
        'fa': ':موفرفری:',
        'id': ':rambut_keriting:',
        'zh': ':卷发:',
        'ru': ':кудрявые_волосы:'
    },
    '\U000027B0': {  # ➰
        'en': ':curly_loop:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':schleife:',
        'es': ':bucle:',
        'fr': ':boucle:',
        'ja': ':ループ:',
        'ko': ':고리:',
        'pt': ':laço_encaracolado:',
        'it': ':occhiello:',
        'fa': ':حلقه_مجعد:',
        'id': ':untaian_bergelombang:',
        'zh': ':卷曲环:',
        'ru': ':петля:'
    },
    '\U0001F4B1': {  # 💱
        'en': ':currency_exchange:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':geldwechsel:',
        'es': ':cambio_de_divisas:',
        'fr': ':conversion_de_devise:',
        'ja': ':外貨両替:',
        'ko': ':환전:',
        'pt': ':câmbio_de_moeda:',
        'it': ':cambio_valuta:',
        'fa': ':تبدیل_ارز:',
        'id': ':kurs:',
        'zh': ':货币兑换:',
        'ru': ':обмен_валюты:'
    },
    '\U0001F35B': {  # 🍛
        'en': ':curry_rice:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':curry:'],
        'de': ':reis_mit_curry:',
        'es': ':arroz_con_curry:',
        'fr': ':riz_au_curry:',
        'ja': ':カレーライス:',
        'ko': ':카레라이스:',
        'pt': ':arroz_com_curry:',
        'it': ':riso_al_curry:',
        'fa': ':برنج_با_کاری:',
        'id': ':nasi_kari:',
        'zh': ':咖喱饭:',
        'ru': ':рис_с_карри:'
    },
    '\U0001F36E': {  # 🍮
        'en': ':custard:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':pudding:',
        'es': ':flan:',
        'fr': ':crème_renversée:',
        'ja': ':プリン:',
        'ko': ':커스타드_푸딩:',
        'pt': ':pudim:',
        'it': ':budino:',
        'fa': ':کاستارد:',
        'id': ':puding_susu:',
        'zh': ':奶黄:',
        'ru': ':пудинг:'
    },
    '\U0001F6C3': {  # 🛃
        'en': ':customs:',
        'status': fully_qualified,
        'E': 1,
        'de': ':zollkontrolle:',
        'es': ':aduana:',
        'fr': ':douane:',
        'ja': ':税関:',
        'ko': ':세관:',
        'pt': ':alfândega:',
        'it': ':simbolo_della_dogana:',
        'fa': ':گمرک:',
        'id': ':bea_cukai:',
        'zh': ':海关:',
        'ru': ':таможня:'
    },
    '\U0001F969': {  # 🥩
        'en': ':cut_of_meat:',
        'status': fully_qualified,
        'E': 5,
        'de': ':fleischstück:',
        'es': ':corte_de_carne:',
        'fr': ':morceau_de_viande:',
        'ja': ':ステーキ肉:',
        'ko': ':고깃덩이:',
        'pt': ':corte_de_carne:',
        'it': ':taglio_di_carne:',
        'fa': ':برش_گوشت:',
        'id': ':potongan_daging:',
        'zh': ':肉块:',
        'ru': ':кусок_мяса:'
    },
    '\U0001F300': {  # 🌀
        'en': ':cyclone:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':wirbel:',
        'es': ':ciclón:',
        'fr': ':cyclone:',
        'ja': ':渦巻き:',
        'ko': ':태풍:',
        'pt': ':ciclone:',
        'it': ':ciclone:',
        'fa': ':چرخند:',
        'id': ':topan:',
        'zh': ':台风:',
        'ru': ':циклон:'
    },
    '\U0001F5E1\U0000FE0F': {  # 🗡️
        'en': ':dagger:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':dagger_knife:'],
        'variant': True,
        'de': ':dolch:',
        'es': ':puñal:',
        'fr': ':dague:',
        'ja': ':短刀:',
        'ko': ':단검:',
        'pt': ':adaga:',
        'it': ':pugnale:',
        'fa': ':خنجر:',
        'id': ':belati:',
        'zh': ':匕首:',
        'ru': ':кинжал:'
    },
    '\U0001F5E1': {  # 🗡
        'en': ':dagger:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':dagger_knife:'],
        'variant': True,
        'de': ':dolch:',
        'es': ':puñal:',
        'fr': ':dague:',
        'ja': ':短刀:',
        'ko': ':단검:',
        'pt': ':adaga:',
        'it': ':pugnale:',
        'fa': ':خنجر:',
        'id': ':belati:',
        'zh': ':匕首:',
        'ru': ':кинжал:'
    },
    '\U0001F361': {  # 🍡
        'en': ':dango:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':dango:',
        'es': ':dango:',
        'fr': ':brochette_de_bonbons:',
        'ja': ':だんご:',
        'ko': ':떡꼬치:',
        'pt': ':dango:',
        'it': ':dango:',
        'fa': ':دانگو:',
        'id': ':dango:',
        'zh': ':团子:',
        'ru': ':данго:'
    },
    '\U0001F3FF': {  # 🏿
        'en': ':dark_skin_tone:',
        'status': component,
        'E': 1,
        'alias': [':emoji_modifier_fitzpatrick_type_6:'],
        'de': ':dunkle_hautfarbe:',
        'es': ':tono_de_piel_oscuro:',
        'fr': ':peau_foncée:',
        'ja': ':濃い肌色:',
        'ko': ':검은색_피부:',
        'pt': ':pele_escura:',
        'it': ':carnagione_scura:',
        'fa': ':پوست_آبنوسی:',
        'id': ':warna_kulit_gelap:',
        'zh': ':较深肤色:',
        'ru': ':очень_темный_тон_кожи:'
    },
    '\U0001F4A8': {  # 💨
        'en': ':dashing_away:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':dash:'],
        'de': ':staubwolke:',
        'es': ':salir_corriendo:',
        'fr': ':décamper:',
        'ja': ':ダッシュ:',
        'ko': ':달려나감:',
        'pt': ':rapidez:',
        'it': ':nuvola_di_polvere:',
        'fa': ':به_سرعت_باد:',
        'id': ':berlari:',
        'zh': ':尾气:',
        'ru': ':исчезновение:'
    },
    '\U0001F9CF\U0000200D\U00002642\U0000FE0F': {  # 🧏‍♂️
        'en': ':deaf_man:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörloser_mann:',
        'es': ':hombre_sordo:',
        'fr': ':homme_sourd:',
        'ja': ':耳の不自由な男性:',
        'ko': ':청각장애가_있는_남자:',
        'pt': ':homem_surdo:',
        'it': ':uomo_con_problemi_di_udito:',
        'fa': ':مرد_ناشنوا:',
        'id': ':pria_tuli:',
        'zh': ':失聪的男人:',
        'ru': ':глухои_мужчина:'
    },
    '\U0001F9CF\U0000200D\U00002642': {  # 🧏‍♂
        'en': ':deaf_man:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörloser_mann:',
        'es': ':hombre_sordo:',
        'fr': ':homme_sourd:',
        'ja': ':耳の不自由な男性:',
        'ko': ':청각장애가_있는_남자:',
        'pt': ':homem_surdo:',
        'it': ':uomo_con_problemi_di_udito:',
        'fa': ':مرد_ناشنوا:',
        'id': ':pria_tuli:',
        'zh': ':失聪的男人:',
        'ru': ':глухой_мужчина:'
    },
    '\U0001F9CF\U0001F3FF\U0000200D\U00002642\U0000FE0F': {  # 🧏🏿‍♂️
        'en': ':deaf_man_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörloser_mann_dunkle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_oscuro:',
        'fr': ':homme_sourd_peau_foncée:',
        'ja': ':耳の不自由な男性_濃い肌色:',
        'ko': ':청각장애가_있는_남자_검은색_피부:',
        'pt': ':homem_surdo_pele_escura:',
        'it': ':uomo_con_problemi_di_udito_carnagione_scura:',
        'fa': ':مرد_ناشنوا_پوست_آبنوسی:',
        'id': ':pria_tuli_warna_kulit_gelap:',
        'zh': ':失聪的男人_较深肤色:',
        'ru': ':глухои_мужчина_очень_темныи_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FF\U0000200D\U00002642': {  # 🧏🏿‍♂
        'en': ':deaf_man_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörloser_mann_dunkle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_oscuro:',
        'fr': ':homme_sourd_peau_foncée:',
        'ja': ':耳の不自由な男性_濃い肌色:',
        'ko': ':청각장애가_있는_남자_검은색_피부:',
        'pt': ':homem_surdo_pele_escura:',
        'it': ':uomo_con_problemi_di_udito_carnagione_scura:',
        'fa': ':مرد_ناشنوا_پوست_آبنوسی:',
        'id': ':pria_tuli_warna_kulit_gelap:',
        'zh': ':失聪的男人_较深肤色:',
        'ru': ':глухой_мужчина_очень_темный_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FB\U0000200D\U00002642\U0000FE0F': {  # 🧏🏻‍♂️
        'en': ':deaf_man_light_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörloser_mann_helle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_claro:',
        'fr': ':homme_sourd_peau_claire:',
        'ja': ':耳の不自由な男性_薄い肌色:',
        'ko': ':청각장애가_있는_남자_하얀_피부:',
        'pt': ':homem_surdo_pele_clara:',
        'it': ':uomo_con_problemi_di_udito_carnagione_chiara:',
        'fa': ':مرد_ناشنوا_پوست_سفید:',
        'id': ':pria_tuli_warna_kulit_cerah:',
        'zh': ':失聪的男人_较浅肤色:',
        'ru': ':глухои_мужчина_очень_светлыи_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FB\U0000200D\U00002642': {  # 🧏🏻‍♂
        'en': ':deaf_man_light_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörloser_mann_helle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_claro:',
        'fr': ':homme_sourd_peau_claire:',
        'ja': ':耳の不自由な男性_薄い肌色:',
        'ko': ':청각장애가_있는_남자_하얀_피부:',
        'pt': ':homem_surdo_pele_clara:',
        'it': ':uomo_con_problemi_di_udito_carnagione_chiara:',
        'fa': ':مرد_ناشنوا_پوست_سفید:',
        'id': ':pria_tuli_warna_kulit_cerah:',
        'zh': ':失聪的男人_较浅肤色:',
        'ru': ':глухой_мужчина_очень_светлый_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FE\U0000200D\U00002642\U0000FE0F': {  # 🧏🏾‍♂️
        'en': ':deaf_man_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörloser_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_oscuro_medio:',
        'fr': ':homme_sourd_peau_mate:',
        'ja': ':耳の不自由な男性_やや濃い肌色:',
        'ko': ':청각장애가_있는_남자_진한_갈색_피부:',
        'pt': ':homem_surdo_pele_morena_escura:',
        'it': ':uomo_con_problemi_di_udito_carnagione_abbastanza_scura:',
        'fa': ':مرد_ناشنوا_پوست_گندمی:',
        'id': ':pria_tuli_warna_kulit_gelap-sedang:',
        'zh': ':失聪的男人_中等深肤色:',
        'ru': ':глухои_мужчина_темныи_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FE\U0000200D\U00002642': {  # 🧏🏾‍♂
        'en': ':deaf_man_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörloser_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_oscuro_medio:',
        'fr': ':homme_sourd_peau_mate:',
        'ja': ':耳の不自由な男性_やや濃い肌色:',
        'ko': ':청각장애가_있는_남자_진한_갈색_피부:',
        'pt': ':homem_surdo_pele_morena_escura:',
        'it': ':uomo_con_problemi_di_udito_carnagione_abbastanza_scura:',
        'fa': ':مرد_ناشنوا_پوست_گندمی:',
        'id': ':pria_tuli_warna_kulit_gelap-sedang:',
        'zh': ':失聪的男人_中等深肤色:',
        'ru': ':глухой_мужчина_темный_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FC\U0000200D\U00002642\U0000FE0F': {  # 🧏🏼‍♂️
        'en': ':deaf_man_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörloser_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_claro_medio:',
        'fr': ':homme_sourd_peau_moyennement_claire:',
        'ja': ':耳の不自由な男性_やや薄い肌色:',
        'ko': ':청각장애가_있는_남자_연한_갈색_피부:',
        'pt': ':homem_surdo_pele_morena_clara:',
        'it': ':uomo_con_problemi_di_udito_carnagione_abbastanza_chiara:',
        'fa': ':مرد_ناشنوا_پوست_روشن:',
        'id': ':pria_tuli_warna_kulit_cerah-sedang:',
        'zh': ':失聪的男人_中等浅肤色:',
        'ru': ':глухои_мужчина_светлыи_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FC\U0000200D\U00002642': {  # 🧏🏼‍♂
        'en': ':deaf_man_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörloser_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_claro_medio:',
        'fr': ':homme_sourd_peau_moyennement_claire:',
        'ja': ':耳の不自由な男性_やや薄い肌色:',
        'ko': ':청각장애가_있는_남자_연한_갈색_피부:',
        'pt': ':homem_surdo_pele_morena_clara:',
        'it': ':uomo_con_problemi_di_udito_carnagione_abbastanza_chiara:',
        'fa': ':مرد_ناشنوا_پوست_روشن:',
        'id': ':pria_tuli_warna_kulit_cerah-sedang:',
        'zh': ':失聪的男人_中等浅肤色:',
        'ru': ':глухой_мужчина_светлый_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FD\U0000200D\U00002642\U0000FE0F': {  # 🧏🏽‍♂️
        'en': ':deaf_man_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörloser_mann_mittlere_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_medio:',
        'fr': ':homme_sourd_peau_légèrement_mate:',
        'ja': ':耳の不自由な男性_中間の肌色:',
        'ko': ':청각장애가_있는_남자_갈색_피부:',
        'pt': ':homem_surdo_pele_morena:',
        'it': ':uomo_con_problemi_di_udito_carnagione_olivastra:',
        'fa': ':مرد_ناشنوا_پوست_طلایی:',
        'id': ':pria_tuli_warna_kulit_sedang:',
        'zh': ':失聪的男人_中等肤色:',
        'ru': ':глухои_мужчина_среднии_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FD\U0000200D\U00002642': {  # 🧏🏽‍♂
        'en': ':deaf_man_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörloser_mann_mittlere_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_medio:',
        'fr': ':homme_sourd_peau_légèrement_mate:',
        'ja': ':耳の不自由な男性_中間の肌色:',
        'ko': ':청각장애가_있는_남자_갈색_피부:',
        'pt': ':homem_surdo_pele_morena:',
        'it': ':uomo_con_problemi_di_udito_carnagione_olivastra:',
        'fa': ':مرد_ناشنوا_پوست_طلایی:',
        'id': ':pria_tuli_warna_kulit_sedang:',
        'zh': ':失聪的男人_中等肤色:',
        'ru': ':глухой_мужчина_средний_тон_кожи:'
    },
    '\U0001F9CF': {  # 🧏
        'en': ':deaf_person:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_person:',
        'es': ':persona_sorda:',
        'fr': ':personne_sourde:',
        'ja': ':耳の不自由な人:',
        'ko': ':청각_장애가_있는_사람:',
        'pt': ':pessoa_surda:',
        'it': ':persona_con_problemi_di_udito:',
        'fa': ':آدم_ناشنوا:',
        'id': ':orang_tuli:',
        'zh': ':失聪者:',
        'ru': ':глухой_человек:'
    },
    '\U0001F9CF\U0001F3FF': {  # 🧏🏿
        'en': ':deaf_person_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_person_dunkle_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_oscuro:',
        'fr': ':personne_sourde_peau_foncée:',
        'ja': ':耳の不自由な人_濃い肌色:',
        'ko': ':청각_장애가_있는_사람_검은색_피부:',
        'pt': ':pessoa_surda_pele_escura:',
        'it': ':persona_con_problemi_di_udito_carnagione_scura:',
        'fa': ':آدم_ناشنوا_پوست_آبنوسی:',
        'id': ':orang_tuli_warna_kulit_gelap:',
        'zh': ':失聪者_较深肤色:',
        'ru': ':глухой_человек_очень_темный_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FB': {  # 🧏🏻
        'en': ':deaf_person_light_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_person_helle_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_claro:',
        'fr': ':personne_sourde_peau_claire:',
        'ja': ':耳の不自由な人_薄い肌色:',
        'ko': ':청각_장애가_있는_사람_하얀_피부:',
        'pt': ':pessoa_surda_pele_clara:',
        'it': ':persona_con_problemi_di_udito_carnagione_chiara:',
        'fa': ':آدم_ناشنوا_پوست_سفید:',
        'id': ':orang_tuli_warna_kulit_cerah:',
        'zh': ':失聪者_较浅肤色:',
        'ru': ':глухой_человек_очень_светлый_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FE': {  # 🧏🏾
        'en': ':deaf_person_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_person_mitteldunkle_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_oscuro_medio:',
        'fr': ':personne_sourde_peau_mate:',
        'ja': ':耳の不自由な人_やや濃い肌色:',
        'ko': ':청각_장애가_있는_사람_진한_갈색_피부:',
        'pt': ':pessoa_surda_pele_morena_escura:',
        'it': ':persona_con_problemi_di_udito_carnagione_abbastanza_scura:',
        'fa': ':آدم_ناشنوا_پوست_گندمی:',
        'id': ':orang_tuli_warna_kulit_gelap-sedang:',
        'zh': ':失聪者_中等深肤色:',
        'ru': ':глухой_человек_темный_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FC': {  # 🧏🏼
        'en': ':deaf_person_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_person_mittelhelle_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_claro_medio:',
        'fr': ':personne_sourde_peau_moyennement_claire:',
        'ja': ':耳の不自由な人_やや薄い肌色:',
        'ko': ':청각_장애가_있는_사람_연한_갈색_피부:',
        'pt': ':pessoa_surda_pele_morena_clara:',
        'it': ':persona_con_problemi_di_udito_carnagione_abbastanza_chiara:',
        'fa': ':آدم_ناشنوا_پوست_روشن:',
        'id': ':orang_tuli_warna_kulit_cerah-sedang:',
        'zh': ':失聪者_中等浅肤色:',
        'ru': ':глухой_человек_светлый_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FD': {  # 🧏🏽
        'en': ':deaf_person_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_person_mittlere_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_medio:',
        'fr': ':personne_sourde_peau_légèrement_mate:',
        'ja': ':耳の不自由な人_中間の肌色:',
        'ko': ':청각_장애가_있는_사람_갈색_피부:',
        'pt': ':pessoa_surda_pele_morena:',
        'it': ':persona_con_problemi_di_udito_carnagione_olivastra:',
        'fa': ':آدم_ناشنوا_پوست_طلایی:',
        'id': ':orang_tuli_warna_kulit_sedang:',
        'zh': ':失聪者_中等肤色:',
        'ru': ':глухой_человек_средний_тон_кожи:'
    },
    '\U0001F9CF\U0000200D\U00002640\U0000FE0F': {  # 🧏‍♀️
        'en': ':deaf_woman:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_frau:',
        'es': ':mujer_sorda:',
        'fr': ':femme_sourde:',
        'ja': ':耳の不自由な女性:',
        'ko': ':청각장애가_있는_여자:',
        'pt': ':mulher_surda:',
        'it': ':donna_con_problemi_di_udito:',
        'fa': ':زن_ناشنوا:',
        'id': ':wanita_tuli:',
        'zh': ':失聪的女人:',
        'ru': ':глухая_женщина:'
    },
    '\U0001F9CF\U0000200D\U00002640': {  # 🧏‍♀
        'en': ':deaf_woman:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörlose_frau:',
        'es': ':mujer_sorda:',
        'fr': ':femme_sourde:',
        'ja': ':耳の不自由な女性:',
        'ko': ':청각장애가_있는_여자:',
        'pt': ':mulher_surda:',
        'it': ':donna_con_problemi_di_udito:',
        'fa': ':زن_ناشنوا:',
        'id': ':wanita_tuli:',
        'zh': ':失聪的女人:',
        'ru': ':глухая_женщина:'
    },
    '\U0001F9CF\U0001F3FF\U0000200D\U00002640\U0000FE0F': {  # 🧏🏿‍♀️
        'en': ':deaf_woman_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_frau_dunkle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_oscuro:',
        'fr': ':femme_sourde_peau_foncée:',
        'ja': ':耳の不自由な女性_濃い肌色:',
        'ko': ':청각장애가_있는_여자_검은색_피부:',
        'pt': ':mulher_surda_pele_escura:',
        'it': ':donna_con_problemi_di_udito_carnagione_scura:',
        'fa': ':زن_ناشنوا_پوست_آبنوسی:',
        'id': ':wanita_tuli_warna_kulit_gelap:',
        'zh': ':失聪的女人_较深肤色:',
        'ru': ':глухая_женщина_очень_темныи_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FF\U0000200D\U00002640': {  # 🧏🏿‍♀
        'en': ':deaf_woman_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörlose_frau_dunkle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_oscuro:',
        'fr': ':femme_sourde_peau_foncée:',
        'ja': ':耳の不自由な女性_濃い肌色:',
        'ko': ':청각장애가_있는_여자_검은색_피부:',
        'pt': ':mulher_surda_pele_escura:',
        'it': ':donna_con_problemi_di_udito_carnagione_scura:',
        'fa': ':زن_ناشنوا_پوست_آبنوسی:',
        'id': ':wanita_tuli_warna_kulit_gelap:',
        'zh': ':失聪的女人_较深肤色:',
        'ru': ':глухая_женщина_очень_темный_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FB\U0000200D\U00002640\U0000FE0F': {  # 🧏🏻‍♀️
        'en': ':deaf_woman_light_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_frau_helle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_claro:',
        'fr': ':femme_sourde_peau_claire:',
        'ja': ':耳の不自由な女性_薄い肌色:',
        'ko': ':청각장애가_있는_여자_하얀_피부:',
        'pt': ':mulher_surda_pele_clara:',
        'it': ':donna_con_problemi_di_udito_carnagione_chiara:',
        'fa': ':زن_ناشنوا_پوست_سفید:',
        'id': ':wanita_tuli_warna_kulit_cerah:',
        'zh': ':失聪的女人_较浅肤色:',
        'ru': ':глухая_женщина_очень_светлыи_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FB\U0000200D\U00002640': {  # 🧏🏻‍♀
        'en': ':deaf_woman_light_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörlose_frau_helle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_claro:',
        'fr': ':femme_sourde_peau_claire:',
        'ja': ':耳の不自由な女性_薄い肌色:',
        'ko': ':청각장애가_있는_여자_하얀_피부:',
        'pt': ':mulher_surda_pele_clara:',
        'it': ':donna_con_problemi_di_udito_carnagione_chiara:',
        'fa': ':زن_ناشنوا_پوست_سفید:',
        'id': ':wanita_tuli_warna_kulit_cerah:',
        'zh': ':失聪的女人_较浅肤色:',
        'ru': ':глухая_женщина_очень_светлый_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FE\U0000200D\U00002640\U0000FE0F': {  # 🧏🏾‍♀️
        'en': ':deaf_woman_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_frau_mitteldunkle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_oscuro_medio:',
        'fr': ':femme_sourde_peau_mate:',
        'ja': ':耳の不自由な女性_やや濃い肌色:',
        'ko': ':청각장애가_있는_여자_진한_갈색_피부:',
        'pt': ':mulher_surda_pele_morena_escura:',
        'it': ':donna_con_problemi_di_udito_carnagione_abbastanza_scura:',
        'fa': ':زن_ناشنوا_پوست_گندمی:',
        'id': ':wanita_tuli_warna_kulit_gelap-sedang:',
        'zh': ':失聪的女人_中等深肤色:',
        'ru': ':глухая_женщина_темныи_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FE\U0000200D\U00002640': {  # 🧏🏾‍♀
        'en': ':deaf_woman_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörlose_frau_mitteldunkle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_oscuro_medio:',
        'fr': ':femme_sourde_peau_mate:',
        'ja': ':耳の不自由な女性_やや濃い肌色:',
        'ko': ':청각장애가_있는_여자_진한_갈색_피부:',
        'pt': ':mulher_surda_pele_morena_escura:',
        'it': ':donna_con_problemi_di_udito_carnagione_abbastanza_scura:',
        'fa': ':زن_ناشنوا_پوست_گندمی:',
        'id': ':wanita_tuli_warna_kulit_gelap-sedang:',
        'zh': ':失聪的女人_中等深肤色:',
        'ru': ':глухая_женщина_темный_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FC\U0000200D\U00002640\U0000FE0F': {  # 🧏🏼‍♀️
        'en': ':deaf_woman_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_frau_mittelhelle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_claro_medio:',
        'fr': ':femme_sourde_peau_moyennement_claire:',
        'ja': ':耳の不自由な女性_やや薄い肌色:',
        'ko': ':청각장애가_있는_여자_연한_갈색_피부:',
        'pt': ':mulher_surda_pele_morena_clara:',
        'it': ':donna_con_problemi_di_udito_carnagione_abbastanza_chiara:',
        'fa': ':زن_ناشنوا_پوست_روشن:',
        'id': ':wanita_tuli_warna_kulit_cerah-sedang:',
        'zh': ':失聪的女人_中等浅肤色:',
        'ru': ':глухая_женщина_светлыи_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FC\U0000200D\U00002640': {  # 🧏🏼‍♀
        'en': ':deaf_woman_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörlose_frau_mittelhelle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_claro_medio:',
        'fr': ':femme_sourde_peau_moyennement_claire:',
        'ja': ':耳の不自由な女性_やや薄い肌色:',
        'ko': ':청각장애가_있는_여자_연한_갈색_피부:',
        'pt': ':mulher_surda_pele_morena_clara:',
        'it': ':donna_con_problemi_di_udito_carnagione_abbastanza_chiara:',
        'fa': ':زن_ناشنوا_پوست_روشن:',
        'id': ':wanita_tuli_warna_kulit_cerah-sedang:',
        'zh': ':失聪的女人_中等浅肤色:',
        'ru': ':глухая_женщина_светлый_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FD\U0000200D\U00002640\U0000FE0F': {  # 🧏🏽‍♀️
        'en': ':deaf_woman_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':gehörlose_frau_mittlere_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_medio:',
        'fr': ':femme_sourde_peau_légèrement_mate:',
        'ja': ':耳の不自由な女性_中間の肌色:',
        'ko': ':청각장애가_있는_여자_갈색_피부:',
        'pt': ':mulher_surda_pele_morena:',
        'it': ':donna_con_problemi_di_udito_carnagione_olivastra:',
        'fa': ':زن_ناشنوا_پوست_طلایی:',
        'id': ':wanita_tuli_warna_kulit_sedang:',
        'zh': ':失聪的女人_中等肤色:',
        'ru': ':глухая_женщина_среднии_тон_кожи:'
    },
    '\U0001F9CF\U0001F3FD\U0000200D\U00002640': {  # 🧏🏽‍♀
        'en': ':deaf_woman_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 12,
        'de': ':gehörlose_frau_mittlere_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_medio:',
        'fr': ':femme_sourde_peau_légèrement_mate:',
        'ja': ':耳の不自由な女性_中間の肌色:',
        'ko': ':청각장애가_있는_여자_갈색_피부:',
        'pt': ':mulher_surda_pele_morena:',
        'it': ':donna_con_problemi_di_udito_carnagione_olivastra:',
        'fa': ':زن_ناشنوا_پوست_طلایی:',
        'id': ':wanita_tuli_warna_kulit_sedang:',
        'zh': ':失聪的女人_中等肤色:',
        'ru': ':глухая_женщина_средний_тон_кожи:'
    },
    '\U0001F333': {  # 🌳
        'en': ':deciduous_tree:',
        'status': fully_qualified,
        'E': 1,
        'de': ':laubbaum:',
        'es': ':árbol_de_hoja_caduca:',
        'fr': ':arbre_à_feuilles_caduques:',
        'ja': ':落葉樹:',
        'ko': ':나무:',
        'pt': ':árvore_caidiça:',
        'it': ':albero_deciduo:',
        'fa': ':درخت_سبز:',
        'id': ':pohon_meranggas:',
        'zh': ':落叶树:',
        'ru': ':дерево:'
    },
    '\U0001F98C': {  # 🦌
        'en': ':deer:',
        'status': fully_qualified,
        'E': 3,
        'de': ':hirsch:',
        'es': ':ciervo:',
        'fr': ':cerf:',
        'ja': ':シカ:',
        'ko': ':사슴:',
        'pt': ':cervo:',
        'it': ':cervo:',
        'fa': ':گوزن:',
        'id': ':rusa:',
        'zh': ':鹿:',
        'ru': ':олень:'
    },
    '\U0001F69A': {  # 🚚
        'en': ':delivery_truck:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':truck:'],
        'de': ':lieferwagen:',
        'es': ':camión_de_reparto:',
        'fr': ':camion_de_livraison:',
        'ja': ':トラック:',
        'ko': ':운송_트럭:',
        'pt': ':caminhão_de_entrega:',
        'it': ':camion:',
        'fa': ':کامیون_پخش:',
        'id': ':truk_pengiriman:',
        'zh': ':货车:',
        'ru': ':грузовик:'
    },
    '\U0001F3EC': {  # 🏬
        'en': ':department_store:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kaufhaus:',
        'es': ':grandes_almacenes:',
        'fr': ':grand_magasin:',
        'ja': ':デパート:',
        'ko': ':백화점:',
        'pt': ':loja_de_departamentos:',
        'it': ':grande_magazzino:',
        'fa': ':فروشگاه_بزرگ:',
        'id': ':toserba:',
        'zh': ':商场:',
        'ru': ':универмаг:'
    },
    '\U0001F3DA\U0000FE0F': {  # 🏚️
        'en': ':derelict_house:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':derelict_house_building:'],
        'variant': True,
        'de': ':verfallenes_haus:',
        'es': ':casa_abandonada:',
        'fr': ':maison_abandonnée:',
        'ja': ':廃屋:',
        'ko': ':낡은_주택_건물:',
        'pt': ':casa_abandonada:',
        'it': ':casa_in_rovina:',
        'fa': ':خانه_متروکه:',
        'id': ':rumah_terlantar:',
        'zh': ':废墟:',
        'ru': ':заброшенныи_дом:'
    },
    '\U0001F3DA': {  # 🏚
        'en': ':derelict_house:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':derelict_house_building:'],
        'variant': True,
        'de': ':verfallenes_haus:',
        'es': ':casa_abandonada:',
        'fr': ':maison_abandonnée:',
        'ja': ':廃屋:',
        'ko': ':낡은_주택_건물:',
        'pt': ':casa_abandonada:',
        'it': ':casa_in_rovina:',
        'fa': ':خانهٔ_متروکه:',
        'id': ':rumah_terlantar:',
        'zh': ':废墟:',
        'ru': ':заброшенный_дом:'
    },
    '\U0001F3DC\U0000FE0F': {  # 🏜️
        'en': ':desert:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':wüste:',
        'es': ':desierto:',
        'fr': ':désert:',
        'ja': ':砂漠:',
        'ko': ':사막:',
        'pt': ':deserto:',
        'it': ':deserto:',
        'fa': ':بیابان:',
        'id': ':gurun:',
        'zh': ':沙漠:',
        'ru': ':пустыня:'
    },
    '\U0001F3DC': {  # 🏜
        'en': ':desert:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':wüste:',
        'es': ':desierto:',
        'fr': ':désert:',
        'ja': ':砂漠:',
        'ko': ':사막:',
        'pt': ':deserto:',
        'it': ':deserto:',
        'fa': ':بیابان:',
        'id': ':gurun:',
        'zh': ':沙漠:',
        'ru': ':пустыня:'
    },
    '\U0001F3DD\U0000FE0F': {  # 🏝️
        'en': ':desert_island:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':einsame_insel:',
        'es': ':isla_desierta:',
        'fr': ':île_déserte:',
        'ja': ':無人島:',
        'ko': ':사막_섬:',
        'pt': ':ilha_deserta:',
        'it': ':isola_deserta:',
        'fa': ':جزیره_خالی_از_سکنه:',
        'id': ':pulau_terpencil:',
        'zh': ':无人荒岛:',
        'ru': ':остров:'
    },
    '\U0001F3DD': {  # 🏝
        'en': ':desert_island:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':einsame_insel:',
        'es': ':isla_desierta:',
        'fr': ':île_déserte:',
        'ja': ':無人島:',
        'ko': ':사막_섬:',
        'pt': ':ilha_deserta:',
        'it': ':isola_deserta:',
        'fa': ':جزیرهٔ_خالی_از_سکنه:',
        'id': ':pulau_terpencil:',
        'zh': ':无人荒岛:',
        'ru': ':остров:'
    },
    '\U0001F5A5\U0000FE0F': {  # 🖥️
        'en': ':desktop_computer:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':desktopcomputer:',
        'es': ':ordenador_de_sobremesa:',
        'fr': ':ordinateur_de_bureau:',
        'ja': ':デスクトップパソコン:',
        'ko': ':데스크톱_컴퓨터:',
        'pt': ':computador_de_mesa:',
        'it': ':computer_fisso:',
        'fa': ':رایانه_میزی:',
        'id': ':komputer_desktop:',
        'zh': ':台式电脑:',
        'ru': ':компьютер:'
    },
    '\U0001F5A5': {  # 🖥
        'en': ':desktop_computer:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':desktopcomputer:',
        'es': ':ordenador_de_sobremesa:',
        'fr': ':ordinateur_de_bureau:',
        'ja': ':デスクトップパソコン:',
        'ko': ':데스크톱_컴퓨터:',
        'pt': ':computador_de_mesa:',
        'it': ':computer_fisso:',
        'fa': ':رایانهٔ_میزی:',
        'id': ':komputer_desktop:',
        'zh': ':台式电脑:',
        'ru': ':компьютер:'
    },
    '\U0001F575\U0000FE0F': {  # 🕵️
        'en': ':detective:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':sleuth_or_spy:'],
        'variant': True,
        'de': ':detektiv(in):',
        'es': ':detective:',
        'fr': ':détective:',
        'ja': ':探偵:',
        'ko': ':탐정:',
        'pt': ':detetive:',
        'it': ':detective:',
        'fa': ':کارآگاه:',
        'id': ':detektif:',
        'zh': ':侦探:',
        'ru': ':детектив:'
    },
    '\U0001F575': {  # 🕵
        'en': ':detective:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':sleuth_or_spy:'],
        'variant': True,
        'de': ':detektiv(in):',
        'es': ':detective:',
        'fr': ':détective:',
        'ja': ':探偵:',
        'ko': ':탐정:',
        'pt': ':detetive:',
        'it': ':detective:',
        'fa': ':کارآگاه:',
        'id': ':detektif:',
        'zh': ':侦探:',
        'ru': ':детектив:'
    },
    '\U0001F575\U0001F3FF': {  # 🕵🏿
        'en': ':detective_dark_skin_tone:',
        'status': fully_qualified,
        'E': 2,
        'de': ':detektiv(in)_dunkle_hautfarbe:',
        'es': ':detective_tono_de_piel_oscuro:',
        'fr': ':détective_peau_foncée:',
        'ja': ':探偵_濃い肌色:',
        'ko': ':탐정_검은색_피부:',
        'pt': ':detetive_pele_escura:',
        'it': ':detective_carnagione_scura:',
        'fa': ':کارآگاه_پوست_آبنوسی:',
        'id': ':detektif_warna_kulit_gelap:',
        'zh': ':侦探_较深肤色:',
        'ru': ':детектив_очень_темный_тон_кожи:'
    },
    '\U0001F575\U0001F3FB': {  # 🕵🏻
        'en': ':detective_light_skin_tone:',
        'status': fully_qualified,
        'E': 2,
        'de': ':detektiv(in)_helle_hautfarbe:',
        'es': ':detective_tono_de_piel_claro:',
        'fr': ':détective_peau_claire:',
        'ja': ':探偵_薄い肌色:',
        'ko': ':탐정_하얀_피부:',
        'pt': ':detetive_pele_clara:',
        'it': ':detective_carnagione_chiara:',
        'fa': ':کارآگاه_پوست_سفید:',
        'id': ':detektif_warna_kulit_cerah:',
        'zh': ':侦探_较浅肤色:',
        'ru': ':детектив_очень_светлый_тон_кожи:'
    },
    '\U0001F575\U0001F3FE': {  # 🕵🏾
        'en': ':detective_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 2,
        'de': ':detektiv(in)_mitteldunkle_hautfarbe:',
        'es': ':detective_tono_de_piel_oscuro_medio:',
        'fr': ':détective_peau_mate:',
        'ja': ':探偵_やや濃い肌色:',
        'ko': ':탐정_진한_갈색_피부:',
        'pt': ':detetive_pele_morena_escura:',
        'it': ':detective_carnagione_abbastanza_scura:',
        'fa': ':کارآگاه_پوست_گندمی:',
        'id': ':detektif_warna_kulit_gelap-sedang:',
        'zh': ':侦探_中等深肤色:',
        'ru': ':детектив_темный_тон_кожи:'
    },
    '\U0001F575\U0001F3FC': {  # 🕵🏼
        'en': ':detective_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 2,
        'de': ':detektiv(in)_mittelhelle_hautfarbe:',
        'es': ':detective_tono_de_piel_claro_medio:',
        'fr': ':détective_peau_moyennement_claire:',
        'ja': ':探偵_やや薄い肌色:',
        'ko': ':탐정_연한_갈색_피부:',
        'pt': ':detetive_pele_morena_clara:',
        'it': ':detective_carnagione_abbastanza_chiara:',
        'fa': ':کارآگاه_پوست_روشن:',
        'id': ':detektif_warna_kulit_cerah-sedang:',
        'zh': ':侦探_中等浅肤色:',
        'ru': ':детектив_светлый_тон_кожи:'
    },
    '\U0001F575\U0001F3FD': {  # 🕵🏽
        'en': ':detective_medium_skin_tone:',
        'status': fully_qualified,
        'E': 2,
        'de': ':detektiv(in)_mittlere_hautfarbe:',
        'es': ':detective_tono_de_piel_medio:',
        'fr': ':détective_peau_légèrement_mate:',
        'ja': ':探偵_中間の肌色:',
        'ko': ':탐정_갈색_피부:',
        'pt': ':detetive_pele_morena:',
        'it': ':detective_carnagione_olivastra:',
        'fa': ':کارآگاه_پوست_طلایی:',
        'id': ':detektif_warna_kulit_sedang:',
        'zh': ':侦探_中等肤色:',
        'ru': ':детектив_средний_тон_кожи:'
    },
    '\U00002666\U0000FE0F': {  # ♦️
        'en': ':diamond_suit:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':diamonds:'],
        'variant': True,
        'de': ':karo:',
        'es': ':palo_de_diamantes:',
        'fr': ':carreau:',
        'ja': ':ダイヤ:',
        'ko': ':다이아몬드:',
        'pt': ':naipe_de_ouros:',
        'it': ':quadri:',
        'fa': ':خال_خشت:',
        'id': ':wajik:',
        'zh': ':方片:',
        'ru': ':бубны:'
    },
    '\U00002666': {  # ♦
        'en': ':diamond_suit:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':diamonds:'],
        'variant': True,
        'de': ':karo:',
        'es': ':palo_de_diamantes:',
        'fr': ':carreau:',
        'ja': ':ダイヤ:',
        'ko': ':다이아몬드:',
        'pt': ':naipe_de_ouros:',
        'it': ':quadri:',
        'fa': ':خال_خشت:',
        'id': ':wajik:',
        'zh': ':方片:',
        'ru': ':бубны:'
    },
    '\U0001F4A0': {  # 💠
        'en': ':diamond_with_a_dot:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':diamond_shape_with_a_dot_inside:'],
        'de': ':rautenform_mit_punkt:',
        'es': ':rombo_con_pétalo:',
        'fr': ':diamant_avec_un_point:',
        'ja': ':ドット付きひし形:',
        'ko': ':점이_찍힌_다이아몬드_모양:',
        'pt': ':diamante_com_um_ponto:',
        'it': ':petalo_di_fiore:',
        'fa': ':لوزی_نقطه_دار:',
        'id': ':wajik_bertitik:',
        'zh': ':带圆点的菱形:',
        'ru': ':ромб_с_точкой:'
    },
    '\U0001F505': {  # 🔅
        'en': ':dim_button:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':low_brightness:'],
        'de': ':taste_dimmen:',
        'es': ':brillo_bajo:',
        'fr': ':luminosité_faible:',
        'ja': ':低輝度:',
        'ko': ':밝기_낮음:',
        'pt': ':botão_de_diminuir_brilho:',
        'it': ':luminosità_bassa:',
        'fa': ':دکمهٔ_تار_کردن_صفحه:',
        'id': ':tombol_redupkan:',
        'zh': ':低亮度按钮:',
        'ru': ':низкая_яркость:'
    },
    '\U0001F61E': {  # 😞
        'en': ':disappointed_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':disappointed:'],
        'de': ':enttäuschtes_gesicht:',
        'es': ':cara_decepcionada:',
        'fr': ':visage_déçu:',
        'ja': ':失望した顔:',
        'ko': ':실망한_얼굴:',
        'pt': ':rosto_desapontado:',
        'it': ':faccina_delusa:',
        'fa': ':ناامید:',
        'id': ':wajah_kecewa:',
        'zh': ':失望:',
        'ru': ':разочарование:'
    },
    '\U0001F978': {  # 🥸
        'en': ':disguised_face:',
        'status': fully_qualified,
        'E': 13,
        'de': ':verkleidet:',
        'es': ':cara_disfrazada:',
        'fr': ':visage_déguisé:',
        'ja': ':変装した顔:',
        'ko': ':변장한_얼굴:',
        'pt': ':rosto_disfarçado:',
        'it': ':faccina_travestita:',
        'fa': ':تغییرقیافه_داده:',
        'id': ':wajah_menyamar:',
        'zh': ':伪装的脸:',
        'ru': ':маскировка:'
    },
    '\U00002797': {  # ➗
        'en': ':divide:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':heavy_division_sign:'],
        'de': ':geteilt_durch:',
        'es': ':división:',
        'fr': ':signe_diviser:',
        'ja': ':割り算:',
        'ko': ':나누기:',
        'pt': ':símbolo_de_divisão:',
        'it': ':divisione:',
        'fa': ':تقسیم_کردن:',
        'id': ':bagi:',
        'zh': ':除:',
        'ru': ':знак_деления:'
    },
    '\U0001F93F': {  # 🤿
        'en': ':diving_mask:',
        'status': fully_qualified,
        'E': 12,
        'de': ':tauchmaske:',
        'es': ':máscara_de_buceo:',
        'fr': ':masque_de_plongée:',
        'ja': ':ダイビング_マスク:',
        'ko': ':다이빙_마스크:',
        'pt': ':máscara_de_mergulho:',
        'it': ':maschera_da_sub:',
        'fa': ':ماسک_غواصی:',
        'id': ':kacamata_selam:',
        'zh': ':潜水面罩:',
        'ru': ':маска_с_трубкой:'
    },
    '\U0001FA94': {  # 🪔
        'en': ':diya_lamp:',
        'status': fully_qualified,
        'E': 12,
        'de': ':öllampe:',
        'es': ':lámpara_de_aceite:',
        'fr': ':diya:',
        'ja': ':ディヤランプ:',
        'ko': ':기름_램프:',
        'pt': ':lâmpada_de_óleo:',
        'it': ':diya:',
        'fa': ':چراغ_دایا:',
        'id': ':lampu_diya:',
        'zh': ':印度油灯:',
        'ru': ':масляная_лампа:'
    },
    '\U0001F4AB': {  # 💫
        'en': ':dizzy:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':schwindlig:',
        'es': ':símbolo_de_mareo:',
        'fr': ':étourdissement:',
        'ja': ':くらくら:',
        'ko': ':어지러움:',
        'pt': ':zonzo:',
        'it': ':stella_con_scia:',
        'fa': ':سرگیجه:',
        'id': ':pusing:',
        'zh': ':头晕:',
        'ru': ':кружащаяся_звезда:'
    },
    '\U0001F9EC': {  # 🧬
        'en': ':dna:',
        'status': fully_qualified,
        'E': 11,
        'de': ':dna:',
        'es': ':adn:',
        'fr': ':adn:',
        'ja': ':dna:',
        'ko': ':dna:',
        'pt': ':dna:',
        'it': ':dna:',
        'fa': ':دی_ان_ای:',
        'id': ':dna:',
        'zh': ':DNA:',
        'ru': ':ДНК:'
    },
    '\U0001F9A4': {  # 🦤
        'en': ':dodo:',
        'status': fully_qualified,
        'E': 13,
        'de': ':dodo:',
        'es': ':dodo:',
        'fr': ':dodo:',
        'ja': ':ドードー:',
        'ko': ':도도새:',
        'pt': ':dodô:',
        'it': ':dodo:',
        'fa': ':دودو:',
        'id': ':dodo:',
        'zh': ':渡渡鸟:',
        'ru': ':дронт:'
    },
    '\U0001F415': {  # 🐕
        'en': ':dog:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':dog2:'],
        'variant': True,
        'de': ':hund:',
        'es': ':perro:',
        'fr': ':chien:',
        'ja': ':イヌ:',
        'ko': ':개:',
        'pt': ':cachorro:',
        'it': ':cane:',
        'fa': ':سگ:',
        'id': ':anjing:',
        'zh': ':狗:',
        'ru': ':собака:'
    },
    '\U0001F436': {  # 🐶
        'en': ':dog_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':dog:'],
        'de': ':hundegesicht:',
        'es': ':cara_de_perro:',
        'fr': ':tête_de_chien:',
        'ja': ':イヌの顔:',
        'ko': ':강아지_얼굴:',
        'pt': ':rosto_de_cachorro:',
        'it': ':muso_di_cane:',
        'fa': ':صورت_سگ:',
        'id': ':wajah_anjing:',
        'zh': ':狗脸:',
        'ru': ':морда_собаки:'
    },
    '\U0001F4B5': {  # 💵
        'en': ':dollar_banknote:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':dollar:'],
        'de': ':dollar-banknote:',
        'es': ':billete_de_dólar:',
        'fr': ':billet_en_dollars:',
        'ja': ':ドル札:',
        'ko': ':달러:',
        'pt': ':nota_de_dólar:',
        'it': ':banconota_dollaro:',
        'fa': ':اسکناس_دلار:',
        'id': ':uang_kertas_dolar:',
        'zh': ':美元:',
        'ru': ':банкнота_доллара:'
    },
    '\U0001F42C': {  # 🐬
        'en': ':dolphin:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flipper:'],
        'de': ':delfin:',
        'es': ':delfín:',
        'fr': ':dauphin:',
        'ja': ':イルカ:',
        'ko': ':돌고래:',
        'pt': ':golfinho:',
        'it': ':delfino:',
        'fa': ':دلفین:',
        'id': ':lumba-lumba:',
        'zh': ':海豚:',
        'ru': ':дельфин:'
    },
    '\U0001FACF': {  # 🫏
        'en': ':donkey:',
        'status': fully_qualified,
        'E': 15,
        'de': ':esel:',
        'es': ':burro:',
        'fr': ':âne:',
        'ja': ':ロバ:',
        'ko': ':당나귀:',
        'pt': ':burro:',
        'it': ':asino:',
        'fa': ':الاغ:',
        'id': ':keledai:',
        'zh': ':驴:',
        'ru': ':осел:'
    },
    '\U0001F6AA': {  # 🚪
        'en': ':door:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':tür:',
        'es': ':puerta:',
        'fr': ':porte:',
        'ja': ':ドア:',
        'ko': ':문:',
        'pt': ':porta:',
        'it': ':porta:',
        'fa': ':در:',
        'id': ':pintu:',
        'zh': ':门:',
        'ru': ':дверь:'
    },
    '\U0001FAE5': {  # 🫥
        'en': ':dotted_line_face:',
        'status': fully_qualified,
        'E': 14,
        'de': ':gestricheltes_gesicht:',
        'es': ':cara_con_línea_de_puntos:',
        'fr': ':visage_en_pointillés:',
        'ja': ':点線の顔:',
        'ko': ':테두리가_점선으로_된_얼굴:',
        'pt': ':rosto_com_linha_pontilhada:',
        'it': ':faccina_tratteggiata:',
        'fa': ':چهره_با_خط_چین:',
        'id': ':wajah_dengan_garis_titik-titik:',
        'zh': ':虚线脸:',
        'ru': ':лицо_пунктиром:'
    },
    '\U0001F52F': {  # 🔯
        'en': ':dotted_six-pointed_star:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':six_pointed_star:', ':dotted_six_pointed_star:'],
        'de': ':hexagramm_mit_punkt:',
        'es': ':estrella_de_seis_puntas:',
        'fr': ':étoile_à_6_branches:',
        'ja': ':六芒星:',
        'ko': ':육각_별:',
        'pt': ':estrela_de_seis_pontas:',
        'it': ':stella_a_sei_punte:',
        'fa': ':ستارهٔ_شش_پر:',
        'id': ':bintang_sudut_enam_bertitik:',
        'zh': ':带中间点的六芒星:',
        'ru': ':шестиконечная_звезда:'
    },
    '\U000027BF': {  # ➿
        'en': ':double_curly_loop:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':loop:'],
        'de': ':doppelschleife:',
        'es': ':bucle_doble:',
        'fr': ':double_boucle:',
        'ja': ':ダブルループ:',
        'ko': ':두_개의_고리:',
        'pt': ':loop_encaracolado_duas_vezes:',
        'it': ':doppio_occhiello:',
        'fa': ':حلقه_مجعد_دوتایی:',
        'id': ':untaian_bergelombang_ganda:',
        'zh': ':双卷曲环:',
        'ru': ':двойная_петля:'
    },
    '\U0000203C\U0000FE0F': {  # ‼️
        'en': ':double_exclamation_mark:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':bangbang:'],
        'variant': True,
        'de': ':doppeltes_ausrufezeichen:',
        'es': ':exclamación_doble:',
        'fr': ':double_point_d’exclamation:',
        'ja': ':二重感嘆符:',
        'ko': ':두_개의_느낌표:',
        'pt': ':dupla_exclamação:',
        'it': ':doppio_punto_esclamativo:',
        'fa': ':علامت_تعجب_دوتایی:',
        'id': ':tanda_seru_ganda:',
        'zh': ':双感叹号:',
        'ru': ':два_восклицательных_знака:'
    },
    '\U0000203C': {  # ‼
        'en': ':double_exclamation_mark:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':bangbang:'],
        'variant': True,
        'de': ':doppeltes_ausrufezeichen:',
        'es': ':exclamación_doble:',
        'fr': ':double_point_d’exclamation:',
        'ja': ':二重感嘆符:',
        'ko': ':두_개의_느낌표:',
        'pt': ':dupla_exclamação:',
        'it': ':doppio_punto_esclamativo:',
        'fa': ':علامت_تعجب_دوتایی:',
        'id': ':tanda_seru_ganda:',
        'zh': ':双感叹号:',
        'ru': ':два_восклицательных_знака:'
    },
    '\U0001F369': {  # 🍩
        'en': ':doughnut:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':donut:',
        'es': ':dónut:',
        'fr': ':doughnut:',
        'ja': ':ドーナツ:',
        'ko': ':도넛:',
        'pt': ':donut:',
        'it': ':ciambella:',
        'fa': ':دونات:',
        'id': ':donat:',
        'zh': ':甜甜圈:',
        'ru': ':пончик:'
    },
    '\U0001F54A\U0000FE0F': {  # 🕊️
        'en': ':dove:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':dove_of_peace:'],
        'variant': True,
        'de': ':taube:',
        'es': ':paloma:',
        'fr': ':colombe:',
        'ja': ':ハト:',
        'ko': ':비둘기:',
        'pt': ':pomba_branca:',
        'it': ':colomba:',
        'fa': ':کبوتر:',
        'id': ':merpati:',
        'zh': ':鸽:',
        'ru': ':голубь:'
    },
    '\U0001F54A': {  # 🕊
        'en': ':dove:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':dove_of_peace:'],
        'variant': True,
        'de': ':taube:',
        'es': ':paloma:',
        'fr': ':colombe:',
        'ja': ':ハト:',
        'ko': ':비둘기:',
        'pt': ':pomba_branca:',
        'it': ':colomba:',
        'fa': ':کبوتر:',
        'id': ':merpati:',
        'zh': ':鸽:',
        'ru': ':голубь:'
    },
    '\U00002199\U0000FE0F': {  # ↙️
        'en': ':down-left_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':arrow_lower_left:', ':down_left_arrow:'],
        'variant': True,
        'de': ':pfeil_nach_links_unten:',
        'es': ':flecha_hacia_la_esquina_inferior_izquierda:',
        'fr': ':flèche_bas_gauche:',
        'ja': ':左下矢印:',
        'ko': ':좌하향_화살표:',
        'pt': ':seta_para_baixo_e_para_a_esquerda:',
        'it': ':freccia_in_basso_a_sinistra:',
        'fa': ':پیکان_پایین_چپ:',
        'id': ':tanda_panah_kiri_bawah:',
        'zh': ':左下箭头:',
        'ru': ':стрелка_влево-вниз:'
    },
    '\U00002199': {  # ↙
        'en': ':down-left_arrow:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':arrow_lower_left:', ':down_left_arrow:'],
        'variant': True,
        'de': ':pfeil_nach_links_unten:',
        'es': ':flecha_hacia_la_esquina_inferior_izquierda:',
        'fr': ':flèche_bas_gauche:',
        'ja': ':左下矢印:',
        'ko': ':좌하향_화살표:',
        'pt': ':seta_para_baixo_e_para_a_esquerda:',
        'it': ':freccia_in_basso_a_sinistra:',
        'fa': ':پیکان_پایین_چپ:',
        'id': ':tanda_panah_kiri_bawah:',
        'zh': ':左下箭头:',
        'ru': ':стрелка_влево_вниз:'
    },
    '\U00002198\U0000FE0F': {  # ↘️
        'en': ':down-right_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':arrow_lower_right:', ':down_right_arrow:'],
        'variant': True,
        'de': ':pfeil_nach_rechts_unten:',
        'es': ':flecha_hacia_la_esquina_inferior_derecha:',
        'fr': ':flèche_bas_droite:',
        'ja': ':右下矢印:',
        'ko': ':우하향_화살표:',
        'pt': ':seta_para_baixo_e_para_a_direita:',
        'it': ':freccia_in_basso_a_destra:',
        'fa': ':پیکان_پایین_راست:',
        'id': ':tanda_panah_kanan_bawah:',
        'zh': ':右下箭头:',
        'ru': ':стрелка_вправо-вниз:'
    },
    '\U00002198': {  # ↘
        'en': ':down-right_arrow:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':arrow_lower_right:', ':down_right_arrow:'],
        'variant': True,
        'de': ':pfeil_nach_rechts_unten:',
        'es': ':flecha_hacia_la_esquina_inferior_derecha:',
        'fr': ':flèche_bas_droite:',
        'ja': ':右下矢印:',
        'ko': ':우하향_화살표:',
        'pt': ':seta_para_baixo_e_para_a_direita:',
        'it': ':freccia_in_basso_a_destra:',
        'fa': ':پیکان_پایین_راست:',
        'id': ':tanda_panah_kanan_bawah:',
        'zh': ':右下箭头:',
        'ru': ':стрелка_вправо_вниз:'
    },
    '\U00002B07\U0000FE0F': {  # ⬇️
        'en': ':down_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':arrow_down:'],
        'variant': True,
        'de': ':pfeil_nach_unten:',
        'es': ':flecha_hacia_abajo:',
        'fr': ':flèche_bas:',
        'ja': ':下矢印:',
        'ko': ':하향_화살표:',
        'pt': ':seta_para_baixo:',
        'it': ':freccia_rivolta_verso_il_basso:',
        'fa': ':پیکان_پایین:',
        'id': ':tanda_panah_bawah:',
        'zh': ':向下箭头:',
        'ru': ':стрелка_вниз:'
    },
    '\U00002B07': {  # ⬇
        'en': ':down_arrow:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':arrow_down:'],
        'variant': True,
        'de': ':pfeil_nach_unten:',
        'es': ':flecha_hacia_abajo:',
        'fr': ':flèche_bas:',
        'ja': ':下矢印:',
        'ko': ':하향_화살표:',
        'pt': ':seta_para_baixo:',
        'it': ':freccia_rivolta_verso_il_basso:',
        'fa': ':پیکان_پایین:',
        'id': ':tanda_panah_bawah:',
        'zh': ':向下箭头:',
        'ru': ':стрелка_вниз:'
    },
    '\U0001F613': {  # 😓
        'en': ':downcast_face_with_sweat:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':sweat:'],
        'de': ':bedrücktes_gesicht_mit_schweiß:',
        'es': ':cara_con_sudor_frío:',
        'fr': ':visage_démoralisé_avec_goutte_de_sueur:',
        'ja': ':冷や汗:',
        'ko': ':식은땀_흘리는_얼굴:',
        'pt': ':rosto_cabisbaixo_com_gota_de_suor:',
        'it': ':faccina_abbattuta_sudata:',
        'fa': ':عرق_سرد:',
        'id': ':wajah_berkeringat:',
        'zh': ':汗:',
        'ru': ':в_холодном_поту:'
    },
    '\U0001F53D': {  # 🔽
        'en': ':downwards_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':arrow_down_small:'],
        'de': ':abwärts-schaltfläche:',
        'es': ':triángulo_hacia_abajo:',
        'fr': ':petit_triangle_bas:',
        'ja': ':下ボタン:',
        'ko': ':아래쪽_삼각형:',
        'pt': ':botão_apontando_para_baixo:',
        'it': ':pulsante_a_triangolo_rivolto_verso_il_basso:',
        'fa': ':دکمهٔ_پایین:',
        'id': ':tombol_turun:',
        'zh': ':向下三角形按钮:',
        'ru': ':треугольник_острием_вниз:'
    },
    '\U0001F409': {  # 🐉
        'en': ':dragon:',
        'status': fully_qualified,
        'E': 1,
        'de': ':drache:',
        'es': ':dragón:',
        'fr': ':dragon:',
        'ja': ':ドラゴン:',
        'ko': ':용:',
        'pt': ':dragão:',
        'it': ':drago:',
        'fa': ':اژدها:',
        'id': ':naga:',
        'zh': ':龙:',
        'ru': ':дракон:'
    },
    '\U0001F432': {  # 🐲
        'en': ':dragon_face:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':drachengesicht:',
        'es': ':cara_de_dragón:',
        'fr': ':tête_de_dragon:',
        'ja': ':ドラゴンの顔:',
        'ko': ':용_얼굴:',
        'pt': ':rosto_de_dragão:',
        'it': ':testa_di_drago:',
        'fa': ':صورت_اژدها:',
        'id': ':wajah_naga:',
        'zh': ':龙头:',
        'ru': ':голова_дракона:'
    },
    '\U0001F457': {  # 👗
        'en': ':dress:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kleid:',
        'es': ':vestido:',
        'fr': ':robe:',
        'ja': ':ワンピース:',
        'ko': ':원피스:',
        'pt': ':vestido:',
        'it': ':vestito:',
        'fa': ':پیراهن_زنانه:',
        'id': ':gaun:',
        'zh': ':连衣裙:',
        'ru': ':платье:'
    },
    '\U0001F924': {  # 🤤
        'en': ':drooling_face:',
        'status': fully_qualified,
        'E': 3,
        'de': ':sabberndes_gesicht:',
        'es': ':cara_babeando:',
        'fr': ':visage_qui_bave:',
        'ja': ':よだれを垂らした顔:',
        'ko': ':침_흘리는_얼굴:',
        'pt': ':rosto_babando:',
        'it': ':faccina_che_sbava:',
        'fa': ':آب_افتادن_دهان:',
        'id': ':ngiler:',
        'zh': ':流口水:',
        'ru': ':пускает_слюни:'
    },
    '\U0001FA78': {  # 🩸
        'en': ':drop_of_blood:',
        'status': fully_qualified,
        'E': 12,
        'de': ':blutstropfen:',
        'es': ':gota_de_sangre:',
        'fr': ':goutte_de_sang:',
        'ja': ':血液:',
        'ko': ':핏방울:',
        'pt': ':gota_de_sangue:',
        'it': ':goccia_di_sangue:',
        'fa': ':قطره_خون:',
        'id': ':tetesan_darah:',
        'zh': ':血滴:',
        'ru': ':капля_крови:'
    },
    '\U0001F4A7': {  # 💧
        'en': ':droplet:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':tropfen:',
        'es': ':gota:',
        'fr': ':goutte_d’eau:',
        'ja': ':水滴:',
        'ko': ':물방울:',
        'pt': ':gota:',
        'it': ':goccia:',
        'fa': ':قطره:',
        'id': ':tetesan_air:',
        'zh': ':水滴:',
        'ru': ':капля:'
    },
    '\U0001F941': {  # 🥁
        'en': ':drum:',
        'status': fully_qualified,
        'E': 3,
        'de': ':trommel:',
        'es': ':tambor:',
        'fr': ':batterie:',
        'ja': ':ドラム:',
        'ko': ':드럼:',
        'pt': ':tambor:',
        'it': ':tamburo:',
        'fa': ':طبل:',
        'id': ':drum:',
        'zh': ':鼓:',
        'ru': ':барабан:'
    },
    '\U0001F986': {  # 🦆
        'en': ':duck:',
        'status': fully_qualified,
        'E': 3,
        'de': ':ente:',
        'es': ':pato:',
        'fr': ':canard:',
        'ja': ':カモ:',
        'ko': ':오리:',
        'pt': ':pato:',
        'it': ':anatra:',
        'fa': ':مرغابی:',
        'id': ':bebek:',
        'zh': ':鸭子:',
        'ru': ':утка:'
    },
    '\U0001F95F': {  # 🥟
        'en': ':dumpling:',
        'status': fully_qualified,
        'E': 5,
        'de': ':teigtasche:',
        'es': ':dumpling:',
        'fr': ':boulette_de_pâte:',
        'ja': ':点心:',
        'ko': ':만두:',
        'pt': ':bolinho_asiático:',
        'it': ':raviolo:',
        'fa': ':کوفته:',
        'id': ':dumpling:',
        'zh': ':饺子:',
        'ru': ':вареник:'
    },
    '\U0001F4C0': {  # 📀
        'en': ':dvd:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':dvd:',
        'es': ':disco_dvd:',
        'fr': ':dvd:',
        'ja': ':dvd:',
        'ko': ':디비디:',
        'pt': ':dvd:',
        'it': ':dvd:',
        'fa': ':دی_وی_دی:',
        'id': ':dvd:',
        'zh': ':DVD:',
        'ru': ':DVD:'
    },
    '\U0001F4E7': {  # 📧
        'en': ':e-mail:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':email:', ':e_mail:'],
        'de': ':e-mail:',
        'es': ':correo_electrónico:',
        'fr': ':e-mail:',
        'ja': ':eメール:',
        'ko': ':이메일:',
        'pt': ':e-mail:',
        'it': ':e-mail:',
        'fa': ':ایمیل:',
        'id': ':email:',
        'zh': ':电子邮件:',
        'ru': ':электронная_почта:'
    },
    '\U0001F985': {  # 🦅
        'en': ':eagle:',
        'status': fully_qualified,
        'E': 3,
        'de': ':adler:',
        'es': ':águila:',
        'fr': ':aigle:',
        'ja': ':ワシ:',
        'ko': ':독수리:',
        'pt': ':águia:',
        'it': ':aquila:',
        'fa': ':عقاب:',
        'id': ':elang:',
        'zh': ':鹰:',
        'ru': ':орел:'
    },
    '\U0001F442': {  # 👂
        'en': ':ear:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':ohr:',
        'es': ':oreja:',
        'fr': ':oreille:',
        'ja': ':耳:',
        'ko': ':귀:',
        'pt': ':orelha:',
        'it': ':orecchio:',
        'fa': ':گوش:',
        'id': ':telinga:',
        'zh': ':耳朵:',
        'ru': ':ухо:'
    },
    '\U0001F442\U0001F3FF': {  # 👂🏿
        'en': ':ear_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ohr_dunkle_hautfarbe:',
        'es': ':oreja_tono_de_piel_oscuro:',
        'fr': ':oreille_peau_foncée:',
        'ja': ':耳_濃い肌色:',
        'ko': ':귀_검은색_피부:',
        'pt': ':orelha_pele_escura:',
        'it': ':orecchio_carnagione_scura:',
        'fa': ':گوش_پوست_آبنوسی:',
        'id': ':telinga_warna_kulit_gelap:',
        'zh': ':耳朵_较深肤色:',
        'ru': ':ухо_очень_темный_тон_кожи:'
    },
    '\U0001F442\U0001F3FB': {  # 👂🏻
        'en': ':ear_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ohr_helle_hautfarbe:',
        'es': ':oreja_tono_de_piel_claro:',
        'fr': ':oreille_peau_claire:',
        'ja': ':耳_薄い肌色:',
        'ko': ':귀_하얀_피부:',
        'pt': ':orelha_pele_clara:',
        'it': ':orecchio_carnagione_chiara:',
        'fa': ':گوش_پوست_سفید:',
        'id': ':telinga_warna_kulit_cerah:',
        'zh': ':耳朵_较浅肤色:',
        'ru': ':ухо_очень_светлый_тон_кожи:'
    },
    '\U0001F442\U0001F3FE': {  # 👂🏾
        'en': ':ear_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ohr_mitteldunkle_hautfarbe:',
        'es': ':oreja_tono_de_piel_oscuro_medio:',
        'fr': ':oreille_peau_mate:',
        'ja': ':耳_やや濃い肌色:',
        'ko': ':귀_진한_갈색_피부:',
        'pt': ':orelha_pele_morena_escura:',
        'it': ':orecchio_carnagione_abbastanza_scura:',
        'fa': ':گوش_پوست_گندمی:',
        'id': ':telinga_warna_kulit_gelap-sedang:',
        'zh': ':耳朵_中等深肤色:',
        'ru': ':ухо_темный_тон_кожи:'
    },
    '\U0001F442\U0001F3FC': {  # 👂🏼
        'en': ':ear_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ohr_mittelhelle_hautfarbe:',
        'es': ':oreja_tono_de_piel_claro_medio:',
        'fr': ':oreille_peau_moyennement_claire:',
        'ja': ':耳_やや薄い肌色:',
        'ko': ':귀_연한_갈색_피부:',
        'pt': ':orelha_pele_morena_clara:',
        'it': ':orecchio_carnagione_abbastanza_chiara:',
        'fa': ':گوش_پوست_روشن:',
        'id': ':telinga_warna_kulit_cerah-sedang:',
        'zh': ':耳朵_中等浅肤色:',
        'ru': ':ухо_светлый_тон_кожи:'
    },
    '\U0001F442\U0001F3FD': {  # 👂🏽
        'en': ':ear_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ohr_mittlere_hautfarbe:',
        'es': ':oreja_tono_de_piel_medio:',
        'fr': ':oreille_peau_légèrement_mate:',
        'ja': ':耳_中間の肌色:',
        'ko': ':귀_갈색_피부:',
        'pt': ':orelha_pele_morena:',
        'it': ':orecchio_carnagione_olivastra:',
        'fa': ':گوش_پوست_طلایی:',
        'id': ':telinga_warna_kulit_sedang:',
        'zh': ':耳朵_中等肤色:',
        'ru': ':ухо_средний_тон_кожи:'
    },
    '\U0001F33D': {  # 🌽
        'en': ':ear_of_corn:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':corn:'],
        'de': ':maiskolben:',
        'es': ':espiga_de_maíz:',
        'fr': ':épi_de_maïs:',
        'ja': ':とうもろこし:',
        'ko': ':옥수수:',
        'pt': ':milho:',
        'it': ':pannocchia:',
        'fa': ':بلال_ذرت:',
        'id': ':bonggol_jagung:',
        'zh': ':玉米:',
        'ru': ':кукурузный_початок:'
    },
    '\U0001F9BB': {  # 🦻
        'en': ':ear_with_hearing_aid:',
        'status': fully_qualified,
        'E': 12,
        'de': ':ohr_mit_hörgerät:',
        'es': ':oreja_con_audífono:',
        'fr': ':oreille_appareillée:',
        'ja': ':補聴器を付けた耳:',
        'ko': ':보청기를_낀_귀:',
        'pt': ':ouvido_com_aparelho_auditivo:',
        'it': ':orecchio_con_apparecchio_acustico:',
        'fa': ':سمعک_در_گوش:',
        'id': ':telinga_dengan_alat_bantu_dengar:',
        'zh': ':戴助听器的耳朵:',
        'ru': ':ухо_со_слуховым_аппаратом:'
    },
    '\U0001F9BB\U0001F3FF': {  # 🦻🏿
        'en': ':ear_with_hearing_aid_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':ohr_mit_hörgerät_dunkle_hautfarbe:',
        'es': ':oreja_con_audífono_tono_de_piel_oscuro:',
        'fr': ':oreille_appareillée_peau_foncée:',
        'ja': ':補聴器を付けた耳_濃い肌色:',
        'ko': ':보청기를_낀_귀_검은색_피부:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_escura:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_scura:',
        'fa': ':سمعک_در_گوش_پوست_آبنوسی:',
        'id': ':telinga_dengan_alat_bantu_dengar_warna_kulit_gelap:',
        'zh': ':戴助听器的耳朵_较深肤色:',
        'ru': ':ухо_со_слуховым_аппаратом_очень_темный_тон_кожи:'
    },
    '\U0001F9BB\U0001F3FB': {  # 🦻🏻
        'en': ':ear_with_hearing_aid_light_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':ohr_mit_hörgerät_helle_hautfarbe:',
        'es': ':oreja_con_audífono_tono_de_piel_claro:',
        'fr': ':oreille_appareillée_peau_claire:',
        'ja': ':補聴器を付けた耳_薄い肌色:',
        'ko': ':보청기를_낀_귀_하얀_피부:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_clara:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_chiara:',
        'fa': ':سمعک_در_گوش_پوست_سفید:',
        'id': ':telinga_dengan_alat_bantu_dengar_warna_kulit_cerah:',
        'zh': ':戴助听器的耳朵_较浅肤色:',
        'ru': ':ухо_со_слуховым_аппаратом_очень_светлый_тон_кожи:'
    },
    '\U0001F9BB\U0001F3FE': {  # 🦻🏾
        'en': ':ear_with_hearing_aid_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':ohr_mit_hörgerät_mitteldunkle_hautfarbe:',
        'es': ':oreja_con_audífono_tono_de_piel_oscuro_medio:',
        'fr': ':oreille_appareillée_peau_mate:',
        'ja': ':補聴器を付けた耳_やや濃い肌色:',
        'ko': ':보청기를_낀_귀_진한_갈색_피부:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_morena_escura:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_abbastanza_scura:',
        'fa': ':سمعک_در_گوش_پوست_گندمی:',
        'id': ':telinga_dengan_alat_bantu_dengar_warna_kulit_gelap-sedang:',
        'zh': ':戴助听器的耳朵_中等深肤色:',
        'ru': ':ухо_со_слуховым_аппаратом_темный_тон_кожи:'
    },
    '\U0001F9BB\U0001F3FC': {  # 🦻🏼
        'en': ':ear_with_hearing_aid_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':ohr_mit_hörgerät_mittelhelle_hautfarbe:',
        'es': ':oreja_con_audífono_tono_de_piel_claro_medio:',
        'fr': ':oreille_appareillée_peau_moyennement_claire:',
        'ja': ':補聴器を付けた耳_やや薄い肌色:',
        'ko': ':보청기를_낀_귀_연한_갈색_피부:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_morena_clara:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_abbastanza_chiara:',
        'fa': ':سمعک_در_گوش_پوست_روشن:',
        'id': ':telinga_dengan_alat_bantu_dengar_warna_kulit_cerah-sedang:',
        'zh': ':戴助听器的耳朵_中等浅肤色:',
        'ru': ':ухо_со_слуховым_аппаратом_светлый_тон_кожи:'
    },
    '\U0001F9BB\U0001F3FD': {  # 🦻🏽
        'en': ':ear_with_hearing_aid_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12,
        'de': ':ohr_mit_hörgerät_mittlere_hautfarbe:',
        'es': ':oreja_con_audífono_tono_de_piel_medio:',
        'fr': ':oreille_appareillée_peau_légèrement_mate:',
        'ja': ':補聴器を付けた耳_中間の肌色:',
        'ko': ':보청기를_낀_귀_갈색_피부:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_morena:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_olivastra:',
        'fa': ':سمعک_در_گوش_پوست_طلایی:',
        'id': ':telinga_dengan_alat_bantu_dengar_warna_kulit_sedang:',
        'zh': ':戴助听器的耳朵_中等肤色:',
        'ru': ':ухо_со_слуховым_аппаратом_средний_тон_кожи:'
    },
    '\U0001F95A': {  # 🥚
        'en': ':egg:',
        'status': fully_qualified,
        'E': 3,
        'alias': [':egg2:'],
        'de': ':ei:',
        'es': ':huevo:',
        'fr': ':œuf:',
        'ja': ':卵:',
        'ko': ':달걀:',
        'pt': ':ovo:',
        'it': ':uovo:',
        'fa': ':تخم_مرغ:',
        'id': ':telur:',
        'zh': ':蛋:',
        'ru': ':яйцо:'
    },
    '\U0001F346': {  # 🍆
        'en': ':eggplant:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':aubergine:',
        'es': ':berenjena:',
        'fr': ':aubergine:',
        'ja': ':ナス:',
        'ko': ':가지:',
        'pt': ':berinjela:',
        'it': ':melanzana:',
        'fa': ':بادمجان:',
        'id': ':terung:',
        'zh': ':茄子:',
        'ru': ':баклажан:'
    },
    '\U00002734\U0000FE0F': {  # ✴️
        'en': ':eight-pointed_star:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':eight_pointed_black_star:', ':eight_pointed_star:'],
        'variant': True,
        'de': ':stern_mit_acht_zacken:',
        'es': ':estrella_de_ocho_puntas:',
        'fr': ':étoile_huit_branches:',
        'ja': ':八角星:',
        'ko': ':팔각_별:',
        'pt': ':estrela_de_oito_pontas:',
        'it': ':stella_stilizzata:',
        'fa': ':ستاره_هشت_پر:',
        'id': ':bintang_berujung_delapan:',
        'zh': ':八角星:',
        'ru': ':восьмиконечная_звезда:'
    },
    '\U00002734': {  # ✴
        'en': ':eight-pointed_star:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':eight_pointed_black_star:', ':eight_pointed_star:'],
        'variant': True,
        'de': ':achtstrahliger_stern:',
        'es': ':estrella_de_ocho_puntas:',
        'fr': ':étoile_huit_branches:',
        'ja': ':八角星:',
        'ko': ':팔각_별:',
        'pt': ':estrela_de_oito_pontas:',
        'it': ':stella_stilizzata:',
        'fa': ':ستاره_هشت_پر:',
        'id': ':bintang_berujung_delapan:',
        'zh': ':八角星:',
        'ru': ':восьмиконечная_звезда:'
    },
    '\U00002733\U0000FE0F': {  # ✳️
        'en': ':eight-spoked_asterisk:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':eight_spoked_asterisk:'],
        'variant': True,
        'de': ':achtzackiger_stern:',
        'es': ':asterisco_de_ocho_puntas:',
        'fr': ':astérisque_huit_branches:',
        'ja': ':八角アスタリスク:',
        'ko': ':여덟_갈래_별표:',
        'pt': ':asterisco_de_oito_pontas:',
        'it': ':asterisco:',
        'fa': ':ستاره_هشت_پره:',
        'id': ':bintang_berujung_lancip_delapan:',
        'zh': ':八轮辐星号:',
        'ru': ':восьмиконечная_звездочка:'
    },
    '\U00002733': {  # ✳
        'en': ':eight-spoked_asterisk:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':eight_spoked_asterisk:'],
        'variant': True,
        'de': ':achtzackiger_stern:',
        'es': ':asterisco_de_ocho_puntas:',
        'fr': ':astérisque_huit_branches:',
        'ja': ':八角アスタリスク:',
        'ko': ':여덟_갈래_별표:',
        'pt': ':asterisco_de_oito_pontas:',
        'it': ':asterisco:',
        'fa': ':ستاره_هشت_پره:',
        'id': ':bintang_berujung_lancip_delapan:',
        'zh': ':八轮辐星号:',
        'ru': ':восьмиконечная_звездочка:'
    },
    '\U0001F563': {  # 🕣
        'en': ':eight-thirty:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':clock830:', ':eight_thirty:'],
        'variant': True,
        'de': ':8.30_uhr:',
        'es': ':ocho_y_media:',
        'fr': ':huit_heures_et_demie:',
        'ja': ':8時半:',
        'ko': ':여덟_시_반:',
        'pt': ':oito_e_meia:',
        'it': ':ore_otto_e_mezza:',
        'fa': ':ساعت_هشت_و_نیم:',
        'id': ':jam_setengah_sembilan:',
        'zh': ':八点半:',
        'ru': ':полдевятого:'
    },
    '\U0001F557': {  # 🕗
        'en': ':eight_o’clock:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':clock8:', ':eight_oclock:'],
        'variant': True,
        'de': ':8.00_uhr:',
        'es': ':8_en_punto:',
        'fr': ':huit_heures:',
        'ja': ':8時:',
        'ko': ':여덟_시:',
        'pt': ':8_horas:',
        'it': ':ore_otto:',
        'fa': ':ساعت_هشت:',
        'id': ':jam_delapan_tepat:',
        'zh': ':八点:',
        'ru': ':восемь_часов:'
    },
    '\U000023CF\U0000FE0F': {  # ⏏️
        'en': ':eject_button:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':eject_symbol:'],
        'variant': True,
        'de': ':auswerfen:',
        'es': ':expulsar:',
        'fr': ':bouton_éjecter:',
        'ja': ':取り出しボタン:',
        'ko': ':꺼냄_버튼:',
        'pt': ':botão_ejetar:',
        'it': ':pulsante_di_espulsione:',
        'fa': ':دکمه_خروج:',
        'id': ':tombol_keluarkan:',
        'zh': ':推出按钮:',
        'ru': ':выброс:'
    },
    '\U000023CF': {  # ⏏
        'en': ':eject_button:',
        'status': unqualified,
        'E': 1,
        'alias': [':eject_symbol:'],
        'variant': True,
        'de': ':auswerfen:',
        'es': ':expulsar:',
        'fr': ':bouton_éjecter:',
        'ja': ':取り出しボタン:',
        'ko': ':꺼냄_버튼:',
        'pt': ':botão_ejetar:',
        'it': ':pulsante_di_espulsione:',
        'fa': ':دکمهٔ_خروج:',
        'id': ':tombol_keluarkan:',
        'zh': ':推出按钮:',
        'ru': ':выброс:'
    },
    '\U0001F50C': {  # 🔌
        'en': ':electric_plug:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':netzstecker:',
        'es': ':enchufe_eléctrico:',
        'fr': ':câble_avec_fiche_électrique:',
        'ja': ':コンセント:',
        'ko': ':전기_플러그:',
        'pt': ':tomada_elétrica:',
        'it': ':spina_elettrica:',
        'fa': ':دوشاخه:',
        'id': ':steker:',
        'zh': ':电源插头:',
        'ru': ':вилка:'
    },
    '\U0001F418': {  # 🐘
        'en': ':elephant:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':elefant:',
        'es': ':elefante:',
        'fr': ':éléphant:',
        'ja': ':ゾウ:',
        'ko': ':코끼리:',
        'pt': ':elefante:',
        'it': ':elefante:',
        'fa': ':فیل:',
        'id': ':gajah:',
        'zh': ':大象:',
        'ru': ':слон:'
    },
    '\U0001F6D7': {  # 🛗
        'en': ':elevator:',
        'status': fully_qualified,
        'E': 13,
        'de': ':fahrstuhl:',
        'es': ':ascensor:',
        'fr': ':ascenseur:',
        'ja': ':エレベーター:',
        'ko': ':엘리베이터:',
        'pt': ':elevador:',
        'it': ':ascensore:',
        'fa': ':بالابر:',
        'id': ':elevator:',
        'zh': ':电梯:',
        'ru': ':лифт:'
    },
    '\U0001F566': {  # 🕦
        'en': ':eleven-thirty:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':clock1130:', ':eleven_thirty:'],
        'variant': True,
        'de': ':11.30_uhr:',
        'es': ':once_y_media:',
        'fr': ':onze_heures_et_demie:',
        'ja': ':11時半:',
        'ko': ':열한_시_반:',
        'pt': ':onze_e_meia:',
        'it': ':ore_undici_e_mezza:',
        'fa': ':ساعت_یازده_و_نیم:',
        'id': ':jam_setengah_dua_belas:',
        'zh': ':十一点半:',
        'ru': ':полдвенадцатого:'
    },
    '\U0001F55A': {  # 🕚
        'en': ':eleven_o’clock:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':clock11:', ':eleven_oclock:'],
        'variant': True,
        'de': ':11.00_uhr:',
        'es': ':11_en_punto:',
        'fr': ':onze_heures:',
        'ja': ':11時:',
        'ko': ':열한_시:',
        'pt': ':11_horas:',
        'it': ':ore_undici:',
        'fa': ':ساعت_یازده:',
        'id': ':jam_sebelas_tepat:',
        'zh': ':十一点:',
        'ru': ':одиннадцать_часов:'
    },
    '\U0001F9DD': {  # 🧝
        'en': ':elf:',
        'status': fully_qualified,
        'E': 5,
        'de': ':elf(e):',
        'es': ':elfo:',
        'fr': ':elfe:',
        'ja': ':エルフ:',
        'ko': ':엘프:',
        'pt': ':elfo:',
        'it': ':elfo:',
        'fa': ':اِلف:',
        'id': ':kurcaci:',
        'zh': ':小精灵:',
        'ru': ':эльф:'
    },
    '\U0001F9DD\U0001F3FF': {  # 🧝🏿
        'en': ':elf_dark_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':elf(e)_dunkle_hautfarbe:',
        'es': ':elfo_tono_de_piel_oscuro:',
        'fr': ':elfe_peau_foncée:',
        'ja': ':エルフ_濃い肌色:',
        'ko': ':엘프_검은색_피부:',
        'pt': ':elfo_pele_escura:',
        'it': ':elfo_carnagione_scura:',
        'fa': ':اِلف_پوست_آبنوسی:',
        'id': ':kurcaci_warna_kulit_gelap:',
        'zh': ':小精灵_较深肤色:',
        'ru': ':эльф_очень_темный_тон_кожи:'
    },
    '\U0001F9DD\U0001F3FB': {  # 🧝🏻
        'en': ':elf_light_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':elf(e)_helle_hautfarbe:',
        'es': ':elfo_tono_de_piel_claro:',
        'fr': ':elfe_peau_claire:',
        'ja': ':エルフ_薄い肌色:',
        'ko': ':엘프_하얀_피부:',
        'pt': ':elfo_pele_clara:',
        'it': ':elfo_carnagione_chiara:',
        'fa': ':اِلف_پوست_سفید:',
        'id': ':kurcaci_warna_kulit_cerah:',
        'zh': ':小精灵_较浅肤色:',
        'ru': ':эльф_очень_светлый_тон_кожи:'
    },
    '\U0001F9DD\U0001F3FE': {  # 🧝🏾
        'en': ':elf_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':elf(e)_mitteldunkle_hautfarbe:',
        'es': ':elfo_tono_de_piel_oscuro_medio:',
        'fr': ':elfe_peau_mate:',
        'ja': ':エルフ_やや濃い肌色:',
        'ko': ':엘프_진한_갈색_피부:',
        'pt': ':elfo_pele_morena_escura:',
        'it': ':elfo_carnagione_abbastanza_scura:',
        'fa': ':اِلف_پوست_گندمی:',
        'id': ':kurcaci_warna_kulit_gelap-sedang:',
        'zh': ':小精灵_中等深肤色:',
        'ru': ':эльф_темный_тон_кожи:'
    },
    '\U0001F9DD\U0001F3FC': {  # 🧝🏼
        'en': ':elf_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':elf(e)_mittelhelle_hautfarbe:',
        'es': ':elfo_tono_de_piel_claro_medio:',
        'fr': ':elfe_peau_moyennement_claire:',
        'ja': ':エルフ_やや薄い肌色:',
        'ko': ':엘프_연한_갈색_피부:',
        'pt': ':elfo_pele_morena_clara:',
        'it': ':elfo_carnagione_abbastanza_chiara:',
        'fa': ':اِلف_پوست_روشن:',
        'id': ':kurcaci_warna_kulit_cerah-sedang:',
        'zh': ':小精灵_中等浅肤色:',
        'ru': ':эльф_светлый_тон_кожи:'
    },
    '\U0001F9DD\U0001F3FD': {  # 🧝🏽
        'en': ':elf_medium_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':elf(e)_mittlere_hautfarbe:',
        'es': ':elfo_tono_de_piel_medio:',
        'fr': ':elfe_peau_légèrement_mate:',
        'ja': ':エルフ_中間の肌色:',
        'ko': ':엘프_갈색_피부:',
        'pt': ':elfo_pele_morena:',
        'it': ':elfo_carnagione_olivastra:',
        'fa': ':اِلف_پوست_طلایی:',
        'id': ':kurcaci_warna_kulit_sedang:',
        'zh': ':小精灵_中等肤色:',
        'ru': ':эльф_средний_тон_кожи:'
    },
    '\U0001FAB9': {  # 🪹
        'en': ':empty_nest:',
        'status': fully_qualified,
        'E': 14,
        'de': ':leeres_nest:',
        'es': ':nido_vacío:',
        'fr': ':nid_vide:',
        'ja': ':空っぽの巣:',
        'ko': ':빈_둥지:',
        'pt': ':ninho_vazio:',
        'it': ':nido_vuoto:',
        'fa': ':آشیانه_خالی:',
        'id': ':sarang_burung_kosong:',
        'zh': ':空巢:',
        'ru': ':пустое_гнездо:'
    },
    '\U0001F621': {  # 😡
        'en': ':enraged_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':rage:', ':pout:'],
        'de': ':schmollendes_gesicht:',
        'es': ':cara_cabreada:',
        'fr': ':visage_boudeur:',
        'ja': ':ふくれっ面:',
        'ko': ':뾰로통한_얼굴:',
        'pt': ':rosto_furioso:',
        'it': ':faccina_accigliata:',
        'fa': ':خشمگین:',
        'id': ':wajah_sangat_marah:',
        'zh': ':怒火中烧:',
        'ru': ':злится:'
    },
    '\U00002709\U0000FE0F': {  # ✉️
        'en': ':envelope:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':briefumschlag:',
        'es': ':sobre:',
        'fr': ':enveloppe:',
        'ja': ':封筒:',
        'ko': ':봉투:',
        'pt': ':envelope:',
        'it': ':busta:',
        'fa': ':پاکت_نامه:',
        'id': ':amplop:',
        'zh': ':信封:',
        'ru': ':конверт:'
    },
    '\U00002709': {  # ✉
        'en': ':envelope:',
        'status': unqualified,
        'E': 0.6,
        'variant': True,
        'de': ':briefumschlag:',
        'es': ':sobre:',
        'fr': ':enveloppe:',
        'ja': ':封筒:',
        'ko': ':봉투:',
        'pt': ':envelope:',
        'it': ':busta:',
        'fa': ':پاکت_نامه:',
        'id': ':amplop:',
        'zh': ':信封:',
        'ru': ':конверт:'
    },
    '\U0001F4E9': {  # 📩
        'en': ':envelope_with_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':umschlag_mit_pfeil:',
        'es': ':sobre_con_flecha:',
        'fr': ':enveloppe_avec_flèche:',
        'ja': ':メール受信:',
        'ko': ':보낸_편지:',
        'pt': ':envelope_com_seta:',
        'it': ':posta_in_uscita:',
        'fa': ':پاکت_نامه_با_پیکان:',
        'id': ':amplop_dengan_panah:',
        'zh': ':收邮件:',
        'ru': ':исходящее_письмо:'
    },
    '\U0001F4B6': {  # 💶
        'en': ':euro_banknote:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':euro:'],
        'de': ':euro-banknote:',
        'es': ':billete_de_euro:',
        'fr': ':billet_en_euros:',
        'ja': ':ユーロ札:',
        'ko': ':유로:',
        'pt': ':nota_de_euro:',
        'it': ':banconota_euro:',
        'fa': ':اسکناس_یورو:',
        'id': ':uang_kertas_euro:',
        'zh': ':欧元:',
        'ru': ':банкнота_евро:'
    },
    '\U0001F332': {  # 🌲
        'en': ':evergreen_tree:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nadelbaum:',
        'es': ':árbol_de_hoja_perenne:',
        'fr': ':conifère:',
        'ja': ':常緑樹:',
        'ko': ':소나무:',
        'pt': ':conífera:',
        'it': ':albero_sempreverde:',
        'fa': ':همیشه_سبز:',
        'id': ':pohon_hijau_abadi:',
        'zh': ':松树:',
        'ru': ':елка:'
    },
    '\U0001F411': {  # 🐑
        'en': ':ewe:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':sheep:'],
        'de': ':schaf:',
        'es': ':oveja:',
        'fr': ':mouton:',
        'ja': ':羊:',
        'ko': ':양:',
        'pt': ':ovelha:',
        'it': ':pecora:',
        'fa': ':میش:',
        'id': ':domba_betina:',
        'zh': ':母羊:',
        'ru': ':овца:'
    },
    '\U00002049\U0000FE0F': {  # ⁉️
        'en': ':exclamation_question_mark:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':interrobang:'],
        'variant': True,
        'de': ':ausrufe-_und_fragezeichen:',
        'es': ':exclamación_e_interrogación:',
        'fr': ':points_d’exclamation_et_d’interrogation:',
        'ja': ':感嘆符疑問符:',
        'ko': ':느낌표와_물음표:',
        'pt': ':exclamação_com_interrogação:',
        'it': ':punto_esclamativo_e_interrogativo:',
        'fa': ':علامت_تعجب_و_سؤال:',
        'id': ':tanda_seru_dan_tanya:',
        'zh': ':感叹疑问号:',
        'ru': ':восклицательныи_и_вопросительныи_знаки:'
    },
    '\U00002049': {  # ⁉
        'en': ':exclamation_question_mark:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':interrobang:'],
        'variant': True,
        'de': ':ausrufe-_und_fragezeichen:',
        'es': ':exclamación_e_interrogación:',
        'fr': ':points_d’exclamation_et_d’interrogation:',
        'ja': ':感嘆符疑問符:',
        'ko': ':느낌표와_물음표:',
        'pt': ':exclamação_com_interrogação:',
        'it': ':punto_esclamativo_e_interrogativo:',
        'fa': ':علامت_تعجب_و_سؤال:',
        'id': ':tanda_seru_dan_tanya:',
        'zh': ':感叹疑问号:',
        'ru': ':восклицательный_и_вопросительный_знаки:'
    },
    '\U0001F92F': {  # 🤯
        'en': ':exploding_head:',
        'status': fully_qualified,
        'E': 5,
        'de': ':explodierender_kopf:',
        'es': ':cabeza_explotando:',
        'fr': ':tête_qui_explose:',
        'ja': ':頭爆発:',
        'ko': ':폭발하는_얼굴:',
        'pt': ':cabeça_explodindo:',
        'it': ':testa_che_esplode:',
        'fa': ':سر_منفجرشده:',
        'id': ':kepala_meledak:',
        'zh': ':爆炸头:',
        'ru': ':взрыв_мозга:'
    },
    '\U0001F611': {  # 😑
        'en': ':expressionless_face:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':expressionless:'],
        'de': ':ausdrucksloses_gesicht:',
        'es': ':cara_sin_expresión:',
        'fr': ':visage_sans_expression:',
        'ja': ':無表情:',
        'ko': ':무표정한_얼굴:',
        'pt': ':rosto_inexpressivo:',
        'it': ':faccina_inespressiva:',
        'fa': ':صورت_بی_حالت:',
        'id': ':wajah_tanpa_ekspresi:',
        'zh': ':无语:',
        'ru': ':без_эмоций:'
    },
    '\U0001F441\U0000FE0F': {  # 👁️
        'en': ':eye:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':auge:',
        'es': ':ojo:',
        'fr': ':œil:',
        'ja': ':片目:',
        'ko': ':눈:',
        'pt': ':olho:',
        'it': ':occhio:',
        'fa': ':چشم:',
        'id': ':satu_mata:',
        'zh': ':眼睛:',
        'ru': ':глаз:'
    },
    '\U0001F441': {  # 👁
        'en': ':eye:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':auge:',
        'es': ':ojo:',
        'fr': ':œil:',
        'ja': ':片目:',
        'ko': ':눈:',
        'pt': ':olho:',
        'it': ':occhio:',
        'fa': ':چشم:',
        'id': ':satu_mata:',
        'zh': ':眼睛:',
        'ru': ':глаз:'
    },
    '\U0001F441\U0000FE0F\U0000200D\U0001F5E8\U0000FE0F': {  # 👁️‍🗨️
        'en': ':eye_in_speech_bubble:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':eye_speech_bubble:'],
        'de': ':auge_in_sprechblase:',
        'es': ':ojo_en_bocadillo_de_texto:',
        'fr': ':œil_dans_une_bulle_de_bd:',
        'ja': ':吹き出しの目:',
        'ko': ':말풍선_속_눈:',
        'pt': ':olho_no_balão_de_diálogo:',
        'it': ':occhio_nel_fumetto:',
        'fa': ':چشم_در_حباب_گفتار:',
        'id': ':mata_di_dalam_gelembung_ucapan:',
        'zh': ':眼睛对话框:',
        'ru': ':значок_свидетеля:'
    },
    '\U0001F441\U0000200D\U0001F5E8\U0000FE0F': {  # 👁‍🗨️
        'en': ':eye_in_speech_bubble:',
        'status': unqualified,
        'E': 2,
        'alias': [':eye_speech_bubble:'],
        'de': ':auge_in_sprechblase:',
        'es': ':ojo_en_bocadillo_de_texto:',
        'fr': ':œil_dans_une_bulle_de_bd:',
        'ja': ':吹き出しの目:',
        'ko': ':말풍선_속_눈:',
        'pt': ':olho_no_balão_de_diálogo:',
        'it': ':occhio_nel_fumetto:',
        'fa': ':چشم_در_حباب_گفتار:',
        'id': ':mata_di_dalam_gelembung_ucapan:',
        'zh': ':眼睛对话框:',
        'ru': ':значок_свидетеля:'
    },
    '\U0001F441\U0000FE0F\U0000200D\U0001F5E8': {  # 👁️‍🗨
        'en': ':eye_in_speech_bubble:',
        'status': minimally_qualified,
        'E': 2,
        'alias': [':eye_speech_bubble:'],
        'de': ':auge_in_sprechblase:',
        'es': ':ojo_en_bocadillo_de_texto:',
        'fr': ':œil_dans_une_bulle_de_bd:',
        'ja': ':吹き出しの目:',
        'ko': ':말풍선_속_눈:',
        'pt': ':olho_no_balão_de_diálogo:',
        'it': ':occhio_nel_fumetto:',
        'fa': ':چشم_در_حباب_گفتار:',
        'id': ':mata_di_dalam_gelembung_ucapan:',
        'zh': ':眼睛对话框:',
        'ru': ':значок_свидетеля:'
    },
    '\U0001F441\U0000200D\U0001F5E8': {  # 👁‍🗨
        'en': ':eye_in_speech_bubble:',
        'status': unqualified,
        'E': 2,
        'alias': [':eye_speech_bubble:'],
        'de': ':auge_in_sprechblase:',
        'es': ':ojo_en_bocadillo_de_texto:',
        'fr': ':œil_dans_une_bulle_de_bd:',
        'ja': ':吹き出しの目:',
        'ko': ':말풍선_속_눈:',
        'pt': ':olho_no_balão_de_diálogo:',
        'it': ':occhio_nel_fumetto:',
        'fa': ':چشم_در_حباب_گفتار:',
        'id': ':mata_di_dalam_gelembung_ucapan:',
        'zh': ':眼睛对话框:',
        'ru': ':значок_свидетеля:'
    },
    '\U0001F440': {  # 👀
        'en': ':eyes:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':augen:',
        'es': ':ojos:',
        'fr': ':yeux:',
        'ja': ':目:',
        'ko': ':왕눈이_눈알:',
        'pt': ':olhos:',
        'it': ':occhi:',
        'fa': ':دو_چشم:',
        'id': ':dua_mata:',
        'zh': ':双眼:',
        'ru': ':глаза:'
    },
    '\U0001F618': {  # 😘
        'en': ':face_blowing_a_kiss:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':kissing_heart:'],
        'de': ':kuss_zuwerfendes_gesicht:',
        'es': ':cara_lanzando_un_beso:',
        'fr': ':visage_envoyant_un_bisou:',
        'ja': ':投げキッス:',
        'ko': ':키스를_보내는_얼굴:',
        'pt': ':rosto_mandando_um_beijo:',
        'it': ':faccina_che_manda_un_bacio:',
        'fa': ':بوسه_فرستادن:',
        'id': ':wajah_memberikan_ciuman_jauh:',
        'zh': ':飞吻:',
        'ru': ':воздушный_поцелуй:'
    },
    '\U0001F62E\U0000200D\U0001F4A8': {  # 😮‍💨
        'en': ':face_exhaling:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':gesicht_das_ausatmet:',
        'es': ':cara_exhalando:',
        'fr': ':visage_expirant:',
        'ja': ':息を吐く顔:',
        'ko': ':날숨_쉬는_얼굴:',
        'pt': ':rosto_exalando:',
        'it': ':faccina_che_espira:',
        'fa': ':صورتک_در_حال_بازدم:',
        'id': ':wajah_mengembuskan_napas:',
        'zh': ':呼气:',
        'ru': ':выдыхает:'
    },
    '\U0001F979': {  # 🥹
        'en': ':face_holding_back_tears:',
        'status': fully_qualified,
        'E': 14,
        'de': ':gesicht_das_tränen_zurückhält:',
        'es': ':cara_aguantándose_las_lágrimas:',
        'fr': ':visage_retenant_ses_larmes:',
        'ja': ':涙をこらえた顔:',
        'ko': ':눈물을_참는_얼굴:',
        'pt': ':rosto_segurando_as_lágrimas:',
        'it': ':faccina_che_trattiene_le_lacrime:',
        'fa': ':چهره_با_چشمان_پراشک:',
        'id': ':wajah_menahan_air_mata:',
        'zh': ':忍住泪水:',
        'ru': ':еле_сдерживает_слезы:'
    },
    '\U0001F636\U0000200D\U0001F32B\U0000FE0F': {  # 😶‍🌫️
        'en': ':face_in_clouds:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':gesicht_in_wolken:',
        'es': ':cara_en_las_nubes:',
        'fr': ':visage_dans_les_nuages:',
        'ja': ':雲の中の顔:',
        'ko': ':공상에_잠긴_얼굴:',
        'pt': ':rosto_nas_nuvens:',
        'it': ':testa_tra_le_nuvole:',
        'fa': ':صورتک_میان_ابر:',
        'id': ':wajah_di_awan:',
        'zh': ':迷茫:',
        'ru': ':лицо_в_облаках:'
    },
    '\U0001F636\U0000200D\U0001F32B': {  # 😶‍🌫
        'en': ':face_in_clouds:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':gesicht_in_wolken:',
        'es': ':cara_en_las_nubes:',
        'fr': ':visage_dans_les_nuages:',
        'ja': ':雲の中の顔:',
        'ko': ':공상에_잠긴_얼굴:',
        'pt': ':rosto_nas_nuvens:',
        'it': ':testa_tra_le_nuvole:',
        'fa': ':صورتک_میان_ابر:',
        'id': ':wajah_di_awan:',
        'zh': ':迷茫:',
        'ru': ':лицо_в_облаках:'
    },
    '\U0001F60B': {  # 😋
        'en': ':face_savoring_food:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':yum:'],
        'de': ':sich_die_lippen_leckendes_gesicht:',
        'es': ':cara_saboreando_comida:',
        'fr': ':miam:',
        'ja': ':にこにこぺろり:',
        'ko': ':맛있는_음식을_음미하는_얼굴:',
        'pt': ':rosto_saboreando_comida:',
        'it': ':faccina_che_si_lecca_i_baffi:',
        'fa': ':صورت_با_زبان_آب_افتاده:',
        'id': ':wajah_menikmati:',
        'zh': ':好吃:',
        'ru': ':вкусно:'
    },
    '\U0001F631': {  # 😱
        'en': ':face_screaming_in_fear:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':scream:'],
        'de': ':vor_angst_schreiendes_gesicht:',
        'es': ':cara_gritando_de_miedo:',
        'fr': ':visage_qui_hurle_de_peur:',
        'ja': ':恐怖:',
        'ko': ':공포에_절규하는_얼굴:',
        'pt': ':rosto_gritando_de_medo:',
        'it': ':faccina_terrorizzata:',
        'fa': ':جیغ_ترس:',
        'id': ':wajah_sangat_ketakutan:',
        'zh': ':吓死了:',
        'ru': ':в_ужасе:'
    },
    '\U0001F92E': {  # 🤮
        'en': ':face_vomiting:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':vomiting_face:'],
        'de': ':kotzendes_gesicht:',
        'es': ':cara_vomitando:',
        'fr': ':visage_qui_vomit:',
        'ja': ':嘔吐する顔:',
        'ko': ':토하는_얼굴:',
        'pt': ':rosto_vomitando:',
        'it': ':faccina_che_vomita:',
        'fa': ':درحال_استفراغ:',
        'id': ':wajah_muntah:',
        'zh': ':呕吐:',
        'ru': ':рвота:'
    },
    '\U0001F635': {  # 😵
        'en': ':face_with_crossed-out_eyes:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':dizzy_face:', ':face_with_crossed_out_eyes:', ':knocked_out_face:'],
        'de': ':benommenes_gesicht:',
        'es': ':cara_mareada:',
        'fr': ':visage_étourdi:',
        'ja': ':めまい:',
        'ko': ':어지러운_얼굴:',
        'pt': ':rosto_atordoado:',
        'it': ':faccina_frastornata:',
        'fa': ':گیج:',
        'id': ':wajah_pusing:',
        'zh': ':晕头转向:',
        'ru': ':головокружение:'
    },
    '\U0001FAE4': {  # 🫤
        'en': ':face_with_diagonal_mouth:',
        'status': fully_qualified,
        'E': 14,
        'de': ':gesicht_mit_schrägem_mund:',
        'es': ':cara_con_boca_diagonal:',
        'fr': ':visage_avec_bouche_en_diagonale:',
        'ja': ':口が斜めの顔:',
        'ko': ':입이_한쪽으로_올라간_얼굴:',
        'pt': ':rosto_com_boca_diagonal:',
        'it': ':faccina_perplessa:',
        'fa': ':صورت_با_دهان_کج:',
        'id': ':wajah_dengan_mulut_diagonal:',
        'zh': ':郁闷:',
        'ru': ':с_перекошенным_ртом:'
    },
    '\U0001F92D': {  # 🤭
        'en': ':face_with_hand_over_mouth:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':hand_over_mouth:'],
        'de': ':verlegen_kicherndes_gesicht:',
        'es': ':cara_con_mano_sobre_la_boca:',
        'fr': ':visage_avec_une_main_sur_la_bouche:',
        'ja': ':口に手を当てた顔:',
        'ko': ':손으로_입을_가린_얼굴:',
        'pt': ':rosto_com_a_mão_sobre_a_boca:',
        'it': ':faccina_con_mano_sulla_bocca:',
        'fa': ':دست_روی_دهن:',
        'id': ':wajah_dengan_tangan_menutup_mulut:',
        'zh': ':不说:',
        'ru': ':прикрывает_рот_рукой:'
    },
    '\U0001F915': {  # 🤕
        'en': ':face_with_head-bandage:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':face_with_head_bandage:'],
        'de': ':gesicht_mit_kopfverband:',
        'es': ':cara_con_la_cabeza_vendada:',
        'fr': ':visage_avec_bandage_autour_de_la_tête:',
        'ja': ':包帯を巻いた顔:',
        'ko': ':머리에_붕대를_감은_얼굴:',
        'pt': ':rosto_com_atadura_na_cabeça:',
        'it': ':faccina_con_la_testa_bendata:',
        'fa': ':سر_باندپیچی_شده:',
        'id': ':wajah_diperban_di_kepala:',
        'zh': ':受伤:',
        'ru': ':с_перевязанной_головой:'
    },
    '\U0001F637': {  # 😷
        'en': ':face_with_medical_mask:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':mask:'],
        'de': ':gesicht_mit_atemschutzmaske:',
        'es': ':cara_con_mascarilla_médica:',
        'fr': ':visage_avec_masque:',
        'ja': ':マスク顔:',
        'ko': ':마스크_낀_얼굴:',
        'pt': ':rosto_com_máscara_médica:',
        'it': ':faccina_con_mascherina:',
        'fa': ':صورت_با_ماسک_پزشکی:',
        'id': ':wajah_bermasker:',
        'zh': ':感冒:',
        'ru': ':в_медицинской_маске:'
    },
    '\U0001F9D0': {  # 🧐
        'en': ':face_with_monocle:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':monocle_face:'],
        'de': ':gesicht_mit_monokel:',
        'es': ':cara_con_monóculo:',
        'fr': ':visage_avec_un_monocle:',
        'ja': ':モノクルを付けた顔:',
        'ko': ':단안경을_쓴_얼굴:',
        'pt': ':rosto_com_monóculo:',
        'it': ':faccina_con_monocolo:',
        'fa': ':صورت_با_عینک_یک_چشمی:',
        'id': ':wajah_dengan_monokel:',
        'zh': ':带单片眼镜的脸:',
        'ru': ':с_моноклем:'
    },
    '\U0001FAE2': {  # 🫢
        'en': ':face_with_open_eyes_and_hand_over_mouth:',
        'status': fully_qualified,
        'E': 14,
        'de': ':gesicht_mit_offenen_augen_und_hand_über_dem_mund:',
        'es': ':cara_con_ojos_abiertos_y_boca_tapada:',
        'fr': ':visage_avec_yeux_ouverts_et_main_sur_la_bouche:',
        'ja': ':目を開けて口に手を当てた顔:',
        'ko': ':눈을_뜨고_손으로_입을_가린_얼굴:',
        'pt': ':rosto_com_olhos_abertos_e_mão_sobre_a_boca:',
        'it': ':faccina_con_occhi_aperti_e_mano_sulla_bocca:',
        'fa': ':چهره_با_چشمان_باز_و_دست_روی_دهان:',
        'id': ':wajah_dengan_mata_terbuka_dan_tangan_menutupi_mulut:',
        'zh': ':睁眼捂嘴:',
        'ru': ':прикрывает_рот_рукой_с_открытыми_глазами:'
    },
    '\U0001F62E': {  # 😮
        'en': ':face_with_open_mouth:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':open_mouth:'],
        'de': ':gesicht_mit_offenem_mund:',
        'es': ':cara_con_la_boca_abierta:',
        'fr': ':visage_avec_bouche_ouverte:',
        'ja': ':口が開いた顔:',
        'ko': ':입벌린_얼굴:',
        'pt': ':rosto_com_boca_aberta:',
        'it': ':faccina_con_bocca_aperta:',
        'fa': ':دهان_باز_از_تعجب:',
        'id': ':wajah_dengan_mulut_terbuka:',
        'zh': ':吃惊:',
        'ru': ':удивление:'
    },
    '\U0001FAE3': {  # 🫣
        'en': ':face_with_peeking_eye:',
        'status': fully_qualified,
        'E': 14,
        'de': ':gesicht_mit_durch_die_finger_linsendem_auge:',
        'es': ':cara_tapada_con_ojo_espiando:',
        'fr': ':visage_qui_regarde_entre_ses_doigts:',
        'ja': ':指の間からのぞき見る顔:',
        'ko': ':한쪽_눈을_가리고_훔쳐보는_얼굴:',
        'pt': ':rosto_com_olho_espiando:',
        'it': ':faccina_che_sbircia_tra_le_dita:',
        'fa': ':چهره_با_نگاه_دزدکی:',
        'id': ':wajah_dengan_mata_mengintip:',
        'zh': ':偷看:',
        'ru': ':подсматривает:'
    },
    '\U0001F928': {  # 🤨
        'en': ':face_with_raised_eyebrow:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':raised_eyebrow:'],
        'de': ':gesicht_mit_hochgezogenen_augenbrauen:',
        'es': ':cara_con_ceja_alzada:',
        'fr': ':visage_avec_les_sourcils_relevés:',
        'ja': ':眉を上げた顔:',
        'ko': ':눈썹을_치켜올린_얼굴:',
        'pt': ':rosto_com_sobrancelha_levantada:',
        'it': ':faccia_con_sopracciglia_alzate:',
        'fa': ':صورت_با_ابروی_بالاداده:',
        'id': ':wajah_dengan_alis_terangkat:',
        'zh': ':挑眉:',
        'ru': ':не_верит:'
    },
    '\U0001F644': {  # 🙄
        'en': ':face_with_rolling_eyes:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':roll_eyes:'],
        'de': ':augen_verdrehendes_gesicht:',
        'es': ':cara_con_ojos_en_blanco:',
        'fr': ':visage_roulant_des_yeux:',
        'ja': ':上を見る顔:',
        'ko': ':눈을_굴리고_있는_얼굴:',
        'pt': ':rosto_com_olhos_revirados:',
        'it': ':faccina_con_occhi_al_cielo:',
        'fa': ':نگاه_کلافه:',
        'id': ':wajah_dengan_mata_ke_atas:',
        'zh': ':翻白眼:',
        'ru': ':закатывает_глаза:'
    },
    '\U0001F635\U0000200D\U0001F4AB': {  # 😵‍💫
        'en': ':face_with_spiral_eyes:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':gesicht_mit_spiralen_als_augen:',
        'es': ':cara_con_ojos_de_espiral:',
        'fr': ':visage_aux_yeux_en_spirales:',
        'ja': ':目を回した顔:',
        'ko': ':현기증_난_얼굴:',
        'pt': ':rosto_com_olhos_em_espiral:',
        'it': ':faccina_con_occhi_a_spirale:',
        'fa': ':صورتک_با_چشمان_چرخشی:',
        'id': ':wajah_dengan_mata_berputar:',
        'zh': ':晕:',
        'ru': ':ошеломление:'
    },
    '\U0001F624': {  # 😤
        'en': ':face_with_steam_from_nose:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':triumph:'],
        'de': ':schnaubendes_gesicht:',
        'es': ':cara_resoplando:',
        'fr': ':visage_avec_fumée_sortant_des_narines:',
        'ja': ':勝ち誇った顔:',
        'ko': ':화내며_씩씩거리는_얼굴:',
        'pt': ':rosto_soltando_vapor_pelo_nariz:',
        'it': ':faccina_che_sbuffa:',
        'fa': ':ژست_برندگی_با_دلخوری:',
        'id': ':wajah_bersungguh-sungguh:',
        'zh': ':傲慢:',
        'ru': ':в_ожидании_успеха:'
    },
    '\U0001F92C': {  # 🤬
        'en': ':face_with_symbols_on_mouth:',
        'status': fully_qualified,
        'E': 5,
        'alias': [':cursing_face:'],
        'de': ':gesicht_mit_symbolen_über_dem_mund:',
        'es': ':cara_con_símbolos_en_la_boca:',
        'fr': ':visage_avec_des_symboles_dans_la_bouche:',
        'ja': ':ののしる:',
        'ko': ':욕하는_얼굴:',
        'pt': ':rosto_com_símbolos_na_boca:',
        'it': ':faccina_con_simboli_sulla_bocca:',
        'fa': ':صورت_با_نمادهایی_روی_دهان:',
        'id': ':wajah_dengan_simbol_pada_mulut:',
        'zh': ':嘴上有符号的脸:',
        'ru': ':цензура:'
    },
    '\U0001F602': {  # 😂
        'en': ':face_with_tears_of_joy:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':joy:'],
        'de': ':gesicht_mit_freudentränen:',
        'es': ':cara_llorando_de_risa:',
        'fr': ':visage_riant_aux_larmes:',
        'ja': ':嬉し泣き:',
        'ko': ':기쁨의_눈물을_흘리는_얼굴:',
        'pt': ':rosto_chorando_de_rir:',
        'it': ':faccina_con_lacrime_di_gioia:',
        'fa': ':روده_بر:',
        'id': ':wajah_gembira_berurai_air_mata:',
        'zh': ':笑哭了:',
        'ru': ':смеется_до_слез:'
    },
    '\U0001F912': {  # 🤒
        'en': ':face_with_thermometer:',
        'status': fully_qualified,
        'E': 1,
        'de': ':gesicht_mit_fieberthermometer:',
        'es': ':cara_con_termómetro:',
        'fr': ':visage_avec_thermomètre:',
        'ja': ':熱がある顔:',
        'ko': ':체온계를_물고_있는_얼굴:',
        'pt': ':rosto_com_termômetro:',
        'it': ':faccina_con_termometro:',
        'fa': ':تب_دار:',
        'id': ':wajah_bertermometer:',
        'zh': ':发烧:',
        'ru': ':с_градусником_во_рту:'
    },
    '\U0001F61B': {  # 😛
        'en': ':face_with_tongue:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':stuck_out_tongue:'],
        'de': ':gesicht_mit_herausgestreckter_zunge:',
        'es': ':cara_sacando_la_lengua:',
        'fr': ':visage_qui_tire_la_langue:',
        'ja': ':舌を出した顔:',
        'ko': ':혀를_내민_얼굴:',
        'pt': ':rosto_mostrando_a_língua:',
        'it': ':faccina_che_mostra_la_lingua:',
        'fa': ':زبونک:',
        'id': ':wajah_menjulurkan_lidah:',
        'zh': ':吐舌:',
        'ru': ':показывает_язык:'
    },
    '\U0001F636': {  # 😶
        'en': ':face_without_mouth:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':no_mouth:'],
        'de': ':gesicht_ohne_mund:',
        'es': ':cara_sin_boca:',
        'fr': ':visage_sans_bouche:',
        'ja': ':口のない顔:',
        'ko': ':입이_없는_얼굴:',
        'pt': ':rosto_sem_boca:',
        'it': ':faccina_senza_bocca:',
        'fa': ':صورت_بدون_لب:',
        'id': ':wajah_tanpa_mulut:',
        'zh': ':沉默:',
        'ru': ':лицо_без_рта:'
    },
    '\U0001F3ED': {  # 🏭
        'en': ':factory:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':fabrik:',
        'es': ':fábrica:',
        'fr': ':usine:',
        'ja': ':工場:',
        'ko': ':공장:',
        'pt': ':fábrica:',
        'it': ':fabbrica:',
        'fa': ':کارخانه:',
        'id': ':pabrik:',
        'zh': ':工厂:',
        'ru': ':завод:'
    },
    '\U0001F9D1\U0000200D\U0001F3ED': {  # 🧑‍🏭
        'en': ':factory_worker:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':fabrikarbeiter(in):',
        'es': ':profesional_industrial:',
        'fr': ':ouvrier_(tous_genres):',
        'ja': ':溶接工:',
        'ko': ':공장_직원:',
        'pt': ':funcionário_de_fábrica:',
        'it': ':persona_che_lavora_in_fabbrica:',
        'fa': ':کارگر_کارخانه:',
        'id': ':pekerja_pabrik:',
        'zh': ':工人:',
        'ru': ':рабочий:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U0001F3ED': {  # 🧑🏿‍🏭
        'en': ':factory_worker_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':fabrikarbeiter(in)_dunkle_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_oscuro:',
        'fr': ':ouvrier_(tous_genres)_peau_foncée:',
        'ja': ':溶接工_濃い肌色:',
        'ko': ':공장_직원_검은색_피부:',
        'pt': ':funcionário_de_fábrica_pele_escura:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_scura:',
        'fa': ':کارگر_کارخانه_پوست_آبنوسی:',
        'id': ':pekerja_pabrik_warna_kulit_gelap:',
        'zh': ':工人_较深肤色:',
        'ru': ':рабочий_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U0001F3ED': {  # 🧑🏻‍🏭
        'en': ':factory_worker_light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':fabrikarbeiter(in)_helle_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_claro:',
        'fr': ':ouvrier_(tous_genres)_peau_claire:',
        'ja': ':溶接工_薄い肌色:',
        'ko': ':공장_직원_하얀_피부:',
        'pt': ':funcionário_de_fábrica_pele_clara:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_chiara:',
        'fa': ':کارگر_کارخانه_پوست_سفید:',
        'id': ':pekerja_pabrik_warna_kulit_cerah:',
        'zh': ':工人_较浅肤色:',
        'ru': ':рабочий_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U0001F3ED': {  # 🧑🏾‍🏭
        'en': ':factory_worker_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':fabrikarbeiter(in)_mitteldunkle_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_oscuro_medio:',
        'fr': ':ouvrier_(tous_genres)_peau_mate:',
        'ja': ':溶接工_やや濃い肌色:',
        'ko': ':공장_직원_진한_갈색_피부:',
        'pt': ':funcionário_de_fábrica_pele_morena_escura:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_abbastanza_scura:',
        'fa': ':کارگر_کارخانه_پوست_گندمی:',
        'id': ':pekerja_pabrik_warna_kulit_gelap-sedang:',
        'zh': ':工人_中等深肤色:',
        'ru': ':рабочий_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U0001F3ED': {  # 🧑🏼‍🏭
        'en': ':factory_worker_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':fabrikarbeiter(in)_mittelhelle_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_claro_medio:',
        'fr': ':ouvrier_(tous_genres)_peau_moyennement_claire:',
        'ja': ':溶接工_やや薄い肌色:',
        'ko': ':공장_직원_연한_갈색_피부:',
        'pt': ':funcionário_de_fábrica_pele_morena_clara:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_abbastanza_chiara:',
        'fa': ':کارگر_کارخانه_پوست_روشن:',
        'id': ':pekerja_pabrik_warna_kulit_cerah-sedang:',
        'zh': ':工人_中等浅肤色:',
        'ru': ':рабочий_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U0001F3ED': {  # 🧑🏽‍🏭
        'en': ':factory_worker_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':fabrikarbeiter(in)_mittlere_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_medio:',
        'fr': ':ouvrier_(tous_genres)_peau_légèrement_mate:',
        'ja': ':溶接工_中間の肌色:',
        'ko': ':공장_직원_갈색_피부:',
        'pt': ':funcionário_de_fábrica_pele_morena:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_olivastra:',
        'fa': ':کارگر_کارخانه_پوست_طلایی:',
        'id': ':pekerja_pabrik_warna_kulit_sedang:',
        'zh': ':工人_中等肤色:',
        'ru': ':рабочий_средний_тон_кожи:'
    },
    '\U0001F9DA': {  # 🧚
        'en': ':fairy:',
        'status': fully_qualified,
        'E': 5,
        'de': ':märchenfee:',
        'es': ':hada:',
        'fr': ':personnage_féérique:',
        'ja': ':妖精:',
        'ko': ':요정:',
        'pt': ':fada:',
        'it': ':fata:',
        'fa': ':پری:',
        'id': ':peri:',
        'zh': ':精灵:',
        'ru': ':фея:'
    },
    '\U0001F9DA\U0001F3FF': {  # 🧚🏿
        'en': ':fairy_dark_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':märchenfee_dunkle_hautfarbe:',
        'es': ':hada_tono_de_piel_oscuro:',
        'fr': ':personnage_féérique_peau_foncée:',
        'ja': ':妖精_濃い肌色:',
        'ko': ':요정_검은색_피부:',
        'pt': ':fada_pele_escura:',
        'it': ':fata_carnagione_scura:',
        'fa': ':پری_پوست_آبنوسی:',
        'id': ':peri_warna_kulit_gelap:',
        'zh': ':精灵_较深肤色:',
        'ru': ':фея_очень_темный_тон_кожи:'
    },
    '\U0001F9DA\U0001F3FB': {  # 🧚🏻
        'en': ':fairy_light_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':märchenfee_helle_hautfarbe:',
        'es': ':hada_tono_de_piel_claro:',
        'fr': ':personnage_féérique_peau_claire:',
        'ja': ':妖精_薄い肌色:',
        'ko': ':요정_하얀_피부:',
        'pt': ':fada_pele_clara:',
        'it': ':fata_carnagione_chiara:',
        'fa': ':پری_پوست_سفید:',
        'id': ':peri_warna_kulit_cerah:',
        'zh': ':精灵_较浅肤色:',
        'ru': ':фея_очень_светлый_тон_кожи:'
    },
    '\U0001F9DA\U0001F3FE': {  # 🧚🏾
        'en': ':fairy_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':märchenfee_mitteldunkle_hautfarbe:',
        'es': ':hada_tono_de_piel_oscuro_medio:',
        'fr': ':personnage_féérique_peau_mate:',
        'ja': ':妖精_やや濃い肌色:',
        'ko': ':요정_진한_갈색_피부:',
        'pt': ':fada_pele_morena_escura:',
        'it': ':fata_carnagione_abbastanza_scura:',
        'fa': ':پری_پوست_گندمی:',
        'id': ':peri_warna_kulit_gelap-sedang:',
        'zh': ':精灵_中等深肤色:',
        'ru': ':фея_темный_тон_кожи:'
    },
    '\U0001F9DA\U0001F3FC': {  # 🧚🏼
        'en': ':fairy_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':märchenfee_mittelhelle_hautfarbe:',
        'es': ':hada_tono_de_piel_claro_medio:',
        'fr': ':personnage_féérique_peau_moyennement_claire:',
        'ja': ':妖精_やや薄い肌色:',
        'ko': ':요정_연한_갈색_피부:',
        'pt': ':fada_pele_morena_clara:',
        'it': ':fata_carnagione_abbastanza_chiara:',
        'fa': ':پری_پوست_روشن:',
        'id': ':peri_warna_kulit_cerah-sedang:',
        'zh': ':精灵_中等浅肤色:',
        'ru': ':фея_светлый_тон_кожи:'
    },
    '\U0001F9DA\U0001F3FD': {  # 🧚🏽
        'en': ':fairy_medium_skin_tone:',
        'status': fully_qualified,
        'E': 5,
        'de': ':märchenfee_mittlere_hautfarbe:',
        'es': ':hada_tono_de_piel_medio:',
        'fr': ':personnage_féérique_peau_légèrement_mate:',
        'ja': ':妖精_中間の肌色:',
        'ko': ':요정_갈색_피부:',
        'pt': ':fada_pele_morena:',
        'it': ':fata_carnagione_olivastra:',
        'fa': ':پری_پوست_طلایی:',
        'id': ':peri_warna_kulit_sedang:',
        'zh': ':精灵_中等肤色:',
        'ru': ':фея_средний_тон_кожи:'
    },
    '\U0001F9C6': {  # 🧆
        'en': ':falafel:',
        'status': fully_qualified,
        'E': 12,
        'de': ':falafel:',
        'es': ':falafel:',
        'fr': ':falafels:',
        'ja': ':ファラフェル:',
        'ko': ':팔라펠:',
        'pt': ':falafel:',
        'it': ':falafel:',
        'fa': ':فلافل:',
        'id': ':falafel:',
        'zh': ':炸豆丸子:',
        'ru': ':фалафель:'
    },
    '\U0001F342': {  # 🍂
        'en': ':fallen_leaf:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':laub:',
        'es': ':hojas_caídas:',
        'fr': ':feuille_morte:',
        'ja': ':落ち葉:',
        'ko': ':낙엽:',
        'pt': ':folhas_caídas:',
        'it': ':foglia_caduta:',
        'fa': ':برگ_ریزان:',
        'id': ':daun_berguguran:',
        'zh': ':落叶:',
        'ru': ':падающие_листья:'
    },
    '\U0001F46A': {  # 👪
        'en': ':family:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':familie:',
        'es': ':familia:',
        'fr': ':famille:',
        'ja': ':家族:',
        'ko': ':가족:',
        'pt': ':família:',
        'it': ':famiglia:',
        'fa': ':خانواده:',
        'id': ':keluarga:',
        'zh': ':家庭:',
        'ru': ':семья:'
    },
    '\U0001F468\U0000200D\U0001F466': {  # 👨‍👦
        'en': ':family_man_boy:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_mann_junge:',
        'es': ':familia_hombre_y_niño:',
        'fr': ':famille_homme_et_garçon:',
        'ja': ':家族_男性_男の子:',
        'ko': ':가족_남자_남자_아이:',
        'pt': ':família_homem_e_menino:',
        'it': ':famiglia_uomo_e_bambino:',
        'fa': ':خانواده_مرد_پسر:',
        'id': ':keluarga_pria_anak_laki-laki:',
        'zh': ':家庭_男人男孩:',
        'ru': ':семья_мужчина_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466': {  # 👨‍👦‍👦
        'en': ':family_man_boy_boy:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_mann_junge_und_junge:',
        'es': ':familia_hombre_niño_niño:',
        'fr': ':famille_homme_garçon_et_garçon:',
        'ja': ':家族_男性_男の子_男の子:',
        'ko': ':가족_남자_남자_아이_남자_아이:',
        'pt': ':família_homem_menino_e_menino:',
        'it': ':famiglia_uomo_bambino_e_bambino:',
        'fa': ':خانواده_مرد_پسر_و_پسر:',
        'id': ':keluarga_pria_anak_laki-laki_anak_laki-laki:',
        'zh': ':家庭_男人男孩男孩:',
        'ru': ':семья_мужчина_мальчик_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F467': {  # 👨‍👧
        'en': ':family_man_girl:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_mann_mädchen:',
        'es': ':familia_hombre_y_niña:',
        'fr': ':famille_homme_et_fille:',
        'ja': ':家族_男性_女の子:',
        'ko': ':가족_남자_여자_아이:',
        'pt': ':família_homem_e_menina:',
        'it': ':famiglia_uomo_e_bambina:',
        'fa': ':خانواده_مرد_دختر:',
        'id': ':keluarga_pria_anak_perempuan:',
        'zh': ':家庭_男人女孩:',
        'ru': ':семья_мужчина_девочка:'
    },
    '\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466': {  # 👨‍👧‍👦
        'en': ':family_man_girl_boy:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_mann_mädchen_und_junge:',
        'es': ':familia_hombre_niña_niño:',
        'fr': ':famille_homme_fille_et_garçon:',
        'ja': ':家族_男性_女の子_男の子:',
        'ko': ':가족_남자_여자_아이_남자_아이:',
        'pt': ':família_homem_menina_e_menino:',
        'it': ':famiglia_uomo_bambina_e_bambino:',
        'fa': ':خانواده_مرد_دختر_و_پسر:',
        'id': ':keluarga_pria_anak_perempuan_anak_laki-laki:',
        'zh': ':家庭_男人女孩男孩:',
        'ru': ':семья_мужчина_девочка_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467': {  # 👨‍👧‍👧
        'en': ':family_man_girl_girl:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_mann_mädchen_und_mädchen:',
        'es': ':familia_hombre_niña_niña:',
        'fr': ':famille_homme_fille_et_fille:',
        'ja': ':家族_男性_女の子_女の子:',
        'ko': ':가족_남자_여자_아이_여자_아이:',
        'pt': ':família_homem_menina_e_menina:',
        'it': ':famiglia_uomo_bambina_e_bambina:',
        'fa': ':خانواده_مرد_دختر_و_دختر:',
        'id': ':keluarga_pria_anak_perempuan_anak_perempuan:',
        'zh': ':家庭_男人女孩女孩:',
        'ru': ':семья_мужчина_девочка_девочка:'
    },
    '\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466': {  # 👨‍👨‍👦
        'en': ':family_man_man_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_mann_und_junge:',
        'es': ':familia_hombre_hombre_niño:',
        'fr': ':famille_homme_homme_et_garçon:',
        'ja': ':家族_男性_男性_男の子:',
        'ko': ':가족_남자_남자_남자_아이:',
        'pt': ':família_homem_homem_e_menino:',
        'it': ':famiglia_uomo_uomo_e_bambino:',
        'fa': ':خانواده_مرد_مرد_و_پسر:',
        'id': ':keluarga_pria_pria_anak_laki-laki:',
        'zh': ':家庭_男人男人男孩:',
        'ru': ':семья_мужчина_мужчина_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466': {  # 👨‍👨‍👦‍👦
        'en': ':family_man_man_boy_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_mann_junge_und_junge:',
        'es': ':familia_hombre_hombre_niño_niño:',
        'fr': ':famille_homme_homme_garçon_et_garçon:',
        'ja': ':家族_男性_男性_男の子_男の子:',
        'ko': ':가족_남자_남자_남자_아이_남자_아이:',
        'pt': ':família_homem_homem_menino_e_menino:',
        'it': ':famiglia_uomo_uomo_bambino_e_bambino:',
        'fa': ':خانواده_مرد_مرد_پسر_و_پسر:',
        'id': ':keluarga_pria_pria_anak_laki-laki_anak_laki-laki:',
        'zh': ':家庭_男人男人男孩男孩:',
        'ru': ':семья_мужчина_мужчина_мальчик_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467': {  # 👨‍👨‍👧
        'en': ':family_man_man_girl:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_mann_und_mädchen:',
        'es': ':familia_hombre_hombre_niña:',
        'fr': ':famille_homme_homme_et_fille:',
        'ja': ':家族_男性_男性_女の子:',
        'ko': ':가족_남자_남자_여자_아이:',
        'pt': ':família_homem_homem_e_menina:',
        'it': ':famiglia_uomo_uomo_e_bambina:',
        'fa': ':خانواده_مرد_مرد_و_دختر:',
        'id': ':keluarga_pria_pria_anak_perempuan:',
        'zh': ':家庭_男人男人女孩:',
        'ru': ':семья_мужчина_мужчина_девочка:'
    },
    '\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466': {  # 👨‍👨‍👧‍👦
        'en': ':family_man_man_girl_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_mann_mädchen_und_junge:',
        'es': ':familia_hombre_hombre_niña_niño:',
        'fr': ':famille_homme_homme_fille_et_garçon:',
        'ja': ':家族_男性_男性_女の子_男の子:',
        'ko': ':가족_남자_남자_여자_아이_남자_아이:',
        'pt': ':família_homem_homem_menina_e_menino:',
        'it': ':famiglia_uomo_uomo_bambina_e_bambino:',
        'fa': ':خانواده_مرد_مرد_دختر_و_پسر:',
        'id': ':keluarga_pria_pria_anak_perempuan_anak_laki-laki:',
        'zh': ':家庭_男人男人女孩男孩:',
        'ru': ':семья_мужчина_мужчина_девочка_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467': {  # 👨‍👨‍👧‍👧
        'en': ':family_man_man_girl_girl:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_mann_mädchen_und_mädchen:',
        'es': ':familia_hombre_hombre_niña_niña:',
        'fr': ':famille_homme_homme_fille_et_fille:',
        'ja': ':家族_男性_男性_女の子_女の子:',
        'ko': ':가족_남자_남자_여자_아이_여자_아이:',
        'pt': ':família_homem_homem_menina_e_menina:',
        'it': ':famiglia_uomo_uomo_bambina_e_bambina:',
        'fa': ':خانواده_مرد_مرد_دختر_و_دختر:',
        'id': ':keluarga_pria_pria_anak_perempuan_anak_perempuan:',
        'zh': ':家庭_男人男人女孩女孩:',
        'ru': ':семья_мужчина_мужчина_девочка_девочка:'
    },
    '\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466': {  # 👨‍👩‍👦
        'en': ':family_man_woman_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_frau_und_junge:',
        'es': ':familia_hombre_mujer_niño:',
        'fr': ':famille_homme_femme_et_garçon:',
        'ja': ':家族_男性_女性_男の子:',
        'ko': ':가족_남자_여자_남자_아이:',
        'pt': ':família_homem_mulher_e_menino:',
        'it': ':famiglia_uomo_donna_e_bambino:',
        'fa': ':خانواده_مرد_زن_و_پسر:',
        'id': ':keluarga_pria_wanita_anak_laki-laki:',
        'zh': ':家庭_男人女人男孩:',
        'ru': ':семья_мужчина_женщина_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': {  # 👨‍👩‍👦‍👦
        'en': ':family_man_woman_boy_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_frau_junge_und_junge:',
        'es': ':familia_hombre_mujer_niño_niño:',
        'fr': ':famille_homme_femme_garçon_et_garçon:',
        'ja': ':家族_男性_女性_男の子_男の子:',
        'ko': ':가족_남자_여자_남자_아이_남자_아이:',
        'pt': ':família_homem_mulher_menino_e_menino:',
        'it': ':famiglia_uomo_donna_bambino_e_bambino:',
        'fa': ':خانواده_مرد_زن_پسر_و_پسر:',
        'id': ':keluarga_pria_wanita_anak_laki-laki_anak_laki-laki:',
        'zh': ':家庭_男人女人男孩男孩:',
        'ru': ':семья_мужчина_женщина_мальчик_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467': {  # 👨‍👩‍👧
        'en': ':family_man_woman_girl:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_frau_und_mädchen:',
        'es': ':familia_hombre_mujer_niña:',
        'fr': ':famille_homme_femme_et_fille:',
        'ja': ':家族_男性_女性_女の子:',
        'ko': ':가족_남자_여자_여자_아이:',
        'pt': ':família_homem_mulher_e_menina:',
        'it': ':famiglia_uomo_donna_e_bambina:',
        'fa': ':خانواده_مرد_زن_و_دختر:',
        'id': ':keluarga_pria_wanita_anak_perempuan:',
        'zh': ':家庭_男人女人女孩:',
        'ru': ':семья_мужчина_женщина_девочка:'
    },
    '\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': {  # 👨‍👩‍👧‍👦
        'en': ':family_man_woman_girl_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_frau_mädchen_und_junge:',
        'es': ':familia_hombre_mujer_niña_niño:',
        'fr': ':famille_homme_femme_fille_et_garçon:',
        'ja': ':家族_男性_女性_女の子_男の子:',
        'ko': ':가족_남자_여자_여자_아이_남자_아이:',
        'pt': ':família_homem_mulher_menina_e_menino:',
        'it': ':famiglia_uomo_donna_bambina_e_bambino:',
        'fa': ':خانواده_مرد_زن_دختر_و_پسر:',
        'id': ':keluarga_pria_wanita_anak_perempuan_anak_laki-laki:',
        'zh': ':家庭_男人女人女孩男孩:',
        'ru': ':семья_мужчина_женщина_девочка_мальчик:'
    },
    '\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': {  # 👨‍👩‍👧‍👧
        'en': ':family_man_woman_girl_girl:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_mann_frau_mädchen_und_mädchen:',
        'es': ':familia_hombre_mujer_niña_niña:',
        'fr': ':famille_homme_femme_fille_et_fille:',
        'ja': ':家族_男性_女性_女の子_女の子:',
        'ko': ':가족_남자_여자_여자_아이_여자_아이:',
        'pt': ':família_homem_mulher_menina_e_menina:',
        'it': ':famiglia_uomo_donna_bambina_e_bambina:',
        'fa': ':خانواده_مرد_زن_دختر_و_دختر:',
        'id': ':keluarga_pria_wanita_anak_perempuan_anak_perempuan:',
        'zh': ':家庭_男人女人女孩女孩:',
        'ru': ':семья_мужчина_женщина_девочка_девочка:'
    },
    '\U0001F469\U0000200D\U0001F466': {  # 👩‍👦
        'en': ':family_woman_boy:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_frau_junge:',
        'es': ':familia_mujer_y_niño:',
        'fr': ':famille_femme_et_garçon:',
        'ja': ':家族_女性_男の子:',
        'ko': ':가족_여자_남자_아이:',
        'pt': ':família_mulher_e_menino:',
        'it': ':famiglia_donna_e_bambino:',
        'fa': ':خانواده_زن_پسر:',
        'id': ':keluarga_wanita_anak_laki-laki:',
        'zh': ':家庭_女人男孩:',
        'ru': ':семья_женщина_мальчик:'
    },
    '\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': {  # 👩‍👦‍👦
        'en': ':family_woman_boy_boy:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_frau_junge_und_junge:',
        'es': ':familia_mujer_niño_niño:',
        'fr': ':famille_femme_garçon_et_garçon:',
        'ja': ':家族_女性_男の子_男の子:',
        'ko': ':가족_여자_남자_아이_남자_아이:',
        'pt': ':família_mulher_menino_e_menino:',
        'it': ':famiglia_donna_bambino_e_bambino:',
        'fa': ':خانواده_زن_پسر_و_پسر:',
        'id': ':keluarga_wanita_anak_laki-laki_anak_laki-laki:',
        'zh': ':家庭_女人男孩男孩:',
        'ru': ':семья_женщина_мальчик_мальчик:'
    },
    '\U0001F469\U0000200D\U0001F467': {  # 👩‍👧
        'en': ':family_woman_girl:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_frau_mädchen:',
        'es': ':familia_mujer_y_niña:',
        'fr': ':famille_femme_et_fille:',
        'ja': ':家族_女性_女の子:',
        'ko': ':가족_여자_여자_아이:',
        'pt': ':família_mulher_e_menina:',
        'it': ':famiglia_donna_e_bambina:',
        'fa': ':خانواده_زن_دختر:',
        'id': ':keluarga_wanita_anak_perempuan:',
        'zh': ':家庭_女人女孩:',
        'ru': ':семья_женщина_девочка:'
    },
    '\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': {  # 👩‍👧‍👦
        'en': ':family_woman_girl_boy:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_frau_mädchen_und_junge:',
        'es': ':familia_mujer_niña_niño:',
        'fr': ':famille_femme_fille_et_garçon:',
        'ja': ':家族_女性_女の子_男の子:',
        'ko': ':가족_여자_여자_아이_남자_아이:',
        'pt': ':família_mulher_menina_e_menino:',
        'it': ':famiglia_donna_bambina_e_bambino:',
        'fa': ':خانواده_زن_دختر_و_پسر:',
        'id': ':keluarga_wanita_anak_perempuan_anak_laki-laki:',
        'zh': ':家庭_女人女孩男孩:',
        'ru': ':семья_женщина_девочка_мальчик:'
    },
    '\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': {  # 👩‍👧‍👧
        'en': ':family_woman_girl_girl:',
        'status': fully_qualified,
        'E': 4,
        'de': ':familie_frau_mädchen_und_mädchen:',
        'es': ':familia_mujer_niña_niña:',
        'fr': ':famille_femme_fille_et_fille:',
        'ja': ':家族_女性_女の子_女の子:',
        'ko': ':가족_여자_여자_아이_여자_아이:',
        'pt': ':família_mulher_menina_e_menina:',
        'it': ':famiglia_donna_bambina_e_bambina:',
        'fa': ':خانواده_زن_دختر_و_دختر:',
        'id': ':keluarga_wanita_anak_perempuan_anak_perempuan:',
        'zh': ':家庭_女人女孩女孩:',
        'ru': ':семья_женщина_девочка_девочка:'
    },
    '\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466': {  # 👩‍👩‍👦
        'en': ':family_woman_woman_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_frau_frau_und_junge:',
        'es': ':familia_mujer_mujer_niño:',
        'fr': ':famille_femme_femme_et_garçon:',
        'ja': ':家族_女性_女性_男の子:',
        'ko': ':가족_여자_여자_남자_아이:',
        'pt': ':família_mulher_mulher_e_menino:',
        'it': ':famiglia_donna_donna_e_bambino:',
        'fa': ':خانواده_زن_زن_و_پسر:',
        'id': ':keluarga_wanita_wanita_anak_laki-laki:',
        'zh': ':家庭_女人女人男孩:',
        'ru': ':семья_женщина_женщина_мальчик:'
    },
    '\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': {  # 👩‍👩‍👦‍👦
        'en': ':family_woman_woman_boy_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_frau_frau_junge_und_junge:',
        'es': ':familia_mujer_mujer_niño_niño:',
        'fr': ':famille_femme_femme_garçon_et_garçon:',
        'ja': ':家族_女性_女性_男の子_男の子:',
        'ko': ':가족_여자_여자_남자_아이_남자_아이:',
        'pt': ':família_mulher_mulher_menino_e_menino:',
        'it': ':famiglia_donna_donna_bambino_e_bambino:',
        'fa': ':خانواده_زن_زن_پسر_و_پسر:',
        'id': ':keluarga_wanita_wanita_anak_laki-laki_anak_laki-laki:',
        'zh': ':家庭_女人女人男孩男孩:',
        'ru': ':семья_женщина_женщина_мальчик_мальчик:'
    },
    '\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467': {  # 👩‍👩‍👧
        'en': ':family_woman_woman_girl:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_frau_frau_und_mädchen:',
        'es': ':familia_mujer_mujer_niña:',
        'fr': ':famille_femme_femme_et_fille:',
        'ja': ':家族_女性_女性_女の子:',
        'ko': ':가족_여자_여자_여자_아이:',
        'pt': ':família_mulher_mulher_e_menina:',
        'it': ':famiglia_donna_donna_e_bambina:',
        'fa': ':خانواده_زن_زن_و_دختر:',
        'id': ':keluarga_wanita_wanita_anak_perempuan:',
        'zh': ':家庭_女人女人女孩:',
        'ru': ':семья_женщина_женщина_девочка:'
    },
    '\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': {  # 👩‍👩‍👧‍👦
        'en': ':family_woman_woman_girl_boy:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_frau_frau_mädchen_und_junge:',
        'es': ':familia_mujer_mujer_niña_niño:',
        'fr': ':famille_femme_femme_fille_et_garçon:',
        'ja': ':家族_女性_女性_女の子_男の子:',
        'ko': ':가족_여자_여자_여자_아이_남자_아이:',
        'pt': ':família_mulher_mulher_menina_e_menino:',
        'it': ':famiglia_donna_donna_bambina_e_bambino:',
        'fa': ':خانواده_زن_زن_دختر_و_پسر:',
        'id': ':keluarga_wanita_wanita_anak_perempuan_anak_laki-laki:',
        'zh': ':家庭_女人女人女孩男孩:',
        'ru': ':семья_женщина_женщина_девочка_мальчик:'
    },
    '\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': {  # 👩‍👩‍👧‍👧
        'en': ':family_woman_woman_girl_girl:',
        'status': fully_qualified,
        'E': 2,
        'de': ':familie_frau_frau_mädchen_und_mädchen:',
        'es': ':familia_mujer_mujer_niña_niña:',
        'fr': ':famille_femme_femme_fille_et_fille:',
        'ja': ':家族_女性_女性_女の子_女の子:',
        'ko': ':가족_여자_여자_여자_아이_여자_아이:',
        'pt': ':família_mulher_mulher_menina_e_menina:',
        'it': ':famiglia_donna_donna_bambina_e_bambina:',
        'fa': ':خانواده_زن_زن_دختر_و_دختر:',
        'id': ':keluarga_wanita_wanita_anak_perempuan_anak_perempuan:',
        'zh': ':家庭_女人女人女孩女孩:',
        'ru': ':семья_женщина_женщина_девочка_девочка:'
    },
    '\U0001F9D1\U0000200D\U0001F33E': {  # 🧑‍🌾
        'en': ':farmer:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':bauer/bäuerin:',
        'es': ':profesional_de_la_agricultura:',
        'fr': ':fermier_(tous_genres):',
        'ja': ':農家:',
        'ko': ':농부:',
        'pt': ':agricultor:',
        'it': ':agricoltore:',
        'fa': ':کشاورز:',
        'id': ':petani:',
        'zh': ':农民:',
        'ru': ':фермер:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U0001F33E': {  # 🧑🏿‍🌾
        'en': ':farmer_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':bauer/bäuerin_dunkle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_oscuro:',
        'fr': ':fermier_(tous_genres)_peau_foncée:',
        'ja': ':農家_濃い肌色:',
        'ko': ':농부_검은색_피부:',
        'pt': ':agricultor_pele_escura:',
        'it': ':agricoltore_carnagione_scura:',
        'fa': ':کشاورز_پوست_آبنوسی:',
        'id': ':petani_warna_kulit_gelap:',
        'zh': ':农民_较深肤色:',
        'ru': ':фермер_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U0001F33E': {  # 🧑🏻‍🌾
        'en': ':farmer_light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':bauer/bäuerin_helle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_claro:',
        'fr': ':fermier_(tous_genres)_peau_claire:',
        'ja': ':農家_薄い肌色:',
        'ko': ':농부_하얀_피부:',
        'pt': ':agricultor_pele_clara:',
        'it': ':agricoltore_carnagione_chiara:',
        'fa': ':کشاورز_پوست_سفید:',
        'id': ':petani_warna_kulit_cerah:',
        'zh': ':农民_较浅肤色:',
        'ru': ':фермер_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U0001F33E': {  # 🧑🏾‍🌾
        'en': ':farmer_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':bauer/bäuerin_mitteldunkle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_oscuro_medio:',
        'fr': ':fermier_(tous_genres)_peau_mate:',
        'ja': ':農家_やや濃い肌色:',
        'ko': ':농부_진한_갈색_피부:',
        'pt': ':agricultor_pele_morena_escura:',
        'it': ':agricoltore_carnagione_abbastanza_scura:',
        'fa': ':کشاورز_پوست_گندمی:',
        'id': ':petani_warna_kulit_gelap-sedang:',
        'zh': ':农民_中等深肤色:',
        'ru': ':фермер_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U0001F33E': {  # 🧑🏼‍🌾
        'en': ':farmer_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':bauer/bäuerin_mittelhelle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_claro_medio:',
        'fr': ':fermier_(tous_genres)_peau_moyennement_claire:',
        'ja': ':農家_やや薄い肌色:',
        'ko': ':농부_연한_갈색_피부:',
        'pt': ':agricultor_pele_morena_clara:',
        'it': ':agricoltore_carnagione_abbastanza_chiara:',
        'fa': ':کشاورز_پوست_روشن:',
        'id': ':petani_warna_kulit_cerah-sedang:',
        'zh': ':农民_中等浅肤色:',
        'ru': ':фермер_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U0001F33E': {  # 🧑🏽‍🌾
        'en': ':farmer_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':bauer/bäuerin_mittlere_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_medio:',
        'fr': ':fermier_(tous_genres)_peau_légèrement_mate:',
        'ja': ':農家_中間の肌色:',
        'ko': ':농부_갈색_피부:',
        'pt': ':agricultor_pele_morena:',
        'it': ':agricoltore_carnagione_olivastra:',
        'fa': ':کشاورز_پوست_طلایی:',
        'id': ':petani_warna_kulit_sedang:',
        'zh': ':农民_中等肤色:',
        'ru': ':фермер_средний_тон_кожи:'
    },
    '\U000023E9': {  # ⏩
        'en': ':fast-forward_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':fast_forward:', ':fast_forward_button:'],
        'variant': True,
        'de': ':doppelpfeile_nach_rechts:',
        'es': ':avance_rápido:',
        'fr': ':bouton_avance_rapide:',
        'ja': ':早送りボタン:',
        'ko': ':빨리_감기:',
        'pt': ':botão_avançar:',
        'it': ':pulsante_di_avanzamento_rapido:',
        'fa': ':دکمهٔ_سریع_به_جلو:',
        'id': ':tombol_maju_cepat:',
        'zh': ':快进按钮:',
        'ru': ':перемотка_вперед:'
    },
    '\U000023EC': {  # ⏬
        'en': ':fast_down_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':arrow_double_down:'],
        'de': ':doppelpfeile_nach_unten:',
        'es': ':triángulo_doble_hacia_abajo:',
        'fr': ':double_flèche_vers_le_bas:',
        'ja': ':高速下ボタン:',
        'ko': ':아래쪽_화살표:',
        'pt': ':botão_de_avanço_para_baixo:',
        'it': ':doppia_freccia_in_basso:',
        'fa': ':دکمهٔ_سریع_به_پایین:',
        'id': ':tombol_turun_cepat:',
        'zh': ':快速下降按钮:',
        'ru': ':два_треугольника_острием_вниз:'
    },
    '\U000023EA': {  # ⏪
        'en': ':fast_reverse_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':rewind:'],
        'variant': True,
        'de': ':zurückspulen:',
        'es': ':retroceso_rápido:',
        'fr': ':bouton_retour_rapide:',
        'ja': ':早戻しボタン:',
        'ko': ':되감기:',
        'pt': ':botão_de_retroceder:',
        'it': ':pulsante_di_riavvolgimento_rapido:',
        'fa': ':دکمهٔ_سریع_به_عقب:',
        'id': ':tombol_mundur_cepat:',
        'zh': ':快退按钮:',
        'ru': ':перемотка_назад:'
    },
    '\U000023EB': {  # ⏫
        'en': ':fast_up_button:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':arrow_double_up:'],
        'de': ':doppelpfeile_nach_oben:',
        'es': ':triángulo_doble_hacia_arriba:',
        'fr': ':double_flèche_vers_le_haut:',
        'ja': ':高速上ボタン:',
        'ko': ':위쪽_화살표:',
        'pt': ':botão_de_avanço_para_cima:',
        'it': ':pulsante_doppia_freccia_in_alto:',
        'fa': ':دکمهٔ_سریع_به_بالا:',
        'id': ':tombol_naik_cepat:',
        'zh': ':快速上升按钮:',
        'ru': ':два_треугольника_острием_вверх:'
    },
    '\U0001F4E0': {  # 📠
        'en': ':fax_machine:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':fax:'],
        'de': ':faxgerät:',
        'es': ':máquina_de_fax:',
        'fr': ':fax:',
        'ja': ':ファックス:',
        'ko': ':팩스:',
        'pt': ':fax:',
        'it': ':fax:',
        'fa': ':دستگاه_نمابر:',
        'id': ':mesin_faks:',
        'zh': ':传真机:',
        'ru': ':факс:'
    },
    '\U0001F628': {  # 😨
        'en': ':fearful_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':fearful:'],
        'de': ':ängstliches_gesicht:',
        'es': ':cara_asustada:',
        'fr': ':visage_effrayé:',
        'ja': ':青ざめ:',
        'ko': ':두려워하는_얼굴:',
        'pt': ':rosto_amedrontado:',
        'it': ':faccina_impaurita:',
        'fa': ':ترسیده:',
        'id': ':wajah_ketakutan:',
        'zh': ':害怕:',
        'ru': ':боится:'
    },
    '\U0001FAB6': {  # 🪶
        'en': ':feather:',
        'status': fully_qualified,
        'E': 13,
        'de': ':feder:',
        'es': ':pluma:',
        'fr': ':plume:',
        'ja': ':羽:',
        'ko': ':깃털:',
        'pt': ':pena:',
        'it': ':piuma:',
        'fa': ':پَر:',
        'id': ':bulu:',
        'zh': ':羽毛:',
        'ru': ':перо:'
    },
    '\U00002640\U0000FE0F': {  # ♀️
        'en': ':female_sign:',
        'status': fully_qualified,
        'E': 4,
        'variant': True,
        'de': ':frauensymbol:',
        'es': ':signo_femenino:',
        'fr': ':symbole_de_la_femme:',
        'ja': ':女性のマーク:',
        'ko': ':여성_기호:',
        'pt': ':símbolo_de_feminino:',
        'it': ':simbolo_genere_femminile:',
        'fa': ':نشان_ناهید:',
        'id': ':simbol_wanita:',
        'zh': ':女性符号:',
        'ru': ':женскии_символ:'
    },
    '\U00002640': {  # ♀
        'en': ':female_sign:',
        'status': unqualified,
        'E': 4,
        'variant': True,
        'de': ':frauensymbol:',
        'es': ':signo_femenino:',
        'fr': ':symbole_de_la_femme:',
        'ja': ':女性のマーク:',
        'ko': ':여성_기호:',
        'pt': ':símbolo_de_feminino:',
        'it': ':simbolo_genere_femminile:',
        'fa': ':نشان_ناهید:',
        'id': ':simbol_wanita:',
        'zh': ':女性符号:',
        'ru': ':женский_символ:'
    },
    '\U0001F3A1': {  # 🎡
        'en': ':ferris_wheel:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':riesenrad:',
        'es': ':noria_de_feria:',
        'fr': ':grande_roue:',
        'ja': ':観覧車:',
        'ko': ':관람차:',
        'pt': ':roda_gigante:',
        'it': ':ruota_panoramica:',
        'fa': ':چرخ_وفلک:',
        'id': ':kincir_ria:',
        'zh': ':摩天轮:',
        'ru': ':колесо_обозрения:'
    },
    '\U000026F4\U0000FE0F': {  # ⛴️
        'en': ':ferry:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':fähre:',
        'es': ':ferri:',
        'fr': ':ferry:',
        'ja': ':フェリー:',
        'ko': ':페리:',
        'pt': ':balsa:',
        'it': ':traghetto:',
        'fa': ':فرابر:',
        'id': ':kapal_feri:',
        'zh': ':渡轮:',
        'ru': ':паром:'
    },
    '\U000026F4': {  # ⛴
        'en': ':ferry:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':fähre:',
        'es': ':ferri:',
        'fr': ':ferry:',
        'ja': ':フェリー:',
        'ko': ':페리:',
        'pt': ':balsa:',
        'it': ':traghetto:',
        'fa': ':فرابر:',
        'id': ':kapal_feri:',
        'zh': ':渡轮:',
        'ru': ':паром:'
    },
    '\U0001F3D1': {  # 🏑
        'en': ':field_hockey:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':field_hockey_stick_and_ball:'],
        'de': ':feldhockey:',
        'es': ':hockey_sobre_hierba:',
        'fr': ':hockey_sur_gazon:',
        'ja': ':ホッケー:',
        'ko': ':필드_하키:',
        'pt': ':hóquei_de_campo:',
        'it': ':hockey_su_prato:',
        'fa': ':هاکی_روی_چمن:',
        'id': ':hoki_lapangan:',
        'zh': ':曲棍球:',
        'ru': ':хоккей_на_траве:'
    },
    '\U0001F5C4\U0000FE0F': {  # 🗄️
        'en': ':file_cabinet:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':aktenschrank:',
        'es': ':archivador:',
        'fr': ':meuble_à_dossiers:',
        'ja': ':ファイルキャビネット:',
        'ko': ':파일_보관_서랍장:',
        'pt': ':gavetas_de_escritório:',
        'it': ':schedario_da_ufficio:',
        'fa': ':کشوی_پرونده:',
        'id': ':lemari_file:',
        'zh': ':文件柜:',
        'ru': ':архивныи_шкаф:'
    },
    '\U0001F5C4': {  # 🗄
        'en': ':file_cabinet:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':aktenschrank:',
        'es': ':archivador:',
        'fr': ':meuble_à_dossiers:',
        'ja': ':ファイルキャビネット:',
        'ko': ':파일_보관_서랍장:',
        'pt': ':gavetas_de_escritório:',
        'it': ':schedario_da_ufficio:',
        'fa': ':کشوی_پرونده:',
        'id': ':lemari_file:',
        'zh': ':文件柜:',
        'ru': ':архивный_шкаф:'
    },
    '\U0001F4C1': {  # 📁
        'en': ':file_folder:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':ordner:',
        'es': ':carpeta_de_archivos:',
        'fr': ':dossier:',
        'ja': ':フォルダー:',
        'ko': ':폴더:',
        'pt': ':pasta_de_arquivos:',
        'it': ':cartella_file:',
        'fa': ':پوشه:',
        'id': ':folder_file:',
        'zh': ':文件夹:',
        'ru': ':папка:'
    },
    '\U0001F39E\U0000FE0F': {  # 🎞️
        'en': ':film_frames:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':film_strip:'],
        'variant': True,
        'de': ':filmstreifen:',
        'es': ':fotograma_de_película:',
        'fr': ':pellicule:',
        'ja': ':映画フィルム:',
        'ko': ':영화_프레임:',
        'pt': ':rolo_de_filmes:',
        'it': ':pellicola_cinematografica:',
        'fa': ':حلقه_فیلم:',
        'id': ':frame_film:',
        'zh': ':影片帧:',
        'ru': ':кинопленка:'
    },
    '\U0001F39E': {  # 🎞
        'en': ':film_frames:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':film_strip:'],
        'variant': True,
        'de': ':filmstreifen:',
        'es': ':fotograma_de_película:',
        'fr': ':pellicule:',
        'ja': ':映画フィルム:',
        'ko': ':영화_프레임:',
        'pt': ':rolo_de_filmes:',
        'it': ':pellicola_cinematografica:',
        'fa': ':حلقهٔ_فیلم:',
        'id': ':frame_film:',
        'zh': ':影片帧:',
        'ru': ':кинопленка:'
    },
    '\U0001F4FD\U0000FE0F': {  # 📽️
        'en': ':film_projector:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':filmprojektor:',
        'es': ':proyector_de_cine:',
        'fr': ':projecteur_cinématographique:',
        'ja': ':映写機:',
        'ko': ':영화_프로젝터:',
        'pt': ':projetor_de_filmes:',
        'it': ':proiettore_cinematografico:',
        'fa': ':پروژکتور:',
        'id': ':proyektor_film:',
        'zh': ':电影放映机:',
        'ru': ':кинопроектор:'
    },
    '\U0001F4FD': {  # 📽
        'en': ':film_projector:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':filmprojektor:',
        'es': ':proyector_de_cine:',
        'fr': ':projecteur_cinématographique:',
        'ja': ':映写機:',
        'ko': ':영화_프로젝터:',
        'pt': ':projetor_de_filmes:',
        'it': ':proiettore_cinematografico:',
        'fa': ':پروژکتور:',
        'id': ':proyektor_film:',
        'zh': ':电影放映机:',
        'ru': ':кинопроектор:'
    },
    '\U0001F525': {  # 🔥
        'en': ':fire:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':feuer:',
        'es': ':fuego:',
        'fr': ':feu:',
        'ja': ':火:',
        'ko': ':불:',
        'pt': ':fogo:',
        'it': ':fuoco:',
        'fa': ':آتش:',
        'id': ':api:',
        'zh': ':火焰:',
        'ru': ':огонь:'
    },
    '\U0001F692': {  # 🚒
        'en': ':fire_engine:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':feuerwehrauto:',
        'es': ':coche_de_bomberos:',
        'fr': ':camion_de_pompier:',
        'ja': ':消防車:',
        'ko': ':소방차:',
        'pt': ':carro_do_corpo_de_bombeiros:',
        'it': ':camion_dei_pompieri:',
        'fa': ':ماشین_آتش_نشانی:',
        'id': ':mobil_pemadam_kebakaran:',
        'zh': ':消防车:',
        'ru': ':пожарная_машина:'
    },
    '\U0001F9EF': {  # 🧯
        'en': ':fire_extinguisher:',
        'status': fully_qualified,
        'E': 11,
        'de': ':feuerlöscher:',
        'es': ':extintor:',
        'fr': ':extincteur:',
        'ja': ':消火器:',
        'ko': ':소화기:',
        'pt': ':extintor_de_incêndio:',
        'it': ':estintore:',
        'fa': ':کپسول_آتش_نشانی:',
        'id': ':alat_pemadam_api_ringan:',
        'zh': ':灭火器:',
        'ru': ':огнетушитель:'
    },
    '\U0001F9E8': {  # 🧨
        'en': ':firecracker:',
        'status': fully_qualified,
        'E': 11,
        'de': ':feuerwerkskörper:',
        'es': ':petardo:',
        'fr': ':pétard:',
        'ja': ':爆竹:',
        'ko': ':폭죽:',
        'pt': ':bombinha:',
        'it': ':petardo:',
        'fa': ':ترقه:',
        'id': ':petasan:',
        'zh': ':爆竹:',
        'ru': ':динамитная_шашка:'
    },
    '\U0001F9D1\U0000200D\U0001F692': {  # 🧑‍🚒
        'en': ':firefighter:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':feuerwehrmann/-frau:',
        'es': ':bombero:',
        'fr': ':pompier:',
        'ja': ':消防士:',
        'ko': ':소방관:',
        'pt': ':bombeiro:',
        'it': ':pompiere:',
        'fa': ':مأمور_آتش_نشانی:',
        'id': ':pemadam_kebakaran:',
        'zh': ':消防员:',
        'ru': ':пожарный:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U0001F692': {  # 🧑🏿‍🚒
        'en': ':firefighter_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':feuerwehrmann/-frau_dunkle_hautfarbe:',
        'es': ':bombero_tono_de_piel_oscuro:',
        'fr': ':pompier_peau_foncée:',
        'ja': ':消防士_濃い肌色:',
        'ko': ':소방관_검은색_피부:',
        'pt': ':bombeiro_pele_escura:',
        'it': ':pompiere_carnagione_scura:',
        'fa': ':مأمور_آتش_نشانی_پوست_آبنوسی:',
        'id': ':pemadam_kebakaran_warna_kulit_gelap:',
        'zh': ':消防员_较深肤色:',
        'ru': ':пожарный_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U0001F692': {  # 🧑🏻‍🚒
        'en': ':firefighter_light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':feuerwehrmann/-frau_helle_hautfarbe:',
        'es': ':bombero_tono_de_piel_claro:',
        'fr': ':pompier_peau_claire:',
        'ja': ':消防士_薄い肌色:',
        'ko': ':소방관_하얀_피부:',
        'pt': ':bombeiro_pele_clara:',
        'it': ':pompiere_carnagione_chiara:',
        'fa': ':مأمور_آتش_نشانی_پوست_سفید:',
        'id': ':pemadam_kebakaran_warna_kulit_cerah:',
        'zh': ':消防员_较浅肤色:',
        'ru': ':пожарный_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U0001F692': {  # 🧑🏾‍🚒
        'en': ':firefighter_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':feuerwehrmann/-frau_mitteldunkle_hautfarbe:',
        'es': ':bombero_tono_de_piel_oscuro_medio:',
        'fr': ':pompier_peau_mate:',
        'ja': ':消防士_やや濃い肌色:',
        'ko': ':소방관_진한_갈색_피부:',
        'pt': ':bombeiro_pele_morena_escura:',
        'it': ':pompiere_carnagione_abbastanza_scura:',
        'fa': ':مأمور_آتش_نشانی_پوست_گندمی:',
        'id': ':pemadam_kebakaran_warna_kulit_gelap-sedang:',
        'zh': ':消防员_中等深肤色:',
        'ru': ':пожарный_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U0001F692': {  # 🧑🏼‍🚒
        'en': ':firefighter_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':feuerwehrmann/-frau_mittelhelle_hautfarbe:',
        'es': ':bombero_tono_de_piel_claro_medio:',
        'fr': ':pompier_peau_moyennement_claire:',
        'ja': ':消防士_やや薄い肌色:',
        'ko': ':소방관_연한_갈색_피부:',
        'pt': ':bombeiro_pele_morena_clara:',
        'it': ':pompiere_carnagione_abbastanza_chiara:',
        'fa': ':مأمور_آتش_نشانی_پوست_روشن:',
        'id': ':pemadam_kebakaran_warna_kulit_cerah-sedang:',
        'zh': ':消防员_中等浅肤色:',
        'ru': ':пожарный_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U0001F692': {  # 🧑🏽‍🚒
        'en': ':firefighter_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':feuerwehrmann/-frau_mittlere_hautfarbe:',
        'es': ':bombero_tono_de_piel_medio:',
        'fr': ':pompier_peau_légèrement_mate:',
        'ja': ':消防士_中間の肌色:',
        'ko': ':소방관_갈색_피부:',
        'pt': ':bombeiro_pele_morena:',
        'it': ':pompiere_carnagione_olivastra:',
        'fa': ':مأمور_آتش_نشانی_پوست_طلایی:',
        'id': ':pemadam_kebakaran_warna_kulit_sedang:',
        'zh': ':消防员_中等肤色:',
        'ru': ':пожарный_средний_тон_кожи:'
    },
    '\U0001F386': {  # 🎆
        'en': ':fireworks:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':feuerwerk:',
        'es': ':fuegos_artificiales:',
        'fr': ':feu_d’artifice:',
        'ja': ':打ち上げ花火:',
        'ko': ':불꽃놀이:',
        'pt': ':fogos_de_artifício:',
        'it': ':fuochi_d’artificio:',
        'fa': ':آتش_بازی:',
        'id': ':kembang_api:',
        'zh': ':焰火:',
        'ru': ':фейерверк:'
    },
    '\U0001F313': {  # 🌓
        'en': ':first_quarter_moon:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':zunehmender_halbmond:',
        'es': ':luna_en_cuarto_creciente:',
        'fr': ':premier_quartier_de_lune:',
        'ja': ':上弦の月:',
        'ko': ':오른쪽_반달:',
        'pt': ':quarto_crescente:',
        'it': ':primo_quarto_di_luna:',
        'fa': ':یک_چهارم_نخست:',
        'id': ':bulan_perbani_awal:',
        'zh': ':上弦月:',
        'ru': ':луна_в_первой_четверти:'
    },
    '\U0001F31B': {  # 🌛
        'en': ':first_quarter_moon_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':first_quarter_moon_with_face:'],
        'de': ':mondsichel_mit_gesicht_links:',
        'es': ':luna_de_cuarto_creciente_con_cara:',
        'fr': ':premier_quartier_de_lune_avec_visage:',
        'ja': ':顔のある上弦の月:',
        'ko': ':얼굴이_있는_초승달:',
        'pt': ':rosto_da_lua_de_quarto_crescente:',
        'it': ':faccina_primo_quarto_di_luna:',
        'fa': ':صورت_یک_چهارم_اول:',
        'id': ':wajah_bulan_perbani_awal:',
        'zh': ':微笑的上弦月:',
        'ru': ':луна_в_первой_четверти_с_лицом:'
    },
    '\U0001F41F': {  # 🐟
        'en': ':fish:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':fisch:',
        'es': ':pez:',
        'fr': ':poisson:',
        'ja': ':魚:',
        'ko': ':물고기:',
        'pt': ':peixe:',
        'it': ':pesce:',
        'fa': ':ماهی:',
        'id': ':ikan:',
        'zh': ':鱼:',
        'ru': ':рыба:'
    },
    '\U0001F365': {  # 🍥
        'en': ':fish_cake_with_swirl:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':fish_cake:'],
        'de': ':fischfrikadelle:',
        'es': ':pastel_de_pescado_japonés:',
        'fr': ':croquette_de_poisson:',
        'ja': ':なると:',
        'ko': ':어묵:',
        'pt': ':bolinho_de_peixe:',
        'it': ':tortino_di_pesce_a_spirale:',
        'fa': ':کیک_ماهی:',
        'id': ':kue_ikan_dengan_desain_pusaran:',
        'zh': ':鱼板:',
        'ru': ':камабоко:'
    },
    '\U0001F3A3': {  # 🎣
        'en': ':fishing_pole:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':fishing_pole_and_fish:'],
        'de': ':angel_mit_fisch:',
        'es': ':caña_de_pescar:',
        'fr': ':pêche_à_la_ligne:',
        'ja': ':釣り:',
        'ko': ':낚싯대:',
        'pt': ':pesca:',
        'it': ':canna_da_pesca:',
        'fa': ':چوب_ماهیگیری:',
        'id': ':tongkat_pancing:',
        'zh': ':钓鱼竿:',
        'ru': ':удочка_с_рыбой:'
    },
    '\U0001F560': {  # 🕠
        'en': ':five-thirty:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':clock530:', ':five_thirty:'],
        'variant': True,
        'de': ':5.30_uhr:',
        'es': ':cinco_y_media:',
        'fr': ':cinq_heures_et_demie:',
        'ja': ':5時半:',
        'ko': ':다섯_시_반:',
        'pt': ':cinco_e_meia:',
        'it': ':ore_cinque_e_mezza:',
        'fa': ':ساعت_پنج_و_نیم:',
        'id': ':jam_setengah_enam:',
        'zh': ':五点半:',
        'ru': ':полшестого:'
    },
    '\U0001F554': {  # 🕔
        'en': ':five_o’clock:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':clock5:', ':five_oclock:'],
        'variant': True,
        'de': ':5.00_uhr:',
        'es': ':5_en_punto:',
        'fr': ':cinq_heures:',
        'ja': ':5時:',
        'ko': ':다섯_시:',
        'pt': ':5_horas:',
        'it': ':ore_cinque:',
        'fa': ':ساعت_پنج:',
        'id': ':jam_lima_tepat:',
        'zh': ':五点:',
        'ru': ':пять_часов:'
    },
    '\U000026F3': {  # ⛳
        'en': ':flag_in_hole:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':golf:'],
        'variant': True,
        'de': ':golffahne:',
        'es': ':banderín_en_hoyo:',
        'fr': ':drapeau_de_golf:',
        'ja': ':ゴルフ:',
        'ko': ':골프:',
        'pt': ':bandeira_no_buraco:',
        'it': ':bandiera_in_buca:',
        'fa': ':پرچم_در_سوراخ:',
        'id': ':bendera_di_lubang_golf:',
        'zh': ':高尔夫球洞:',
        'ru': ':гольф:'
    },
    '\U0001F9A9': {  # 🦩
        'en': ':flamingo:',
        'status': fully_qualified,
        'E': 12,
        'de': ':flamingo:',
        'es': ':flamenco:',
        'fr': ':flamant:',
        'ja': ':フラミンゴ:',
        'ko': ':홍학:',
        'pt': ':flamingo:',
        'it': ':fenicottero:',
        'fa': ':فلامینگو:',
        'id': ':flamingo:',
        'zh': ':火烈鸟:',
        'ru': ':фламинго:'
    },
    '\U0001F526': {  # 🔦
        'en': ':flashlight:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':taschenlampe:',
        'es': ':linterna:',
        'fr': ':torche:',
        'ja': ':懐中電灯:',
        'ko': ':손전등:',
        'pt': ':lanterna:',
        'it': ':torcia:',
        'fa': ':چراغ_قوه:',
        'id': ':lampu_senter:',
        'zh': ':手电筒:',
        'ru': ':фонарик:'
    },
    '\U0001F97F': {  # 🥿
        'en': ':flat_shoe:',
        'status': fully_qualified,
        'E': 11,
        'de': ':flacher_schuh:',
        'es': ':bailarina:',
        'fr': ':chaussure_plate:',
        'ja': ':フラットシューズ:',
        'ko': ':플랫_슈즈:',
        'pt': ':sapatilha:',
        'it': ':ballerina:',
        'fa': ':کفش_بی_پاشنه:',
        'id': ':sepatu_flat:',
        'zh': ':平底鞋:',
        'ru': ':балетки:'
    },
    '\U0001FAD3': {  # 🫓
        'en': ':flatbread:',
        'status': fully_qualified,
        'E': 13,
        'de': ':fladenbrot:',
        'es': ':pan_sin_levadura:',
        'fr': ':galette:',
        'ja': ':フラットブレッド:',
        'ko': ':플랫브레드:',
        'pt': ':pão_sírio:',
        'it': ':focaccia:',
        'fa': ':نان_مسطح:',
        'id': ':roti_pipih:',
        'zh': ':扁面包:',
        'ru': ':лепешка:'
    },
    '\U0000269C\U0000FE0F': {  # ⚜️
        'en': ':fleur-de-lis:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':fleur_de_lis:'],
        'variant': True,
        'de': ':lilie:',
        'es': ':flor_de_lis:',
        'fr': ':fleur_de_lys:',
        'ja': ':フルール・ド・リス:',
        'ko': ':백합_문장:',
        'pt': ':flor-de-lis:',
        'it': ':giglio_araldico:',
        'fa': ':نماد_گل_زنبق:',
        'id': ':fleur-de-lis:',
        'zh': ':百合花饰:',
        'ru': ':геральдическая_лилия:'
    },
    '\U0000269C': {  # ⚜
        'en': ':fleur-de-lis:',
        'status': unqualified,
        'E': 1,
        'alias': [':fleur_de_lis:'],
        'variant': True,
        'de': ':lilie:',
        'es': ':flor_de_lis:',
        'fr': ':fleur_de_lys:',
        'ja': ':フルール・ド・リス:',
        'ko': ':백합_문장:',
        'pt': ':flor-de-lis:',
        'it': ':giglio_araldico:',
        'fa': ':نماد_گل_زنبق:',
        'id': ':fleur-de-lis:',
        'zh': ':百合花饰:',
        'ru': ':геральдическая_лилия:'
    },
    '\U0001F4AA': {  # 💪
        'en': ':flexed_biceps:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':muscle:'],
        'de': ':angespannter_bizeps:',
        'es': ':bíceps_flexionado:',
        'fr': ':biceps_contracté:',
        'ja': ':力こぶ:',
        'ko': ':알통:',
        'pt': ':bíceps:',
        'it': ':bicipite:',
        'fa': ':عضلهٔ_بازو:',
        'id': ':lengan_berotot:',
        'zh': ':肌肉:',
        'ru': ':бицепс:'
    },
    '\U0001F4AA\U0001F3FF': {  # 💪🏿
        'en': ':flexed_biceps_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':angespannter_bizeps_dunkle_hautfarbe:',
        'es': ':bíceps_flexionado_tono_de_piel_oscuro:',
        'fr': ':biceps_contracté_peau_foncée:',
        'ja': ':力こぶ_濃い肌色:',
        'ko': ':알통_검은색_피부:',
        'pt': ':bíceps_pele_escura:',
        'it': ':bicipite_carnagione_scura:',
        'fa': ':عضلهٔ_بازو_پوست_آبنوسی:',
        'id': ':lengan_berotot_warna_kulit_gelap:',
        'zh': ':肌肉_较深肤色:',
        'ru': ':бицепс_очень_темный_тон_кожи:'
    },
    '\U0001F4AA\U0001F3FB': {  # 💪🏻
        'en': ':flexed_biceps_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':angespannter_bizeps_helle_hautfarbe:',
        'es': ':bíceps_flexionado_tono_de_piel_claro:',
        'fr': ':biceps_contracté_peau_claire:',
        'ja': ':力こぶ_薄い肌色:',
        'ko': ':알통_하얀_피부:',
        'pt': ':bíceps_pele_clara:',
        'it': ':bicipite_carnagione_chiara:',
        'fa': ':عضلهٔ_بازو_پوست_سفید:',
        'id': ':lengan_berotot_warna_kulit_cerah:',
        'zh': ':肌肉_较浅肤色:',
        'ru': ':бицепс_очень_светлый_тон_кожи:'
    },
    '\U0001F4AA\U0001F3FE': {  # 💪🏾
        'en': ':flexed_biceps_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':angespannter_bizeps_mitteldunkle_hautfarbe:',
        'es': ':bíceps_flexionado_tono_de_piel_oscuro_medio:',
        'fr': ':biceps_contracté_peau_mate:',
        'ja': ':力こぶ_やや濃い肌色:',
        'ko': ':알통_진한_갈색_피부:',
        'pt': ':bíceps_pele_morena_escura:',
        'it': ':bicipite_carnagione_abbastanza_scura:',
        'fa': ':عضلهٔ_بازو_پوست_گندمی:',
        'id': ':lengan_berotot_warna_kulit_gelap-sedang:',
        'zh': ':肌肉_中等深肤色:',
        'ru': ':бицепс_темный_тон_кожи:'
    },
    '\U0001F4AA\U0001F3FC': {  # 💪🏼
        'en': ':flexed_biceps_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':angespannter_bizeps_mittelhelle_hautfarbe:',
        'es': ':bíceps_flexionado_tono_de_piel_claro_medio:',
        'fr': ':biceps_contracté_peau_moyennement_claire:',
        'ja': ':力こぶ_やや薄い肌色:',
        'ko': ':알통_연한_갈색_피부:',
        'pt': ':bíceps_pele_morena_clara:',
        'it': ':bicipite_carnagione_abbastanza_chiara:',
        'fa': ':عضلهٔ_بازو_پوست_روشن:',
        'id': ':lengan_berotot_warna_kulit_cerah-sedang:',
        'zh': ':肌肉_中等浅肤色:',
        'ru': ':бицепс_светлый_тон_кожи:'
    },
    '\U0001F4AA\U0001F3FD': {  # 💪🏽
        'en': ':flexed_biceps_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':angespannter_bizeps_mittlere_hautfarbe:',
        'es': ':bíceps_flexionado_tono_de_piel_medio:',
        'fr': ':biceps_contracté_peau_légèrement_mate:',
        'ja': ':力こぶ_中間の肌色:',
        'ko': ':알통_갈색_피부:',
        'pt': ':bíceps_pele_morena:',
        'it': ':bicipite_carnagione_olivastra:',
        'fa': ':عضلهٔ_بازو_پوست_طلایی:',
        'id': ':lengan_berotot_warna_kulit_sedang:',
        'zh': ':肌肉_中等肤色:',
        'ru': ':бицепс_средний_тон_кожи:'
    },
    '\U0001F4BE': {  # 💾
        'en': ':floppy_disk:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':diskette:',
        'es': ':disquete:',
        'fr': ':disquette:',
        'ja': ':フロッピー:',
        'ko': ':플로피_디스크:',
        'pt': ':disquete:',
        'it': ':floppy_disc:',
        'fa': ':فلاپی_دیسک:',
        'id': ':disket:',
        'zh': ':软盘:',
        'ru': ':дискета:'
    },
    '\U0001F3B4': {  # 🎴
        'en': ':flower_playing_cards:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':japanische_blumenkarte:',
        'es': ':cartas_de_flores:',
        'fr': ':jeu_des_fleurs:',
        'ja': ':花札:',
        'ko': ':화투:',
        'pt': ':carta_de_flores:',
        'it': ':carta_da_gioco_hanafuda:',
        'fa': ':کارت_بازی_گل_دار:',
        'id': ':kartu_bunga:',
        'zh': ':花札:',
        'ru': ':цветочная_карта:'
    },
    '\U0001F633': {  # 😳
        'en': ':flushed_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':flushed:'],
        'de': ':errötetes_gesicht_mit_großen_augen:',
        'es': ':cara_sonrojada:',
        'fr': ':visage_qui_rougit:',
        'ja': ':赤面:',
        'ko': ':상기된_얼굴:',
        'pt': ':rosto_ruborizado:',
        'it': ':faccina_imbarazzata:',
        'fa': ':سرخ_شده:',
        'id': ':wajah_tersipu:',
        'zh': ':脸红:',
        'ru': ':краснеет:'
    },
    '\U0001FA88': {  # 🪈
        'en': ':flute:',
        'status': fully_qualified,
        'E': 15,
        'de': ':flöte:',
        'es': ':flauta:',
        'fr': ':flûte:',
        'ja': ':笛:',
        'ko': ':플루트:',
        'pt': ':flauta:',
        'it': ':flauto:',
        'fa': ':فلوت:',
        'id': ':seruling:',
        'zh': ':长笛:',
        'ru': ':флейта:'
    },
    '\U0001FAB0': {  # 🪰
        'en': ':fly:',
        'status': fully_qualified,
        'E': 13,
        'de': ':fliege:',
        'es': ':mosca:',
        'fr': ':mouche:',
        'ja': ':ハエ:',
        'ko': ':파리:',
        'pt': ':mosca:',
        'it': ':mosca:',
        'fa': ':مگس:',
        'id': ':lalat:',
        'zh': ':苍蝇:',
        'ru': ':муха:'
    },
    '\U0001F94F': {  # 🥏
        'en': ':flying_disc:',
        'status': fully_qualified,
        'E': 11,
        'de': ':frisbee:',
        'es': ':disco_volador:',
        'fr': ':disque_volant:',
        'ja': ':フリスビー:',
        'ko': ':원반:',
        'pt': ':frisbee:',
        'it': ':frisbee:',
        'fa': ':فریزبی:',
        'id': ':lempar_cakram:',
        'zh': ':飞盘:',
        'ru': ':летающий_диск:'
    },
    '\U0001F6F8': {  # 🛸
        'en': ':flying_saucer:',
        'status': fully_qualified,
        'E': 5,
        'de': ':fliegende_untertasse:',
        'es': ':platillo_volante:',
        'fr': ':soucoupe_volante:',
        'ja': ':空飛ぶ円盤:',
        'ko': ':비행접시:',
        'pt': ':disco_voador:',
        'it': ':disco_volante:',
        'fa': ':بشقاب_پرنده:',
        'id': ':piring_terbang:',
        'zh': ':飞碟:',
        'ru': ':летающая_тарелка:'
    },
    '\U0001F32B\U0000FE0F': {  # 🌫️
        'en': ':fog:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':nebel:',
        'es': ':niebla:',
        'fr': ':brouillard:',
        'ja': ':霧:',
        'ko': ':안개:',
        'pt': ':neblina:',
        'it': ':nebbia:',
        'fa': ':مه:',
        'id': ':kabut:',
        'zh': ':雾:',
        'ru': ':густои_туман:'
    },
    '\U0001F32B': {  # 🌫
        'en': ':fog:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':nebel:',
        'es': ':niebla:',
        'fr': ':brouillard:',
        'ja': ':霧:',
        'ko': ':안개:',
        'pt': ':nevoeiro:',
        'it': ':nebbia:',
        'fa': ':مه:',
        'id': ':kabut:',
        'zh': ':雾:',
        'ru': ':густой_туман:'
    },
    '\U0001F301': {  # 🌁
        'en': ':foggy:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':neblig:',
        'es': ':bruma:',
        'fr': ':brume:',
        'ja': ':霧の都会:',
        'ko': ':자욱한_안개:',
        'pt': ':enevoado:',
        'it': ':nebbioso:',
        'fa': ':مه_آلود:',
        'id': ':berkabut:',
        'zh': ':有雾:',
        'ru': ':туман:'
    },
    '\U0001F64F': {  # 🙏
        'en': ':folded_hands:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':pray:'],
        'de': ':zusammengelegte_handflächen:',
        'es': ':manos_en_oración:',
        'fr': ':mains_en_prière:',
        'ja': ':祈り:',
        'ko': ':기도:',
        'pt': ':mãos_juntas:',
        'it': ':mani_giunte:',
        'fa': ':خواهش_می_کنم:',
        'id': ':melipat_tangan:',
        'zh': ':双手合十:',
        'ru': ':сложенные_руки:'
    },
    '\U0001F64F\U0001F3FF': {  # 🙏🏿
        'en': ':folded_hands_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':zusammengelegte_handflächen_dunkle_hautfarbe:',
        'es': ':manos_en_oración_tono_de_piel_oscuro:',
        'fr': ':mains_en_prière_peau_foncée:',
        'ja': ':祈り_濃い肌色:',
        'ko': ':기도_검은색_피부:',
        'pt': ':mãos_juntas_pele_escura:',
        'it': ':mani_giunte_carnagione_scura:',
        'fa': ':خواهش_می_کنم_پوست_آبنوسی:',
        'id': ':melipat_tangan_warna_kulit_gelap:',
        'zh': ':双手合十_较深肤色:',
        'ru': ':сложенные_руки_очень_темный_тон_кожи:'
    },
    '\U0001F64F\U0001F3FB': {  # 🙏🏻
        'en': ':folded_hands_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':zusammengelegte_handflächen_helle_hautfarbe:',
        'es': ':manos_en_oración_tono_de_piel_claro:',
        'fr': ':mains_en_prière_peau_claire:',
        'ja': ':祈り_薄い肌色:',
        'ko': ':기도_하얀_피부:',
        'pt': ':mãos_juntas_pele_clara:',
        'it': ':mani_giunte_carnagione_chiara:',
        'fa': ':خواهش_می_کنم_پوست_سفید:',
        'id': ':melipat_tangan_warna_kulit_cerah:',
        'zh': ':双手合十_较浅肤色:',
        'ru': ':сложенные_руки_очень_светлый_тон_кожи:'
    },
    '\U0001F64F\U0001F3FE': {  # 🙏🏾
        'en': ':folded_hands_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':zusammengelegte_handflächen_mitteldunkle_hautfarbe:',
        'es': ':manos_en_oración_tono_de_piel_oscuro_medio:',
        'fr': ':mains_en_prière_peau_mate:',
        'ja': ':祈り_やや濃い肌色:',
        'ko': ':기도_진한_갈색_피부:',
        'pt': ':mãos_juntas_pele_morena_escura:',
        'it': ':mani_giunte_carnagione_abbastanza_scura:',
        'fa': ':خواهش_می_کنم_پوست_گندمی:',
        'id': ':melipat_tangan_warna_kulit_gelap-sedang:',
        'zh': ':双手合十_中等深肤色:',
        'ru': ':сложенные_руки_темный_тон_кожи:'
    },
    '\U0001F64F\U0001F3FC': {  # 🙏🏼
        'en': ':folded_hands_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':zusammengelegte_handflächen_mittelhelle_hautfarbe:',
        'es': ':manos_en_oración_tono_de_piel_claro_medio:',
        'fr': ':mains_en_prière_peau_moyennement_claire:',
        'ja': ':祈り_やや薄い肌色:',
        'ko': ':기도_연한_갈색_피부:',
        'pt': ':mãos_juntas_pele_morena_clara:',
        'it': ':mani_giunte_carnagione_abbastanza_chiara:',
        'fa': ':خواهش_می_کنم_پوست_روشن:',
        'id': ':melipat_tangan_warna_kulit_cerah-sedang:',
        'zh': ':双手合十_中等浅肤色:',
        'ru': ':сложенные_руки_светлый_тон_кожи:'
    },
    '\U0001F64F\U0001F3FD': {  # 🙏🏽
        'en': ':folded_hands_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':zusammengelegte_handflächen_mittlere_hautfarbe:',
        'es': ':manos_en_oración_tono_de_piel_medio:',
        'fr': ':mains_en_prière_peau_légèrement_mate:',
        'ja': ':祈り_中間の肌色:',
        'ko': ':기도_갈색_피부:',
        'pt': ':mãos_juntas_pele_morena:',
        'it': ':mani_giunte_carnagione_olivastra:',
        'fa': ':خواهش_می_کنم_پوست_طلایی:',
        'id': ':melipat_tangan_warna_kulit_sedang:',
        'zh': ':双手合十_中等肤色:',
        'ru': ':сложенные_руки_средний_тон_кожи:'
    },
    '\U0001FAAD': {  # 🪭
        'en': ':folding_hand_fan:',
        'status': fully_qualified,
        'E': 15,
        'de': ':faltfächer:',
        'es': ':abanico_abierto:',
        'fr': ':éventail:',
        'ja': ':扇子:',
        'ko': ':접이식_손_부채:',
        'pt': ':leque_dobrável:',
        'it': ':ventaglio:',
        'fa': ':بادبزن:',
        'id': ':kipas_tangan_lipat:',
        'zh': ':折扇:',
        'ru': ':веер:'
    },
    '\U0001FAD5': {  # 🫕
        'en': ':fondue:',
        'status': fully_qualified,
        'E': 13,
        'de': ':fondue:',
        'es': ':fondue:',
        'fr': ':fondue:',
        'ja': ':フォンデュ:',
        'ko': ':퐁듀:',
        'pt': ':fondue:',
        'it': ':fonduta:',
        'fa': ':فوندو:',
        'id': ':fondue:',
        'zh': ':奶酪火锅:',
        'ru': ':фондю:'
    },
    '\U0001F9B6': {  # 🦶
        'en': ':foot:',
        'status': fully_qualified,
        'E': 11,
        'de': ':fuß:',
        'es': ':pie:',
        'fr': ':pied:',
        'ja': ':足:',
        'ko': ':한쪽_발:',
        'pt': ':pé:',
        'it': ':piede:',
        'fa': ':پا:',
        'id': ':telapak_kaki:',
        'zh': ':脚:',
        'ru': ':щиколотка:'
    },
    '\U0001F9B6\U0001F3FF': {  # 🦶🏿
        'en': ':foot_dark_skin_tone:',
        'status': fully_qualified,
        'E': 11,
        'de': ':fuß_dunkle_hautfarbe:',
        'es': ':pie_tono_de_piel_oscuro:',
        'fr': ':pied_peau_foncée:',
        'ja': ':足_濃い肌色:',
        'ko': ':한쪽_발_검은색_피부:',
        'pt': ':pé_pele_escura:',
        'it': ':piede_carnagione_scura:',
        'fa': ':پا_پوست_آبنوسی:',
        'id': ':telapak_kaki_warna_kulit_gelap:',
        'zh': ':脚_较深肤色:',
        'ru': ':щиколотка_очень_темный_тон_кожи:'
    },
    '\U0001F9B6\U0001F3FB': {  # 🦶🏻
        'en': ':foot_light_skin_tone:',
        'status': fully_qualified,
        'E': 11,
        'de': ':fuß_helle_hautfarbe:',
        'es': ':pie_tono_de_piel_claro:',
        'fr': ':pied_peau_claire:',
        'ja': ':足_薄い肌色:',
        'ko': ':한쪽_발_하얀_피부:',
        'pt': ':pé_pele_clara:',
        'it': ':piede_carnagione_chiara:',
        'fa': ':پا_پوست_سفید:',
        'id': ':telapak_kaki_warna_kulit_cerah:',
        'zh': ':脚_较浅肤色:',
        'ru': ':щиколотка_очень_светлый_тон_кожи:'
    },
    '\U0001F9B6\U0001F3FE': {  # 🦶🏾
        'en': ':foot_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 11,
        'de': ':fuß_mitteldunkle_hautfarbe:',
        'es': ':pie_tono_de_piel_oscuro_medio:',
        'fr': ':pied_peau_mate:',
        'ja': ':足_やや濃い肌色:',
        'ko': ':한쪽_발_진한_갈색_피부:',
        'pt': ':pé_pele_morena_escura:',
        'it': ':piede_carnagione_abbastanza_scura:',
        'fa': ':پا_پوست_گندمی:',
        'id': ':telapak_kaki_warna_kulit_gelap-sedang:',
        'zh': ':脚_中等深肤色:',
        'ru': ':щиколотка_темный_тон_кожи:'
    },
    '\U0001F9B6\U0001F3FC': {  # 🦶🏼
        'en': ':foot_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 11,
        'de': ':fuß_mittelhelle_hautfarbe:',
        'es': ':pie_tono_de_piel_claro_medio:',
        'fr': ':pied_peau_moyennement_claire:',
        'ja': ':足_やや薄い肌色:',
        'ko': ':한쪽_발_연한_갈색_피부:',
        'pt': ':pé_pele_morena_clara:',
        'it': ':piede_carnagione_abbastanza_chiara:',
        'fa': ':پا_پوست_روشن:',
        'id': ':telapak_kaki_warna_kulit_cerah-sedang:',
        'zh': ':脚_中等浅肤色:',
        'ru': ':щиколотка_светлый_тон_кожи:'
    },
    '\U0001F9B6\U0001F3FD': {  # 🦶🏽
        'en': ':foot_medium_skin_tone:',
        'status': fully_qualified,
        'E': 11,
        'de': ':fuß_mittlere_hautfarbe:',
        'es': ':pie_tono_de_piel_medio:',
        'fr': ':pied_peau_légèrement_mate:',
        'ja': ':足_中間の肌色:',
        'ko': ':한쪽_발_갈색_피부:',
        'pt': ':pé_pele_morena:',
        'it': ':piede_carnagione_olivastra:',
        'fa': ':پا_پوست_طلایی:',
        'id': ':telapak_kaki_warna_kulit_sedang:',
        'zh': ':脚_中等肤色:',
        'ru': ':щиколотка_средний_тон_кожи:'
    },
    '\U0001F463': {  # 👣
        'en': ':footprints:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':fußabdrücke:',
        'es': ':huellas_de_pies:',
        'fr': ':traces_de_pas:',
        'ja': ':足あと:',
        'ko': ':발자국:',
        'pt': ':pegadas:',
        'it': ':impronta_di_piedi:',
        'fa': ':رد_پا:',
        'id': ':jejak_kaki:',
        'zh': ':脚印:',
        'ru': ':следы:'
    },
    '\U0001F374': {  # 🍴
        'en': ':fork_and_knife:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':messer_und_gabel:',
        'es': ':tenedor_y_cuchillo:',
        'fr': ':fourchette_et_couteau:',
        'ja': ':ナイフとフォーク:',
        'ko': ':포크와_나이프:',
        'pt': ':garfo_e_faca:',
        'it': ':forchetta_e_coltello:',
        'fa': ':کارد_و_چنگال:',
        'id': ':garpu_dan_pisau:',
        'zh': ':刀叉:',
        'ru': ':вилка_и_нож:'
    },
    '\U0001F37D\U0000FE0F': {  # 🍽️
        'en': ':fork_and_knife_with_plate:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':plate_with_cutlery:'],
        'variant': True,
        'de': ':teller_mit_messer_und_gabel:',
        'es': ':cuchillo_y_tenedor_con_un_plato:',
        'fr': ':fourchette_et_couteau_avec_assiette:',
        'ja': ':ナイフとフォークと皿:',
        'ko': ':포크와_나이프가_있는_접시:',
        'pt': ':prato_com_talheres:',
        'it': ':piatto_e_posate:',
        'fa': ':بشقاب_و_کارد_و_چنگال:',
        'id': ':garpu_dan_pisau_dengan_piring:',
        'zh': ':餐具:',
        'ru': ':тарелка_с_приборами:'
    },
    '\U0001F37D': {  # 🍽
        'en': ':fork_and_knife_with_plate:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':plate_with_cutlery:'],
        'variant': True,
        'de': ':teller_mit_messer_und_gabel:',
        'es': ':cuchillo_y_tenedor_con_un_plato:',
        'fr': ':fourchette_et_couteau_avec_assiette:',
        'ja': ':ナイフとフォークと皿:',
        'ko': ':포크와_나이프가_있는_접시:',
        'pt': ':prato_com_talheres:',
        'it': ':piatto_e_posate:',
        'fa': ':بشقاب_و_کارد_و_چنگال:',
        'id': ':garpu_dan_pisau_dengan_piring:',
        'zh': ':餐具:',
        'ru': ':тарелка_с_приборами:'
    },
    '\U0001F960': {  # 🥠
        'en': ':fortune_cookie:',
        'status': fully_qualified,
        'E': 5,
        'de': ':glückskeks:',
        'es': ':galleta_de_la_fortuna:',
        'fr': ':biscuit_chinois:',
        'ja': ':フォーチュンクッキー:',
        'ko': ':포츈_쿠키:',
        'pt': ':biscoito_da_sorte:',
        'it': ':biscotto_della_fortuna:',
        'fa': ':شیرینی_شانس:',
        'id': ':kue_keberuntungan:',
        'zh': ':幸运饼干:',
        'ru': ':печенье_с_предсказанием:'
    },
    '\U000026F2': {  # ⛲
        'en': ':fountain:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':springbrunnen:',
        'es': ':fuente:',
        'fr': ':fontaine:',
        'ja': ':噴水:',
        'ko': ':분수:',
        'pt': ':fonte:',
        'it': ':fontana:',
        'fa': ':فواره:',
        'id': ':air_mancur:',
        'zh': ':喷泉:',
        'ru': ':фонтан:'
    },
    '\U0001F58B\U0000FE0F': {  # 🖋️
        'en': ':fountain_pen:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':lower_left_fountain_pen:'],
        'variant': True,
        'de': ':füllhalter:',
        'es': ':estilográfica:',
        'fr': ':stylo_plume:',
        'ja': ':万年筆:',
        'ko': ':만년필:',
        'pt': ':caneta_tinteiro:',
        'it': ':penna_stilografica:',
        'fa': ':خودنویس:',
        'id': ':pena_tinta:',
        'zh': ':钢笔:',
        'ru': ':перьевая_ручка:'
    },
    '\U0001F58B': {  # 🖋
        'en': ':fountain_pen:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':lower_left_fountain_pen:'],
        'variant': True,
        'de': ':füllhalter:',
        'es': ':estilográfica:',
        'fr': ':stylo_plume:',
        'ja': ':万年筆:',
        'ko': ':만년필:',
        'pt': ':caneta_tinteiro:',
        'it': ':penna_stilografica:',
        'fa': ':خودنویس:',
        'id': ':pena_tinta:',
        'zh': ':钢笔:',
        'ru': ':перьевая_ручка:'
    },
    '\U0001F55F': {  # 🕟
        'en': ':four-thirty:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':clock430:', ':four_thirty:'],
        'variant': True,
        'de': ':4.30_uhr:',
        'es': ':cuatro_y_media:',
        'fr': ':quatre_heures_et_demie:',
        'ja': ':4時半:',
        'ko': ':네_시_반:',
        'pt': ':quatro_e_meia:',
        'it': ':ore_quattro_e_mezza:',
        'fa': ':ساعت_چهار_و_نیم:',
        'id': ':jam_setengah_lima:',
        'zh': ':四点半:',
        'ru': ':полпятого:'
    },
    '\U0001F340': {  # 🍀
        'en': ':four_leaf_clover:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':glücksklee:',
        'es': ':trébol_de_cuatro_hojas:',
        'fr': ':trèfle_à_quatre_feuilles:',
        'ja': ':四つ葉のクローバー:',
        'ko': ':네잎클로버:',
        'pt': ':trevo_de_quatro_folhas:',
        'it': ':quadrifoglio:',
        'fa': ':شبدر_چهاربرگ:',
        'id': ':empat_daun_semanggi:',
        'zh': ':四叶草:',
        'ru': ':четырехлистный_клевер:'
    },
    '\U0001F553': {  # 🕓
        'en': ':four_o’clock:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':clock4:', ':four_oclock:'],
        'variant': True,
        'de': ':4.00_uhr:',
        'es': ':4_en_punto:',
        'fr': ':quatre_heures:',
        'ja': ':4時:',
        'ko': ':네_시:',
        'pt': ':4_horas:',
        'it': ':ore_quattro:',
        'fa': ':ساعت_چهار:',
        'id': ':jam_empat_tepat:',
        'zh': ':四点:',
        'ru': ':четыре_часа:'
    },
    '\U0001F98A': {  # 🦊
        'en': ':fox:',
        'status': fully_qualified,
        'E': 3,
        'alias': [':fox_face:'],
        'de': ':fuchs:',
        'es': ':zorro:',
        'fr': ':renard:',
        'ja': ':キツネの顔:',
        'ko': ':여우_얼굴:',
        'pt': ':rosto_de_raposa:',
        'it': ':volpe:',
        'fa': ':روباه:',
        'id': ':rubah:',
        'zh': ':狐狸:',
        'ru': ':морда_лисицы:'
    },
    '\U0001F5BC\U0000FE0F': {  # 🖼️
        'en': ':framed_picture:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':frame_with_picture:'],
        'variant': True,
        'de': ':gerahmtes_bild:',
        'es': ':cuadro_enmarcado:',
        'fr': ':cadre_avec_image:',
        'ja': ':絵画:',
        'ko': ':그림_액자:',
        'pt': ':quadro_emoldurado:',
        'it': ':quadro_incorniciato:',
        'fa': ':عکس_در_قاب:',
        'id': ':bingkai_bergambar:',
        'zh': ':带框的画:',
        'ru': ':картина:'
    },
    '\U0001F5BC': {  # 🖼
        'en': ':framed_picture:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':frame_with_picture:'],
        'variant': True,
        'de': ':gerahmtes_bild:',
        'es': ':cuadro_enmarcado:',
        'fr': ':cadre_avec_image:',
        'ja': ':絵画:',
        'ko': ':그림_액자:',
        'pt': ':quadro_emoldurado:',
        'it': ':quadro_incorniciato:',
        'fa': ':عکس_در_قاب:',
        'id': ':bingkai_bergambar:',
        'zh': ':带框的画:',
        'ru': ':картина:'
    },
    '\U0001F35F': {  # 🍟
        'en': ':french_fries:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':fries:'],
        'de': ':pommes_frites:',
        'es': ':patatas_fritas:',
        'fr': ':frites:',
        'ja': ':フライドポテト:',
        'ko': ':감자튀김:',
        'pt': ':batata_frita:',
        'it': ':patatine:',
        'fa': ':سیب_زمینی_سرخ_کرده:',
        'id': ':kentang_goreng:',
        'zh': ':薯条:',
        'ru': ':картофель_фри:'
    },
    '\U0001F364': {  # 🍤
        'en': ':fried_shrimp:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':frittierte_garnele:',
        'es': ':gamba_frita:',
        'fr': ':beignet_de_crevette:',
        'ja': ':エビフライ:',
        'ko': ':새우튀김:',
        'pt': ':camarão_frito:',
        'it': ':gambero_fritto:',
        'fa': ':میگو_سوخاری:',
        'id': ':udang_goreng:',
        'zh': ':天妇罗:',
        'ru': ':жареная_креветка:'
    },
    '\U0001F438': {  # 🐸
        'en': ':frog:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':frosch:',
        'es': ':rana:',
        'fr': ':grenouille:',
        'ja': ':カエルの顔:',
        'ko': ':개구리_얼굴:',
        'pt': ':sapo:',
        'it': ':rana:',
        'fa': ':قورباغه:',
        'id': ':katak:',
        'zh': ':青蛙:',
        'ru': ':голова_лягушки:'
    },
    '\U0001F425': {  # 🐥
        'en': ':front-facing_baby_chick:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':hatched_chick:', ':front_facing_baby_chick:'],
        'de': ':küken_von_vorne:',
        'es': ':pollito_de_frente:',
        'fr': ':poussin_de_face:',
        'ja': ':前を向いているひよこ:',
        'ko': ':정면을_향해_날개를_편_병아리:',
        'pt': ':pintinho_de_frente:',
        'it': ':pulcino_visto_di_fronte:',
        'fa': ':جوجه_از_روبرو:',
        'id': ':itik_bayi_menghadap_ke_depan:',
        'zh': ':正面朝向的小鸡:',
        'ru': ':цыпленок:'
    },
    '\U00002639\U0000FE0F': {  # ☹️
        'en': ':frowning_face:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':white_frowning_face:'],
        'variant': True,
        'de': ':düsteres_gesicht:',
        'es': ':cara_con_el_ceño_fruncido:',
        'fr': ':visage_mécontent:',
        'ja': ':困った顔:',
        'ko': ':찡그린_얼굴:',
        'pt': ':rosto_descontente:',
        'it': ':faccina_imbronciata:',
        'fa': ':اخمو:',
        'id': ':wajah_cemberut:',
        'zh': ':不满:',
        'ru': ':грустит:'
    },
    '\U00002639': {  # ☹
        'en': ':frowning_face:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':white_frowning_face:'],
        'variant': True,
        'de': ':düsteres_gesicht:',
        'es': ':cara_con_el_ceño_fruncido:',
        'fr': ':visage_mécontent:',
        'ja': ':困った顔:',
        'ko': ':찡그린_얼굴:',
        'pt': ':rosto_descontente:',
        'it': ':faccina_imbronciata:',
        'fa': ':اخمو:',
        'id': ':wajah_cemberut:',
        'zh': ':不满:',
        'ru': ':грустит:'
    },
    '\U0001F626': {  # 😦
        'en': ':frowning_face_with_open_mouth:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':frowning:'],
        'de': ':entsetztes_gesicht:',
        'es': ':cara_con_el_ceño_fruncido_y_la_boca_abierta:',
        'fr': ':visage_mécontent_avec_bouche_ouverte:',
        'ja': ':あきれ顔:',
        'ko': ':입_벌리고_찌푸린_얼굴:',
        'pt': ':rosto_franzido_com_boca_aberta:',
        'it': ':faccina_imbronciata_con_bocca_aperta:',
        'fa': ':چهره_درهم_کشیده_با_دهان_باز:',
        'id': ':wajah_mengerutkan_kening_dengan_mulut_terbuka:',
        'zh': ':啊:',
        'ru': ':в_изумлении:'
    },
    '\U000026FD': {  # ⛽
        'en': ':fuel_pump:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':fuelpump:'],
        'variant': True,
        'de': ':tanksäule:',
        'es': ':surtidor_de_gasolina:',
        'fr': ':pompe_à_essence:',
        'ja': ':ガソリンスタンド:',
        'ko': ':주유소:',
        'pt': ':posto_de_gasolina:',
        'it': ':stazione_di_servizio:',
        'fa': ':پمپ_بنزین:',
        'id': ':pompa_bahan_bakar:',
        'zh': ':油泵:',
        'ru': ':заправка:'
    },
    '\U0001F315': {  # 🌕
        'en': ':full_moon:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':vollmond:',
        'es': ':luna_llena:',
        'fr': ':pleine_lune:',
        'ja': ':満月:',
        'ko': ':보름달:',
        'pt': ':lua_cheia:',
        'it': ':luna_piena:',
        'fa': ':ماه_کامل:',
        'id': ':bulan_purnama:',
        'zh': ':满月:',
        'ru': ':полнолуние:'
    },
    '\U0001F31D': {  # 🌝
        'en': ':full_moon_face:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':full_moon_with_face:'],
        'de': ':vollmond_mit_gesicht:',
        'es': ':luna_llena_con_cara:',
        'fr': ':pleine_lune_avec_visage:',
        'ja': ':顔のある満月:',
        'ko': ':얼굴이_있는_보름달:',
        'pt': ':rosto_da_lua_cheia:',
        'it': ':faccina_luna_piena:',
        'fa': ':صورت_ماه_کامل:',
        'id': ':wajah_bulan_purnama:',
        'zh': ':微笑的月亮:',
        'ru': ':полная_луна_с_лицом:'
    },
    '\U000026B1\U0000FE0F': {  # ⚱️
        'en': ':funeral_urn:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':urne:',
        'es': ':urna_funeraria:',
        'fr': ':urne_funéraire:',
        'ja': ':骨壺:',
        'ko': ':납골_단지:',
        'pt': ':urna_funerária:',
        'it': ':urna_funeraria:',
        'fa': ':خاکستردان:',
        'id': ':guci_abu:',
        'zh': ':骨灰缸:',
        'ru': ':урна_с_прахом:'
    },
    '\U000026B1': {  # ⚱
        'en': ':funeral_urn:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':urne:',
        'es': ':urna_funeraria:',
        'fr': ':urne_funéraire:',
        'ja': ':骨壺:',
        'ko': ':납골_단지:',
        'pt': ':urna_funerária:',
        'it': ':urna_funeraria:',
        'fa': ':خاکستردان:',
        'id': ':guci_abu:',
        'zh': ':骨灰缸:',
        'ru': ':урна_с_прахом:'
    },
    '\U0001F3B2': {  # 🎲
        'en': ':game_die:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':spielwürfel:',
        'es': ':dado:',
        'fr': ':dés:',
        'ja': ':サイコロ:',
        'ko': ':주사위:',
        'pt': ':jogo_de_dado:',
        'it': ':dado:',
        'fa': ':تاس_بازی:',
        'id': ':dadu:',
        'zh': ':骰子:',
        'ru': ':игральная_кость:'
    },
    '\U0001F9C4': {  # 🧄
        'en': ':garlic:',
        'status': fully_qualified,
        'E': 12,
        'de': ':knoblauch:',
        'es': ':ajo:',
        'fr': ':ail:',
        'ja': ':ニンニク:',
        'ko': ':마늘:',
        'pt': ':alho:',
        'it': ':aglio:',
        'fa': ':سیر:',
        'id': ':bawang_putih:',
        'zh': ':蒜:',
        'ru': ':чеснок:'
    },
    '\U00002699\U0000FE0F': {  # ⚙️
        'en': ':gear:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':zahnrad:',
        'es': ':engranaje:',
        'fr': ':roue_dentée:',
        'ja': ':歯車:',
        'ko': ':톱니바퀴:',
        'pt': ':engrenagem:',
        'it': ':ingranaggio:',
        'fa': ':چرخ_دنده:',
        'id': ':roda_gigi:',
        'zh': ':齿轮:',
        'ru': ':шестеренка:'
    },
    '\U00002699': {  # ⚙
        'en': ':gear:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':zahnrad:',
        'es': ':engranaje:',
        'fr': ':roue_dentée:',
        'ja': ':歯車:',
        'ko': ':톱니바퀴:',
        'pt': ':engrenagem:',
        'it': ':ingranaggio:',
        'fa': ':چرخ_دنده:',
        'id': ':roda_gigi:',
        'zh': ':齿轮:',
        'ru': ':шестеренка:'
    },
    '\U0001F48E': {  # 💎
        'en': ':gem_stone:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':gem:'],
        'de': ':edelstein:',
        'es': ':piedra_preciosa:',
        'fr': ':pierre_précieuse:',
        'ja': ':宝石:',
        'ko': ':원석:',
        'pt': ':pedra_preciosa:',
        'it': ':gemma:',
        'fa': ':جواهر:',
        'id': ':batu_permata:',
        'zh': ':宝石:',
        'ru': ':драгоценный_камень:'
    },
    '\U0001F9DE': {  # 🧞
        'en': ':genie:',
        'status': fully_qualified,
        'E': 5,
        'de': ':flaschengeist:',
        'es': ':genio:',
        'fr': ':génie:',
        'ja': ':精霊:',
        'ko': ':지니:',
        'pt': ':gênio:',
        'it': ':genio:',
        'fa': ':جن:',
        'id': ':jin:',
        'zh': ':妖怪:',
        'ru': ':джинн:'
    },
    '\U0001F47B': {  # 👻
        'en': ':ghost:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':gespenst:',
        'es': ':fantasma:',
        'fr': ':fantôme:',
        'ja': ':お化け:',
        'ko': ':유령:',
        'pt': ':fantasma:',
        'it': ':fantasma:',
        'fa': ':روح:',
        'id': ':hantu:',
        'zh': ':鬼:',
        'ru': ':привидение:'
    },
    '\U0001FADA': {  # 🫚
        'en': ':ginger_root:',
        'status': fully_qualified,
        'E': 15,
        'de': ':ingwer:',
        'es': ':raíz_de_jengibre:',
        'fr': ':racine_de_gingembre:',
        'ja': ':ショウガ:',
        'ko': ':생강_뿌리:',
        'pt': ':gengibre:',
        'it': ':radice_di_zenzero:',
        'fa': ':زنجبیل:',
        'id': ':akar_jahe:',
        'zh': ':姜:',
        'ru': ':корень_имбиря:'
    },
    '\U0001F992': {  # 🦒
        'en': ':giraffe:',
        'status': fully_qualified,
        'E': 5,
        'de': ':giraffe:',
        'es': ':jirafa:',
        'fr': ':girafe:',
        'ja': ':キリン:',
        'ko': ':기린:',
        'pt': ':girafa:',
        'it': ':giraffa:',
        'fa': ':زرافه:',
        'id': ':jerapah:',
        'zh': ':长颈鹿:',
        'ru': ':жираф:'
    },
    '\U0001F467': {  # 👧
        'en': ':girl:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':mädchen:',
        'es': ':niña:',
        'fr': ':fille:',
        'ja': ':女の子:',
        'ko': ':여자_아이:',
        'pt': ':menina:',
        'it': ':bambina:',
        'fa': ':دختر:',
        'id': ':anak_perempuan:',
        'zh': ':女孩:',
        'ru': ':девочка:'
    },
    '\U0001F467\U0001F3FF': {  # 👧🏿
        'en': ':girl_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':mädchen_dunkle_hautfarbe:',
        'es': ':niña_tono_de_piel_oscuro:',
        'fr': ':fille_peau_foncée:',
        'ja': ':女の子_濃い肌色:',
        'ko': ':여자_아이_검은색_피부:',
        'pt': ':menina_pele_escura:',
        'it': ':bambina_carnagione_scura:',
        'fa': ':دختر_پوست_آبنوسی:',
        'id': ':anak_perempuan_warna_kulit_gelap:',
        'zh': ':女孩_较深肤色:',
        'ru': ':девочка_очень_темный_тон_кожи:'
    },
    '\U0001F467\U0001F3FB': {  # 👧🏻
        'en': ':girl_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':mädchen_helle_hautfarbe:',
        'es': ':niña_tono_de_piel_claro:',
        'fr': ':fille_peau_claire:',
        'ja': ':女の子_薄い肌色:',
        'ko': ':여자_아이_하얀_피부:',
        'pt': ':menina_pele_clara:',
        'it': ':bambina_carnagione_chiara:',
        'fa': ':دختر_پوست_سفید:',
        'id': ':anak_perempuan_warna_kulit_cerah:',
        'zh': ':女孩_较浅肤色:',
        'ru': ':девочка_очень_светлый_тон_кожи:'
    },
    '\U0001F467\U0001F3FE': {  # 👧🏾
        'en': ':girl_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':mädchen_mitteldunkle_hautfarbe:',
        'es': ':niña_tono_de_piel_oscuro_medio:',
        'fr': ':fille_peau_mate:',
        'ja': ':女の子_やや濃い肌色:',
        'ko': ':여자_아이_진한_갈색_피부:',
        'pt': ':menina_pele_morena_escura:',
        'it': ':bambina_carnagione_abbastanza_scura:',
        'fa': ':دختر_پوست_گندمی:',
        'id': ':anak_perempuan_warna_kulit_gelap-sedang:',
        'zh': ':女孩_中等深肤色:',
        'ru': ':девочка_темный_тон_кожи:'
    },
    '\U0001F467\U0001F3FC': {  # 👧🏼
        'en': ':girl_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':mädchen_mittelhelle_hautfarbe:',
        'es': ':niña_tono_de_piel_claro_medio:',
        'fr': ':fille_peau_moyennement_claire:',
        'ja': ':女の子_やや薄い肌色:',
        'ko': ':여자_아이_연한_갈색_피부:',
        'pt': ':menina_pele_morena_clara:',
        'it': ':bambina_carnagione_abbastanza_chiara:',
        'fa': ':دختر_پوست_روشن:',
        'id': ':anak_perempuan_warna_kulit_cerah-sedang:',
        'zh': ':女孩_中等浅肤色:',
        'ru': ':девочка_светлый_тон_кожи:'
    },
    '\U0001F467\U0001F3FD': {  # 👧🏽
        'en': ':girl_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':mädchen_mittlere_hautfarbe:',
        'es': ':niña_tono_de_piel_medio:',
        'fr': ':fille_peau_légèrement_mate:',
        'ja': ':女の子_中間の肌色:',
        'ko': ':여자_아이_갈색_피부:',
        'pt': ':menina_pele_morena:',
        'it': ':bambina_carnagione_olivastra:',
        'fa': ':دختر_پوست_طلایی:',
        'id': ':anak_perempuan_warna_kulit_sedang:',
        'zh': ':女孩_中等肤色:',
        'ru': ':девочка_средний_тон_кожи:'
    },
    '\U0001F95B': {  # 🥛
        'en': ':glass_of_milk:',
        'status': fully_qualified,
        'E': 3,
        'alias': [':milk_glass:'],
        'de': ':glas_milch:',
        'es': ':vaso_de_leche:',
        'fr': ':verre_de_lait:',
        'ja': ':牛乳入りのコップ:',
        'ko': ':우유_한잔:',
        'pt': ':copo_de_leite:',
        'it': ':bicchiere_di_latte:',
        'fa': ':لیوان_شیر:',
        'id': ':gelas_susu:',
        'zh': ':一杯奶:',
        'ru': ':стакан_молока:'
    },
    '\U0001F453': {  # 👓
        'en': ':glasses:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':eyeglasses:'],
        'variant': True,
        'de': ':brille:',
        'es': ':gafas:',
        'fr': ':lunettes_de_vue:',
        'ja': ':メガネ:',
        'ko': ':안경:',
        'pt': ':óculos:',
        'it': ':occhiali_da_vista:',
        'fa': ':عینک_طبی:',
        'id': ':kacamata:',
        'zh': ':眼镜:',
        'ru': ':очки:'
    },
    '\U0001F30E': {  # 🌎
        'en': ':globe_showing_Americas:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':earth_americas:', ':globe_showing_americas:'],
        'variant': True,
        'de': ':globus_mit_amerika:',
        'es': ':globo_terráqueo_mostrando_américa:',
        'fr': ':globe_tourné_sur_les_amériques:',
        'ja': ':地球(アメリカ大陸):',
        'ko': ':미대륙이_보이는_지구:',
        'pt': ':globo_mostrando_as_américas:',
        'it': ':america:',
        'fa': ':نقشهٔ_قارهٔ_امریکا_روی_کره:',
        'id': ':bola_dunia_amerika:',
        'zh': ':地球上的美洲:',
        'ru': ':Америка_на_земном_шаре:'
    },
    '\U0001F30F': {  # 🌏
        'en': ':globe_showing_Asia-Australia:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':earth_asia:', ':globe_showing_asia_australia:'],
        'variant': True,
        'de': ':globus_mit_asien_und_australien:',
        'es': ':globo_terráqueo_mostrando_asia_y_australia:',
        'fr': ':globe_tourné_sur_l’asie_et_l’australie:',
        'ja': ':地球(アジアとオーストラリア):',
        'ko': ':아시아와_호주가_보이는_지구:',
        'pt': ':globo_mostrando_ásia_e_oceania:',
        'it': ':asia_e_australia:',
        'fa': ':نقشهٔ_آسیا-اقیانوسیه_روی_کره:',
        'id': ':bola_dunia_asia-australia:',
        'zh': ':地球上的亚洲澳洲:',
        'ru': ':Азия_и_Австралия_на_земном_шаре:'
    },
    '\U0001F30D': {  # 🌍
        'en': ':globe_showing_Europe-Africa:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':earth_africa:', ':globe_showing_europe_africa:'],
        'variant': True,
        'de': ':globus_mit_europa_und_afrika:',
        'es': ':globo_terráqueo_mostrando_europa_y_áfrica:',
        'fr': ':globe_tourné_sur_l’afrique_et_l’europe:',
        'ja': ':地球(ヨーロッパとアフリカ):',
        'ko': ':유럽과_아프리카가_보이는_지구:',
        'pt': ':globo_mostrando_europa_e_áfrica:',
        'it': ':europa_e_africa:',
        'fa': ':نقشهٔ_اروپا-افریقا_روی_کره:',
        'id': ':bola_dunia_eropa-afrika:',
        'zh': ':地球上的欧洲非洲:',
        'ru': ':Европа_и_Африка_на_земном_шаре:'
    },
    '\U0001F310': {  # 🌐
        'en': ':globe_with_meridians:',
        'status': fully_qualified,
        'E': 1,
        'de': ':globus_mit_meridianen:',
        'es': ':globo_terráqueo_con_meridianos:',
        'fr': ':globe_avec_méridiens:',
        'ja': ':子午線のある地球:',
        'ko': ':자오선_지구:',
        'pt': ':globo_com_meridianos:',
        'it': ':globo_con_meridiani:',
        'fa': ':کره_با_نصف_النهارها:',
        'id': ':bola_dunia_dengan_garis_meridian:',
        'zh': ':带经纬线的地球:',
        'ru': ':меридианы:'
    },
    '\U0001F9E4': {  # 🧤
        'en': ':gloves:',
        'status': fully_qualified,
        'E': 5,
        'de': ':handschuhe:',
        'es': ':guantes:',
        'fr': ':gants:',
        'ja': ':手袋:',
        'ko': ':장갑:',
        'pt': ':luvas:',
        'it': ':guanti:',
        'fa': ':دستکش:',
        'id': ':sarung_tangan:',
        'zh': ':手套:',
        'ru': ':перчатки:'
    },
    '\U0001F31F': {  # 🌟
        'en': ':glowing_star:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':star2:'],
        'de': ':funkelnder_stern:',
        'es': ':estrella_brillante:',
        'fr': ':étoile_brillante:',
        'ja': ':きらきら星:',
        'ko': ':반짝이는_별:',
        'pt': ':estrela_brilhante:',
        'it': ':stella_che_brilla:',
        'fa': ':ستاره_درخشان:',
        'id': ':bintang_bersinar:',
        'zh': ':闪亮的星星:',
        'ru': ':сияющая_звезда:'
    },
    '\U0001F945': {  # 🥅
        'en': ':goal_net:',
        'status': fully_qualified,
        'E': 3,
        'de': ':tor:',
        'es': ':portería:',
        'fr': ':cage:',
        'ja': ':ゴールネット:',
        'ko': ':골대:',
        'pt': ':gol:',
        'it': ':porta_da_calcio:',
        'fa': ':دروازه:',
        'id': ':gawang:',
        'zh': ':球门:',
        'ru': ':спортивные_ворота:'
    },
    '\U0001F410': {  # 🐐
        'en': ':goat:',
        'status': fully_qualified,
        'E': 1,
        'de': ':ziege:',
        'es': ':cabra:',
        'fr': ':chèvre:',
        'ja': ':山羊:',
        'ko': ':염소:',
        'pt': ':cabra:',
        'it': ':capra:',
        'fa': ':بز:',
        'id': ':kambing:',
        'zh': ':山羊:',
        'ru': ':коза:'
    },
    '\U0001F47A': {  # 👺
        'en': ':goblin:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':japanese_goblin:'],
        'de': ':kobold:',
        'es': ':demonio_japonés_tengu:',
        'fr': ':monstre_japonais:',
        'ja': ':天狗:',
        'ko': ':코주부_도깨비_가면:',
        'pt': ':duende_japonês:',
        'it': ':goblin:',
        'fa': ':جن_کوتوله:',
        'id': ':topeng_hidung_panjang:',
        'zh': ':小妖精:',
        'ru': ':японский_гоблин:'
    },
    '\U0001F97D': {  # 🥽
        'en': ':goggles:',
        'status': fully_qualified,
        'E': 11,
        'de': ':schutzbrille:',
        'es': ':gafas_de_protección:',
        'fr': ':lunettes:',
        'ja': ':ゴーグル:',
        'ko': ':고글:',
        'pt': ':óculos_de_proteção:',
        'it': ':maschera:',
        'fa': ':عینک_ایمنی:',
        'id': ':kacamata_renang:',
        'zh': ':护目镜:',
        'ru': ':защитные_очки:'
    },
    '\U0001FABF': {  # 🪿
        'en': ':goose:',
        'status': fully_qualified,
        'E': 15,
        'de': ':gans:',
        'es': ':oca:',
        'fr': ':oie:',
        'ja': ':ガチョウ:',
        'ko': ':거위:',
        'pt': ':ganso:',
        'it': ':oca:',
        'fa': ':غاز:',
        'id': ':angsa_leher_pendek:',
        'zh': ':鹅:',
        'ru': ':гусь:'
    },
    '\U0001F98D': {  # 🦍
        'en': ':gorilla:',
        'status': fully_qualified,
        'E': 3,
        'alias': [':harambe:'],
        'de': ':gorilla:',
        'es': ':gorila:',
        'fr': ':gorille:',
        'ja': ':ゴリラ:',
        'ko': ':고릴라:',
        'pt': ':gorila:',
        'it': ':gorilla:',
        'fa': ':گوریل:',
        'id': ':gorila:',
        'zh': ':大猩猩:',
        'ru': ':горилла:'
    },
    '\U0001F393': {  # 🎓
        'en': ':graduation_cap:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':mortar_board:'],
        'variant': True,
        'de': ':doktorhut:',
        'es': ':birrete:',
        'fr': ':toque_universitaire:',
        'ja': ':角帽:',
        'ko': ':졸업_모자:',
        'pt': ':chapéu_de_formatura:',
        'it': ':cappello_da_laureato:',
        'fa': ':کلاه_فارغ_التحصیلی:',
        'id': ':topi_wisuda:',
        'zh': ':毕业帽:',
        'ru': ':шапочка_выпускника:'
    },
    '\U0001F347': {  # 🍇
        'en': ':grapes:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':trauben:',
        'es': ':uvas:',
        'fr': ':raisin:',
        'ja': ':ぶどう:',
        'ko': ':포도:',
        'pt': ':uvas:',
        'it': ':uva:',
        'fa': ':انگور:',
        'id': ':anggur:',
        'zh': ':葡萄:',
        'ru': ':виноград:'
    },
    '\U0001F34F': {  # 🍏
        'en': ':green_apple:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':grüner_apfel:',
        'es': ':manzana_verde:',
        'fr': ':pomme_verte:',
        'ja': ':青リンゴ:',
        'ko': ':초록_사과:',
        'pt': ':maçã_verde:',
        'it': ':mela_verde:',
        'fa': ':سیب_سبز:',
        'id': ':apel_hijau:',
        'zh': ':青苹果:',
        'ru': ':зеленое_яблоко:'
    },
    '\U0001F4D7': {  # 📗
        'en': ':green_book:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':grünes_buch:',
        'es': ':libro_verde:',
        'fr': ':livre_vert:',
        'ja': ':緑の本:',
        'ko': ':초록색_책:',
        'pt': ':livro_verde:',
        'it': ':libro_verde:',
        'fa': ':کتاب_سبز:',
        'id': ':buku_hijau:',
        'zh': ':绿色书本:',
        'ru': ':зеленая_книга:'
    },
    '\U0001F7E2': {  # 🟢
        'en': ':green_circle:',
        'status': fully_qualified,
        'E': 12,
        'de': ':grüner_punkt:',
        'es': ':círculo_verde:',
        'fr': ':disque_vert:',
        'ja': ':緑の丸:',
        'ko': ':녹색_원:',
        'pt': ':círculo_verde:',
        'it': ':cerchio_verde:',
        'fa': ':دایره_سبز:',
        'id': ':lingkaran_hijau:',
        'zh': ':绿色圆:',
        'ru': ':зеленый_круг:'
    },
    '\U0001F49A': {  # 💚
        'en': ':green_heart:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':grünes_herz:',
        'es': ':corazón_verde:',
        'fr': ':cœur_vert:',
        'ja': ':緑のハート:',
        'ko': ':초록색_하트:',
        'pt': ':coração_verde:',
        'it': ':cuore_verde:',
        'fa': ':قلب_سبز:',
        'id': ':hati_hijau:',
        'zh': ':绿心:',
        'ru': ':зеленое_сердце:'
    },
    '\U0001F957': {  # 🥗
        'en': ':green_salad:',
        'status': fully_qualified,
        'E': 3,
        'de': ':salat:',
        'es': ':ensalada:',
        'fr': ':salade_verte:',
        'ja': ':グリーンサラダ:',
        'ko': ':야채샐러드:',
        'pt': ':salada_verde:',
        'it': ':insalata_verde:',
        'fa': ':سالاد_سبزیجات:',
        'id': ':salad_hijau:',
        'zh': ':绿色沙拉:',
        'ru': ':овощной_салат:'
    },
    '\U0001F7E9': {  # 🟩
        'en': ':green_square:',
        'status': fully_qualified,
        'E': 12,
        'de': ':grünes_quadrat:',
        'es': ':cuadrado_verde:',
        'fr': ':carré_vert:',
        'ja': ':緑の四角:',
        'ko': ':녹색_사각형:',
        'pt': ':quadrado_verde:',
        'it': ':quadrato_verde:',
        'fa': ':مربع_سبز:',
        'id': ':persegi_hijau:',
        'zh': ':绿色方块:',
        'ru': ':зеленый_квадрат:'
    },
    '\U0001FA76': {  # 🩶
        'en': ':grey_heart:',
        'status': fully_qualified,
        'E': 15,
        'de': ':graues_herz:',
        'es': ':corazón_gris:',
        'fr': ':cœur_gris:',
        'ja': ':灰色のハート:',
        'ko': ':회색_하트:',
        'pt': ':coração_cinza:',
        'it': ':cuore_grigio:',
        'fa': ':قلب_خاکستری:',
        'id': ':hati_abu-abu:',
        'zh': ':灰心:',
        'ru': ':серое_сердце:'
    },
    '\U0001F62C': {  # 😬
        'en': ':grimacing_face:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':grimacing:'],
        'de': ':grimassen_schneidendes_gesicht:',
        'es': ':cara_haciendo_una_mueca:',
        'fr': ':visage_grimaçant:',
        'ja': ':しかめ面:',
        'ko': ':찌푸린_얼굴:',
        'pt': ':rosto_expressando_desagrado:',
        'it': ':faccina_con_smorfia:',
        'fa': ':دهن_کجی:',
        'id': ':wajah_menyeringai:',
        'zh': ':龇牙咧嘴:',
        'ru': ':гримаса:'
    },
    '\U0001F63A': {  # 😺
        'en': ':grinning_cat:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':smiley_cat:'],
        'de': ':grinsende_katze:',
        'es': ':gato_sonriendo:',
        'fr': ':chat_qui_sourit:',
        'ja': ':にっこり笑う猫:',
        'ko': ':활짝_웃는_고양이_얼굴:',
        'pt': ':rosto_de_gato_sorrindo:',
        'it': ':gatto_che_sorride:',
        'fa': ':گربهٔ_خندان:',
        'id': ':wajah_kucing_tersenyum_lebar:',
        'zh': ':大笑的猫:',
        'ru': ':улыбающийся_кот:'
    },
    '\U0001F638': {  # 😸
        'en': ':grinning_cat_with_smiling_eyes:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':smile_cat:'],
        'de': ':grinsende_katze_mit_lachenden_augen:',
        'es': ':gato_sonriendo_con_ojos_sonrientes:',
        'fr': ':chat_qui_sourit_avec_des_yeux_rieurs:',
        'ja': ':にやっと笑う猫:',
        'ko': ':미소_짓는_눈으로_활짝_웃는_고양이_얼굴:',
        'pt': ':rosto_de_gato_sorrindo_com_olhos_sorridentes:',
        'it': ':gatto_che_sogghigna:',
        'fa': ':گربه_با_چشم_های_خندان:',
        'id': ':wajah_kucing_gembira_dengan_mata_bahagia:',
        'zh': ':微笑的猫:',
        'ru': ':смеющийся_кот:'
    },
    '\U0001F600': {  # 😀
        'en': ':grinning_face:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':grinning:'],
        'de': ':grinsendes_gesicht:',
        'es': ':cara_sonriendo:',
        'fr': ':visage_rieur:',
        'ja': ':にっこり笑う:',
        'ko': ':활짝_웃는_얼굴:',
        'pt': ':rosto_risonho:',
        'it': ':faccina_con_un_gran_sorriso:',
        'fa': ':خنده:',
        'id': ':wajah_gembira:',
        'zh': ':嘿嘿:',
        'ru': ':широко_улыбается:'
    },
    '\U0001F603': {  # 😃
        'en': ':grinning_face_with_big_eyes:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':smiley:'],
        'de': ':grinsendes_gesicht_mit_großen_augen:',
        'es': ':cara_sonriendo_con_ojos_grandes:',
        'fr': ':visage_souriant_avec_de_grands_yeux:',
        'ja': ':わーい:',
        'ko': ':눈을_크게_뜨고_웃는_얼굴:',
        'pt': ':rosto_risonho_com_olhos_bem_abertos:',
        'it': ':faccina_con_un_gran_sorriso_e_occhi_spalancati:',
        'fa': ':صورت_خندان:',
        'id': ':wajah_tersenyum_lebar:',
        'zh': ':哈哈:',
        'ru': ':смеется:'
    },
    '\U0001F604': {  # 😄
        'en': ':grinning_face_with_smiling_eyes:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':smile:'],
        'de': ':grinsendes_gesicht_mit_lachenden_augen:',
        'es': ':cara_sonriendo_con_ojos_sonrientes:',
        'fr': ':visage_très_souriant_aux_yeux_rieurs:',
        'ja': ':笑顔:',
        'ko': ':미소_짓는_눈으로_활짝_웃는_얼굴:',
        'pt': ':rosto_risonho_com_olhos_sorridentes:',
        'it': ':faccina_con_sorriso_e_occhi_sorridenti:',
        'fa': ':خندان:',
        'id': ':wajah_tersenyum_lebar_dan_mata_tertutup:',
        'zh': ':大笑:',
        'ru': ':смеется_с_закрытыми_глазами:'
    },
    '\U0001F605': {  # 😅
        'en': ':grinning_face_with_sweat:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':sweat_smile:'],
        'de': ':grinsendes_gesicht_mit_schweißtropfen:',
        'es': ':cara_sonriendo_con_sudor_frío:',
        'fr': ':visage_souriant_avec_une_goutte_de_sueur:',
        'ja': ':冷や汗笑顔:',
        'ko': ':땀_흘리며_웃는_얼굴:',
        'pt': ':rosto_risonho_com_gota_de_suor:',
        'it': ':faccina_con_un_gran_sorriso_e_goccia_di_sudore:',
        'fa': ':صورت_خندان_و_عرق_کرده:',
        'id': ':wajah_tersenyum_lebar_dan_berkeringat:',
        'zh': ':苦笑:',
        'ru': ':нервно_смеется:'
    },
    '\U0001F606': {  # 😆
        'en': ':grinning_squinting_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':satisfied:', ':laughing:'],
        'de': ':grinsegesicht_mit_zugekniffenen_augen:',
        'es': ':cara_sonriendo_con_los_ojos_cerrados:',
        'fr': ':visage_souriant_avec_yeux_plissés:',
        'ja': ':きゃー:',
        'ko': ':눈웃음짓는_얼굴:',
        'pt': ':rosto_risonho_com_olhos_semicerrados:',
        'it': ':sorriso_a_bocca_aperta_con_occhi_chiusi:',
        'fa': ':قاه_قاه:',
        'id': ':wajah_tersenyum_lebar_dan_memejamkan_mata:',
        'zh': ':斜眼笑:',
        'ru': ':смеется_с_прищуренными_глазами:'
    },
    '\U0001F497': {  # 💗
        'en': ':growing_heart:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':heartpulse:'],
        'de': ':wachsendes_herz:',
        'es': ':corazón_creciente:',
        'fr': ':cœur_grandissant:',
        'ja': ':大きくなるハート:',
        'ko': ':커지는_하트:',
        'pt': ':coração_crescendo:',
        'it': ':cuore_che_cresce:',
        'fa': ':قلب_هیجان_زده:',
        'id': ':hati_membesar:',
        'zh': ':搏动的心:',
        'ru': ':растущее_сердце:'
    },
    '\U0001F482': {  # 💂
        'en': ':guard:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':wache:',
        'es': ':guardia:',
        'fr': ':garde:',
        'ja': ':衛兵:',
        'ko': ':근위병:',
        'pt': ':guarda:',
        'it': ':guardia:',
        'fa': ':گارد_نظامی:',
        'id': ':penjaga:',
        'zh': ':卫兵:',
        'ru': ':гвардеец:'
    },
    '\U0001F482\U0001F3FF': {  # 💂🏿
        'en': ':guard_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':wache_dunkle_hautfarbe:',
        'es': ':guardia_tono_de_piel_oscuro:',
        'fr': ':garde_peau_foncée:',
        'ja': ':衛兵_濃い肌色:',
        'ko': ':근위병_검은색_피부:',
        'pt': ':guarda_pele_escura:',
        'it': ':guardia_carnagione_scura:',
        'fa': ':گارد_نظامی_پوست_آبنوسی:',
        'id': ':penjaga_warna_kulit_gelap:',
        'zh': ':卫兵_较深肤色:',
        'ru': ':гвардеец_очень_темный_тон_кожи:'
    },
    '\U0001F482\U0001F3FB': {  # 💂🏻
        'en': ':guard_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':wache_helle_hautfarbe:',
        'es': ':guardia_tono_de_piel_claro:',
        'fr': ':garde_peau_claire:',
        'ja': ':衛兵_薄い肌色:',
        'ko': ':근위병_하얀_피부:',
        'pt': ':guarda_pele_clara:',
        'it': ':guardia_carnagione_chiara:',
        'fa': ':گارد_نظامی_پوست_سفید:',
        'id': ':penjaga_warna_kulit_cerah:',
        'zh': ':卫兵_较浅肤色:',
        'ru': ':гвардеец_очень_светлый_тон_кожи:'
    },
    '\U0001F482\U0001F3FE': {  # 💂🏾
        'en': ':guard_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':wache_mitteldunkle_hautfarbe:',
        'es': ':guardia_tono_de_piel_oscuro_medio:',
        'fr': ':garde_peau_mate:',
        'ja': ':衛兵_やや濃い肌色:',
        'ko': ':근위병_진한_갈색_피부:',
        'pt': ':guarda_pele_morena_escura:',
        'it': ':guardia_carnagione_abbastanza_scura:',
        'fa': ':گارد_نظامی_پوست_گندمی:',
        'id': ':penjaga_warna_kulit_gelap-sedang:',
        'zh': ':卫兵_中等深肤色:',
        'ru': ':гвардеец_темный_тон_кожи:'
    },
    '\U0001F482\U0001F3FC': {  # 💂🏼
        'en': ':guard_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':wache_mittelhelle_hautfarbe:',
        'es': ':guardia_tono_de_piel_claro_medio:',
        'fr': ':garde_peau_moyennement_claire:',
        'ja': ':衛兵_やや薄い肌色:',
        'ko': ':근위병_연한_갈색_피부:',
        'pt': ':guarda_pele_morena_clara:',
        'it': ':guardia_carnagione_abbastanza_chiara:',
        'fa': ':گارد_نظامی_پوست_روشن:',
        'id': ':penjaga_warna_kulit_cerah-sedang:',
        'zh': ':卫兵_中等浅肤色:',
        'ru': ':гвардеец_светлый_тон_кожи:'
    },
    '\U0001F482\U0001F3FD': {  # 💂🏽
        'en': ':guard_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':wache_mittlere_hautfarbe:',
        'es': ':guardia_tono_de_piel_medio:',
        'fr': ':garde_peau_légèrement_mate:',
        'ja': ':衛兵_中間の肌色:',
        'ko': ':근위병_갈색_피부:',
        'pt': ':guarda_pele_morena:',
        'it': ':guardia_carnagione_olivastra:',
        'fa': ':گارد_نظامی_پوست_طلایی:',
        'id': ':penjaga_warna_kulit_sedang:',
        'zh': ':卫兵_中等肤色:',
        'ru': ':гвардеец_средний_тон_кожи:'
    },
    '\U0001F9AE': {  # 🦮
        'en': ':guide_dog:',
        'status': fully_qualified,
        'E': 12,
        'de': ':blindenhund:',
        'es': ':perro_guía:',
        'fr': ':chien_guide:',
        'ja': ':盲導犬:',
        'ko': ':안내견:',
        'pt': ':cão-guia:',
        'it': ':cane_guida:',
        'fa': ':سگ_راهنما:',
        'id': ':anjing_pemandu:',
        'zh': ':导盲犬:',
        'ru': ':собака_поводырь:'
    },
    '\U0001F3B8': {  # 🎸
        'en': ':guitar:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':gitarre:',
        'es': ':guitarra:',
        'fr': ':guitare:',
        'ja': ':ギター:',
        'ko': ':기타:',
        'pt': ':guitarra:',
        'it': ':chitarra:',
        'fa': ':گیتار:',
        'id': ':gitar:',
        'zh': ':吉他:',
        'ru': ':гитара:'
    },
    '\U0001FAAE': {  # 🪮
        'en': ':hair_pick:',
        'status': fully_qualified,
        'E': 15,
        'de': ':haarkamm:',
        'es': ':peineta:',
        'fr': ':peigne_afro:',
        'ja': ':アフロコーム:',
        'ko': ':머리_빗:',
        'pt': ':pente_de_cabelo:',
        'it': ':pettine_afro:',
        'fa': ':شانه_جداکننده:',
        'id': ':sisir_garpu:',
        'zh': ':发夹:',
        'ru': ':афро_гребень:'
    },
    '\U0001F354': {  # 🍔
        'en': ':hamburger:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':hamburger:',
        'es': ':hamburguesa:',
        'fr': ':hamburger:',
        'ja': ':ハンバーガー:',
        'ko': ':햄버거:',
        'pt': ':hambúrguer:',
        'it': ':hamburger:',
        'fa': ':همبرگر:',
        'id': ':hamburger:',
        'zh': ':汉堡:',
        'ru': ':гамбургер:'
    },
    '\U0001F528': {  # 🔨
        'en': ':hammer:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':hammer:',
        'es': ':martillo:',
        'fr': ':marteau:',
        'ja': ':ハンマー:',
        'ko': ':망치:',
        'pt': ':martelo:',
        'it': ':martello:',
        'fa': ':چکش:',
        'id': ':palu:',
        'zh': ':锤子:',
        'ru': ':молоток:'
    },
    '\U00002692\U0000FE0F': {  # ⚒️
        'en': ':hammer_and_pick:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':hammer_und_pickel:',
        'es': ':martillo_y_pico:',
        'fr': ':marteau_et_pioche:',
        'ja': ':ハンマーとつるはし:',
        'ko': ':망치와_곡괭이:',
        'pt': ':martelo_e_picareta:',
        'it': ':piccone_e_martello:',
        'fa': ':چکش_و_کلنگ:',
        'id': ':palu_dan_beliung:',
        'zh': ':锤子与镐:',
        'ru': ':кирка_и_молот:'
    },
    '\U00002692': {  # ⚒
        'en': ':hammer_and_pick:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':hammer_und_pickel:',
        'es': ':martillo_y_pico:',
        'fr': ':marteau_et_pioche:',
        'ja': ':ハンマーとつるはし:',
        'ko': ':망치와_곡괭이:',
        'pt': ':martelo_e_picareta:',
        'it': ':piccone_e_martello:',
        'fa': ':چکش_و_کلنگ:',
        'id': ':palu_dan_beliung:',
        'zh': ':锤子与镐:',
        'ru': ':кирка_и_молот:'
    },
    '\U0001F6E0\U0000FE0F': {  # 🛠️
        'en': ':hammer_and_wrench:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':hammer_und_schraubenschlüssel:',
        'es': ':martillo_y_llave_inglesa:',
        'fr': ':marteau_et_clé_à_molette:',
        'ja': ':ハンマーとレンチ:',
        'ko': ':망치와_렌치:',
        'pt': ':martelo_e_chave-inglesa:',
        'it': ':martello_e_chiave_inglese:',
        'fa': ':چکش_و_آچار:',
        'id': ':palu_dan_kunci_inggris:',
        'zh': ':锤子与扳手:',
        'ru': ':молот_и_гаечныи_ключ:'
    },
    '\U0001F6E0': {  # 🛠
        'en': ':hammer_and_wrench:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':hammer_und_schraubenschlüssel:',
        'es': ':martillo_y_llave_inglesa:',
        'fr': ':marteau_et_clé_à_molette:',
        'ja': ':ハンマーとレンチ:',
        'ko': ':망치와_렌치:',
        'pt': ':martelo_e_chave-inglesa:',
        'it': ':martello_e_chiave_inglese:',
        'fa': ':چکش_و_آچار:',
        'id': ':palu_dan_kunci_inggris:',
        'zh': ':锤子与扳手:',
        'ru': ':молот_и_гаечный_ключ:'
    },
    '\U0001FAAC': {  # 🪬
        'en': ':hamsa:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hamsa:',
        'es': ':hamsa:',
        'fr': ':main_de_fatma:',
        'ja': ':ハムサ:',
        'ko': ':함사:',
        'pt': ':hamsá:',
        'it': ':mano_di_fatima:',
        'fa': ':خمسه:',
        'id': ':hamsa:',
        'zh': ':法蒂玛之手:',
        'ru': ':хамса:'
    },
    '\U0001F439': {  # 🐹
        'en': ':hamster:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':hamster:',
        'es': ':hámster:',
        'fr': ':hamster:',
        'ja': ':ハムスターの顔:',
        'ko': ':햄스터_얼굴:',
        'pt': ':rosto_de_hamster:',
        'it': ':criceto:',
        'fa': ':همستر:',
        'id': ':hamster:',
        'zh': ':仓鼠:',
        'ru': ':морда_хомяка:'
    },
    '\U0001F590\U0000FE0F': {  # 🖐️
        'en': ':hand_with_fingers_splayed:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':raised_hand_with_fingers_splayed:'],
        'variant': True,
        'de': ':hand_mit_gespreizten_fingern:',
        'es': ':mano_abierta:',
        'fr': ':main_levée_doigts_écartés:',
        'ja': ':開いた手:',
        'ko': ':쫙_편_손바닥:',
        'pt': ':mão_aberta_com_os_dedos_separados:',
        'it': ':mano_aperta:',
        'fa': ':دست_با_انگشتان_باز:',
        'id': ':telapak_tangan_terbuka:',
        'zh': ':手掌:',
        'ru': ':раскрытая_ладонь:'
    },
    '\U0001F590': {  # 🖐
        'en': ':hand_with_fingers_splayed:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':raised_hand_with_fingers_splayed:'],
        'variant': True,
        'de': ':hand_mit_gespreizten_fingern:',
        'es': ':mano_abierta:',
        'fr': ':main_levée_doigts_écartés:',
        'ja': ':開いた手:',
        'ko': ':쫙_편_손바닥:',
        'pt': ':mão_aberta_com_os_dedos_separados:',
        'it': ':mano_aperta:',
        'fa': ':دست_با_انگشتان_باز:',
        'id': ':telapak_tangan_terbuka:',
        'zh': ':手掌:',
        'ru': ':раскрытая_ладонь:'
    },
    '\U0001F590\U0001F3FF': {  # 🖐🏿
        'en': ':hand_with_fingers_splayed_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':hand_mit_gespreizten_fingern_dunkle_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_oscuro:',
        'fr': ':main_levée_doigts_écartés_peau_foncée:',
        'ja': ':開いた手_濃い肌色:',
        'ko': ':쫙_편_손바닥_검은색_피부:',
        'pt': ':mão_aberta_com_os_dedos_separados_pele_escura:',
        'it': ':mano_aperta_carnagione_scura:',
        'fa': ':دست_با_انگشتان_باز_پوست_آبنوسی:',
        'id': ':telapak_tangan_terbuka_warna_kulit_gelap:',
        'zh': ':手掌_较深肤色:',
        'ru': ':раскрытая_ладонь_очень_темный_тон_кожи:'
    },
    '\U0001F590\U0001F3FB': {  # 🖐🏻
        'en': ':hand_with_fingers_splayed_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':hand_mit_gespreizten_fingern_helle_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_claro:',
        'fr': ':main_levée_doigts_écartés_peau_claire:',
        'ja': ':開いた手_薄い肌色:',
        'ko': ':쫙_편_손바닥_하얀_피부:',
        'pt': ':mão_aberta_com_os_dedos_separados_pele_clara:',
        'it': ':mano_aperta_carnagione_chiara:',
        'fa': ':دست_با_انگشتان_باز_پوست_سفید:',
        'id': ':telapak_tangan_terbuka_warna_kulit_cerah:',
        'zh': ':手掌_较浅肤色:',
        'ru': ':раскрытая_ладонь_очень_светлый_тон_кожи:'
    },
    '\U0001F590\U0001F3FE': {  # 🖐🏾
        'en': ':hand_with_fingers_splayed_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':hand_mit_gespreizten_fingern_mitteldunkle_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_oscuro_medio:',
        'fr': ':main_levée_doigts_écartés_peau_mate:',
        'ja': ':開いた手_やや濃い肌色:',
        'ko': ':쫙_편_손바닥_진한_갈색_피부:',
        'pt': ':mão_aberta_com_os_dedos_separados_pele_morena_escura:',
        'it': ':mano_aperta_carnagione_abbastanza_scura:',
        'fa': ':دست_با_انگشتان_باز_پوست_گندمی:',
        'id': ':telapak_tangan_terbuka_warna_kulit_gelap-sedang:',
        'zh': ':手掌_中等深肤色:',
        'ru': ':раскрытая_ладонь_темный_тон_кожи:'
    },
    '\U0001F590\U0001F3FC': {  # 🖐🏼
        'en': ':hand_with_fingers_splayed_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':hand_mit_gespreizten_fingern_mittelhelle_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_claro_medio:',
        'fr': ':main_levée_doigts_écartés_peau_moyennement_claire:',
        'ja': ':開いた手_やや薄い肌色:',
        'ko': ':쫙_편_손바닥_연한_갈색_피부:',
        'pt': ':mão_aberta_com_os_dedos_separados_pele_morena_clara:',
        'it': ':mano_aperta_carnagione_abbastanza_chiara:',
        'fa': ':دست_با_انگشتان_باز_پوست_روشن:',
        'id': ':telapak_tangan_terbuka_warna_kulit_cerah-sedang:',
        'zh': ':手掌_中等浅肤色:',
        'ru': ':раскрытая_ладонь_светлый_тон_кожи:'
    },
    '\U0001F590\U0001F3FD': {  # 🖐🏽
        'en': ':hand_with_fingers_splayed_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':hand_mit_gespreizten_fingern_mittlere_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_medio:',
        'fr': ':main_levée_doigts_écartés_peau_légèrement_mate:',
        'ja': ':開いた手_中間の肌色:',
        'ko': ':쫙_편_손바닥_갈색_피부:',
        'pt': ':mão_aberta_com_os_dedos_separados_pele_morena:',
        'it': ':mano_aperta_carnagione_olivastra:',
        'fa': ':دست_با_انگشتان_باز_پوست_طلایی:',
        'id': ':telapak_tangan_terbuka_warna_kulit_sedang:',
        'zh': ':手掌_中等肤色:',
        'ru': ':раскрытая_ладонь_средний_тон_кожи:'
    },
    '\U0001FAF0': {  # 🫰
        'en': ':hand_with_index_finger_and_thumb_crossed:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hand_mit_gekreuztem_zeigefinger_und_daumen:',
        'es': ':mano_con_dedo_índice_y_pulgar_cruzados:',
        'fr': ':main_avec_index_et_pouce_croisés:',
        'ja': ':親指と人差し指をクロス:',
        'ko': ':엄지와_중지를_교차한_손:',
        'pt': ':mão_com_dedo_indicador_e_polegar_cruzados:',
        'it': ':mano_con_indice_e_pollice_incrociati:',
        'fa': ':دست_با_انگشت_اشاره_و_شست_متقاطع:',
        'id': ':tangan_dengan_jari_telunjuk_dan_ibu_jari_menyilang:',
        'zh': ':食指与拇指交叉的手:',
        'ru': ':жест_«деньги»:'
    },
    '\U0001FAF0\U0001F3FF': {  # 🫰🏿
        'en': ':hand_with_index_finger_and_thumb_crossed_dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hand_mit_gekreuztem_zeigefinger_und_daumen_dunkle_hautfarbe:',
        'es': ':mano_con_dedo_índice_y_pulgar_cruzados_tono_de_piel_oscuro:',
        'fr': ':main_avec_index_et_pouce_croisés_peau_foncée:',
        'ja': ':親指と人差し指をクロス_濃い肌色:',
        'ko': ':엄지와_중지를_교차한_손_검은색_피부:',
        'pt': ':mão_com_dedo_indicador_e_polegar_cruzados_pele_escura:',
        'it': ':mano_con_indice_e_pollice_incrociati_carnagione_scura:',
        'fa': ':دست_با_انگشت_اشاره_و_شست_متقاطع_پوست_آبنوسی:',
        'id': ':tangan_dengan_jari_telunjuk_dan_ibu_jari_menyilang_warna_kulit_gelap:',
        'zh': ':食指与拇指交叉的手_较深肤色:',
        'ru': ':жест_«деньги»_очень_темный_тон_кожи:'
    },
    '\U0001FAF0\U0001F3FB': {  # 🫰🏻
        'en': ':hand_with_index_finger_and_thumb_crossed_light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hand_mit_gekreuztem_zeigefinger_und_daumen_helle_hautfarbe:',
        'es': ':mano_con_dedo_índice_y_pulgar_cruzados_tono_de_piel_claro:',
        'fr': ':main_avec_index_et_pouce_croisés_peau_claire:',
        'ja': ':親指と人差し指をクロス_薄い肌色:',
        'ko': ':엄지와_중지를_교차한_손_하얀_피부:',
        'pt': ':mão_com_dedo_indicador_e_polegar_cruzados_pele_clara:',
        'it': ':mano_con_indice_e_pollice_incrociati_carnagione_chiara:',
        'fa': ':دست_با_انگشت_اشاره_و_شست_متقاطع_پوست_سفید:',
        'id': ':tangan_dengan_jari_telunjuk_dan_ibu_jari_menyilang_warna_kulit_cerah:',
        'zh': ':食指与拇指交叉的手_较浅肤色:',
        'ru': ':жест_«деньги»_очень_светлый_тон_кожи:'
    },
    '\U0001FAF0\U0001F3FE': {  # 🫰🏾
        'en': ':hand_with_index_finger_and_thumb_crossed_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hand_mit_gekreuztem_zeigefinger_und_daumen_mitteldunkle_hautfarbe:',
        'es': ':mano_con_dedo_índice_y_pulgar_cruzados_tono_de_piel_oscuro_medio:',
        'fr': ':main_avec_index_et_pouce_croisés_peau_mate:',
        'ja': ':親指と人差し指をクロス_やや濃い肌色:',
        'ko': ':엄지와_중지를_교차한_손_진한_갈색_피부:',
        'pt': ':mão_com_dedo_indicador_e_polegar_cruzados_pele_morena_escura:',
        'it': ':mano_con_indice_e_pollice_incrociati_carnagione_abbastanza_scura:',
        'fa': ':دست_با_انگشت_اشاره_و_شست_متقاطع_پوست_گندمی:',
        'id': ':tangan_dengan_jari_telunjuk_dan_ibu_jari_menyilang_warna_kulit_gelap-sedang:',
        'zh': ':食指与拇指交叉的手_中等深肤色:',
        'ru': ':жест_«деньги»_темный_тон_кожи:'
    },
    '\U0001FAF0\U0001F3FC': {  # 🫰🏼
        'en': ':hand_with_index_finger_and_thumb_crossed_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hand_mit_gekreuztem_zeigefinger_und_daumen_mittelhelle_hautfarbe:',
        'es': ':mano_con_dedo_índice_y_pulgar_cruzados_tono_de_piel_claro_medio:',
        'fr': ':main_avec_index_et_pouce_croisés_peau_moyennement_claire:',
        'ja': ':親指と人差し指をクロス_やや薄い肌色:',
        'ko': ':엄지와_중지를_교차한_손_연한_갈색_피부:',
        'pt': ':mão_com_dedo_indicador_e_polegar_cruzados_pele_morena_clara:',
        'it': ':mano_con_indice_e_pollice_incrociati_carnagione_abbastanza_chiara:',
        'fa': ':دست_با_انگشت_اشاره_و_شست_متقاطع_پوست_روشن:',
        'id': ':tangan_dengan_jari_telunjuk_dan_ibu_jari_menyilang_warna_kulit_cerah-sedang:',
        'zh': ':食指与拇指交叉的手_中等浅肤色:',
        'ru': ':жест_«деньги»_светлый_тон_кожи:'
    },
    '\U0001FAF0\U0001F3FD': {  # 🫰🏽
        'en': ':hand_with_index_finger_and_thumb_crossed_medium_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hand_mit_gekreuztem_zeigefinger_und_daumen_mittlere_hautfarbe:',
        'es': ':mano_con_dedo_índice_y_pulgar_cruzados_tono_de_piel_medio:',
        'fr': ':main_avec_index_et_pouce_croisés_peau_légèrement_mate:',
        'ja': ':親指と人差し指をクロス_中間の肌色:',
        'ko': ':엄지와_중지를_교차한_손_갈색_피부:',
        'pt': ':mão_com_dedo_indicador_e_polegar_cruzados_pele_morena:',
        'it': ':mano_con_indice_e_pollice_incrociati_carnagione_olivastra:',
        'fa': ':دست_با_انگشت_اشاره_و_شست_متقاطع_پوست_طلایی:',
        'id': ':tangan_dengan_jari_telunjuk_dan_ibu_jari_menyilang_warna_kulit_sedang:',
        'zh': ':食指与拇指交叉的手_中等肤色:',
        'ru': ':жест_«деньги»_средний_тон_кожи:'
    },
    '\U0001F45C': {  # 👜
        'en': ':handbag:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':handtasche:',
        'es': ':bolso:',
        'fr': ':sac_à_main:',
        'ja': ':ハンドバッグ:',
        'ko': ':핸드백:',
        'pt': ':bolsa_de_mão:',
        'it': ':borsa:',
        'fa': ':کیف_دوشی:',
        'id': ':tas_jinjing:',
        'zh': ':手提包:',
        'ru': ':сумка:'
    },
    '\U0001F91D': {  # 🤝
        'en': ':handshake:',
        'status': fully_qualified,
        'E': 3,
        'de': ':handschlag:',
        'es': ':apretón_de_manos:',
        'fr': ':poignée_de_main:',
        'ja': ':握手:',
        'ko': ':악수:',
        'pt': ':aperto_de_mãos:',
        'it': ':stretta_di_mano:',
        'fa': ':دست_دادن:',
        'id': ':jabat_tangan:',
        'zh': ':握手:',
        'ru': ':рукопожатие:'
    },
    '\U0001F91D\U0001F3FF': {  # 🤝🏿
        'en': ':handshake_dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_dunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro:',
        'fr': ':poignée_de_main_peau_foncée:',
        'ja': ':握手_濃い肌色:',
        'ko': ':악수_검은색_피부:',
        'pt': ':aperto_de_mãos_pele_escura:',
        'it': ':stretta_di_mano_carnagione_scura:',
        'fa': ':دست_دادن_پوست_آبنوسی:',
        'id': ':jabat_tangan_warna_kulit_gelap:',
        'zh': ':握手_较深肤色:',
        'ru': ':рукопожатие_очень_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FB': {  # 🫱🏿‍🫲🏻
        'en': ':handshake_dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_dunkle_hautfarbe_helle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_y_tono_de_piel_claro:',
        'fr': ':poignée_de_main_peau_foncée_et_peau_claire:',
        'ja': ':握手_濃い肌色_薄い肌色:',
        'ko': ':악수_검은색_피부_하얀_피부:',
        'pt': ':aperto_de_mãos_pele_escura_e_pele_clara:',
        'it': ':stretta_di_mano_carnagione_scura_e_carnagione_chiara:',
        'fa': ':دست_دادن_پوست_آبنوسی_پوست_سفید:',
        'id': ':jabat_tangan_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':握手_较深肤色较浅肤色:',
        'ru': ':рукопожатие_очень_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FE': {  # 🫱🏿‍🫲🏾
        'en': ':handshake_dark_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_dunkle_hautfarbe_mitteldunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_y_tono_de_piel_oscuro_medio:',
        'fr': ':poignée_de_main_peau_foncée_et_peau_mate:',
        'ja': ':握手_濃い肌色_やや濃い肌色:',
        'ko': ':악수_검은색_피부_진한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_escura_e_pele_morena_escura:',
        'it': ':stretta_di_mano_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':دست_دادن_پوست_آبنوسی_پوست_گندمی:',
        'id': ':jabat_tangan_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':握手_较深肤色中等深肤色:',
        'ru': ':рукопожатие_очень_темный_тон_кожи_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FC': {  # 🫱🏿‍🫲🏼
        'en': ':handshake_dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_dunkle_hautfarbe_mittelhelle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_y_tono_de_piel_claro_medio:',
        'fr': ':poignée_de_main_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':握手_濃い肌色_やや薄い肌色:',
        'ko': ':악수_검은색_피부_연한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_escura_e_pele_morena_clara:',
        'it': ':stretta_di_mano_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':دست_دادن_پوست_آبنوسی_پوست_روشن:',
        'id': ':jabat_tangan_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':握手_较深肤色中等浅肤色:',
        'ru': ':рукопожатие_очень_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FD': {  # 🫱🏿‍🫲🏽
        'en': ':handshake_dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_dunkle_hautfarbe_mittlere_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_y_tono_de_piel_medio:',
        'fr': ':poignée_de_main_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':握手_濃い肌色_中間の肌色:',
        'ko': ':악수_검은색_피부_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_escura_e_pele_morena:',
        'it': ':stretta_di_mano_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':دست_دادن_پوست_آبنوسی_پوست_طلایی:',
        'id': ':jabat_tangan_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':握手_较深肤色中等肤色:',
        'ru': ':рукопожатие_очень_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F91D\U0001F3FB': {  # 🤝🏻
        'en': ':handshake_light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_helle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro:',
        'fr': ':poignée_de_main_peau_claire:',
        'ja': ':握手_薄い肌色:',
        'ko': ':악수_하얀_피부:',
        'pt': ':aperto_de_mãos_pele_clara:',
        'it': ':stretta_di_mano_carnagione_chiara:',
        'fa': ':دست_دادن_پوست_سفید:',
        'id': ':jabat_tangan_warna_kulit_cerah:',
        'zh': ':握手_较浅肤色:',
        'ru': ':рукопожатие_очень_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FF': {  # 🫱🏻‍🫲🏿
        'en': ':handshake_light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_helle_hautfarbe_dunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_y_tono_de_piel_oscuro:',
        'fr': ':poignée_de_main_peau_claire_et_peau_foncée:',
        'ja': ':握手_薄い肌色_濃い肌色:',
        'ko': ':악수_하얀_피부_검은색_피부:',
        'pt': ':aperto_de_mãos_pele_clara_e_pele_escura:',
        'it': ':stretta_di_mano_carnagione_chiara_e_carnagione_scura:',
        'fa': ':دست_دادن_پوست_سفید_پوست_آبنوسی:',
        'id': ':jabat_tangan_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':握手_较浅肤色较深肤色:',
        'ru': ':рукопожатие_очень_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FE': {  # 🫱🏻‍🫲🏾
        'en': ':handshake_light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_helle_hautfarbe_mitteldunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_y_tono_de_piel_oscuro_medio:',
        'fr': ':poignée_de_main_peau_claire_et_peau_mate:',
        'ja': ':握手_薄い肌色_やや濃い肌色:',
        'ko': ':악수_하얀_피부_진한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_clara_e_pele_morena_escura:',
        'it': ':stretta_di_mano_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':دست_دادن_پوست_سفید_پوست_گندمی:',
        'id': ':jabat_tangan_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':握手_较浅肤色中等深肤色:',
        'ru': ':рукопожатие_очень_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FC': {  # 🫱🏻‍🫲🏼
        'en': ':handshake_light_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_helle_hautfarbe_mittelhelle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_y_tono_de_piel_claro_medio:',
        'fr': ':poignée_de_main_peau_claire_et_peau_moyennement_claire:',
        'ja': ':握手_薄い肌色_やや薄い肌色:',
        'ko': ':악수_하얀_피부_연한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_clara_e_pele_morena_clara:',
        'it': ':stretta_di_mano_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':دست_دادن_پوست_سفید_پوست_روشن:',
        'id': ':jabat_tangan_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':握手_较浅肤色中等浅肤色:',
        'ru': ':рукопожатие_очень_светлый_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FD': {  # 🫱🏻‍🫲🏽
        'en': ':handshake_light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_helle_hautfarbe_mittlere_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_y_tono_de_piel_medio:',
        'fr': ':poignée_de_main_peau_claire_et_peau_légèrement_mate:',
        'ja': ':握手_薄い肌色_中間の肌色:',
        'ko': ':악수_하얀_피부_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_clara_e_pele_morena:',
        'it': ':stretta_di_mano_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':دست_دادن_پوست_سفید_پوست_طلایی:',
        'id': ':jabat_tangan_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':握手_较浅肤色中等肤色:',
        'ru': ':рукопожатие_очень_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F91D\U0001F3FE': {  # 🤝🏾
        'en': ':handshake_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mitteldunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_medio:',
        'fr': ':poignée_de_main_peau_mate:',
        'ja': ':握手_やや濃い肌色:',
        'ko': ':악수_진한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_escura:',
        'it': ':stretta_di_mano_carnagione_abbastanza_scura:',
        'fa': ':دست_دادن_پوست_گندمی:',
        'id': ':jabat_tangan_warna_kulit_gelap-sedang:',
        'zh': ':握手_中等深肤色:',
        'ru': ':рукопожатие_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FF': {  # 🫱🏾‍🫲🏿
        'en': ':handshake_medium-dark_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mitteldunkle_hautfarbe_dunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_medio_y_tono_de_piel_oscuro:',
        'fr': ':poignée_de_main_peau_mate_et_peau_foncée:',
        'ja': ':握手_やや濃い肌色_濃い肌色:',
        'ko': ':악수_진한_갈색_피부_검은색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_escura_e_pele_escura:',
        'it': ':stretta_di_mano_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':دست_دادن_پوست_گندمی_پوست_آبنوسی:',
        'id': ':jabat_tangan_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':握手_中等深肤色较深肤色:',
        'ru': ':рукопожатие_темный_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FB': {  # 🫱🏾‍🫲🏻
        'en': ':handshake_medium-dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mitteldunkle_hautfarbe_helle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_medio_y_tono_de_piel_claro:',
        'fr': ':poignée_de_main_peau_mate_et_peau_claire:',
        'ja': ':握手_やや濃い肌色_薄い肌色:',
        'ko': ':악수_진한_갈색_피부_하얀_피부:',
        'pt': ':aperto_de_mãos_pele_morena_escura_e_pele_clara:',
        'it': ':stretta_di_mano_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':دست_دادن_پوست_گندمی_پوست_سفید:',
        'id': ':jabat_tangan_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':握手_中等深肤色较浅肤色:',
        'ru': ':рукопожатие_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FC': {  # 🫱🏾‍🫲🏼
        'en': ':handshake_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mitteldunkle_hautfarbe_mittelhelle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_medio_y_tono_de_piel_claro_medio:',
        'fr': ':poignée_de_main_peau_mate_et_peau_moyennement_claire:',
        'ja': ':握手_やや濃い肌色_やや薄い肌色:',
        'ko': ':악수_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_escura_e_pele_morena_clara:',
        'it': ':stretta_di_mano_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':دست_دادن_پوست_گندمی_پوست_روشن:',
        'id': ':jabat_tangan_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':握手_中等深肤色中等浅肤色:',
        'ru': ':рукопожатие_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FD': {  # 🫱🏾‍🫲🏽
        'en': ':handshake_medium-dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mitteldunkle_hautfarbe_mittlere_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_oscuro_medio_y_tono_de_piel_medio:',
        'fr': ':poignée_de_main_peau_mate_et_peau_légèrement_mate:',
        'ja': ':握手_やや濃い肌色_中間の肌色:',
        'ko': ':악수_진한_갈색_피부_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_escura_e_pele_morena:',
        'it': ':stretta_di_mano_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':دست_دادن_پوست_گندمی_پوست_طلایی:',
        'id': ':jabat_tangan_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':握手_中等深肤色中等肤色:',
        'ru': ':рукопожатие_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F91D\U0001F3FC': {  # 🤝🏼
        'en': ':handshake_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittelhelle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_medio:',
        'fr': ':poignée_de_main_peau_moyennement_claire:',
        'ja': ':握手_やや薄い肌色:',
        'ko': ':악수_연한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_clara:',
        'it': ':stretta_di_mano_carnagione_abbastanza_chiara:',
        'fa': ':دست_دادن_پوست_روشن:',
        'id': ':jabat_tangan_warna_kulit_cerah-sedang:',
        'zh': ':握手_中等浅肤色:',
        'ru': ':рукопожатие_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FF': {  # 🫱🏼‍🫲🏿
        'en': ':handshake_medium-light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittelhelle_hautfarbe_dunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_medio_y_tono_de_piel_oscuro:',
        'fr': ':poignée_de_main_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':握手_やや薄い肌色_濃い肌色:',
        'ko': ':악수_연한_갈색_피부_검은색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_clara_e_pele_escura:',
        'it': ':stretta_di_mano_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':دست_دادن_پوست_روشن_پوست_آبنوسی:',
        'id': ':jabat_tangan_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':握手_中等浅肤色较深肤色:',
        'ru': ':рукопожатие_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FB': {  # 🫱🏼‍🫲🏻
        'en': ':handshake_medium-light_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittelhelle_hautfarbe_helle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_medio_y_tono_de_piel_claro:',
        'fr': ':poignée_de_main_peau_moyennement_claire_et_peau_claire:',
        'ja': ':握手_やや薄い肌色_薄い肌色:',
        'ko': ':악수_연한_갈색_피부_하얀_피부:',
        'pt': ':aperto_de_mãos_pele_morena_clara_e_pele_clara:',
        'it': ':stretta_di_mano_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':دست_دادن_پوست_روشن_پوست_سفید:',
        'id': ':jabat_tangan_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':握手_中等浅肤色较浅肤色:',
        'ru': ':рукопожатие_светлый_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FE': {  # 🫱🏼‍🫲🏾
        'en': ':handshake_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittelhelle_hautfarbe_mitteldunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_medio_y_tono_de_piel_oscuro_medio:',
        'fr': ':poignée_de_main_peau_moyennement_claire_et_peau_mate:',
        'ja': ':握手_やや薄い肌色_やや濃い肌色:',
        'ko': ':악수_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_clara_e_pele_morena_escura:',
        'it': ':stretta_di_mano_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':دست_دادن_پوست_روشن_پوست_گندمی:',
        'id': ':jabat_tangan_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':握手_中等浅肤色中等深肤色:',
        'ru': ':рукопожатие_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FD': {  # 🫱🏼‍🫲🏽
        'en': ':handshake_medium-light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittelhelle_hautfarbe_mittlere_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_claro_medio_y_tono_de_piel_medio:',
        'fr': ':poignée_de_main_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':握手_やや薄い肌色_中間の肌色:',
        'ko': ':악수_연한_갈색_피부_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_clara_e_pele_morena:',
        'it': ':stretta_di_mano_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':دست_دادن_پوست_روشن_پوست_طلایی:',
        'id': ':jabat_tangan_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':握手_中等浅肤色中等肤色:',
        'ru': ':рукопожатие_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F91D\U0001F3FD': {  # 🤝🏽
        'en': ':handshake_medium_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittlere_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_medio:',
        'fr': ':poignée_de_main_peau_légèrement_mate:',
        'ja': ':握手_中間の肌色:',
        'ko': ':악수_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena:',
        'it': ':stretta_di_mano_carnagione_olivastra:',
        'fa': ':دست_دادن_پوست_طلایی:',
        'id': ':jabat_tangan_warna_kulit_sedang:',
        'zh': ':握手_中等肤色:',
        'ru': ':рукопожатие_средний_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FF': {  # 🫱🏽‍🫲🏿
        'en': ':handshake_medium_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittlere_hautfarbe_dunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_medio_y_tono_de_piel_oscuro:',
        'fr': ':poignée_de_main_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':握手_中間の肌色_濃い肌色:',
        'ko': ':악수_갈색_피부_검은색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_e_pele_escura:',
        'it': ':stretta_di_mano_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':دست_دادن_پوست_طلایی_پوست_آبنوسی:',
        'id': ':jabat_tangan_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':握手_中等肤色较深肤色:',
        'ru': ':рукопожатие_средний_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FB': {  # 🫱🏽‍🫲🏻
        'en': ':handshake_medium_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittlere_hautfarbe_helle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_medio_y_tono_de_piel_claro:',
        'fr': ':poignée_de_main_peau_légèrement_mate_et_peau_claire:',
        'ja': ':握手_中間の肌色_薄い肌色:',
        'ko': ':악수_갈색_피부_하얀_피부:',
        'pt': ':aperto_de_mãos_pele_morena_e_pele_clara:',
        'it': ':stretta_di_mano_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':دست_دادن_پوست_طلایی_پوست_سفید:',
        'id': ':jabat_tangan_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':握手_中等肤色较浅肤色:',
        'ru': ':рукопожатие_средний_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FE': {  # 🫱🏽‍🫲🏾
        'en': ':handshake_medium_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittlere_hautfarbe_mitteldunkle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_medio_y_tono_de_piel_oscuro_medio:',
        'fr': ':poignée_de_main_peau_légèrement_mate_et_peau_mate:',
        'ja': ':握手_中間の肌色_やや濃い肌色:',
        'ko': ':악수_갈색_피부_진한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_e_pele_morena_escura:',
        'it': ':stretta_di_mano_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':دست_دادن_پوست_طلایی_پوست_گندمی:',
        'id': ':jabat_tangan_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':握手_中等肤色中等深肤色:',
        'ru': ':рукопожатие_средний_тон_кожи_темный_тон_кожи:'
    },
    '\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FC': {  # 🫱🏽‍🫲🏼
        'en': ':handshake_medium_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':handschlag_mittlere_hautfarbe_mittelhelle_hautfarbe:',
        'es': ':apretón_de_manos_tono_de_piel_medio_y_tono_de_piel_claro_medio:',
        'fr': ':poignée_de_main_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':握手_中間の肌色_やや薄い肌色:',
        'ko': ':악수_갈색_피부_연한_갈색_피부:',
        'pt': ':aperto_de_mãos_pele_morena_e_pele_morena_clara:',
        'it': ':stretta_di_mano_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':دست_دادن_پوست_طلایی_پوست_روشن:',
        'id': ':jabat_tangan_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':握手_中等肤色中等浅肤色:',
        'ru': ':рукопожатие_средний_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F423': {  # 🐣
        'en': ':hatching_chick:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':schlüpfendes_küken:',
        'es': ':pollito_rompiendo_el_cascarón:',
        'fr': ':poussin_qui_éclôt:',
        'ja': ':卵からかえったひよこ:',
        'ko': ':알에서_깬_병아리:',
        'pt': ':pintinho_chocando:',
        'it': ':pulcino_che_nasce:',
        'fa': ':جوجه_ازتخم_بیرون_آمده:',
        'id': ':itik_menetas:',
        'zh': ':小鸡破壳:',
        'ru': ':цыпленок_в_яйце:'
    },
    '\U0001F3A7': {  # 🎧
        'en': ':headphone:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':headphones:'],
        'variant': True,
        'de': ':kopfhörer:',
        'es': ':auricular:',
        'fr': ':casque:',
        'ja': ':ヘッドホン:',
        'ko': ':헤드폰:',
        'pt': ':fones_de_ouvido:',
        'it': ':cuffie:',
        'fa': ':هدفون:',
        'id': ':headphone:',
        'zh': ':耳机:',
        'ru': ':наушники:'
    },
    '\U0001FAA6': {  # 🪦
        'en': ':headstone:',
        'status': fully_qualified,
        'E': 13,
        'de': ':grabstein:',
        'es': ':lápida:',
        'fr': ':pierre_tombale:',
        'ja': ':墓石:',
        'ko': ':묘비:',
        'pt': ':lápide:',
        'it': ':lapide:',
        'fa': ':سنگ_قبر:',
        'id': ':batu_nisan:',
        'zh': ':墓碑:',
        'ru': ':надгробие:'
    },
    '\U0001F9D1\U0000200D\U00002695\U0000FE0F': {  # 🧑‍⚕️
        'en': ':health_worker:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin:',
        'es': ':profesional_sanitario:',
        'fr': ':professionnel_de_la_santé_(tous_genres):',
        'ja': ':医者:',
        'ko': ':의료인:',
        'pt': ':profissional_de_saúde:',
        'it': ':persona_che_lavora_nella_sanità:',
        'fa': ':مأمور_بهداشت:',
        'id': ':petugas_kesehatan:',
        'zh': ':卫生工作者:',
        'ru': ':медицинскии_работник:'
    },
    '\U0001F9D1\U0000200D\U00002695': {  # 🧑‍⚕
        'en': ':health_worker:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin:',
        'es': ':profesional_sanitario:',
        'fr': ':professionnel_de_la_santé_(tous_genres):',
        'ja': ':医者:',
        'ko': ':의료인:',
        'pt': ':profissional_de_saúde:',
        'it': ':persona_che_lavora_nella_sanità:',
        'fa': ':مأمور_بهداشت:',
        'id': ':petugas_kesehatan:',
        'zh': ':卫生工作者:',
        'ru': ':медицинский_работник:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002695\U0000FE0F': {  # 🧑🏿‍⚕️
        'en': ':health_worker_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_dunkle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_oscuro:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_foncée:',
        'ja': ':医者_濃い肌色:',
        'ko': ':의료인_검은색_피부:',
        'pt': ':profissional_de_saúde_pele_escura:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_scura:',
        'fa': ':مأمور_بهداشت_پوست_آبنوسی:',
        'id': ':petugas_kesehatan_warna_kulit_gelap:',
        'zh': ':卫生工作者_较深肤色:',
        'ru': ':медицинскии_работник_очень_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002695': {  # 🧑🏿‍⚕
        'en': ':health_worker_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_dunkle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_oscuro:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_foncée:',
        'ja': ':医者_濃い肌色:',
        'ko': ':의료인_검은색_피부:',
        'pt': ':profissional_de_saúde_pele_escura:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_scura:',
        'fa': ':مأمور_بهداشت_پوست_آبنوسی:',
        'id': ':petugas_kesehatan_warna_kulit_gelap:',
        'zh': ':卫生工作者_较深肤色:',
        'ru': ':медицинский_работник_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002695\U0000FE0F': {  # 🧑🏻‍⚕️
        'en': ':health_worker_light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_helle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_claro:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_claire:',
        'ja': ':医者_薄い肌色:',
        'ko': ':의료인_하얀_피부:',
        'pt': ':profissional_de_saúde_pele_clara:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_chiara:',
        'fa': ':مأمور_بهداشت_پوست_سفید:',
        'id': ':petugas_kesehatan_warna_kulit_cerah:',
        'zh': ':卫生工作者_较浅肤色:',
        'ru': ':медицинскии_работник_очень_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002695': {  # 🧑🏻‍⚕
        'en': ':health_worker_light_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_helle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_claro:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_claire:',
        'ja': ':医者_薄い肌色:',
        'ko': ':의료인_하얀_피부:',
        'pt': ':profissional_de_saúde_pele_clara:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_chiara:',
        'fa': ':مأمور_بهداشت_پوست_سفید:',
        'id': ':petugas_kesehatan_warna_kulit_cerah:',
        'zh': ':卫生工作者_较浅肤色:',
        'ru': ':медицинский_работник_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002695\U0000FE0F': {  # 🧑🏾‍⚕️
        'en': ':health_worker_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_mitteldunkle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_oscuro_medio:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_mate:',
        'ja': ':医者_やや濃い肌色:',
        'ko': ':의료인_진한_갈색_피부:',
        'pt': ':profissional_de_saúde_pele_morena_escura:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_abbastanza_scura:',
        'fa': ':مأمور_بهداشت_پوست_گندمی:',
        'id': ':petugas_kesehatan_warna_kulit_gelap-sedang:',
        'zh': ':卫生工作者_中等深肤色:',
        'ru': ':медицинскии_работник_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002695': {  # 🧑🏾‍⚕
        'en': ':health_worker_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_mitteldunkle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_oscuro_medio:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_mate:',
        'ja': ':医者_やや濃い肌色:',
        'ko': ':의료인_진한_갈색_피부:',
        'pt': ':profissional_de_saúde_pele_morena_escura:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_abbastanza_scura:',
        'fa': ':مأمور_بهداشت_پوست_گندمی:',
        'id': ':petugas_kesehatan_warna_kulit_gelap-sedang:',
        'zh': ':卫生工作者_中等深肤色:',
        'ru': ':медицинский_работник_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002695\U0000FE0F': {  # 🧑🏼‍⚕️
        'en': ':health_worker_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_mittelhelle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_claro_medio:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_moyennement_claire:',
        'ja': ':医者_やや薄い肌色:',
        'ko': ':의료인_연한_갈색_피부:',
        'pt': ':profissional_de_saúde_pele_morena_clara:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_abbastanza_chiara:',
        'fa': ':مأمور_بهداشت_پوست_روشن:',
        'id': ':petugas_kesehatan_warna_kulit_cerah-sedang:',
        'zh': ':卫生工作者_中等浅肤色:',
        'ru': ':медицинскии_работник_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002695': {  # 🧑🏼‍⚕
        'en': ':health_worker_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_mittelhelle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_claro_medio:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_moyennement_claire:',
        'ja': ':医者_やや薄い肌色:',
        'ko': ':의료인_연한_갈색_피부:',
        'pt': ':profissional_de_saúde_pele_morena_clara:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_abbastanza_chiara:',
        'fa': ':مأمور_بهداشت_پوست_روشن:',
        'id': ':petugas_kesehatan_warna_kulit_cerah-sedang:',
        'zh': ':卫生工作者_中等浅肤色:',
        'ru': ':медицинский_работник_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002695\U0000FE0F': {  # 🧑🏽‍⚕️
        'en': ':health_worker_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_mittlere_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_medio:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_légèrement_mate:',
        'ja': ':医者_中間の肌色:',
        'ko': ':의료인_갈색_피부:',
        'pt': ':profissional_de_saúde_pele_morena:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_olivastra:',
        'fa': ':مأمور_بهداشت_پوست_طلایی:',
        'id': ':petugas_kesehatan_warna_kulit_sedang:',
        'zh': ':卫生工作者_中等肤色:',
        'ru': ':медицинскии_работник_среднии_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002695': {  # 🧑🏽‍⚕
        'en': ':health_worker_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':arzt/ärztin_mittlere_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_medio:',
        'fr': ':professionnel_de_la_santé_(tous_genres)_peau_légèrement_mate:',
        'ja': ':医者_中間の肌色:',
        'ko': ':의료인_갈색_피부:',
        'pt': ':profissional_de_saúde_pele_morena:',
        'it': ':persona_che_lavora_nella_sanità_carnagione_olivastra:',
        'fa': ':مأمور_بهداشت_پوست_طلایی:',
        'id': ':petugas_kesehatan_warna_kulit_sedang:',
        'zh': ':卫生工作者_中等肤色:',
        'ru': ':медицинский_работник_средний_тон_кожи:'
    },
    '\U0001F649': {  # 🙉
        'en': ':hear-no-evil_monkey:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':hear_no_evil:', ':hear_no_evil_monkey:'],
        'de': ':sich_die_ohren_zuhaltendes_affengesicht:',
        'es': ':mono_con_los_oídos_tapados:',
        'fr': ':singe_ne_rien_entendre:',
        'ja': ':聞かざる:',
        'ko': ':귀를_막고_있는_원숭이:',
        'pt': ':macaco_que_não_ouve_nada:',
        'it': ':non_sento:',
        'fa': ':میمون_شرنشنو:',
        'id': ':tidak_mendengar:',
        'zh': ':非礼勿听:',
        'ru': ':ничего_не_слышу:'
    },
    '\U0001F49F': {  # 💟
        'en': ':heart_decoration:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':herzdekoration:',
        'es': ':adorno_de_corazón:',
        'fr': ':décoration_avec_cœur:',
        'ja': ':ハートデコ:',
        'ko': ':하트_장식:',
        'pt': ':coração_decorativo:',
        'it': ':decorazione_con_cuore:',
        'fa': ':قلب_تزئینی:',
        'id': ':dekorasi_hati:',
        'zh': ':心型装饰:',
        'ru': ':значок_«сердце»:'
    },
    '\U00002763\U0000FE0F': {  # ❣️
        'en': ':heart_exclamation:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':heavy_heart_exclamation:', ':heavy_heart_exclamation_mark_ornament:'],
        'variant': True,
        'de': ':herz_als_ausrufezeichen:',
        'es': ':exclamación_de_corazón:',
        'fr': ':cœur_point_d’exclamation:',
        'ja': ':ハートのびっくり:',
        'ko': ':하트_모양_느낌표:',
        'pt': ':exclamação_de_coração:',
        'it': ':punto_esclamativo_a_cuore:',
        'fa': ':علامت_تعجب_قلبی:',
        'id': ':tanda_seru_berbentuk_hati:',
        'zh': ':心叹号:',
        'ru': ':восклицательныи_знак_в_виде_сердца:'
    },
    '\U00002763': {  # ❣
        'en': ':heart_exclamation:',
        'status': unqualified,
        'E': 1,
        'alias': [':heavy_heart_exclamation:', ':heavy_heart_exclamation_mark_ornament:'],
        'variant': True,
        'de': ':herz_als_ausrufezeichen:',
        'es': ':exclamación_de_corazón:',
        'fr': ':cœur_point_d’exclamation:',
        'ja': ':ハートのびっくり:',
        'ko': ':하트_모양_느낌표:',
        'pt': ':exclamação_de_coração:',
        'it': ':punto_esclamativo_a_cuore:',
        'fa': ':علامت_تعجب_قلبی:',
        'id': ':tanda_seru_berbentuk_hati:',
        'zh': ':心叹号:',
        'ru': ':восклицательный_знак_в_виде_сердца:'
    },
    '\U0001FAF6': {  # 🫶
        'en': ':heart_hands:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hände_die_herz_bilden:',
        'es': ':manos_formando_un_corazón:',
        'fr': ':mains_qui_forment_un_cœur:',
        'ja': ':ハート形の手:',
        'ko': ':손_하트:',
        'pt': ':mãos_de_coração:',
        'it': ':mani_a_cuore:',
        'fa': ':شکل_قلب_با_دست:',
        'id': ':tangan_membentuk_hati:',
        'zh': ':做成心形的双手:',
        'ru': ':жест_«сердце»:'
    },
    '\U0001FAF6\U0001F3FF': {  # 🫶🏿
        'en': ':heart_hands_dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hände_die_herz_bilden_dunkle_hautfarbe:',
        'es': ':manos_formando_un_corazón_tono_de_piel_oscuro:',
        'fr': ':mains_qui_forment_un_cœur_peau_foncée:',
        'ja': ':ハート形の手_濃い肌色:',
        'ko': ':손_하트_검은색_피부:',
        'pt': ':mãos_de_coração_pele_escura:',
        'it': ':mani_a_cuore_carnagione_scura:',
        'fa': ':شکل_قلب_با_دست_پوست_آبنوسی:',
        'id': ':tangan_membentuk_hati_warna_kulit_gelap:',
        'zh': ':做成心形的双手_较深肤色:',
        'ru': ':жест_«сердце»_очень_темный_тон_кожи:'
    },
    '\U0001FAF6\U0001F3FB': {  # 🫶🏻
        'en': ':heart_hands_light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hände_die_herz_bilden_helle_hautfarbe:',
        'es': ':manos_formando_un_corazón_tono_de_piel_claro:',
        'fr': ':mains_qui_forment_un_cœur_peau_claire:',
        'ja': ':ハート形の手_薄い肌色:',
        'ko': ':손_하트_하얀_피부:',
        'pt': ':mãos_de_coração_pele_clara:',
        'it': ':mani_a_cuore_carnagione_chiara:',
        'fa': ':شکل_قلب_با_دست_پوست_سفید:',
        'id': ':tangan_membentuk_hati_warna_kulit_cerah:',
        'zh': ':做成心形的双手_较浅肤色:',
        'ru': ':жест_«сердце»_очень_светлый_тон_кожи:'
    },
    '\U0001FAF6\U0001F3FE': {  # 🫶🏾
        'en': ':heart_hands_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hände_die_herz_bilden_mitteldunkle_hautfarbe:',
        'es': ':manos_formando_un_corazón_tono_de_piel_oscuro_medio:',
        'fr': ':mains_qui_forment_un_cœur_peau_mate:',
        'ja': ':ハート形の手_やや濃い肌色:',
        'ko': ':손_하트_진한_갈색_피부:',
        'pt': ':mãos_de_coração_pele_morena_escura:',
        'it': ':mani_a_cuore_carnagione_abbastanza_scura:',
        'fa': ':شکل_قلب_با_دست_پوست_گندمی:',
        'id': ':tangan_membentuk_hati_warna_kulit_gelap-sedang:',
        'zh': ':做成心形的双手_中等深肤色:',
        'ru': ':жест_«сердце»_темный_тон_кожи:'
    },
    '\U0001FAF6\U0001F3FC': {  # 🫶🏼
        'en': ':heart_hands_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hände_die_herz_bilden_mittelhelle_hautfarbe:',
        'es': ':manos_formando_un_corazón_tono_de_piel_claro_medio:',
        'fr': ':mains_qui_forment_un_cœur_peau_moyennement_claire:',
        'ja': ':ハート形の手_やや薄い肌色:',
        'ko': ':손_하트_연한_갈색_피부:',
        'pt': ':mãos_de_coração_pele_morena_clara:',
        'it': ':mani_a_cuore_carnagione_abbastanza_chiara:',
        'fa': ':شکل_قلب_با_دست_پوست_روشن:',
        'id': ':tangan_membentuk_hati_warna_kulit_cerah-sedang:',
        'zh': ':做成心形的双手_中等浅肤色:',
        'ru': ':жест_«сердце»_светлый_тон_кожи:'
    },
    '\U0001FAF6\U0001F3FD': {  # 🫶🏽
        'en': ':heart_hands_medium_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':hände_die_herz_bilden_mittlere_hautfarbe:',
        'es': ':manos_formando_un_corazón_tono_de_piel_medio:',
        'fr': ':mains_qui_forment_un_cœur_peau_légèrement_mate:',
        'ja': ':ハート形の手_中間の肌色:',
        'ko': ':손_하트_갈색_피부:',
        'pt': ':mãos_de_coração_pele_morena:',
        'it': ':mani_a_cuore_carnagione_olivastra:',
        'fa': ':شکل_قلب_با_دست_پوست_طلایی:',
        'id': ':tangan_membentuk_hati_warna_kulit_sedang:',
        'zh': ':做成心形的双手_中等肤色:',
        'ru': ':жест_«сердце»_средний_тон_кожи:'
    },
    '\U00002764\U0000FE0F\U0000200D\U0001F525': {  # ❤️‍🔥
        'en': ':heart_on_fire:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':herz_in_flammen:',
        'es': ':corazón_en_llamas:',
        'fr': ':cœur_enflammé:',
        'ja': ':燃えるハート:',
        'ko': ':불_난_하트:',
        'pt': ':coração_em_chamas:',
        'it': ':cuore_in_fiamme:',
        'fa': ':قلب_در_آتش:',
        'id': ':hati_berapi-api:',
        'zh': ':火上之心:',
        'ru': ':горящее_сердце:'
    },
    '\U00002764\U0000200D\U0001F525': {  # ❤‍🔥
        'en': ':heart_on_fire:',
        'status': unqualified,
        'E': 13.1,
        'de': ':herz_in_flammen:',
        'es': ':corazón_en_llamas:',
        'fr': ':cœur_enflammé:',
        'ja': ':燃えるハート:',
        'ko': ':불_난_하트:',
        'pt': ':coração_em_chamas:',
        'it': ':cuore_in_fiamme:',
        'fa': ':قلب_در_آتش:',
        'id': ':hati_berapi-api:',
        'zh': ':火上之心:',
        'ru': ':горящее_сердце:'
    },
    '\U00002665\U0000FE0F': {  # ♥️
        'en': ':heart_suit:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':hearts:'],
        'variant': True,
        'de': ':herz:',
        'es': ':palo_de_corazones:',
        'fr': ':cœur_cartes:',
        'ja': ':ハート:',
        'ko': ':하트:',
        'pt': ':naipe_de_copas:',
        'it': ':cuori:',
        'fa': ':خال_دل:',
        'id': ':hati:',
        'zh': ':红桃:',
        'ru': ':червы:'
    },
    '\U00002665': {  # ♥
        'en': ':heart_suit:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':hearts:'],
        'variant': True,
        'de': ':herz:',
        'es': ':palo_de_corazones:',
        'fr': ':cœur_cartes:',
        'ja': ':ハート:',
        'ko': ':하트:',
        'pt': ':naipe_de_copas:',
        'it': ':cuori:',
        'fa': ':خال_دل:',
        'id': ':hati:',
        'zh': ':红桃:',
        'ru': ':червы:'
    },
    '\U0001F498': {  # 💘
        'en': ':heart_with_arrow:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':cupid:'],
        'de': ':herz_mit_pfeil:',
        'es': ':corazón_con_flecha:',
        'fr': ':cœur_et_flèche:',
        'ja': ':ハートに矢:',
        'ko': ':화살이_꽂힌_하트:',
        'pt': ':coração_com_flecha:',
        'it': ':cuore_con_freccia:',
        'fa': ':قلب_تیرخورده:',
        'id': ':hati_terpanah:',
        'zh': ':心中箭了:',
        'ru': ':сердце_пронзенное_стрелой:'
    },
    '\U0001F49D': {  # 💝
        'en': ':heart_with_ribbon:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':gift_heart:'],
        'de': ':herz_mit_schleife:',
        'es': ':corazón_con_lazo:',
        'fr': ':cœur_avec_ruban:',
        'ja': ':ハートにリボン:',
        'ko': ':리본_달린_하트:',
        'pt': ':coração_com_fita:',
        'it': ':cuore_con_fiocco:',
        'fa': ':قلب_با_روبان:',
        'id': ':hati_berpita:',
        'zh': ':系有缎带的心:',
        'ru': ':сердце_с_лентой:'
    },
    '\U0001F4B2': {  # 💲
        'en': ':heavy_dollar_sign:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':dollarzeichen_extrafett:',
        'es': ':símbolo_de_dólar:',
        'fr': ':symbole_dollar:',
        'ja': ':ドル記号:',
        'ko': ':미국_달러:',
        'pt': ':cifrão:',
        'it': ':dollaro:',
        'fa': ':نماد_دلار:',
        'id': ':tanda_dolar_tebal:',
        'zh': ':粗美元符号:',
        'ru': ':жирный_знак_доллара:'
    },
    '\U0001F7F0': {  # 🟰
        'en': ':heavy_equals_sign:',
        'status': fully_qualified,
        'E': 14,
        'de': ':gleichheitszeichen_extrafett:',
        'es': ':signo_igual_grueso:',
        'fr': ':signe_égal_gras:',
        'ja': ':太字の等号:',
        'ko': ':등호_기호:',
        'pt': ':sinal_de_igual:',
        'it': ':segno_di_uguaglianza:',
        'fa': ':علامت_تساوی_پررنگ:',
        'id': ':tanda_sama_dengan_tebal:',
        'zh': ':粗等号:',
        'ru': ':жирный_знак_равенства:'
    },
    '\U0001F994': {  # 🦔
        'en': ':hedgehog:',
        'status': fully_qualified,
        'E': 5,
        'de': ':igel:',
        'es': ':erizo:',
        'fr': ':hérisson:',
        'ja': ':ハリネズミ:',
        'ko': ':고슴도치:',
        'pt': ':porco-espinho:',
        'it': ':riccio:',
        'fa': ':جوجه_تیغی:',
        'id': ':landak:',
        'zh': ':刺猬:',
        'ru': ':еж:'
    },
    '\U0001F681': {  # 🚁
        'en': ':helicopter:',
        'status': fully_qualified,
        'E': 1,
        'de': ':hubschrauber:',
        'es': ':helicóptero:',
        'fr': ':hélicoptère:',
        'ja': ':ヘリコプター:',
        'ko': ':헬리콥터:',
        'pt': ':helicóptero:',
        'it': ':elicottero:',
        'fa': ':هلیکوپتر:',
        'id': ':helikopter:',
        'zh': ':直升机:',
        'ru': ':вертолет:'
    },
    '\U0001F33F': {  # 🌿
        'en': ':herb:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kräuter:',
        'es': ':hierba:',
        'fr': ':feuille:',
        'ja': ':ハーブ:',
        'ko': ':풀:',
        'pt': ':erva:',
        'it': ':pianta:',
        'fa': ':گیاه_دارویی:',
        'id': ':herba:',
        'zh': ':药草:',
        'ru': ':веточка:'
    },
    '\U0001F33A': {  # 🌺
        'en': ':hibiscus:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':hibiskus:',
        'es': ':flor_de_hibisco:',
        'fr': ':hibiscus:',
        'ja': ':ハイビスカス:',
        'ko': ':무궁화:',
        'pt': ':hibisco:',
        'it': ':ibisco:',
        'fa': ':گل_ختمی:',
        'id': ':kembang_sepatu:',
        'zh': ':芙蓉:',
        'ru': ':гибискус:'
    },
    '\U0001F460': {  # 👠
        'en': ':high-heeled_shoe:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':high_heel:', ':high_heeled_shoe:'],
        'de': ':stöckelschuh:',
        'es': ':zapato_de_tacón:',
        'fr': ':chaussure_à_talon_haut:',
        'ja': ':ハイヒール:',
        'ko': ':하이힐:',
        'pt': ':sapato_de_salto_alto:',
        'it': ':scarpa_con_il_tacco:',
        'fa': ':کفش_پاشنه_بلند:',
        'id': ':sepatu_hak_tinggi:',
        'zh': ':高跟鞋:',
        'ru': ':туфли_на_каблуке:'
    },
    '\U0001F684': {  # 🚄
        'en': ':high-speed_train:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':bullettrain_side:', ':high_speed_train:'],
        'de': ':hochgeschwindigkeitszug_mit_spitzer_nase:',
        'es': ':tren_de_alta_velocidad:',
        'fr': ':tgv:',
        'ja': ':新幹線:',
        'ko': ':고속열차:',
        'pt': ':trem_de_alta_velocidade:',
        'it': ':treno_alta_velocità:',
        'fa': ':قطار_سریع_السیر:',
        'id': ':kereta_ekspres:',
        'zh': ':高速列车:',
        'ru': ':скоростной_поезд:'
    },
    '\U000026A1': {  # ⚡
        'en': ':high_voltage:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':zap:'],
        'variant': True,
        'de': ':hochspannung:',
        'es': ':alto_voltaje:',
        'fr': ':haute_tension:',
        'ja': ':高電圧:',
        'ko': ':고압_주의:',
        'pt': ':alta_tensão:',
        'it': ':alta_tensione:',
        'fa': ':ولتاژ_بالا:',
        'id': ':tegangan_tinggi:',
        'zh': ':高压:',
        'ru': ':высокое_напряжение:'
    },
    '\U0001F97E': {  # 🥾
        'en': ':hiking_boot:',
        'status': fully_qualified,
        'E': 11,
        'de': ':wanderstiefel:',
        'es': ':bota_de_senderismo:',
        'fr': ':chaussure_de_randonnée:',
        'ja': ':ハイキングシューズ:',
        'ko': ':등산화:',
        'pt': ':bota_de_trekking:',
        'it': ':stivale_da_trekking:',
        'fa': ':چکمه_راه_پیمایی:',
        'id': ':sepatu_boot_untuk_mendaki:',
        'zh': ':登山鞋:',
        'ru': ':треккинговый_ботинок:'
    },
    '\U0001F6D5': {  # 🛕
        'en': ':hindu_temple:',
        'status': fully_qualified,
        'E': 12,
        'de': ':hindutempel:',
        'es': ':templo_hindú:',
        'fr': ':temple_hindou:',
        'ja': ':ヒンドゥー教の寺院:',
        'ko': ':힌두교_사원:',
        'pt': ':templo_hindu:',
        'it': ':tempio_indù:',
        'fa': ':معبد_هندو:',
        'id': ':pura:',
        'zh': ':印度寺庙:',
        'ru': ':индуистский_храм:'
    },
    '\U0001F99B': {  # 🦛
        'en': ':hippopotamus:',
        'status': fully_qualified,
        'E': 11,
        'de': ':nilpferd:',
        'es': ':hipopótamo:',
        'fr': ':hippopotame:',
        'ja': ':カバ:',
        'ko': ':하마:',
        'pt': ':hipopótamo:',
        'it': ':ippopotamo:',
        'fa': ':اسب_آبی:',
        'id': ':kuda_nil:',
        'zh': ':河马:',
        'ru': ':гиппопотам:'
    },
    '\U0001F573\U0000FE0F': {  # 🕳️
        'en': ':hole:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':loch:',
        'es': ':agujero:',
        'fr': ':trou:',
        'ja': ':穴:',
        'ko': ':구멍:',
        'pt': ':buraco:',
        'it': ':buco:',
        'fa': ':چاله:',
        'id': ':lubang:',
        'zh': ':洞:',
        'ru': ':дыра:'
    },
    '\U0001F573': {  # 🕳
        'en': ':hole:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':loch:',
        'es': ':agujero:',
        'fr': ':trou:',
        'ja': ':穴:',
        'ko': ':구멍:',
        'pt': ':buraco:',
        'it': ':buco:',
        'fa': ':چاله:',
        'id': ':lubang:',
        'zh': ':洞:',
        'ru': ':дыра:'
    },
    '\U00002B55': {  # ⭕
        'en': ':hollow_red_circle:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':o:'],
        'variant': True,
        'de': ':hohler_roter_kreis:',
        'es': ':círculo_rojo_hueco:',
        'fr': ':cercle_rouge:',
        'ja': ':太い丸:',
        'ko': ':크고_진한_동그라미:',
        'pt': ':círculo_grande_oco:',
        'it': ':cerchio_rosso_vuoto:',
        'fa': ':دایره_قرمز_توخالی:',
        'id': ':lingkaran_besar_tebal:',
        'zh': ':红色空心圆圈:',
        'ru': ':красное_кольцо:'
    },
    '\U0001F36F': {  # 🍯
        'en': ':honey_pot:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':honigtopf:',
        'es': ':tarro_de_miel:',
        'fr': ':pot_de_miel:',
        'ja': ':はちみつ:',
        'ko': ':꿀:',
        'pt': ':pote_de_mel:',
        'it': ':barattolo_di_miele:',
        'fa': ':ظرف_عسل:',
        'id': ':toples_madu:',
        'zh': ':蜂蜜:',
        'ru': ':мед:'
    },
    '\U0001F41D': {  # 🐝
        'en': ':honeybee:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':bee:'],
        'de': ':biene:',
        'es': ':abeja:',
        'fr': ':abeille:',
        'ja': ':ミツバチ:',
        'ko': ':꿀벌:',
        'pt': ':abelha:',
        'it': ':ape:',
        'fa': ':زنبور_عسل:',
        'id': ':lebah_madu:',
        'zh': ':蜜蜂:',
        'ru': ':пчела:'
    },
    '\U0001FA9D': {  # 🪝
        'en': ':hook:',
        'status': fully_qualified,
        'E': 13,
        'de': ':haken:',
        'es': ':gancho:',
        'fr': ':crochet:',
        'ja': ':フック:',
        'ko': ':갈고리:',
        'pt': ':gancho:',
        'it': ':gancio:',
        'fa': ':قلاب:',
        'id': ':kait:',
        'zh': ':挂钩:',
        'ru': ':крюк:'
    },
    '\U0001F6A5': {  # 🚥
        'en': ':horizontal_traffic_light:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':traffic_light:'],
        'de': ':horizontale_verkehrsampel:',
        'es': ':semáforo_horizontal:',
        'fr': ':feu_tricolore_horizontal:',
        'ja': ':信号横:',
        'ko': ':가로_신호등:',
        'pt': ':semáforo_horizontal:',
        'it': ':semaforo_orizzontale:',
        'fa': ':چراغ_راهنمایی_افقی:',
        'id': ':lampu_lalu_lintas_horizontal:',
        'zh': ':横向的红绿灯:',
        'ru': ':горизонтальный_светофор:'
    },
    '\U0001F40E': {  # 🐎
        'en': ':horse:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':racehorse:'],
        'de': ':pferd:',
        'es': ':caballo:',
        'fr': ':cheval:',
        'ja': ':馬:',
        'ko': ':말:',
        'pt': ':cavalo:',
        'it': ':cavallo:',
        'fa': ':اسب:',
        'id': ':kuda:',
        'zh': ':马:',
        'ru': ':лошадь:'
    },
    '\U0001F434': {  # 🐴
        'en': ':horse_face:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':horse:'],
        'de': ':pferdegesicht:',
        'es': ':cara_de_caballo:',
        'fr': ':tête_de_cheval:',
        'ja': ':馬の顔:',
        'ko': ':말_얼굴:',
        'pt': ':rosto_de_cavalo:',
        'it': ':muso_di_cavallo:',
        'fa': ':صورت_اسب:',
        'id': ':wajah_kuda:',
        'zh': ':马头:',
        'ru': ':морда_лошади:'
    },
    '\U0001F3C7': {  # 🏇
        'en': ':horse_racing:',
        'status': fully_qualified,
        'E': 1,
        'de': ':pferderennen:',
        'es': ':carrera_de_caballos:',
        'fr': ':course_hippique:',
        'ja': ':競馬:',
        'ko': ':승마:',
        'pt': ':corrida_de_cavalos:',
        'it': ':ippica:',
        'fa': ':مسابقه_اسب_سواری:',
        'id': ':balap_kuda:',
        'zh': ':赛马:',
        'ru': ':скачки:'
    },
    '\U0001F3C7\U0001F3FF': {  # 🏇🏿
        'en': ':horse_racing_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':pferderennen_dunkle_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_oscuro:',
        'fr': ':course_hippique_peau_foncée:',
        'ja': ':競馬_濃い肌色:',
        'ko': ':승마_검은색_피부:',
        'pt': ':corrida_de_cavalos_pele_escura:',
        'it': ':ippica_carnagione_scura:',
        'fa': ':مسابقه_اسب_سواری_پوست_آبنوسی:',
        'id': ':balap_kuda_warna_kulit_gelap:',
        'zh': ':赛马_较深肤色:',
        'ru': ':скачки_очень_темный_тон_кожи:'
    },
    '\U0001F3C7\U0001F3FB': {  # 🏇🏻
        'en': ':horse_racing_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':pferderennen_helle_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_claro:',
        'fr': ':course_hippique_peau_claire:',
        'ja': ':競馬_薄い肌色:',
        'ko': ':승마_하얀_피부:',
        'pt': ':corrida_de_cavalos_pele_clara:',
        'it': ':ippica_carnagione_chiara:',
        'fa': ':مسابقه_اسب_سواری_پوست_سفید:',
        'id': ':balap_kuda_warna_kulit_cerah:',
        'zh': ':赛马_较浅肤色:',
        'ru': ':скачки_очень_светлый_тон_кожи:'
    },
    '\U0001F3C7\U0001F3FE': {  # 🏇🏾
        'en': ':horse_racing_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':pferderennen_mitteldunkle_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_oscuro_medio:',
        'fr': ':course_hippique_peau_mate:',
        'ja': ':競馬_やや濃い肌色:',
        'ko': ':승마_진한_갈색_피부:',
        'pt': ':corrida_de_cavalos_pele_morena_escura:',
        'it': ':ippica_carnagione_abbastanza_scura:',
        'fa': ':مسابقه_اسب_سواری_پوست_گندمی:',
        'id': ':balap_kuda_warna_kulit_gelap-sedang:',
        'zh': ':赛马_中等深肤色:',
        'ru': ':скачки_темный_тон_кожи:'
    },
    '\U0001F3C7\U0001F3FC': {  # 🏇🏼
        'en': ':horse_racing_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':pferderennen_mittelhelle_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_claro_medio:',
        'fr': ':course_hippique_peau_moyennement_claire:',
        'ja': ':競馬_やや薄い肌色:',
        'ko': ':승마_연한_갈색_피부:',
        'pt': ':corrida_de_cavalos_pele_morena_clara:',
        'it': ':ippica_carnagione_abbastanza_chiara:',
        'fa': ':مسابقه_اسب_سواری_پوست_روشن:',
        'id': ':balap_kuda_warna_kulit_cerah-sedang:',
        'zh': ':赛马_中等浅肤色:',
        'ru': ':скачки_светлый_тон_кожи:'
    },
    '\U0001F3C7\U0001F3FD': {  # 🏇🏽
        'en': ':horse_racing_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':pferderennen_mittlere_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_medio:',
        'fr': ':course_hippique_peau_légèrement_mate:',
        'ja': ':競馬_中間の肌色:',
        'ko': ':승마_갈색_피부:',
        'pt': ':corrida_de_cavalos_pele_morena:',
        'it': ':ippica_carnagione_olivastra:',
        'fa': ':مسابقه_اسب_سواری_پوست_طلایی:',
        'id': ':balap_kuda_warna_kulit_sedang:',
        'zh': ':赛马_中等肤色:',
        'ru': ':скачки_средний_тон_кожи:'
    },
    '\U0001F3E5': {  # 🏥
        'en': ':hospital:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':krankenhaus:',
        'es': ':hospital:',
        'fr': ':hôpital:',
        'ja': ':病院:',
        'ko': ':병원:',
        'pt': ':hospital:',
        'it': ':ospedale:',
        'fa': ':بیمارستان:',
        'id': ':rumah_sakit:',
        'zh': ':医院:',
        'ru': ':больница:'
    },
    '\U00002615': {  # ☕
        'en': ':hot_beverage:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':coffee:'],
        'variant': True,
        'de': ':heißgetränk:',
        'es': ':bebida_caliente:',
        'fr': ':boisson_chaude:',
        'ja': ':温かい飲み物:',
        'ko': ':뜨거운_음료:',
        'pt': ':café:',
        'it': ':bevanda_calda:',
        'fa': ':نوشیدنی_گرم:',
        'id': ':minuman_hangat:',
        'zh': ':热饮:',
        'ru': ':горячий_напиток:'
    },
    '\U0001F32D': {  # 🌭
        'en': ':hot_dog:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':hotdog:'],
        'de': ':hotdog:',
        'es': ':perrito_caliente:',
        'fr': ':hot_dog:',
        'ja': ':ホットドッグ:',
        'ko': ':핫도그:',
        'pt': ':cachorro-quente:',
        'it': ':hot_dog:',
        'fa': ':هات_داگ:',
        'id': ':hot_dog:',
        'zh': ':热狗:',
        'ru': ':хот_дог:'
    },
    '\U0001F975': {  # 🥵
        'en': ':hot_face:',
        'status': fully_qualified,
        'E': 11,
        'de': ':schwitzendes_gesicht:',
        'es': ':cara_con_calor:',
        'fr': ':visage_rouge_et_chaud:',
        'ja': ':暑い顔:',
        'ko': ':더운_얼굴:',
        'pt': ':rosto_fervendo_de_calor:',
        'it': ':faccina_accaldata:',
        'fa': ':خیلی_گرم:',
        'id': ':wajah_kepanasan:',
        'zh': ':脸发烧:',
        'ru': ':жар:'
    },
    '\U0001F336\U0000FE0F': {  # 🌶️
        'en': ':hot_pepper:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':peperoni:',
        'es': ':chile_picante:',
        'fr': ':piment_rouge:',
        'ja': ':とうがらし:',
        'ko': ':홍고추:',
        'pt': ':pimenta:',
        'it': ':peperoncino:',
        'fa': ':فلفل_قرمز:',
        'id': ':cabai_pedas:',
        'zh': ':红辣椒:',
        'ru': ':острыи_перец:'
    },
    '\U0001F336': {  # 🌶
        'en': ':hot_pepper:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':peperoni:',
        'es': ':chile_picante:',
        'fr': ':piment_rouge:',
        'ja': ':とうがらし:',
        'ko': ':홍고추:',
        'pt': ':pimenta:',
        'it': ':peperoncino:',
        'fa': ':فلفل_قرمز:',
        'id': ':cabai_pedas:',
        'zh': ':红辣椒:',
        'ru': ':острый_перец:'
    },
    '\U00002668\U0000FE0F': {  # ♨️
        'en': ':hot_springs:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':hotsprings:'],
        'variant': True,
        'de': ':heiße_quellen:',
        'es': ':aguas_termales:',
        'fr': ':sources_chaudes:',
        'ja': ':温泉マーク:',
        'ko': ':온천:',
        'pt': ':chamas:',
        'it': ':sorgenti_termali:',
        'fa': ':چشمه_آب_گرم:',
        'id': ':sumber_air_panas:',
        'zh': ':温泉:',
        'ru': ':горячие_источники:'
    },
    '\U00002668': {  # ♨
        'en': ':hot_springs:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':hotsprings:'],
        'variant': True,
        'de': ':heiße_quellen:',
        'es': ':aguas_termales:',
        'fr': ':sources_chaudes:',
        'ja': ':温泉マーク:',
        'ko': ':온천:',
        'pt': ':chamas:',
        'it': ':sorgenti_termali:',
        'fa': ':چشمهٔ_آب_گرم:',
        'id': ':sumber_air_panas:',
        'zh': ':温泉:',
        'ru': ':горячие_источники:'
    },
    '\U0001F3E8': {  # 🏨
        'en': ':hotel:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':hotel:',
        'es': ':hotel:',
        'fr': ':hôtel:',
        'ja': ':ホテル:',
        'ko': ':호텔:',
        'pt': ':hotel:',
        'it': ':hotel:',
        'fa': ':هتل:',
        'id': ':hotel:',
        'zh': ':酒店:',
        'ru': ':гостиница:'
    },
    '\U0000231B': {  # ⌛
        'en': ':hourglass_done:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':hourglass:'],
        'variant': True,
        'de': ':sanduhr:',
        'es': ':reloj_de_arena_sin_tiempo:',
        'fr': ':sablier:',
        'ja': ':砂時計:',
        'ko': ':시간이_다_된_모래시계:',
        'pt': ':ampulheta:',
        'it': ':clessidra:',
        'fa': ':ساعت_شنی:',
        'id': ':jam_pasir:',
        'zh': ':沙漏:',
        'ru': ':песочные_часы:'
    },
    '\U000023F3': {  # ⏳
        'en': ':hourglass_not_done:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':hourglass_flowing_sand:'],
        'variant': True,
        'de': ':laufende_sanduhr:',
        'es': ':reloj_de_arena_con_tiempo:',
        'fr': ':sablier_avec_sable_qui_coule:',
        'ja': ':砂が落ちている砂時計:',
        'ko': ':시간이_남은_모래시계:',
        'pt': ':ampulheta_contando_o_tempo:',
        'it': ':clessidra_che_scorre:',
        'fa': ':ساعت_شنی_با_شن_روان:',
        'id': ':jam_pasir_yang_bergerak:',
        'zh': ':沙正往下流的沙漏:',
        'ru': ':время_пошло:'
    },
    '\U0001F3E0': {  # 🏠
        'en': ':house:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':haus:',
        'es': ':casa:',
        'fr': ':maison:',
        'ja': ':家:',
        'ko': ':집:',
        'pt': ':casa:',
        'it': ':casa:',
        'fa': ':خانه:',
        'id': ':rumah:',
        'zh': ':房子:',
        'ru': ':дом:'
    },
    '\U0001F3E1': {  # 🏡
        'en': ':house_with_garden:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':haus_mit_garten:',
        'es': ':casa_con_jardín:',
        'fr': ':maison_avec_jardin:',
        'ja': ':庭付きの家:',
        'ko': ':정원이_있는_집:',
        'pt': ':casa_com_jardim:',
        'it': ':casa_con_giardino:',
        'fa': ':خانهٔ_حیاط_دار:',
        'id': ':rumah_dengan_taman:',
        'zh': ':别墅:',
        'ru': ':дом_с_садом:'
    },
    '\U0001F3D8\U0000FE0F': {  # 🏘️
        'en': ':houses:',
        'status': fully_qualified,
        'E': 0.7,
        'alias': [':house_buildings:'],
        'variant': True,
        'de': ':wohnhäuser:',
        'es': ':casas:',
        'fr': ':maisons:',
        'ja': ':住宅街:',
        'ko': ':주택_건물:',
        'pt': ':casas:',
        'it': ':case:',
        'fa': ':منازل:',
        'id': ':perumahan:',
        'zh': ':房屋建筑:',
        'ru': ':дома:'
    },
    '\U0001F3D8': {  # 🏘
        'en': ':houses:',
        'status': unqualified,
        'E': 0.7,
        'alias': [':house_buildings:'],
        'variant': True,
        'de': ':wohnhäuser:',
        'es': ':casas:',
        'fr': ':maisons:',
        'ja': ':住宅街:',
        'ko': ':주택_건물:',
        'pt': ':casas:',
        'it': ':case:',
        'fa': ':منازل:',
        'id': ':perumahan:',
        'zh': ':房屋建筑:',
        'ru': ':дома:'
    },
    '\U0001F4AF': {  # 💯
        'en': ':hundred_points:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':100:'],
        'de': ':100_punkte:',
        'es': ':cien_puntos:',
        'fr': ':cent_points:',
        'ja': ':100点満点:',
        'ko': ':백점:',
        'pt': ':cem_pontos:',
        'it': ':100_punti:',
        'fa': ':۱۰۰_امتیاز:',
        'id': ':nilai_seratus:',
        'zh': ':一百分:',
        'ru': ':сто_баллов:'
    },
    '\U0001F62F': {  # 😯
        'en': ':hushed_face:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':hushed:'],
        'de': ':verdutztes_gesicht:',
        'es': ':cara_estupefacta:',
        'fr': ':visage_ébahi:',
        'ja': ':ぽかーん:',
        'ko': ':숨죽인_얼굴:',
        'pt': ':rosto_surpreso:',
        'it': ':faccina_sorpresa:',
        'fa': ':مات_و_مبهوت:',
        'id': ':wajah_hening:',
        'zh': ':缄默:',
        'ru': ':неприятный_сюрприз:'
    },
    '\U0001F6D6': {  # 🛖
        'en': ':hut:',
        'status': fully_qualified,
        'E': 13,
        'de': ':hütte:',
        'es': ':cabaña:',
        'fr': ':hutte:',
        'ja': ':わらぶき小屋:',
        'ko': ':오두막:',
        'pt': ':cabana:',
        'it': ':capanna:',
        'fa': ':کلبه:',
        'id': ':pondok:',
        'zh': ':小屋:',
        'ru': ':хижина:'
    },
    '\U0001FABB': {  # 🪻
        'en': ':hyacinth:',
        'status': fully_qualified,
        'E': 15,
        'de': ':hyazinthe:',
        'es': ':campanilla:',
        'fr': ':jacinthe:',
        'ja': ':ヒヤシンス:',
        'ko': ':히아신스:',
        'pt': ':jacinto:',
        'it': ':giacinto:',
        'fa': ':سنبل:',
        'id': ':hyacinthus:',
        'zh': ':风信子:',
        'ru': ':гиацинт:'
    },
    '\U0001F9CA': {  # 🧊
        'en': ':ice:',
        'status': fully_qualified,
        'E': 12,
        'alias': [':ice_cube:'],
        'de': ':eiswürfel:',
        'es': ':cubito_de_hielo:',
        'fr': ':glaçon:',
        'ja': ':角氷:',
        'ko': ':얼음:',
        'pt': ':cubo_de_gelo:',
        'it': ':cubetto_di_ghiaccio:',
        'fa': ':قالب_یخ:',
        'id': ':es_batu:',
        'zh': ':冰块:',
        'ru': ':кубик_льда:'
    },
    '\U0001F368': {  # 🍨
        'en': ':ice_cream:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':eiscreme:',
        'es': ':helado:',
        'fr': ':glace:',
        'ja': ':アイスクリーム:',
        'ko': ':아이스크림:',
        'pt': ':sorvete:',
        'it': ':coppa_di_gelato:',
        'fa': ':بستنی:',
        'id': ':es_krim:',
        'zh': ':冰淇淋:',
        'ru': ':мороженое:'
    },
    '\U0001F3D2': {  # 🏒
        'en': ':ice_hockey:',
        'status': fully_qualified,
        'E': 1,
        'alias': [':ice_hockey_stick_and_puck:'],
        'de': ':eishockey:',
        'es': ':hockey_sobre_hielo:',
        'fr': ':hockey_sur_glace:',
        'ja': ':アイスホッケー:',
        'ko': ':아이스_하키와_퍽:',
        'pt': ':hóquei_no_gelo:',
        'it': ':hockey_su_ghiaccio:',
        'fa': ':هاکی_روی_یخ:',
        'id': ':hoki_es:',
        'zh': ':冰球:',
        'ru': ':хоккей:'
    },
    '\U000026F8\U0000FE0F': {  # ⛸️
        'en': ':ice_skate:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':schlittschuh:',
        'es': ':patín_de_hielo:',
        'fr': ':patin_à_glace:',
        'ja': ':アイススケート:',
        'ko': ':아이스_스케이트:',
        'pt': ':patins_de_gelo:',
        'it': ':pattinaggio_su_ghiaccio:',
        'fa': ':اسکی_روی_یخ:',
        'id': ':seluncur_es:',
        'zh': ':滑冰:',
        'ru': ':коньки:'
    },
    '\U000026F8': {  # ⛸
        'en': ':ice_skate:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':schlittschuh:',
        'es': ':patín_de_hielo:',
        'fr': ':patin_à_glace:',
        'ja': ':アイススケート:',
        'ko': ':아이스_스케이트:',
        'pt': ':patins_de_gelo:',
        'it': ':pattinaggio_su_ghiaccio:',
        'fa': ':اسکی_روی_یخ:',
        'id': ':seluncur_es:',
        'zh': ':滑冰:',
        'ru': ':коньки:'
    },
    '\U0001FAAA': {  # 🪪
        'en': ':identification_card:',
        'status': fully_qualified,
        'E': 14,
        'de': ':ausweis:',
        'es': ':carné_de_identidad:',
        'fr': ':carte_d’identité:',
        'ja': ':身分証:',
        'ko': ':신분증:',
        'pt': ':cartão_de_identificação:',
        'it': ':carta_d’identità:',
        'fa': ':کارت_شناسایی:',
        'id': ':kartu_identitas:',
        'zh': ':身份证:',
        'ru': ':удостоверение_личности:'
    },
    '\U0001F4E5': {  # 📥
        'en': ':inbox_tray:',
        'status': fully_qualified,
        'E': 0.6,
        'variant': True,
        'de': ':posteingang:',
        'es': ':bandeja_de_entrada:',
        'fr': ':boîte_de_réception:',
        'ja': ':受信トレイ:',
        'ko': ':받은_편지함:',
        'pt': ':bandeja_de_entrada:',
        'it': ':posta_ricevuta:',
        'fa': ':صندوق_ورودی:',
        'id': ':baki_kotak_masuk:',
        'zh': ':收件箱:',
        'ru': ':входящие:'
    },
    '\U0001F4E8': {  # 📨
        'en': ':incoming_envelope:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':eingehender_briefumschlag:',
        'es': ':sobre_entrante:',
        'fr': ':message_reçu:',
        'ja': ':メール受信中:',
        'ko': ':받은_편지:',
        'pt': ':envelope_chegando:',
        'it': ':posta_in_arrivo:',
        'fa': ':پاکت_نامهٔ_ورودی:',
        'id': ':amplop_datang:',
        'zh': ':来信:',
        'ru': ':входящее_письмо:'
    },
    '\U0001FAF5': {  # 🫵
        'en': ':index_pointing_at_the_viewer:',
        'status': fully_qualified,
        'E': 14,
        'de': ':auf_betrachter_zeigender_zeigefinger:',
        'es': ':dedo_índice_apuntándote_a_ti:',
        'fr': ':index_pointant_vers_l’utilisateur:',
        'ja': ':人を指差している手:',
        'ko': ':보는_사람을_가리키는_검지:',
        'pt': ':indicador_apontando_para_o_visualizador:',
        'it': ':indice_verso_l’osservatore:',
        'fa': ':اشاره_به_بیننده:',
        'id': ':jari_telunjuk_menunjuk_penonton:',
        'zh': ':指向观察者的食指:',
        'ru': ':указательный_палец_на_себя:'
    },
    '\U0001FAF5\U0001F3FF': {  # 🫵🏿
        'en': ':index_pointing_at_the_viewer_dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':auf_betrachter_zeigender_zeigefinger_dunkle_hautfarbe:',
        'es': ':dedo_índice_apuntándote_a_ti_tono_de_piel_oscuro:',
        'fr': ':index_pointant_vers_l’utilisateur_peau_foncée:',
        'ja': ':人を指差している手_濃い肌色:',
        'ko': ':보는_사람을_가리키는_검지_검은색_피부:',
        'pt': ':indicador_apontando_para_o_visualizador_pele_escura:',
        'it': ':indice_verso_l’osservatore_carnagione_scura:',
        'fa': ':اشاره_به_بیننده_پوست_آبنوسی:',
        'id': ':jari_telunjuk_menunjuk_penonton_warna_kulit_gelap:',
        'zh': ':指向观察者的食指_较深肤色:',
        'ru': ':указательный_палец_на_себя_очень_темный_тон_кожи:'
    },
    '\U0001FAF5\U0001F3FB': {  # 🫵🏻
        'en': ':index_pointing_at_the_viewer_light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':auf_betrachter_zeigender_zeigefinger_helle_hautfarbe:',
        'es': ':dedo_índice_apuntándote_a_ti_tono_de_piel_claro:',
        'fr': ':index_pointant_vers_l’utilisateur_peau_claire:',
        'ja': ':人を指差している手_薄い肌色:',
        'ko': ':보는_사람을_가리키는_검지_하얀_피부:',
        'pt': ':indicador_apontando_para_o_visualizador_pele_clara:',
        'it': ':indice_verso_l’osservatore_carnagione_chiara:',
        'fa': ':اشاره_به_بیننده_پوست_سفید:',
        'id': ':jari_telunjuk_menunjuk_penonton_warna_kulit_cerah:',
        'zh': ':指向观察者的食指_较浅肤色:',
        'ru': ':указательный_палец_на_себя_очень_светлый_тон_кожи:'
    },
    '\U0001FAF5\U0001F3FE': {  # 🫵🏾
        'en': ':index_pointing_at_the_viewer_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':auf_betrachter_zeigender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': ':dedo_índice_apuntándote_a_ti_tono_de_piel_oscuro_medio:',
        'fr': ':index_pointant_vers_l’utilisateur_peau_mate:',
        'ja': ':人を指差している手_やや濃い肌色:',
        'ko': ':보는_사람을_가리키는_검지_진한_갈색_피부:',
        'pt': ':indicador_apontando_para_o_visualizador_pele_morena_escura:',
        'it': ':indice_verso_l’osservatore_carnagione_abbastanza_scura:',
        'fa': ':اشاره_به_بیننده_پوست_گندمی:',
        'id': ':jari_telunjuk_menunjuk_penonton_warna_kulit_gelap-sedang:',
        'zh': ':指向观察者的食指_中等深肤色:',
        'ru': ':указательный_палец_на_себя_темный_тон_кожи:'
    },
    '\U0001FAF5\U0001F3FC': {  # 🫵🏼
        'en': ':index_pointing_at_the_viewer_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':auf_betrachter_zeigender_zeigefinger_mittelhelle_hautfarbe:',
        'es': ':dedo_índice_apuntándote_a_ti_tono_de_piel_claro_medio:',
        'fr': ':index_pointant_vers_l’utilisateur_peau_moyennement_claire:',
        'ja': ':人を指差している手_やや薄い肌色:',
        'ko': ':보는_사람을_가리키는_검지_연한_갈색_피부:',
        'pt': ':indicador_apontando_para_o_visualizador_pele_morena_clara:',
        'it': ':indice_verso_l’osservatore_carnagione_abbastanza_chiara:',
        'fa': ':اشاره_به_بیننده_پوست_روشن:',
        'id': ':jari_telunjuk_menunjuk_penonton_warna_kulit_cerah-sedang:',
        'zh': ':指向观察者的食指_中等浅肤色:',
        'ru': ':указательный_палец_на_себя_светлый_тон_кожи:'
    },
    '\U0001FAF5\U0001F3FD': {  # 🫵🏽
        'en': ':index_pointing_at_the_viewer_medium_skin_tone:',
        'status': fully_qualified,
        'E': 14,
        'de': ':auf_betrachter_zeigender_zeigefinger_mittlere_hautfarbe:',
        'es': ':dedo_índice_apuntándote_a_ti_tono_de_piel_medio:',
        'fr': ':index_pointant_vers_l’utilisateur_peau_légèrement_mate:',
        'ja': ':人を指差している手_中間の肌色:',
        'ko': ':보는_사람을_가리키는_검지_갈색_피부:',
        'pt': ':indicador_apontando_para_o_visualizador_pele_morena:',
        'it': ':indice_verso_l’osservatore_carnagione_olivastra:',
        'fa': ':اشاره_به_بیننده_پوست_طلایی:',
        'id': ':jari_telunjuk_menunjuk_penonton_warna_kulit_sedang:',
        'zh': ':指向观察者的食指_中等肤色:',
        'ru': ':указательный_палец_на_себя_средний_тон_кожи:'
    },
    '\U0000261D\U0000FE0F': {  # ☝️
        'en': ':index_pointing_up:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':point_up:'],
        'variant': True,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne:',
        'es': ':dedo_índice_hacia_arriba:',
        'fr': ':index_pointant_vers_le_haut:',
        'ja': ':上指差し:',
        'ko': ':위쪽을_가리키는_손:',
        'pt': ':indicador_apontando_para_cima:',
        'it': ':indice_verso_l’alto:',
        'fa': ':اشاره_به_بالا:',
        'id': ':menunjuk_ke_atas:',
        'zh': ':食指向上指:',
        'ru': ':указательныи_палец_вверх:'
    },
    '\U0000261D': {  # ☝
        'en': ':index_pointing_up:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':point_up:'],
        'variant': True,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne:',
        'es': ':dedo_índice_hacia_arriba:',
        'fr': ':index_pointant_vers_le_haut:',
        'ja': ':上指差し:',
        'ko': ':위쪽을_가리키는_손:',
        'pt': ':indicador_apontando_para_cima:',
        'it': ':indice_verso_l’alto:',
        'fa': ':اشاره_به_بالا:',
        'id': ':menunjuk_ke_atas:',
        'zh': ':食指向上指:',
        'ru': ':указательный_палец_вверх:'
    },
    '\U0000261D\U0001F3FF': {  # ☝🏿
        'en': ':index_pointing_up_dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_dunkle_hautfarbe:',
        'es': ':dedo_índice_hacia_arriba_tono_de_piel_oscuro:',
        'fr': ':index_pointant_vers_le_haut_peau_foncée:',
        'ja': ':上指差し_濃い肌色:',
        'ko': ':위쪽을_가리키는_손_검은색_피부:',
        'pt': ':indicador_apontando_para_cima_pele_escura:',
        'it': ':indice_verso_l’alto_carnagione_scura:',
        'fa': ':اشاره_به_بالا_پوست_آبنوسی:',
        'id': ':menunjuk_ke_atas_warna_kulit_gelap:',
        'zh': ':食指向上指_较深肤色:',
        'ru': ':указательный_палец_вверх_очень_темный_тон_кожи:'
    },
    '\U0000261D\U0001F3FB': {  # ☝🏻
        'en': ':index_pointing_up_light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_helle_hautfarbe:',
        'es': ':dedo_índice_hacia_arriba_tono_de_piel_claro:',
        'fr': ':index_pointant_vers_le_haut_peau_claire:',
        'ja': ':上指差し_薄い肌色:',
        'ko': ':위쪽을_가리키는_손_하얀_피부:',
        'pt': ':indicador_apontando_para_cima_pele_clara:',
        'it': ':indice_verso_l’alto_carnagione_chiara:',
        'fa': ':اشاره_به_بالا_پوست_سفید:',
        'id': ':menunjuk_ke_atas_warna_kulit_cerah:',
        'zh': ':食指向上指_较浅肤色:',
        'ru': ':указательный_палец_вверх_очень_светлый_тон_кожи:'
    },
    '\U0000261D\U0001F3FE': {  # ☝🏾
        'en': ':index_pointing_up_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_mitteldunkle_hautfarbe:',
        'es': ':dedo_índice_hacia_arriba_tono_de_piel_oscuro_medio:',
        'fr': ':index_pointant_vers_le_haut_peau_mate:',
        'ja': ':上指差し_やや濃い肌色:',
        'ko': ':위쪽을_가리키는_손_진한_갈색_피부:',
        'pt': ':indicador_apontando_para_cima_pele_morena_escura:',
        'it': ':indice_verso_l’alto_carnagione_abbastanza_scura:',
        'fa': ':اشاره_به_بالا_پوست_گندمی:',
        'id': ':menunjuk_ke_atas_warna_kulit_gelap-sedang:',
        'zh': ':食指向上指_中等深肤色:',
        'ru': ':указательный_палец_вверх_темный_тон_кожи:'
    },
    '\U0000261D\U0001F3FC': {  # ☝🏼
        'en': ':index_pointing_up_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_mittelhelle_hautfarbe:',
        'es': ':dedo_índice_hacia_arriba_tono_de_piel_claro_medio:',
        'fr': ':index_pointant_vers_le_haut_peau_moyennement_claire:',
        'ja': ':上指差し_やや薄い肌色:',
        'ko': ':위쪽을_가리키는_손_연한_갈색_피부:',
        'pt': ':indicador_apontando_para_cima_pele_morena_clara:',
        'it': ':indice_verso_l’alto_carnagione_abbastanza_chiara:',
        'fa': ':اشاره_به_بالا_پوست_روشن:',
        'id': ':menunjuk_ke_atas_warna_kulit_cerah-sedang:',
        'zh': ':食指向上指_中等浅肤色:',
        'ru': ':указательный_палец_вверх_светлый_тон_кожи:'
    },
    '\U0000261D\U0001F3FD': {  # ☝🏽
        'en': ':index_pointing_up_medium_skin_tone:',
        'status': fully_qualified,
        'E': 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_mittlere_hautfarbe:',
        'es': ':dedo_índice_hacia_arriba_tono_de_piel_medio:',
        'fr': ':index_pointant_vers_le_haut_peau_légèrement_mate:',
        'ja': ':上指差し_中間の肌色:',
        'ko': ':위쪽을_가리키는_손_갈색_피부:',
        'pt': ':indicador_apontando_para_cima_pele_morena:',
        'it': ':indice_verso_l’alto_carnagione_olivastra:',
        'fa': ':اشاره_به_بالا_پوست_طلایی:',
        'id': ':menunjuk_ke_atas_warna_kulit_sedang:',
        'zh': ':食指向上指_中等肤色:',
        'ru': ':указательный_палец_вверх_средний_тон_кожи:'
    },
    '\U0000267E\U0000FE0F': {  # ♾️
        'en': ':infinity:',
        'status': fully_qualified,
        'E': 11,
        'variant': True,
        'de': ':unendlichkeit:',
        'es': ':infinito:',
        'fr': ':infini:',
        'ja': ':無限大:',
        'ko': ':무한:',
        'pt': ':infinito:',
        'it': ':simbolo_dell’infinito:',
        'fa': ':بی_نهایت:',
        'id': ':tak_terhingga:',
        'zh': ':无穷大:',
        'ru': ':бесконечность:'
    },
    '\U0000267E': {  # ♾
        'en': ':infinity:',
        'status': unqualified,
        'E': 11,
        'variant': True,
        'de': ':unendlichkeit:',
        'es': ':infinito:',
        'fr': ':infini:',
        'ja': ':無限大:',
        'ko': ':무한:',
        'pt': ':infinito:',
        'it': ':simbolo_dell’infinito:',
        'fa': ':بی_نهایت:',
        'id': ':tak_terhingga:',
        'zh': ':无穷大:',
        'ru': ':бесконечность:'
    },
    '\U00002139\U0000FE0F': {  # ℹ️
        'en': ':information:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':information_source:'],
        'variant': True,
        'de': ':buchstabe_i_in_blauem_quadrat:',
        'es': ':información:',
        'fr': ':source_d’informations:',
        'ja': ':iマーク:',
        'ko': ':정보_출처:',
        'pt': ':informações:',
        'it': ':punto_informazioni:',
        'fa': ':اطلاعات:',
        'id': ':informasi:',
        'zh': ':信息:',
        'ru': ':значок_информации:'
    },
    '\U00002139': {  # ℹ
        'en': ':information:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':information_source:'],
        'variant': True,
        'de': ':buchstabe_i_in_blauem_quadrat:',
        'es': ':información:',
        'fr': ':source_d’informations:',
        'ja': ':iマーク:',
        'ko': ':정보_출처:',
        'pt': ':informações:',
        'it': ':punto_informazioni:',
        'fa': ':اطلاعات:',
        'id': ':informasi:',
        'zh': ':信息:',
        'ru': ':значок_информации:'
    },
    '\U0001F524': {  # 🔤
        'en': ':input_latin_letters:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':abc:'],
        'de': ':eingabesymbol_lateinische_buchstaben:',
        'es': ':alfabeto_latino:',
        'fr': ':alphabet_latin:',
        'ja': ':英字の入力:',
        'ko': ':라틴어_알파벳:',
        'pt': ':letras_latinas:',
        'it': ':pulsante_con_lettere_latine:',
        'fa': ':ورودی_حروف_لاتین:',
        'id': ':masukan_huruf_latin:',
        'zh': ':输入拉丁字母:',
        'ru': ':ввод_латиницей:'
    },
    '\U0001F521': {  # 🔡
        'en': ':input_latin_lowercase:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':abcd:'],
        'de': ':eingabesymbol_lateinische_kleinbuchstaben:',
        'es': ':letras_latinas_minúsculas:',
        'fr': ':minuscules:',
        'ja': ':英小文字の入力:',
        'ko': ':소문자:',
        'pt': ':letras_latinas_minúsculas:',
        'it': ':pulsante_con_lettere_latine_minuscole:',
        'fa': ':ورودی_حروف_کوچک:',
        'id': ':masukan_huruf_kecil_latin:',
        'zh': ':输入小写拉丁字母:',
        'ru': ':ввод_строчными:'
    },
    '\U0001F520': {  # 🔠
        'en': ':input_latin_uppercase:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':capital_abcd:'],
        'de': ':eingabesymbol_lateinische_großbuchstaben:',
        'es': ':letras_latinas_mayúsculas:',
        'fr': ':majuscules:',
        'ja': ':英大文字の入力:',
        'ko': ':대문자:',
        'pt': ':letras_latinas_maiúsculas:',
        'it': ':pulsante_con_lettere_latine_maiuscole:',
        'fa': ':ورودی_حروف_بزرگ:',
        'id': ':masukan_huruf_kapital_latin:',
        'zh': ':输入大写拉丁字母:',
        'ru': ':ввод_прописными:'
    },
    '\U0001F522': {  # 🔢
        'en': ':input_numbers:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':1234:'],
        'de': ':eingabesymbol_zahlen:',
        'es': ':números:',
        'fr': ':saisie_de_chiffres:',
        'ja': ':数字の入力:',
        'ko': ':숫자:',
        'pt': ':números:',
        'it': ':pulsante_con_numeri:',
        'fa': ':ورودی_عددی:',
        'id': ':masukan_angka:',
        'zh': ':输入数字:',
        'ru': ':ввод_цифр:'
    },
    '\U0001F523': {  # 🔣
        'en': ':input_symbols:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':symbols:'],
        'de': ':eingabesymbol_sonderzeichen:',
        'es': ':símbolos:',
        'fr': ':saisie_de_symboles:',
        'ja': ':記号の入力:',
        'ko': ':기호:',
        'pt': ':símbolos:',
        'it': ':pulsante_con_simboli:',
        'fa': ':ورودی_نماد:',
        'id': ':masukan_simbol:',
        'zh': ':输入符号:',
        'ru': ':ввод_символов:'
    },
    '\U0001F383': {  # 🎃
        'en': ':jack-o-lantern:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':jack_o_lantern:'],
        'de': ':halloweenkürbis:',
        'es': ':calabaza_de_halloween:',
        'fr': ':citrouille:',
        'ja': ':ハロウィンかぼちゃ:',
        'ko': ':할로윈:',
        'pt': ':abóbora_de_halloween:',
        'it': ':zucca_di_halloween:',
        'fa': ':فانوس_کدو_تنبل:',
        'id': ':jack-o-lantern:',
        'zh': ':南瓜灯:',
        'ru': ':тыква_фонарь:'
    },
    '\U0001FAD9': {  # 🫙
        'en': ':jar:',
        'status': fully_qualified,
        'E': 14,
        'de': ':einmachglas:',
        'es': ':tarro:',
        'fr': ':bocal:',
        'ja': ':瓶:',
        'ko': ':작은_병:',
        'pt': ':jarro:',
        'it': ':barattolo:',
        'fa': ':شیشه_دهان_گشاد:',
        'id': ':toples:',
        'zh': ':罐:',
        'ru': ':банка:'
    },
    '\U0001F456': {  # 👖
        'en': ':jeans:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':jeans:',
        'es': ':vaqueros:',
        'fr': ':jean:',
        'ja': ':ジーンズ:',
        'ko': ':바지:',
        'pt': ':jeans:',
        'it': ':jeans:',
        'fa': ':شلوار_جین:',
        'id': ':celana_jeans:',
        'zh': ':牛仔裤:',
        'ru': ':джинсы:'
    },
    '\U0001FABC': {  # 🪼
        'en': ':jellyfish:',
        'status': fully_qualified,
        'E': 15,
        'de': ':qualle:',
        'es': ':medusa:',
        'fr': ':méduse:',
        'ja': ':クラゲ:',
        'ko': ':해파리:',
        'pt': ':água-viva:',
        'it': ':medusa:',
        'fa': ':عروس_دریایی:',
        'id': ':ubur-ubur:',
        'zh': ':水母:',
        'ru': ':медуза:'
    },
    '\U0001F0CF': {  # 🃏
        'en': ':joker:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':black_joker:'],
        'de': ':jokerkarte:',
        'es': ':comodín:',
        'fr': ':carte_joker:',
        'ja': ':ジョーカー:',
        'ko': ':조커:',
        'pt': ':curinga:',
        'it': ':jolly:',
        'fa': ':جوکر:',
        'id': ':joker:',
        'zh': ':大小王:',
        'ru': ':джокер:'
    },
    '\U0001F579\U0000FE0F': {  # 🕹️
        'en': ':joystick:',
        'status': fully_qualified,
        'E': 0.7,
        'variant': True,
        'de': ':joystick:',
        'es': ':joystick:',
        'fr': ':manette_de_jeu:',
        'ja': ':ジョイスティック:',
        'ko': ':조이스틱:',
        'pt': ':joystick:',
        'it': ':joystick:',
        'fa': ':دسته_بازی:',
        'id': ':joystick:',
        'zh': ':游戏操控杆:',
        'ru': ':джоистик:'
    },
    '\U0001F579': {  # 🕹
        'en': ':joystick:',
        'status': unqualified,
        'E': 0.7,
        'variant': True,
        'de': ':joystick:',
        'es': ':joystick:',
        'fr': ':manette_de_jeu:',
        'ja': ':ジョイスティック:',
        'ko': ':조이스틱:',
        'pt': ':joystick:',
        'it': ':joystick:',
        'fa': ':دسته_بازی:',
        'id': ':joystick:',
        'zh': ':游戏操控杆:',
        'ru': ':джойстик:'
    },
    '\U0001F9D1\U0000200D\U00002696\U0000FE0F': {  # 🧑‍⚖️
        'en': ':judge:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':richter(in):',
        'es': ':fiscal:',
        'fr': ':juge:',
        'ja': ':裁判官:',
        'ko': ':판사:',
        'pt': ':juiz_no_tribunal:',
        'it': ':giudice:',
        'fa': ':قاضی:',
        'id': ':hakim:',
        'zh': ':法官:',
        'ru': ':судья:'
    },
    '\U0001F9D1\U0000200D\U00002696': {  # 🧑‍⚖
        'en': ':judge:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':richter(in):',
        'es': ':fiscal:',
        'fr': ':juge:',
        'ja': ':裁判官:',
        'ko': ':판사:',
        'pt': ':juiz_no_tribunal:',
        'it': ':giudice:',
        'fa': ':قاضی:',
        'id': ':hakim:',
        'zh': ':法官:',
        'ru': ':судья:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002696\U0000FE0F': {  # 🧑🏿‍⚖️
        'en': ':judge_dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':richter(in)_dunkle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_oscuro:',
        'fr': ':juge_peau_foncée:',
        'ja': ':裁判官_濃い肌色:',
        'ko': ':판사_검은색_피부:',
        'pt': ':juiz_no_tribunal_pele_escura:',
        'it': ':giudice_carnagione_scura:',
        'fa': ':قاضی_پوست_آبنوسی:',
        'id': ':hakim_warna_kulit_gelap:',
        'zh': ':法官_较深肤色:',
        'ru': ':судья_очень_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002696': {  # 🧑🏿‍⚖
        'en': ':judge_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':richter(in)_dunkle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_oscuro:',
        'fr': ':juge_peau_foncée:',
        'ja': ':裁判官_濃い肌色:',
        'ko': ':판사_검은색_피부:',
        'pt': ':juiz_no_tribunal_pele_escura:',
        'it': ':giudice_carnagione_scura:',
        'fa': ':قاضی_پوست_آبنوسی:',
        'id': ':hakim_warna_kulit_gelap:',
        'zh': ':法官_较深肤色:',
        'ru': ':судья_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002696\U0000FE0F': {  # 🧑🏻‍⚖️
        'en': ':judge_light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':richter(in)_helle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_claro:',
        'fr': ':juge_peau_claire:',
        'ja': ':裁判官_薄い肌色:',
        'ko': ':판사_하얀_피부:',
        'pt': ':juiz_no_tribunal_pele_clara:',
        'it': ':giudice_carnagione_chiara:',
        'fa': ':قاضی_پوست_سفید:',
        'id': ':hakim_warna_kulit_cerah:',
        'zh': ':法官_较浅肤色:',
        'ru': ':судья_очень_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002696': {  # 🧑🏻‍⚖
        'en': ':judge_light_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':richter(in)_helle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_claro:',
        'fr': ':juge_peau_claire:',
        'ja': ':裁判官_薄い肌色:',
        'ko': ':판사_하얀_피부:',
        'pt': ':juiz_no_tribunal_pele_clara:',
        'it': ':giudice_carnagione_chiara:',
        'fa': ':قاضی_پوست_سفید:',
        'id': ':hakim_warna_kulit_cerah:',
        'zh': ':法官_较浅肤色:',
        'ru': ':судья_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002696\U0000FE0F': {  # 🧑🏾‍⚖️
        'en': ':judge_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':richter(in)_mitteldunkle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_oscuro_medio:',
        'fr': ':juge_peau_mate:',
        'ja': ':裁判官_やや濃い肌色:',
        'ko': ':판사_진한_갈색_피부:',
        'pt': ':juiz_no_tribunal_pele_morena_escura:',
        'it': ':giudice_carnagione_abbastanza_scura:',
        'fa': ':قاضی_پوست_گندمی:',
        'id': ':hakim_warna_kulit_gelap-sedang:',
        'zh': ':法官_中等深肤色:',
        'ru': ':судья_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002696': {  # 🧑🏾‍⚖
        'en': ':judge_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':richter(in)_mitteldunkle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_oscuro_medio:',
        'fr': ':juge_peau_mate:',
        'ja': ':裁判官_やや濃い肌色:',
        'ko': ':판사_진한_갈색_피부:',
        'pt': ':juiz_no_tribunal_pele_morena_escura:',
        'it': ':giudice_carnagione_abbastanza_scura:',
        'fa': ':قاضی_پوست_گندمی:',
        'id': ':hakim_warna_kulit_gelap-sedang:',
        'zh': ':法官_中等深肤色:',
        'ru': ':судья_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002696\U0000FE0F': {  # 🧑🏼‍⚖️
        'en': ':judge_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':richter(in)_mittelhelle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_claro_medio:',
        'fr': ':juge_peau_moyennement_claire:',
        'ja': ':裁判官_やや薄い肌色:',
        'ko': ':판사_연한_갈색_피부:',
        'pt': ':juiz_no_tribunal_pele_morena_clara:',
        'it': ':giudice_carnagione_abbastanza_chiara:',
        'fa': ':قاضی_پوست_روشن:',
        'id': ':hakim_warna_kulit_cerah-sedang:',
        'zh': ':法官_中等浅肤色:',
        'ru': ':судья_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FC\U0000200D\U00002696': {  # 🧑🏼‍⚖
        'en': ':judge_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':richter(in)_mittelhelle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_claro_medio:',
        'fr': ':juge_peau_moyennement_claire:',
        'ja': ':裁判官_やや薄い肌色:',
        'ko': ':판사_연한_갈색_피부:',
        'pt': ':juiz_no_tribunal_pele_morena_clara:',
        'it': ':giudice_carnagione_abbastanza_chiara:',
        'fa': ':قاضی_پوست_روشن:',
        'id': ':hakim_warna_kulit_cerah-sedang:',
        'zh': ':法官_中等浅肤色:',
        'ru': ':судья_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002696\U0000FE0F': {  # 🧑🏽‍⚖️
        'en': ':judge_medium_skin_tone:',
        'status': fully_qualified,
        'E': 12.1,
        'de': ':richter(in)_mittlere_hautfarbe:',
        'es': ':fiscal_tono_de_piel_medio:',
        'fr': ':juge_peau_légèrement_mate:',
        'ja': ':裁判官_中間の肌色:',
        'ko': ':판사_갈색_피부:',
        'pt': ':juiz_no_tribunal_pele_morena:',
        'it': ':giudice_carnagione_olivastra:',
        'fa': ':قاضی_پوست_طلایی:',
        'id': ':hakim_warna_kulit_sedang:',
        'zh': ':法官_中等肤色:',
        'ru': ':судья_среднии_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FD\U0000200D\U00002696': {  # 🧑🏽‍⚖
        'en': ':judge_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 12.1,
        'de': ':richter(in)_mittlere_hautfarbe:',
        'es': ':fiscal_tono_de_piel_medio:',
        'fr': ':juge_peau_légèrement_mate:',
        'ja': ':裁判官_中間の肌色:',
        'ko': ':판사_갈색_피부:',
        'pt': ':juiz_no_tribunal_pele_morena:',
        'it': ':giudice_carnagione_olivastra:',
        'fa': ':قاضی_پوست_طلایی:',
        'id': ':hakim_warna_kulit_sedang:',
        'zh': ':法官_中等肤色:',
        'ru': ':судья_средний_тон_кожи:'
    },
    '\U0001F54B': {  # 🕋
        'en': ':kaaba:',
        'status': fully_qualified,
        'E': 1,
        'de': ':kaaba:',
        'es': ':kaaba:',
        'fr': ':kaaba:',
        'ja': ':カーバ:',
        'ko': ':카바:',
        'pt': ':caaba_islã:',
        'it': ':kaaba:',
        'fa': ':کعبه:',
        'id': ':ka’bah:',
        'zh': ':克尔白:',
        'ru': ':Кааба:'
    },
    '\U0001F998': {  # 🦘
        'en': ':kangaroo:',
        'status': fully_qualified,
        'E': 11,
        'de': ':känguru:',
        'es': ':canguro:',
        'fr': ':kangourou:',
        'ja': ':カンガルー:',
        'ko': ':캥거루:',
        'pt': ':canguru:',
        'it': ':canguro:',
        'fa': ':کانگورو:',
        'id': ':kanguru:',
        'zh': ':袋鼠:',
        'ru': ':кенгуру:'
    },
    '\U0001F511': {  # 🔑
        'en': ':key:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':schlüssel:',
        'es': ':llave:',
        'fr': ':clé:',
        'ja': ':鍵:',
        'ko': ':열쇠:',
        'pt': ':chave:',
        'it': ':chiave:',
        'fa': ':کلید:',
        'id': ':kunci:',
        'zh': ':钥匙:',
        'ru': ':ключ:'
    },
    '\U00002328\U0000FE0F': {  # ⌨️
        'en': ':keyboard:',
        'status': fully_qualified,
        'E': 1,
        'variant': True,
        'de': ':tastatur:',
        'es': ':teclado:',
        'fr': ':clavier:',
        'ja': ':キーボード:',
        'ko': ':키보드:',
        'pt': ':teclado:',
        'it': ':tastiera:',
        'fa': ':صفحه_کلید:',
        'id': ':keyboard:',
        'zh': ':键盘:',
        'ru': ':клавиатура:'
    },
    '\U00002328': {  # ⌨
        'en': ':keyboard:',
        'status': unqualified,
        'E': 1,
        'variant': True,
        'de': ':tastatur:',
        'es': ':teclado:',
        'fr': ':clavier:',
        'ja': ':キーボード:',
        'ko': ':키보드:',
        'pt': ':teclado:',
        'it': ':tastiera:',
        'fa': ':صفحه_کلید:',
        'id': ':keyboard:',
        'zh': ':键盘:',
        'ru': ':клавиатура:'
    },
    '\U00000023\U0000FE0F\U000020E3': {  # #️⃣
        'en': ':keycap_#:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':hash:'],
        'de': ':taste_#:',
        'es': ':teclas_#:',
        'fr': ':touches_#:',
        'ja': ':囲み数字_#:',
        'ko': ':키_캡_#:',
        'pt': ':tecla_#:',
        'it': ':tasto_#:',
        'fa': ':جلد_کلید_#:',
        'id': ':keycap_#:',
        'zh': ':按键_#:',
        'ru': ':клавиши_#:'
    },
    '\U00000023\U000020E3': {  # #⃣
        'en': ':keycap_#:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':hash:'],
        'de': ':taste_#:',
        'es': ':teclas_#:',
        'fr': ':touches_#:',
        'ja': ':囲み数字_#:',
        'ko': ':키_캡_#:',
        'pt': ':tecla_#:',
        'it': ':tasto_#:',
        'fa': ':جلد_کلید_#:',
        'id': ':keycap_#:',
        'zh': ':按键_#:',
        'ru': ':клавиши_#:'
    },
    '\U0000002A\U0000FE0F\U000020E3': {  # *️⃣
        'en': ':keycap_*:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':asterisk:'],
        'de': ':taste_*:',
        'es': ':teclas_*:',
        'fr': ':touches_*:',
        'ja': ':囲み数字_*:',
        'ko': ':키_캡_*:',
        'pt': ':tecla_*:',
        'it': ':tasto_*:',
        'fa': ':جلد_کلید_*:',
        'id': ':keycap_*:',
        'zh': ':按键_*:',
        'ru': ':клавиши_*:'
    },
    '\U0000002A\U000020E3': {  # *⃣
        'en': ':keycap_*:',
        'status': unqualified,
        'E': 2,
        'alias': [':asterisk:'],
        'de': ':taste_*:',
        'es': ':teclas_*:',
        'fr': ':touches_*:',
        'ja': ':囲み数字_*:',
        'ko': ':키_캡_*:',
        'pt': ':tecla_*:',
        'it': ':tasto_*:',
        'fa': ':جلد_کلید_*:',
        'id': ':keycap_*:',
        'zh': ':按键_*:',
        'ru': ':клавиши_*:'
    },
    '\U00000030\U0000FE0F\U000020E3': {  # 0️⃣
        'en': ':keycap_0:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':zero:'],
        'de': ':taste_0:',
        'es': ':teclas_0:',
        'fr': ':touches_0:',
        'ja': ':囲み数字_0:',
        'ko': ':키_캡_0:',
        'pt': ':tecla_0:',
        'it': ':tasto_0:',
        'fa': ':جلد_کلید_0:',
        'id': ':keycap_0:',
        'zh': ':按键_0:',
        'ru': ':клавиши_0:'
    },
    '\U00000030\U000020E3': {  # 0⃣
        'en': ':keycap_0:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':zero:'],
        'de': ':taste_0:',
        'es': ':teclas_0:',
        'fr': ':touches_0:',
        'ja': ':囲み数字_0:',
        'ko': ':키_캡_0:',
        'pt': ':tecla_0:',
        'it': ':tasto_0:',
        'fa': ':جلد_کلید_0:',
        'id': ':keycap_0:',
        'zh': ':按键_0:',
        'ru': ':клавиши_0:'
    },
    '\U00000031\U0000FE0F\U000020E3': {  # 1️⃣
        'en': ':keycap_1:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':one:'],
        'de': ':taste_1:',
        'es': ':teclas_1:',
        'fr': ':touches_1:',
        'ja': ':囲み数字_1:',
        'ko': ':키_캡_1:',
        'pt': ':tecla_1:',
        'it': ':tasto_1:',
        'fa': ':جلد_کلید_1:',
        'id': ':keycap_1:',
        'zh': ':按键_1:',
        'ru': ':клавиши_1:'
    },
    '\U00000031\U000020E3': {  # 1⃣
        'en': ':keycap_1:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':one:'],
        'de': ':taste_1:',
        'es': ':teclas_1:',
        'fr': ':touches_1:',
        'ja': ':囲み数字_1:',
        'ko': ':키_캡_1:',
        'pt': ':tecla_1:',
        'it': ':tasto_1:',
        'fa': ':جلد_کلید_1:',
        'id': ':keycap_1:',
        'zh': ':按键_1:',
        'ru': ':клавиши_1:'
    },
    '\U0001F51F': {  # 🔟
        'en': ':keycap_10:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':ten:', ':keycap_ten:'],
        'de': ':taste_10:',
        'es': ':teclas_10:',
        'fr': ':touches_10:',
        'ja': ':囲み数字_10:',
        'ko': ':키_캡_10:',
        'pt': ':tecla_10:',
        'it': ':tasto_10:',
        'fa': ':جلد_کلید_10:',
        'id': ':keycap_10:',
        'zh': ':按键_10:',
        'ru': ':клавиши_10:'
    },
    '\U00000032\U0000FE0F\U000020E3': {  # 2️⃣
        'en': ':keycap_2:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':two:'],
        'de': ':taste_2:',
        'es': ':teclas_2:',
        'fr': ':touches_2:',
        'ja': ':囲み数字_2:',
        'ko': ':키_캡_2:',
        'pt': ':tecla_2:',
        'it': ':tasto_2:',
        'fa': ':جلد_کلید_2:',
        'id': ':keycap_2:',
        'zh': ':按键_2:',
        'ru': ':клавиши_2:'
    },
    '\U00000032\U000020E3': {  # 2⃣
        'en': ':keycap_2:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':two:'],
        'de': ':taste_2:',
        'es': ':teclas_2:',
        'fr': ':touches_2:',
        'ja': ':囲み数字_2:',
        'ko': ':키_캡_2:',
        'pt': ':tecla_2:',
        'it': ':tasto_2:',
        'fa': ':جلد_کلید_2:',
        'id': ':keycap_2:',
        'zh': ':按键_2:',
        'ru': ':клавиши_2:'
    },
    '\U00000033\U0000FE0F\U000020E3': {  # 3️⃣
        'en': ':keycap_3:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':three:'],
        'de': ':taste_3:',
        'es': ':teclas_3:',
        'fr': ':touches_3:',
        'ja': ':囲み数字_3:',
        'ko': ':키_캡_3:',
        'pt': ':tecla_3:',
        'it': ':tasto_3:',
        'fa': ':جلد_کلید_3:',
        'id': ':keycap_3:',
        'zh': ':按键_3:',
        'ru': ':клавиши_3:'
    },
    '\U00000033\U000020E3': {  # 3⃣
        'en': ':keycap_3:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':three:'],
        'de': ':taste_3:',
        'es': ':teclas_3:',
        'fr': ':touches_3:',
        'ja': ':囲み数字_3:',
        'ko': ':키_캡_3:',
        'pt': ':tecla_3:',
        'it': ':tasto_3:',
        'fa': ':جلد_کلید_3:',
        'id': ':keycap_3:',
        'zh': ':按键_3:',
        'ru': ':клавиши_3:'
    },
    '\U00000034\U0000FE0F\U000020E3': {  # 4️⃣
        'en': ':keycap_4:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':four:'],
        'de': ':taste_4:',
        'es': ':teclas_4:',
        'fr': ':touches_4:',
        'ja': ':囲み数字_4:',
        'ko': ':키_캡_4:',
        'pt': ':tecla_4:',
        'it': ':tasto_4:',
        'fa': ':جلد_کلید_4:',
        'id': ':keycap_4:',
        'zh': ':按键_4:',
        'ru': ':клавиши_4:'
    },
    '\U00000034\U000020E3': {  # 4⃣
        'en': ':keycap_4:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':four:'],
        'de': ':taste_4:',
        'es': ':teclas_4:',
        'fr': ':touches_4:',
        'ja': ':囲み数字_4:',
        'ko': ':키_캡_4:',
        'pt': ':tecla_4:',
        'it': ':tasto_4:',
        'fa': ':جلد_کلید_4:',
        'id': ':keycap_4:',
        'zh': ':按键_4:',
        'ru': ':клавиши_4:'
    },
    '\U00000035\U0000FE0F\U000020E3': {  # 5️⃣
        'en': ':keycap_5:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':five:'],
        'de': ':taste_5:',
        'es': ':teclas_5:',
        'fr': ':touches_5:',
        'ja': ':囲み数字_5:',
        'ko': ':키_캡_5:',
        'pt': ':tecla_5:',
        'it': ':tasto_5:',
        'fa': ':جلد_کلید_5:',
        'id': ':keycap_5:',
        'zh': ':按键_5:',
        'ru': ':клавиши_5:'
    },
    '\U00000035\U000020E3': {  # 5⃣
        'en': ':keycap_5:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':five:'],
        'de': ':taste_5:',
        'es': ':teclas_5:',
        'fr': ':touches_5:',
        'ja': ':囲み数字_5:',
        'ko': ':키_캡_5:',
        'pt': ':tecla_5:',
        'it': ':tasto_5:',
        'fa': ':جلد_کلید_5:',
        'id': ':keycap_5:',
        'zh': ':按键_5:',
        'ru': ':клавиши_5:'
    },
    '\U00000036\U0000FE0F\U000020E3': {  # 6️⃣
        'en': ':keycap_6:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':six:'],
        'de': ':taste_6:',
        'es': ':teclas_6:',
        'fr': ':touches_6:',
        'ja': ':囲み数字_6:',
        'ko': ':키_캡_6:',
        'pt': ':tecla_6:',
        'it': ':tasto_6:',
        'fa': ':جلد_کلید_6:',
        'id': ':keycap_6:',
        'zh': ':按键_6:',
        'ru': ':клавиши_6:'
    },
    '\U00000036\U000020E3': {  # 6⃣
        'en': ':keycap_6:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':six:'],
        'de': ':taste_6:',
        'es': ':teclas_6:',
        'fr': ':touches_6:',
        'ja': ':囲み数字_6:',
        'ko': ':키_캡_6:',
        'pt': ':tecla_6:',
        'it': ':tasto_6:',
        'fa': ':جلد_کلید_6:',
        'id': ':keycap_6:',
        'zh': ':按键_6:',
        'ru': ':клавиши_6:'
    },
    '\U00000037\U0000FE0F\U000020E3': {  # 7️⃣
        'en': ':keycap_7:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':seven:'],
        'de': ':taste_7:',
        'es': ':teclas_7:',
        'fr': ':touches_7:',
        'ja': ':囲み数字_7:',
        'ko': ':키_캡_7:',
        'pt': ':tecla_7:',
        'it': ':tasto_7:',
        'fa': ':جلد_کلید_7:',
        'id': ':keycap_7:',
        'zh': ':按键_7:',
        'ru': ':клавиши_7:'
    },
    '\U00000037\U000020E3': {  # 7⃣
        'en': ':keycap_7:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':seven:'],
        'de': ':taste_7:',
        'es': ':teclas_7:',
        'fr': ':touches_7:',
        'ja': ':囲み数字_7:',
        'ko': ':키_캡_7:',
        'pt': ':tecla_7:',
        'it': ':tasto_7:',
        'fa': ':جلد_کلید_7:',
        'id': ':keycap_7:',
        'zh': ':按键_7:',
        'ru': ':клавиши_7:'
    },
    '\U00000038\U0000FE0F\U000020E3': {  # 8️⃣
        'en': ':keycap_8:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':eight:'],
        'de': ':taste_8:',
        'es': ':teclas_8:',
        'fr': ':touches_8:',
        'ja': ':囲み数字_8:',
        'ko': ':키_캡_8:',
        'pt': ':tecla_8:',
        'it': ':tasto_8:',
        'fa': ':جلد_کلید_8:',
        'id': ':keycap_8:',
        'zh': ':按键_8:',
        'ru': ':клавиши_8:'
    },
    '\U00000038\U000020E3': {  # 8⃣
        'en': ':keycap_8:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':eight:'],
        'de': ':taste_8:',
        'es': ':teclas_8:',
        'fr': ':touches_8:',
        'ja': ':囲み数字_8:',
        'ko': ':키_캡_8:',
        'pt': ':tecla_8:',
        'it': ':tasto_8:',
        'fa': ':جلد_کلید_8:',
        'id': ':keycap_8:',
        'zh': ':按键_8:',
        'ru': ':клавиши_8:'
    },
    '\U00000039\U0000FE0F\U000020E3': {  # 9️⃣
        'en': ':keycap_9:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':nine:'],
        'de': ':taste_9:',
        'es': ':teclas_9:',
        'fr': ':touches_9:',
        'ja': ':囲み数字_9:',
        'ko': ':키_캡_9:',
        'pt': ':tecla_9:',
        'it': ':tasto_9:',
        'fa': ':جلد_کلید_9:',
        'id': ':keycap_9:',
        'zh': ':按键_9:',
        'ru': ':клавиши_9:'
    },
    '\U00000039\U000020E3': {  # 9⃣
        'en': ':keycap_9:',
        'status': unqualified,
        'E': 0.6,
        'alias': [':nine:'],
        'de': ':taste_9:',
        'es': ':teclas_9:',
        'fr': ':touches_9:',
        'ja': ':囲み数字_9:',
        'ko': ':키_캡_9:',
        'pt': ':tecla_9:',
        'it': ':tasto_9:',
        'fa': ':جلد_کلید_9:',
        'id': ':keycap_9:',
        'zh': ':按键_9:',
        'ru': ':клавиши_9:'
    },
    '\U0001FAAF': {  # 🪯
        'en': ':khanda:',
        'status': fully_qualified,
        'E': 15,
        'de': ':khanda:',
        'es': ':khanda:',
        'fr': ':khanda:',
        'ja': ':カンダ:',
        'ko': ':칸다:',
        'pt': ':khanda:',
        'it': ':khanda:',
        'fa': ':خندا:',
        'id': ':khanda:',
        'zh': ':坎达:',
        'ru': ':кханда:'
    },
    '\U0001F6F4': {  # 🛴
        'en': ':kick_scooter:',
        'status': fully_qualified,
        'E': 3,
        'de': ':tretroller:',
        'es': ':patinete:',
        'fr': ':trottinette:',
        'ja': ':キックボード:',
        'ko': ':킥보드:',
        'pt': ':patinete:',
        'it': ':monopattino:',
        'fa': ':چرخ_پایی:',
        'id': ':otoped:',
        'zh': ':滑板车:',
        'ru': ':самокат:'
    },
    '\U0001F458': {  # 👘
        'en': ':kimono:',
        'status': fully_qualified,
        'E': 0.6,
        'de': ':kimono:',
        'es': ':kimono:',
        'fr': ':kimono:',
        'ja': ':着物:',
        'ko': ':기모노:',
        'pt': ':quimono:',
        'it': ':kimono:',
        'fa': ':کیمونو:',
        'id': ':kimono:',
        'zh': ':和服:',
        'ru': ':кимоно:'
    },
    '\U0001F48F': {  # 💏
        'en': ':kiss:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':couplekiss:'],
        'de': ':sich_küssendes_paar:',
        'es': ':beso:',
        'fr': ':bisou:',
        'ja': ':2人でキス:',
        'ko': ':키스:',
        'pt': ':beijo:',
        'it': ':bacio_tra_coppia:',
        'fa': ':بوسه:',
        'id': ':berciuman:',
        'zh': ':亲吻:',
        'ru': ':поцелуй:'
    },
    '\U0001F48F\U0001F3FF': {  # 💏🏿
        'en': ':kiss_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_dunkle_hautfarbe:',
        'es': ':beso_tono_de_piel_oscuro:',
        'fr': ':bisou_peau_foncée:',
        'ja': ':2人でキス_濃い肌色:',
        'ko': ':키스_검은색_피부:',
        'pt': ':beijo_pele_escura:',
        'it': ':bacio_tra_coppia_carnagione_scura:',
        'fa': ':بوسه_پوست_آبنوسی:',
        'id': ':berciuman_warna_kulit_gelap:',
        'zh': ':亲吻_较深肤色:',
        'ru': ':поцелуй_очень_темный_тон_кожи:'
    },
    '\U0001F48F\U0001F3FB': {  # 💏🏻
        'en': ':kiss_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_helle_hautfarbe:',
        'es': ':beso_tono_de_piel_claro:',
        'fr': ':bisou_peau_claire:',
        'ja': ':2人でキス_薄い肌色:',
        'ko': ':키스_하얀_피부:',
        'pt': ':beijo_pele_clara:',
        'it': ':bacio_tra_coppia_carnagione_chiara:',
        'fa': ':بوسه_پوست_سفید:',
        'id': ':berciuman_warna_kulit_cerah:',
        'zh': ':亲吻_较浅肤色:',
        'ru': ':поцелуй_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468': {  # 👨‍❤️‍💋‍👨
        'en': ':kiss_man_man:',
        'status': fully_qualified,
        'E': 2,
        'alias': [':couplekiss_man_man:'],
        'de': ':sich_küssendes_paar_mann,_mann:',
        'es': ':beso_hombre_y_hombre:',
        'fr': ':bisou_homme_et_homme:',
        'ja': ':2人でキス_男性_男性:',
        'ko': ':키스_남자_남자:',
        'pt': ':beijo_homem_e_homem:',
        'it': ':bacio_tra_coppia_uomo_e_uomo:',
        'fa': ':بوسه_مرد_مرد:',
        'id': ':berciuman_pria_pria:',
        'zh': ':亲吻_男人男人:',
        'ru': ':поцелуи_мужчина_мужчина:'
    },
    '\U0001F468\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468': {  # 👨‍❤‍💋‍👨
        'en': ':kiss_man_man:',
        'status': minimally_qualified,
        'E': 2,
        'alias': [':couplekiss_man_man:'],
        'de': ':sich_küssendes_paar_mann_mann:',
        'es': ':beso_hombre_y_hombre:',
        'fr': ':bisou_homme_et_homme:',
        'ja': ':2人でキス_男性_男性:',
        'ko': ':키스_남자_남자:',
        'pt': ':beijo_homem_e_homem:',
        'it': ':bacio_tra_coppia_uomo_e_uomo:',
        'fa': ':بوسه_مرد_مرد:',
        'id': ':berciuman_pria_pria:',
        'zh': ':亲吻_男人男人:',
        'ru': ':поцелуй_мужчина_мужчина:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏿‍❤️‍💋‍👨🏿
        'en': ':kiss_man_man_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_濃い肌色:',
        'ko': ':키스_남자_남자_검은색_피부:',
        'pt': ':beijo_homem_homem_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人较深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏿‍❤‍💋‍👨🏿
        'en': ':kiss_man_man_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_濃い肌色:',
        'ko': ':키스_남자_남자_검은색_피부:',
        'pt': ':beijo_homem_homem_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人较深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏿‍❤️‍💋‍👨🏻
        'en': ':kiss_man_man_dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_peau_foncée_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_濃い肌色_薄い肌色:',
        'ko': ':키스_남자_남자_검은색_피부_하얀_피부:',
        'pt': ':beijo_homem_homem_pele_escura_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_scura_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人较深肤色较浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏿‍❤‍💋‍👨🏻
        'en': ':kiss_man_man_dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_peau_foncée_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_濃い肌色_薄い肌色:',
        'ko': ':키스_남자_남자_검은색_피부_하얀_피부:',
        'pt': ':beijo_homem_homem_pele_escura_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_scura_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人较深肤色较浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏿‍❤️‍💋‍👨🏾
        'en': ':kiss_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_peau_foncée_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_濃い肌色_やや濃い肌色:',
        'ko': ':키스_남자_남자_검은색_피부_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_escura_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人较深肤色中等深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_темныи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏿‍❤‍💋‍👨🏾
        'en': ':kiss_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_peau_foncée_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_濃い肌色_やや濃い肌色:',
        'ko': ':키스_남자_남자_검은색_피부_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_escura_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人较深肤色中等深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_темный_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏿‍❤️‍💋‍👨🏼
        'en': ':kiss_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_濃い肌色_やや薄い肌色:',
        'ko': ':키스_남자_남자_검은색_피부_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_escura_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人较深肤色中等浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏿‍❤‍💋‍👨🏼
        'en': ':kiss_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_濃い肌色_やや薄い肌色:',
        'ko': ':키스_남자_남자_검은색_피부_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_escura_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人较深肤色中等浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏿‍❤️‍💋‍👨🏽
        'en': ':kiss_man_man_dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_濃い肌色_中間の肌色:',
        'ko': ':키스_남자_남자_검은색_피부_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_escura_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人较深肤色中等肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏿‍❤‍💋‍👨🏽
        'en': ':kiss_man_man_dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_濃い肌色_中間の肌色:',
        'ko': ':키스_남자_남자_검은색_피부_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_escura_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人较深肤色中等肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏻‍❤️‍💋‍👨🏻
        'en': ':kiss_man_man_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_薄い肌色:',
        'ko': ':키스_남자_남자_하얀_피부:',
        'pt': ':beijo_homem_homem_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人较浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏻‍❤‍💋‍👨🏻
        'en': ':kiss_man_man_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_薄い肌色:',
        'ko': ':키스_남자_남자_하얀_피부:',
        'pt': ':beijo_homem_homem_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人较浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏻‍❤️‍💋‍👨🏿
        'en': ':kiss_man_man_light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_peau_claire_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_薄い肌色_濃い肌色:',
        'ko': ':키스_남자_남자_하얀_피부_검은색_피부:',
        'pt': ':beijo_homem_homem_pele_clara_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_chiara_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人较浅肤色较深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏻‍❤‍💋‍👨🏿
        'en': ':kiss_man_man_light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_peau_claire_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_薄い肌色_濃い肌色:',
        'ko': ':키스_남자_남자_하얀_피부_검은색_피부:',
        'pt': ':beijo_homem_homem_pele_clara_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_chiara_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人较浅肤色较深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏻‍❤️‍💋‍👨🏾
        'en': ':kiss_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_peau_claire_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_薄い肌色_やや濃い肌色:',
        'ko': ':키스_남자_남자_하얀_피부_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_clara_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_سفید_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人较浅肤色中等深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏻‍❤‍💋‍👨🏾
        'en': ':kiss_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_peau_claire_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_薄い肌色_やや濃い肌色:',
        'ko': ':키스_남자_남자_하얀_피부_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_clara_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_سفید_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人较浅肤色中等深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏻‍❤️‍💋‍👨🏼
        'en': ':kiss_man_man_light_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_peau_claire_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_薄い肌色_やや薄い肌色:',
        'ko': ':키스_남자_남자_하얀_피부_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_clara_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_سفید_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人较浅肤色中等浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_светлыи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏻‍❤‍💋‍👨🏼
        'en': ':kiss_man_man_light_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_peau_claire_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_薄い肌色_やや薄い肌色:',
        'ko': ':키스_남자_남자_하얀_피부_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_clara_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_سفید_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人较浅肤色中等浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_светлый_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏻‍❤️‍💋‍👨🏽
        'en': ':kiss_man_man_light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_peau_claire_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_薄い肌色_中間の肌色:',
        'ko': ':키스_남자_남자_하얀_피부_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_clara_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_پوست_سفید_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人较浅肤色中等肤色:',
        'ru': ':поцелуи_мужчина_мужчина_очень_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏻‍❤‍💋‍👨🏽
        'en': ':kiss_man_man_light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_peau_claire_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_薄い肌色_中間の肌色:',
        'ko': ':키스_남자_남자_하얀_피부_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_clara_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_پوست_سفید_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人较浅肤色中等肤色:',
        'ru': ':поцелуй_мужчина_мужчина_очень_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏾‍❤️‍💋‍👨🏾
        'en': ':kiss_man_man_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人中等深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏾‍❤‍💋‍👨🏾
        'en': ':kiss_man_man_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人中等深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏾‍❤️‍💋‍👨🏿
        'en': ':kiss_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_peau_mate_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色_濃い肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부_검은색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_escura_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人中等深肤色较深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_темныи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏾‍❤‍💋‍👨🏿
        'en': ':kiss_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_peau_mate_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色_濃い肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부_검은색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_escura_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人中等深肤色较深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_темный_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏾‍❤️‍💋‍👨🏻
        'en': ':kiss_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_peau_mate_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色_薄い肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부_하얀_피부:',
        'pt': ':beijo_homem_homem_pele_morena_escura_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_گندمی_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人中等深肤色较浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏾‍❤‍💋‍👨🏻
        'en': ':kiss_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_peau_mate_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色_薄い肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부_하얀_피부:',
        'pt': ':beijo_homem_homem_pele_morena_escura_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_گندمی_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人中等深肤色较浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏾‍❤️‍💋‍👨🏼
        'en': ':kiss_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_peau_mate_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色_やや薄い肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_escura_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_گندمی_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人中等深肤色中等浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏾‍❤‍💋‍👨🏼
        'en': ':kiss_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mitteldunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_peau_mate_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色_やや薄い肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_escura_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_گندمی_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人中等深肤色中等浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏾‍❤️‍💋‍👨🏽
        'en': ':kiss_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_peau_mate_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色_中間の肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_escura_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_پوست_گندمی_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人中等深肤色中等肤色:',
        'ru': ':поцелуи_мужчина_мужчина_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏾‍❤‍💋‍👨🏽
        'en': ':kiss_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mitteldunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_oscuro_medio_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_peau_mate_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_やや濃い肌色_中間の肌色:',
        'ko': ':키스_남자_남자_진한_갈색_피부_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_escura_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_scura_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_پوست_گندمی_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_gelap-sedang_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人中等深肤色中等肤色:',
        'ru': ':поцелуй_мужчина_мужчина_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏼‍❤️‍💋‍👨🏼
        'en': ':kiss_man_man_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人中等浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏼‍❤‍💋‍👨🏼
        'en': ':kiss_man_man_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人中等浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏼‍❤️‍💋‍👨🏿
        'en': ':kiss_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色_濃い肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부_검은색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_clara_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人中等浅肤色较深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏼‍❤‍💋‍👨🏿
        'en': ':kiss_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittelhelle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_peau_moyennement_claire_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色_濃い肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부_검은색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_clara_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_روشن_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人中等浅肤色较深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏼‍❤️‍💋‍👨🏻
        'en': ':kiss_man_man_medium-light_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_peau_moyennement_claire_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色_薄い肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부_하얀_피부:',
        'pt': ':beijo_homem_homem_pele_morena_clara_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_روشن_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人中等浅肤色较浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_светлыи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏼‍❤‍💋‍👨🏻
        'en': ':kiss_man_man_medium-light_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittelhelle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_peau_moyennement_claire_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色_薄い肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부_하얀_피부:',
        'pt': ':beijo_homem_homem_pele_morena_clara_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_روشن_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人中等浅肤色较浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_светлый_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏼‍❤️‍💋‍👨🏾
        'en': ':kiss_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_peau_moyennement_claire_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色_やや濃い肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_clara_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_روشن_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人中等浅肤色中等深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏼‍❤‍💋‍👨🏾
        'en': ':kiss_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittelhelle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_peau_moyennement_claire_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色_やや濃い肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_clara_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_روشن_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人中等浅肤色中等深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏼‍❤️‍💋‍👨🏽
        'en': ':kiss_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色_中間の肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_clara_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_پوست_روشن_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人中等浅肤色中等肤色:',
        'ru': ':поцелуи_мужчина_мужчина_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏼‍❤‍💋‍👨🏽
        'en': ':kiss_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittelhelle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_claro_medio_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_peau_moyennement_claire_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_やや薄い肌色_中間の肌色:',
        'ko': ':키스_남자_남자_연한_갈색_피부_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_clara_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_abbastanza_chiara_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_پوست_روشن_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_cerah-sedang_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人中等浅肤色中等肤色:',
        'ru': ':поцелуй_мужчина_мужчина_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏽‍❤️‍💋‍👨🏽
        'en': ':kiss_man_man_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_中間の肌色:',
        'ko': ':키스_남자_남자_갈색_피부:',
        'pt': ':beijo_homem_homem_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人中等肤色:',
        'ru': ':поцелуи_мужчина_мужчина_среднии_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': {  # 👨🏽‍❤‍💋‍👨🏽
        'en': ':kiss_man_man_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_und_mittlere_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio:',
        'fr': ':bisou_homme_homme_et_peau_légèrement_mate:',
        'ja': ':2人でキス_男性_男性_中間の肌色:',
        'ko': ':키스_남자_남자_갈색_피부:',
        'pt': ':beijo_homem_homem_e_pele_morena:',
        'it': ':bacio_tra_coppia_uomo_uomo_e_carnagione_olivastra:',
        'fa': ':بوسه_مرد_مرد_و_پوست_طلایی:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang:',
        'zh': ':亲吻_男人男人中等肤色:',
        'ru': ':поцелуй_мужчина_мужчина_средний_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏽‍❤️‍💋‍👨🏿
        'en': ':kiss_man_man_medium_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_中間の肌色_濃い肌色:',
        'ko': ':키스_남자_남자_갈색_피부_검은색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人中等肤色较深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_среднии_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': {  # 👨🏽‍❤‍💋‍👨🏿
        'en': ':kiss_man_man_medium_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittlere_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio_tono_de_piel_oscuro:',
        'fr': ':bisou_homme_homme_peau_légèrement_mate_et_peau_foncée:',
        'ja': ':2人でキス_男性_男性_中間の肌色_濃い肌色:',
        'ko': ':키스_남자_남자_갈색_피부_검은색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_e_pele_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_olivastra_e_carnagione_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_طلایی_و_پوست_آبنوسی:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang_warna_kulit_gelap:',
        'zh': ':亲吻_男人男人中等肤色较深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_средний_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏽‍❤️‍💋‍👨🏻
        'en': ':kiss_man_man_medium_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_peau_légèrement_mate_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_中間の肌色_薄い肌色:',
        'ko': ':키스_남자_남자_갈색_피부_하얀_피부:',
        'pt': ':beijo_homem_homem_pele_morena_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_طلایی_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人中等肤色较浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_среднии_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': {  # 👨🏽‍❤‍💋‍👨🏻
        'en': ':kiss_man_man_medium_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittlere_hautfarbe,helle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio_tono_de_piel_claro:',
        'fr': ':bisou_homme_homme_peau_légèrement_mate_et_peau_claire:',
        'ja': ':2人でキス_男性_男性_中間の肌色_薄い肌色:',
        'ko': ':키스_남자_남자_갈색_피부_하얀_피부:',
        'pt': ':beijo_homem_homem_pele_morena_e_pele_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_olivastra_e_carnagione_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_طلایی_و_پوست_سفید:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang_warna_kulit_cerah:',
        'zh': ':亲吻_男人男人中等肤色较浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_средний_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏽‍❤️‍💋‍👨🏾
        'en': ':kiss_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_peau_légèrement_mate_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_中間の肌色_やや濃い肌色:',
        'ko': ':키스_남자_남자_갈색_피부_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_طلایی_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人中等肤色中等深肤色:',
        'ru': ':поцелуи_мужчина_мужчина_среднии_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': {  # 👨🏽‍❤‍💋‍👨🏾
        'en': ':kiss_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittlere_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_homme_homme_peau_légèrement_mate_et_peau_mate:',
        'ja': ':2人でキス_男性_男性_中間の肌色_やや濃い肌色:',
        'ko': ':키스_남자_남자_갈색_피부_진한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_olivastra_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_مرد_مرد_پوست_طلایی_و_پوست_گندمی:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_男人男人中等肤色中等深肤色:',
        'ru': ':поцелуй_мужчина_мужчина_средний_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏽‍❤️‍💋‍👨🏼
        'en': ':kiss_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_中間の肌色_やや薄い肌色:',
        'ko': ':키스_남자_남자_갈색_피부_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_طلایی_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人中等肤色中等浅肤色:',
        'ru': ':поцелуи_мужчина_мужчина_среднии_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': {  # 👨🏽‍❤‍💋‍👨🏼
        'en': ':kiss_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mann_mann_mittlere_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_hombre_hombre_tono_de_piel_medio_tono_de_piel_claro_medio:',
        'fr': ':bisou_homme_homme_peau_légèrement_mate_et_peau_moyennement_claire:',
        'ja': ':2人でキス_男性_男性_中間の肌色_やや薄い肌色:',
        'ko': ':키스_남자_남자_갈색_피부_연한_갈색_피부:',
        'pt': ':beijo_homem_homem_pele_morena_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_uomo_uomo_carnagione_olivastra_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_مرد_مرد_پوست_طلایی_و_پوست_روشن:',
        'id': ':berciuman_pria_pria_warna_kulit_sedang_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_男人男人中等肤色中等浅肤色:',
        'ru': ':поцелуй_мужчина_мужчина_средний_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F48B': {  # 💋
        'en': ':kiss_mark:',
        'status': fully_qualified,
        'E': 0.6,
        'alias': [':kiss:'],
        'de': ':kussabdruck:',
        'es': ':marca_de_beso:',
        'fr': ':trace_de_rouge_à_lèvres:',
        'ja': ':キスマーク:',
        'ko': ':입술:',
        'pt': ':marca_de_beijo:',
        'it': ':impronta_della_bocca:',
        'fa': ':بوسه_ماتیکی:',
        'id': ':tanda_ciuman:',
        'zh': ':唇印:',
        'ru': ':след_от_поцелуя:'
    },
    '\U0001F48F\U0001F3FE': {  # 💏🏾
        'en': ':kiss_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mitteldunkle_hautfarbe:',
        'es': ':beso_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_peau_mate:',
        'ja': ':2人でキス_やや濃い肌色:',
        'ko': ':키스_진한_갈색_피부:',
        'pt': ':beijo_pele_morena_escura:',
        'it': ':bacio_tra_coppia_carnagione_abbastanza_scura:',
        'fa': ':بوسه_پوست_گندمی:',
        'id': ':berciuman_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_中等深肤色:',
        'ru': ':поцелуй_темный_тон_кожи:'
    },
    '\U0001F48F\U0001F3FC': {  # 💏🏼
        'en': ':kiss_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mittelhelle_hautfarbe:',
        'es': ':beso_tono_de_piel_claro_medio:',
        'fr': ':bisou_peau_moyennement_claire:',
        'ja': ':2人でキス_やや薄い肌色:',
        'ko': ':키스_연한_갈색_피부:',
        'pt': ':beijo_pele_morena_clara:',
        'it': ':bacio_tra_coppia_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_پوست_روشن:',
        'id': ':berciuman_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_中等浅肤色:',
        'ru': ':поцелуй_светлый_тон_кожи:'
    },
    '\U0001F48F\U0001F3FD': {  # 💏🏽
        'en': ':kiss_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_mittlere_hautfarbe:',
        'es': ':beso_tono_de_piel_medio:',
        'fr': ':bisou_peau_légèrement_mate:',
        'ja': ':2人でキス_中間の肌色:',
        'ko': ':키스_갈색_피부:',
        'pt': ':beijo_pele_morena:',
        'it': ':bacio_tra_coppia_carnagione_olivastra:',
        'fa': ':بوسه_پوست_طلایی:',
        'id': ':berciuman_warna_kulit_sedang:',
        'zh': ':亲吻_中等肤色:',
        'ru': ':поцелуй_средний_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏿‍❤️‍💋‍🧑🏻
        'en': ':kiss_person_person_dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':bisou_adulte_adulte_peau_foncée_et_peau_claire:',
        'ja': ':2人でキス_大人_大人_濃い肌色_薄い肌色:',
        'ko': ':키스_사람_사람_검은색_피부_하얀_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_escura_e_pele_clara:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_scura_e_carnagione_chiara:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':亲吻_成人成人较深肤色较浅肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_очень_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏿‍❤‍💋‍🧑🏻
        'en': ':kiss_person_person_dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_dunkle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_claro:',
        'fr': ':bisou_adulte_adulte_peau_foncée_et_peau_claire:',
        'ja': ':2人でキス_大人_大人_濃い肌色_薄い肌色:',
        'ko': ':키스_사람_사람_검은색_피부_하얀_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_escura_e_pele_clara:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_scura_e_carnagione_chiara:',
        'fa': ':بوسه_شخص_شخص_پوست_آبنوسی_و_پوست_سفید:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap_warna_kulit_cerah:',
        'zh': ':亲吻_成人成人较深肤色较浅肤色:',
        'ru': ':поцелуй_взрослый_взрослый_очень_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏿‍❤️‍💋‍🧑🏾
        'en': ':kiss_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_adulte_adulte_peau_foncée_et_peau_mate:',
        'ja': ':2人でキス_大人_大人_濃い肌色_やや濃い肌色:',
        'ko': ':키스_사람_사람_검은색_피부_진한_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_escura_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_成人成人较深肤色中等深肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_очень_темныи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏿‍❤‍💋‍🧑🏾
        'en': ':kiss_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_dunkle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_adulte_adulte_peau_foncée_et_peau_mate:',
        'ja': ':2人でキス_大人_大人_濃い肌色_やや濃い肌色:',
        'ko': ':키스_사람_사람_검은색_피부_진한_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_escura_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_scura_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_شخص_شخص_پوست_آبنوسی_و_پوست_گندمی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_成人成人较深肤色中等深肤色:',
        'ru': ':поцелуй_взрослый_взрослый_очень_темный_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏿‍❤️‍💋‍🧑🏼
        'en': ':kiss_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':bisou_adulte_adulte_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':2人でキス_大人_大人_濃い肌色_やや薄い肌色:',
        'ko': ':키스_사람_사람_검은색_피부_연한_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_escura_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_成人成人较深肤色中等浅肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_очень_темныи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏿‍❤‍💋‍🧑🏼
        'en': ':kiss_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_dunkle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_claro_medio:',
        'fr': ':bisou_adulte_adulte_peau_foncée_et_peau_moyennement_claire:',
        'ja': ':2人でキス_大人_大人_濃い肌色_やや薄い肌色:',
        'ko': ':키스_사람_사람_검은색_피부_연한_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_escura_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_scura_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_شخص_شخص_پوست_آبنوسی_و_پوست_روشن:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_成人成人较深肤色中等浅肤色:',
        'ru': ':поцелуй_взрослый_взрослый_очень_темный_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏿‍❤️‍💋‍🧑🏽
        'en': ':kiss_person_person_dark_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':bisou_adulte_adulte_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':2人でキス_大人_大人_濃い肌色_中間の肌色:',
        'ko': ':키스_사람_사람_검은색_피부_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_escura_e_pele_morena:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':亲吻_成人成人较深肤色中等肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_очень_темныи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏿‍❤‍💋‍🧑🏽
        'en': ':kiss_person_person_dark_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_dunkle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_tono_de_piel_medio:',
        'fr': ':bisou_adulte_adulte_peau_foncée_et_peau_légèrement_mate:',
        'ja': ':2人でキス_大人_大人_濃い肌色_中間の肌色:',
        'ko': ':키스_사람_사람_검은색_피부_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_escura_e_pele_morena:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_scura_e_carnagione_olivastra:',
        'fa': ':بوسه_شخص_شخص_پوست_آبنوسی_و_پوست_طلایی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap_warna_kulit_sedang:',
        'zh': ':亲吻_成人成人较深肤色中等肤色:',
        'ru': ':поцелуй_взрослый_взрослый_очень_темный_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏻‍❤️‍💋‍🧑🏿
        'en': ':kiss_person_person_light_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':bisou_adulte_adulte_peau_claire_et_peau_foncée:',
        'ja': ':2人でキス_大人_大人_薄い肌色_濃い肌色:',
        'ko': ':키스_사람_사람_하얀_피부_검은색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_clara_e_pele_escura:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_chiara_e_carnagione_scura:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':亲吻_成人成人较浅肤色较深肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_очень_светлыи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏻‍❤‍💋‍🧑🏿
        'en': ':kiss_person_person_light_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_helle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_oscuro:',
        'fr': ':bisou_adulte_adulte_peau_claire_et_peau_foncée:',
        'ja': ':2人でキス_大人_大人_薄い肌色_濃い肌色:',
        'ko': ':키스_사람_사람_하얀_피부_검은색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_clara_e_pele_escura:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_chiara_e_carnagione_scura:',
        'fa': ':بوسه_شخص_شخص_پوست_سفید_و_پوست_آبنوسی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_cerah_warna_kulit_gelap:',
        'zh': ':亲吻_成人成人较浅肤色较深肤色:',
        'ru': ':поцелуй_взрослый_взрослый_очень_светлый_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏻‍❤️‍💋‍🧑🏾
        'en': ':kiss_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_adulte_adulte_peau_claire_et_peau_mate:',
        'ja': ':2人でキス_大人_大人_薄い肌色_やや濃い肌色:',
        'ko': ':키스_사람_사람_하얀_피부_진한_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_clara_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_سفید_و_پوست_گندمی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_成人成人较浅肤色中等深肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_очень_светлыи_тон_кожи_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': {  # 🧑🏻‍❤‍💋‍🧑🏾
        'en': ':kiss_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_helle_hautfarbe,mitteldunkle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_oscuro_medio:',
        'fr': ':bisou_adulte_adulte_peau_claire_et_peau_mate:',
        'ja': ':2人でキス_大人_大人_薄い肌色_やや濃い肌色:',
        'ko': ':키스_사람_사람_하얀_피부_진한_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_clara_e_pele_morena_escura:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_chiara_e_carnagione_abbastanza_scura:',
        'fa': ':بوسه_شخص_شخص_پوست_سفید_و_پوست_گندمی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_cerah_warna_kulit_gelap-sedang:',
        'zh': ':亲吻_成人成人较浅肤色中等深肤色:',
        'ru': ':поцелуй_взрослый_взрослый_очень_светлый_тон_кожи_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏻‍❤️‍💋‍🧑🏼
        'en': ':kiss_person_person_light_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':bisou_adulte_adulte_peau_claire_et_peau_moyennement_claire:',
        'ja': ':2人でキス_大人_大人_薄い肌色_やや薄い肌色:',
        'ko': ':키스_사람_사람_하얀_피부_연한_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_clara_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_سفید_و_پوست_روشن:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_成人成人较浅肤色中等浅肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_очень_светлыи_тон_кожи_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏻‍❤‍💋‍🧑🏼
        'en': ':kiss_person_person_light_skin_tone_medium-light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_helle_hautfarbe,mittelhelle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_claro_medio:',
        'fr': ':bisou_adulte_adulte_peau_claire_et_peau_moyennement_claire:',
        'ja': ':2人でキス_大人_大人_薄い肌色_やや薄い肌色:',
        'ko': ':키스_사람_사람_하얀_피부_연한_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_clara_e_pele_morena_clara:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_chiara_e_carnagione_abbastanza_chiara:',
        'fa': ':بوسه_شخص_شخص_پوست_سفید_و_پوست_روشن:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_cerah_warna_kulit_cerah-sedang:',
        'zh': ':亲吻_成人成人较浅肤色中等浅肤色:',
        'ru': ':поцелуй_взрослый_взрослый_очень_светлый_тон_кожи_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏻‍❤️‍💋‍🧑🏽
        'en': ':kiss_person_person_light_skin_tone_medium_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':bisou_adulte_adulte_peau_claire_et_peau_légèrement_mate:',
        'ja': ':2人でキス_大人_大人_薄い肌色_中間の肌色:',
        'ko': ':키스_사람_사람_하얀_피부_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_clara_e_pele_morena:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_سفید_و_پوست_طلایی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':亲吻_成人成人较浅肤色中等肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_очень_светлыи_тон_кожи_среднии_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': {  # 🧑🏻‍❤‍💋‍🧑🏽
        'en': ':kiss_person_person_light_skin_tone_medium_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_helle_hautfarbe,mittlere_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_claro_tono_de_piel_medio:',
        'fr': ':bisou_adulte_adulte_peau_claire_et_peau_légèrement_mate:',
        'ja': ':2人でキス_大人_大人_薄い肌色_中間の肌色:',
        'ko': ':키스_사람_사람_하얀_피부_갈색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_clara_e_pele_morena:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_chiara_e_carnagione_olivastra:',
        'fa': ':بوسه_شخص_شخص_پوست_سفید_و_پوست_طلایی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_cerah_warna_kulit_sedang:',
        'zh': ':亲吻_成人成人较浅肤色中等肤色:',
        'ru': ':поцелуй_взрослый_взрослый_очень_светлый_тон_кожи_средний_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏾‍❤️‍💋‍🧑🏿
        'en': ':kiss_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':bisou_adulte_adulte_peau_mate_et_peau_foncée:',
        'ja': ':2人でキス_大人_大人_やや濃い肌色_濃い肌色:',
        'ko': ':키스_사람_사람_진한_갈색_피부_검은색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_morena_escura_e_pele_escura:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':亲吻_成人成人中等深肤色较深肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_темныи_тон_кожи_очень_темныи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': {  # 🧑🏾‍❤‍💋‍🧑🏿
        'en': ':kiss_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_mitteldunkle_hautfarbe,dunkle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_oscuro:',
        'fr': ':bisou_adulte_adulte_peau_mate_et_peau_foncée:',
        'ja': ':2人でキス_大人_大人_やや濃い肌色_濃い肌色:',
        'ko': ':키스_사람_사람_진한_갈색_피부_검은색_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_morena_escura_e_pele_escura:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_abbastanza_scura_e_carnagione_scura:',
        'fa': ':بوسه_شخص_شخص_پوست_گندمی_و_پوست_آبنوسی:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_gelap:',
        'zh': ':亲吻_成人成人中等深肤色较深肤色:',
        'ru': ':поцелуй_взрослый_взрослый_темный_тон_кожи_очень_темный_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏾‍❤️‍💋‍🧑🏻
        'en': ':kiss_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_erwachsener_erwachsener_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':bisou_adulte_adulte_peau_mate_et_peau_claire:',
        'ja': ':2人でキス_大人_大人_やや濃い肌色_薄い肌色:',
        'ko': ':키스_사람_사람_진한_갈색_피부_하얀_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_morena_escura_e_pele_clara:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':بوسه_بزرگسال_بزرگسال_پوست_گندمی_و_پوست_سفید:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':亲吻_成人成人中等深肤色较浅肤色:',
        'ru': ':поцелуи_взрослыи_взрослыи_темныи_тон_кожи_очень_светлыи_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': {  # 🧑🏾‍❤‍💋‍🧑🏻
        'en': ':kiss_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status': minimally_qualified,
        'E': 13.1,
        'de': ':sich_küssendes_paar_person_person_mitteldunkle_hautfarbe,helle_hautfarbe:',
        'es': ':beso_persona_adulta_persona_adulta_tono_de_piel_oscuro_medio_tono_de_piel_claro:',
        'fr': ':bisou_adulte_adulte_peau_mate_et_peau_claire:',
        'ja': ':2人でキス_大人_大人_やや濃い肌色_薄い肌色:',
        'ko': ':키스_사람_사람_진한_갈색_피부_하얀_피부:',
        'pt': ':beijo_pessoa_pessoa_pele_morena_escura_e_pele_clara:',
        'it': ':bacio_tra_coppia_persona_persona_carnagione_abbastanza_scura_e_carnagione_chiara:',
        'fa': ':بوسه_شخص_شخص_پوست_گندمی_و_پوست_سفید:',
        'id': ':berciuman_dewasa_dewasa_warna_kulit_gelap-sedang_warna_kulit_cerah:',
        'zh': ':亲吻_成人成人中等深肤色较浅肤色:',
        'ru': ':поцелуй_взрослый_взрослый_темный_тон_кожи_очень_светлый_тон_кожи:'
    },
    '\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': {  # 🧑🏾‍❤️‍💋‍🧑🏼
        'en': ':kiss_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status': fully_qualified,
        'E': 13.1,
        'de': ':sich_küs