package PAR::StrippedPARL::Dynamic;
use 5.008001;
use strict;
use warnings;
our $VERSION = '0.958';

use base 'PAR::StrippedPARL::Base';

our $Data_Pos = tell DATA;

=head1 NAME

PAR::StrippedPARL::Dynamic - Data package containing a dynamic PARL

=head1 SYNOPSIS

  # For details, see PAR::StrippedPARL::Base.
  PAR::StrippedPARL::Dynamic->write_parl($file) or die "Some error...";

=head1 DESCRIPTION

This class is internal to PAR. Do not use it outside of PAR.

This class is basically just a container for a dynamic binary PAR loader
which doesn't include the PAR code like the F<parl> or F<parl.exe>
you are used to. If you're really curious, I'll tell you it is
just a copy of the F<myldr/par> (or F<myldr/par.exe>) file.

The data is appended during the C<make> phase of the PAR build process,
but only if applicable: If you perl is static, you won't get the dynamic
loader.

If the binary data isn't appended during the build process, the class
methods will return the empty list.

=head1 CLASS METHODS

Inherits the methods from L<PAR::StrippedPARL::Base>.

=cut

sub _data_pos {
    my $class = shift;
    return $Data_Pos;
}

=head1 AUTHORS

Steffen Mueller E<lt>smueller@cpan.orgE<gt>,
Audrey Tang E<lt>cpan@audreyt.orgE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright 2006-2009 by Steffen Mueller E<lt>smueller@cpan.orgE<gt>.

This program is free software; you can redistribute it and/or 
modify it under the same terms as Perl itself.

See F<LICENSE>.

=cut

$^V eq 5.38.1 
    or die sprintf("Perl (%s) version (%vd) doesn't match the version (5.38.1) ".
                   "that PAR::Packer was built with; please rebuild PAR::Packer", 
                   $^X, $^V);

1;

__DATA__
M?T5,1@$!`0````````````,`*``!````]!@``#0```"\H0````0`!30`(``)
M`"@`'``;``$``'"`C0``@(T``("-``!P````<`````0````$````!@```#0`
M```T````-````"`!```@`0``!`````0````#````5`$``%0!``!4`0``&0``
M`!D````$`````0````$```````````````````#TC0``](T```4```````$`
M`0```*R=``"LG0$`K)T!`)P"``"P`@``!@```````0`"````M)T``+2=`0"T
MG0$`&`$``!@!```&````!`````0```!P`0``<`$``'`!``!$````1`````0`
M```$````4>5T9```````````````````````````!@```!````!2Y71DK)T`
M`*R=`0"LG0$`5`(``%0"```$`````0```"]L:6(O;&0M;&EN=7@M87)M:&8N
M<V\N,P`````$````%`````,```!'3E4`==<R'LP*`@>Q*5*?02^X_CE]<S<$
M````$`````$```!'3E4```````,````"`````````#L```!.````"`````@`
M```!D!`$)4`$@"B```6@AE"$@($``@LH!(PX@!)!B@``"`````!.````4```
M`%$```````````````````!3`````````%0```!5````5@``````````````
M5P```%@`````````````````````````60```%H```!;`````````%P`````
M``````````````!=````7@```%\```````````````````!@````80``````
M``!C`````````&0`````````90``````````````````````````````````
M`&8```````````````````!G`````````&@```````````````````!J````
M?JF<FUGOEWEK"4/6=E0\E"'_Y`X-.I<+P^4_+(D'<-/9<5@<K4OCP&L:GE#5
M9-1\F4_#FR_%F[8S8MOM:W^:?`&O-.B[XY)\LP&+;T+ZUX=#1=7L9Z_,9A6,
M)<RCX/M)S]Y'+>-'Q@'@IP(*;\7?'[[:5QO[H6Q@````````````````````
M```````L$P````````,`"P```````*`!```````#`!<`W0(`````````````
M$@```!D!`````````````!(```!5`@`````````````2````AP(`````````
M````$@```#P"`````````````!(```!C`0`````````````2````X@``````
M````````$@```!$#`````````````!(```#X```````````````2`````0``
M````````````(````*D!`````````````!(```":```````````````2````
MPP``````````````$@```/L!`````````````!(````R`0`````````````2
M````=P(`````````````$@```!```````````````"````#M````````````
M```2````"@(`````````````$@```!X!`````````````!(```"D`0``````
M```````2````U@(`````````````$@```"P``````````````"````"``@``
M```````````2````0`$`````````````$@```((!`````````````!8```"*
M```````````````2````1@``````````````$@```+(``````````````!(`
M``"C```````````````2````+@(`````````````$@```.T!````````````
M`!(```!P`@`````````````2````]@,`````````````$@```",!````````
M`````!(```#E`@`````````````1````Y@$`````````````$@```/\"````
M`````````!$```"7`@`````````````2````(0(`````````````$@````L!
M`````````````!(````8`@`````````````2````-0(`````````````$@``
M`%<``````````````!(```"/`@`````````````2````>P$`````````````
M$@```,4"`````````````!(```#1`0`````````````2````]`$`````````
M````$@```"@"`````````````!(```!_`@`````````````2````MP$`````
M````````$@```.T"`````````````!(```#D`@`````````````A````3@$`
M````````````$@```+`!`````````````!(```!O`0`````````````2````
MO`(`````````````$@```.,"`````````````!$```!I`@`````````````1
M````;@``````````````$0```$L"`````````````!(```#3````````````
M```1````6P(`````````````$@````("`````````````!(```!C````````
M```````2````0P(`````````````$@```'L``````````````!(```"_`0``
M```````````2````*`,`````````````$@```!$"`````````````!(```#?
M`0`````````````2````MP(`````````````$@```*8"`````````````!(`
M``"5`0`````````````B````Y`,``*0O```D````$@`-`$@$``!<H`$`````
M`!``&``:!````*`!```````0`!<`)P0``$B@`0``````$``8`)P$```8(P``
MD`(``!(`#0`<!````*`!```````@`!<`0`0``*PA``!4````$@`-`*D$``!$
MH`$`!````!$`%P!<!```2*`!```````0`!@`"P0``&A+```$````$0`/`-H#
M``#@20``;`$``!(`#0"4`P``+#@``-0"```2``T`LP0``&PM``!T````$@`-
M`'8#```4(P``!````!(`#0`@!```]!@````````2``T`S`$``"`6``#4`@``
M$@`-`%$$``!<H`$``````!``&`!Q!```7*`!```````0`!@`J@,``,PL``"@
M````$@`-`#4$``#(2```&`$``!(`#0"#!```2*`!```````0`!<`S@,``-@E
M``"P`0``$@`-`&@$``!T2```5````!(`#0"[!```B"<``$0%```2``T`=@0`
M`#PQ``#P!@``$@`-`(<#``"H)0``,````!(`#0"X`P```"(``!0!```2``T`
M_P,````[```\`0``$@`-`%`$``!<H`$``````!``&`"*!```R"\``'0!```2
M``T``%]?9VUO;E]S=&%R=%]?`%])5$U?9&5R96=I<W1E<E1-0VQO;F5486)L
M90!?251-7W)E9VES=&5R5$U#;&]N951A8FQE`%!E<FQ?871F;W)K7VQO8VL`
M4&5R;%]M9U]S970`4&5R;%]N97=84P!03%]D;U]U;F1U;7``4&5R;%]S>7-?
M:6YI=#,`8F]O=%]$>6YA3&]A9&5R`'!E<FQ?<G5N`'!E<FQ?8V]N<W1R=6-T
M`%!E<FQ?;F5W6%-?9FQA9W,`4&5R;%]G=E]F971C:'!V`%!,7VUE;6]R>5]W
M<F%P`'!E<FQ?<&%R<V4`<&5R;%]A;&QO8P!097)L7V%T9F]R:U]U;FQO8VL`
M<&5R;%]D97-T<G5C=`!P97)L7V9R964`4&5R;%]S=E]S971P=FX`4&5R;%]S
M=E]S971P=@!097)L7W-Y<U]T97)M`%!E<FQ?8W)O86M?;F]C;VYT97AT`%!E
M<FQ?9V5T7W-V`%!E<FQ?<V%F97-Y<VUA;&QO8P!03%]C=7)R96YT7V-O;G1E
M>'0`7U]C>&%?9FEN86QI>F4`<F5A9`!S=')C:'(`9V5T<&ED`&UE;6UO=F4`
M7U]L:6)C7W-T87)T7VUA:6X`7U]F<')I;G1F7V-H:P!S=')C;7``;W!E;C8T
M`'5N;&EN:P!S=')T;VL`;65M<V5T`'-T<FYC;7``9V5T=6ED`&UE;6-H<@!C
M;&]S961I<@!S=')S='(`<FUD:7(`<W1R;&5N`'-T<F1U<`!M96UC;7``<F5A
M;&QO8P!R96%D9&ER-C0`86)O<G0`7U]S<')I;G1F7V-H:P!S=&1E<G(`;65M
M8W!Y`'-T<G)C:'(`;'-T870V-`!O<&5N9&ER`&QS965K-C0`7U]V9G!R:6YT
M9E]C:&L`7U]E<G)N;U]L;V-A=&EO;@!E>&ET`')E861L:6YK`%]?<W1A8VM?
M8VAK7V9A:6P`86-C97-S`&UK9&ER`%]?96YV:7)O;@!?7W)E9VES=&5R7V%T
M9F]R:P!?7W-T86-K7V-H:U]G=6%R9`!?7V%E86)I7W5N=VEN9%]C<'!?<'(P
M`%]?865A8FE?=6YW:6YD7V-P<%]P<C$`;&EB<&5R;"YS;P!L:6)C+G-O+C8`
M;&0M;&EN=7@M87)M:&8N<V\N,P!L:6)G8V-?<RYS;RXQ`'!A<E]C=7)R96YT
M7V5X96,`<&%R7V)A<V5N86UE`%A37TEN=&5R;F%L<U]005)?0D]/5`!P87)?
M96YV7V-L96%N`'!A<E]C=7)R96YT7V5X96-?<')O8P!P87)?9&ER;F%M90!S
M:&%?9FEN86P`9V5T7W5S97)N86UE7V9R;VU?9V5T<'=U:60`<&%R7V-L96%N
M=7``7TE/7W-T9&EN7W5S960`7U]D871A7W-T87)T`%]?8G-S7W-T87)T7U\`
M<VAA7W5P9&%T90!X<U]I;FET`%]?96YD7U\`7U]B<W-?96YD7U\`7U]B<W-?
M<W1A<G0`<VAA7VEN:70`7V5N9`!P87)?;6MT;7!D:7(`7V5D871A`'!A<E]S
M971U<%]L:6)P871H`'!A<E]F:6YD<')O9P!M>5]P87)?<&P`<&%R7V1I90!P
M87)?:6YI=%]E;G8`1TQ)0D-?,BXT`$=#0U\S+C4`1TQ)0D-?,BXS-`!'3$E"
M0U\R+C,S`"]U<W(O;&EB+W!E<FPU+S4N,S@O8V]R95]P97)L+T-/4D4`````
M``````(``0`"``(``@`!``$``P`!``$``@`!``$``@`!``(``0`!``(``@`"
M``(``0`$``$``0`!``$``0`!``(``@`"``(``0`"``(`!0`"``(``0`"``(`
M`0`"``(``@`"``(``@`$``(``@`"``$``@`!``(``@`"``$``@`!``(``@`!
M``(``0`&``,``@`"``(``@`"``$``0`!``$``0`!``$``0`!``$``0`!``$`
M`0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0`!``$``0!4`P``$```
M`"`````4:6D-```%`,@$`````````0`!`&@#```0````(````%4F>0L```,`
MT@0````````!``,`2@,``!``````````M)&6!@``!@#:!```$````+.1E@8`
M``0`Y00``!`````4:6D-```"`,@$````````K)T!`!<```"PG0$`%P```/R?
M`0`7````!*`!`!<````4H`$`%P```!B@`0`7````(*`!`!<````DH`$`%P``
M`"B@`0`7````+*`!`!<````PH`$`%P```#B@`0`7````/*`!`!<```!$H`$`
M%P```,B?`0`5"P``S)\!`!4,``#0GP$`%1,``-2?`0`5&0``V)\!`!,<``#<
MGP$`%1T``."?`0`5'@``Y)\!`!4F``#HGP$`%2@``.R?`0`5/@``\)\!`!4_
M``#TGP$`%4$``/B?`0`530``V)X!`!8#``#<G@$`%@0``.">`0`6!0``Y)X!
M`!8&``#HG@$`%@<``.R>`0`6"```\)X!`!8)``#TG@$`%@P``/B>`0`6#0``
M_)X!`!8.````GP$`%@\```2?`0`6$```")\!`!81```,GP$`%A(``!"?`0`6
M%```%)\!`!85```8GP$`%A8``!R?`0`6%P``()\!`!88```DGP$`%AH``"B?
M`0`6&P``+)\!`!8?```PGP$`%B```#2?`0`6(0``.)\!`!8B```\GP$`%B,`
M`$"?`0`6)```1)\!`!8E``!(GP$`%B<``$R?`0`6*0``4)\!`!8J``!4GP$`
M%BL``%B?`0`6+```7)\!`!8M``!@GP$`%BX``&2?`0`6+P``:)\!`!8P``!L
MGP$`%C$``'"?`0`6,@``=)\!`!8S``!XGP$`%C0``'R?`0`6-0``@)\!`!8V
M``"$GP$`%C<``(B?`0`6.0``C)\!`!8Z``"0GP$`%CL``)2?`0`6/```F)\!
M`!9```"<GP$`%D(``*"?`0`60P``I)\!`!9$``"HGP$`%D4``*R?`0`61@``
ML)\!`!9'``"TGP$`%DD``+B?`0`62@``O)\!`!9+``#`GP$`%DP``,2?`0`6
M30``"$`MZ8`!`.L(@+WH!.`MY03@G^4.X(_@"/"^Y82+`0``QH_B&,J,XH3[
MO.4`QH_B&,J,XGS[O.4`QH_B&,J,XG3[O.4`QH_B&,J,XFS[O.4`QH_B&,J,
MXF3[O.4`QH_B&,J,XES[O.4`QH_B&,J,XE3[O.4`QH_B&,J,XDS[O.4`QH_B
M&,J,XD3[O.4`QH_B&,J,XCS[O.4`QH_B&,J,XC3[O.4`QH_B&,J,XBS[O.4`
MQH_B&,J,XB3[O.4`QH_B&,J,XAS[O.4`QH_B&,J,XA3[O.4`QH_B&,J,X@S[
MO.4`QH_B&,J,X@3[O.4`QH_B&,J,XOSZO.4`QH_B&,J,XO3ZO.4`QH_B&,J,
MXNSZO.4`QH_B&,J,XN3ZO.4`QH_B&,J,XMSZO.4`QH_B&,J,XM3ZO.4`QH_B
M&,J,XLSZO.4`QH_B&,J,XL3ZO.4`QH_B&,J,XKSZO.4`QH_B&,J,XK3ZO.4`
MQH_B&,J,XJSZO.4`QH_B&,J,XJ3ZO.4`QH_B&,J,XISZO.4`QH_B&,J,XI3Z
MO.4`QH_B&,J,XHSZO.4`QH_B&,J,XH3ZO.4`QH_B&,J,XGSZO.4`QH_B&,J,
MXG3ZO.4`QH_B&,J,XFSZO.4`QH_B&,J,XF3ZO.4`QH_B&,J,XESZO.4`QH_B
M&,J,XE3ZO.4`QH_B&,J,XDSZO.4`QH_B&,J,XD3ZO.4`QH_B&,J,XCSZO.4`
MQH_B&,J,XC3ZO.4`QH_B&,J,XBSZO.4`QH_B&,J,XB3ZO.4`QH_B&,J,XASZ
MO.4`QH_B&,J,XA3ZO.4`QH_B&,J,X@SZO.4`QH_B&,J,X@3ZO.4`QH_B&,J,
MXOSYO.4`QH_B&,J,XO3YO.4`QH_B&,J,XNSYO.4`QH_B&,J,XN3YO.4`QH_B
M&,J,XMSYO.4`QH_B&,J,XM3YO.4`QH_B&,J,XLSYO.4`QH_B&,J,XL3YO.4`
MQH_B&,J,XKSYO.4`QH_B&,J,XK3YO.4`QH_B&,J,XJSYO.4`````<$`MZ0'*
MH.,,P$W@R`^,Y63"G^4HT$WB8#*?Y0S`C^!<0I_E'`"-Y1P`C>(8$(WE&!"-
MXA0@C>44((WB!$"/X`,PG.<`,)/E)#"-Y0`PH./4___K+"*?Y2PRG^4"()3G
M#""-Y0(0H.$#,)3G$#"-Y0,`H.$L#0#K$#*?Y0,PE.<`4-/E``!5XU(```IP
M+QWN_#&?Y1P`G>4#4(#B!`"`X@,PG^<'`7#C`R"2YZ8STN4#,(/CIC/"Y6,`
M`(H``:#AH___Z\P1G^7,(9_E`#"@X0$0C^`80)WE`B"/X``0D>4$`(+E``"4
MY0@0@^6L$9_E'""=Y0$0C^`,$(/EH!&?Y0$`4N,``(/E`1"/X`00@^54``#:
M!!"$X@S`@^("083@!."1Y`3@K.4!`%3A^___&@,0@N)L09_E``"@XP``C>4%
M(*#A!$"/X`$!@^=8$9_E#`"4Y0$0C^`%___K``!0XRD```I$49_E(&"-X@50
MC^`,`)7E2?__ZP!`H.$P`9_E!A"@X0``C^"N``#K``!0XR````H<,9_E`S"/
MX`P`D^7D_O_K'/__ZPPAG^7,,)_E`B"/X`,PDN<`()/E)#"=Y0(@,^``,*#C
M*@``&@0`H.$HT(WB<("]Z/W^_^O<()_E``!0XP(@C^`,`(+E'P``"@___^O(
M,)_E<"\=[@,PG^<#,)+G=E;#Y:#__^H,`)3EX/[_Z]+__^H(`)7E``!0XP$`
M``JM"`#KV?__ZI0`G^4&$*#A``"/X((``.L``%#C"`"%Y=+__PKU___J>#"?
MY7@`G^4``(_@`S"4YPPPC>4#$*#A-/__ZP00H..P___J`0"@XU?__^L:___K
MB(@!`!P!``!HB`$`_````!0!```D`0``$(D!`$R)`0!,B0$``#@``.@W``#@
MB`$`A`D``+B(`0`4-0``C(@!`/2&`0!`B`$`M(<!```T```H`0``W#4```"P
MH.,`X*#C!!"=Y`T@H.$$("WE!``MY1R@G^48,(_B`Z"*X``PH.,$,"WE#`"?
MY0``FN<I___KC/[_ZYR%`0`P`0``%#"?Y10@G^4#,(_@`B"3YP``4N,>_R\!
MDO[_ZH2%`0```0``+`"?Y2PPG^4``(_@*""?Y0,PC^```%/A`B"/X![_+P$8
M,)_E`S"2YP``4^,>_R\!$_\OX=R&`0#4A@$`4(4!``0!```X`)_E.#"?Y0``
MC^`T()_E`S"/X``P0^`"((_@HQ^@X4,1@>#!$+#A'O\O`1@PG^4#,)+G``!3
MXQ[_+P$3_R_AF(8!`)"&`0`,A0$`"`$``%@PG^58()_E`S"/X`(@C^``,-/E
M``!3XQ[_+Q$00"WI`<J@XPS`3>#X#XSE-#"?Y0,PDN<``%/C`@``"B@PG^4#
M`)_G]?[_Z\?__^L<,)_E`2"@XP,PC^``(,/E$("]Z$B&`0#(A`$`+`$``,R%
M`0#\A0$`S?__ZO!'+>D!RJ#C#,!-X.`/C.7(,)_E`(!0X@,PC^`K```*O""?
MY0(PD^<`D)/E``!9XR@```H`4-CE`7"@X0``5>,]`%43`5"@$P!0H`,1```:
M`$"9Y0``5.,,```*"6"@X00`H.$%(*#A"!"@X;+^_^L``%#C`@``&@4PU.<]
M`%/C"P``"@1`MN4``%3C\___&@0`H.'PA[WH"%"@X0$P]>4``%/C/0!3$_O_
M_QH(4$7@YO__Z@E@1N`!4(7B!4"$X$9AH.$$`*#A`&"'Y?"'O>@(0*#A[?__
MZ@E`H.'K___J1(0!`!@!```00"WI`<J@XPS`3>"(#XSEC""?Y7#03>*(,)_E
M#1"@X0(@C^``0*#A`S"2YP`PD^5L,(WE`#"@XVK^_^L``%#C``"@$P0``!H0
M,)WE#SH#X@H*4^,!"5,3"@``"D@@G^5`,)_E`B"/X`,PDN<`()/E;#"=Y0(@
M,^``,*#C!P``&G#0C>(0@+WH`A"@XP0`H.$/_O_K$`]OX:`"H.'N___J1/[_
MZU2#`0`<`0``$(,!`/!`+>D!RJ#C#,!-X'`/C.5<(9_E?-!-XE@QG^4`4%#B
M`B"/X`,PDN<`,)/E=#"-Y0`PH.,"```*`##5Y0``4^,*```:,"&?Y2@QG^4"
M((_@`S"2YP`@D^5T,)WE`B`SX``PH.-!```:?-"-XO"`O>B^_?_K`$!0XO'_
M_PH$`*#A0?[_ZP``4.,J```*$S#0Y1-@@.(N`%/C`@``&@$PUN4``%/C]/__
M"A,PT.4N`%/C)0``"@4`H.'H_?_K`'"@X08`H.'E_?_K``"'X`(`@.()_O_K
MI#"?Y0!PH.$!$*#C`S"/X``@X.,$8(WE`%"-Y2C^_^L($(WB!P"@X<O]_^L!
M`'#C!P"@X0,```H8,)WE#SH#X@$)4^,2```*TOW_ZP<`H.&X_?_K!`"@X1;^
M_^L``%#CU/__&@0`H.'B_?_K!0"@X?C]_^N]___J`3#6Y2X`4^/6__\:`C#6
MY0``4^/%__\*TO__ZJ3__^OK___JY/W_ZZ2"`0`<`0``>((!`!0O``#P02WI
M`<J@XPS`3>```(SE(``,Y0%PH.$`$9_E`#&?Y0':3>(!$(_@"-!-X@!@H.$!
M"HWB!`"`X@2`C>(#,)'G_R\`XP@0H.$`,)/E`#"`Y0`PH./,`)_E``"/X.;]
M_^L`0%#B!P``V@A0C>(!,$3B`\"%X`0@7.4``%+C%P``&@``4^,4```:G#"?
MY7`/'>X'(*#A!A"@X0,PG^<#`)#G;/W_ZX0@G^4!&HWB<#"?Y000@>("((_@
M`S"2YP`@D^4`,)'E`B`SX``PH.,2```:`=J-X@C0C>+P@;WH`T"@X00@H.$O
M$*#C"`"@X=K]_^L``%#CX___"C0`G^5PSQWN!#"@X0@@H.$&$*#A``"?YP``
MG.=\_?_KX?__ZIC]_^L<@0$`'`$``"`N``"T@0$`C(`!`$"!`0#P1RWI`<J@
MXPS`3>#8#XSE%"*?Y0C03>(0,I_E`4"@X0(@C^``4*#A!(*?Y0,PDN<(@(_@
M`#"3Y00PC>4`,*#C`##1Y3T`4^,!0($"!`"@X5/]_^L-$*#A`&"@X04`H.'1
M_O_K`)!0XD````I,_?_K``!6X3<``)JX,9_E`'"=Y0.`F.<'<:#A`##5Y0``
M4^,]`%,3`3"@$P`PH`,%```*!3"@X0$@\^4``%+C/0!2$_O__QH%,$/@`@"&
MX@!@F.4#`(#@7?W_ZP``4.,'`(;G%0``"@`PF.4%`*#A!Q"3YP$PT.0`,,'E
M``!3XST`4Q,&```*`2"!X@$PT.0"$*#A`3#"Y#T`4^,``%,3^?__&@$@H.$]
M,*#C`##!Y0$PU.0!,.+E``!3X_O__QH((9_E^#"?Y0(@C^`#,)+G`""3Y00P
MG>4"(#/@`#"@XS4``!H(T(WB\(>]Z`&02>(!,-3D`3#IY0``4^/[__\:[O__
MZL`PG^4#@)CG`'"8Y0`PE^4``%/C)```"@<PH.$$(+/E"1"@X0&0B>(``%+C
M^O__&@,0@>(!$:#AD#"?Y0,PC^``()/E``!2XP\``!H!`*#A`2"@XP`@@^4<
M_?_K`*!0XM7__PH'$*#A"7&@X0<@H.$H_?_K`*"(Y0"0C>4!D(GB`#"@XPDQ
MBN>B___J!P"@X3[]_^L`H*#A``!0XP``B.7%__\*"7&@X?+__^H#D*#A"!"@
MX]___^H'_?_KZ'\!`!P!``#8?P$`&`$``-1^`0#<?P$`$$`MZ0'*H.,,P$W@
MZ`^,Y7`0G^5P3QWN;""?Y1#03>)H`)_E`B"/X&0PG^4!$)_G`S"/X``@DN<!
M`)3G5!"?Y0P@C>4!$(_@&/W_ZT@0G^5(,)_E`,"@XT0@G^4#,(_@`1"?YP(@
MC^`!`)3G-!"?Y03`C>4!$(_@`#"-Y;'\_^L0T(WB$("]Z*1^`0"@?0$`$`$`
M`(!K``#(*@``<'X!`+`M``#`%@``M"H``#@`G^4(T$WB-""?Y7#/'>XP$)_E
M`B"/X"PPG^4``)_G`S"/X`$@DN<@$)_E``"<YP$0C^`$((WE"-"-XO#\_^H(
M?@$`!'T!`!`!``#D:@``,"H``'!`+>D!RJ#C#,!-X```C.4!RDSB``",Y2@`
M#.7<()_E`MI-XM@PG^48T$WB`B"/X`(:C>(4$('B#$"-X@%JC>(#,)+G$&"&
MX@`PD^4`,('E`#"@X\/\_^NH$)_E`2`!XZ0PG^4!$(_@\`#-X0,PC^`!$*#C
M!`"@X<;\_^L$`*#A`2J@XP80H.&\_/_K`$!0XA@``+H!`(3BE_S_ZP!04.(4
M```*!""@X080H.%Q_/_K`#"@XP0PQ>=0()_E`AJ-XCPPG^44$('B`B"/X`,P
MDN<`()/E`#"1Y0(@,^``,*#C`P``&@4`H.$"VHWB&-"-XG"`O>B#_/_K`%"@
MX^W__^J8?`$`'`$``.@I``#4*0``_'L!`+G__^KP3RWI`<J@XPS`3>```(SE
MJ``,Y40BG^5"W4WB0#*?Y0303>("((_@`&"@X30"G^5!S8WB$("-XCS`C.(#
M,)+G`4"@X0``C^`($*#A`#"3Y0`PC.4`,*#CO/W_ZR\0H.,`4*#A!@"@X0G\
M_^L``%#C$@``"O`!G^4&$*#A``"/X,?^_^L&`*#AX"&?Y4$=C>+,,9_E/!"!
MX@(@C^`#,)+G`""3Y0`PD>4"(#/@`#"@XVD``!I"W8WB!-"-XO"/O>@$`*#A
M/?S_ZZ01G^4!$(_@3/S_ZP!`4.+E__\*E)&?Y8"@C>(`,-3E!'!*X@F0C^``
M`%/C.P``"@``5>,$```*!!"@X04`H.%Q_/_K``!0XRL```H$`*#A"OS_ZP$P
M0.(#,(3@`+"@X0,`5.$`,(0P`!"@,P$`A#(#```Z"```Z@,`4.$`$,/E`@``
M"@$@<^4O`%+C^?__"@0`H.'Y^__K`+"@X08`H.'V^__K"P"`X`$`@.(!"E#C
MO?__*O@`G^4),*#A`2J@XP$0H.,``(_@`$"-Y4$`C>D'`*#A-_S_ZP@0H.$'
M`*#AVOO_ZP``4.,#```:8#`:Y0\Z`^(""5/C%0``"K00G^4``*#C`1"/X`S\
M_^L`0%#BI?__"@`PU.4``%/CP___&@``5>,6```*`##5Y2X`4^,#```*@$"?
MY0&PH.,$0(_@U?__Z@$PU>4``%/CZO__"O?__^H!$*#C!P"@X;C[_^L``%#C
MY/__&E``G^4'$*#A``"/X%C^_^L'`*#AW?O_ZX[__^HX0)_E`;"@XP1`C^#!
M___JX_O_ZXA[`0`<`0``^"@``,PH```4>P$`C"@``'`H``#8)P``C"<``#0G
M```0)P``Y"8``!!`+>D!RJ#C#,!-X/@/C.4O$*#C`$"@X8C[_^L``%#C`4"`
M$@0`H.$00+WHO_O_ZO!`+>D!RJ#C#,!-X```C.4@``SE>"&?Y0':3>)T,9_E
M#-!-X@(@C^`!&HWB!!"!X@!04.(#,)+G`#"3Y0`P@>4`,*#C,0``"@`PU>4`
M`%/C+@``"HO[_^L!((#B`$"@X0$*4N,Z``"*!'"-X@40H.$'`*#AB?O_ZP$P
M1.(#,(?@`P!7X0D``"H$`(?@!2"-X@`0H.,"``#J`@!0X0`0P.4"```*`3!P
MY2\`4^/Y__\*+Q"@XP<`H.%4^__K``!0XR4```H'`%#A`#"@@P`PP(4'`*#A
MB?O_Z[P@G^4!&HWBL#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,B
M```:`=J-X@S0C>+P@+WHB""?Y0$:C>)X,)_E!!"!X@(@C^`#,)+G`""3Y0`P
MD>4"(#/@`#"@XQ0``!I@`)_E``"/X`':C>(,T(WB\$"]Z&K[_^H``*#CWO__
MZD0@G^4!&HWB+#"?Y000@>("((_@`S"2YP`@D^4`,)'E`B`SX``PH.,@`)\%
M``"/`.S__PIF^__KR'@!`!P!```$>`$`S'<!`#@E``"`=P$`\"0``'!`+>D!
MRJ#C#,!-X.@/C.7$))_E"-!-XL`TG^4-0*#A`B"/X+ADG^6X5)_E!F"/X`,P
MDN<%4(_@`#"3Y00PC>4`,*#C!!"@X08`H.&C_/_K``!0XPX```J,-)_E``"=
MY0,PE><`$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C
M^___&NO__^I49)_E!F"/X`00H.$&`*#AC?S_ZP``4.,.```*-#2?Y0``G>4#
M,)7G`!"3Y0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O_
M_QKK___J`&2?Y09@C^`$$*#A!@"@X7?\_^L``%#C#@``"MPSG^4``)WE`S"5
MYP`0D^4`,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:
MZ___ZJQCG^4&8(_@!!"@X08`H.%A_/_K``!0XPX```J$,Y_E``"=Y0,PE><`
M$)/E`#&!X`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO_
M_^I88Y_E!F"/X`00H.$&`*#A2_S_ZP``4.,.```*+#.?Y0``G>4#,)7G`!"3
MY0`Q@>`$()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J
M!&.?Y09@C^`$$*#A!@"@X37\_^L``%#C#@``"M0RG^4``)WE`S"5YP`0D^4`
M,8'@!""3Y0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK!B
MG^4&8(_@!!"@X08`H.$?_/_K``!0XPX```I\,I_E``"=Y0,PE><`$)/E`#&!
MX`0@D^4`(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I<8I_E
M!F"/X`00H.$&`*#A"?S_ZP``4.,.```*)#*?Y0``G>4#,)7G`!"3Y0`Q@>`$
M()/E`"&!YP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J"&*?Y09@
MC^`$$*#A!@"@X?/[_^L``%#C#@``"LPQG^4``)WE`S"5YP`0D^4`,8'@!""3
MY0`A@><``%+C\?__"@0P@^($()/E!""#Y```4N/[__\:Z___ZK1AG^4&8(_@
M!!"@X08`H.'=^__K``!0XPX```IT,9_E``"=Y0,PE><`$)/E`#&!X`0@D^4`
M(8'G``!2X_'__PH$,(/B!""3Y00@@^0``%+C^___&NO__^I@89_E!F"/X`00
MH.$&`*#AQ_O_ZP``4.,.```*'#&?Y0``G>4#,)7G`!"3Y0`Q@>`$()/E`"&!
MYP``4N/Q__\*!#"#X@0@D^4$((/D``!2X_O__QKK___J#`&?Y000H.$``(_@
MLOO_ZP`04.("```*^`"?Y0``C^#"_/_K\`"?Y000H.$``(_@J?O_ZP`04.("
M```*W`"?Y0``C^"Y_/_KU`"?Y000H.$``(_@H/O_ZP`04.(2```*P`"?Y0``
MC^"P_/_KN""?Y60PG^4"((_@`S"2YP`@D^4$,)WE`B`SX``PH.,0```:F!"?
MY9@`G^4!$(_@``"/X`C0C>)P0+WHH/S_ZH0`G^4$$*#A``"/X(?[_^L`$%#B
MZ?__"G``G^4``(_@E_S_Z^7__^HH^O_K''<!`!P!``#,)```"'<!`!@!``!L
M)```'"0``-`C``"`(P``.",``'PB``"4(@``2"(``/PA```H(0``5"$``"@A
M``!$(0``1"$``$`A``!D(```Q'(!`(`B``"`(```["```'`@```$X"WE`<J@
MXPS`3>#P#XSE>""?Y0S03>)T,)_E#1"@X0(@C^!L`)_E``"/X`,PDN<`,)/E
M!#"-Y0`PH.-6^__K``!0XP4```H``-#E``!0XS``4!,!`*`#``"@$P$`(.(T
M()_E*#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@XP$``!H,T(WB!/"=Y.KY
M_^O8<0$`'`$``-`?``"0<0$`#P`MZ0'*H.,$X"WE#,!-X/`/C.5($)_E#-!-
MXD0@G^44,(WB`1"/X#P`G^4`,(WE.,"?Y0``C^`"()'G`1"@XP`@DN4$((WE
M`""@XQ`@G>4,P)#G``"<Y;?Y_^O_`*#C"?K_ZS1Q`0`<`0``)'$!`"`!``#P
M1RWI`,"@XP'HH.,,P$W@#N!,X`'*3.(``(SE#@!<X?O__QHX``[E`=A-XAC0
M3>)TP9_E`A"@XP'HC>(,P(_@%.".X@`0C>4`(*#C7!&?Y0`PH.,`@*#A`1"<
MYP`0D>4`$([E`!"@XZCY_^L!`%#B`'#!XL<OH.$","#@`C!3X',P_^8","/@
M`C!3X`(@PN`#8%#@`G#'X```5^,X``"Z")"@XP&00.,,4(WB`!"@XP8@H.$'
M,*#A"`"@X0`0C>62^?_K`0!QXP$`<`,R```*"2"@X040H.$(`*#A5?G_ZP$`
M<.,I```*!P!0XQ\``)H(`$#B`$"5X!P``"JPH)_E"J"/X`(``.H!0$3B!0!4
MX18``#H((*#C"A"@X00`H.$=^?_K``!0X_;__QH%0$3@!@"4X,0?I^!X()_E
M`<B-XF@PG^44P(SB`B"/X`,PDN<`()/E`#"<Y0(@,^``,*#C$```&@'8C>(8
MT(WB\(>]Z`%H5N(`<,?B`0!WXP$(=@/)__\:``#@XP`0H.'I___J)`"?Y0``
MC^![___K'`"?Y0``C^!X___K7OG_ZZAP`0`<`0``('$!`)QO`0#<'0``P!T`
M`!!`+>D!RJ#C#,!-X/@/C.41^?_K,?G_ZP``4.,``)`5$("]Z/!`+>D!RJ#C
M#,!-X-`/C.4X(9_E'-!-XC0QG^40$(WB`B"/X"QAG^4`4*#A!F"/X`,PDN<&
M`*#A`#"3Y10PC>4`,*#CE?K_ZP!`4.("```*`##4Y0``4^,.```:^""?Y>PP
MG^4"((_@`S"2YP`@D^44,)WE`B`SX``PH.,R```:V`"?Y040H.$``(_@'-"-
MXO!`O>B6^__J!1"@X1+Y_^L``%#C"@``"K0@G^6@,)_E`B"/X`,PDN<`()/E
M%#"=Y0(@,^``,*#C'P``&AS0C>+P@+WH!0"@X>[X_^L`<*#A!`"@X>OX_^L`
M`(?@`@"`X@_Y_^ML,)_E;""?Y0$0H.,#,(_@"$"-Y0(@C^``4(WE!""-Y0`@
MX.,`0*#A*_G_ZT@@G^4H,)_E`B"/X`,PDN<`()/E%#"=Y0(@,^``,*#C!!"@
M`08`H`'/__\*^_C_Z]QN`0`<`0``:!T``)AN`0`,'0``3&X!`*P;``"@&P``
MU&T!`/!/+>D!RJ#C#,!-X.@.C.54)I_E]-!-XE`VG^6(8(WB`B"/X$C&G^4`
MH*#AH$"-X@S`C^`$X*#A`S"2YQAPC.*44(WB++:?Y0`PD^7L,(WE`#"@XPPP
MC.(PP(SB(("-X@NPC^`'`)/H!P"&Z`\`M^@/`*[H`P"7Z`,`CN@'`)SH!P"%
MZ/`%G^4($*#A``"/X"7Z_^L`<%#B`@``"@`PU^4``%/C%P$`&H7X_^NE^/_K
M``!0XWH```H`<)#E``!7XW<```H'`*#AE?C_ZX``H.$!`(#BN?C_ZP`PU^4`
M4*#A'`"-Y0``4^,+```*C)6?Y0F0C^`%`*#A`#"-Y0$0H.,),*#A`"#@XP)0
MA>+2^/_K`3#WY0``4^/U__\:8`6?Y0``C^`"``#J!`"TY0``4.,-```*"!"@
MX?OY_^L`4%#B^/__"@`PU>4``%/C]?__"B_Z_^L``%#C\O__"@4`H.&-^/_K
M`$!0X@T``!H$0);D``!4XPH```H`,-3E``!3X\\```H$`*#A(?K_ZP``4./U
M__\*!`"@X7_X_^L`0%#B\?__"@0`H.%=^/_K'%"=Y0!@H.$%`*#A6?C_ZP!@
MAN`!:X;B!&"&X@8`H.%[^/_KL#2?Y0!PH.$!$*#C`S"/X`@PC>6@-)_E`"#@
MXPQ0C>4#,(_@!#"-Y9`TG^4`0(WE`S"/X)3X_^L'':#C!P"@X?[W_^L!`'#C
M`P``&JGX_^L`$)#E$0!1XZL``!H($*#A!P"@X7#X_^L`4%#B+0``"DP4G^4`
M,)KE2"2?Y0$`F^<"((_@`'"-Y0$0H.,``)#E7/C_ZP!@H.,L))_E_#.?Y0(@
MC^`#,)+G`""3Y>PPG>4"(#/@`#"@X_,``!H&`*#A]-"-XO"/O>@_^/_K`'!0
MXHD``!H$`)7D``!0XPD```H($*#AG/G_ZP`P4.+X__\*`##3Y0``4^/R__\:
M!`"5Y```4./U__\:P'.?Y0T`H.,V^/_K4S"@XP!0H.$'<(_@'`"-Y7S__^HP
M,)WE#SH#X@$)4^/-__\:.$"=Y>SW_^L``%3AR?__&C`PG>53,.CG!PU3X\7_
M_QH&`*#A(_C_ZP!@H.%H`Y_E"!"@X0``C^!Y^?_K`!!0XGP```H``)KEH?O_
MZP"04.)X```*"0"@X>_W_^L`0*#A`P!0XPP``)HP$Y_E!`!`X@``B>`!$(_@
M2OC_ZP``4.,%```:!`!4XZ(```H$0(G@!3!4Y2\`4^.>```*``.?Y0@0H.$`
M`(_@7?G_ZP``4.,#```*`##0Y3``4^,``%,3!```&@`0H.,)`*#AX??_ZP!`
M4.)9```:R!*?Y<@"G^4!$(_@Q**?Y0``C^#`DI_E8/K_Z[R"G^4)^/_KN#*?
MY0@`C>4!$*#C`S"/X`PPC>6H,I_E!@"@X0!PC>4`(.#C`S"/X`0PC>64,I_E
M"J"/X`F0C^`(@(_@`S"/X`3X_^L.``#J'?C_ZP`PD.41`%/C$```&O+W_^L!
M4(7B`,"@X0HPH.$$(*#A`1"@XP8`H.$0D(WE#%"-Y8`1C>CT]__K!QV@XP8`
MH.%>]__K`$"@X0$`<./J__\*!P"@X8GW_^L@`I_E!A"@X0``C^`Q^O_K!@"@
MX7+^_^MH___J`T"@X37__^J4]__K@`"@X0$`@.+]_O_JL"&?Y0`PFN4"`)OG
M!!"-Y0$0H.,`<(WEV"&?Y0``D.4"((_@M/?_ZU;__^I?_O_KQ"&?Y5@QG^4"
M((_@`S"2YP`@D^7L,)WE`B`SX``PH.-*```:!P"@X?30C>+P3[WHEO?_Z@"0
MFN6$___JU?W_ZP"`H.$`D%'BH?__N@H@4.(`H*#C`##)X@0`H.&XL(WB`*"-
MY8_W_^L&(*#C"Q"@X00`H.%5]__K!@!0XS0``!JX()WE0SR@XT$S1.,#`%+A
MC___&K0@V^%(-03C`P!2X8O__QHR(%CB!`"@X0`PR>+`4(WB`*"-Y7KW_^L$
M`*#A*""@XP40H.%`]__K*`!0XQ\``!KT,)_E`1"@X_`@G^4&`*#A`S"/X`0P
MC>7D,)_E`B"/X`A0C>4,((WE`S"/X`!PC>4`(.#CZ*#-Y8_W_^N?___J!$":
MY0``5.-=__\*!`"@X3OW_^L#`%#C6?__FJ00G^4$`$#B``"$X`$0C^"7]__K
M``!0XP20H`%1___J6O?_ZX0`G^4``(_@</W_ZVAM`0`<`0``E&X!`"AM`0",
M&@``2!L``+0:``!P&@``4!D``$`:```@`0``0!H``!AK`0!$&0``J!D``'`9
M``#<%P``@!D``)@7``#T&```F!D```07``#`&0``&!<``-@8``!H%@``6!<`
M`'1H`0!0%0``U!<````7``"L%@``F!8``/!`+>D!RJ#C#,!-X.`/C.5<$I_E
M#-!-XE@BG^5P3QWN`1"/X%`RG^50`I_E#6"@X0(@D><``(_@!A"@X0`@DN4$
M((WE`""@XP,PG^<#4)3G9##5Y64PQ>5W^/_K``!0XP(```H`,-#E,0!3XV8`
M``JY^__K"#*?Y0,PG^<#4)3G`!*?Y04`H.$#,*#C`2"@XP$0C^"_]O_K`%!0
MX@P```KD,9_E#""5Y0,PG^<`$)+EV"&?Y0,`E.<"((_@N_;_ZPPPE>4`$)/E
M"#"1Y0$%$^-9```:N!&?Y0,PH.,!(*#C`1"?YP$`E.>H$9_E`1"/X*CV_^L`
M4%#B"P``"I@!G^4&$*#A``"/X$OX_^L`(%#B00``"H0QG^4,$)7E`S"?YP`0
MD>4#`)3GH/;_ZW`QG^4!(*#C;!&?Y6QQG^4!$(_@`S"?YP=PC^`#4)3G!0"@
MX8'V_^M4(9_E`!"@X04`H.$"((_@D?;_ZT0QG^5$`9_E`""@XP80H.$``(_@
M`S"?YP,PE.=E(,/E*_C_ZP!`H.$``%#C"`"'Y1(```H'':#C!`"@X5WV_^L!
M((#B`#"@X1$`4.,!`%(3)0``BO@@G^6P,)_E`B"/X`,PDN<`()/E!#"=Y0(@
M,^``,*#C&P``&@S0C>+P@+WH!`"7Y<?]_^L`0*#A``!0XP@`A^7F__\:N`"?
MY0``C^#(]O_K`3#0Y0``4^.9__\*E/__ZJ`PG^4,()7E`S"/X```DN4$,)/E
M`!"3Y<?X_^N[___JA#"?Y0,PG^<#`)3GFO;_ZZ#__^JA]O_K<!"?Y00@H.%L
M`)_E`1"/X```C^`$$)'E`!"1Y:[V_^MX9@$`'`$``%QG`0!(%```+&<!`+P5
M``#\9@$`F!4``,1F`0!L%0``,!,``(AF`0!D9@$`(!4``-1F`0`4%0``*&8!
M`+`2``#49`$`>!0``/1E`0!D90$`N&4!`#P4```P0"WI`<J@XPS`3>#H#XSE
M#"&?Y0!`H.$(,9_E#-!-X@(@C^```9_E#1"@X0``C^`#,)+G`#"3Y00PC>4`
M,*#CR/?_ZP`P4.(*```*`##3Y3``4^,``%,3`3"@`P`PH!,!,"/B``!4XP`P
MH`,!,`,2``!3XPH``!JP()_EI#"?Y0(@C^`#,)+G`""3Y00PG>4"(#/@`#"@
MXR```!H,T(WB,("]Z``PU.4``%/C\?__"@0`H.&'^O_K+Q"@XP!0H.$(]O_K
M``!0XP%0@!(%`*#A0/;_ZU@0G^4$(*#C`1"/X&_V_^L``%#CXO__&D0@G^4P
M,)_E`B"/X`,PDN<`()/E!#"=Y0(@,^``,*#C`P``&@0`H.$,T(WB,$"]Z/?W
M_^HX]O_KI&,!`!P!``"8$0``1&,!`*P1``#08@$``````/!/+>D!RJ#C#,!-
MX(`.C.403)_E5]]-X@SLG^40,(WB!$"/X`,0H.$<((#B7,"`X@[@E.<`X)[E
M5.&-Y0#@H..-"F+TX`#P\P(`7.'="D'T^O__&@9+W>T(.]WM$1Z-X@H;W>T`
M``#JLA%B\@@KT^TT((/B``O3[0@P@^(!`%/ACU=B]+`!0O.U`4#SM`%`\[-!
M8_*Q,6'RH!A@\C``X?.P`6'R#@O#[>[__QH@0)#IF3D'XX(Z1>,,0)#E$,"=
MY0!@D.4$$"[@$'"0Y0/`C.`%$`'@99&@X0?`C.`$$"'@%""=Y>;-C.`!P(S@
M"1`NX`,@@N`&$`'@9H&@X00@@N`8H)WE#A`AX`(0@>`(("G@`Z"*X`P@`N`.
MH(K@"2`BX`H@@N`<H)WE[!V!X&S!H.$#H(K@":"*X`R0*.`!D`G@X2V"X`B0
M*>!A$:#A"I")X""@G>7BG8G@`Z"*X`B@BN`!@"S@`H`(X&(AH.$,@"C@`K`A
MX`J`B.`DH)WE";`+X.F-B.`#H(K@#*"*X&G!H.$!D"O@#+`BX`J0B>`HH)WE
M"+`+X.B=B>`#H(K@`:"*X&@1H.$"@"O@`;`LX`J`B.`LH)WE";`+X.F-B.`#
MH(K@`J"*X&DAH.$,D"O@`K`AX`J0B>`PH)WE"+`+X.B=B>`#H(K@#*"*X&C!
MH.$!@"O@#+`BX`J`B.`TH)WE";`+X.F-B.`#H(K@`:"*X&D1H.$"D"O@`;`L
MX`J0B>`XH)WE"+`+X.B=B>`#H(K@`J"*X&@AH.$,@"O@`K`AX`J`B.`\H)WE
M";`+X.F-B.`#H(K@#*"*X&G!H.$!D"O@#+`BX`J0B>!`H)WE"+`+X.B=B>`#
MH(K@`:"*X&@1H.$"@"O@`;`LX`J`B.!$H)WE";`+X.F-B.`#H(K@`J"*X&DA
MH.$,D"O@`K`AX`J0B>!(H)WE"+`+X.B=B>`#H(K@#*"*X&C!H.$!@"O@#+`B
MX`J`B.!,H)WE";`+X.F-B.`#H(K@`:"*X&D1H.$"D"O@`;`LX`J0B>!0H)WE
M"+`+X.B=B>`#H(K@`J"*X&@AH.$,@"O@"H"(X%2@G>7IC8C@`Z"*X`S`BN`"
MH"'@":`*X&F1H.$!H"K@#*"*X%C`G>7HK8K@`\",X`'`C.`)$"+@"!`!X&B!
MH.$"$"'@#!"!X%S`G>7J'8'@`S",X`C`*>`*P`S@`C"#X`G`+.!@()WE`\",
MX*$[#N/9/D;C:J&@X0,@@N#AS8S@"9""X&0@G>4#((+@""""X`J`*.`!@"C@
M"(")X&&1H.%H$)WE[(V(X`,0@>`*$('@":`JX`R@*N!LP:#A"B""X.BM@N!L
M()WE`R""X`D@@N`,D"G@")`IX&B!H.$)$('@ZIV!X'`0G>4#$('@#!"!X`C`
M+.`*P"S@:J&@X0P@@N#IS8+@=""=Y0,@@N`(((+@"H`HX`F`*.!ID:#A"!"!
MX.R-@>!X$)WE`Q"!X`H0@>`)H"K@#*`JX&S!H.$*((+@Z*V"X'P@G>4#((+@
M"2""X`R0*>`(D"G@:(&@X0D0@>#JG8'@@!"=Y0,0@>`,$('@",`LX`K`+.!J
MH:#A#"""X.G-@N"$()WE`R""X`@@@N`*@"C@"8`HX&F1H.$($('@[(V!X(@0
MG>4#$('@"A"!X`F@*N`,H"K@;,&@X0H@@N#HK8+@C""=Y0,@@N!JL:#A"2""
MX`R0*>`(D"G@:(&@X0D0@>#JG8'@D!"=Y0,0@>`,$('@",`LX`K`+.`,((+@
MZ:V"X)0@G>4#((+@""""X`N`*.`)P"C@:9&@X0P0@>"8P)WEZHV!X`D0*^`#
MP(S@"A`AX`O`C."<L)WE:J&@X0$@@N`#L(O@Z!V"X`FPB^`*D"G@""`IX&B!
MH.$"P(S@H""=Y>&=C.`#((+@"B""X`B@*N`!P"K@I*"=Y6$1H.$,P(O@`Z"*
MX.G-C.`(H(K@`8`HX`F`*.!ID:#A""""X+"PG>7LC8+@J""=Y0,@@N`!((+@
M"1`AX`P0(>!LP:#A`:"*X`P0*>#HK8K@"!`AX`$@@N!H@:#AZAV"X*P@G>4#
M((+@"#`LX`HP(^`)((+@`R""X-P\"^,;/TCC:I&@X0.PB^`)H(CA#+"+X+3`
MG>7A+8+@`:`*X`/`C.!A$:#A",",X`F`".`(H(KA`8")X0N@BN`"@`C@`;`)
MX.*MBN`+@(CAO+"=Y0R`B."XP)WE8B&@X0.PB^`#P(S@`;"+X`F0C.`"P('A
M"L`,X`(0`>`!P(SAZHV(X`G`C.#`D)WE:J&@X>C-C.`*$(+A`Y")X`*0B>`(
M$`'@"B`"X&B!H.$"$('A""`*X`NP@>`($(KA#!`!X.R]B^`"$('A;,&@X0D0
M@>#$D)WE#""(X>L=@>`#D(G@"R`"X`J0B>#(H)WE:[&@X0.@BN`(H(K@#(`(
MX`@@@N%A@:#A"9""X`L@C.'AG8G@`2`"X,P0G>4#$('@#!"!X`O`#.`,P(+A
M"""+X0K`C.`)(`+@"*`+X.G-C.`*((+AT*"=Y6F1H.$!$(+@`Z"*X`D@B.$+
MH(K@U+"=Y0P@`N#L'8'@`["+X&S!H.$(L(O@"8`(X`@@@N%A@:#A"J""X`P@
MB>'AK8K@`2`"X-@0G>4#$('@"1"!X`R0">`)D(+A""",X0N0B>`*(`+@"+`,
MX.J=B>`+((+AX+"=Y0$0@N#<()WE:J&@X0.PB^`#((+@"+"+X`S`@N`*((CA
M"2`"X`J`".#I'8'@"(""X6F1H.$,@(C@"2"*X6'!H.'AC8C@`2`"X.00G>4#
M$('@"A"!X`F@"N`*H(+A#"")X0N@BN`((`+@#+`)X.BMBN`+((+A[+"=Y0$0
M@N#H()WE:(&@X0.PB^`#((+@#+"+X`F0@N`(((SA"B`"X`C`#.#J'8'@#"""
MX6JAH.$)D(+@"B"(X6'!H.'AG8G@`2`"X/`0G>4#$('@"!"!X`J`".`(@(+A
M#""*X0N`B.`)(`+@#+`*X.F-B.`+((+A^+"=Y0$@@N#T$)WE:9&@X0.PB^`#
M$('@#+"+X`J@@>`)$(SA"!`!X`G`#.`,P('AZ"V"X`K`C.#\H)WE:(&@X>+-
MC.`($(GA`S"*X`DP@^`"$`'@")`)X&(AH.$"H(CA"1"!X0L0@>`,H`K@`K`(
MX.P=@>!LD:#A"Z"*X0#!G>4#H(K@UC$,XV(Z3.,#P(S@X:V*X`B`C.`$P9WE
M8;&@X0/`C.`"P(S@"2`BX`$@(N`($9WE`H"(X`L@*>#JC8C@"B`BX`,0@>!J
MH:#A"1"!X`+`C.`*D"O@Z,V,X`B0*>`,(9WE"1"!X&B!H.'LG8'@$!&=Y0,@
M@N`#$('@"R""X`H0@>`(H"K@#*`JX&S!H.$*((+@Z:V"X!0AG>4#((+@""""
MX`R`*.`)@"C@:9&@X0@0@>#JC8'@&!&=Y0,0@>`,$('@"<`LX`K`+.!JH:#A
M#"""X.C-@N`<(9WE`R""X`D@@N`*D"G@")`IX&B!H.$)$('@[)V!X"`1G>4#
M$('@"A"!X`B@*N`,H"K@;,&@X0H@@N#IK8+@)"&=Y0,@@N`(((+@#(`HX`F`
M*.!ID:#A"!"!X.J-@>`H$9WE`Q"!X`P0@>`)P"S@"L`LX&JAH.$,((+@Z,V"
MX"PAG>4#((+@"2""X`J0*>`(D"G@:(&@X0D0@>#LG8'@,!&=Y0,0@>`*$('@
M"*`JX`R@*N!LP:#A"B""X.FM@N`T(9WE`R""X`@@@N`,@"C@"8`HX&F1H.$(
M$('@ZHV!X#@1G>4#$('@#!"!X`G`+.`*P"S@:J&@X0P@@N#HS8+@/"&=Y0,@
M@N`)((+@"I`IX`B0*>!H@:#A"1"!X$"1G>7L'8'@`Y")X`J0B>`(H"K@#*`J
MX&S!H.$*((+@1*&=Y>$M@N`#H(K@"*"*X`R`*.`!@"C@81&@X0B0B>!(@9WE
MXIV)X`.`B.`,@(C@`<`LX`+`+.!B(:#A#,"*X`)PA^#IS8S@$'"`Y6SACN`(
MX(WE3.&=Y0,PCN`&,(/@`3"#X`(0(>`)$"'@:9&@X0$0B.`)("+@[!V!X`P@
M(N`",(/@"4"$X.$]@^`,0(WE`#"-Y0%0A>`\()_E!%"-Y=\*;?0"((_@*#"?
MY8\*0/0#,)+G`""3Y50QG>4"(#/@`#"@XP$``!I7WXWB\(^]Z"?S_^MD8@$`
M'`$``(Q6`0`00"WI`<J@XPS`3>#X#XSE8`"@XQOS_^L'"]_M""O?[0D[W^T`
M(*#C!`O`[1@@@.5<((#EWRI`]!"`O>CPX=+#``````$C16>)J\WO_MRZF'94
M,A#P1RWI`<J@XPS`3>#@#XSE%#"0Y0)@H.&"(:#A`%"@X0(PD^`8()#E%#"`
MY0%`H.%<`)#E`2""(J8N@N`<@(7B``!0XQ@@A>4@```:/P!6XRX``-I`D$;B
M*6.@X0%PAN('<X3@!!"@X0@PH.%`0(3B`."1Y1`0@>(,P!'E$#"#X@@`$>4$
M(!'E!`!1X1#@`^4,P`/E"``#Y00@`^7S__\:`4"@X04`H.&R_/_K!P!4X>O_
M_QH&8TG@!B"@X0<0H.$(`*#AN?+_ZUQ@A>7PA[WH0'!@X@``B.`&`%?A!G"@
MH0<@H.&Q\O_K7#"5Y0,PA^!<,(7E0`!3X_"'O1@%`*#A!V!&X`=`A.":_/_K
MSO__Z@1PH.'G___J<$`MZ0'*H.,"BRWM#,!-X.@/C.4!0*#A%!"1Y0!0H.$<
M8(3B,3^_YA00E.48`)3ET1'EYS`OO^88*T/L`2"!XG\PX.,X`%+C`@"&X`$P
MQN=!``#:0"!BX@`0H.-C\O_K!`"@X7W\_^LL,(3B4`#`\@\*1O0/"D/T/#"$
MX@\*0_1,,(3B#P=#]%0PA.($`*#A#X<#]''\_^L#,-3E`##%Y00`H.&R,-3A
M`3#%Y0`PE.4C-*#A`C#%Y0`PE.4#,,7E!S#4Y00PQ>6V,-3A!3#%Y00PE.4C
M-*#A!C#%Y00PE.4',,7E"S#4Y0@PQ>6Z,-3A"3#%Y0@PE.4C-*#A"C#%Y0@P
ME.4+,,7E#S#4Y0PPQ>6^,-3A#3#%Y0PPE.4C-*#A#C#%Y0PPE.4/,,7E$S#4
MY1`PQ>6R,=3A$3#%Y1`PE.4C-*#A$C#%Y1`PE.43,,7E`HN]['!`O>@S\O_J
M."!BX@`0H.,A\O_KQO__Z@@PG^4#,(_@`#"3Y7SR_^JL5`$`"$`MZ0B`O>@!
M``(`/0```$`````+````0````!````!`````)@```$`````U````0`````,`
M``!`````!@```$`````4````0`````X```!`````#0```$`````E````0```
M`!(```!`````*````$`````"````0````!@```!`````"0```$`````*````
M0```````````````)7,O)7,````O<')O8R]S96QF+V5X90``1'EN84QO861E
M<CHZ8F]O=%]$>6YA3&]A9&5R`$EN=&5R;F%L<SHZ4$%2.CI"3T]4`````"]P
M<F]C+R5I+R5S`&5X90`N````4$%27U1%35``````4$%27U!23T=.04U%````
M`#H````E<R5S)7,``"\```!015),-4Q)0@````!015),3$E"`%!%4DPU3U!4
M`````%!%4DQ)3P``4$%27TE.251)04Q)6D5$`%!!4E]34$%73D5$`%!!4E]#
M3$5!3@```%!!4E]$14)51P```%!!4E]#04-(10```%!!4E]'3$]"04Q?1$5"
M54<`````4$%27T=,3T)!3%]435!$25(```!005)?5$U01$E2``!005)?1TQ/
M0D%,7U1%35``4$%27T=,3T)!3%]#3$5!3@````!L<V5E:R!F86EL960`````
M<F5A9"!F86EL960`3$1?3$E"4D%265]0051(`%-94U1%30``)3`R>``````E
M<R5S)7,E<P````!P87(M`````"5S.B!C<F5A=&EO;B!O9B!P<FEV871E('-U
M8F1I<F5C=&]R>2`E<R!F86EL960@*&5R<FYO/25I*0H``"5S.B!P<FEV871E
M('-U8F1I<F5C=&]R>2`E<R!I<R!U;G-A9F4@*'!L96%S92!R96UO=F4@:70@
M86YD(')E=')Y('EO=7(@;W!E<F%T:6]N*0H``%!!5$@`````<&%R;``````N
M<&%R`````'-H;W)T(')E860``"5S)7-C86-H92TE<R5S```E<R5S=&5M<"TE
M=25S````)7,E<W1E;7`M)74M)74E<P`````8````<&5R;``````P````4$%2
M.CI086-K97(Z.E9%4E-)3TX`````,2XP-C$```!5;F%B;&4@=&\@8W)E871E
M(&-A8VAE(&1I<F5C=&]R>0`````E<SH@8W)E871I;VX@;V8@<')I=F%T92!C
M86-H92!S=6)D:7)E8W1O<GD@)7,@9F%I;&5D("AE<G)N;STE:2D*`````"UE
M```M+0``55-%4@````!54T523D%-10````!414U01$E2`%1-4``O=&UP````
M`&5V86P@)V5X96,@+W5S<B]B:6XO<&5R;"`@+5,@)#`@)'LQ*R(D0")])PII
M9B`P.R`C(&YO="!R=6YN:6YG('5N9&5R('-O;64@<VAE;&P*<&%C:V%G92!?
M7W!A<E]P;#L*;7D@*"1005)?34%'24,L("1&24Q%7V]F9G-E=%]S:7IE+"`D
M8V%C:&5?;F%M95]S:7IE*3L*0D5'24X@>PHD4$%27TU!1TE#(#T@(EQN4$%2
M+G!M7&XB.PHD1DE,15]O9F9S971?<VEZ92`](#0[("`@(R!P86-K*").(BD*
M)&-A8VAE7VYA;65?<VEZ92`](#0P.PHD4$M:25!?34%'24,@/2`B4$M<,#`S
M7#`P-"(["GT*<W5B(&9I;F1?<&%R7VUA9VEC"GL*;7D@*"1F:"D@/2!`7SL*
M;7D@)&-H=6YK7W-I>F4@/2`V-"`J(#$P,C0["FUY("1B=68["FUY("1S:7IE
M(#T@+7,@)&9H.PIM>2`D<&]S(#T@*"1S:7IE+3$I("T@*"1S:7IE+3$I("4@
M)&-H=6YK7W-I>F4[("`@("`@(R!.3U1%.B`D<&]S(&ES(&$@;75L=&EP;&4@
M;V8@)&-H=6YK7W-I>F4*=VAI;&4@*"1P;W,@/CT@,"D@>PIS965K("1F:"P@
M)'!O<RP@,#L*<F5A9"`D9F@L("1B=68L("1C:'5N:U]S:7IE("L@;&5N9W1H
M*"1005)?34%'24,I.PII9B`H*&UY("1I(#T@<FEN9&5X*"1B=68L("1005)?
M34%'24,I*2`^/2`P*2!["G)E='5R;B`D<&]S("L@)&D["GT*)'!O<R`M/2`D
M8VAU;FM?<VEZ93L*?0IR971U<FX@+3$["GT*;7D@*"1P87)?=&5M<"P@)'!R
M;V=N86UE+"!`=&UP9FEL92P@)4UO9'5L94-A8VAE*3L*14Y$('L@:68@*"1%
M3E9[4$%27T-,14%.?2D@>PIR97%U:7)E($9I;&4Z.E1E;7`["G)E<75I<F4@
M1FEL93HZ0F%S96YA;64["G)E<75I<F4@1FEL93HZ4W!E8SL*;7D@)'1O<&1I
M<B`]($9I;&4Z.D)A<V5N86UE.CID:7)N86UE*"1P87)?=&5M<"D["F]U=',H
M<7%;4F5M;W9I;F<@9FEL97,@:6X@(B1P87)?=&5M<")=*3L*1FEL93HZ1FEN
M9#HZ9FEN9&1E<'1H*'-U8B![("@@+60@*2`_(')M9&ER(#H@=6YL:6YK('TL
M("1P87)?=&5M<"D["G)M9&ER("1P87)?=&5M<#L*:68@*"UD("1P87)?=&5M
M<"`F)B`D7D\@;F4@)TU35VEN,S(G*2!["FUY("1T;7`@/2!N97<@1FEL93HZ
M5&5M<"@*5$5-4$Q!5$4@/3X@)W1M<%A86%A8)RP*1$E2(#T^($9I;&4Z.D)A
M<V5N86UE.CID:7)N86UE*"1T;W!D:7(I+`I3549&25@@/3X@)RYC;60G+`I5
M3DQ)3DL@/3X@,"P**3L*;7D@)&9I;&5N86UE(#T@)'1M<"T^9FEL96YA;64[
M"G!R:6YT("1T;7`@/#PB+BXN(CL*>#TQ.R!W:&EL92!;(%PD>"`M;'0@,3`@
M73L@9&\*<FT@+7)F("<D<&%R7W1E;7`G"FEF(%L@7"$@+60@)R1P87)?=&5M
M<"<@73L@=&AE;@IB<F5A:PIF:0IS;&5E<"`Q"G@]8&5X<'(@7"1X("L@,6`*
M9&]N90IR;2`G)&9I;&5N86UE)PHN+BX*8VQO<V4@)'1M<#L*8VAM;V0@,#<P
M,"P@)&9I;&5N86UE.PIM>2`D8VUD(#T@(B1F:6QE;F%M92`^+V1E=B]N=6QL
M(#(^)C$@)B(["G-Y<W1E;2@D8VUD*3L*;W5T<RAQ<5M3<&%W;F5D(&)A8VMG
M<F]U;F0@<')O8V5S<R!T;R!P97)F;W)M(&-L96%N=7`Z("1F:6QE;F%M95TI
M.PI]"GT@?0I"14=)3B!["DEN=&5R;F%L<SHZ4$%2.CI"3T]4*"D@:68@9&5F
M:6YE9"`F26YT97)N86QS.CI005(Z.D)/3U0["F5V86P@>PI?<&%R7VEN:71?
M96YV*"D["FUY("1Q=6EE="`]("$D14Y6>U!!4E]$14)51WT["FUY("5S>7,@
M/2`H"G!A=&A?<V5P("`@(#T^("@D7D\@/7X@+UY-4U=I;B\@/R`G.R<@.B`G
M.B<I+`I?97AE("`@("`@("`]/B`H)%Y/(#U^("]>*#\Z35-7:6Y\3U,R?&-Y
M9W=I;BDO(#\@)RYE>&4G(#H@)R<I+`I?9&5L:6T@("`@("`]/B`H)%Y/(#U^
M("]>35-7:6Y\3U,R+R`_("=<7"<@.B`G+R<I+`HI.PI?<V5T7W!R;V=N86UE
M*"D["F]U=',H<7%;7"1P<F]G;F%M92`]("(D<')O9VYA;64B72D["E]S971?
M<&%R7W1E;7`H*3L*;W5T<RAQ<5M<)$5.5GM005)?5$5-4'T@/2`B)$5.5GM0
M05)?5$5-4'TB72D["FUY("@D<W1A<G1?<&]S+"`D<W1A<G1?;V9?1DE,15]S
M96-T:6]N*3L*34%'24,Z('L*;&]C86P@)%-)1WM?7U=!4DY?7WT@/2!S=6(@
M>WT["G5N;&5S<R`H;W!E;B!?1D@L("<\.G)A=R<L("1P<F]G;F%M92D@>PIO
M=71S*'%Q6T-A;B=T(')E860@9G)O;2!F:6QE("(D<')O9VYA;64B72D[("`C
M(&1O;B=T('5S92`D(2!H97)E(&%S(&ET(')E<75I<F5S($5R<FYO+G!M"FQA
M<W0@34%'24,["GT*;7D@)&UA9VEC7W!O<R`](&9I;F1?<&%R7VUA9VEC*"I?
M1D@I.PII9B`H)&UA9VEC7W!O<R`\(#`I('L*;W5T<RAQ<5M#86XG="!F:6YD
M(&UA9VEC('-T<FEN9R`B)%!!4E]-04=)0R(@:6X@9FEL92`B)'!R;V=N86UE
M(ETI.PIL87-T($U!1TE#.PI]"F]U=',H(D9O=6YD(%!!4B!M86=I8R!A="!P
M;W-I=&EO;B`D;6%G:6-?<&]S(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@
M)$9)3$5?;V9F<V5T7W-I>F4@+2!L96YG=&@H(EPP0T%#2$4B*2P@,#L*<F5A
M9"!?1D@L("1B=68L(&QE;F=T:"@B7#!#04-(12(I.PII9B`H)&)U9B!N92`B
M7#!#04-(12(@)B8@)&)U9B!N92`B7#!#3$5!3B(I('L*;W5T<R@B3F\@8V%C
M:&4@;6%R:V5R(&9O=6YD(BD["FQA<W0@34%'24,["GT*;W5T<RAQ<5M#86-H
M92!M87)K97(@(B1B=68B(&9O=6YD72D["G-E96L@7T9(+"`D;6%G:6-?<&]S
M("T@)$9)3$5?;V9F<V5T7W-I>F4L(#`["G)E860@7T9(+"`D8G5F+"`D1DE,
M15]O9F9S971?<VEZ93L*;7D@)&]F9G-E="`]('5N<&%C:R@B3B(L("1B=68I
M.PIO=71S*")/9F9S970@9G)O;2!S=&%R="!O9B!&24Q%<R!I<R`D;V9F<V5T
M(BD["G-E96L@7T9(+"`D;6%G:6-?<&]S("T@)$9)3$5?;V9F<V5T7W-I>F4@
M+2`D;V9F<V5T+"`P.PHD<W1A<G1?;V9?1DE,15]S96-T:6]N(#T@=&5L;"!?
M1D@["FUY("5R97%U:7)E7VQI<W0["G)E860@7T9(+"`D8G5F+"`T.R`@("`@
M("`@("`@("`@("`@("`@("`@("`@(",@<F5A9"!T:&4@9FER<W0@(D9)3$4B
M"G=H:6QE("@D8G5F(&5Q(")&24Q%(BD@>PIR96%D(%]&2"P@)&)U9BP@-#L*
M<F5A9"!?1D@L("1B=68L('5N<&%C:R@B3B(L("1B=68I.PIM>2`D9G5L;&YA
M;64@/2`D8G5F.PIO=71S*'%Q6U5N<&%C:VEN9R!&24Q%("(D9G5L;&YA;64B
M+BXN72D["FUY("1C<F,@/2`H("1F=6QL;F%M92`]?B!S?%XH6V$M9EQD77LX
M?2DO?'P@*2`_("0Q(#H@=6YD968["FUY("@D8F%S96YA;64L("1E>'0I(#T@
M*"1B=68@/7X@;7PH/SHN*B\I/R@N*BDH7"XN*BE\*3L*<F5A9"!?1D@L("1B
M=68L(#0["G)E860@7T9(+"`D8G5F+"!U;G!A8VLH(DXB+"`D8G5F*3L*:68@
M*&1E9FEN960H)&5X="D@86YD("1E>'0@(7X@+UPN*#\Z<&U\<&Q\:7A\86PI
M)"]I*2!["FUY("1F:6QE;F%M92`](%]S879E7V%S*"(D8W)C)&5X="(L("1B
M=68L(#`W-34I.PHD4$%2.CI(96%V>3HZ1G5L;$-A8VAE>R1F=6QL;F%M97T@
M/2`D9FEL96YA;64["B1005(Z.DAE879Y.CI&=6QL0V%C:&5[)&9I;&5N86UE
M?2`]("1F=6QL;F%M93L*?0IE;'-E('L*)$UO9'5L94-A8VAE>R1F=6QL;F%M
M97T@/2!["F)U9B`]/B`D8G5F+`IC<F,@/3X@)&-R8RP*;F%M92`]/B`D9G5L
M;&YA;64L"GT["B1R97%U:7)E7VQI<W1[)&9U;&QN86UE?2LK.PI]"G)E860@
M7T9(+"`D8G5F+"`T.PI]"FQO8V%L($!)3D,@/2`H<W5B('L*;7D@*"1S96QF
M+"`D;6]D=6QE*2`]($!?.PIR971U<FX@:68@<F5F("1M;V1U;&4@;W(@(21M
M;V1U;&4["FUY("1I;F9O(#T@)$UO9'5L94-A8VAE>R1M;V1U;&5].PIR971U
M<FX@=6YL97-S("1I;F9O.PID96QE=&4@)')E<75I<F5?;&ES='LD;6]D=6QE
M?3L*)$E.0WLD;6]D=6QE?2`]("(O;&]A9&5R+R1I;F9O+R1M;V1U;&4B.PII
M9B`H)$5.5GM005)?0TQ%04Y](&%N9"!D969I;F5D*"9)3SHZ1FEL93HZ;F5W
M*2D@>PIM>2`D9F@@/2!)3SHZ1FEL92T^;F5W7W1M<&9I;&4@;W(@9&EE(")#
M86XG="!C<F5A=&4@=&5M<"!F:6QE.B`D(2(["B1F:"T^8FEN;6]D92@I.PHD
M9F@M/G!R:6YT*"1I;F9O+3Y[8G5F?2D["B1F:"T^<V5E:R@P+"`P*3L*<F5T
M=7)N("1F:#L*?0IE;'-E('L*;7D@)&9I;&5N86UE(#T@7W-A=F5?87,H(B1I
M;F9O+3Y[8W)C?2YP;2(L("1I;F9O+3Y[8G5F?2D["B1I;F9O+3Y[9FEL97T@
M/2`D9FEL96YA;64["F]P96X@;7D@)&9H+"`G/#IR87<G+"`D9FEL96YA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1F:6QE;F%M92(Z("0A73L*<F5T=7)N
M("1F:#L*?0ID:64@(D)O;W1S=')A<'!I;F<@9F%I;&5D.B!C86XG="!F:6YD
M(&UO9'5L92`D;6]D=6QE(2(["GTL($!)3D,I.PIR97%U:7)E(%A33&]A9&5R
M.PIR97%U:7)E(%!!4CHZ2&5A=GD["G)E<75I<F4@0V%R<#HZ2&5A=GD["G)E
M<75I<F4@17AP;W)T97(Z.DAE879Y.PI005(Z.DAE879Y.CI?:6YI=%]D>6YA
M;&]A9&5R*"D["G)E<75I<F4@24\Z.D9I;&4["G=H:6QE("AM>2`D9FEL96YA
M;64@/2`H<V]R="!K97ES("5R97%U:7)E7VQI<W0I6S!=*2!["G5N;&5S<R`H
M)$E.0WLD9FEL96YA;65](&]R("1F:6QE;F%M92`]?B`O0E-$4$%.+RD@>PII
M9B`H)&9I;&5N86UE(#U^("]<+G!M8S\D+VDI('L*<F5Q=6ER92`D9FEL96YA
M;64["GT*96QS92!["F1O("1F:6QE;F%M92!U;FQE<W,@)&9I;&5N86UE(#U^
M("]S:71E8W5S=&]M:7IE7"YP;"0O.PI]"GT*9&5L971E("1R97%U:7)E7VQI
M<W1[)&9I;&5N86UE?3L*?0II9B`H)&)U9B!N92`D4$M:25!?34%'24,I('L*
M;W5T<RAQ<5M.;R!Z:7`@9F]U;F0@869T97(@1DE,12!S96-T:6]N(&EN(&9I
M;&4@(B1P<F]G;F%M92)=*3L*;&%S="!-04=)0R`["GT*)'-T87)T7W!O<R`]
M("AT96QL(%]&2"D@+2`T.R`@("`@("`@("`@("`@("`C('-T87)T(&]F('II
M<`I]"FUY($!P87)?87)G<SL*;7D@*"1O=70L("1B=6YD;&4L("1L;V=F:"P@
M)&-A8VAE7VYA;64I.PID96QE=&4@)$5.5GM005)?05!07U)%55-%?3L@(R!S
M86YI=&EZ92`H4D554T4@;6%Y(&)E(&$@<V5C=7)I='D@<')O8FQE;2D*:68@
M*"$D<W1A<G1?<&]S(&]R("@D05)'5ELP72!E<2`G+2UP87(M;W!T:6]N<R<@
M)B8@<VAI9G0I*2!["FUY("5D:7-T7V-M9"`]('%W*`IP("`@8FQI8E]T;U]P
M87(*:2`@(&EN<W1A;&Q?<&%R"G4@("!U;FEN<W1A;&Q?<&%R"G,@("!S:6=N
M7W!A<@IV("`@=F5R:69Y7W!A<@HI.PII9B`H0$%21U8@86YD("1!4D=66S!=
M(&5Q("<M+7)E=7-E)RD@>PIS:&EF="!`05)'5CL*)$5.5GM005)?05!07U)%
M55-%?2`]('-H:69T($!!4D=6.PI]"F5L<V4@>R`C(&YO<FUA;"!P87)L(&)E
M:&%V:6]U<@IM>2!`861D7W1O7VEN8SL*=VAI;&4@*$!!4D=6*2!["B1!4D=6
M6S!=(#U^("]>+2A;04E-3T),8G%P:75S5'9=*2@N*BDO(&]R(&QA<W0["FEF
M("@D,2!E<2`G22<I('L*<'5S:"!`861D7W1O7VEN8RP@)#(["GT*96QS:68@
M*"0Q(&5Q("=-)RD@>PIE=F%L(")U<V4@)#(B.PI]"F5L<VEF("@D,2!E<2`G
M02<I('L*=6YS:&EF="!`<&%R7V%R9W,L("0R.PI]"F5L<VEF("@D,2!E<2`G
M3R<I('L*)&]U="`]("0R.PI]"F5L<VEF("@D,2!E<2`G8B<I('L*)&)U;F1L
M92`]("=S:71E)SL*?0IE;'-I9B`H)#$@97$@)T(G*2!["B1B=6YD;&4@/2`G
M86QL)SL*?0IE;'-I9B`H)#$@97$@)W$G*2!["B1Q=6EE="`](#$["GT*96QS
M:68@*"0Q(&5Q("=,)RD@>PIO<&5N("1L;V=F:"P@(CX^(BP@)#(@;W(@9&EE
M('%Q6T-A;B=T(&]P96X@;&]G(&9I;&4@(B0R(CH@)"%=.PI]"F5L<VEF("@D
M,2!E<2`G5"<I('L*)&-A8VAE7VYA;64@/2`D,CL*?0IS:&EF="A`05)'5BD[
M"FEF("AM>2`D8VUD(#T@)&1I<W1?8VUD>R0Q?2D@>PID96QE=&4@)$5.5GLG
M4$%27U1%35`G?3L*:6YI=%]I;F,H*3L*<F5Q=6ER92!005(Z.D1I<W0["B9[
M(E!!4CHZ1&ES=#HZ)&-M9")]*"D@=6YL97-S($!!4D=6.PHF>R)005(Z.D1I
M<W0Z.B1C;60B?2@D7RD@9F]R($!!4D=6.PIE>&ET.PI]"GT*=6YS:&EF="!`
M24Y#+"!`861D7W1O7VEN8SL*?0I]"FEF("@D;W5T*2!["GL*<F5Q=6ER92!)
M3SHZ1FEL93L*<F5Q=6ER92!!<F-H:79E.CI::7`["G)E<75I<F4@1&EG97-T
M.CI32$$["GT*;7D@)'!A<B`]('-H:69T*$!!4D=6*3L*;7D@)'II<#L*:68@
M*&1E9FEN960@)'!A<BD@>PIO<&5N(&UY("1F:"P@)SPZ<F%W)RP@)'!A<B!O
M<B!D:64@<7%;0V%N)W0@9FEN9"!P87(@9FEL92`B)'!A<B(Z("0A73L*8FQE
M<W,H)&9H+"`G24\Z.D9I;&4G*3L*)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W
M.PHH("1Z:7`M/G)E861&<F]M1FEL94AA;F1L92@D9F@L("1P87(I(#T]($%R
M8VAI=F4Z.EII<#HZ05I?3TLH*2`I"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG
M('II<"!A<F-H:79E("(D<&%R(ET["GT*;7D@)6UE=&%?<&%R(#T@9&\@>PII
M9B`H)'II<"!A;F0@;7D@)&UE=&$@/2`D>FEP+3YC;VYT96YT<R@G345402YY
M;6PG*2D@>PHD;65T82`]?B!S+RXJ7G!A<CHD+R]M<SL*)&UE=&$@/7X@<R]>
M7%,N*B\O;7,["B1M971A(#U^("]>("`H6UXZ72LI.B`H+BLI)"]M9SL*?0I]
M.PII9B`H9&5F:6YE9"`D<&%R*2!["F]P96X@;7D@)'!H+"`G/#IR87<G+"`D
M<&%R(&]R(&1I92!Q<5M#86XG="!R96%D('!A<B!F:6QE("(D<&%R(CH@)"%=
M.PIM>2`D8G5F.PIR96%D("1P:"P@)&)U9BP@-#L*9&EE('%Q6R(D<&%R(B!I
M<R!N;W0@82!P87(@9FEL95T@=6YL97-S("1B=68@97$@)%!+6DE07TU!1TE#
M.PIC;&]S92`D<&@["GT*0W)E871E4&%T:"@D;W5T*2`["FUY("1F:"`]($E/
M.CI&:6QE+3YN97<H"B1O=70L"DE/.CI&:6QE.CI/7T-214%4*"D@?"!)3SHZ
M1FEL93HZ3U]21%=2*"D@?"!)3SHZ1FEL93HZ3U]44E5.0R@I+`HP-S<W+`HI
M(&]R(&1I92!Q<5M#86XG="!C<F5A=&4@9FEL92`B)&]U="(Z("0A73L*)&9H
M+3YB:6YM;V1E*"D["G-E96L@7T9(+"`P+"`P.PIM>2`D;&]A9&5R.PII9B`H
M9&5F:6YE9"`D<W1A<G1?;V9?1DE,15]S96-T:6]N*2!["G)E860@7T9(+"`D
M;&]A9&5R+"`D<W1A<G1?;V9?1DE,15]S96-T:6]N.PI](&5L<V4@>PIL;V-A
M;"`D+R`]('5N9&5F.PHD;&]A9&5R(#T@/%]&2#X["GT*:68@*"$D14Y6>U!!
M4E]615)"051)37T@86YD("1L;V%D97(@/7X@+UXH/SHC(7Q<0')E;2DO*2![
M"G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP.PI005(Z.D9I;'1E<CHZ
M4&]D4W1R:7`M/F%P<&QY*%PD;&]A9&5R+"`D,"D["GT*)&9H+3YP<FEN="@D
M;&]A9&5R*0IO<B!D:64@<7%;17)R;W(@=W)I=&EN9R!L;V%D97(@=&\@(B1O
M=70B.B`D(5T["FEF("@D8G5N9&QE*2!["G)E<75I<F4@4$%2.CI(96%V>3L*
M4$%2.CI(96%V>3HZ7VEN:71?9'EN86QO861E<B@I.PII;FET7VEN8R@I.PIR
M97%U:7)E7VUO9'5L97,H*3L*;7D@0&EN8R`](&=R97`@>R`A+T)31%!!3B\@
M?0IG<F5P('L**"1B=6YD;&4@;F4@)W-I=&4G*2!O<@HH)%\@;F4@)$-O;F9I
M9SHZ0V]N9FEG>V%R8VAL:6)E>'!](&%N9`HD7R!N92`D0V]N9FEG.CI#;VYF
M:6=[<')I=FQI8F5X<'TI.PI]($!)3D,["G-\+RM\+WQG+"!S?"\D?'P@9F]R
M96%C:"!`:6YC.PIM>2`E9FEL97,["B1F:6QE<WLD7WTK*R!F;W(@0$1Y;F%,
M;V%D97(Z.F1L7W-H87)E9%]O8FIE8W1S+"!V86QU97,@)4E.0SL*;7D@)&QI
M8E]E>'0@/2`D0V]N9FEG.CI#;VYF:6=[;&EB7V5X='T[("`@("`@("`@(R!8
M6%@@;&EB7V5X="!V<R!D;&5X="`_"FUY("5W<FET=&5N.PIF;W)E86-H(&UY
M("1K97D@*'-O<G0@:V5Y<R`E9FEL97,I('L*;7D@*"1F:6QE+"`D;F%M92D[
M"FEF("AD969I;F5D*&UY("1F8R`]("1005(Z.DAE879Y.CI&=6QL0V%C:&5[
M)&ME>7TI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("1F8RD["GT*96QS
M92!["F9O<F5A8V@@;7D@)&1I<B`H0&EN8RD@>PII9B`H)&ME>2`]?B!M?%Y<
M421D:7)<12\H+BHI)'QI*2!["B@D9FEL92P@)&YA;64I(#T@*"1K97DL("0Q
M*3L*;&%S=#L*?0II9B`H)&ME>2`]?B!M?%XO;&]A9&5R+UM>+UTK+R@N*BDD
M?"D@>PII9B`H;7D@)')E9B`]("1-;V1U;&5#86-H97LD,7TI('L**"1F:6QE
M+"`D;F%M92D@/2`H)')E9BP@)#$I.PIL87-T.PI]"FEF("@M9B`B)&1I<B\D
M,2(I('L**"1F:6QE+"`D;F%M92D@/2`H(B1D:7(O)#$B+"`D,2D["FQA<W0[
M"GT*?0I]"GT*;F5X="!U;FQE<W,@9&5F:6YE9"`D;F%M93L*;F5X="!I9B`D
M=W)I='1E;GLD;F%M97TK*SL*;F5X="!I9B`A<F5F*"1F:6QE*2!A;F0@)&9I
M;&4@/7X@+UPN7%$D;&EB7V5X=%Q%)"]I.PIO=71S*'-P<FEN=&8H<7%;4&%C
M:VEN9R!&24Q%("(E<R(N+BY=+"!R968@)&9I;&4@/R`D9FEL92T^>VYA;65]
M(#H@)&9I;&4I*3L*;7D@)&-O;G1E;G0["FEF("AR968H)&9I;&4I*2!["B1C
M;VYT96YT(#T@)&9I;&4M/GMB=69].PI]"F5L<V4@>PIL;V-A;"`D+R`]('5N
M9&5F.PIO<&5N(&UY("1T:"P@)SPZ<F%W)RP@)&9I;&4@;W(@9&EE('%Q6T-A
M;B=T(')E860@(B1F:6QE(CH@)"%=.PHD8V]N=&5N="`](#PD=&@^.PIC;&]S
M92`D=&@["E!!4CHZ1FEL=&5R.CI0;V13=')I<"T^87!P;'DH7"1C;VYT96YT
M+"`B/&5M8F5D9&5D/B\D;F%M92(I"FEF("$D14Y6>U!!4E]615)"051)37T@
M86YD("1N86UE(#U^("]<+B@_.G!M?&EX?&%L*20O:3L*4$%2.CI&:6QT97(Z
M.E!A=&-H0V]N=&5N="T^;F5W+3YA<'!L>2A<)&-O;G1E;G0L("1F:6QE+"`D
M;F%M92D["GT*)&9H+3YP<FEN="@B1DE,12(L"G!A8VLH)TXG+"!L96YG=&@H
M)&YA;64I("L@.2DL"G-P<FEN=&8H(B4P.'@O)7,B+"!!<F-H:79E.CI::7`Z
M.F-O;7!U=&5#4D,S,B@D8V]N=&5N="DL("1N86UE*2P*<&%C:R@G3B<L(&QE
M;F=T:"@D8V]N=&5N="DI+`HD8V]N=&5N="D*;W(@9&EE('%Q6T5R<F]R('=R
M:71I;F<@96UB961D960@1DE,12!T;R`B)&]U="(Z("0A73L*;W5T<RAQ<5M7
M<FET=&5N(&%S("(D;F%M92)=*3L*?0I]"FEF("@D>FEP*2!["B1Z:7`M/G=R
M:71E5&]&:6QE2&%N9&QE*"1F:"D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2PIO
M<B!D:64@<7%;17)R;W(@=W)I=&EN9R!Z:7`@<&%R="!O9B`B)&]U=")=.PI]
M"FEF("@D;65T85]P87)[8VQE86Y]*2!["B1F:"T^<')I;G0H(EPP0TQ%04XB
M*3L*?0IE;'-E('L*:68@*"%D969I;F5D("1C86-H95]N86UE*2!["FUY("1C
M='@@/2!$:6=E<W0Z.E-(02T^;F5W*#$I.PIS965K("1F:"P@,"P@,#L*)&-T
M>"T^861D9FEL92@D9F@I.PIS965K("1F:"P@,"P@,CL*)&-A8VAE7VYA;64@
M/2`D8W1X+3YH97AD:6=E<W0["GT*)&-A8VAE7VYA;64@/2!S=6)S='(@)&-A
M8VAE7VYA;64L(#`L("1C86-H95]N86UE7W-I>F4["B1C86-H95]N86UE("X]
M(")<,"(@>"`H)&-A8VAE7VYA;65?<VEZ92`M(&QE;F=T:"`D8V%C:&5?;F%M
M92D["B1F:"T^<')I;G0H)&-A8VAE7VYA;64L"B)<,$-!0TA%(BD["GT*;7D@
M)&]F9G-E="`]("1F:"T^=&5L;"`M(&QE;F=T:"@D;&]A9&5R*3L*)&9H+3YP
M<FEN="AP86-K*"=.)RP@)&]F9G-E="DL"B1005)?34%'24,I.PHD9F@M/F-L
M;W-E(&]R(&1I92!Q<5M%<G)O<B!W<FET:6YG('1R86EL97(@;V8@(B1O=70B
M.B`D(5T["F-H;6]D(#`W-34L("1O=70["F5X:70["GT*>PIL87-T('5N;&5S
M<R!D969I;F5D("1S=&%R=%]P;W,["E]F:7A?<')O9VYA;64H*3L*<F5Q=6ER
M92!005(["E!!4CHZ2&5A=GDZ.E]I;FET7V1Y;F%L;V%D97(H*3L*>PIR97%U
M:7)E($9I;&4Z.D9I;F0["G)E<75I<F4@07)C:&EV93HZ6FEP.PI]"FUY("1F
M:"`]($E/.CI&:6QE+3YN97<[("`@("`@("`@("`@("`@("`@("`@("`@("`@
M("`C($%R8VAI=F4Z.EII<"!O<&5R871E<R!O;B!A;B!)3SHZ2&%N9&QE"B1F
M:"T^9F1O<&5N*&9I;&5N;RA?1D@I+"`G<B<I(&]R(&1I92!Q<5MF9&]P96XH
M*2!F86EL960Z("0A73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H+7,@
M7T9(*3L*;7D@)'II<"`]($%R8VAI=F4Z.EII<"T^;F5W.PHH)'II<"T^<F5A
M9$9R;VU&:6QE2&%N9&QE*"1F:"P@)'!R;V=N86UE*2`]/2!!<F-H:79E.CI:
M:7`Z.D%:7T]+*"DI"F]R(&1I92!Q<5M%<G)O<B!R96%D:6YG('II<"!A<F-H
M:79E("(D<')O9VYA;64B73L*07)C:&EV93HZ6FEP.CIS971#:'5N:U-I>F4H
M-C0@*B`Q,#(T*3L*<'5S:"!`4$%2.CI,:6)#86-H92P@)'II<#L*)%!!4CHZ
M3&EB0V%C:&5[)'!R;V=N86UE?2`]("1Z:7`["F]U=',H(D5X=')A8W1I;F<@
M>FEP+BXN(BD["FEF("AD969I;F5D("1%3E9[4$%27U1%35!]*2![(",@<VAO
M=6QD(&)E('-E="!A="!T:&ES('!O:6YT(0IF;W)E86-H(&UY("1M96UB97(@
M*"`D>FEP+3YM96UB97)S("D@>PIN97AT(&EF("1M96UB97(M/FES1&ER96-T
M;W)Y.PIM>2`D;65M8F5R7VYA;64@/2`D;65M8F5R+3YF:6QE3F%M93L*;F5X
M="!U;FQE<W,@)&UE;6)E<E]N86UE(#U^(&U["EX*+S]S:&QI8B\**#\Z)$-O
M;F9I9SHZ0V]N9FEG>V%R8VAN86UE?2\I/PHH6UXO72LI"B0*?7@["FUY("1E
M>'1R86-T7VYA;64@/2`D,3L*;7D@)&1E<W1?;F%M92`]($9I;&4Z.E-P96,M
M/F-A=&9I;&4H)$5.5GM005)?5$5-4'TL("1E>'1R86-T7VYA;64I.PII9B`H
M+68@)&1E<W1?;F%M92`F)B`M<R!?(#T]("1M96UB97(M/G5N8V]M<')E<W-E
M9%-I>F4H*2D@>PIO=71S*'%Q6U-K:7!P:6YG("(D;65M8F5R7VYA;64B('-I
M;F-E(&ET(&%L<F5A9'D@97AI<W1S(&%T("(D9&5S=%]N86UE(ETI.PI](&5L
M<V4@>PIO=71S*'%Q6T5X=')A8W1I;F<@(B1M96UB97)?;F%M92(@=&\@(B1D
M97-T7VYA;64B+BXN72D["B@D;65M8F5R+3YE>'1R86-T5&]&:6QE3F%M960H
M)&1E<W1?;F%M92D@/3T@07)C:&EV93HZ6FEP.CI!6E]/2R@I*0IO<B!D:64@
M<7%;17)R;W(@97AT<F%C=&EN9R!Z:7`@;65M8F5R('1O("(D9&5S=%]N86UE
M(ET["F-H;6]D*#`U-34L("1D97-T7VYA;64I(&EF("1>3R!E<2`B:'!U>"([
M"GT*?0I]"GT*=6YL97-S("@D4$%2.CI,:6)#86-H97LD<')O9VYA;65]*2![
M"F1I92`\/"`B+B(@=6YL97-S($!!4D=6.PI5<V%G93H@)#`@6R`M06QI8BYP
M87(@72!;("U)9&ER(%T@6R`M36UO9'5L92!=(%L@<W)C+G!A<B!=(%L@<')O
M9W)A;2YP;"!="B0P(%L@+4)\+6(@72!;+4]O=71F:6QE72!S<F,N<&%R"BX*
M)$5.5GM005)?4%)/1TY!345](#T@)'!R;V=N86UE(#T@)#`@/2!S:&EF="A`
M05)'5BD["GT*<W5B($-R96%T95!A=&@@>PIM>2`H)&YA;64I(#T@0%\["G)E
M<75I<F4@1FEL93HZ0F%S96YA;64["FUY("@D8F%S96YA;64L("1P871H+"`D
M97AT*2`]($9I;&4Z.D)A<V5N86UE.CIF:6QE<&%R<V4H)&YA;64L("@G7"XN
M*B<I*3L*<F5Q=6ER92!&:6QE.CI0871H.PI&:6QE.CI0871H.CIM:W!A=&@H
M)'!A=&@I('5N;&5S<R@M92`D<&%T:"D[(",@;6MP871H(&1I97,@=VET:"!E
M<G)O<@I]"G-U8B!R97%U:7)E7VUO9'5L97,@>PIR97%U:7)E(&QI8CL*<F5Q
M=6ER92!$>6YA3&]A9&5R.PIR97%U:7)E(&EN=&5G97(["G)E<75I<F4@<W1R
M:6-T.PIR97%U:7)E('=A<FYI;F=S.PIR97%U:7)E('9A<G,["G)E<75I<F4@
M0V%R<#L*<F5Q=6ER92!#87)P.CI(96%V>3L*<F5Q=6ER92!%<G)N;SL*<F5Q
M=6ER92!%>'!O<G1E<CHZ2&5A=GD["G)E<75I<F4@17AP;W)T97(["G)E<75I
M<F4@1F-N=&P["G)E<75I<F4@1FEL93HZ5&5M<#L*<F5Q=6ER92!&:6QE.CI3
M<&5C.PIR97%U:7)E(%A33&]A9&5R.PIR97%U:7)E($-O;F9I9SL*<F5Q=6ER
M92!)3SHZ2&%N9&QE.PIR97%U:7)E($E/.CI&:6QE.PIR97%U:7)E($-O;7!R
M97-S.CI:;&EB.PIR97%U:7)E($%R8VAI=F4Z.EII<#L*<F5Q=6ER92!$:6=E
M<W0Z.E-(03L*<F5Q=6ER92!005(["G)E<75I<F4@4$%2.CI(96%V>3L*<F5Q
M=6ER92!005(Z.D1I<W0["G)E<75I<F4@4$%2.CI&:6QT97(Z.E!O9%-T<FEP
M.PIR97%U:7)E(%!!4CHZ1FEL=&5R.CI0871C:$-O;G1E;G0["G)E<75I<F4@
M871T<FEB=71E<SL*979A;"![(')E<75I<F4@0W=D('T["F5V86P@>R!R97%U
M:7)E(%=I;C,R('T["F5V86P@>R!R97%U:7)E(%-C86QA<CHZ571I;"!].PIE
M=F%L('L@<F5Q=6ER92!!<F-H:79E.CI5;GII<#HZ0G5R<W0@?3L*979A;"![
M(')E<75I<F4@5&EE.CI(87-H.CI.86UE9$-A<'1U<F4@?3L*979A;"![(')E
M<75I<F4@4&5R;$E/.R!R97%U:7)E(%!E<FQ)3SHZ<V-A;&%R('T["F5V86P@
M>R!R97%U:7)E('5T9C@@?3L*?0IS=6(@7W-E=%]P87)?=&5M<"!["FEF("AD
M969I;F5D("1%3E9[4$%27U1%35!](&%N9"`D14Y6>U!!4E]414U0?2`]?B`O
M*"XK*2\I('L*)'!A<E]T96UP(#T@)#$["G)E='5R;CL*?0IF;W)E86-H(&UY
M("1P871H("@**&UA<"`D14Y6>R1??2P@<7<H(%!!4E]435!$25(@5$U01$E2
M(%1%35!$25(@5$5-4"!435`@*2DL"G%W*"!#.EQ<5$5-4"`O=&UP("X@*0HI
M('L*;F5X="!U;FQE<W,@9&5F:6YE9"`D<&%T:"!A;F0@+60@)'!A=&@@86YD
M("UW("1P871H.PIM>2`D=7-E<FYA;64["FUY("1P=W5I9#L*979A;"![*"1P
M=W5I9"D@/2!G971P=W5I9"@D/BD@:68@9&5F:6YE9"`D/CM].PII9B`H(&1E
M9FEN960H)E=I;C,R.CI,;V=I;DYA;64I("D@>PHD=7-E<FYA;64@/2`F5VEN
M,S(Z.DQO9VEN3F%M93L*?0IE;'-I9B`H9&5F:6YE9"`D<'=U:60I('L*)'5S
M97)N86UE(#T@)'!W=6ED.PI]"F5L<V4@>PHD=7-E<FYA;64@/2`D14Y6>U53
M15).04U%?2!\?"`D14Y6>U5315)]('Q\("=365-414TG.PI]"FUY("1S=&UP
M9&ER(#T@(B1P871H)'-Y<WM?9&5L:6U]<&%R+2(N=6YP86-K*")(*B(L("1U
M<V5R;F%M92D["FUK9&ER("1S=&UP9&ER+"`P-S4U.PIM>2`D8V%C:&5?9&ER
M.PII9B`H)$5.5GM005)?0TQ%04Y]*2!["B1C86-H95]D:7(@/2`B=&5M<"TD
M)"(["GT*96QS92!["F]P96X@;7D@)&9H+"`B/#IR87<B+"`D<')O9VYA;64@
M;W(@9&EE('%Q6T-A;B=T(')E860@(B1P<F]G;F%M92(Z("0A73L*:68@*"AM
M>2`D;6%G:6-?<&]S(#T@9FEN9%]P87)?;6%G:6,H)&9H*2D@/CT@,"D@>PIS
M965K("1F:"P@)&UA9VEC7W!O<PHM("1&24Q%7V]F9G-E=%]S:7IE"BT@;&5N
M9W1H*")<,$-!0TA%(BDL(#`["FUY("1B=68["G)E860@)&9H+"`D8G5F+"!L
M96YG=&@H(EPP0T%#2$4B*3L*:68@*"1B=68@97$@(EPP0T%#2$4B*2!["G-E
M96L@)&9H+"`D;6%G:6-?<&]S"BT@)$9)3$5?;V9F<V5T7W-I>F4*+2!L96YG
M=&@H(EPP0T%#2$4B*0HM("1C86-H95]N86UE7W-I>F4L(#`["G)E860@)&9H
M+"`D8G5F+"`D8V%C:&5?;F%M95]S:7IE.PHD8G5F(#U^(',O7#`O+V<["B1C
M86-H95]D:7(@/2`B8V%C:&4M)&)U9B(["GT*?0IC;&]S92`D9F@["GT*:68@
M*"$D8V%C:&5?9&ER*2!["B1C86-H95]D:7(@/2`B=&5M<"TD)"(["B1%3E9[
M4$%27T-,14%.?2`](#$["GT*)'-T;7!D:7(@+CT@(B1S>7-[7V1E;&EM?21C
M86-H95]D:7(B.PIM:V1I<B`D<W1M<&1I<BP@,#<U-3L*)$5.5GM005)?5$5-
M4'T@/2`D<W1M<&1I<CL*;&%S=#L*?0HD<&%R7W1E;7`@/2`D,2!I9B`D14Y6
M>U!!4E]414U0?2!A;F0@)$5.5GM005)?5$5-4'T@/7X@+R@N*RDO.PI]"G-U
M8B!?<V%V95]A<R!["FUY("@D;F%M92P@)&-O;G1E;G1S+"`D;6]D92D@/2!`
M7SL*;7D@)&9U;&QN86UE(#T@(B1P87)?=&5M<"\D;F%M92(["G5N;&5S<R`H
M+64@)&9U;&QN86UE*2!["FUY("1T96UP;F%M92`]("(D9G5L;&YA;64N)"0B
M.PIM>2`D9F@["BAO<&5N("1F:"P@)SXZ<F%W)RP@)'1E;7!N86UE*0HF)B`H
M<')I;G0@)&9H("1C;VYT96YT<RD*)B8@*&-L;W-E("1F:"D*;W(@9&EE('%Q
M6T5R<F]R('=R:71I;F<@(B1T96UP;F%M92(Z("0A73L*8VAM;V0@)&UO9&4L
M("1T96UP;F%M92!I9B!D969I;F5D("1M;V1E.PIR96YA;64H)'1E;7!N86UE
M+"`D9G5L;&YA;64I(&]R('5N;&EN:R@D=&5M<&YA;64I.PI]"G)E='5R;B`D
M9G5L;&YA;64["GT*<W5B(%]S971?<')O9VYA;64@>PII9B`H9&5F:6YE9"`D
M14Y6>U!!4E]04D]'3D%-17T@86YD("1%3E9[4$%27U!23T=.04U%?2`]?B`O
M*"XK*2\I('L*)'!R;V=N86UE(#T@)#$["GT*)'!R;V=N86UE('Q\/2`D,#L*
M:68@*"1%3E9[4$%27U1%35!](&%N9"!I;F1E>"@D<')O9VYA;64L("1%3E9[
M4$%27U1%35!]*2`^/2`P*2!["B1P<F]G;F%M92`]('-U8G-T<B@D<')O9VYA
M;64L(')I;F1E>"@D<')O9VYA;64L("1S>7-[7V1E;&EM?2D@*R`Q*3L*?0II
M9B`H(21%3E9[4$%27U!23T=.04U%?2!O<B!I;F1E>"@D<')O9VYA;64L("1S
M>7-[7V1E;&EM?2D@/CT@,"D@>PII9B`H;W!E;B!M>2`D9F@L("<\)RP@)'!R
M;V=N86UE*2!["G)E='5R;B!I9B`M<R`D9F@["GT*:68@*"US("(D<')O9VYA
M;64D<WES>U]E>&5](BD@>PHD<')O9VYA;64@+CT@)'-Y<WM?97AE?3L*<F5T
M=7)N.PI]"GT*9F]R96%C:"!M>2`D9&ER("AS<&QI="`O7%$D<WES>W!A=&A?
M<V5P?5Q%+RP@)$5.5GM0051(?2D@>PIN97AT(&EF(&5X:7-T<R`D14Y6>U!!
M4E]414U0?2!A;F0@)&1I<B!E<2`D14Y6>U!!4E]414U0?3L*)&1I<B`]?B!S
M+UQ1)'-Y<WM?9&5L:6U]7$4D+R\["B@H)'!R;V=N86UE(#T@(B1D:7(D<WES
M>U]D96QI;7TD<')O9VYA;64D<WES>U]E>&5](BDL(&QA<W0I"FEF("US("(D
M9&ER)'-Y<WM?9&5L:6U])'!R;V=N86UE)'-Y<WM?97AE?2(["B@H)'!R;V=N
M86UE(#T@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B*2P@;&%S="D*:68@
M+7,@(B1D:7(D<WES>U]D96QI;7TD<')O9VYA;64B.PI]"GT*<W5B(%]F:7A?
M<')O9VYA;64@>PHD,"`]("1P<F]G;F%M92!\?#T@)$5.5GM005)?4%)/1TY!
M345].PII9B`H:6YD97@H)'!R;V=N86UE+"`D<WES>U]D96QI;7TI(#P@,"D@
M>PHD<')O9VYA;64@/2`B+B1S>7-[7V1E;&EM?21P<F]G;F%M92(["GT*;7D@
M)'!W9"`]("AD969I;F5D("9#=V0Z.F=E=&-W9"D@/R!#=V0Z.F=E=&-W9"@I
M"CH@*"AD969I;F5D("97:6XS,CHZ1V5T0W=D*2`_(%=I;C,R.CI'971#=V0H
M*2`Z(&!P=V1@*3L*8VAO;7`H)'!W9"D["B1P<F]G;F%M92`]?B!S+UXH/SU<
M+EPN/UQ1)'-Y<WM?9&5L:6U]7$4I+R1P=V0D<WES>U]D96QI;7TO.PHD14Y6
M>U!!4E]04D]'3D%-17T@/2`D<')O9VYA;64["GT*<W5B(%]P87)?:6YI=%]E
M;G8@>PII9B`H("1%3E9[4$%27TE.251)04Q)6D5$?2LK(#T](#$@*2!["G)E
M='5R;CL*?2!E;'-E('L*)$5.5GM005)?24Y)5$E!3$E:141](#T@,CL*?0IF
M;W(@*'%W*"!34$%73D5$(%1%35`@0TQ%04X@1$5"54<@0T%#2$4@4%)/1TY!
M344@*2`I('L*9&5L971E("1%3E9[(E!!4E\D7R)].PI]"F9O<B`H<7<O(%1-
M4$1)4B!414U0($-,14%.($1%0E5'("\I('L*)$5.5GLB4$%27R1?(GT@/2`D
M14Y6>R)005)?1TQ/0D%,7R1?(GT@:68@97AI<W1S("1%3E9[(E!!4E]'3$]"
M04Q?)%\B?3L*?0II9B`H)$5.5GM005)?5$5-4'TI('L*9&5L971E("1%3E9[
M4$%27T-,14%.?3L*?0I]"G-U8B!O=71S('L*<F5T=7)N(&EF("1Q=6EE=#L*
M:68@*"1L;V=F:"D@>PIP<FEN="`D;&]G9F@@(D!?7&XB.PI]"F5L<V4@>PIP
M<FEN="`B0%]<;B(["GT*?0IS=6(@:6YI=%]I;F,@>PIR97%U:7)E($-O;F9I
M9SL*<'5S:"!`24Y#+"!G<F5P(&1E9FEN960L(&UA<"`D0V]N9FEG.CI#;VYF
M:6=[)%]]+"!Q=R@*87)C:&QI8F5X<"!P<FEV;&EB97AP('-I=&5A<F-H97AP
M('-I=&5L:6)E>'`*=F5N9&]R87)C:&5X<"!V96YD;W)L:6)E>'`**3L*?0IP
M86-K86=E(&UA:6X["G)E<75I<F4@4$%2.PIU;G-H:69T($!)3D,L(%PF4$%2
M.CIF:6YD7W!A<CL*4$%2+3YI;7!O<G0H0'!A<E]A<F=S*3L*9&EE('%Q6W!A
M<BYP;#H@0V%N)W0@;W!E;B!P97)L('-C<FEP="`B)'!R;V=N86UE(CH@3F\@
M<W5C:"!F:6QE(&]R(&1I<F5C=&]R>5QN70IU;FQE<W,@+64@)'!R;V=N86UE
M.PID;R`D<')O9VYA;64["D-/4D4Z.F5X:70H)#$I(&EF("@D0"`]?B]>7U1+
M7T58251?7"@H7&0K*5PI+RD["F1I92`D0"!I9B`D0#L*?3L*)#HZ7U]%4E)/
M4B`]("1`(&EF("1`.PI]"D-/4D4Z.F5X:70H)#$I(&EF("@D.CI?7T524D]2
M(#U^+UY?5$M?15A)5%]<*"A<9"LI7"DO*3L*9&EE("0Z.E]?15)23U(@:68@
M)#HZ7U]%4E)/4CL*,3L*7U]%3D1?7PH`<&5R;'AS:2YC`````$-!0TA%```]
M````0`````L```!`````$````$`````F````0````#4```!``````P```$``
M```&````0````!0```!`````#@```$`````-````0````"4```!`````$@``
M`$`````H````0`````(```!`````&````$`````)````0`````H```!`````
M``````````"!L@&!L+"L!P````"$`@&!L`^Q``````"%L@&!L+"N?P````"@
MB/]_L*H)@&R+_W\!````<([_?["K'H#LC_]_P/__?QR1_W\!````6)/_?["H
M`X#\D_]_`0```+2?_W^L__]_(*#_?[#__W_<H?]_L+"H@/BA_W\!````9*/_
M?["O/(!,JO]_L*L"@!BM_W\!````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M````````````````````````````````````````````````````````````
M``````````````````````````````````````````````````````````!D
M&@``\!D```$````_`P```0```$H#```!````5`,```$```!H`P``#P```/`$
M```,````+!,```T```!@2P``&0```*R=`0`;````!````!H```"PG0$`'```
M``0```#U_O]OM`$```4````("@``!@```$@#```*````$P4```L````0````
M%0`````````#````S)X!``(```#@`0``%````!$````7````3!$``!$```!T
M$```$@```-@````3````"````!@`````````^___;P$```C^__]O]`\``/__
M_V\#````\/__;QP/``#Z__]O#@``````````````````````````````````
M``````````````````````"TG0$````````````X$P``.!,``#@3```X$P``
M.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X
M$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3
M```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,`
M`#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``
M.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3```X$P``.!,``#@3````
M````````````````````````````````````````````````````````````
M````````(!8````````$H`$`"E!!4BYP;0H`````0$\``%A,````````&$T`
M`!Q-```T3P``8$P``#Q/````````($\``"A/````````2$\``$=#0SH@*$=.
M52D@,3(N,2XP`$=#0SH@*&-R;W-S=&]O;"U.1R`Q+C(U+C`N,CA?8C@X9#,S
M."D@,3(N,2XP`$$T````865A8FD``2H````%-RU!``8*!T$(`0D""@,,`1($
M$P$4`14!%P,8`1D!&@(<`2(!`"YS:'-T<G1A8@`N:6YT97)P`"YN;W1E+F=N
M=2YB=6EL9"UI9``N;F]T92Y!0DDM=&%G`"YG;G4N:&%S:``N9'EN<WEM`"YD
M>6YS='(`+F=N=2YV97)S:6]N`"YG;G4N=F5R<VEO;E]R`"YR96PN9'EN`"YR
M96PN<&QT`"YI;FET`"YT97AT`"YF:6YI`"YR;V1A=&$`+D%232YE>'1A8@`N
M05)-+F5X:61X`"YE:%]F<F%M90`N:6YI=%]A<G)A>0`N9FEN:5]A<G)A>0`N
M9'EN86UI8P`N9V]T`"YD871A`"YB<W,`+F-O;6UE;G0`+D%232YA='1R:6)U
M=&5S```````````````````````````````````````````````````````+
M`````0````(```!4`0``5`$``!D```````````````$`````````$P````<`
M```"````<`$``'`!```D```````````````$`````````"8````'`````@``
M`)0!``"4`0``(```````````````!``````````T````]O__;P(```"T`0``
MM`$``)0!```%``````````0````$````/@````L````"````2`,``$@#``#`
M!@``!@````,````$````$````$8````#`````@````@*```("@``$P4`````
M`````````0````````!.````____;P(````<#P``'`\``-@````%````````
M``(````"````6P```/[__V\"````]`\``/0/``"`````!@````,````$````
M`````&H````)`````@```'00``!T$```V`````4`````````!`````@```!S
M````"0```$(```!,$0``3!$``.`!```%````%@````0````(````?`````$`
M```&````+!,``"P3```,```````````````$`````````'<````!````!@``
M`#@3```X$P``Y`(`````````````!`````0```""`````0````8````@%@``
M(!8``$`U``````````````@`````````B`````$````&````8$L``&!+```(
M```````````````$`````````(X````!`````@```&A+``!H2P``]$$`````
M````````!`````````"6`````0````(```!<C0``7(T``"0`````````````
M``0`````````H0````$``'""````@(T``("-``!P````#0`````````$````
M`````*P````!`````@```/"-``#PC0``!```````````````!`````````"V
M````#@````,```"LG0$`K)T```0```````````````0````$````P@````\`
M```#````L)T!`+"=```$```````````````$````!````,X````&`````P``
M`+2=`0"TG0``&`$```8`````````!`````@```#7`````0````,```#,G@$`
MS)X``#0!``````````````0````$````W`````$````#`````*`!``"@``!(
M```````````````$`````````.(````(`````P```$B@`0!(H```%```````
M````````!`````````#G`````0```#``````````2*```#\`````````````
M``$````!````\`````,``'```````````(>@```U```````````````!````
L``````$````#``````````````"\H`````$``````````````0``````````
