"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DiagnosticsManager = void 0;
const utils_1 = require("../utils");
class DiagnosticsManager {
    constructor(sendDiagnostics, docManager, getDiagnostics) {
        this.sendDiagnostics = sendDiagnostics;
        this.docManager = docManager;
        this.getDiagnostics = getDiagnostics;
        this.pendingUpdates = new Set();
        this.scheduleUpdateAll = (0, utils_1.debounceThrottle)(() => this.updateAll(), 1000);
        this.scheduleBatchUpdate = (0, utils_1.debounceThrottle)(() => {
            this.pendingUpdates.forEach((doc) => {
                this.update(doc);
            });
            this.pendingUpdates.clear();
        }, 750);
    }
    updateAll() {
        this.docManager.getAllOpenedByClient().forEach((doc) => {
            this.update(doc[1]);
        });
        this.pendingUpdates.clear();
    }
    async update(document) {
        const diagnostics = await this.getDiagnostics({ uri: document.getURL() });
        this.sendDiagnostics({
            uri: document.getURL(),
            diagnostics
        });
    }
    removeDiagnostics(document) {
        this.pendingUpdates.delete(document);
        this.sendDiagnostics({
            uri: document.getURL(),
            diagnostics: []
        });
    }
    scheduleUpdate(document) {
        if (!this.docManager.isOpenedInClient(document.getURL())) {
            return;
        }
        this.pendingUpdates.add(document);
        this.scheduleBatchUpdate();
    }
}
exports.DiagnosticsManager = DiagnosticsManager;
//# sourceMappingURL=DiagnosticsManager.js.map