//// (c) 1992-2023 Intel Corporation.                            
// Intel, the Intel logo, Intel, MegaCore, NIOS II, Quartus and TalkBack words    
// and logos are trademarks of Intel Corporation or its subsidiaries in the U.S.  
// and/or other countries. Other marks and brands may be claimed as the property  
// of others. See Trademarks on intel.com for full list of Intel trademarks or    
// the Trademarks & Brands Names Database (if Intel) or See www.Intel.com/legal (if Altera) 
// Your use of Intel Corporation's design tools, logic functions and other        
// software and tools, and its AMPP partner logic functions, and any output       
// files any of the foregoing (including device programming or simulation         
// files), and any associated documentation or information are expressly subject  
// to the terms and conditions of the Altera Program License Subscription         
// Agreement, Intel MegaCore Function License Agreement, or other applicable      
// license agreement, including, without limitation, that your use is for the     
// sole purpose of programming logic devices manufactured by Intel and sold by    
// Intel or its authorized distributors.  Please refer to the applicable          
// agreement for further details.                                                 


`default_nettype none

module acl_printf_buffer_address_generator #(
parameter ASYNC_RESET=1,                              // set to '1' to consume the incoming reset signal asynchronously (use ACLR port on registers), '0' to use synchronous reset (SCLR port on registers)
parameter SYNCHRONIZE_RESET=0                         // set to '1' to pass the incoming reset signal through a synchronizer before use
) (
// Standard global signals
input wire clock,
input wire resetn,

// Upstream interface
output wire o_stall,
input wire i_valid,
input wire i_predicate,
input wire [31:0] i_increment,

// Downstream interface
input wire i_stall,
output reg o_valid,
output reg [63:0] o_result,

// Avalon interface
output wire avm_enable,
output wire avm_read,
output wire avm_write,
output wire [5:0] avm_burstcount,
output wire [31:0] avm_address,
output wire [255:0] avm_writedata,
output wire [31:0] avm_byteenable,
input wire avm_waitrequest,
input wire [255:0] avm_readdata,
input wire avm_readdatavalid,
input wire avm_writeack
);

reg busy; // sent a request, waiting for response

wire predicated_read;
wire down_stall;

   localparam                    NUM_RESET_COPIES = 1;
   localparam                    RESET_PIPE_DEPTH = 3;
   logic                         aclrn;
   logic [NUM_RESET_COPIES-1:0]  sclrn;
   logic                         resetn_synchronized;
   acl_reset_handler #(
      .ASYNC_RESET            (ASYNC_RESET),
      .USE_SYNCHRONIZER       (SYNCHRONIZE_RESET),
      .SYNCHRONIZE_ACLRN      (SYNCHRONIZE_RESET),
      .PIPE_DEPTH             (RESET_PIPE_DEPTH),
      .NUM_COPIES             (NUM_RESET_COPIES)
   ) acl_reset_handler_inst (
      .clk                    (clock),
      .i_resetn               (resetn),
      .o_aclrn                (aclrn),
      .o_sclrn                (sclrn),
      .o_resetn_synchronized  (resetn_synchronized)
   );


assign avm_enable = 1'b1;

// stalled from down-stream when output is valid and stall is high
assign down_stall = ( o_valid & i_stall ); 

/**************
* AVM Signals *
**************/

// send a interconnect request if not stalled down-stream, not busy, nor predicated 
assign avm_read = ( i_valid & ~down_stall & ~busy & ~i_predicate); 
assign avm_address = i_increment;

assign avm_write = 1'b0;
assign avm_writedata = 256'b0;
assign avm_byteenable = 32'b1;
assign avm_burstcount = 1;

/****************
* Other signals *
****************/

// act on predicated inputs (o_valid is high next cycle, result does not matter)
// if not stalled down-stream, not busy and predicate is high
assign predicated_read = ( i_valid & ~down_stall & ~busy & i_predicate ); 

/*******************
* Valid/Stall Logic *
********************/

 always @( posedge clock or negedge aclrn ) begin

   if( ~aclrn ) begin
     o_valid <= 1'b0;
     busy <= 1'b0;
     o_result <= 64'dx;
   end
   else begin

     // busy when read signal is sent, not-busy after readdatavalid is received
     busy <= (busy & ~avm_readdatavalid) | (~busy & avm_read & ~avm_waitrequest);

     // dont change the output, if I get stalled by down-stream
     if( ~down_stall )
     begin     
       // readdatavalid stays high only one cycle
       o_valid <= avm_readdatavalid | predicated_read;

       if( predicated_read ) begin
         o_result <= 64'b0; // assign 0 so I can see change when debugging
       end
       else begin
         o_result <= avm_readdata[63:0];
       end
     end
     
     if (~sclrn[0]) begin
       o_valid <= 1'b0;
       busy <= 1'b0;
     end
     
   end
end

// stall up-stream if I get stalled by down-stream, or by the IC, or just busy servicing a request
assign o_stall = ( down_stall | ( avm_read & avm_waitrequest ) | busy );

endmodule

`default_nettype wire
