"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDateHistogramBucketAgg = void 0;
exports.isDateHistogramBucketAggConfig = isDateHistogramBucketAggConfig;

var _lodash = require("lodash");

var _momentTimezone = _interopRequireDefault(require("moment-timezone"));

var _i18n = require("@osd/i18n");

var _common = require("../../../../common");

var _interval_options = require("./_interval_options");

var _date_histogram = require("./create_filter/date_histogram");

var _bucket_agg_type = require("./bucket_agg_type");

var _bucket_agg_types = require("./bucket_agg_types");

var _time_buckets = require("./lib/time_buckets");

var _agg_params = require("../agg_params");

var _metric_agg_type = require("../metrics/metric_agg_type");

var _utils = require("../utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const updateTimeBuckets = (agg, calculateBounds, customBuckets) => {
  const bounds = agg.params.timeRange && (agg.fieldIsTimeField() || (0, _interval_options.isAutoInterval)(agg.params.interval)) ? calculateBounds(agg.params.timeRange) : undefined;
  const buckets = customBuckets || agg.buckets;
  buckets.setBounds(bounds);
  buckets.setInterval(agg.params.interval);
};

function isDateHistogramBucketAggConfig(agg) {
  return Boolean(agg.buckets);
}

const getDateHistogramBucketAgg = ({
  calculateBounds,
  isDefaultTimezone,
  getConfig
}) => new _bucket_agg_type.BucketAggType({
  name: _bucket_agg_types.BUCKET_TYPES.DATE_HISTOGRAM,
  title: _i18n.i18n.translate('data.search.aggs.buckets.dateHistogramTitle', {
    defaultMessage: 'Date Histogram'
  }),
  ordered: {
    date: true
  },

  makeLabel(agg) {
    let output = {};

    if (this.params) {
      output = (0, _agg_params.writeParams)(this.params, agg);
    }

    const field = agg.getFieldDisplayName();
    return _i18n.i18n.translate('data.search.aggs.buckets.dateHistogramLabel', {
      defaultMessage: '{fieldName} per {intervalDescription}',
      values: {
        fieldName: field,
        intervalDescription: output.metricScaleText || output.bucketInterval.description
      }
    });
  },

  createFilter: _date_histogram.createFilterDateHistogram,

  decorateAggConfig() {
    let buckets;
    return {
      buckets: {
        configurable: true,

        get() {
          if (buckets) return buckets;
          buckets = new _time_buckets.TimeBuckets({
            'histogram:maxBars': getConfig(_common.UI_SETTINGS.HISTOGRAM_MAX_BARS),
            'histogram:barTarget': getConfig(_common.UI_SETTINGS.HISTOGRAM_BAR_TARGET),
            dateFormat: getConfig('dateFormat'),
            'dateFormat:scaled': getConfig('dateFormat:scaled')
          });
          updateTimeBuckets(this, calculateBounds, buckets);
          return buckets;
        }

      }
    };
  },

  getSerializedFormat(agg) {
    return {
      id: 'date',
      params: {
        pattern: agg.buckets.getScaledDateFormat()
      }
    };
  },

  params: [{
    name: 'field',
    type: 'field',
    filterFieldTypes: _common.OSD_FIELD_TYPES.DATE,

    default(agg) {
      return agg.getIndexPattern().timeFieldName;
    },

    onChange(agg) {
      if ((0, _interval_options.isAutoInterval)((0, _lodash.get)(agg, 'params.interval')) && !agg.fieldIsTimeField()) {
        delete agg.params.interval;
      }
    }

  }, {
    name: 'timeRange',
    default: null,
    write: _lodash.noop
  }, {
    name: 'useNormalizedOpenSearchInterval',
    default: true,
    write: _lodash.noop
  }, {
    name: 'scaleMetricValues',
    default: false,
    write: _lodash.noop,
    advanced: true
  }, {
    name: 'interval',

    deserialize(state, agg) {
      // For upgrading from 7.0.x to 7.1.x - intervals are now stored as key of options or custom value
      if (state === 'custom') {
        return (0, _lodash.get)(agg, 'params.customInterval');
      }

      const interval = (0, _lodash.find)(_interval_options.intervalOptions, {
        val: state
      }); // For upgrading from 4.0.x to 4.1.x - intervals are now stored as 'y' instead of 'year',
      // but this maps the old values to the new values

      if (!interval && state === 'year') {
        return 'y';
      }

      return state;
    },

    default: _interval_options.autoInterval,
    options: _interval_options.intervalOptions,

    write(agg, output, aggs) {
      updateTimeBuckets(agg, calculateBounds);
      const {
        useNormalizedOpenSearchInterval,
        scaleMetricValues
      } = agg.params;
      const interval = agg.buckets.getInterval(useNormalizedOpenSearchInterval);
      output.bucketInterval = interval;

      if (interval.expression === '0ms') {
        // We are hitting this code a couple of times while configuring in editor
        // with an interval of 0ms because the overall time range has not yet been
        // set. Since 0ms is not a valid OpenSearch interval, we cannot pass it through dateHistogramInterval
        // below, since it would throw an exception. So in the cases we still have an interval of 0ms
        // here we simply skip the rest of the method and never write an interval into the DSL, since
        // this DSL will anyway not be used before we're passing this code with an actual interval.
        return;
      }

      output.params = { ...output.params,
        ...(0, _utils.dateHistogramInterval)(interval.expression)
      };
      const scaleMetrics = scaleMetricValues && interval.scaled && interval.scale && interval.scale < 1;

      if (scaleMetrics && aggs) {
        const metrics = aggs.aggs.filter(a => (0, _metric_agg_type.isMetricAggType)(a.type));
        const all = (0, _lodash.every)(metrics, a => {
          const {
            type
          } = a;

          if ((0, _metric_agg_type.isMetricAggType)(type)) {
            return type.isScalable();
          }
        });

        if (all) {
          var _interval$preScaled;

          output.metricScale = interval.scale;
          output.metricScaleText = ((_interval$preScaled = interval.preScaled) === null || _interval$preScaled === void 0 ? void 0 : _interval$preScaled.description) || '';
        }
      }
    }

  }, {
    name: 'time_zone',
    default: undefined,
    // We don't ever want this parameter to be serialized out (when saving or to URLs)
    // since we do all the logic handling it "on the fly" in the `write` method, to prevent
    // time_zones being persisted into saved_objects
    serialize: _lodash.noop,

    write(agg, output) {
      // If a time_zone has been set explicitly always prefer this.
      let tz = agg.params.time_zone;

      if (!tz && agg.params.field) {
        // If a field has been configured check the index pattern's typeMeta if a date_histogram on that
        // field requires a specific time_zone
        tz = (0, _lodash.get)(agg.getIndexPattern(), ['typeMeta', 'aggs', 'date_histogram', agg.params.field.name, 'time_zone']);
      }

      if (!tz) {
        // If the index pattern typeMeta data, didn't had a time zone assigned for the selected field use the configured tz
        const detectedTimezone = _momentTimezone.default.tz.guess();

        const tzOffset = (0, _momentTimezone.default)().format('Z');
        tz = isDefaultTimezone() ? detectedTimezone || tzOffset : getConfig('dateFormat:tz');
      }

      output.params.time_zone = tz;
    }

  }, {
    name: 'drop_partials',
    default: false,
    write: _lodash.noop,
    shouldShow: agg => {
      const field = agg.params.field;
      return field && field.name && field.name === agg.getIndexPattern().timeFieldName;
    }
  }, {
    name: 'format'
  }, {
    name: 'min_doc_count',
    default: 1
  }, {
    name: 'extended_bounds',
    default: {},

    write(agg, output) {
      const val = agg.params.extended_bounds;

      if (val.min != null || val.max != null) {
        output.params.extended_bounds = {
          min: (0, _momentTimezone.default)(val.min).valueOf(),
          max: (0, _momentTimezone.default)(val.max).valueOf()
        };
        return;
      }
    }

  }]
});

exports.getDateHistogramBucketAgg = getDateHistogramBucketAgg;