"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.visualReportSchema = exports.triggerSchema = exports.scheduleSchema = exports.reportSchema = exports.reportParamsSchema = exports.reportDefinitionSchema = exports.opensearchDashboardsUserSchema = exports.intervalSchema = exports.deliverySchema = exports.dataReportSchema = exports.cronSchema = exports.channelSchema = void 0;

var _configSchema = require("@osd/config-schema");

var _validationHelper = require("../utils/validationHelper");

var _cronValidator = require("cron-validator");

var _constants = require("../routes/utils/constants");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const dataReportSchema = _configSchema.schema.object({
  // Need this to build the links in email
  origin: _configSchema.schema.uri(),
  //e.g. https://xxxxx.com
  base_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },

    minLength: 1
  }),
  saved_search_id: _configSchema.schema.string(),
  //ISO duration format. 'PT10M' means 10 min
  time_duration: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexDuration.test(value)) {
        return `invalid time duration: ${value}`;
      }
    }

  }),
  //TODO: future support schema.literal('xlsx')
  report_format: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.FORMAT.csv)]),
  limit: _configSchema.schema.number({
    defaultValue: _constants.DEFAULT_MAX_SIZE,
    min: 0
  }),
  excel: _configSchema.schema.boolean({
    defaultValue: true
  })
});

exports.dataReportSchema = dataReportSchema;

const visualReportSchema = _configSchema.schema.object({
  // Need this to build the links in email
  origin: _configSchema.schema.uri(),
  //e.g. https://xxxxx.com
  base_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },

    minLength: 1
  }),
  window_width: _configSchema.schema.number({
    defaultValue: 1600,
    min: 0
  }),
  window_height: _configSchema.schema.number({
    defaultValue: 800,
    min: 0
  }),
  report_format: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.FORMAT.pdf), _configSchema.schema.literal(_constants.FORMAT.png)]),
  header: _configSchema.schema.maybe(_configSchema.schema.string()),
  footer: _configSchema.schema.maybe(_configSchema.schema.string()),
  time_duration: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexDuration.test(value)) {
        return `invalid time duration: ${value}`;
      }
    }

  })
});

exports.visualReportSchema = visualReportSchema;

const intervalSchema = _configSchema.schema.object({
  interval: _configSchema.schema.object({
    period: _configSchema.schema.number({
      min: 0
    }),
    // Refer to job scheduler SPI https://github.com/opensearch-project/job-scheduler/blob/main/spi/src/main/java/org/opensearch/jobscheduler/spi/schedule/IntervalSchedule.java
    unit: _configSchema.schema.oneOf([_configSchema.schema.literal('MINUTES'), _configSchema.schema.literal('HOURS'), _configSchema.schema.literal('DAYS'), // Job scheduler in reporting OpenSearch plugin always saves as following format
    _configSchema.schema.literal('Minutes'), _configSchema.schema.literal('Hours'), _configSchema.schema.literal('Days')]),
    // timestamp
    start_time: _configSchema.schema.number()
  })
});

exports.intervalSchema = intervalSchema;

const cronSchema = _configSchema.schema.object({
  cron: _configSchema.schema.object({
    expression: _configSchema.schema.string({
      validate(value) {
        if (!(0, _cronValidator.isValidCron)(value)) {
          return `invalid cron expression: ${value}`;
        }
      }

    }),
    //TODO: add more validation once we add full support of timezone
    timezone: _configSchema.schema.string()
  })
});

exports.cronSchema = cronSchema;

const scheduleSchema = _configSchema.schema.object({
  schedule_type: _configSchema.schema.oneOf([
  /*
  TODO: Future Date option will be added in the future.
  Currently @osd/config-schema has no support for more than 2 conditions, keep an eye on library update
  */
  _configSchema.schema.literal(_constants.SCHEDULE_TYPE.recurring), _configSchema.schema.literal(_constants.SCHEDULE_TYPE.cron)]),
  schedule: _configSchema.schema.conditional(_configSchema.schema.siblingRef('schedule_type'), _constants.SCHEDULE_TYPE.recurring, intervalSchema, cronSchema),
  enabled_time: _configSchema.schema.number(),
  enabled: _configSchema.schema.boolean()
});

exports.scheduleSchema = scheduleSchema;

const opensearchDashboardsUserSchema = _configSchema.schema.object({
  opensearch_dashboards_recipients: _configSchema.schema.arrayOf(_configSchema.schema.string())
});

exports.opensearchDashboardsUserSchema = opensearchDashboardsUserSchema;

const channelSchema = _configSchema.schema.object({
  recipients: _configSchema.schema.arrayOf(_configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexEmailAddress.test(value)) {
        return `invalid email address ${value}`;
      }
    }

  }), {
    minSize: 1
  }),
  title: _configSchema.schema.string(),
  textDescription: _configSchema.schema.string(),
  htmlDescription: _configSchema.schema.maybe(_configSchema.schema.string()),
  configIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
});

exports.channelSchema = channelSchema;

const triggerSchema = _configSchema.schema.object({
  trigger_type: _configSchema.schema.oneOf([
  /*
    TODO: Alerting will be added in the future.
    Currently @osd/config-schema has no support for more than 2 conditions, keep an eye on library update
  */
  _configSchema.schema.literal(_constants.TRIGGER_TYPE.schedule), _configSchema.schema.literal(_constants.TRIGGER_TYPE.onDemand)]),
  trigger_params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('trigger_type'), _constants.TRIGGER_TYPE.onDemand, _configSchema.schema.never(), scheduleSchema)
});

exports.triggerSchema = triggerSchema;

const deliverySchema = _configSchema.schema.object({
  configIds: _configSchema.schema.arrayOf(_configSchema.schema.string()),
  title: _configSchema.schema.string(),
  textDescription: _configSchema.schema.string(),
  htmlDescription: _configSchema.schema.string()
});

exports.deliverySchema = deliverySchema;

const reportParamsSchema = _configSchema.schema.object({
  report_name: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexReportName.test(value)) {
        return `invald report name ${value}.\nMust be non-empty, allow a-z, A-Z, 0-9, (), [], ',' - and _ and spaces`;
      }
    }

  }),
  report_source: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_TYPE.dashboard), _configSchema.schema.literal(_constants.REPORT_TYPE.visualization), _configSchema.schema.literal(_constants.REPORT_TYPE.savedSearch), _configSchema.schema.literal(_constants.REPORT_TYPE.notebook)]),
  description: _configSchema.schema.string(),
  core_params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('report_source'), _constants.REPORT_TYPE.savedSearch, dataReportSchema, visualReportSchema)
});

exports.reportParamsSchema = reportParamsSchema;

const reportDefinitionSchema = _configSchema.schema.object({
  report_params: reportParamsSchema,
  delivery: deliverySchema,
  trigger: triggerSchema,
  time_created: _configSchema.schema.maybe(_configSchema.schema.number()),
  last_updated: _configSchema.schema.maybe(_configSchema.schema.number()),
  status: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_DEFINITION_STATUS.active), _configSchema.schema.literal(_constants.REPORT_DEFINITION_STATUS.disabled)]))
});

exports.reportDefinitionSchema = reportDefinitionSchema;

const reportSchema = _configSchema.schema.object({
  query_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },

    minLength: 1
  }),
  time_from: _configSchema.schema.number(),
  time_to: _configSchema.schema.number(),
  report_definition: reportDefinitionSchema,
  time_created: _configSchema.schema.maybe(_configSchema.schema.number()),
  last_updated: _configSchema.schema.maybe(_configSchema.schema.number()),
  state: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_STATE.created), _configSchema.schema.literal(_constants.REPORT_STATE.error), _configSchema.schema.literal(_constants.REPORT_STATE.pending), _configSchema.schema.literal(_constants.REPORT_STATE.shared)]))
});

exports.reportSchema = reportSchema;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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