"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToCSV = exports.buildRequestBody = void 0;

var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));

var _json2Csv = _interopRequireDefault(require("json-2-csv"));

var _lodash = _interopRequireDefault(require("lodash"));

var _moment = _interopRequireDefault(require("moment"));

var _common = require("../../../../../src/plugins/data/common");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
var metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
}; // Get the selected columns by the user.

exports.metaData = metaData;

const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;

  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }

  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
}; // Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query


exports.getSelectedFields = getSelectedFields;

const buildRequestBody = (report, allowLeadingWildcards, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();

  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const savedObjectConfig = {
    allowLeadingWildcards: allowLeadingWildcards,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter, savedObjectConfig); // Add time range

  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }

  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  } // Add sorting to the query


  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);

  if (report._source.sorting.length > 0) {
    const sortings = report._source.sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });

    esbSearchQuery.sorts(sortings);
  } // add selected fields to query


  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  } // Add a customizer to merge queries to generate request body


  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });

  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
}; // Fetch the data from OpenSearch


exports.buildRequestBody = buildRequestBody;

const getOpenSearchData = (arrayHits, report, params, dateFormat) => {
  let hits = [];

  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields; // get all the fields of type date and format them to excel format

      for (let dateField of report._source.dateFields) {
        const dateValue = data._source[dateField];

        if (dateValue && dateValue.length !== 0) {
          if (dateValue instanceof Array) {
            // loop through array
            dateValue.forEach((element, index) => {
              data._source[dateField][index] = (0, _moment.default)(fields[dateField][index]).format(dateFormat);
            });
          } else {
            // The fields response always returns an array of values for each field
            // https://www.elastic.co/guide/en/elasticsearch/reference/master/search-fields.html#search-fields-response
            data._source[dateField] = (0, _moment.default)(fields[dateField][0]).format(dateFormat);
          }
        }
      }

      delete data['fields'];

      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        hits.push(params.excel ? sanitize(data) : data);
      } // Truncate to expected limit size


      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }

  return hits;
}; //Convert the data to Csv format


exports.getOpenSearchData = getOpenSearchData;

const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};

exports.convertToCSV = convertToCSV;

function flattenHits(hits, result = {}, prefix = '') {
  for (const [key, value] of Object.entries(hits)) {
    if (!hits.hasOwnProperty(key)) continue;

    if (value != null && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenHits(value, result, prefix + key + '.');
    } else {
      result[prefix.replace(/^_source\./, '') + key] = value;
    }
  }

  return result;
} //Return only the selected fields


function traverse(data, keys, result = {}) {
  data = flattenHits(data);
  const sourceKeys = Object.keys(data);
  keys.forEach(key => {
    const value = _lodash.default.get(data, key, undefined);

    if (value !== undefined) result[key] = value;else {
      Object.keys(data).filter(sourceKey => sourceKey.startsWith(key + '.')).forEach(sourceKey => result[sourceKey] = data[sourceKey]);
    }
  });
  return result;
}
/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */


function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;

    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }

  return doc;
}

const addDocValueFields = (report, requestBody) => {
  const docValues = [];

  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  } // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.


  requestBody = { ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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