"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Integration = void 0;

var fs = _interopRequireWildcard(require("fs/promises"));

var _path = _interopRequireDefault(require("path"));

var _validators = require("../validators");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/**
 * Helper function to compare version numbers.
 * Assumes that the version numbers are valid, produces undefined behavior otherwise.
 *
 * @param a Left-hand number
 * @param b Right-hand number
 * @returns -1 if a > b, 1 if a < b, 0 otherwise.
 */
function compareVersions(a, b) {
  const aParts = a.split('.').map(Number.parseInt);
  const bParts = b.split('.').map(Number.parseInt);

  for (let i = 0; i < Math.max(aParts.length, bParts.length); i++) {
    const aValue = i < aParts.length ? aParts[i] : 0;
    const bValue = i < bParts.length ? bParts[i] : 0;

    if (aValue > bValue) {
      return -1; // a > b
    } else if (aValue < bValue) {
      return 1; // a < b
    }
  }

  return 0; // a == b
}
/**
 * Helper function to check if the given path is a directory
 *
 * @param dirPath The directory to check.
 * @returns True if the path is a directory.
 */


async function isDirectory(dirPath) {
  try {
    const stats = await fs.stat(dirPath);
    return stats.isDirectory();
  } catch {
    return false;
  }
}
/**
 * Helper function to log validation errors.
 * Relies on the `ajv` package for validation error logs..
 *
 * @param integration The name of the component that failed validation.
 * @param validator A failing ajv validator.
 */


function logValidationErrors(integration, validator) {
  var _validator$errors;

  const errors = (_validator$errors = validator.errors) === null || _validator$errors === void 0 ? void 0 : _validator$errors.map(e => e.message);
  console.error(`Validation errors in ${integration}`, errors);
}
/**
 * The Integration class represents the data for Integration Templates.
 * It is backed by the repository file system.
 * It includes accessor methods for integration configs, as well as helpers for nested components.
 */


class Integration {
  constructor(directory) {
    _defineProperty(this, "directory", void 0);

    _defineProperty(this, "name", void 0);

    this.directory = directory;
    this.name = _path.default.basename(directory);
  }
  /**
   * Check the integration for validity.
   * This is not a deep check, but a quick check to verify that the integration is a valid directory and has a config file.
   *
   * @returns true if the integration is valid.
   */


  async check() {
    if (!(await isDirectory(this.directory))) {
      return false;
    }

    return (await this.getConfig()) !== null;
  }
  /**
   * Like check(), but thoroughly checks all nested integration dependencies.
   *
   * @returns true if the integration is valid.
   */


  async deepCheck() {
    if (!(await this.check())) {
      console.error('check failed');
      return false;
    }

    try {
      // An integration must have at least one mapping
      const schemas = await this.getSchemas();

      if (Object.keys(schemas.mappings).length === 0) {
        return false;
      } // An integration must have at least one asset


      const assets = await this.getAssets();

      if (Object.keys(assets).length === 0) {
        return false;
      }
    } catch (err) {
      // Any loading errors are considered invalid
      console.error('Deep check failed for exception', err);
      return false;
    }

    return true;
  }
  /**
   * Get the latest version of the integration available.
   * This method relies on the fact that integration configs have their versions in their name.
   * Any files that don't match the config naming convention will be ignored.
   *
   * @returns A string with the latest version, or null if no versions are available.
   */


  async getLatestVersion() {
    const files = await fs.readdir(this.directory);
    const versions = [];

    for (const file of files) {
      if (_path.default.extname(file) === '.json' && file.startsWith(`${this.name}-`)) {
        const version = file.substring(this.name.length + 1, file.length - 5);

        if (!version.match(/^\d+(\.\d+)*$/)) {
          continue;
        }

        versions.push(version);
      }
    }

    versions.sort((a, b) => compareVersions(a, b));
    return versions.length > 0 ? versions[0] : null;
  }
  /**
   * Get the configuration of the current integration.
   *
   * @param version The version of the config to retrieve.
   * @returns The config if a valid config matching the version is present, otherwise null.
   */


  async getConfig(version) {
    const maybeVersion = version ? version : await this.getLatestVersion();

    if (maybeVersion === null) {
      return null;
    }

    const configFile = `${this.name}-${maybeVersion}.json`;

    const configPath = _path.default.join(this.directory, configFile);

    try {
      const config = await fs.readFile(configPath, {
        encoding: 'utf-8'
      });
      const possibleTemplate = JSON.parse(config);

      if (!(0, _validators.templateValidator)(possibleTemplate)) {
        logValidationErrors(configFile, _validators.templateValidator);
        return null;
      }

      return possibleTemplate;
    } catch (err) {
      if (err instanceof SyntaxError) {
        console.error(`Syntax errors in ${configFile}`, err);
        return null;
      }

      if (err instanceof Error && err.code === 'ENOENT') {
        console.error(`Attempted to retrieve non-existent config ${configFile}`);
        return null;
      }

      throw new Error('Could not load integration', {
        cause: err
      });
    }
  }
  /**
   * Retrieve assets associated with the integration.
   * This method greedily retrieves all assets.
   * If the version is invalid, an error is thrown.
   * If an asset is invalid, it will be skipped.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */


  async getAssets(version) {
    const config = await this.getConfig(version);

    if (config === null) {
      return Promise.reject(new Error('Attempted to get assets of invalid config'));
    }

    const result = {};

    if (config.assets.savedObjects) {
      const sobjPath = _path.default.join(this.directory, 'assets', `${config.assets.savedObjects.name}-${config.assets.savedObjects.version}.ndjson`);

      try {
        const ndjson = await fs.readFile(sobjPath, {
          encoding: 'utf-8'
        });
        const asJson = '[' + ndjson.replace(/\n/g, ',') + ']';
        const parsed = JSON.parse(asJson);
        result.savedObjects = parsed;
      } catch (err) {
        console.error("Failed to load saved object assets, proceeding as if it's absent", err);
      }
    }

    return result;
  }
  /**
   * Retrieve sample data associated with the integration.
   * If the version is invalid, an error is thrown.
   * If the sample data is invalid, null will be returned
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing a list of sample data with adjusted timestamps.
   */


  async getSampleData(version) {
    const config = await this.getConfig(version);

    if (config === null) {
      return Promise.reject(new Error('Attempted to get assets of invalid config'));
    }

    const result = {
      sampleData: null
    };

    if (config.sampleData) {
      var _config$sampleData;

      const sobjPath = _path.default.join(this.directory, 'data', (_config$sampleData = config.sampleData) === null || _config$sampleData === void 0 ? void 0 : _config$sampleData.path);

      try {
        const jsonContent = await fs.readFile(sobjPath, {
          encoding: 'utf-8'
        });
        const parsed = JSON.parse(jsonContent);

        for (const value of parsed) {
          if (!('@timestamp' in value)) {
            continue;
          } // Randomly scatter timestamps across last 10 minutes
          // Assume for now that the ordering of events isn't important, can change to a sequence if needed
          // Also doesn't handle fields like `observedTimestamp` if present


          Object.assign(value, {
            '@timestamp': new Date(Date.now() - Math.floor(Math.random() * 1000 * 60 * 10)).toISOString()
          });
        }

        result.sampleData = parsed;
      } catch (err) {
        console.error("Failed to load saved object assets, proceeding as if it's absent", err);
      }
    }

    return result;
  }
  /**
   * Retrieve schema data associated with the integration.
   * This method greedily retrieves all mappings and schemas.
   * It's assumed that a valid version will be provided.
   * If the version is invalid, an error is thrown.
   * If a schema is invalid, an error will be thrown.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */


  async getSchemas(version) {
    const config = await this.getConfig(version);

    if (config === null) {
      return Promise.reject(new Error('Attempted to get assets of invalid config'));
    }

    const result = {
      mappings: {}
    };

    try {
      for (const component of config.components) {
        const schemaFile = `${component.name}-${component.version}.mapping.json`;
        const rawSchema = await fs.readFile(_path.default.join(this.directory, 'schemas', schemaFile), {
          encoding: 'utf-8'
        });
        const parsedSchema = JSON.parse(rawSchema);
        result.mappings[component.name] = parsedSchema;
      }
    } catch (err) {
      // It's not clear that an invalid schema can be recovered from.
      // For integrations to function, we need schemas to be valid.
      console.error('Error loading schema', err);
      return Promise.reject(new Error('Could not load schema', {
        cause: err
      }));
    }

    return result;
  }
  /**
   * Retrieves the data for a static file associated with the integration.
   *
   * @param staticPath The path of the static to retrieve.
   * @returns A buffer with the static's data if present, otherwise null.
   */


  async getStatic(staticPath) {
    const fullStaticPath = _path.default.join(this.directory, 'static', staticPath);

    try {
      return await fs.readFile(fullStaticPath);
    } catch (err) {
      if (err instanceof Error && err.code === 'ENOENT') {
        console.error(`Static not found: ${staticPath}`);
        return null;
      }

      throw err;
    }
  }

}

exports.Integration = Integration;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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