"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;
var _opensearch_security_client = require("../../backend/opensearch_security_client");
var _tenant_resolver = require("../../multitenancy/tenant_resolver");
var _errors = require("../../errors");
var _common = require("../../../common");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;
    _defineProperty(this, "type", void 0);
    _defineProperty(this, "securityClient", void 0);
    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;
      // skip auth for APIs that do not require auth
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      }
      const authState = {};

      // if browser request, auth logic is:
      //   1. check if request includes auth header or parameter(e.g. jwt in url params) is present, if so, authenticate with auth header.
      //   2. if auth header not present, check if auth cookie is present, if no cookie, send to authentication workflow
      //   3. verify whether auth cookie is valid, if not valid, send to authentication workflow
      //   4. if cookie is valid, pass to route handlers
      const authHeaders = {};
      let cookie;
      let authInfo;
      // if this is an REST API call, suppose the request includes necessary auth header
      // see https://www.elastic.co/guide/en/opensearch-dashboards/master/using-api.html
      if (this.requestIncludesAuthInfo(request)) {
        try {
          const additionalAuthHeader = await this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additionalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additionalAuthHeader);
          cookie = this.getCookie(request, authInfo);

          // set tenant from cookie if exist
          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();
          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // no auth header in request, try cookie
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }
        if (!cookie || !(await this.isValidCookie(cookie, request))) {
          // clear cookie
          this.sessionStorageFactory.asScoped(request).clear();

          // for assets, we can still pass it to resource handler as notHandled.
          // marking it as authenticated may result in login pop up when auth challenge
          // is enabled.
          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          }

          // send to auth workflow
          return this.handleUnauthedRequest(request, response, toolkit);
        }

        // extend session expiration time
        if (this.config.session.keepalive) {
          cookie.expiryTime = Date.now() + this.config.session.ttl;
          this.sessionStorageFactory.asScoped(request).set(cookie);
        }
        // cookie is valid
        // build auth header
        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie, request);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additionalAuthHeader = await this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additionalAuthHeader);
      }

      // resolve tenant if necessary
      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo);
          // return 401 if no tenant available
          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          }
          authState.selectedTenant = tenant;

          // set tenant in header
          if (this.config.multitenancy.enabled && this.config.multitenancy.enable_aggregation_view) {
            // Store all saved objects in a single kibana index.
            Object.assign(authHeaders, {
              securitytenant: _common.GLOBAL_TENANT_SYMBOL
            });
          } else {
            Object.assign(authHeaders, {
              securitytenant: tenant
            });
          }

          // set tenant to cookie
          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);
          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }
            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }
          throw error;
        }
      }
      if (!authInfo) {
        authInfo = await this.securityClient.authinfo(request, authHeaders);
      }
      authState.authInfo = authInfo;
      return toolkit.authenticated({
        requestHeaders: authHeaders,
        state: authState
      });
    });
    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
    this.config = config;
  }
  authNotRequired(request) {
    const pathname = request.url.pathname;
    if (!pathname) {
      return false;
    }
    // allow requests to ignored routes
    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    }
    // allow requests to routes that doesn't require authentication
    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }
    return false;
  }
  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }
    const dashboardsInfo = await this.securityClient.dashboardsinfo(request, authHeader);
    return (0, _tenant_resolver.resolveTenant)({
      request,
      username: authInfo.user_name,
      roles: authInfo.roles,
      availableTenants: authInfo.tenants,
      config: this.config,
      cookie,
      multitenancyEnabled: dashboardsInfo.multitenancy_enabled,
      privateTenantEnabled: dashboardsInfo.private_tenant_enabled,
      defaultTenant: dashboardsInfo.default_tenant
    });
  }
  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  }

  // abstract functions for concrete auth types to implement
}
exports.AuthenticationType = AuthenticationType;
_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/api/core/capabilities',
// FIXME: need to figure out how to bypass this API call
'/app/login']);
_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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