"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.addTenantParameterToResolvedShortLink = addTenantParameterToResolvedShortLink;
exports.isValidTenant = isValidTenant;
exports.resolve = resolve;
exports.resolveTenant = resolveTenant;
var _lodash = require("lodash");
var _std = require("@osd/std");
var _common = require("../../common");
var _router = require("../../../../src/core/server/http/router");
var _short_url_routes = require("../../../../src/plugins/share/common/short_url_routes");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

const PRIVATE_TENANTS = exports.PRIVATE_TENANTS = [_common.PRIVATE_TENANT_SYMBOL, 'private'];
const GLOBAL_TENANTS = exports.GLOBAL_TENANTS = ['global', _common.GLOBAL_TENANT_SYMBOL, 'Global'];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param username
 * @param roles
 * @param availableTenants
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param multitenancyEnabled
 * @param privateTenantEnabled
 * @param defaultTenant
 *
 * @returns user preferred tenant of the request.
 */
function resolveTenant({
  request,
  username,
  roles,
  availableTenants,
  config,
  cookie,
  multitenancyEnabled,
  privateTenantEnabled,
  defaultTenant
}) {
  var _request$url, _request$url2, _request$url3, _config$multitenancy, _config$multitenancy2;
  const DEFAULT_READONLY_ROLES = ['kibana_read_only'];
  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 || (_request$url = request.url) === null || _request$url === void 0 || (_request$url = _request$url.searchParams) === null || _request$url === void 0 ? void 0 : _request$url.get('securityTenant_');
  const securitytenant = request === null || request === void 0 || (_request$url2 = request.url) === null || _request$url2 === void 0 || (_request$url2 = _request$url2.searchParams) === null || _request$url2 === void 0 ? void 0 : _request$url2.get('securitytenant');
  // eslint-disable-next-line @typescript-eslint/naming-convention
  const security_tenant = request === null || request === void 0 || (_request$url3 = request.url) === null || _request$url3 === void 0 || (_request$url3 = _request$url3.searchParams) === null || _request$url3 === void 0 ? void 0 : _request$url3.get('security_tenant');
  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (security_tenant) {
    selectedTenant = security_tenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else if (defaultTenant && multitenancyEnabled) {
    selectedTenant = defaultTenant;
  } else {
    selectedTenant = undefined;
  }
  const isReadonly = roles === null || roles === void 0 ? void 0 : roles.some(role => {
    var _config$readonly_mode;
    return ((_config$readonly_mode = config.readonly_mode) === null || _config$readonly_mode === void 0 ? void 0 : _config$readonly_mode.roles.includes(role)) || DEFAULT_READONLY_ROLES.includes(role);
  });
  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = (_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global;
  return resolve(username, selectedTenant, preferredTenants, availableTenants, globalTenantEnabled, multitenancyEnabled, privateTenantEnabled);
}
function resolve(username, requestedTenant, preferredTenants, availableTenants,
// is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, multitenancyEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];
  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }
  if (!multitenancyEnabled) {
    if (!globalTenantEnabled) {
      return undefined;
    }
    return _common.GLOBAL_TENANT_SYMBOL;
  }
  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;
    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }
    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.PRIVATE_TENANT_SYMBOL;
    }
    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return _common.GLOBAL_TENANT_SYMBOL;
    }
  }
  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();
      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return _common.GLOBAL_TENANT_SYMBOL;
      }
      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return _common.PRIVATE_TENANT_SYMBOL;
      }
      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }
  if (globalTenantEnabled) {
    return _common.GLOBAL_TENANT_SYMBOL;
  }
  if (privateTenantEnabled) {
    return _common.PRIVATE_TENANT_SYMBOL;
  }

  /**
   * Fall back to the first tenant in the available tenants
   * Under the condition of enabling multitenancy, if the user has disabled both 'Global' and 'Private' tenants:
   * it will remove the default global tenant key for custom tenant.
   */
  if (Object.keys(availableTenantsClone).length > 1 && availableTenantsClone.hasOwnProperty(_common.globalTenantName)) {
    delete availableTenantsClone[_common.globalTenantName];
  }
  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}

/**
 * Return true if tenant parameter is a valid tenant.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */
function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}

/**
 * If multitenancy is enabled & the URL entered starts with /goto,
 * We will modify the rawResponse to add a new parameter to the URL, the security_tenant (or value for tenant when in multitenancy)
 * With the security_tenant added, the resolved short URL now contains the security_tenant information (so the short URL retains the tenant information).
 **/

function addTenantParameterToResolvedShortLink(request) {
  if (request.url.pathname.startsWith(`${_short_url_routes.GOTO_PREFIX}/`)) {
    const rawRequest = (0, _router.ensureRawRequest)(request);
    const rawResponse = rawRequest.response;
    const responsePath = rawResponse.headers.location;

    // Make sure the request really should redirect
    if (rawResponse.headers.location && !responsePath.includes('security_tenant') && request.headers.securitytenant) {
      // Mutating the headers toolkit.next({headers: ...}) logs a warning about headers being overwritten
      rawResponse.headers.location = (0, _std.modifyUrl)(responsePath, parts => {
        if (parts.query.security_tenant === undefined) {
          parts.query.security_tenant = request.headers.securitytenant;
        }
      });
    }
  }
  return request;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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