"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.visualReportSchema = exports.triggerSchema = exports.scheduleSchema = exports.reportSchema = exports.reportParamsSchema = exports.reportDefinitionSchema = exports.opensearchDashboardsUserSchema = exports.intervalSchema = exports.deliverySchema = exports.dataReportSchema = exports.cronSchema = exports.channelSchema = void 0;
var _configSchema = require("@osd/config-schema");
var _validationHelper = require("../utils/validationHelper");
var _cronValidator = require("cron-validator");
var _constants = require("../routes/utils/constants");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const dataReportSchema = exports.dataReportSchema = _configSchema.schema.object({
  // Need this to build the links in email
  origin: _configSchema.schema.uri(),
  //e.g. https://xxxxx.com
  base_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  saved_search_id: _configSchema.schema.string(),
  //ISO duration format. 'PT10M' means 10 min
  time_duration: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexDuration.test(value)) {
        return `invalid time duration: ${value}`;
      }
    }
  }),
  //TODO: future support schema.literal('xlsx')
  report_format: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.FORMAT.csv)]),
  limit: _configSchema.schema.number({
    defaultValue: _constants.DEFAULT_MAX_SIZE,
    min: 0
  }),
  excel: _configSchema.schema.boolean({
    defaultValue: true
  })
});
const visualReportSchema = exports.visualReportSchema = _configSchema.schema.object({
  // Need this to build the links in email
  origin: _configSchema.schema.uri(),
  //e.g. https://xxxxx.com
  base_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  window_width: _configSchema.schema.number({
    defaultValue: 1600,
    min: 0
  }),
  window_height: _configSchema.schema.number({
    defaultValue: 800,
    min: 0
  }),
  report_format: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.FORMAT.pdf), _configSchema.schema.literal(_constants.FORMAT.png)]),
  header: _configSchema.schema.maybe(_configSchema.schema.string()),
  footer: _configSchema.schema.maybe(_configSchema.schema.string()),
  time_duration: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexDuration.test(value)) {
        return `invalid time duration: ${value}`;
      }
    }
  })
});
const intervalSchema = exports.intervalSchema = _configSchema.schema.object({
  interval: _configSchema.schema.object({
    period: _configSchema.schema.number({
      min: 0
    }),
    // Refer to job scheduler SPI https://github.com/opensearch-project/job-scheduler/blob/main/spi/src/main/java/org/opensearch/jobscheduler/spi/schedule/IntervalSchedule.java
    unit: _configSchema.schema.oneOf([_configSchema.schema.literal('MINUTES'), _configSchema.schema.literal('HOURS'), _configSchema.schema.literal('DAYS'),
    // Job scheduler in reporting OpenSearch plugin always saves as following format
    _configSchema.schema.literal('Minutes'), _configSchema.schema.literal('Hours'), _configSchema.schema.literal('Days')]),
    // timestamp
    start_time: _configSchema.schema.number()
  })
});
const cronSchema = exports.cronSchema = _configSchema.schema.object({
  cron: _configSchema.schema.object({
    expression: _configSchema.schema.string({
      validate(value) {
        if (!(0, _cronValidator.isValidCron)(value)) {
          return `invalid cron expression: ${value}`;
        }
      }
    }),
    //TODO: add more validation once we add full support of timezone
    timezone: _configSchema.schema.string()
  })
});
const scheduleSchema = exports.scheduleSchema = _configSchema.schema.object({
  schedule_type: _configSchema.schema.oneOf([
  /*
  TODO: Future Date option will be added in the future.
  Currently @osd/config-schema has no support for more than 2 conditions, keep an eye on library update
  */
  _configSchema.schema.literal(_constants.SCHEDULE_TYPE.recurring), _configSchema.schema.literal(_constants.SCHEDULE_TYPE.cron)]),
  schedule: _configSchema.schema.conditional(_configSchema.schema.siblingRef('schedule_type'), _constants.SCHEDULE_TYPE.recurring, intervalSchema, cronSchema),
  enabled_time: _configSchema.schema.number(),
  enabled: _configSchema.schema.boolean()
});
const opensearchDashboardsUserSchema = exports.opensearchDashboardsUserSchema = _configSchema.schema.object({
  opensearch_dashboards_recipients: _configSchema.schema.arrayOf(_configSchema.schema.string())
});
const channelSchema = exports.channelSchema = _configSchema.schema.object({
  recipients: _configSchema.schema.arrayOf(_configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexEmailAddress.test(value)) {
        return `invalid email address ${value}`;
      }
    }
  }), {
    minSize: 1
  }),
  title: _configSchema.schema.string(),
  textDescription: _configSchema.schema.string(),
  htmlDescription: _configSchema.schema.maybe(_configSchema.schema.string()),
  configIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
});
const triggerSchema = exports.triggerSchema = _configSchema.schema.object({
  trigger_type: _configSchema.schema.oneOf([
  /*
    TODO: Alerting will be added in the future.
    Currently @osd/config-schema has no support for more than 2 conditions, keep an eye on library update
  */
  _configSchema.schema.literal(_constants.TRIGGER_TYPE.schedule), _configSchema.schema.literal(_constants.TRIGGER_TYPE.onDemand)]),
  trigger_params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('trigger_type'), _constants.TRIGGER_TYPE.onDemand, _configSchema.schema.never(), scheduleSchema)
});
const deliverySchema = exports.deliverySchema = _configSchema.schema.object({
  configIds: _configSchema.schema.arrayOf(_configSchema.schema.string()),
  title: _configSchema.schema.string(),
  textDescription: _configSchema.schema.string(),
  htmlDescription: _configSchema.schema.string()
});
const reportParamsSchema = exports.reportParamsSchema = _configSchema.schema.object({
  report_name: _configSchema.schema.string({
    validate(value) {
      if (!_validationHelper.regexReportName.test(value)) {
        return `invald report name ${value}.\nMust be non-empty, allow a-z, A-Z, 0-9, (), [], ',' - and _ and spaces`;
      }
    }
  }),
  report_source: _configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_TYPE.dashboard), _configSchema.schema.literal(_constants.REPORT_TYPE.visualization), _configSchema.schema.literal(_constants.REPORT_TYPE.savedSearch), _configSchema.schema.literal(_constants.REPORT_TYPE.notebook)]),
  description: _configSchema.schema.string(),
  core_params: _configSchema.schema.conditional(_configSchema.schema.siblingRef('report_source'), _constants.REPORT_TYPE.savedSearch, dataReportSchema, visualReportSchema)
});
const reportDefinitionSchema = exports.reportDefinitionSchema = _configSchema.schema.object({
  report_params: reportParamsSchema,
  delivery: deliverySchema,
  trigger: triggerSchema,
  time_created: _configSchema.schema.maybe(_configSchema.schema.number()),
  last_updated: _configSchema.schema.maybe(_configSchema.schema.number()),
  status: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_DEFINITION_STATUS.active), _configSchema.schema.literal(_constants.REPORT_DEFINITION_STATUS.disabled)]))
});
const reportSchema = exports.reportSchema = _configSchema.schema.object({
  query_url: _configSchema.schema.string({
    validate(value) {
      if (!(0, _validationHelper.isValidRelativeUrl)(value)) {
        return `invalid relative url: ${value}`;
      }
    },
    minLength: 1
  }),
  time_from: _configSchema.schema.number(),
  time_to: _configSchema.schema.number(),
  report_definition: reportDefinitionSchema,
  time_created: _configSchema.schema.maybe(_configSchema.schema.number()),
  last_updated: _configSchema.schema.maybe(_configSchema.schema.number()),
  state: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.literal(_constants.REPORT_STATE.created), _configSchema.schema.literal(_constants.REPORT_STATE.error), _configSchema.schema.literal(_constants.REPORT_STATE.pending), _configSchema.schema.literal(_constants.REPORT_STATE.shared)]))
});
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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