"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.metaData = exports.getSelectedFields = exports.getOpenSearchData = exports.convertToCSV = exports.buildRequestBody = void 0;
var _elasticBuilder = _interopRequireDefault(require("elastic-builder"));
var _json2Csv = _interopRequireDefault(require("json-2-csv"));
var _lodash = _interopRequireDefault(require("lodash"));
var _momentTimezone = _interopRequireDefault(require("moment-timezone"));
var _common = require("../../../../../src/plugins/data/common");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

var metaData = exports.metaData = {
  saved_search_id: null,
  report_format: null,
  start: null,
  end: null,
  fields: null,
  type: null,
  timeFieldName: null,
  sorting: null,
  fields_exist: false,
  selectedFields: [],
  paternName: null,
  searchSourceJSON: [],
  dateFields: []
};

// Get the selected columns by the user.
const getSelectedFields = async columns => {
  const selectedFields = [];
  let fields_exist = false;
  for (let column of columns) {
    if (column !== '_source') {
      fields_exist = true;
      selectedFields.push(column);
    } else {
      fields_exist = false;
      selectedFields.push('_source');
    }
  }
  metaData.fields_exist = fields_exist;
  metaData.selectedFields = selectedFields;
};

// Build the OpenSearch query from the meta data
// is_count is set to 1 if we building the count query but 0 if we building the fetch data query
exports.getSelectedFields = getSelectedFields;
const buildRequestBody = (report, allowLeadingWildcards, is_count) => {
  let esbBoolQuery = _elasticBuilder.default.boolQuery();
  const searchSourceJSON = report._source.searchSourceJSON;
  const savedObjectQuery = JSON.parse(searchSourceJSON).query;
  const savedObjectFilter = JSON.parse(searchSourceJSON).filter;
  const savedObjectConfig = {
    allowLeadingWildcards: allowLeadingWildcards,
    queryStringOptions: {},
    ignoreFilterIfFieldNotInIndex: false
  };
  const QueryFromSavedObject = (0, _common.buildOpenSearchQuery)(undefined, savedObjectQuery, savedObjectFilter, savedObjectConfig);
  // Add time range
  if (report._source.timeFieldName && report._source.timeFieldName.length > 0) {
    esbBoolQuery.must(_elasticBuilder.default.rangeQuery(report._source.timeFieldName).format('epoch_millis').gte(report._source.start - 1).lte(report._source.end + 1));
  }
  if (is_count) {
    return _elasticBuilder.default.requestBodySearch().query(esbBoolQuery);
  }

  // Add sorting to the query
  let esbSearchQuery = _elasticBuilder.default.requestBodySearch().query(esbBoolQuery).version(true);
  if (report._source.sorting.length > 0) {
    const sortings = report._source.sorting.map(element => {
      return _elasticBuilder.default.sort(element[0], element[1]);
    });
    esbSearchQuery.sorts(sortings);
  }

  // add selected fields to query
  if (report._source.fields_exist) {
    esbSearchQuery.source({
      includes: report._source.selectedFields
    });
  }
  // Add a customizer to merge queries to generate request body
  let requestBody = _lodash.default.mergeWith({
    query: QueryFromSavedObject
  }, esbSearchQuery.toJSON(), (objValue, srcValue) => {
    if (_lodash.default.isArray(objValue)) {
      return objValue.concat(srcValue);
    }
  });
  requestBody = addDocValueFields(report, requestBody);
  return requestBody;
};

// Fetch the data from OpenSearch
exports.buildRequestBody = buildRequestBody;
const getOpenSearchData = (arrayHits, report, params, dateFormat, timezone) => {
  let hits = [];
  for (let valueRes of arrayHits) {
    for (let data of valueRes.hits) {
      const fields = data.fields;
      // get all the fields of type date and format them to excel format
      let tempKeyElement = [];
      for (let dateField of report._source.dateFields) {
        let keys;
        keys = dateField.split('.');
        const dateValue = data._source[dateField];
        const fieldDateValue = fields[dateField];
        const isDateFieldPresent = isKeyPresent(data._source, dateField);
        if (isDateFieldPresent) {
          // if its not a nested date field
          if (keys.length === 1) {
            // if conditions to determine if the date field's value is an array or a string
            if (typeof dateValue === 'string') {
              data._source[keys] = _momentTimezone.default.utc(dateValue).tz(timezone).format(dateFormat);
            } else if (fieldDateValue.length !== 0 && fieldDateValue instanceof Array) {
              fieldDateValue.forEach((element, index) => {
                data._source[keys][index] = _momentTimezone.default.utc(element).tz(timezone).format(dateFormat);
              });
            } else {
              data._source[keys] = [];
            }
            // else to cover cases with nested date fields
          } else {
            let keyElement = keys.shift();
            // if conditions to determine if the date field's value is an array or a string
            if (typeof fieldDateValue === 'string') {
              keys.push(_momentTimezone.default.utc(fieldDateValue).tz(timezone).format(dateFormat));
            } else if (fieldDateValue.length !== 0 && fieldDateValue instanceof Array) {
              let tempArray = [];
              fieldDateValue.forEach(index => {
                tempArray.push(_momentTimezone.default.utc(index).tz(timezone).format(dateFormat));
              });
              keys.push(tempArray);
            } else {
              keys.push([]);
            }
            const nestedJSON = arrayToNestedJSON(keys);
            let keyLength = Object.keys(data._source);
            // to check if the nested field have anyother keys apart from date field
            if (tempKeyElement.includes(keyElement) || keyLength.length > 1) {
              data._source[keyElement] = {
                ...data._source[keyElement],
                ...nestedJSON
              };
            } else {
              data._source[keyElement] = nestedJSON;
              tempKeyElement.push(keyElement);
            }
          }
        }
      }
      delete data['fields'];
      if (report._source.fields_exist === true) {
        let result = traverse(data, report._source.selectedFields);
        hits.push(params.excel ? sanitize(result) : result);
      } else {
        hits.push(params.excel ? sanitize(data) : data);
      }
      // Truncate to expected limit size
      if (hits.length >= params.limit) {
        return hits;
      }
    }
  }
  return hits;
};

//Convert the data to Csv format
exports.getOpenSearchData = getOpenSearchData;
const convertToCSV = async (dataset, csvSeparator) => {
  let convertedData = [];
  const options = {
    delimiter: {
      field: csvSeparator,
      eol: '\n'
    },
    emptyFieldValue: ' '
  };
  await _json2Csv.default.json2csvAsync(dataset[0], options).then(csv => {
    convertedData = csv;
  });
  return convertedData;
};
exports.convertToCSV = convertToCSV;
function flattenHits(hits, result = {}, prefix = '') {
  for (const [key, value] of Object.entries(hits)) {
    if (!hits.hasOwnProperty(key)) continue;
    if (value != null && typeof value === 'object' && !Array.isArray(value) && Object.keys(value).length > 0) {
      flattenHits(value, result, prefix + key + '.');
    } else {
      result[prefix.replace(/^_source\./, '') + key] = value;
    }
  }
  return result;
}

//Return only the selected fields
function traverse(data, keys, result = {}) {
  data = flattenHits(data);
  const sourceKeys = Object.keys(data);
  keys.forEach(key => {
    const value = _lodash.default.get(data, key, undefined);
    if (value !== undefined) result[key] = value;else {
      Object.keys(data).filter(sourceKey => sourceKey.startsWith(key + '.')).forEach(sourceKey => result[sourceKey] = data[sourceKey]);
    }
  });
  return result;
}

/**
 * Escape special characters if field value prefixed with.
 * This is intend to avoid CSV injection in Microsoft Excel.
 * @param doc   document
 */
function sanitize(doc) {
  for (const field in doc) {
    if (doc[field] == null) continue;
    if (doc[field].toString().startsWith('+') || doc[field].toString().startsWith('-') && typeof doc[field] !== 'number' || doc[field].toString().startsWith('=') || doc[field].toString().startsWith('@')) {
      doc[field] = "'" + doc[field];
    }
  }
  return doc;
}
function arrayToNestedJSON(arr) {
  if (arr.length === 0) {
    return null;
  } else if (arr.length === 1) {
    return arr[0];
  } else {
    const key = arr[0];
    const rest = arr.slice(1);
    return {
      [key]: arrayToNestedJSON(rest)
    };
  }
}
function isKeyPresent(data, key) {
  if (typeof data === 'object' && data !== null) {
    if (key in data) {
      return true;
    }
    for (const value of Object.values(data)) {
      if (isKeyPresent(value, key)) {
        return true;
      }
    }
  }
  return false;
}
const addDocValueFields = (report, requestBody) => {
  const docValues = [];
  for (const dateType of report._source.dateFields) {
    docValues.push({
      field: dateType,
      format: 'date_hour_minute_second_fraction'
    });
  }
  // elastic-builder doesn't provide function to build docvalue_fields with format,
  // this is a workaround which appends docvalues field to the request body.
  requestBody = {
    ...requestBody,
    docvalue_fields: docValues
  };
  return requestBody;
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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