"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecurityPlugin = void 0;
var _operators = require("rxjs/operators");
var _routes = require("./routes");
var _opensearch_security_configuration_plugin = _interopRequireDefault(require("./backend/opensearch_security_configuration_plugin"));
var _opensearch_security_plugin = _interopRequireDefault(require("./backend/opensearch_security_plugin"));
var _security_cookie = require("./session/security_cookie");
var _opensearch_security_client = require("./backend/opensearch_security_client");
var _tenant_index = require("./multitenancy/tenant_index");
var _auth_handler_factory = require("./auth/auth_handler_factory");
var _routes2 = require("./multitenancy/routes");
var _auth_type_routes = require("./routes/auth_type_routes");
var _core = require("../../../src/core/server/saved_objects/migrations/core");
var _saved_objects_wrapper = require("./saved_objects/saved_objects_wrapper");
var _tenant_resolver = require("./multitenancy/tenant_resolver");
var _readonly_service = require("./readonly/readonly_service");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class SecurityPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    // FIXME: keep an reference of admin client so that it can be used in start(), better to figureout a
    //        decent way to get adminClient in start. (maybe using getStartServices() from setup?)
    // @ts-ignore: property not initialzied in constructor
    _defineProperty(this, "securityClient", void 0);
    _defineProperty(this, "savedObjectClientWrapper", void 0);
    this.logger = initializerContext.logger.get();
    this.savedObjectClientWrapper = new _saved_objects_wrapper.SecuritySavedObjectsClientWrapper();
  }
  async setup(core) {
    var _config$multitenancy, _config$multitenancy2;
    this.logger.debug('opendistro_security: Setup');
    const config$ = this.initializerContext.config.create();
    const config = await config$.pipe((0, _operators.first)()).toPromise();
    const router = core.http.createRouter();
    const esClient = core.opensearch.legacy.createClient('opendistro_security', {
      plugins: [_opensearch_security_configuration_plugin.default, _opensearch_security_plugin.default]
    });
    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    const securitySessionStorageFactory = await core.http.createCookieSessionStorageFactory((0, _security_cookie.getSecurityCookieOptions)(config));

    // put logger into route handler context, so that we don't need to pass througth parameters
    core.http.registerRouteHandlerContext('security_plugin', (context, request) => {
      return {
        logger: this.logger,
        esClient
      };
    });

    // setup auth
    const auth = await (0, _auth_handler_factory.getAuthenticationHandler)(config.auth.type, router, config, core, esClient, securitySessionStorageFactory, this.logger);
    core.http.registerAuth(auth.authHandler);

    /* Here we check if multitenancy is enabled to ensure if it is, we insert the tenant info (security_tenant) into the resolved, short URL so the page can correctly load with the right tenant information [Fix for issue 1203](https://github.com/opensearch-project/security-dashboards-plugin/issues/1203 */
    if ((_config$multitenancy = config.multitenancy) !== null && _config$multitenancy !== void 0 && _config$multitenancy.enabled) {
      core.http.registerOnPreResponse((request, preResponse, toolkit) => {
        (0, _tenant_resolver.addTenantParameterToResolvedShortLink)(request);
        return toolkit.next();
      });
    }

    // Register server side APIs
    (0, _routes.defineRoutes)(router);
    (0, _auth_type_routes.defineAuthTypeRoutes)(router, config);

    // set up multi-tenant routes
    if ((_config$multitenancy2 = config.multitenancy) !== null && _config$multitenancy2 !== void 0 && _config$multitenancy2.enabled) {
      (0, _routes2.setupMultitenantRoutes)(router, securitySessionStorageFactory, this.securityClient);
    }
    if (config.multitenancy.enabled && config.multitenancy.enable_aggregation_view) {
      core.savedObjects.addClientWrapper(2, 'security-saved-object-client-wrapper', this.savedObjectClientWrapper.wrapperFactory);
    }
    const service = new _readonly_service.ReadonlyService(this.logger, this.securityClient, auth, securitySessionStorageFactory, config);
    core.security.registerReadonlyService(service);
    return {
      config$,
      securityConfigClient: esClient
    };
  }

  // TODO: add more logs
  async start(core) {
    var _config$multitenancy3;
    this.logger.debug('opendistro_security: Started');
    const config$ = this.initializerContext.config.create();
    const config = await config$.pipe((0, _operators.first)()).toPromise();
    this.savedObjectClientWrapper.httpStart = core.http;
    this.savedObjectClientWrapper.config = config;
    if ((_config$multitenancy3 = config.multitenancy) !== null && _config$multitenancy3 !== void 0 && _config$multitenancy3.enabled) {
      const globalConfig$ = this.initializerContext.config.legacy.globalConfig$;
      const globalConfig = await globalConfig$.pipe((0, _operators.first)()).toPromise();
      const opensearchDashboardsIndex = globalConfig.opensearchDashboards.index;
      const typeRegistry = core.savedObjects.getTypeRegistry();
      const esClient = core.opensearch.client.asInternalUser;
      const migrationClient = (0, _core.createMigrationOpenSearchClient)(esClient, this.logger);
      (0, _tenant_index.setupIndexTemplate)(esClient, opensearchDashboardsIndex, typeRegistry, this.logger);
      const serializer = core.savedObjects.createSerializer();
      const opensearchDashboardsVersion = this.initializerContext.env.packageInfo.version;
      (0, _tenant_index.migrateTenantIndices)(opensearchDashboardsVersion, migrationClient, this.securityClient, typeRegistry, serializer, this.logger);
    }
    return {
      http: core.http,
      es: core.opensearch.legacy
    };
  }
  stop() {}
}
exports.SecurityPlugin = SecurityPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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