/**
 * @license
 * Copyright Google Inc. All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
(function (factory) {
    if (typeof module === "object" && typeof module.exports === "object") {
        var v = factory(require, exports);
        if (v !== undefined) module.exports = v;
    }
    else if (typeof define === "function" && define.amd) {
        define("@angular/compiler-cli/src/transformers/node_emitter_transform", ["require", "exports", "tslib", "typescript", "@angular/compiler-cli/src/transformers/node_emitter", "@angular/compiler-cli/src/transformers/util"], factory);
    }
})(function (require, exports) {
    "use strict";
    Object.defineProperty(exports, "__esModule", { value: true });
    var tslib_1 = require("tslib");
    var ts = require("typescript");
    var node_emitter_1 = require("@angular/compiler-cli/src/transformers/node_emitter");
    var util_1 = require("@angular/compiler-cli/src/transformers/util");
    function getPreamble(original) {
        return "/**\n * @fileoverview This file was generated by the Angular template compiler. Do not edit.\n * " + original + "\n * @suppress {suspiciousCode,uselessCode,missingProperties,missingOverride,checkTypes}\n * tslint:disable\n */";
    }
    /**
     * Returns a transformer that does two things for generated files (ngfactory etc):
     * - adds a fileoverview JSDoc comment containing Closure Compiler specific "suppress"ions in JSDoc.
     *   The new comment will contain any fileoverview comment text from the original source file this
     *   file was generated from.
     * - updates generated files that are not in the given map of generatedFiles to have an empty
     *   list of statements as their body.
     */
    function getAngularEmitterTransformFactory(generatedFiles, program) {
        return function () {
            var emitter = new node_emitter_1.TypeScriptNodeEmitter();
            return function (sourceFile) {
                var g = generatedFiles.get(sourceFile.fileName);
                var orig = g && program.getSourceFile(g.srcFileUrl);
                var originalComment = '';
                if (orig)
                    originalComment = getFileoverviewComment(orig);
                var preamble = getPreamble(originalComment);
                if (g && g.stmts) {
                    var orig_1 = program.getSourceFile(g.srcFileUrl);
                    var originalComment_1 = '';
                    if (orig_1)
                        originalComment_1 = getFileoverviewComment(orig_1);
                    var _a = tslib_1.__read(emitter.updateSourceFile(sourceFile, g.stmts, preamble), 1), newSourceFile = _a[0];
                    return newSourceFile;
                }
                else if (util_1.GENERATED_FILES.test(sourceFile.fileName)) {
                    // The file should be empty, but emitter.updateSourceFile would still add imports
                    // and various minutiae.
                    // Clear out the source file entirely, only including the preamble comment, so that
                    // ngc produces an empty .js file.
                    return ts.updateSourceFileNode(sourceFile, [emitter.createCommentStatement(sourceFile, preamble)]);
                }
                return sourceFile;
            };
        };
    }
    exports.getAngularEmitterTransformFactory = getAngularEmitterTransformFactory;
    /**
     * Parses and returns the comment text (without start and end markers) of a \@fileoverview comment
     * in the given source file. Returns the empty string if no such comment can be found.
     */
    function getFileoverviewComment(sourceFile) {
        var trivia = sourceFile.getFullText().substring(0, sourceFile.getStart());
        var leadingComments = ts.getLeadingCommentRanges(trivia, 0);
        if (!leadingComments || leadingComments.length === 0)
            return '';
        var comment = leadingComments[0];
        if (comment.kind !== ts.SyntaxKind.MultiLineCommentTrivia)
            return '';
        // Only comments separated with a \n\n from the file contents are considered file-level comments
        // in TypeScript.
        if (sourceFile.getFullText().substring(comment.end, comment.end + 2) !== '\n\n')
            return '';
        var commentText = sourceFile.getFullText().substring(comment.pos, comment.end);
        // Closure Compiler ignores @suppress and similar if the comment contains @license.
        if (commentText.indexOf('@license') !== -1)
            return '';
        return commentText.replace(/^\/\*\*/, '').replace(/ ?\*\/$/, '');
    }
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibm9kZV9lbWl0dGVyX3RyYW5zZm9ybS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvbXBpbGVyLWNsaS9zcmMvdHJhbnNmb3JtZXJzL25vZGVfZW1pdHRlcl90cmFuc2Zvcm0udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUE7Ozs7OztHQU1HOzs7Ozs7Ozs7Ozs7O0lBR0gsK0JBQWlDO0lBRWpDLG9GQUFxRDtJQUNyRCxvRUFBdUM7SUFFdkMsU0FBUyxXQUFXLENBQUMsUUFBZ0I7UUFDbkMsT0FBTyxzR0FFSixRQUFRLHFIQUdULENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7T0FPRztJQUNILFNBQWdCLGlDQUFpQyxDQUM3QyxjQUEwQyxFQUFFLE9BQW1CO1FBRWpFLE9BQU87WUFDTCxJQUFNLE9BQU8sR0FBRyxJQUFJLG9DQUFxQixFQUFFLENBQUM7WUFDNUMsT0FBTyxVQUFTLFVBQXlCO2dCQUN2QyxJQUFNLENBQUMsR0FBRyxjQUFjLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztnQkFDbEQsSUFBTSxJQUFJLEdBQUcsQ0FBQyxJQUFJLE9BQU8sQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxDQUFDO2dCQUN0RCxJQUFJLGVBQWUsR0FBRyxFQUFFLENBQUM7Z0JBQ3pCLElBQUksSUFBSTtvQkFBRSxlQUFlLEdBQUcsc0JBQXNCLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQ3pELElBQU0sUUFBUSxHQUFHLFdBQVcsQ0FBQyxlQUFlLENBQUMsQ0FBQztnQkFDOUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLEtBQUssRUFBRTtvQkFDaEIsSUFBTSxNQUFJLEdBQUcsT0FBTyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsVUFBVSxDQUFDLENBQUM7b0JBQ2pELElBQUksaUJBQWUsR0FBRyxFQUFFLENBQUM7b0JBQ3pCLElBQUksTUFBSTt3QkFBRSxpQkFBZSxHQUFHLHNCQUFzQixDQUFDLE1BQUksQ0FBQyxDQUFDO29CQUNuRCxJQUFBLCtFQUF5RSxFQUF4RSxxQkFBd0UsQ0FBQztvQkFDaEYsT0FBTyxhQUFhLENBQUM7aUJBQ3RCO3FCQUFNLElBQUksc0JBQWUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxFQUFFO29CQUNwRCxpRkFBaUY7b0JBQ2pGLHdCQUF3QjtvQkFDeEIsbUZBQW1GO29CQUNuRixrQ0FBa0M7b0JBQ2xDLE9BQU8sRUFBRSxDQUFDLG9CQUFvQixDQUMxQixVQUFVLEVBQUUsQ0FBQyxPQUFPLENBQUMsc0JBQXNCLENBQUMsVUFBVSxFQUFFLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQztpQkFDekU7Z0JBQ0QsT0FBTyxVQUFVLENBQUM7WUFDcEIsQ0FBQyxDQUFDO1FBQ0osQ0FBQyxDQUFDO0lBQ0osQ0FBQztJQTVCRCw4RUE0QkM7SUFFRDs7O09BR0c7SUFDSCxTQUFTLHNCQUFzQixDQUFDLFVBQXlCO1FBQ3ZELElBQU0sTUFBTSxHQUFHLFVBQVUsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxFQUFFLFVBQVUsQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQzVFLElBQU0sZUFBZSxHQUFHLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDOUQsSUFBSSxDQUFDLGVBQWUsSUFBSSxlQUFlLENBQUMsTUFBTSxLQUFLLENBQUM7WUFBRSxPQUFPLEVBQUUsQ0FBQztRQUNoRSxJQUFNLE9BQU8sR0FBRyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbkMsSUFBSSxPQUFPLENBQUMsSUFBSSxLQUFLLEVBQUUsQ0FBQyxVQUFVLENBQUMsc0JBQXNCO1lBQUUsT0FBTyxFQUFFLENBQUM7UUFDckUsZ0dBQWdHO1FBQ2hHLGlCQUFpQjtRQUNqQixJQUFJLFVBQVUsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxPQUFPLENBQUMsR0FBRyxHQUFHLENBQUMsQ0FBQyxLQUFLLE1BQU07WUFBRSxPQUFPLEVBQUUsQ0FBQztRQUMzRixJQUFNLFdBQVcsR0FBRyxVQUFVLENBQUMsV0FBVyxFQUFFLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2pGLG1GQUFtRjtRQUNuRixJQUFJLFdBQVcsQ0FBQyxPQUFPLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQUUsT0FBTyxFQUFFLENBQUM7UUFDdEQsT0FBTyxXQUFXLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsU0FBUyxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQ25FLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIEBsaWNlbnNlXG4gKiBDb3B5cmlnaHQgR29vZ2xlIEluYy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCB7R2VuZXJhdGVkRmlsZX0gZnJvbSAnQGFuZ3VsYXIvY29tcGlsZXInO1xuaW1wb3J0ICogYXMgdHMgZnJvbSAndHlwZXNjcmlwdCc7XG5cbmltcG9ydCB7VHlwZVNjcmlwdE5vZGVFbWl0dGVyfSBmcm9tICcuL25vZGVfZW1pdHRlcic7XG5pbXBvcnQge0dFTkVSQVRFRF9GSUxFU30gZnJvbSAnLi91dGlsJztcblxuZnVuY3Rpb24gZ2V0UHJlYW1ibGUob3JpZ2luYWw6IHN0cmluZykge1xuICByZXR1cm4gYC8qKlxuICogQGZpbGVvdmVydmlldyBUaGlzIGZpbGUgd2FzIGdlbmVyYXRlZCBieSB0aGUgQW5ndWxhciB0ZW1wbGF0ZSBjb21waWxlci4gRG8gbm90IGVkaXQuXG4gKiAke29yaWdpbmFsfVxuICogQHN1cHByZXNzIHtzdXNwaWNpb3VzQ29kZSx1c2VsZXNzQ29kZSxtaXNzaW5nUHJvcGVydGllcyxtaXNzaW5nT3ZlcnJpZGUsY2hlY2tUeXBlc31cbiAqIHRzbGludDpkaXNhYmxlXG4gKi9gO1xufVxuXG4vKipcbiAqIFJldHVybnMgYSB0cmFuc2Zvcm1lciB0aGF0IGRvZXMgdHdvIHRoaW5ncyBmb3IgZ2VuZXJhdGVkIGZpbGVzIChuZ2ZhY3RvcnkgZXRjKTpcbiAqIC0gYWRkcyBhIGZpbGVvdmVydmlldyBKU0RvYyBjb21tZW50IGNvbnRhaW5pbmcgQ2xvc3VyZSBDb21waWxlciBzcGVjaWZpYyBcInN1cHByZXNzXCJpb25zIGluIEpTRG9jLlxuICogICBUaGUgbmV3IGNvbW1lbnQgd2lsbCBjb250YWluIGFueSBmaWxlb3ZlcnZpZXcgY29tbWVudCB0ZXh0IGZyb20gdGhlIG9yaWdpbmFsIHNvdXJjZSBmaWxlIHRoaXNcbiAqICAgZmlsZSB3YXMgZ2VuZXJhdGVkIGZyb20uXG4gKiAtIHVwZGF0ZXMgZ2VuZXJhdGVkIGZpbGVzIHRoYXQgYXJlIG5vdCBpbiB0aGUgZ2l2ZW4gbWFwIG9mIGdlbmVyYXRlZEZpbGVzIHRvIGhhdmUgYW4gZW1wdHlcbiAqICAgbGlzdCBvZiBzdGF0ZW1lbnRzIGFzIHRoZWlyIGJvZHkuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBnZXRBbmd1bGFyRW1pdHRlclRyYW5zZm9ybUZhY3RvcnkoXG4gICAgZ2VuZXJhdGVkRmlsZXM6IE1hcDxzdHJpbmcsIEdlbmVyYXRlZEZpbGU+LCBwcm9ncmFtOiB0cy5Qcm9ncmFtKTogKCkgPT5cbiAgICAoc291cmNlRmlsZTogdHMuU291cmNlRmlsZSkgPT4gdHMuU291cmNlRmlsZSB7XG4gIHJldHVybiBmdW5jdGlvbigpIHtcbiAgICBjb25zdCBlbWl0dGVyID0gbmV3IFR5cGVTY3JpcHROb2RlRW1pdHRlcigpO1xuICAgIHJldHVybiBmdW5jdGlvbihzb3VyY2VGaWxlOiB0cy5Tb3VyY2VGaWxlKTogdHMuU291cmNlRmlsZSB7XG4gICAgICBjb25zdCBnID0gZ2VuZXJhdGVkRmlsZXMuZ2V0KHNvdXJjZUZpbGUuZmlsZU5hbWUpO1xuICAgICAgY29uc3Qgb3JpZyA9IGcgJiYgcHJvZ3JhbS5nZXRTb3VyY2VGaWxlKGcuc3JjRmlsZVVybCk7XG4gICAgICBsZXQgb3JpZ2luYWxDb21tZW50ID0gJyc7XG4gICAgICBpZiAob3JpZykgb3JpZ2luYWxDb21tZW50ID0gZ2V0RmlsZW92ZXJ2aWV3Q29tbWVudChvcmlnKTtcbiAgICAgIGNvbnN0IHByZWFtYmxlID0gZ2V0UHJlYW1ibGUob3JpZ2luYWxDb21tZW50KTtcbiAgICAgIGlmIChnICYmIGcuc3RtdHMpIHtcbiAgICAgICAgY29uc3Qgb3JpZyA9IHByb2dyYW0uZ2V0U291cmNlRmlsZShnLnNyY0ZpbGVVcmwpO1xuICAgICAgICBsZXQgb3JpZ2luYWxDb21tZW50ID0gJyc7XG4gICAgICAgIGlmIChvcmlnKSBvcmlnaW5hbENvbW1lbnQgPSBnZXRGaWxlb3ZlcnZpZXdDb21tZW50KG9yaWcpO1xuICAgICAgICBjb25zdCBbbmV3U291cmNlRmlsZV0gPSBlbWl0dGVyLnVwZGF0ZVNvdXJjZUZpbGUoc291cmNlRmlsZSwgZy5zdG10cywgcHJlYW1ibGUpO1xuICAgICAgICByZXR1cm4gbmV3U291cmNlRmlsZTtcbiAgICAgIH0gZWxzZSBpZiAoR0VORVJBVEVEX0ZJTEVTLnRlc3Qoc291cmNlRmlsZS5maWxlTmFtZSkpIHtcbiAgICAgICAgLy8gVGhlIGZpbGUgc2hvdWxkIGJlIGVtcHR5LCBidXQgZW1pdHRlci51cGRhdGVTb3VyY2VGaWxlIHdvdWxkIHN0aWxsIGFkZCBpbXBvcnRzXG4gICAgICAgIC8vIGFuZCB2YXJpb3VzIG1pbnV0aWFlLlxuICAgICAgICAvLyBDbGVhciBvdXQgdGhlIHNvdXJjZSBmaWxlIGVudGlyZWx5LCBvbmx5IGluY2x1ZGluZyB0aGUgcHJlYW1ibGUgY29tbWVudCwgc28gdGhhdFxuICAgICAgICAvLyBuZ2MgcHJvZHVjZXMgYW4gZW1wdHkgLmpzIGZpbGUuXG4gICAgICAgIHJldHVybiB0cy51cGRhdGVTb3VyY2VGaWxlTm9kZShcbiAgICAgICAgICAgIHNvdXJjZUZpbGUsIFtlbWl0dGVyLmNyZWF0ZUNvbW1lbnRTdGF0ZW1lbnQoc291cmNlRmlsZSwgcHJlYW1ibGUpXSk7XG4gICAgICB9XG4gICAgICByZXR1cm4gc291cmNlRmlsZTtcbiAgICB9O1xuICB9O1xufVxuXG4vKipcbiAqIFBhcnNlcyBhbmQgcmV0dXJucyB0aGUgY29tbWVudCB0ZXh0ICh3aXRob3V0IHN0YXJ0IGFuZCBlbmQgbWFya2Vycykgb2YgYSBcXEBmaWxlb3ZlcnZpZXcgY29tbWVudFxuICogaW4gdGhlIGdpdmVuIHNvdXJjZSBmaWxlLiBSZXR1cm5zIHRoZSBlbXB0eSBzdHJpbmcgaWYgbm8gc3VjaCBjb21tZW50IGNhbiBiZSBmb3VuZC5cbiAqL1xuZnVuY3Rpb24gZ2V0RmlsZW92ZXJ2aWV3Q29tbWVudChzb3VyY2VGaWxlOiB0cy5Tb3VyY2VGaWxlKTogc3RyaW5nIHtcbiAgY29uc3QgdHJpdmlhID0gc291cmNlRmlsZS5nZXRGdWxsVGV4dCgpLnN1YnN0cmluZygwLCBzb3VyY2VGaWxlLmdldFN0YXJ0KCkpO1xuICBjb25zdCBsZWFkaW5nQ29tbWVudHMgPSB0cy5nZXRMZWFkaW5nQ29tbWVudFJhbmdlcyh0cml2aWEsIDApO1xuICBpZiAoIWxlYWRpbmdDb21tZW50cyB8fCBsZWFkaW5nQ29tbWVudHMubGVuZ3RoID09PSAwKSByZXR1cm4gJyc7XG4gIGNvbnN0IGNvbW1lbnQgPSBsZWFkaW5nQ29tbWVudHNbMF07XG4gIGlmIChjb21tZW50LmtpbmQgIT09IHRzLlN5bnRheEtpbmQuTXVsdGlMaW5lQ29tbWVudFRyaXZpYSkgcmV0dXJuICcnO1xuICAvLyBPbmx5IGNvbW1lbnRzIHNlcGFyYXRlZCB3aXRoIGEgXFxuXFxuIGZyb20gdGhlIGZpbGUgY29udGVudHMgYXJlIGNvbnNpZGVyZWQgZmlsZS1sZXZlbCBjb21tZW50c1xuICAvLyBpbiBUeXBlU2NyaXB0LlxuICBpZiAoc291cmNlRmlsZS5nZXRGdWxsVGV4dCgpLnN1YnN0cmluZyhjb21tZW50LmVuZCwgY29tbWVudC5lbmQgKyAyKSAhPT0gJ1xcblxcbicpIHJldHVybiAnJztcbiAgY29uc3QgY29tbWVudFRleHQgPSBzb3VyY2VGaWxlLmdldEZ1bGxUZXh0KCkuc3Vic3RyaW5nKGNvbW1lbnQucG9zLCBjb21tZW50LmVuZCk7XG4gIC8vIENsb3N1cmUgQ29tcGlsZXIgaWdub3JlcyBAc3VwcHJlc3MgYW5kIHNpbWlsYXIgaWYgdGhlIGNvbW1lbnQgY29udGFpbnMgQGxpY2Vuc2UuXG4gIGlmIChjb21tZW50VGV4dC5pbmRleE9mKCdAbGljZW5zZScpICE9PSAtMSkgcmV0dXJuICcnO1xuICByZXR1cm4gY29tbWVudFRleHQucmVwbGFjZSgvXlxcL1xcKlxcKi8sICcnKS5yZXBsYWNlKC8gP1xcKlxcLyQvLCAnJyk7XG59XG4iXX0=