"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultipleAuthentication = void 0;
var _authentication_type = require("../authentication_type");
var _common = require("../../../../common");
var _next_url = require("../../../utils/next_url");
var _routes = require("./routes");
var _types = require("../../types");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class MultipleAuthentication extends _authentication_type.AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    super(config, sessionStorageFactory, router, esClient, coreSetup, logger);
    _defineProperty(this, "authTypes", void 0);
    _defineProperty(this, "authHandlers", void 0);
    this.authTypes = this.config.auth.type;
    this.authHandlers = new Map();
  }
  async init() {
    const routes = new _routes.MultiAuthRoutes(this.router, this.sessionStorageFactory);
    routes.setupRoutes();
    for (let i = 0; i < this.authTypes.length; i++) {
      switch (this.authTypes[i].toLowerCase()) {
        case _common.AuthType.BASIC:
          {
            const BasicAuth = new _types.BasicAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await BasicAuth.init();
            this.authHandlers.set(_common.AuthType.BASIC, BasicAuth);
            break;
          }
        case _common.AuthType.OPEN_ID:
          {
            const OidcAuth = new _types.OpenIdAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await OidcAuth.init();
            this.authHandlers.set(_common.AuthType.OPEN_ID, OidcAuth);
            break;
          }
        case _common.AuthType.SAML:
          {
            const SamlAuth = new _types.SamlAuthentication(this.config, this.sessionStorageFactory, this.router, this.esClient, this.coreSetup, this.logger);
            await SamlAuth.init();
            this.authHandlers.set(_common.AuthType.SAML, SamlAuth);
            break;
          }
        default:
          {
            throw new Error(`Unsupported authentication type: ${this.authTypes[i]}`);
          }
      }
    }
  }

  // override functions inherited from AuthenticationType
  requestIncludesAuthInfo(request) {
    for (const key of this.authHandlers.keys()) {
      if (this.authHandlers.get(key).requestIncludesAuthInfo(request)) {
        return true;
      }
    }
    return false;
  }
  async getAdditionalAuthHeader(request) {
    var _cookie$authType;
    // To Do: refactor this method to improve the effiency to get cookie, get cookie from input parameter
    const cookie = await this.sessionStorageFactory.asScoped(request).get();
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType = cookie.authType) === null || _cookie$authType === void 0 ? void 0 : _cookie$authType.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return await this.authHandlers.get(reqAuthType).getAdditionalAuthHeader(request);
    } else {
      return {};
    }
  }
  getCookie(request, authInfo) {
    return {};
  }
  async isValidCookie(cookie, request) {
    var _cookie$authType2;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType2 = cookie.authType) === null || _cookie$authType2 === void 0 ? void 0 : _cookie$authType2.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).isValidCookie(cookie, request);
    } else {
      return false;
    }
  }
  handleUnauthedRequest(request, response, toolkit) {
    if (this.isPageRequest(request)) {
      const nextUrlParam = (0, _next_url.composeNextUrlQueryParam)(request, this.coreSetup.http.basePath.serverBasePath);
      if (this.config.auth.anonymous_auth_enabled) {
        const redirectLocation = `${this.coreSetup.http.basePath.serverBasePath}${_common.ANONYMOUS_AUTH_LOGIN}?${nextUrlParam}`;
        return response.redirected({
          headers: {
            location: `${redirectLocation}`
          }
        });
      }
      return response.redirected({
        headers: {
          location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}?${nextUrlParam}`
        }
      });
    } else {
      return response.unauthorized();
    }
  }
  buildAuthHeaderFromCookie(cookie, request) {
    var _cookie$authType3;
    const reqAuthType = cookie === null || cookie === void 0 || (_cookie$authType3 = cookie.authType) === null || _cookie$authType3 === void 0 ? void 0 : _cookie$authType3.toLowerCase();
    if (reqAuthType && this.authHandlers.has(reqAuthType)) {
      return this.authHandlers.get(reqAuthType).buildAuthHeaderFromCookie(cookie, request);
    } else {
      return {};
    }
  }
}
exports.MultipleAuthentication = MultipleAuthentication;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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