"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.MdWorkspaceSymbolProvider = void 0;
const lsp = require("vscode-languageserver-types");
const dispose_1 = require("../util/dispose");
const workspaceCache_1 = require("../workspaceCache");
const string_1 = require("../util/string");
class MdWorkspaceSymbolProvider extends dispose_1.Disposable {
    #cache;
    #symbolProvider;
    constructor(workspace, symbolProvider) {
        super();
        this.#symbolProvider = symbolProvider;
        this.#cache = this._register(new workspaceCache_1.MdWorkspaceInfoCache(workspace, (doc, token) => this.provideDocumentSymbolInformation(doc, token)));
    }
    async provideWorkspaceSymbols(query, token) {
        const allSymbols = await this.#cache.values();
        if (token.isCancellationRequested) {
            return [];
        }
        const normalizedQueryStr = query.toLowerCase();
        return allSymbols.flat().filter(symbolInformation => {
            return (0, string_1.fuzzyContains)(symbolInformation.name.toLowerCase(), normalizedQueryStr);
        });
    }
    async provideDocumentSymbolInformation(document, token) {
        const docSymbols = await this.#symbolProvider.provideDocumentSymbols(document, {}, token);
        if (token.isCancellationRequested) {
            return [];
        }
        return Array.from(this.#toSymbolInformation(document.uri, docSymbols));
    }
    *#toSymbolInformation(uri, docSymbols) {
        for (const symbol of docSymbols) {
            yield {
                name: symbol.name,
                kind: lsp.SymbolKind.String,
                location: { uri, range: symbol.selectionRange }
            };
            if (symbol.children) {
                yield* this.#toSymbolInformation(uri, symbol.children);
            }
        }
    }
}
exports.MdWorkspaceSymbolProvider = MdWorkspaceSymbolProvider;
//# sourceMappingURL=workspaceSymbols.js.map