"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateReportDefinition = exports.validateReport = exports.regexReportName = exports.regexRelativeUrl = exports.regexEmailAddress = exports.regexDuration = exports.isValidRelativeUrl = void 0;
var _path = _interopRequireDefault(require("path"));
var _model = require("../../server/model");
var _constants = require("../../server/routes/utils/constants");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const isValidRelativeUrl = relativeUrl => {
  let normalizedRelativeUrl = relativeUrl;
  if (!relativeUrl.includes('observability#/notebooks') && !relativeUrl.includes('notebooks-dashboards')) {
    normalizedRelativeUrl = _path.default.posix.normalize(relativeUrl);
  }

  // check pattern
  // ODFE pattern: /app/dashboards#/view/7adfa750-4c81-11e8-b3d7-01146121b73d?_g
  // AES pattern: /_plugin/kibana/app/dashboards#/view/7adfa750-4c81-11e8-b3d7-01146121b73d?_g
  const isValid = regexRelativeUrl.test(normalizedRelativeUrl);
  return isValid;
};

/**
 * moment.js isValid() API fails to validate time duration, so use regex
 * https://github.com/moment/moment/issues/1805
 **/
exports.isValidRelativeUrl = isValidRelativeUrl;
const regexDuration = exports.regexDuration = /^(-?)P(?=\d|T\d)(?:(\d+)Y)?(?:(\d+)M)?(?:(\d+)([DW]))?(?:T(?:(\d+)H)?(?:(\d+)M)?(?:(\d+(?:\.\d+)?)S)?)?$/;
const regexEmailAddress = exports.regexEmailAddress = /\S+@\S+\.\S+/;
const regexReportName = exports.regexReportName = /^[\w\-\s\(\)\[\]\,\_\-+]+$/;
const regexRelativeUrl = exports.regexRelativeUrl = /^\/(_plugin\/kibana\/|_dashboards\/)?app\/(dashboards|visualize|discover|discoverLegacy|data-explorer\/discover\/?|observability-dashboards|observability-notebooks|notebooks-dashboards\?view=output_only(&security_tenant=.+)?)(\?security_tenant=.+)?#\/(notebooks\/|view\/|edit\/)?[^\/]+$/;
const validateReport = async (client, report, basePath) => {
  report.query_url = report.query_url.replace(basePath, '');
  report.report_definition.report_params.core_params.base_url = report.report_definition.report_params.core_params.base_url.replace(basePath, '');
  // validate basic schema
  report = _model.reportSchema.validate(report);
  // parse to retrieve data
  const {
    query_url: queryUrl,
    report_definition: {
      report_params: {
        report_source: reportSource
      }
    }
  } = report;
  // Check if saved object actually exists
  await validateSavedObject(client, queryUrl, reportSource);
  return report;
};
exports.validateReport = validateReport;
const validateReportDefinition = async (client, reportDefinition, basePath) => {
  reportDefinition.report_params.core_params.base_url = reportDefinition.report_params.core_params.base_url.replace(basePath, '');
  // validate basic schema
  reportDefinition = _model.reportDefinitionSchema.validate(reportDefinition);
  // parse to retrieve data
  const {
    report_params: {
      report_source: reportSource,
      core_params: {
        base_url: baseUrl
      }
    }
  } = reportDefinition;
  // Check if saved object actually exists
  await validateSavedObject(client, baseUrl, reportSource);
  return reportDefinition;
};
exports.validateReportDefinition = validateReportDefinition;
const validateSavedObject = async (client, url, source) => {
  const getId = url => {
    var _url$split$pop;
    return (_url$split$pop = url.split('/').pop()) === null || _url$split$pop === void 0 ? void 0 : _url$split$pop.replace(/\?\S+$/, '');
  };
  const getType = source => {
    switch (source) {
      case _constants.REPORT_TYPE.dashboard:
        return 'dashboard';
      case _constants.REPORT_TYPE.savedSearch:
        return 'search';
      case _constants.REPORT_TYPE.visualization:
        return 'visualization';
      case _constants.REPORT_TYPE.notebook:
        return 'notebook';
    }
  };
  let exist = false;
  let savedObjectId = '';
  if (getType(source) === 'notebook') {
    // no backend check for notebooks because we would just be checking against the notebooks api again
    exist = true;
  } else {
    savedObjectId = `${getType(source)}:${getId(url)}`;
    const params = {
      index: '.kibana',
      id: savedObjectId
    };
    exist = await client.callAsCurrentUser('exists', params);
  }
  if (!exist) {
    throw Error(`saved object with id ${savedObjectId} does not exist`);
  }
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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