"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createSavedSearchReport = createSavedSearchReport;
var _dataReportHelpers = require("./dataReportHelpers");
var _helpers = require("./helpers");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Specify how long scroll context should be maintained for scrolled search
 */
const scrollTimeout = '1m';
async function createSavedSearchReport(report, client, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask = true, logger, timezone) {
  const params = report.report_definition.report_params;
  const reportFormat = params.core_params.report_format;
  const reportName = params.report_name;
  await populateMetaData(client, report, isScheduledTask, logger);
  const data = await generateReportData(client, params.core_params, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask, logger, timezone);
  const curTime = new Date();
  const timeCreated = curTime.valueOf();
  const fileName = (0, _helpers.getFileName)(reportName, curTime) + '.' + reportFormat;
  return {
    timeCreated,
    dataUrl: data,
    fileName
  };
}

/**
 * Populate parameters and saved search info related to meta data object.
 * @param client  OpenSearch client
 * @param report  Report input
 */
async function populateMetaData(client, report, isScheduledTask, logger) {
  _dataReportHelpers.metaData.saved_search_id = report.report_definition.report_params.core_params.saved_search_id;
  _dataReportHelpers.metaData.report_format = report.report_definition.report_params.core_params.report_format;
  _dataReportHelpers.metaData.start = report.time_from;
  _dataReportHelpers.metaData.end = report.time_to;

  // Get saved search info
  let resIndexPattern = {};
  const ssParams = {
    index: '.kibana',
    id: 'search:' + _dataReportHelpers.metaData.saved_search_id
  };
  const ssInfos = await (0, _helpers.callCluster)(client, 'get', ssParams, isScheduledTask);
  _dataReportHelpers.metaData.sorting = ssInfos._source.search.sort;
  _dataReportHelpers.metaData.type = ssInfos._source.type;
  _dataReportHelpers.metaData.searchSourceJSON = ssInfos._source.search.kibanaSavedObjectMeta.searchSourceJSON;

  // Get the list of selected columns in the saved search.Otherwise select all the fields under the _source
  await (0, _dataReportHelpers.getSelectedFields)(ssInfos._source.search.columns);

  // Get index name
  for (const item of ssInfos._source.references) {
    if (item.name === JSON.parse(_dataReportHelpers.metaData.searchSourceJSON).indexRefName) {
      // Get index-pattern information
      const indexPattern = await (0, _helpers.callCluster)(client, 'get', {
        index: '.kibana',
        id: 'index-pattern:' + item.id
      }, isScheduledTask);
      resIndexPattern = indexPattern._source['index-pattern'];
      _dataReportHelpers.metaData.paternName = resIndexPattern.title;
      _dataReportHelpers.metaData.timeFieldName = resIndexPattern.timeFieldName, _dataReportHelpers.metaData.fields = resIndexPattern.fields; // Get all fields
      // Getting fields of type Date
      const dateFields = [];
      for (const item of JSON.parse(_dataReportHelpers.metaData.fields)) {
        if (item.type === 'date') {
          dateFields.push(item.name);
        }
      }
      _dataReportHelpers.metaData.dateFields = dateFields;
    }
  }
}

/**
 * Generate CSV data by query and convert OpenSearch data set.
 * @param client  OpenSearch client
 * @param limit   limit size of result data set
 */
async function generateReportData(client, params, dateFormat, csvSeparator, allowLeadingWildcards, isScheduledTask, logger, timezone) {
  let opensearchData = {};
  const arrayHits = [];
  const report = {
    _source: _dataReportHelpers.metaData
  };
  const indexPattern = report._source.paternName;
  const maxResultSize = await getMaxResultSize();
  const opensearchCount = await getOpenSearchDataSize();
  const total = Math.min(opensearchCount.count, params.limit);
  if (total === 0) {
    return '';
  }
  const reqBody = (0, _dataReportHelpers.buildRequestBody)(report, allowLeadingWildcards, 0);
  logger.info(`[Reporting csv module] DSL request body: ${JSON.stringify(reqBody)}`);
  if (total > maxResultSize) {
    await getOpenSearchDataByScroll();
  } else {
    await getOpenSearchDataBySearch();
  }
  return convertOpenSearchDataToCsv();

  // Fetch OpenSearch query max size windows to decide search or scroll
  async function getMaxResultSize() {
    const settings = await (0, _helpers.callCluster)(client, 'indices.getSettings', {
      index: indexPattern,
      includeDefaults: true
    }, isScheduledTask);
    let maxResultSize = Number.MAX_SAFE_INTEGER;
    for (let indexName in settings) {
      // The location of max result window differs if default overridden.
      maxResultSize = Math.min(maxResultSize, settings[indexName].settings.index.max_result_window || settings[indexName].defaults.index.max_result_window);
    }
    return maxResultSize;
  }

  // Build the OpenSearch Count query to count the size of result
  async function getOpenSearchDataSize() {
    const countReq = (0, _dataReportHelpers.buildRequestBody)(report, allowLeadingWildcards, 1);
    return await (0, _helpers.callCluster)(client, 'count', {
      index: indexPattern,
      body: countReq
    }, isScheduledTask);
  }
  async function getOpenSearchDataByScroll() {
    const searchParams = {
      index: report._source.paternName,
      scroll: scrollTimeout,
      body: reqBody,
      size: maxResultSize
    };
    // Open scroll context by fetching first batch
    opensearchData = await (0, _helpers.callCluster)(client, 'search', searchParams, isScheduledTask);
    arrayHits.push(opensearchData.hits);

    // Start scrolling till the end
    const nbScroll = Math.floor(total / maxResultSize);
    for (let i = 0; i < nbScroll; i++) {
      const resScroll = await (0, _helpers.callCluster)(client, 'scroll', {
        scrollId: opensearchData._scroll_id,
        scroll: scrollTimeout
      }, isScheduledTask);
      if (Object.keys(resScroll.hits.hits).length > 0) {
        arrayHits.push(resScroll.hits);
      }
    }

    // Clear scroll context
    await (0, _helpers.callCluster)(client, 'clearScroll', {
      scrollId: opensearchData._scroll_id
    }, isScheduledTask);
  }
  async function getOpenSearchDataBySearch() {
    const searchParams = {
      index: report._source.paternName,
      body: reqBody,
      size: total
    };
    opensearchData = await (0, _helpers.callCluster)(client, 'search', searchParams, isScheduledTask);
    arrayHits.push(opensearchData.hits);
  }

  // Parse OpenSearch data and convert to CSV
  async function convertOpenSearchDataToCsv() {
    const dataset = [];
    dataset.push((0, _dataReportHelpers.getOpenSearchData)(arrayHits, report, params, dateFormat, timezone));
    return await (0, _dataReportHelpers.convertToCSV)(dataset, csvSeparator);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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