"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationInstanceBuilder = void 0;
var _uuid = require("uuid");
var _utils = require("./repository/utils");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IntegrationInstanceBuilder {
  constructor(client) {
    _defineProperty(this, "client", void 0);
    this.client = client;
  }
  build(integration, options) {
    const instance = (0, _utils.deepCheck)(integration).then(result => {
      if (!result.ok) {
        return Promise.reject(result.error);
      }
      return integration.getAssets();
    }).then(assets => {
      if (!assets.ok) {
        return Promise.reject(assets.error);
      }
      return assets.value;
    }).then(assets => this.remapIDs(assets.savedObjects)).then(assets => this.remapDataSource(assets, options.dataSource)).then(assets => this.postAssets(assets)).then(refs => this.buildInstance(integration, refs, options));
    return instance;
  }
  remapDataSource(assets, dataSource) {
    if (!dataSource) return assets;
    assets = assets.map(asset => {
      if (asset.type === 'index-pattern') {
        asset.attributes.title = dataSource;
      }
      return asset;
    });
    return assets;
  }
  remapIDs(assets) {
    const toRemap = assets.filter(asset => asset.id);
    const idMap = new Map();
    return toRemap.map(item => {
      if (!idMap.has(item.id)) {
        idMap.set(item.id, (0, _uuid.v4)());
      }
      item.id = idMap.get(item.id);
      for (let ref = 0; ref < item.references.length; ref++) {
        const refId = item.references[ref].id;
        if (!idMap.has(refId)) {
          idMap.set(refId, (0, _uuid.v4)());
        }
        item.references[ref].id = idMap.get(refId);
      }
      return item;
    });
  }
  async postAssets(assets) {
    try {
      const response = await this.client.bulkCreate(assets);
      const refs = response.saved_objects.map(obj => {
        var _obj$attributes;
        return {
          assetType: obj.type,
          assetId: obj.id,
          status: 'available',
          // Assuming a successfully created object is available
          isDefaultAsset: obj.type === 'dashboard',
          // Assuming for now that dashboards are default
          description: (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title
        };
      });
      return Promise.resolve(refs);
    } catch (err) {
      return Promise.reject(err);
    }
  }
  async buildInstance(integration, refs, options) {
    const config = await integration.getConfig();
    if (!config.ok) {
      return Promise.reject(new Error('Attempted to create instance with invalid template', config.error));
    }
    return Promise.resolve({
      name: options.name,
      templateName: config.value.name,
      dataSource: options.dataSource,
      creationDate: new Date().toISOString(),
      assets: refs
    });
  }
}
exports.IntegrationInstanceBuilder = IntegrationInstanceBuilder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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