"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationsManager = void 0;
var _path = _interopRequireDefault(require("path"));
var _metrics_helper = require("../../common/metrics/metrics_helper");
var _integrations_builder = require("./integrations_builder");
var _repository = require("./repository/repository");
var _fs_data_adaptor = require("./repository/fs_data_adaptor");
var _index_data_adaptor = require("./repository/index_data_adaptor");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IntegrationsManager {
  constructor(client, repository) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "instanceBuilder", void 0);
    _defineProperty(this, "repository", void 0);
    _defineProperty(this, "deleteIntegrationInstance", async id => {
      let children;
      try {
        children = await this.client.get('integration-instance', id);
      } catch (err) {
        var _err$output;
        return ((_err$output = err.output) === null || _err$output === void 0 ? void 0 : _err$output.statusCode) === 404 ? Promise.resolve([id]) : Promise.reject(err);
      }
      const toDelete = children.attributes.assets.filter(i => i.assetId).map(i => {
        return {
          id: i.assetId,
          type: i.assetType
        };
      });
      toDelete.push({
        id,
        type: 'integration-instance'
      });
      const result = Promise.all(toDelete.map(async asset => {
        try {
          await this.client.delete(asset.type, asset.id);
          return Promise.resolve(asset.id);
        } catch (err) {
          var _err$output2;
          (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', err);
          return ((_err$output2 = err.output) === null || _err$output2 === void 0 ? void 0 : _err$output2.statusCode) === 404 ? Promise.resolve(asset.id) : Promise.reject(err);
        }
      }));
      (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', 'count');
      return result;
    });
    // Internal; use getIntegrationTemplates.
    _defineProperty(this, "_getAllIntegrationTemplates", async () => {
      const integrationList = await this.repository.getIntegrationList();
      const configResults = await Promise.all(integrationList.map(x => x.getConfig()));
      const configs = configResults.filter(cfg => cfg.ok);
      return Promise.resolve({
        hits: configs.map(cfg => cfg.value)
      });
    });
    // Internal; use getIntegrationTemplates.
    _defineProperty(this, "_getIntegrationTemplatesByName", async name => {
      const integration = await this.repository.getIntegration(name);
      const config = await (integration === null || integration === void 0 ? void 0 : integration.getConfig());
      if (!config || !config.ok) {
        return Promise.resolve({
          hits: []
        });
      }
      return Promise.resolve({
        hits: [config.value]
      });
    });
    _defineProperty(this, "getIntegrationTemplates", async query => {
      if (query !== null && query !== void 0 && query.name) {
        return this._getIntegrationTemplatesByName(query.name);
      }
      return this._getAllIntegrationTemplates();
    });
    _defineProperty(this, "getIntegrationInstances", async _query => {
      var _result$saved_objects;
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.find({
        type: 'integration-instance'
      });
      return Promise.resolve({
        total: result.total,
        hits: (_result$saved_objects = result.saved_objects) === null || _result$saved_objects === void 0 ? void 0 : _result$saved_objects.map(x => ({
          ...x.attributes,
          id: x.id
        }))
      });
    });
    _defineProperty(this, "getIntegrationInstance", async query => {
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.get('integration-instance', `${query.id}`);
      return Promise.resolve(this.buildInstanceResponse(result));
    });
    _defineProperty(this, "buildInstanceResponse", async savedObj => {
      const assets = savedObj.attributes.assets;
      const status = assets ? await this.getAssetStatus(assets) : 'available';
      return {
        id: savedObj.id,
        status,
        ...savedObj.attributes
      };
    });
    _defineProperty(this, "getAssetStatus", async assets => {
      const statuses = await Promise.all(assets.map(async asset => {
        try {
          await this.client.get(asset.assetType, asset.assetId);
          return {
            id: asset.assetId,
            status: 'available'
          };
        } catch (err) {
          var _err$output3;
          const statusCode = (_err$output3 = err.output) === null || _err$output3 === void 0 ? void 0 : _err$output3.statusCode;
          if (statusCode && 400 <= statusCode && statusCode < 500) {
            return {
              id: asset.assetId,
              status: 'unavailable'
            };
          }
          console.error('Failed to get asset status', err);
          return {
            id: asset.assetId,
            status: 'unknown'
          };
        }
      }));
      const [available, unavailable, unknown] = [statuses.filter(x => x.status === 'available').length, statuses.filter(x => x.status === 'unavailable').length, statuses.filter(x => x.status === 'unknown').length];
      if (unknown > 0) return 'unknown';
      if (unavailable > 0 && available > 0) return 'partially-available';
      if (unavailable > 0) return 'unavailable';
      return 'available';
    });
    _defineProperty(this, "loadIntegrationInstance", async (templateName, name, dataSource) => {
      const template = await this.repository.getIntegration(templateName);
      if (template === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      try {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', 'count');
        const result = await this.instanceBuilder.build(template, {
          name,
          dataSource
        });
        const test = await this.client.create('integration-instance', result);
        return Promise.resolve({
          ...result,
          id: test.id
        });
      } catch (err) {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', err);
        return Promise.reject({
          message: err.message,
          statusCode: 500
        });
      }
    });
    _defineProperty(this, "getStatic", async (templateName, staticPath) => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const data = await integration.getStatic(staticPath);
      if (data.ok) {
        return data.value;
      }
      const is404 = data.error.code === 'ENOENT';
      return Promise.reject({
        message: data.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getSchemas", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const result = await integration.getSchemas();
      if (result.ok) {
        return result.value;
      }
      const is404 = result.error.code === 'ENOENT';
      return Promise.reject({
        message: result.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getAssets", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const assets = await integration.getAssets();
      if (assets.ok) {
        return assets.value;
      }
      const is404 = assets.error.code === 'ENOENT';
      return Promise.reject({
        message: assets.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    _defineProperty(this, "getSampleData", async templateName => {
      const integration = await this.repository.getIntegration(templateName);
      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }
      const sampleData = await integration.getSampleData();
      if (sampleData.ok) {
        return sampleData.value;
      }
      const is404 = sampleData.error.code === 'ENOENT';
      return Promise.reject({
        message: sampleData.error.message,
        statusCode: is404 ? 404 : 500
      });
    });
    this.client = client;
    this.repository = repository !== null && repository !== void 0 ? repository : new _repository.TemplateManager([new _index_data_adaptor.IndexDataAdaptor(this.client), new _fs_data_adaptor.FileSystemDataAdaptor(_path.default.join(__dirname, '__data__/repository'))]);
    this.instanceBuilder = new _integrations_builder.IntegrationInstanceBuilder(this.client);
  }
}
exports.IntegrationsManager = IntegrationsManager;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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