"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationReader = void 0;
var _path = _interopRequireDefault(require("path"));
var _semver = _interopRequireDefault(require("semver"));
var _validators = require("../validators");
var _fs_data_adaptor = require("./fs_data_adaptor");
var _utils = require("./utils");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 * The Integration class represents the data for Integration Templates.
 * It is backed by the repository file system.
 * It includes accessor methods for integration configs, as well as helpers for nested components.
 */
class IntegrationReader {
  constructor(directory, reader) {
    _defineProperty(this, "reader", void 0);
    _defineProperty(this, "directory", void 0);
    _defineProperty(this, "name", void 0);
    this.directory = directory;
    this.name = _path.default.basename(directory);
    this.reader = reader !== null && reader !== void 0 ? reader : new _fs_data_adaptor.FileSystemDataAdaptor(directory);
  }

  /**
   * Retrieve data from correct source regardless of if reader is config-localized or not.
   *
   * TODO refactor to assemble filename from `type` instead of requiring caller to format it.
   *
   * @param item An item which may have data in it.
   * @param fileParams Information about the file to read if the config is not localized.
   * @param format How to package the returned data.
   *               If 'json', return `object | object[]`. If 'binary', return `Buffer`.
   * @returns A result with the data, with a format based on the format field.
   */

  async fetchDataOrReadFile(item, fileParams, format) {
    if (this.reader.isConfigLocalized) {
      if (!item.data) {
        return {
          ok: false,
          error: new Error('The config for the provided reader is localized, but no data field is present. ' + JSON.stringify(item))
        };
      }
      try {
        if (format === 'json') {
          return {
            ok: true,
            value: JSON.parse(item.data)
          };
        } else {
          return {
            ok: true,
            value: Buffer.from(item.data, 'base64')
          };
        }
      } catch (error) {
        return {
          ok: false,
          error
        };
      }
    }
    if (format === 'json') {
      return this.reader.readFile(fileParams.filename, fileParams.type);
    } else {
      return this.reader.readFileRaw(fileParams.filename, fileParams.type);
    }
  }

  /**
   * Get the latest version of the integration available.
   * This method relies on the fact that integration configs have their versions in their name.
   * Any files that don't match the config naming convention will be ignored.
   *
   * @returns A string with the latest version, or null if no versions are available.
   */
  async getLatestVersion() {
    const versions = await this.reader.findIntegrationVersions();
    if (!versions.ok) {
      return null;
    }
    if (versions.value.length === 0) {
      return null;
    }
    // Sort descending
    versions.value.sort(_semver.default.rcompare);
    return versions.value[0];
  }

  // Get config without pruning or validation.
  async getRawConfig(version) {
    if ((await this.reader.getDirectoryType()) !== 'integration') {
      return {
        ok: false,
        error: new Error(`${this.directory} is not a valid integration directory`)
      };
    }
    const maybeVersion = version ? version : await this.getLatestVersion();
    if (maybeVersion === null) {
      return {
        ok: false,
        error: new Error(`No valid config matching version ${version} is available`)
      };
    }
    const configFile = `${this.name}-${maybeVersion}.json`;

    // Even config-localized readers must support config-read.
    const config = await this.reader.readFile(configFile);
    if (!config.ok) {
      return config;
    }
    return (0, _validators.validateTemplate)(config.value);
  }

  /**
   * Get the configuration of the current integration.
   *
   * @param version The version of the config to retrieve.
   * @returns The config if a valid config matching the version is present, otherwise null.
   */
  async getConfig(version) {
    const maybeConfig = await this.getRawConfig(version);
    if (!maybeConfig.ok) {
      return maybeConfig;
    }
    return (0, _validators.validateTemplate)((0, _utils.pruneConfig)(maybeConfig.value));
  }
  async getQueries(queriesList) {
    const queries = await Promise.all(queriesList.map(async item => {
      const query = await this.fetchDataOrReadFile(item, {
        filename: `${item.name}-${item.version}.${item.language}`,
        type: 'assets'
      }, 'binary');
      if (!query.ok) {
        return query;
      }
      return {
        ok: true,
        value: {
          language: item.language,
          query: query.value.toString('utf8')
        }
      };
    }));
    return (0, _utils.foldResults)(queries);
  }

  /**
   * Retrieve assets associated with the integration.
   * This method greedily retrieves all assets.
   * If the version is invalid, an error is thrown.
   * If an asset is invalid, it will be skipped.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */
  async getAssets(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = {};
    if (config.assets.savedObjects) {
      const assets = await this.fetchDataOrReadFile(config.assets.savedObjects, {
        filename: `${config.assets.savedObjects.name}-${config.assets.savedObjects.version}.ndjson`,
        type: 'assets'
      }, 'json');
      if (!assets.ok) {
        return assets;
      }
      resultValue.savedObjects = assets.value;
    }
    if (config.assets.queries) {
      const queries = await this.getQueries(config.assets.queries);
      if (!queries.ok) {
        return queries;
      }
      resultValue.queries = queries.value;
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieve sample data associated with the integration.
   * If the version is invalid, an error is thrown.
   * If the sample data is invalid, null will be returned
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing a list of sample data with adjusted timestamps.
   */
  async getSampleData(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = {
      sampleData: null
    };
    if (config.sampleData) {
      const jsonContent = await this.fetchDataOrReadFile(config.sampleData, {
        filename: config.sampleData.path,
        type: 'data'
      }, 'json');
      if (!jsonContent.ok) {
        return jsonContent;
      }
      for (const value of jsonContent.value) {
        if (!('@timestamp' in value)) {
          continue;
        }
        // Randomly scatter timestamps across last 10 minutes
        // Assume for now that the ordering of events isn't important, can change to a sequence if needed
        // Also doesn't handle fields like `observedTimestamp` if present
        const newTime = new Date(Date.now() - Math.floor(Math.random() * 1000 * 60 * 10)).toISOString();
        Object.assign(value, {
          '@timestamp': newTime
        });
        if ('observedTimestamp' in value) {
          Object.assign(value, {
            observedTimestamp: newTime
          });
        }
      }
      resultValue.sampleData = jsonContent.value;
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieve schema data associated with the integration.
   * This method greedily retrieves all mappings and schemas.
   * It's assumed that a valid version will be provided.
   * If the version is invalid, an error is thrown.
   * If a schema is invalid, an error will be thrown.
   *
   * @param version The version of the integration to retrieve assets for.
   * @returns An object containing the different types of assets.
   */
  async getSchemas(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }
    const config = configResult.value;
    const resultValue = {
      mappings: {}
    };
    for (const component of config.components) {
      const schemaFile = `${component.name}-${component.version}.mapping.json`;
      const schema = await this.fetchDataOrReadFile(component, {
        filename: schemaFile,
        type: 'schemas'
      }, 'json');
      if (!schema.ok) {
        return schema;
      }
      resultValue.mappings[component.name] = schema.value;
    }
    return {
      ok: true,
      value: resultValue
    };
  }

  /**
   * Retrieves the data for a static file associated with the integration.
   *
   * @param staticPath The path of the static to retrieve.
   * @returns A buffer with the static's data if present, otherwise null.
   */
  async getStatic(staticPath) {
    // Statics were originally designed to read straight from file system,
    // so we use direct access if possible.
    if (!this.reader.isConfigLocalized) {
      return await this.reader.readFileRaw(staticPath, 'static');
    }

    // Otherwise, we need to search for the right static, by checking each version.
    const versions = await this.reader.findIntegrationVersions();
    if (!versions.ok) {
      return versions;
    }
    for (const version of versions.value) {
      var _statics$logo, _statics$darkModeLogo;
      const config = await this.getRawConfig(version);
      if (!config.ok || !config.value.statics) {
        continue;
      }
      const statics = config.value.statics;
      if (((_statics$logo = statics.logo) === null || _statics$logo === void 0 ? void 0 : _statics$logo.path) === staticPath) {
        if (!('data' in statics.logo)) {
          return {
            ok: false,
            error: new Error('Localized config missing static data')
          };
        }
        return {
          ok: true,
          value: Buffer.from(statics.logo.data, 'base64')
        };
      }
      if ((statics === null || statics === void 0 || (_statics$darkModeLogo = statics.darkModeLogo) === null || _statics$darkModeLogo === void 0 ? void 0 : _statics$darkModeLogo.path) === staticPath) {
        if (!('data' in statics.darkModeLogo)) {
          return {
            ok: false,
            error: new Error('Localized config missing static data')
          };
        }
        return {
          ok: true,
          value: Buffer.from(statics.darkModeLogo.data, 'base64')
        };
      }
      for (const iterStatic of [...((_statics$gallery = statics === null || statics === void 0 ? void 0 : statics.gallery) !== null && _statics$gallery !== void 0 ? _statics$gallery : []), ...((_statics$darkModeGall = statics === null || statics === void 0 ? void 0 : statics.darkModeGallery) !== null && _statics$darkModeGall !== void 0 ? _statics$darkModeGall : [])]) {
        var _statics$gallery, _statics$darkModeGall;
        if (iterStatic.path === staticPath) {
          if (!('data' in iterStatic)) {
            return {
              ok: false,
              error: new Error('Localized config missing static data')
            };
          }
          return {
            ok: true,
            value: Buffer.from(iterStatic.data, 'base64')
          };
        }
      }
    }
    return {
      ok: false,
      error: new Error(`Static not found: ${staticPath}`, {
        code: 'ENOENT'
      })
    };
  }
  async serializeStaticAsset(asset) {
    const data = await this.getStatic(asset.path);
    if (!data.ok) {
      return data;
    }
    return {
      ok: true,
      value: {
        ...asset,
        data: data.value.toString('base64')
      }
    };
  }
  async serializeStatics(statics) {
    const serialized = {};
    if (statics.logo) {
      const serializeResult = await this.serializeStaticAsset(statics.logo);
      serialized.logo = serializeResult.value;
    }
    if (statics.darkModeLogo) {
      const serializeResult = await this.serializeStaticAsset(statics.darkModeLogo);
      serialized.darkModeLogo = serializeResult.value;
    }
    if (statics.gallery) {
      const results = await Promise.all(statics.gallery.map(asset => this.serializeStaticAsset(asset)));
      const foldedResult = (0, _utils.foldResults)(results);
      serialized.gallery = foldedResult.value;
    }
    if (statics.darkModeGallery) {
      const results = await Promise.all(statics.darkModeGallery.map(asset => this.serializeStaticAsset(asset)));
      const foldedResult = (0, _utils.foldResults)(results);
      serialized.darkModeGallery = foldedResult.value;
    }
    return {
      ok: true,
      value: serialized
    };
  }

  /**
   * Serialize the referenced integration as a flat JSON object.
   * Useful for normalizing the format for sending to other locations.
   * This method implements the serialization scheme expected by `JsonCatalogDataAdaptor`.
   *
   * @param version The version of the integration to serialize.
   * @returns A large object which includes all of the integration's data.
   */
  async serialize(version) {
    const configResult = await this.getRawConfig(version);
    if (!configResult.ok) {
      return configResult;
    }

    // Type cast safety: all serializable properties must have the 'data' field.
    // The remainder of the method is populating all such fields.
    const config = configResult.value;
    const componentResults = await Promise.all(config.components.map(component => this.fetchDataOrReadFile(component, {
      filename: `${component.name}-${component.version}.mapping.json`,
      type: 'schemas'
    }, 'json')));
    const componentsResult = (0, _utils.foldResults)(componentResults);
    if (!componentsResult.ok) {
      return componentsResult;
    }
    config.components = config.components.map((component, idx) => {
      return {
        ...component,
        data: JSON.stringify(componentsResult.value[idx])
      };
    });
    if (config.assets.savedObjects) {
      const soMetadata = config.assets.savedObjects;
      const soResult = await this.fetchDataOrReadFile(config.assets.savedObjects, {
        filename: `${soMetadata.name}-${soMetadata.version}.ndjson`,
        type: 'assets'
      }, 'json');
      if (!soResult.ok) {
        return soResult;
      }
      config.assets.savedObjects = {
        ...soMetadata,
        data: JSON.stringify(soResult.value)
      };
    }
    if (config.assets.queries) {
      const queryResults = await Promise.all(config.assets.queries.map(query => this.fetchDataOrReadFile(query, {
        filename: `${query.name}-${query.version}.${query.language}`,
        type: 'assets'
      }, 'binary')));
      const queriesResult = (0, _utils.foldResults)(queryResults);
      if (!queriesResult.ok) {
        return queriesResult;
      }
      config.assets.queries = config.assets.queries.map((query, idx) => {
        return {
          ...query,
          data: JSON.stringify(queriesResult.value[idx].toString('utf8'))
        };
      });
    }
    if (config.statics) {
      const staticsResult = await this.serializeStatics(config.statics);
      if (!staticsResult.ok) {
        return staticsResult;
      }
      config.statics = staticsResult.value;
    }
    if (config.sampleData) {
      const dataResult = await this.getSampleData(version);
      if (!dataResult.ok) {
        return dataResult;
      }
      config.sampleData = {
        ...config.sampleData,
        data: JSON.stringify(dataResult.value.sampleData)
      };
    }
    return {
      ok: true,
      value: config
    };
  }
}
exports.IntegrationReader = IntegrationReader;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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